/**
 * @file    control_panel.c
 * @brief   Routines for constructing and handling the control panel.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ygraph.h"

/**
 * @brief    Set the current frame, ie. the global_current_frame.
 *
 * @param    val  The index of the frame to be set.
 */
void
current_frame_set(gint val)
{
  global_current_frame = val;
}

/**
 * @brief    Add a value to the current_frame value.
 * 
 * @param    incr  The increment to be added to the global_current_frame.
 */
void
current_frame_increment(gint incr)
{
  global_current_frame += incr;

  if (global_current_frame < 0)
    {
      global_current_frame = 0;
      global_display_mode = PAUSE_MODE;
    }
  else if (global_current_frame > global_last_frame)
    {
      global_current_frame = global_last_frame;
      global_display_mode = PAUSE_MODE;
    }
}

/**
 * @brief    Display the value of the current_frame in the control panel.
 */
void
current_time_display_value(void)
{
  gdouble current_time;
  gchar time_str[TIME_STR_SIZE];

  if (global_time_list == NULL)
    current_time = 0;
  else
    current_time = g_array_index(global_time_list, gdouble, 
                                 global_current_frame);
  g_snprintf(time_str, TIME_STR_SIZE, "%g", current_time);

  if (global_control_panel != NULL)
    gtk_entry_set_text(GTK_ENTRY(global_control_panel->time_entry), time_str);
}

/**
 * @brief    Draw the control panel window.
 *
 * @returns  The control panel date structure.
 */
ControlPanel*
create_control_panel(void)
{
  ControlPanel* control_panel;
  GtkWidget* window;
  GtkWidget* vbox;
  GtkWidget* hbox_top;
  GtkWidget* hbox_bottom;
  GtkWidget* start_button;
  GtkWidget* step_back_button;
  GtkWidget* play_button;
  GtkWidget* step_fwd_button;
  GtkWidget* end_button;
  GtkWidget* show_all_button;
  GtkWidget* time_label;
  GtkWidget* time_entry;
  GtkWidget* delay_label;
  GtkWidget* delay_entry;
  GtkWidget* new_button;
  GtkWidget* reload_button;
  GtkWidget* quit_button;
  gchar delay_str[DELAY_STR_SIZE];

  control_panel = g_malloc(sizeof(ControlPanel));

  /*
   * Control panel window.
   */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
                     GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
  gtk_window_set_title(GTK_WINDOW(window), CONTROL_PANEL_NAME);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, TRUE);

  /*
   * Vertical box.
   */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(window), vbox);

  /*
   * Horizontal boxes.
   */
  hbox_top = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_top);
  gtk_box_pack_start(GTK_BOX(vbox), hbox_top, FALSE, FALSE, 0);

  hbox_bottom = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_bottom);
  gtk_box_pack_start(GTK_BOX(vbox), hbox_bottom, FALSE, FALSE, 0);

  /*
   * Top row of buttons.
   */
  start_button = gtk_button_new_with_label(START_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(start_button), "clicked",
                     GTK_SIGNAL_FUNC(start_button_click), NULL);
  gtk_widget_show(start_button);
  gtk_box_pack_start(GTK_BOX(hbox_top), start_button, FALSE, FALSE, 0);

  step_back_button = gtk_button_new_with_label(STEP_BACK_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(step_back_button), "clicked",
                     GTK_SIGNAL_FUNC(step_back_button_click), NULL);
  gtk_widget_show(step_back_button);
  gtk_box_pack_start(GTK_BOX(hbox_top), step_back_button, FALSE, FALSE, 0);

  play_button = gtk_button_new_with_label(PLAY_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(play_button), "clicked",
                     GTK_SIGNAL_FUNC(pause_play_button_click), NULL);
  gtk_widget_set_usize(play_button, 60, -2);
  gtk_widget_show(play_button);
  gtk_box_pack_start(GTK_BOX(hbox_top), play_button, FALSE, FALSE, 0);

  step_fwd_button = gtk_button_new_with_label(STEP_FWD_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(step_fwd_button), "clicked",
                     GTK_SIGNAL_FUNC(step_fwd_button_click), NULL);
  gtk_widget_show(step_fwd_button);
  gtk_box_pack_start(GTK_BOX(hbox_top), step_fwd_button, FALSE, FALSE, 0);

  end_button = gtk_button_new_with_label(END_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(end_button), "clicked",
                     GTK_SIGNAL_FUNC(end_button_click), NULL);
  gtk_widget_show(end_button);
  gtk_box_pack_start(GTK_BOX(hbox_top), end_button, FALSE, FALSE, 0);

  show_all_button = gtk_button_new_with_label(SHOW_ALL_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(show_all_button), "clicked",
                     GTK_SIGNAL_FUNC(show_all_button_click), NULL);
  gtk_widget_show(show_all_button);
  gtk_box_pack_start(GTK_BOX(hbox_top), show_all_button, TRUE, TRUE, 0);

  /*
   * Bottom row of buttons.
   */
  time_label = gtk_label_new(TIME_LABEL);
  gtk_widget_show(time_label);
  gtk_box_pack_start(GTK_BOX(hbox_bottom), time_label, FALSE, FALSE, 0);

  time_entry = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(time_entry), TIME_STR_SIZE);
  gtk_widget_show(time_entry);
  gtk_box_pack_start(GTK_BOX(hbox_bottom), time_entry, FALSE, FALSE, 0);
  gtk_widget_set_usize(time_entry, 60, -2);
  gtk_signal_connect(GTK_OBJECT(time_entry), "activate",
                     GTK_SIGNAL_FUNC(current_time_set), NULL);

  delay_label = gtk_label_new(DELAY_LABEL);
  gtk_widget_show(delay_label);
  gtk_box_pack_start(GTK_BOX(hbox_bottom), delay_label, FALSE, FALSE, 0);

  delay_entry = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(delay_entry), DELAY_STR_SIZE);
  g_snprintf(delay_str, DELAY_STR_SIZE, "%d", option_animate_delay);
  gtk_entry_set_text(GTK_ENTRY(delay_entry), delay_str);
  gtk_widget_show(delay_entry);
  gtk_box_pack_start(GTK_BOX(hbox_bottom), delay_entry, FALSE, FALSE, 0);
  gtk_widget_set_usize(delay_entry, 60, -2);
  gtk_signal_connect(GTK_OBJECT(delay_entry), "activate",
                     GTK_SIGNAL_FUNC(delay_set), NULL);

  new_button = gtk_button_new_with_label(NEW_BUTTON_LABEL);
  gtk_widget_show(new_button);
  gtk_box_pack_start(GTK_BOX(hbox_bottom), new_button, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(new_button), "clicked",
                     GTK_SIGNAL_FUNC(plot_window_empty_new), NULL);

  reload_button = gtk_button_new_with_label(RELOAD_BUTTON_LABEL);
  gtk_widget_show(reload_button);
  gtk_box_pack_start(GTK_BOX(hbox_bottom), reload_button, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(reload_button), "clicked",
                     GTK_SIGNAL_FUNC(data_reload_all), NULL);

  quit_button = gtk_button_new_with_label(QUIT_BUTTON_LABEL);
  gtk_signal_connect_object(GTK_OBJECT(quit_button), "clicked",
                            GTK_SIGNAL_FUNC(gtk_widget_destroy),
                            GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(hbox_bottom), quit_button, TRUE, TRUE, 0);
  gtk_widget_show(quit_button);

  /*
   * Link control panel handles so we have easy access to them later.
   */
  control_panel->window = window;
  control_panel->play_button = play_button;
  control_panel->time_entry = time_entry;
  control_panel->delay_entry = delay_entry;

  return control_panel;
}

/**
 * @brief    Create a new control panel, or raise it if it already exists.
 *
 */
void
control_panel_raise(void)
{
  if (global_control_panel != NULL)
    gdk_window_raise(global_control_panel->window->window);
  else
    {
      global_control_panel = create_control_panel();
      gtk_widget_show (global_control_panel->window);
      current_time_display_value();
    }
}
