/*
 * yauap - A simple commandline frontend for GStreamer
 * Copyright (c) 2006 Sascha Sommer <ssommer@suse.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 */


#ifndef YAUAP_H
#define YAUAP_H 1


/* the player struct: must be plain C code do not introdue glib or dbus specific things here 
* function pointers get filled by main.c 
* this struct should not be changed from within the frontends 
*/
typedef struct player_s{
    /* private data used by main.c - ignore */
    void* yauap_priv;
    /* player verbosity set to  1 to become more verbose */
    int verbose;
    /* methods (if not explained otherwise return 1 on success, 0 on failure) */
    int (*quit)(struct player_s* player);
    int (*pause)(struct player_s* player);
    /* check if a url can be decoded (files need to be prefixed with the file:// protocol)  
    * this function can be called at any time */ 
    int (*can_decode)(const char* url);
    int (*load)(struct player_s* player,const char* url);
    int (*stop)(struct player_s* player);
    int (*start)(struct player_s* player);
    /* get the current track length /  position in ms */
    unsigned int (*get_time_length)(struct player_s* player);
    unsigned int (*get_time_position)(struct player_s* player);
    /* seek to offset in ms */
    int (*seek)(struct player_s* player,unsigned int offset);
    /* return current volume in the range [0.0 - 100.0] */
    float (*get_volume)(struct player_s* player);
    int (*set_volume)(struct player_s* player,float volume);
    /* get a NULL terminated list of metadata for example:
    * author=blah 
    * title=test 
    * 
    * the name of the tags is similar to the name of the tracks in GStreamer   
    */
    int (*get_metadata)(struct player_s* player,char ***metadata);
    /* get audio cd contents: overwrites the set cdrom_device when device is NULL 
    * cdrom_device shall be an unix device 
    * the NULL terminated track list will look like 
    * 01=210 
    * 02=44  
    * 
    * eg. the track numer followed by = and the track length in seconds 
    * this function can be called at any time but remember that it overwrites the used cdrom_device 
    */ 
    int (*get_audio_cd_contents)(struct player_s* player,char* device,char*** tracks);
#define SCOPE_SIZE 2048
    /* get current scope buffer - data has to be at least SCOPE_SIZE bytes */
    /* scope data is raw 16 bit little endian 2 channel audio data */
    int (*get_scopedata)(struct player_s* player,char * data);
} player_t;


/* example call order for the various functions 

1. can_decode() / get_audio_cd_contents() to check if a file is playable
2. load() to load a file
3. start() to start playback
4. a combination of the following functions
   get_volume()
   set_volume()
   get_time_length() 
   seek()
   pause() (call start or pause again to unpause)
   get_metadata() to get the metadata for the current track
   get_scopedata()  get scopedata for audio visualization etc.
   get_time_position() to update postion sliders whatever
5. stop() to stop playback
6. either
   start() to restart playback of the current track from the beginning
   quit() to terminate the player and its control frontends
   or can_decode() / get_audio_cd_contents() again for new files
*/ 
   



/* yauap can be controlled by frontends 
   currently there is a commandline frontend in the commandline/ dir
   and a dbus service in the dbus-service/ dir
   these frontends use the player_t struct to control the main player code 
   They should return the following struct (currently this is hardwired in main.c):
*/

typedef struct yauap_frontend_s {
    /* deallocate the frontend resources (and the memory for the passed frontend_struct) */
    void (*free)(struct yauap_frontend_s* frontend);
    /* signal_cb() the player core in main.c will call this to inform the frontends when one of the following events happened: */
#define SIGNAL_METADATA 1       /* the streams metadata changed and should be refeteched with get_metadata() */
#define SIGNAL_EOS      2       /* end of stream reached or unplayable file */
#define SIGNAL_ERROR    3       /* an error occured: message will contain the error message */
    void (*signal_cb)(struct yauap_frontend_s* frontend,unsigned int signal,char* message);
    /* priv can be used to store private data from the frontend */
    void* priv;
    /* pointer to the player object (use this to control it */
    struct player_s* player;
} yauap_frontend_t;


#endif

