/*
 * @(#)BevelGL.c
 *
 * Copyright 2023 - 2024  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/*-
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 *event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Thanks goes also to Brian Paul for making it possible and inexpensive
 * to use OpenGL at home.
 *
 * Based on rubik.c by Marcelo F. Vianna <mfvianna@centroin.com.br>
 */

#ifdef HAVE_OPENGL

/* Methods file for BevelGL */

#include "BevelP.h"
#include "BevelGLP.h"

static float front_shininess[] =
{60.0};
static float front_specular[] =
{(float) 0.7, (float) 0.7, (float) 0.7, 1.0};
static float ambient[] =
{0.0, 0.0, 0.0, 1.0};
static float diffuse[] =
{1.0, 1.0, 1.0, 1.0};
static float position0[] =
{1.0, 1.0, 1.0, 0.0};
static float position1[] =
{-1.0, -1.0, 1.0, 0.0};
static float lmodel_ambient[] =
{0.5, 0.5, 0.5, 1.0};
static float lmodel_twoside[] =
{GL_TRUE};

static float face_material[MAX_FACES + 2][4] =
{{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{0.0, 0.0, 0.0, 1.0},
{0.0, 0.0, 0.0, 1.0}};

/*-
 * Color labels mapping:
 * =====================
 *
 *	  +------+
 *	  |3    0|
 *	  |      |
 *	  | TOP  |
 *	  | (0)  |
 *	  | RED  |
 *	  |2    1|
 * +------+------+------+
 * |3    0|3    0|3    0|
 * |      |      |      |
 * | LEFT |FRONT |RIGHT |
 * | (1)  | (2)  | (3)  |
 * |YELLOW| BLUE |ORANGE|
 * |2    1|2    1|2    1|
 * +------+------+------+
 *	  |3    0|
 *	  |      |
 *	  |BOTTOM|
 *	  | (4)  |
 *	  |WHITE |
 *	  |2    1|
 *	  +------+	 +-------+
 *	  |3    0|	 |3 /|\ 0|
 *	  |      |	 | / | \ |
 *	  | BACK |	 |/ 7|4 \|
 *	  | (5)  |       |-------|
 *	  |GREEN |	 |\ 6|5 /|
 *	  |2    1|	 | \ | / |
 *	  +------+	 |2 \|/ 1|
 *			 +-------+
 *
 *  Map to 3d
 *  FRONT  => X, Y
 *  BACK   => X, Y
 *  LEFT   => Z, Y
 *  RIGHT  => Z, Y
 *  TOP    => X, Z
 *  BOTTOM => X, Z
 */

static Boolean madeCurrent = False;

#ifdef WINVER
static HGLRC hRC = NULL;
#else
static GLXContext *glXContext = (GLXContext *) NULL;
static Boolean setValuesPuzzleGL(Widget current, Widget request, Widget renew);
static void resizePuzzleGL(BevelGLWidget w);
static void initializePuzzleGL(Widget request, Widget renew);
static void exposePuzzleGL(Widget renew, XEvent *event, Region region);
static void movePuzzleGLTl(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLTop(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLTr(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLLeft(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLRight(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBl(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBottom(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBr(BevelGLWidget w,
	XEvent *event, char **args, int nArgs);

static char translationsGL[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>F11: MoveCcw()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>F12: MoveCw()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>f: Find()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListGL[] =
{
	{(char *) "Quit", (XtActionProc) quitPuzzle},
	{(char *) "Hide", (XtActionProc) hidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) movePuzzleCcw},
	{(char *) "MoveTl", (XtActionProc) movePuzzleGLTl},
	{(char *) "MoveTop", (XtActionProc) movePuzzleGLTop},
	{(char *) "MoveTr", (XtActionProc) movePuzzleGLTr},
	{(char *) "MoveLeft", (XtActionProc) movePuzzleGLLeft},
	{(char *) "MoveCw", (XtActionProc) movePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) movePuzzleGLRight},
	{(char *) "MoveBl", (XtActionProc) movePuzzleGLBl},
	{(char *) "MoveBottom", (XtActionProc) movePuzzleGLBottom},
	{(char *) "MoveBr", (XtActionProc) movePuzzleGLBr},
	{(char *) "Select", (XtActionProc) selectPuzzle},
	{(char *) "Release", (XtActionProc) releasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) practicePuzzleWithQuery},
	{(char *) "Practice2", (XtActionProc) practicePuzzleWithDoubleClick},
	{(char *) "RandomizeMaybe", (XtActionProc) randomizePuzzleWithQuery},
	{(char *) "Randomize2", (XtActionProc) randomizePuzzleWithDoubleClick},
	{(char *) "Get", (XtActionProc) getPuzzle},
	{(char *) "Write", (XtActionProc) writePuzzle},
	{(char *) "Undo", (XtActionProc) undoPuzzle},
	{(char *) "Redo", (XtActionProc) redoPuzzle},
	{(char *) "Clear", (XtActionProc) clearPuzzle},
	{(char *) "Randomize", (XtActionProc) randomizePuzzle},
	{(char *) "Solve", (XtActionProc) solvePuzzle},
	{(char *) "Find", (XtActionProc) findPuzzle},
	{(char *) "Practice", (XtActionProc) practicePuzzle},
	{(char *) "Orientize", (XtActionProc) orientizePuzzle},
	{(char *) "View", (XtActionProc) viewPuzzle},
	{(char *) "Speed", (XtActionProc) speedUpPuzzle},
	{(char *) "Slow", (XtActionProc) slowDownPuzzle},
	{(char *) "Sound", (XtActionProc) toggleSoundPuzzle},
	{(char *) "Enter", (XtActionProc) enterPuzzle},
	{(char *) "Leave", (XtActionProc) leavePuzzle}
};

static XtResource resourcesGL[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(BevelWidget, core.width),
	 XtRString, (caddr_t) "300"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(BevelWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(BevelWidget, bevel.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(BevelWidget, bevel.background),
	 XtRString, (caddr_t) "#AEB2C3" /*XtDefaultBackground*/},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(BevelWidget, bevel.frameColor),
	 XtRString, (caddr_t) "Cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.faceName[1]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.faceName[2]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.faceName[3]),
	 XtRString, (caddr_t) "Orange"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.faceName[4]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.faceName[5]),
	 XtRString, (caddr_t) "Green"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(BevelWidget, bevel.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNview, XtCView, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.view),
	 XtRString, (caddr_t) "1"},
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULT_ORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(BevelWidget, bevel.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(BevelWidget, bevel.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.currentControl),
	 XtRString, (caddr_t) "0"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(BevelWidget, bevel.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(BevelWidget, bevel.select),
	 XtRCallback, (caddr_t) NULL}
};

BevelGLClassRec bevelGLClassRec =
{
	{
		(WidgetClass) & bevelClassRec,	/* superclass */
		(char *) "BevelGL",	/* class name */
		sizeof (BevelGLRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) initializePuzzleGL,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListGL,	/* actions */
		XtNumber(actionsListGL),	/* num actions */
		resourcesGL,	/* resources */
		XtNumber(resourcesGL),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) resizePuzzleGL,	/* resize */
		(XtExposeProc) exposePuzzleGL,	/* expose */
		(XtSetValuesFunc) setValuesPuzzleGL,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		translationsGL,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass bevelGLWidgetClass = (WidgetClass) & bevelGLClassRec;
#endif

static void
pickColor(BevelGLWidget w, int c, int mono, float *material)
{
#ifdef WINVER
	struct tagColor {
		int red, green, blue;
	} color;
#define MAX_INTENSITY 0xFF
#else
	XColor color;
#define MAX_INTENSITY 0xFFFF
#endif

	switch (c) {
	case TOP_FACE:
	case LEFT_FACE:
	case FRONT_FACE:
	case RIGHT_FACE:
	case BOTTOM_FACE:
	case BACK_FACE:
#ifdef WINVER
		color.red = GetRValue(w->bevel.faceGC[c]);
		color.green = GetGValue(w->bevel.faceGC[c]);
		color.blue = GetBValue(w->bevel.faceGC[c]);
#else
		color.pixel = w->bevel.faceColor[c];
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case BORDER_FACE:
#ifdef WINVER
		color.red = GetRValue(w->bevel.borderGC);
		color.green = GetGValue(w->bevel.borderGC);
		color.blue = GetBValue(w->bevel.borderGC);
#else
		color.pixel = w->bevel.borderColor;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case NO_FACE:
	default:
#ifdef WINVER
		color.red = GetRValue(w->bevel.inverseGC);
		color.green = GetGValue(w->bevel.inverseGC);
		color.blue = GetBValue(w->bevel.inverseGC);
#else
		color.pixel = w->bevel.background;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	}
#ifdef DEBUG
	(void) printf("%d: i%d %d %d\n", c, color.red, color.green, color.blue);
#endif
	if (mono) {
		/* really GrayScale */
		float intensity = (float) (0.3 * color.red +
			0.59 * color.green + 0.11 * color.blue);
#ifdef DEBUG
		(void) printf("m%g\n", intensity);
#endif
		material[0] = (float) intensity / MAX_INTENSITY;
		material[1] = material[0];
		material[2] = material[0];
	} else {
		material[0] = (float) color.red / MAX_INTENSITY;
		material[1] = (float) color.green / MAX_INTENSITY;
		material[2] = (float) color.blue / MAX_INTENSITY;
	}
#ifdef DEBUG
	(void) printf("%d: f%g %g %g\n", c, material[0], material[1], material[2]);
#endif
}

typedef float Matrix[3][3];
typedef float Vector[3];

static void
matrixMatrix(Matrix a, Matrix b, Matrix x)
{
	int i, j;

	for (i = 0; i < 3; i++) {
		for (j = 0; j < 3; j++) {
			x[i][j] = a[i][0] * b[0][j] +
				a[i][1] * b[1][j] +
				a[i][2] * b[2][j];
		}
	}
}

static void
matrixVector(Matrix a, Vector v, Vector x)
{
	int i;

	for (i = 0; i < 3; i++) {
		x[i] = a[i][0] * v[0] +
			a[i][1] * v[1] +
			a[i][2] * v[2];
	}
}

static void
setMatrix(float theta, float x, float y, float z, Matrix a)
{
	float s = (float) sin(theta * M_PI / 180.0);
	float c = (float) cos(theta * M_PI / 180.0);

	if (x != 0.0 && y == 0.0 && z == 0.0) {
		a[0][0] = 1.0, a[0][1] = 0.0; a[0][2] = 0.0;
		a[1][0] = 0.0, a[1][1] = c; a[1][2] = -s;
		a[2][0] = 0.0, a[2][1] = s; a[2][2] = c;
	} else if (y != 0.0 && x == 0.0 && z == 0.0) {
		a[0][0] = c, a[0][1] = 0.0; a[0][2] = s;
		a[1][0] = 0.0, a[1][1] = 1.0; a[1][2] = 0.0;
		a[2][0] = -s, a[2][1] = 0.0; a[2][2] = c;
	} else if (z != 0.0 && x == 0.0 && y == 0.0) {
		a[0][0] = c, a[0][1] = -s; a[0][2] = 0.0;
		a[1][0] = s, a[1][1] = c; a[1][2] = 0.0;
		a[2][0] = 0.0, a[2][1] = 0.0; a[2][2] = 1.0;
	}
}

static void
drawOuterStickerlessCubit(void)
{
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	/* Edge of cubit */
	glNormal3f(1.0, 1.0, 0.0);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glNormal3f(0.0, 1.0, 1.0);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glNormal3f(1.0, 0.0, 1.0);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glEnd();
	glBegin(GL_TRIANGLES);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	/* Put sticker here */
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glNormal3f(0.0, 1.0, 0.0);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	/* Corner of cubit */
	glNormal3f(1.0, 1.0, 1.0);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	/* Sharper corners of cubit */
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_LENGTH, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glNormal3f(1.0, -1.0, 1.0);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_LENGTH, CUBE_ROUND);
	glNormal3f(1.0, 1.0, -1.0);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_LENGTH);
	glEnd();
	glBegin(GL_POLYGON);
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glEnd();
}

static void
drawInnerStickerlessCubit(void)
{
	glBegin(GL_TRIANGLES);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(FACEROUND, CUT2, FACELEN);
	glVertex3f(CUT2, FACEROUND, FACELEN);
	glVertex3f(CUT2, CUT2, FACELEN);

	glVertex3f(FACEROUND, CUT2, FACELEN);
	glVertex3f(CUT2, FACEROUND, FACELEN);
	glVertex3f(CUT2, CUT2, CUBE_ROUND);

	glVertex3f(CUT2, FACEROUND, FACELEN);
	glVertex3f(CUT2, CUT2, FACELEN);
	glVertex3f(CUT2, CUT2, CUBE_ROUND);

	glVertex3f(CUT2, CUT2, FACELEN);
	glVertex3f(FACEROUND, CUT2, FACELEN);
	glVertex3f(CUT2, CUT2, CUBE_ROUND);
	glEnd();
}

static void
drawOrientLine(BevelGLWidget w, int face, int rotation,
		float size, float x, float y, float z)
{
	int err = 0;
	char *buf1, *buf2;
	float stickerLong = STICKER_LONG * size;

	glPushMatrix();
	glTranslatef(x, y, z);
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	switch (face) {
	case 0:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -stickerLong);
			glVertex3f(-CUT, 0.0, -stickerLong);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(stickerLong, 0.0, CUT);
			glVertex3f(stickerLong, 0.0, -CUT);
			break;
		case 2:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, stickerLong);
			glVertex3f(CUT, 0.0, stickerLong);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-stickerLong, 0.0, -CUT);
			glVertex3f(-stickerLong, 0.0, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 1:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, stickerLong, CUT);
			glVertex3f(0.0, stickerLong, -CUT);
			break;
		case 1:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, stickerLong);
			glVertex3f(0.0, CUT, stickerLong);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -stickerLong, -CUT);
			glVertex3f(0.0, -stickerLong, CUT);
			break;
		case 3:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, -stickerLong);
			glVertex3f(0.0, -CUT, -stickerLong);
			break;
		default:
			err = 1;
		}
		break;
	case 2:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, stickerLong, 0.0);
			glVertex3f(-CUT, stickerLong, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(stickerLong, -CUT, 0.0);
			glVertex3f(stickerLong, CUT, 0.0);
			break;
		case 2:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, -stickerLong, 0.0);
			glVertex3f(CUT, -stickerLong, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-stickerLong, CUT, 0.0);
			glVertex3f(-stickerLong, -CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	case 3:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, stickerLong, -CUT);
			glVertex3f(0.0, stickerLong, CUT);
			glVertex3f(0.0, CUT, CUT);
			break;
		case 1:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -CUT, -stickerLong);
			glVertex3f(0.0, CUT, -stickerLong);
			glVertex3f(0.0, CUT, -CUT);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -stickerLong, CUT);
			glVertex3f(0.0, -stickerLong, -CUT);
			glVertex3f(0.0, -CUT, -CUT);
			break;
		case 3:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, CUT, stickerLong);
			glVertex3f(0.0, -CUT, stickerLong);
			glVertex3f(0.0, -CUT, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 4:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(CUT, 0.0, stickerLong);
			glVertex3f(-CUT, 0.0, stickerLong);
			glVertex3f(-CUT, 0.0, CUT);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(stickerLong, 0.0, -CUT);
			glVertex3f(stickerLong, 0.0, CUT);
			glVertex3f(CUT, 0.0, CUT);
			break;
		case 2:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -stickerLong);
			glVertex3f(CUT, 0.0, -stickerLong);
			glVertex3f(CUT, 0.0, -CUT);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-stickerLong, 0.0, CUT);
			glVertex3f(-stickerLong, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 5:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(CUT, -stickerLong, 0.0);
			glVertex3f(-CUT, -stickerLong, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(stickerLong, CUT, 0.0);
			glVertex3f(stickerLong, -CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			break;
		case 2:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-CUT, stickerLong, 0.0);
			glVertex3f(CUT, stickerLong, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-stickerLong, -CUT, 0.0);
			glVertex3f(-stickerLong, CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	}
	glEnd();
	glPopMatrix();
	if (err == 1) {
		intCat(&buf1, "draw_orient_line: face ", face);
		stringCat(&buf2, buf1, ", rotation ");
		free(buf1);
		intCat(&buf1, buf2, rotation);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
}

static void
drawInnerCubit(BevelGLWidget w, int face, int position)
{
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[(int) w->bevel.cubeLoc[face][position + MAX_ORIENT].face]);
#ifdef DEBUG
		(void) printf("face=%d position=%d\n", face, position);
#endif
	switch (face) {
	case TOP_FACE:
		switch (position) {
		case 0:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 1.0, 0.0);
			glVertex3f(FSTICKER_LONG, FSTICKER_DEPTH, -CUT);
			glVertex3f(CUT, FSTICKER_DEPTH, -FSTICKER_LONG);
			glVertex3f(CUT, FSTICKER_DEPTH, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, FCUT_DEPTH, -FSTICKER_HALF);
			}
			break;
		case 1:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 1.0, 0.0);
			glVertex3f(CUT, FSTICKER_DEPTH, FSTICKER_LONG);
			glVertex3f(FSTICKER_LONG, FSTICKER_DEPTH, CUT);
			glVertex3f(CUT, FSTICKER_DEPTH, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, FCUT_DEPTH, FSTICKER_HALF);
			}
			break;
		case 2:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 1.0, 0.0);
			glVertex3f(-FSTICKER_LONG, FSTICKER_DEPTH, CUT);
			glVertex3f(-CUT, FSTICKER_DEPTH, FSTICKER_LONG);
			glVertex3f(-CUT, FSTICKER_DEPTH, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, FCUT_DEPTH, FSTICKER_HALF);
			}
			break;
		case 3:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 1.0, 0.0);
			glVertex3f(-CUT, FSTICKER_DEPTH, -FSTICKER_LONG);
			glVertex3f(-FSTICKER_LONG, FSTICKER_DEPTH, -CUT);
			glVertex3f(-CUT, FSTICKER_DEPTH, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, FCUT_DEPTH, -FSTICKER_HALF);
			}
			break;
		}
		break;
	case LEFT_FACE:
		switch (position) {
		case 0:
			glBegin(GL_TRIANGLES);
			glNormal3f(-1.0, 0.0, 0.0);
			glVertex3f(-FSTICKER_DEPTH, CUT, FSTICKER_LONG);
			glVertex3f(-FSTICKER_DEPTH, FSTICKER_LONG, CUT);
			glVertex3f(-FSTICKER_DEPTH, CUT, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FCUT_DEPTH, FTENTH, FSTICKER_HALF);
			}
			break;
		case 1:
			glBegin(GL_TRIANGLES);
			glNormal3f(-1.0, 0.0, 0.0);
			glVertex3f(-FSTICKER_DEPTH, -FSTICKER_LONG, CUT);
			glVertex3f(-FSTICKER_DEPTH, -CUT, FSTICKER_LONG);
			glVertex3f(-FSTICKER_DEPTH, -CUT, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FCUT_DEPTH, -FTENTH, FSTICKER_HALF);
			}
			break;
		case 2:
			glBegin(GL_TRIANGLES);
			glNormal3f(-1.0, 0.0, 0.0);
			glVertex3f(-FSTICKER_DEPTH, -CUT, -FSTICKER_LONG);
			glVertex3f(-FSTICKER_DEPTH, -FSTICKER_LONG, -CUT);
			glVertex3f(-FSTICKER_DEPTH, -CUT, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FCUT_DEPTH, -FTENTH, -FSTICKER_HALF);
			}
			break;
		case 3:
			glBegin(GL_TRIANGLES);
			glNormal3f(-1.0, 0.0, 0.0);
			glVertex3f(-FSTICKER_DEPTH, FSTICKER_LONG, -CUT);
			glVertex3f(-FSTICKER_DEPTH, CUT, -FSTICKER_LONG);
			glVertex3f(-FSTICKER_DEPTH, CUT, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FCUT_DEPTH, FTENTH, -FSTICKER_HALF);
			}
			break;
		}
		break;
	case FRONT_FACE:
		switch (position) {
		case 0:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, 1.0);
			glVertex3f(FSTICKER_LONG, CUT, FSTICKER_DEPTH);
			glVertex3f(CUT, FSTICKER_LONG, FSTICKER_DEPTH);
			glVertex3f(CUT, CUT, FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, FSTICKER_HALF, FCUT_DEPTH);
			}
			break;
		case 1:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, 1.0);
			glVertex3f(CUT, -FSTICKER_LONG, FSTICKER_DEPTH);
			glVertex3f(FSTICKER_LONG, -CUT, FSTICKER_DEPTH);
			glVertex3f(CUT, -CUT, FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, -FSTICKER_HALF, FCUT_DEPTH);
			}
			break;
		case 2:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, 1.0);
			glVertex3f(-FSTICKER_LONG, -CUT, FSTICKER_DEPTH);
			glVertex3f(-CUT, -FSTICKER_LONG, FSTICKER_DEPTH);
			glVertex3f(-CUT, -CUT, FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, -FSTICKER_HALF, FCUT_DEPTH);
			}
			break;
		case 3:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, 1.0);
			glVertex3f(-CUT, FSTICKER_LONG, FSTICKER_DEPTH);
			glVertex3f(-FSTICKER_LONG, CUT, FSTICKER_DEPTH);
			glVertex3f(-CUT, CUT, FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, FSTICKER_HALF, FCUT_DEPTH);
			}
			break;
		}
		break;
	case RIGHT_FACE:
		switch (position) {
		case 0:
			glBegin(GL_TRIANGLES);
			glNormal3f(1.0, 0.0, 0.0);
			glVertex3f(FSTICKER_DEPTH, CUT, -FSTICKER_LONG);
			glVertex3f(FSTICKER_DEPTH, FSTICKER_LONG, -CUT);
			glVertex3f(FSTICKER_DEPTH, CUT, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FCUT_DEPTH, FTENTH, -FSTICKER_HALF);
			}
			break;
		case 1:
			glBegin(GL_TRIANGLES);
			glNormal3f(1.0, 0.0, 0.0);
			glVertex3f(FSTICKER_DEPTH, -FSTICKER_LONG, -CUT);
			glVertex3f(FSTICKER_DEPTH, -CUT, -FSTICKER_LONG);
			glVertex3f(FSTICKER_DEPTH, -CUT, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FCUT_DEPTH, -FTENTH, -FSTICKER_HALF);
			}
			break;
		case 2:
			glBegin(GL_TRIANGLES);
			glNormal3f(1.0, 0.0, 0.0);
			glVertex3f(FSTICKER_DEPTH, -CUT, FSTICKER_LONG);
			glVertex3f(FSTICKER_DEPTH, -FSTICKER_LONG, CUT);
			glVertex3f(FSTICKER_DEPTH, -CUT, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FCUT_DEPTH, -FTENTH, FSTICKER_HALF);
			}
			break;
		case 3:
			glBegin(GL_TRIANGLES);
			glNormal3f(1.0, 0.0, 0.0);
			glVertex3f(FSTICKER_DEPTH, FSTICKER_LONG, CUT);
			glVertex3f(FSTICKER_DEPTH, CUT, FSTICKER_LONG);
			glVertex3f(FSTICKER_DEPTH, CUT, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FCUT_DEPTH, FTENTH, FSTICKER_HALF);
			}
			break;
		}
		break;
	case BOTTOM_FACE:
		switch (position) {
		case 0:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, -1.0, 0.0);
			glVertex3f(FSTICKER_LONG, -FSTICKER_DEPTH, CUT);
			glVertex3f(CUT, -FSTICKER_DEPTH, FSTICKER_LONG);
			glVertex3f(CUT, -FSTICKER_DEPTH, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, -FCUT_DEPTH, FSTICKER_HALF);
			}
			break;
		case 1:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, -1.0, 0.0);
			glVertex3f(CUT, -FSTICKER_DEPTH, -FSTICKER_LONG);
			glVertex3f(FSTICKER_LONG, -FSTICKER_DEPTH, -CUT);
			glVertex3f(CUT, -FSTICKER_DEPTH, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, -FCUT_DEPTH, -FSTICKER_HALF);
			}
			break;
		case 2:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, -1.0, 0.0);
			glVertex3f(-FSTICKER_LONG, -FSTICKER_DEPTH, -CUT);
			glVertex3f(-CUT, -FSTICKER_DEPTH, -FSTICKER_LONG);
			glVertex3f(-CUT, -FSTICKER_DEPTH, -CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, -FCUT_DEPTH, -FSTICKER_HALF);
			}
			break;
		case 3:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, -1.0, 0.0);
			glVertex3f(-CUT, -FSTICKER_DEPTH, FSTICKER_LONG);
			glVertex3f(-FSTICKER_LONG, -FSTICKER_DEPTH, CUT);
			glVertex3f(-CUT, -FSTICKER_DEPTH, CUT);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, -FCUT_DEPTH, FSTICKER_HALF);
			}
			break;
		}
		break;
	case BACK_FACE:
		switch (position) {
		case 0:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, -1.0);
			glVertex3f(FSTICKER_LONG, -CUT, -FSTICKER_DEPTH);
			glVertex3f(CUT, -FSTICKER_LONG, -FSTICKER_DEPTH);
			glVertex3f(CUT, -CUT, -FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, -FSTICKER_HALF, -FCUT_DEPTH);
			}
			break;
		case 1:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, -1.0);
			glVertex3f(CUT, FSTICKER_LONG, -FSTICKER_DEPTH);
			glVertex3f(FSTICKER_LONG, CUT, -FSTICKER_DEPTH);
			glVertex3f(CUT, CUT, -FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					FTENTH, FSTICKER_HALF, -FCUT_DEPTH);
			}
			break;
		case 2:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, -1.0);
			glVertex3f(-FSTICKER_LONG, CUT, -FSTICKER_DEPTH);
			glVertex3f(-CUT, FSTICKER_LONG, -FSTICKER_DEPTH);
			glVertex3f(-CUT, CUT, -FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, FSTICKER_HALF, -FCUT_DEPTH);
			}
			break;
		case 3:
			glBegin(GL_TRIANGLES);
			glNormal3f(0.0, 0.0, -1.0);
			glVertex3f(-CUT, -FSTICKER_LONG, -FSTICKER_DEPTH);
			glVertex3f(-FSTICKER_LONG, -CUT, -FSTICKER_DEPTH);
			glVertex3f(-CUT, -CUT, -FSTICKER_DEPTH);
			glEnd();
			if (w->bevel.orient) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][position + MAX_ORIENT].rotation,
					CUBE_LENGTH,
					-FTENTH, -FSTICKER_HALF, -FCUT_DEPTH);
			}
			break;
		}
		break;
	}
}

static int
positionCubit(int face,
	int back, int front, int left, int right, int bottom, int top) {

	if (back != NO_FACE && face == BACK_FACE) { /* Green */
		if (right != NO_FACE && bottom != NO_FACE)
			return 0;
		if (right != NO_FACE && top != NO_FACE)
			return 1;
		if (left != NO_FACE && top != NO_FACE)
			return 2;
		if (left != NO_FACE && bottom != NO_FACE)
			return 3;
	}
	if (front != NO_FACE && face == FRONT_FACE) { /* Blue */
		if (right != NO_FACE && top != NO_FACE)
			return 0;
		if (right != NO_FACE && bottom != NO_FACE)
			return 1;
		if (left != NO_FACE && bottom != NO_FACE)
			return 2;
		if (left != NO_FACE && top != NO_FACE)
			return 3;
	}
	if (left != NO_FACE && face == LEFT_FACE) { /* Yellow */
		if (front != NO_FACE && top != NO_FACE)
			return 0;
		if (front != NO_FACE && bottom != NO_FACE)
			return 1;
		if (back != NO_FACE && bottom != NO_FACE)
			return 2;
		if (back != NO_FACE && top != NO_FACE)
			return 3;
	}
	if (right != NO_FACE && face == RIGHT_FACE) { /* Orange */
		if (back != NO_FACE && top != NO_FACE)
			return 0;
		if (back != NO_FACE && bottom != NO_FACE)
			return 1;
		if (front != NO_FACE && bottom != NO_FACE)
			return 2;
		if (front != NO_FACE && top != NO_FACE)
			return 3;
	}
	if (bottom != NO_FACE && face == BOTTOM_FACE) { /* White */
		if (right != NO_FACE && front != NO_FACE)
			return 0;
		if (right != NO_FACE && back != NO_FACE)
			return 1;
		if (left != NO_FACE && back != NO_FACE)
			return 2;
		if (left != NO_FACE && front != NO_FACE)
			return 3;
	}
	if (top != NO_FACE && face == TOP_FACE) { /* Red */
		if (right != NO_FACE && back != NO_FACE)
			return 0;
		if (right != NO_FACE && front != NO_FACE)
			return 1;
		if (left != NO_FACE && front != NO_FACE)
			return 2;
		if (left != NO_FACE && back != NO_FACE)
			return 3;
	}
	return MAX_ORIENT;
}

/* so puzzle will not fall apart */
static void
drawCenter(void)
{
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	glBegin(GL_QUADS);
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(-CUBE_ROUND2, -CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, -CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, CUBE_ROUND2, CUBE_ROUND2);
	glNormal3f(0.0, 0.0, -1.0);
	glVertex3f(-CUBE_ROUND2, CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, -CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, -CUBE_ROUND2, -CUBE_ROUND2);
	glNormal3f(-1.0, 0.0, 0.0);
	glVertex3f(-CUBE_ROUND2, -CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, -CUBE_ROUND2, -CUBE_ROUND2);
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(CUBE_ROUND2, -CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, -CUBE_ROUND2, CUBE_ROUND2);
	glNormal3f(0.0, -1.0, 0.0);
	glVertex3f(CUBE_ROUND2, -CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, -CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, -CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, -CUBE_ROUND2, -CUBE_ROUND2);
	glNormal3f(0.0, 1.0, 0.0);
	glVertex3f(-CUBE_ROUND2, CUBE_ROUND2, -CUBE_ROUND2);
	glVertex3f(-CUBE_ROUND2, CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, CUBE_ROUND2, CUBE_ROUND2);
	glVertex3f(CUBE_ROUND2, CUBE_ROUND2, -CUBE_ROUND2);
	glEnd();
}

static void
drawOuterCubit(BevelGLWidget w,
		int back, int front, int left, int right, int bottom, int top)
{
	int face, position;
	if (back != NO_FACE) { /* Green */
		face = BACK_FACE;
		position = positionCubit(face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[back]);
		glNormal3f(0.0, 0.0, -1.0);
		if (top != NO_FACE) {
			glVertex3f(-STICKER_SHORT, STICKER_LONG, -STICKER_DEPTH);
			glVertex3f(STICKER_SHORT, STICKER_LONG, -STICKER_DEPTH);
		}
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, -STICKER_SHORT, -STICKER_DEPTH);
			glVertex3f(-STICKER_LONG, STICKER_SHORT, -STICKER_DEPTH);
		}
		if (bottom != NO_FACE) {
			glVertex3f(STICKER_SHORT, -STICKER_LONG, -STICKER_DEPTH);
			glVertex3f(-STICKER_SHORT, -STICKER_LONG, -STICKER_DEPTH);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, STICKER_SHORT, -STICKER_DEPTH);
			glVertex3f(STICKER_LONG, -STICKER_SHORT, -STICKER_DEPTH);
		}
		glEnd();
		if (w->bevel.currentFace == face &&
				w->bevel.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, -1.0);
			if (top != NO_FACE) {
				glVertex3f(-SELECT_SHORT, SELECT_LONG, -SELECT_DEPTH);
				glVertex3f(SELECT_SHORT, SELECT_LONG, -SELECT_DEPTH);
			}
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, -SELECT_SHORT, -SELECT_DEPTH);
				glVertex3f(-SELECT_LONG, SELECT_SHORT, -SELECT_DEPTH);
			}
			if (bottom != NO_FACE) {
				glVertex3f(SELECT_SHORT, -SELECT_LONG, -SELECT_DEPTH);
				glVertex3f(-SELECT_SHORT, -SELECT_LONG, -SELECT_DEPTH);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, SELECT_SHORT, -SELECT_DEPTH);
				glVertex3f(SELECT_LONG, -SELECT_SHORT, -SELECT_DEPTH);
			}
			glEnd();
		}
		if (w->bevel.orient) {
			if (right != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][0].rotation,
					CUBE_LENGTH,
					TENTH, -STICKER_HALF, -CUT_DEPTH);
			}
			if (right != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][1].rotation,
					CUBE_LENGTH,
					TENTH, STICKER_HALF, -CUT_DEPTH);
			}
			if (left != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][2].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, TENTH, -CUT_DEPTH);
			}
			if (left != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][3].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, -TENTH, -CUT_DEPTH);
			}
		}
	}
	if (front != NO_FACE) { /* Blue */
		face = FRONT_FACE;
		position = positionCubit(face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[front]);
		glNormal3f(0.0, 0.0, 1.0);
		if (top != NO_FACE) {
			glVertex3f(STICKER_SHORT, STICKER_LONG, STICKER_DEPTH);
			glVertex3f(-STICKER_SHORT, STICKER_LONG, STICKER_DEPTH);
		}
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, STICKER_SHORT, STICKER_DEPTH);
			glVertex3f(-STICKER_LONG, -STICKER_SHORT, STICKER_DEPTH);
		}
		if (bottom != NO_FACE) {
			glVertex3f(-STICKER_SHORT, -STICKER_LONG, STICKER_DEPTH);
			glVertex3f(STICKER_SHORT, -STICKER_LONG, STICKER_DEPTH);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, -STICKER_SHORT, STICKER_DEPTH);
			glVertex3f(STICKER_LONG, STICKER_SHORT, STICKER_DEPTH);
		}
		glEnd();
		if (w->bevel.currentFace == face &&
				w->bevel.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, 1.0);
			if (top != NO_FACE) {
				glVertex3f(SELECT_SHORT, SELECT_LONG, SELECT_DEPTH);
				glVertex3f(-SELECT_SHORT, SELECT_LONG, SELECT_DEPTH);
			}
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, SELECT_SHORT, SELECT_DEPTH);
				glVertex3f(-SELECT_LONG, -SELECT_SHORT, SELECT_DEPTH);
			}
			if (bottom != NO_FACE) {
				glVertex3f(-SELECT_SHORT, -SELECT_LONG, SELECT_DEPTH);
				glVertex3f(SELECT_SHORT, -SELECT_LONG, SELECT_DEPTH);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, -SELECT_SHORT, SELECT_DEPTH);
				glVertex3f(SELECT_LONG, SELECT_SHORT, SELECT_DEPTH);
			}
			glEnd();
		}
		if (w->bevel.orient) {
			if (right != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][0].rotation,
					CUBE_LENGTH,
					TENTH, STICKER_HALF, CUT_DEPTH);
			}
			if (right != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][1].rotation,
					CUBE_LENGTH,
					TENTH, -STICKER_HALF, CUT_DEPTH);
			}
			if (left != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][2].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, -TENTH, CUT_DEPTH);
			}
			if (left != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][3].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, TENTH, CUT_DEPTH);
			}
		}
	}
	if (left != NO_FACE) { /* Yellow */
		face = LEFT_FACE;
		position = positionCubit(face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[left]);
		glNormal3f(-1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, -STICKER_SHORT, STICKER_LONG);
			glVertex3f(-STICKER_DEPTH, STICKER_SHORT, STICKER_LONG);
		}
		if (top != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, STICKER_LONG, STICKER_SHORT);
			glVertex3f(-STICKER_DEPTH, STICKER_LONG, -STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, STICKER_SHORT, -STICKER_LONG);
			glVertex3f(-STICKER_DEPTH, -STICKER_SHORT, -STICKER_LONG);
		}
		if (bottom != NO_FACE) {
			glVertex3f(-STICKER_DEPTH, -STICKER_LONG, -STICKER_SHORT);
			glVertex3f(-STICKER_DEPTH, -STICKER_LONG, STICKER_SHORT);
		}
		glEnd();
		if (w->bevel.currentFace == face &&
				w->bevel.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(-1.0, 0.0, 0.0);
			if (front != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, -SELECT_SHORT, SELECT_LONG);
				glVertex3f(-SELECT_DEPTH, SELECT_SHORT, SELECT_LONG);
			}
			if (top != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, SELECT_LONG, SELECT_SHORT);
				glVertex3f(-SELECT_DEPTH, SELECT_LONG, -SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, SELECT_SHORT, -SELECT_LONG);
				glVertex3f(-SELECT_DEPTH, -SELECT_SHORT, -SELECT_LONG);
			}
			if (bottom != NO_FACE) {
				glVertex3f(-SELECT_DEPTH, -SELECT_LONG, -SELECT_SHORT);
				glVertex3f(-SELECT_DEPTH, -SELECT_LONG, SELECT_SHORT);
			}
			glEnd();
		}
		if (w->bevel.orient) {
			if (front != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][0].rotation,
					CUBE_LENGTH,
					-CUT_DEPTH, TENTH, STICKER_HALF);
			}
			if (front != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][1].rotation,
					CUBE_LENGTH,
					-CUT_DEPTH, -TENTH, STICKER_HALF);
			}
			if (back != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][2].rotation,
					CUBE_LENGTH,
					-CUT_DEPTH, -STICKER_HALF, -TENTH);
			}
			if (back != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][3].rotation,
					CUBE_LENGTH,
					-CUT_DEPTH,  STICKER_HALF, -TENTH);
			}
		}
	}
	if (right != NO_FACE) { /* Orange */
		face = RIGHT_FACE;
		position = positionCubit(face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[right]);
		glNormal3f(1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glVertex3f(STICKER_DEPTH, STICKER_SHORT, STICKER_LONG);
			glVertex3f(STICKER_DEPTH, -STICKER_SHORT, STICKER_LONG);
		}
		if (top != NO_FACE) {
			glVertex3f(STICKER_DEPTH, STICKER_LONG, -STICKER_SHORT);
			glVertex3f(STICKER_DEPTH, STICKER_LONG, STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(STICKER_DEPTH, -STICKER_SHORT, -STICKER_LONG);
			glVertex3f(STICKER_DEPTH, STICKER_SHORT, -STICKER_LONG);
		}
		if (bottom != NO_FACE) {
			glVertex3f(STICKER_DEPTH, -STICKER_LONG, STICKER_SHORT);
			glVertex3f(STICKER_DEPTH, -STICKER_LONG, -STICKER_SHORT);
		}
		glEnd();
		if (w->bevel.currentFace == face &&
				w->bevel.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(1.0, 0.0, 0.0);
			if (front != NO_FACE) {
				glVertex3f(SELECT_DEPTH, SELECT_SHORT, SELECT_LONG);
				glVertex3f(SELECT_DEPTH, -SELECT_SHORT, SELECT_LONG);
			}
			if (top != NO_FACE) {
				glVertex3f(SELECT_DEPTH, SELECT_LONG, -SELECT_SHORT);
				glVertex3f(SELECT_DEPTH, SELECT_LONG, SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(SELECT_DEPTH, -SELECT_SHORT, -SELECT_LONG);
				glVertex3f(SELECT_DEPTH, SELECT_SHORT, -SELECT_LONG);
			}
			if (bottom != NO_FACE) {
				glVertex3f(SELECT_DEPTH, -SELECT_LONG, SELECT_SHORT);
				glVertex3f(SELECT_DEPTH, -SELECT_LONG, -SELECT_SHORT);
			}
			glEnd();
		}
		if (w->bevel.orient) {
			if (back != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][0].rotation,
					CUBE_LENGTH,
					CUT_DEPTH, STICKER_HALF, -TENTH);
			}
			if (back != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][1].rotation,
					CUBE_LENGTH,
					CUT_DEPTH, -STICKER_HALF, -TENTH);
			}
			if (front != NO_FACE && bottom != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][2].rotation,
					CUBE_LENGTH,
					CUT_DEPTH, -TENTH, STICKER_HALF);
			}
			if (front != NO_FACE && top != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[RIGHT_FACE][3].rotation,
					CUBE_LENGTH,
					CUT_DEPTH, TENTH, STICKER_HALF);
			}
		}
	}
	if (bottom != NO_FACE) { /* White */
		face = BOTTOM_FACE;
		position = positionCubit(face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[bottom]);
		glNormal3f(0.0, -1.0, 0.0);
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, -STICKER_DEPTH, STICKER_SHORT);
			glVertex3f(-STICKER_LONG, -STICKER_DEPTH, -STICKER_SHORT);
		}
		if (front != NO_FACE) {
			glVertex3f(STICKER_SHORT, -STICKER_DEPTH, STICKER_LONG);
			glVertex3f(-STICKER_SHORT, -STICKER_DEPTH, STICKER_LONG);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, -STICKER_DEPTH, -STICKER_SHORT);
			glVertex3f(STICKER_LONG, -STICKER_DEPTH, STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(-STICKER_SHORT, -STICKER_DEPTH, -STICKER_LONG);
			glVertex3f(STICKER_SHORT, -STICKER_DEPTH, -STICKER_LONG);
		}
		glEnd();
		if (w->bevel.currentFace == face &&
				w->bevel.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, -1.0, 0.0);
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, -SELECT_DEPTH, SELECT_SHORT);
				glVertex3f(-SELECT_LONG, -SELECT_DEPTH, -SELECT_SHORT);
			}
			if (front != NO_FACE) {
				glVertex3f(SELECT_SHORT, -SELECT_DEPTH, SELECT_LONG);
				glVertex3f(-SELECT_SHORT, -SELECT_DEPTH, SELECT_LONG);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, -SELECT_DEPTH, -SELECT_SHORT);
				glVertex3f(SELECT_LONG, -SELECT_DEPTH, SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(-SELECT_SHORT, -SELECT_DEPTH, -SELECT_LONG);
				glVertex3f(SELECT_SHORT, -SELECT_DEPTH, -SELECT_LONG);
			}
			glEnd();
		}
		if (w->bevel.orient) {
			if (right != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][0].rotation,
					CUBE_LENGTH,
					STICKER_HALF, -CUT_DEPTH, TENTH);
			}
			if (right != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][1].rotation,
					CUBE_LENGTH,
					STICKER_HALF, -CUT_DEPTH, -TENTH);
			}
			if (left != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][2].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, -CUT_DEPTH, -TENTH);
			}
			if (left != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][3].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, -CUT_DEPTH, TENTH);
			}
		}
	}
	if (top != NO_FACE) { /* Red */
		face = TOP_FACE;
		position = positionCubit(face, back, front, left, right, bottom, top);
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[top]);
		glNormal3f(0.0, 1.0, 0.0);
		if (left != NO_FACE) {
			glVertex3f(-STICKER_LONG, STICKER_DEPTH, -STICKER_SHORT);
			glVertex3f(-STICKER_LONG, STICKER_DEPTH, STICKER_SHORT);
		}
		if (front != NO_FACE) {
			glVertex3f(-STICKER_SHORT, STICKER_DEPTH, STICKER_LONG);
			glVertex3f(STICKER_SHORT, STICKER_DEPTH, STICKER_LONG);
		}
		if (right != NO_FACE) {
			glVertex3f(STICKER_LONG, STICKER_DEPTH, STICKER_SHORT);
			glVertex3f(STICKER_LONG, STICKER_DEPTH, -STICKER_SHORT);
		}
		if (back != NO_FACE) {
			glVertex3f(STICKER_SHORT, STICKER_DEPTH, -STICKER_LONG);
			glVertex3f(-STICKER_SHORT, STICKER_DEPTH, -STICKER_LONG);
		}
		glEnd();
		if (w->bevel.currentFace == face &&
				w->bevel.currentPosition == position) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 1.0, 0.0);
			if (left != NO_FACE) {
				glVertex3f(-SELECT_LONG, SELECT_DEPTH, -SELECT_SHORT);
				glVertex3f(-SELECT_LONG, SELECT_DEPTH, SELECT_SHORT);
			}
			if (front != NO_FACE) {
				glVertex3f(-SELECT_SHORT, SELECT_DEPTH, SELECT_LONG);
				glVertex3f(SELECT_SHORT, SELECT_DEPTH, SELECT_LONG);
			}
			if (right != NO_FACE) {
				glVertex3f(SELECT_LONG, SELECT_DEPTH, SELECT_SHORT);
				glVertex3f(SELECT_LONG, SELECT_DEPTH, -SELECT_SHORT);
			}
			if (back != NO_FACE) {
				glVertex3f(SELECT_SHORT, SELECT_DEPTH, -SELECT_LONG);
				glVertex3f(-SELECT_SHORT, SELECT_DEPTH, -SELECT_LONG);
			}
			glEnd();
		}
		if (w->bevel.orient) {
			if (right != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][0].rotation,
					CUBE_LENGTH,
					STICKER_HALF, CUT_DEPTH, -TENTH);
			}
			if (right != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][1].rotation,
					CUBE_LENGTH,
					STICKER_HALF, CUT_DEPTH, TENTH);
			}
			if (left != NO_FACE && front != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][2].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, CUT_DEPTH, TENTH);
			}
			if (left != NO_FACE && back != NO_FACE) {
				drawOrientLine(w, face,
					w->bevel.cubeLoc[face][3].rotation,
					CUBE_LENGTH,
					-STICKER_HALF, CUT_DEPTH, -TENTH);
			}
		}
	}
}

static void
drawWholeInnerCubit(BevelGLWidget w, int x, int y, int z, int x1, int y1, int z1)
{
	if (x < 0) {
		if (y1 > 0 && z1 > 0) {
			/* LEFT_FACE, Yellow, 1 (red blue) */
			glPushMatrix();
			glRotatef(-90.0, 0, 1, 0);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 1, 0);
			glPopMatrix();
		} else if (y1 < 0 && z1 > 0) {
			/* LEFT_FACE, Yellow, 1 (white blue)*/
			glPushMatrix();
			glRotatef(-90.0, 0, 1, 0);
			glRotatef(-90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 1, 1);
			glPopMatrix();
		} else if (y1 < 0 && z1 < 0) {
			/* LEFT_FACE, Yellow, 1 (white green)*/
			glPushMatrix();
			glRotatef(-90.0, 0, 1, 0);
			glRotatef(180.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 1, 2);
			glPopMatrix();
		} else if (y1 > 0 && z1 < 0) {
			/* LEFT_FACE, Yellow, 1 (red green)*/
			glPushMatrix();
			glRotatef(-90.0, 0, 1, 0);
			glRotatef(90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 1, 3);
			glPopMatrix();
		}
	} else if (x > 0) {
		if (y1 > 0 && z1 < 0) {
			 /* RIGHT_FACE, Orange, 3 (green red)*/
			glPushMatrix();
			glRotatef(90.0, 0, 1, 0);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 3, 0);
			glPopMatrix();
		} else if (y1 < 0 && z1 < 0) {
			 /* RIGHT_FACE, Orange, 3 (green white)*/
			glPushMatrix();
			glRotatef(90.0, 0, 1, 0);
			glRotatef(-90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 3, 1);
			glPopMatrix();
		} else if (y1 < 0 && z1 > 0) {
			 /* RIGHT_FACE, Orange, 3 (blue white)*/
			glPushMatrix();
			glRotatef(90.0, 0, 1, 0);
			glRotatef(180.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 3, 2);
			glPopMatrix();
		} else if (y1 > 0 && z1 > 0) {
			 /* RIGHT_FACE, Orange, 3 (blue red)*/
			glPushMatrix();
			glRotatef(90.0, 0, 1, 0);
			glRotatef(90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 3, 3);
			glPopMatrix();
		}
	} else if (y < 0) {
		if (x1 > 0 && z1 > 0) {
			/* BOTTOM_FACE, White 4 (blue orange)*/
			glPushMatrix();
			glRotatef(90.0, 1, 0, 0);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 4, 0);
			glPopMatrix();
		} else if (x1 > 0 && z1 < 0) {
			/* BOTTOM_FACE, White 4 (green orange)*/
			glPushMatrix();
			glRotatef(90.0, 1, 0, 0);
			glRotatef(-90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 4, 1);
			glPopMatrix();
		} else if (x1 < 0 && z1 < 0) {
			/* BOTTOM_FACE, White 4 (green yellow)*/
			glPushMatrix();
			glRotatef(90.0, 1, 0, 0);
			glRotatef(180.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 4, 2);
			glPopMatrix();
		} else if (x1 < 0 && z1 > 0) {
			/* BOTTOM_FACE, White 4 (blue yellow)*/
			glPushMatrix();
			glRotatef(90.0, 1, 0, 0);
			glRotatef(90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 4, 3);
			glPopMatrix();
		}
	} else if (y > 0) {
		if (x1 > 0 && z1 < 0) {
			/* TOP_FACE, Red 0 (green orange)*/
			glPushMatrix();
			glRotatef(-90.0, 1, 0, 0);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 0, 0);
			glPopMatrix();
		} else if (x1 > 0 && z1 > 0) {
			/* TOP_FACE, Red 0 (blue orange)*/
			glPushMatrix();
			glRotatef(-90.0, 1, 0, 0);
			glRotatef(-90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 0, 1);
			glPopMatrix();
		} else if (x1 < 0 && z1 > 0) {
			/* TOP_FACE, Red 0 (blue yellow)*/
			glPushMatrix();
			glRotatef(-90.0, 1, 0, 0);
			glRotatef(180.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 0, 2);
			glPopMatrix();
		} else if (x1 < 0 && z1 < 0) {
			/* TOP_FACE, Red 0 (green yellow)*/
			glPushMatrix();
			glRotatef(-90.0, 1, 0, 0);
			glRotatef(90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 0, 3);
			glPopMatrix();
		}
	} else if (z < 0) {
		if (x1 > 0 && y1 < 0) {
			/* BACK_FACE, Green 5 (white orange)*/
			glPushMatrix();
			glRotatef(180.0, 0, 1, 0);
			glRotatef(180.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 5, 0);
			glPopMatrix();
		} else if (x1 > 0 && y1 > 0) {
			/* BACK_FACE, Green 5 (red orange)*/
			glPushMatrix();
			glRotatef(180.0, 0, 1, 0);
			glRotatef(90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 5, 1);
			glPopMatrix();
		} else if (x1 < 0 && y1 > 0) {
			/* BACK_FACE, Green 5 (red yellow)*/
			glPushMatrix();
			glRotatef(180.0, 0, 2, 0);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 5, 2);
			glPopMatrix();
		} else if (x1 < 0 && y1 < 0) {
			/* BACK_FACE, Green 5 (white yellow)*/
			glPushMatrix();
			glRotatef(180.0, 0, 1, 0);
			glRotatef(-90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 5, 3);
			glPopMatrix();
		}
	} else if (z > 0) {
		if (x1 > 0 && y1 > 0) {
			/* FRONT_FACE, Blue, 2 (orange white)*/
			glPushMatrix();
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 2, 0);
			glPopMatrix();
		} else if (x1 > 0 && y1 < 0) {
			/* FRONT_FACE, Blue, 2 (orange red)*/
			glPushMatrix();
			glRotatef(-90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 2, 1);
			glPopMatrix();
		} else if (x1 < 0 && y1 < 0) {
			/* FRONT_FACE, Blue, 2 (yellow red)*/
			glPushMatrix();
			glRotatef(180.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 2, 2);
			glPopMatrix();
		} else if (x1 < 0 && y1 > 0) {
			/* FRONT_FACE, Blue, 2 (yellow white)*/
			glPushMatrix();
			glRotatef(90.0, 0, 0, 1);
			drawInnerStickerlessCubit();
			glPopMatrix();
			glPushMatrix();
			drawInnerCubit(w, 2, 3);
			glPopMatrix();
		}
	}
}

static void
drawWholeOuterCubit(BevelGLWidget w, int x, int y, int z)
{
	const float h = 0.5;

	if (x < 0) {
		if (y < 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(-h, -h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, 1, 0);
				glRotatef(180.0, 1, 0, 0);
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* BLL */
					w->bevel.cubeLoc[BACK_FACE][3].face,
					NO_FACE,
					w->bevel.cubeLoc[LEFT_FACE][2].face,
					NO_FACE,
					w->bevel.cubeLoc[BOTTOM_FACE][2].face,
					NO_FACE);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(-h, -h, h);
				glPushMatrix();
				glRotatef(180.0, 0, 0, 1);
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* FLL */
					NO_FACE,
					w->bevel.cubeLoc[FRONT_FACE][2].face,
					w->bevel.cubeLoc[LEFT_FACE][1].face,
					NO_FACE,
					w->bevel.cubeLoc[BOTTOM_FACE][3].face,
					NO_FACE);
				glPopMatrix();
			}
		} else if (y > 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(-h, h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, -1, 0);
				glRotatef(90.0, 0, 0, 1);
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* BUL */
					w->bevel.cubeLoc[BACK_FACE][2].face,
					NO_FACE,
					w->bevel.cubeLoc[LEFT_FACE][3].face,
					NO_FACE,
					NO_FACE,
					w->bevel.cubeLoc[TOP_FACE][3].face);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(-h, h, h);
				glPushMatrix();
				glRotatef(90.0, 0, 0, 1);
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* FUL */
					NO_FACE,
					w->bevel.cubeLoc[FRONT_FACE][3].face,
					w->bevel.cubeLoc[LEFT_FACE][0].face,
					NO_FACE,
					NO_FACE,
					w->bevel.cubeLoc[TOP_FACE][2].face);
				glPopMatrix();
			}
		}
	} else if (x > 0) {
		if (y < 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(h, -h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, 1, 0);
				glRotatef(90.0, 1, 0, 0);
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* BLR */
					w->bevel.cubeLoc[BACK_FACE][0].face,
					NO_FACE,
					NO_FACE,
					w->bevel.cubeLoc[RIGHT_FACE][1].face,
					w->bevel.cubeLoc[BOTTOM_FACE][1].face,
					NO_FACE);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(h, -h, h);
				glPushMatrix();
				glRotatef(90.0, 1, 0, 0);
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* FLR */
					NO_FACE,
					w->bevel.cubeLoc[FRONT_FACE][1].face,
					NO_FACE,
					w->bevel.cubeLoc[RIGHT_FACE][2].face,
					w->bevel.cubeLoc[BOTTOM_FACE][0].face,
					NO_FACE);
				glPopMatrix();
			}
		} else if (y > 0) {
			if (z < 0) {
				glPushMatrix();
				glTranslatef(h, h, -h);
				glPushMatrix();
				glRotatef(90.0, 0, 1, 0);
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* BUR */
					w->bevel.cubeLoc[BACK_FACE][1].face,
					NO_FACE,
					NO_FACE,
					w->bevel.cubeLoc[RIGHT_FACE][0].face,
					NO_FACE,
					w->bevel.cubeLoc[TOP_FACE][0].face);
				glPopMatrix();
			} else if (z > 0) {
				glPushMatrix();
				glTranslatef(h, h, h);
				glPushMatrix();
				drawOuterStickerlessCubit();
				glPopMatrix();
				drawOuterCubit(w, /* FUR */
					NO_FACE,
					w->bevel.cubeLoc[FRONT_FACE][0].face,
					NO_FACE,
					w->bevel.cubeLoc[RIGHT_FACE][3].face,
					NO_FACE,
					w->bevel.cubeLoc[TOP_FACE][1].face);
				glPopMatrix();
			}
		}
	}
}

static int nextFace[MAX_FACES][MAX_ORIENT] =
{
	{-4, -1, 4, 1},
	{-1, 2, 1, -2},
	{-4, 2, 4, -2},
	{1, 2, -1, -2},
	{-4, 1, 4, -1},
	{-4, -2, 4, 2}
};

static void
convertMove(BevelMove move, BevelSlice * slice)
{
	int position = move.position;

	switch (move.face) {
	case 0:
		if (move.direction % 2 == 0) {
			if (position == 0 || position == 1) {
				(*slice).edge = 11;
				(*slice).rotation = (move.direction == 2) ?
					CCW : CW;
			} else if (position == 2 || position == 3) {
				(*slice).edge = 9;
				(*slice).rotation = (move.direction == 0) ?
					CCW : CW;
			}
		} else {
			if (position == 0 || position == 3) {
				(*slice).edge = 2;
				(*slice).rotation = (move.direction == 1) ?
					CCW : CW;
			} else if (position == 1 || position == 2) {
				(*slice).edge = 3;
				(*slice).rotation = (move.direction == 3) ?
					CCW : CW;
			}
		}
		break;
	case 1:
		if (move.direction % 2 == 0) {
			if (position == 0 || position == 1) {
				(*slice).edge = 5;
				(*slice).rotation = (move.direction == 2) ?
					CCW : CW;
			} else if (position == 2 || position == 3) {
				(*slice).edge = 4;
				(*slice).rotation = (move.direction == 0) ?
					CCW : CW;
			}
		} else {
			if (position == 0 || position == 3) {
				(*slice).edge = 9;
				(*slice).rotation = (move.direction == 1) ?
					CCW : CW;
			} else if (position == 1 || position == 2) {
				(*slice).edge = 8;
				(*slice).rotation = (move.direction == 3) ?
					CCW : CW;
			}
		}
		break;
	case 2:
		if (move.direction % 2 == 0) {
			if (position == 0 || position == 1) {
				(*slice).edge = 7;
				(*slice).rotation = (move.direction == 2) ?
					CCW : CW;
			} else if (position == 2 || position == 3) {
				(*slice).edge = 5;
				(*slice).rotation = (move.direction == 0) ?
					CCW : CW;
			}
		} else {
			if (position == 0 || position == 3) {
				(*slice).edge = 3;
				(*slice).rotation = (move.direction == 1) ?
					CCW : CW;
			} else if (position == 1 || position == 2) {
				(*slice).edge = 1;
				(*slice).rotation = (move.direction == 3) ?
					CCW : CW;
			}
		}
		break;
	case 3:
		if (move.direction % 2 == 0) {
			if (position == 0 || position == 1) {
				(*slice).edge = 6;
				(*slice).rotation = (move.direction == 2) ?
					CCW : CW;
			} else if (position == 2 || position == 3) {
				(*slice).edge = 7;
				(*slice).rotation = (move.direction == 0) ?
					CCW : CW;
			}
		} else {
			if (position == 0 || position == 3) {
				(*slice).edge = 11;
				(*slice).rotation = (move.direction == 1) ?
					CCW : CW;
			} else if (position == 1 || position == 2) {
				(*slice).edge = 10;
				(*slice).rotation = (move.direction == 3) ?
					CCW : CW;
			}
		}
		break;
	case 4:
		if (move.direction % 2 == 0) {
			if (position == 0 || position == 1) {
				(*slice).edge = 10;
				(*slice).rotation = (move.direction == 2) ?
					CCW : CW;
			} else if (position == 2 || position == 3) {
				(*slice).edge = 8;
				(*slice).rotation = (move.direction == 0) ?
					CCW : CW;
			}
		} else {
			if (position == 0 || position == 3) {
				(*slice).edge = 1;
				(*slice).rotation = (move.direction == 1) ?
					CCW : CW;
			} else if (position == 1 || position == 2) {
				(*slice).edge = 0;
				(*slice).rotation = (move.direction == 3) ?
					CCW : CW;
			}
		}
		break;
	case 5:
		if (move.direction % 2 == 0) {
			if (position == 0 || position == 1) {
				(*slice).edge = 6;
				(*slice).rotation = (move.direction == 2) ?
					CCW : CW;
			} else if (position == 2 || position == 3) {
				(*slice).edge = 4;
				(*slice).rotation = (move.direction == 0) ?
					CCW : CW;
			}
		} else {
			if (position == 0 || position == 3) {
				(*slice).edge = 0;
				(*slice).rotation = (move.direction == 1) ?
					CCW : CW;
			} else if (position == 1 || position == 2) {
				(*slice).edge = 2;
				(*slice).rotation = (move.direction == 3) ?
					CCW : CW;
			}
		}
		break;
	}
}

/*-
 * If you want to rotate a slice, you can draw that at the anglestep,
  then the without anglestep.
 * There is a sequence for drawing cubies for each axis being moved...
 */
static void
edgeRotate(BevelGLWidget w, int edge, GLfloat rotateStep)
{ /* simiar to Dino period 2 */
	int Xf = 0, Yf = 0, Zf = 0;
	int X1 = 0, Y1 = 0, Z1 = 0;
	int Xn = 0, Yn = 0, Zn = 0;

	if (edge < 4) {
		Yf = (edge / 2 == 1) ? 1.0 : -1.0;
		Zf = (edge % 2 == 1) ? 1.0 : -1.0;
		X1 = -1;
		Xn = 1;
	} else if (edge >= 8) {
		Xf = ((edge % 4) / 2 == 1) ? 1.0 : -1.0;
		Yf = (edge % 2 == 1) ? 1.0 : -1.0;
		Z1 = -1;
		Zn = 1;
	} else {
		Xf = ((edge % 4) / 2 == 1) ? 1.0 : -1.0;
		Zf = (edge % 2 == 1) ? 1.0 : -1.0;
		Y1 = -1;
		Yn = 1;
	}
#ifdef DEBUG
	(void) printf("edge=%d  Xf=%d Yf=%d Zf=%d X1=%d Xn=%d Y1=%d Yn=%d Z1=%d Zn=%d\n",
		edge, Xf, Yf, Zf, X1, Xn, Y1, Yn, Z1, Zn);
#endif
	drawCenter();
	glPushMatrix();
	glRotatef(rotateStep, (float) Xf, (float) Yf, (float) Zf);
	/*drawEdge(w, x, y, z);*/
	if (edge < 4) {
		drawWholeOuterCubit(w, X1, Yf, Zf);
		drawWholeOuterCubit(w, Xn, Yf, Zf);
		drawWholeInnerCubit(w, 0, 0, Zf, X1, Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Zf, Xn, Yf, 0);
		drawWholeInnerCubit(w, 0, Yf, 0, X1, 0, Zf);
		drawWholeInnerCubit(w, 0, Yf, 0, Xn, 0, Zf);
	} else if (edge >= 8) {
		drawWholeOuterCubit(w, Xf, Yf, Z1);
		drawWholeOuterCubit(w, Xf, Yf, Zn);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, Yf, Z1);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, Yf, Zn);
		drawWholeInnerCubit(w, 0, Yf, 0, Xf, 0, Z1);
		drawWholeInnerCubit(w, 0, Yf, 0, Xf, 0, Zn);
	} else {
		drawWholeOuterCubit(w, Xf, Y1, Zf);
		drawWholeOuterCubit(w, Xf, Yn, Zf);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, Y1, Zf);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, Yn, Zf);
		drawWholeInnerCubit(w, 0, 0, Zf, Xf, Y1, 0);
		drawWholeInnerCubit(w, 0, 0, Zf, Xf, Yn, 0);
	}
	if (!w->bevelGL.movement.control)
		glPopMatrix();
	if (edge < 4) {
		drawWholeOuterCubit(w, X1, -Yf, -Zf);
		drawWholeOuterCubit(w, Xn, -Yf, -Zf);
		drawWholeOuterCubit(w, X1, Yf, -Zf);
		drawWholeOuterCubit(w, Xn, Yf, -Zf);
		drawWholeOuterCubit(w, X1, -Yf, Zf);
		drawWholeOuterCubit(w, Xn, -Yf, Zf);
		drawWholeInnerCubit(w, 0, 0, -Zf, X1, -Yf, 0);
		drawWholeInnerCubit(w, 0, 0, -Zf, Xn, -Yf, 0);
		drawWholeInnerCubit(w, 0, -Yf, 0, X1, 0, -Zf);
		drawWholeInnerCubit(w, 0, -Yf, 0, Xn, 0, -Zf);
		drawWholeInnerCubit(w, 0, 0, -Zf, X1, Yf, 0);
		drawWholeInnerCubit(w, 0, 0, -Zf, Xn, Yf, 0);
		drawWholeInnerCubit(w, 0, Yf, 0, X1, 0, -Zf);
		drawWholeInnerCubit(w, 0, Yf, 0, Xn, 0, -Zf);
		drawWholeInnerCubit(w, 0, 0, Zf, X1, -Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Zf, Xn, -Yf, 0);
		drawWholeInnerCubit(w, 0, -Yf, 0, X1, 0, Zf);
		drawWholeInnerCubit(w, 0, -Yf, 0, Xn, 0, Zf);
		drawWholeInnerCubit(w, X1, 0, 0, 0, Yf, Zf);
		drawWholeInnerCubit(w, X1, 0, 0, 0, -Yf, Zf);
		drawWholeInnerCubit(w, X1, 0, 0, 0, Yf, -Zf);
		drawWholeInnerCubit(w, X1, 0, 0, 0, -Yf, -Zf);
		drawWholeInnerCubit(w, Xn, 0, 0, 0, Yf, Zf);
		drawWholeInnerCubit(w, Xn, 0, 0, 0, -Yf, Zf);
		drawWholeInnerCubit(w, Xn, 0, 0, 0, Yf, -Zf);
		drawWholeInnerCubit(w, Xn, 0, 0, 0, -Yf, -Zf);
	} else if (edge >= 8) {
		drawWholeOuterCubit(w, -Xf, -Yf, Z1);
		drawWholeOuterCubit(w, -Xf, -Yf, Zn);
		drawWholeOuterCubit(w, Xf, -Yf, Z1);
		drawWholeOuterCubit(w, Xf, -Yf, Zn);
		drawWholeOuterCubit(w, -Xf, Yf, Z1);
		drawWholeOuterCubit(w, -Xf, Yf, Zn);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, -Yf, Z1);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, -Yf, Zn);
		drawWholeInnerCubit(w, 0, -Yf, 0, -Xf, 0, Z1);
		drawWholeInnerCubit(w, 0, -Yf, 0, -Xf, 0, Zn);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, -Yf, Z1);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, -Yf, Zn);
		drawWholeInnerCubit(w, 0, -Yf, 0, Xf, 0, Z1);
		drawWholeInnerCubit(w, 0, -Yf, 0, Xf, 0, Zn);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, Yf, Z1);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, Yf, Zn);
		drawWholeInnerCubit(w, 0, Yf, 0, -Xf, 0, Z1);
		drawWholeInnerCubit(w, 0, Yf, 0, -Xf, 0, Zn);
		drawWholeInnerCubit(w, 0, 0, Z1, Xf, Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Z1, -Xf, Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Z1, Xf, -Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Z1, -Xf, -Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Zn, Xf, Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Zn, Xf, -Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Zn, -Xf, Yf, 0);
		drawWholeInnerCubit(w, 0, 0, Zn, -Xf, -Yf, 0);
	} else {
		drawWholeOuterCubit(w, -Xf, Y1, -Zf);
		drawWholeOuterCubit(w, -Xf, Yn, -Zf);
		drawWholeOuterCubit(w, Xf, Y1, -Zf);
		drawWholeOuterCubit(w, Xf, Yn, -Zf);
		drawWholeOuterCubit(w, -Xf, Y1, Zf);
		drawWholeOuterCubit(w, -Xf, Yn, Zf);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, Y1, -Zf);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, Yn, -Zf);
		drawWholeInnerCubit(w, 0, 0, -Zf, -Xf, Y1, 0);
		drawWholeInnerCubit(w, 0, 0, -Zf, -Xf, Yn, 0);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, Y1, -Zf);
		drawWholeInnerCubit(w, Xf, 0, 0, 0, Yn, -Zf);
		drawWholeInnerCubit(w, 0, 0, -Zf, Xf, Y1, 0);
		drawWholeInnerCubit(w, 0, 0, -Zf, Xf, Yn, 0);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, Y1, Zf);
		drawWholeInnerCubit(w, -Xf, 0, 0, 0, Yn, Zf);
		drawWholeInnerCubit(w, 0, 0, Zf, -Xf, Y1, 0);
		drawWholeInnerCubit(w, 0, 0, Zf, -Xf, Yn, 0);
		drawWholeInnerCubit(w, 0, Y1, 0, Xf, 0, Zf);
		drawWholeInnerCubit(w, 0, Y1, 0, -Xf, 0, Zf);
		drawWholeInnerCubit(w, 0, Y1, 0, Xf, 0, -Zf);
		drawWholeInnerCubit(w, 0, Y1, 0, -Xf, 0, -Zf);
		drawWholeInnerCubit(w, 0, Yn, 0, Xf, 0, Zf);
		drawWholeInnerCubit(w, 0, Yn, 0, -Xf, 0, Zf);
		drawWholeInnerCubit(w, 0, Yn, 0, Xf, 0, -Zf);
		drawWholeInnerCubit(w, 0, Yn, 0, -Xf, 0, -Zf);
	}
	if (w->bevelGL.movement.control)
		glPopMatrix();
}

/*-
 * This rotates whole cube.
 */
static void
faceRotate(BevelGLWidget w, int face, GLfloat rotateStep)
{
#define X1 (((corner & 4) != 0) ? 1 : -1)
#define Y1 (((corner & 2) != 0) ? 1 : -1)
#define Z1 (((corner & 1) != 0) ? 1 : -1)
#define Xn (((corner & 4) != 0) ? -1 : 1)
#define Yn (((corner & 2) != 0) ? -1 : 1)
#define Zn (((corner & 1) != 0) ? -1 : 1)
#define XP ((face == 4) ? 1 : ((face == -4) ? -1 : 0))
#define YP ((face == 2) ? 1 : ((face == -2) ? -1 : 0))
#define ZP ((face == 1) ? 1 : ((face == -1) ? -1 : 0))
	int corner = 0;

	drawCenter();
	glPushMatrix();
	glRotatef(rotateStep, XP, YP, ZP);
	drawWholeInnerCubit(w, X1, 0, 0, 0, Y1, Z1);
	drawWholeInnerCubit(w, X1, 0, 0, 0, Yn, Z1);
	drawWholeInnerCubit(w, X1, 0, 0, 0, Y1, Zn);
	drawWholeInnerCubit(w, X1, 0, 0, 0, Yn, Zn);
	drawWholeInnerCubit(w, 0, Y1, 0, X1, 0, Z1);
	drawWholeInnerCubit(w, 0, Y1, 0, Xn, 0, Z1);
	drawWholeInnerCubit(w, 0, Y1, 0, X1, 0, Zn);
	drawWholeInnerCubit(w, 0, Y1, 0, Xn, 0, Zn);
	drawWholeInnerCubit(w, 0, 0, Z1, X1, Y1, 0);
	drawWholeInnerCubit(w, 0, 0, Z1, Xn, Y1, 0);
	drawWholeInnerCubit(w, 0, 0, Z1, X1, Yn, 0);
	drawWholeInnerCubit(w, 0, 0, Z1, Xn, Yn, 0);

	drawWholeOuterCubit(w, X1, Y1, Z1);
	drawWholeOuterCubit(w, Xn, Y1, Z1);
	drawWholeOuterCubit(w, X1, Yn, Z1);
	drawWholeOuterCubit(w, X1, Y1, Zn);

	drawWholeInnerCubit(w, Xn, 0, 0, 0, Y1, Z1);
	drawWholeInnerCubit(w, Xn, 0, 0, 0, Yn, Z1);
	drawWholeInnerCubit(w, Xn, 0, 0, 0, Y1, Zn);
	drawWholeInnerCubit(w, Xn, 0, 0, 0, Yn, Zn);
	drawWholeInnerCubit(w, 0, Yn, 0, X1, 0, Z1);
	drawWholeInnerCubit(w, 0, Yn, 0, Xn, 0, Z1);
	drawWholeInnerCubit(w, 0, Yn, 0, X1, 0, Zn);
	drawWholeInnerCubit(w, 0, Yn, 0, Xn, 0, Zn);
	drawWholeInnerCubit(w, 0, 0, Zn, X1, Y1, 0);
	drawWholeInnerCubit(w, 0, 0, Zn, Xn, Y1, 0);
	drawWholeInnerCubit(w, 0, 0, Zn, X1, Yn, 0);
	drawWholeInnerCubit(w, 0, 0, Zn, Xn, Yn, 0);

	drawWholeOuterCubit(w, Xn, Yn, Zn);
	drawWholeOuterCubit(w, X1, Yn, Zn);
	drawWholeOuterCubit(w, Xn, Y1, Zn);
	drawWholeOuterCubit(w, Xn, Yn, Z1);
	glPopMatrix();
}

static void
drawCube(BevelGLWidget w)
{
	BevelSlice slice;
	GLfloat rotateStep = 0.0;

	slice.edge = NO_EDGE;
	slice.rotation = CW;
	if (w->bevelGL.movement.face == NO_FACE) {
		slice.rotation = NO_ROTATION;
		edgeRotate(w, slice.edge, rotateStep);
	} else if (!w->bevelGL.movement.control) {
		convertMove(w->bevelGL.movement, &slice);
		if (slice.edge != NO_EDGE) {
			rotateStep = (float) ((slice.rotation == CCW) ?
				w->bevelGL.rotateStep : -w->bevelGL.rotateStep);
		}
		edgeRotate(w, slice.edge, rotateStep);
	} else {
		int face = nextFace[w->bevelGL.movement.face][w->bevelGL.movement.direction % MAX_ORIENT];

		rotateStep = (float) w->bevelGL.rotateStep;
		faceRotate(w, face, rotateStep);
	}
}

void
movePiecesGL(BevelGLWidget w, int face, int position, int direction,
		Boolean control, int fast)
{
	int i, f;

	w->bevelGL.movement.face = face;
	w->bevelGL.movement.position = position;
	w->bevelGL.movement.direction = direction;
	w->bevelGL.movement.control = control;
	w->bevelGL.rotateStep = 0.0;
	w->bevelGL.angleStep = 0.0;
	f = (w->bevelGL.numTurnInc < fast) ? 1 : fast;
	for (i = 0; i < w->bevelGL.numTurnInc / f; i++) {
		double increment;
		if (control) {
			increment = 90.0 * f / w->bevelGL.numTurnInc;
		} else {
			increment = 180.0 * f / w->bevelGL.numTurnInc;
		}
		w->bevelGL.rotateStep += increment;
		/* rubik has 6 possibles... need 8 here */
		/*if (rotateDir[face][direction] == -1)*/
		drawAllPiecesGL(w);
		Sleep((unsigned int) ((w->bevel.delay /
			(w->bevelGL.numTurnInc * f)) * 40.0));
		w->bevel.currentFace = -1;
		w->bevel.currentPosition = -1;
	}
	w->bevelGL.angleStep = 0.0;
	w->bevelGL.movement.face = NO_FACE;
	w->bevelGL.movement.position = NO_POSITION;
	w->bevelGL.movement.direction = NO_DEPTH;
	w->bevelGL.movement.control = False;
}

void
drawAllPiecesGL(BevelGLWidget w)
{
#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("Draw All GL error");
	}
#endif
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPushMatrix();
	glTranslatef(0.0, 0.0, -10.0);
	if (w->core.height > w->core.width)
		glScalef(SCALE, (SCALE * (float) w->core.width / (float) w->core.height), SCALE);
	else
		glScalef((SCALE * (float) w->core.height / (float) w->core.width),  SCALE, SCALE);
	switch (w->bevel.view) {
	case 0: /* >-  vertical */
		glRotatef((GLfloat) w->bevelGL.step / (GLfloat) 2.0, 0, 1, 0);
		glRotatef((GLfloat) w->bevelGL.step / (GLfloat) 2.0, 1, 0, 0);
		break;
	case 1: /* Y  !vertical */
		glRotatef((GLfloat) w->bevelGL.step / (GLfloat) 2.0, 1, 0, 0);
		glRotatef((GLfloat) w->bevelGL.step / (GLfloat) 2.0, 0, 1, 0);
		break;
	case 2: /* -<  vertical */
		glRotatef((GLfloat) (-w->bevelGL.step) / (GLfloat) 2.0, 0, 1, 0);
		glRotatef((GLfloat) (-3 * w->bevelGL.step) / (GLfloat) 2.0, 1, 0, 0);
		break;
	case 3: /* ^  !vertical */
		glRotatef((GLfloat) (-w->bevelGL.step) / (GLfloat) 2.0, 1, 0, 0);
		glRotatef((GLfloat) (-3 * w->bevelGL.step) / (GLfloat) 2.0, 0, 1, 0);
		break;
	default:
		break;
	}
	drawCube(w);
	glPopMatrix();
	glFlush();
#ifdef WINVER
	SwapBuffers(w->core.hDC);
#else
	glXSwapBuffers(XtDisplay(w), XtWindow(w));
#endif
}

/* UNUSED */
/*void
drawFrameGL(BevelGLWidget w, Boolean focus)
{
}*/

static int
normalizePosition(float x, float y)
{
	if (x >= 0.0 && y < 0.0)
		return 0;
	else if (x >= 0.0 && y >= 0.0)
		return 1;
	else if (x < 0.0 && y >= 0.0)
		return 2;
	else /*if (x < 0.0 && y < 0.0)*/
		return 3;
}

Boolean
selectPiecesGL(BevelGLWidget w, int x, int y, int *face, int *position)
{
	/* Using gluUnProject By: Luke Benstead */
	/* http://nehe.gamedev.net/data/articles/article.asp?article=13 */

	GLint viewport[4];
	GLdouble modelview[16];
	GLdouble projection[16];
	GLfloat winX, winY, winZ;
	GLdouble posX, posY, posZ;
	Matrix a, b, r;
	Vector t, world, nw;

	glGetDoublev(GL_MODELVIEW_MATRIX, modelview);
	glGetDoublev(GL_PROJECTION_MATRIX, projection);
	glGetIntegerv(GL_VIEWPORT, viewport);

	winX = (float) x;
	winY = (float) viewport[3] - (float) y;
	glReadPixels(x, (int) winY, 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT,
		&winZ);
	gluUnProject(winX, winY, winZ, modelview, projection, viewport,
		&(posX), &(posY), &(posZ));
	t[0] = (float) posX;
	t[1] = (float) posY;
	t[2] = (float) (posZ + 10);
	if (w->core.height > w->core.width)
		t[1] *= (float) w->core.height / (float) w->core.width;
	else
		t[0] *= (float) w->core.width / (float) w->core.height;
	*face = -1;
	*position = -1;
	if (winZ == 1) {
		return False;
	}
	world[0] = t[0], world[1] = t[1], world[2] = t[2];
	switch (w->bevel.view) {
	case 0: /* >-  vertical */
		setMatrix((float) (-w->bevelGL.step) / (float) 2.0, 1.0, 0.0, 0.0, a);
		setMatrix((float) (-w->bevelGL.step) / (float) 2.0, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / (float) 2.0;
		nw[1] = world[1] / (float) 2.0;
		nw[2] = world[2] / (float) 2.0;
		if (nw[1] >= nw[0] && nw[1] >= nw[2]) {
			*face = 0;
			*position = normalizePosition(world[0], world[2]);
		} else if (nw[0] >= nw[1] && nw[0] >= nw[2]) {
			*face = 1;
			*position = normalizePosition(world[2], -world[1]);
		} else if (nw[2] >= nw[0] && nw[2] >= nw[1]) {
			*face = 2;
			*position = normalizePosition(world[0], -world[1]);
		}
		break;
	case 1: /* Y  !vertical */
		setMatrix((float) (-w->bevelGL.step) / (float) 2.0, 0.0, 1.0, 0.0, a);
		setMatrix((float) (-w->bevelGL.step) / (float) 2.0, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / (float) 2.0;
		nw[1] = world[1] / (float) 2.0;
		nw[2] = world[2] / (float) 2.0;
		if (nw[1] >= nw[0] && nw[1] >= nw[2]) {
			*face = 0;
			*position = normalizePosition(world[0], world[2]);
		} else if (nw[0] >= nw[1] && nw[0] >= nw[2]) {
			*face = 1;
			*position = normalizePosition(world[2], -world[1]);
		} else if (nw[2] >= nw[0] && nw[2] >= nw[1]) {
			*face = 2;
			*position = normalizePosition(world[0], -world[1]);
		}
		break;
	case 2: /* -<  vertical */
		setMatrix((float) (3 * w->bevelGL.step) / (float) 2.0, 1.0, 0.0, 0.0, a);
		setMatrix((float) w->bevelGL.step / (float) 2.0, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / (float) 2.0;
		nw[1] = world[1] / (float) 2.0;
		nw[2] = world[2] / (float) 2.0;
		if (nw[0] <= nw[1] && nw[0] <= nw[2]) {
			*face = 3;
			*position = normalizePosition(-world[2], -world[1]);
		} else if (nw[1] <= nw[0] && nw[1] <= nw[2]) {
			*face = 4;
			*position = normalizePosition(world[0], -world[2]);
		} else if (nw[2] <= nw[0] && nw[2] <= nw[1]) {
			*face = 5;
			*position = normalizePosition(world[0], world[1]);
		}
		break;
	case 3: /* ^  !vertical */
		setMatrix((float) (3 * w->bevelGL.step) / (float) 2.0, 0.0, 1.0, 0.0, a);
		setMatrix((float) w->bevelGL.step / (float) 2.0, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / (float) 2.0;
		nw[1] = world[1] / (float) 2.0;
		nw[2] = world[2] / (float) 2.0;
		if (nw[0] <= nw[1] && nw[0] <= nw[2]) {
			*face = 3;
			*position = normalizePosition(-world[2], -world[1]);
		} else if (nw[1] <= nw[0] && nw[1] <= nw[2]) {
			*face = 4;
			*position = normalizePosition(world[0], -world[2]);
		} else if (nw[2] <= nw[0] && nw[2] <= nw[1]) {
			*face = 5;
			*position = normalizePosition(world[0], world[1]);
		}
		break;
	default:
		break;
	}
#ifdef DEBUG
	(void) printf("view %d, x %d, y %d\n", w->bevel.view, x, y);
	(void) printf(" wx %g, wy %g, wz %g\n", winX, winY, winZ);
	(void) printf("  px %g, py %g, pz %g\n", posX, posY, posZ);
	(void) printf("   tx %g, ty %g, tz %g\n", t[0], t[1], t[2]);
	(void) printf("    rx %g, ry %g, rz %g\n",
		world[0], world[1], world[2]);
	(void) printf("     face %d, pos %d\n", *face, *position);
#endif
	return True;
}

Boolean
narrowSelectionGL(BevelGLWidget w, int *face, int *direction)
{
	switch (*direction) {
	case TR:
	case BR:
	case BL:
	case TL:
		if (w->bevel.view % 2 == 0) {
			/* if (*face == 1) do nothing */
			if (*face == 3)
				*direction = (*direction + 3) % MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (*direction + 1) % MAX_ORIENT;
			else if (*face == 4)
				*direction = (*direction + 2) % MAX_ORIENT;
			else if (*face == 5)
				*direction = (*direction + 2) % MAX_ORIENT + 2 * MAX_ORIENT;
		}
		break;
	case CCW:
	case CW:
		break;
	case TOP:
	case RIGHT:
	case BOTTOM:
	case LEFT:
		if (w->bevel.view % 2 == 0) {
			if (*face == 1)
				*direction = (TL + *direction) % MAX_ORIENT + 2 * MAX_ORIENT;
			else if (*face == 3)
				*direction = (BL + *direction) % MAX_ORIENT + 2 * MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (TR + *direction) % MAX_ORIENT + 2 * MAX_ORIENT;
			else if (*face == 4)
				*direction = (BR + *direction) % MAX_ORIENT + 2 * MAX_ORIENT;
			else if (*face == 5)
				*direction = (BL + *direction) % MAX_ORIENT;
		}
		break;
	default:
		return False;
	}
#if 0
	if (*position != MAX_ORIENT) {
		if (*direction == CW)
			*direction = (*position + 3) % MAX_ORIENT;
		else if (*direction == CCW)
			*direction = (*position + 1) % MAX_ORIENT;
		else if (*direction < MAX_ORIENT &&
				!((*direction + *position) % 2))
			return False;
	}
#endif
	return True;
}

#ifdef WINVER
static Boolean
setupPixelFormat(BevelGLWidget w, BYTE type, DWORD flags)
{
	PIXELFORMATDESCRIPTOR pfd;
	int pixelFormat;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | flags;
	pfd.dwLayerMask = PFD_MAIN_PLANE;
	/*pfd.cColorBits = 8;
	pfd.cDepthBits = 16;*/
	pfd.iPixelType = type;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;

	if ((pixelFormat = ChoosePixelFormat(w->core.hDC, &pfd)) == 0) {
		DWORD err;
		char *buf1;

		err = GetLastError();
		/* 6 ERROR_INVALID_HANDLE */
		intCat(&buf1, "ChoosePixelFormat failed ", (int) err);
		DISPLAY_WARNING(buf1);
		free(buf1);
		return FALSE;
	}

	if (SetPixelFormat(w->core.hDC, pixelFormat, &pfd) == FALSE) {
		MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
		return FALSE;
	}
	DescribePixelFormat(w->core.hDC, pixelFormat,
		sizeof(PIXELFORMATDESCRIPTOR), &pfd);
	return TRUE;
}

#else

static Boolean
setValuesPuzzleGL(Widget current, Widget request, Widget renew)
{
	BevelGLWidget c = (BevelGLWidget) current, w = (BevelGLWidget) renew;
	Boolean redraw = False;

	if (w->bevel.view != c->bevel.view) {
		resizePuzzleGL(w);
		redraw = True;
	}
	if (w->bevel.delay != c->bevel.delay) {
		w->bevelGL.numTurnInc = ((w->bevel.delay < MAX_TURN_INC) ?
			w->bevel.delay + 1 : MAX_TURN_INC);
	}
	return (redraw);
}

static GLXContext *
initGL(BevelGLWidget w) {
	XVisualInfo xviIn, *xviOut;
	int numVis;

	/*XGetWindowAttributes(XtDisplay(w), XtWindow(w), &xgwa); */
	xviIn.screen = DefaultScreen(XtDisplay(w));
	xviIn.visualid = XVisualIDFromVisual(DefaultVisual(XtDisplay(w),
		xviIn.screen));
	xviOut = XGetVisualInfo(XtDisplay(w), VisualScreenMask | VisualIDMask,
		&xviIn, &numVis);
	if (!xviOut) {
		XtWarning("Could not get XVisualInfo");
		return (GLXContext *) NULL;
	}
	if (glXContext)
		free(glXContext);
	if ((glXContext = (GLXContext *) malloc(sizeof (GLXContext))) ==
			NULL) {
		DISPLAY_ERROR("Not enough memory for glx info, exiting.");
	}
	/* assertion "glx_dpy" failed on Cygwin */
	if (!glXQueryExtension(XtDisplay(w), NULL, NULL)) {
		XtWarning("Could not find GLX");
		return (GLXContext *) NULL;
	}
	*glXContext = glXCreateContext(XtDisplay(w), xviOut, 0, GL_TRUE);
	(void) XFree((char *) xviOut);
	if (!*glXContext) {
		XtWarning("Could not create rendering context");
		return (GLXContext *) NULL;
	}
	return glXContext;
}
#endif

#ifndef WINVER
static
#endif
void
resizePuzzleGL(BevelGLWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	glViewport(0, 0, (GLint) w->core.width, (GLint) w->core.height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-1.0, 1.0, -1.0, 1.0, 5.0, 15.0);
	glMatrixMode(GL_MODELVIEW);
}

static void
makeCurrentGL(BevelGLWidget w)
{
	GLboolean rgbaMode;

#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("GL error");
	}
#endif
	madeCurrent = True;
	/* True Color junk */
	glGetBooleanv(GL_RGBA_MODE, &rgbaMode);
	if (!rgbaMode) {
#ifdef WINVER
		glClearIndex(0.0);
#else
		glIndexi(WhitePixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
		glClearIndex((float) BlackPixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
#endif
	}
	resizePuzzleGL(w);

	glDrawBuffer(GL_BACK);
	glClearDepth(1.0);
	glClearColor(face_material[NO_FACE][0], face_material[NO_FACE][1],
		face_material[NO_FACE][2], face_material[NO_FACE][3]);
	glColor3f(1.0, 1.0, 1.0);

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, position0);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, position1);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lmodel_twoside);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_NORMALIZE);
	glEnable(GL_CULL_FACE);
	glShadeModel(GL_FLAT);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, front_shininess);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, front_specular);

	w->bevelGL.step = 90;
	w->bevelGL.rotateStep = 0.0;
}

#ifndef WINVER
static void
initializePuzzleGL(Widget request, Widget renew)
{
	BevelGLWidget w = (BevelGLWidget) renew;
	int i;

	setAllColors((BevelWidget) w);
	if (initGL(w) == (GLXContext *) NULL) {
		w->bevel.dim = 0;
		return;
	}
	w->bevel.dim = 4; /* 2 and 3 already taken */
	for (i = 0; i < MAX_FACES + 2; i++)
		pickColor(w, i, w->bevel.mono, face_material[i]);
	w->bevelGL.numTurnInc = ((w->bevel.delay < MAX_TURN_INC) ?
		w->bevel.delay + 1 : MAX_TURN_INC);
	resizePuzzleGL(w);
}
#endif

#ifndef WINVER
static
#endif
void
exposePuzzleGL(
#ifdef WINVER
BevelGLWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifdef WINVER
	if (hRC == NULL) {
		BYTE type;
		int i;
		DWORD flags = PFD_DOUBLEBUFFER;

#if 1
		type = PFD_TYPE_RGBA;
#else
		type = PFD_TYPE_COLORINDEX;
#endif
		(void) setupPixelFormat(w, type, flags);
		hRC = wglCreateContext(w->core.hDC);
		w->bevelGL.movement.face = NO_FACE;
		w->bevelGL.movement.position = NO_POSITION;
		w->bevelGL.movement.direction = NO_DEPTH;
		w->bevelGL.movement.control = False;
		w->bevelGL.rotateStep = 0.0;
		w->bevelGL.angleStep = 0.0;
		w->bevel.dim = 4; /* 2 and 3 already taken */
		for (i = 0; i < MAX_FACES + 2; i++)
			pickColor(w, i, w->bevel.mono, face_material[i]);
		w->bevelGL.numTurnInc = ((w->bevel.delay < MAX_TURN_INC) ?
			w->bevel.delay + 1 : MAX_TURN_INC);
		resizePuzzleGL(w);
	}
#else
	BevelGLWidget w = (BevelGLWidget) renew;

	if (!w->core.visible)
		return;
	if (w->bevel.dim != 4)
		return;
#endif
	if (!madeCurrent) {
		makeCurrentGL(w);
	}
	/*drawFrameGL(w, w->bevel.focus);*/
	drawAllPiecesGL(w);
}

#ifndef WINVER
static void
movePuzzleGLTl(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		TL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLTop(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		TOP,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLTr(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		TR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLLeft(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLRight(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLBl(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		BL,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLBottom(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
movePuzzleGLBr(BevelGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((BevelWidget) w, event->xbutton.x, event->xbutton.y,
		BR,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & (Mod1Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}
#endif
#else
int x = 0; /* non empty source */
#endif
