/*---------------------------------------------------------------------------*\

    FILE....: DSPFIFO.H
    TYPE....: C Header File
    AUTHOR..: David Rowe
    DATE....: 20/11/97
    AUTHOR..: Ron Lee
    DATE....: 5/3/07

    Functions used to read and write to First In First Out (DSPFIFO) queues in
    the DSP, thus facilitating PC to DSP data transfer.  The word length of
    the DSPFIFOs is 16 bits.


         Voicetronix Voice Processing Board (VPB) Software
         Copyright (C) 1999-2007 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
         MA  02110-1301  USA

\*---------------------------------------------------------------------------*/

#ifndef __VTDSPFIFO_H__
#define __VTDSPFIFO_H__

#include "hip.h"
#include "fifo.h"


#pragma pack(1)		// force member alignment same as DSP

class V4PCIDspFifo : public Fifo
{ //{{{
private:

	// offsets of each param from base of DSP FIFO structure in DSP memory 
	enum Offsets
	{ //{{{
	    PSTART    = 0,
	    PEND      = 1,
	    PWR       = 2,
	    PRD       = 3,
	    SIZE      = 4,

	    FIFO_SIZE = 5	// size of structure to upload
	}; //}}}

	Hip	       *m_hip;		// hip object used to talk to DSP
	unsigned short  m_board;	// VPB board number
	uint16_t	m_adspfifo;	// address of structure in DSP memory

	// NOTE: these 5 members _must_ not be reordered and must be packed.
	uint16_t	m_start;	// first word in FIFO
	uint16_t	m_end;		// one after last word in FIFO
	uint16_t	m_wr;		// write pointer
	uint16_t	m_rd; 		// read pointer
	uint16_t	m_size;		// total storage in FIFO

	pthread_mutex_t m_mutex;


	//! Performs a few sanity checks on DSP FIFO structure.
	void CheckDspFifo();


public:

	//! Open a link to a DSP FIFO.  Assumes the DSP has been booted.
	V4PCIDspFifo(Hip *hip, unsigned short board, unsigned short fifo_addr);

	~V4PCIDspFifo();


	//! Write a block of words from the PC to a DSP FIFO.
	//
	//! @return @c Fifo::OK	if successful, or Fifo::FULL if @a size words
	//!         could not be written to it atomically.
	Status Write(uint16_t *buf, size_t len);

	//! Read a block of words from a DSP FIFO to the PC.
	//
	//! @return @c Fifo::OK	if successful, or Fifo::EMPTY if @a size words
	//!         could not be read from it atomically.
	Status Read(uint16_t *buf, size_t len);

	//! Return the number of words used in the DSP FIFO.
	size_t HowFull();

	//! Returns the number of words free in the DSP FIFO.
	size_t HowEmpty();

	void Flush();

}; //}}}

#pragma pack()	// normal member alignment


#endif	// #ifndef __VTDSPFIFO_H__

