/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       PTHICommand.cpp
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include "HECILinux.h"
#include "HECI_if.h"
#include "PTHICommand.h"
#include "StatusCodeDefinitions.h"


PTHICommand::PTHICommand( bool verbose): PTHIClient(HECI_PTHI,verbose)
{
}

PTHICommand::~PTHICommand(void)
{
}

/*
* Confirms the correctness of the response message header
* and the response message size
* Arguments:
*	command	- appropriate Host interface command
*	response_header	- pointer to the response message header
*	response_size	- value that holds the actual size of the
*                         response message
*	expected_size	- value that holds the expected size of the
*                         response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
UINT32 PTHICommand::_verifyResponseHeader(const UINT32 command,
                            const PTHI_MESSAGE_HEADER *response_header,
                            UINT32 response_size)
{
    PT_STATUS status = PT_STATUS_SUCCESS;


    if ( response_size != (response_header->Length + sizeof(PTHI_MESSAGE_HEADER)))
    {

        status = PT_STATUS_INTERNAL_ERROR;
    }
    else if(	response_header->Command.cmd.val != command )
    {

        status = PT_STATUS_INTERNAL_ERROR;
    }
    else if(response_header->Reserved != 0 )
    {

        status = PT_STATUS_INTERNAL_ERROR;
    }
    else if(response_header->Version.MajorNumber != MAJOR_VERSION || response_header->Version.MinorNumber < MINOR_VERSION)
    {

        status = PT_STATUS_INTERNAL_ERROR;
    }

    return status;
}

/*
* Confirms the correctness of the GetCodeVersions response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
PT_STATUS PTHICommand::_verifyCodeVersions(const CFG_GET_CODE_VERSIONS_RESPONSE *response)
{
	PT_STATUS status = PT_STATUS_SUCCESS;
	UINT32 codeVerLen;
	UINT32 ptVerTypeCount;
    UINT32 len = 0;
    UINT32 i;

	do
	{
		codeVerLen = response->Header.Length - sizeof(PT_STATUS);
		ptVerTypeCount = codeVerLen - sizeof(response->CodeVersions.BiosVersion)- sizeof(response->CodeVersions.VersionsCount);
		if ( response->CodeVersions.VersionsCount != (ptVerTypeCount/sizeof( PT_VERSION_TYPE )))
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		for ( i = 0; i < (response->CodeVersions.VersionsCount); i ++ )
		{
			len = response->CodeVersions.Versions[i].Description.Length;

			if(len > UNICODE_STRING_LEN)
			{
				status = PT_STATUS_INTERNAL_ERROR;
				break;
			}

			len = response->CodeVersions.Versions[i].Version.Length;
			if ( response->CodeVersions.Versions[i].Version.String[len] != '\0' ||
				(len != strlen((PCHAR)(response->CodeVersions.Versions[i].Version.String))) )
			{
				status = PT_STATUS_INTERNAL_ERROR;
				break;
			}
		}
	}while ( 0 );


	return status;
}
/*
 * GetVersions response message PTHI command
 * Arguments:
 *	response - pointer to the CODE_VERSIONS struct
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PT_STATUS_INTERNAL_ERROR - on failure
 */
UINT32 PTHICommand::GetCodeVersions(CODE_VERSIONS *CodeVersions, HECI_VERSION *hecversion)
{
//	printf("\nIn GetCodeVersions( ) function...");
	if ( CodeVersions == NULL )
	{
		printf("\nNull pointer passed. Exit 1");
		printf("\nstatus returned is PTSDK_STATUS_INVALID_PARAM");
		return PTSDK_STATUS_INVALID_PARAM;
	}
	PT_STATUS status;
	UCHAR command[sizeof(GET_CODE_VERSION_HEADER)];
    memcpy(command ,&GET_CODE_VERSION_HEADER,sizeof(GET_CODE_VERSION_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_CODE_VERSION_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
    CFG_GET_CODE_VERSIONS_RESPONSE *tmp_response;
    PTHI_MESSAGE_HEADER *resp_header;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
    	printf("\nSend message to Heci failed. Exit 2");
	printf("\nstatus returned is PT_STATUS_INTERNAL_ERROR");
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			printf("\nread from heci failed, outbuffsize = 0. exit 3");
			break;
		}

		tmp_response = (CFG_GET_CODE_VERSIONS_RESPONSE*)readBuffer;

		status = tmp_response->Status;
		if( status != PT_STATUS_SUCCESS )
		{
			printf("\nstatus bit in response from heci is not success. exit 4");
			break;
		}

		resp_header = &(tmp_response->Header);

		status = _verifyResponseHeader(CODE_VERSIONS_RESPONSE,
					resp_header,outBuffSize);
		if ( status != PT_STATUS_SUCCESS )
		{
			printf("\nverification of response header failed. exit 5");
			break;
		}

		status = _verifyCodeVersions(tmp_response);
		if ( status != PT_STATUS_SUCCESS )
		{
			printf("\nverification of code versions failed. exit 6");
			break;
		}

		memcpy( CodeVersions,&(tmp_response->CodeVersions),sizeof(CODE_VERSIONS)) ;
	}while(0);

        if ( readBuffer != NULL )
	{
		free(readBuffer);
		readBuffer = NULL;
	}

	*hecversion = PTHIClient.version;
//	printf("\nstatus returned is %d",status);
	return status;
}
/*
 * DisplayCodeVersions print code version
 * Arguments:
 *	response - pointer to the CODE_VERSIONS struct
 * Return values:
 *	NONE
 */
VOID PTHICommand::DisplayCodeVersions(const CODE_VERSIONS *versions)
{
    UINT32 i,j;
    UINT16 len;

    // BIOS version is free format null terminated string,
    // max length of BIOS version string is 64 byte
	fprintf(stdout,"\nBIOS Version:");
	fprintf(stdout,"\t\t %s\n\n",versions->BiosVersion);
	fprintf(stdout,"\nAMT Code Version:\n");
    for ( i = 0; i < versions->VersionsCount; i ++ )
    {
        fprintf(stdout,"\t");
        len = versions->Versions[i].Description.Length;
        for ( j = 0; j < len; j ++ )
        {
             fprintf(stdout,"%c",versions->Versions[i].Description.String[j]);
        }
        fprintf(stdout,":");
        for ( j = 0 ; j < (UINT32)(24 - len); j ++ )
        {
            fprintf(stdout," ");
        }

        fprintf(stdout,"%s\n", versions->Versions[i].Version.String);
    }
     fprintf(stdout,"\n");
}
/*
 * Calls to GetProvisioningMode Host interface command
 * Arguments:
 *	legacy - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetProvisioningMode(AMT_BOOLEAN *legacy)
{
	if ( legacy == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
    UCHAR command[sizeof(GET_PROVISIONING_MODE_HEADER)];
    memcpy(command ,&GET_PROVISIONING_MODE_HEADER,sizeof(GET_PROVISIONING_MODE_HEADER));
	UINT8 *readBuffer;
	UINT32 inBuffSize =0;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;
	ULONG command_size = sizeof(GET_PROVISIONING_MODE_HEADER);
	AMT_STATUS status;


	bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);

	do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}
		CFG_GET_PROVISIONING_MODE_RESPONSE *tmp_response =
		(CFG_GET_PROVISIONING_MODE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(PROVISIONING_MODE_RESPONSE,resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyProvisioningMode(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}


        *legacy = tmp_response->LegacyMode;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
		readBuffer = NULL;
	}
	return status;
}

/*
* Confirms the correctness of the GetProvisioningMode response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyProvisioningMode(const CFG_GET_PROVISIONING_MODE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_PROVISIONING_MODE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}
/*
 * DisplayAMTMode print legacy mode
 * Arguments:
 *	*legacy - pointer to the AMT_BOOLEAN enum
 * Return values:
 *	NONE
 */
void PTHICommand::DisplayAMTMode(const AMT_BOOLEAN *legacy)
{
	fprintf(stdout,"\nAMT Mode:");
	fprintf(stdout,"\t\t\t");
    if (*legacy)
	{
	fprintf(stdout,"\nAMT Mode 1");
        fprintf(stdout,"\n");
	}
    else
	{
	fprintf(stdout,"\nAMT Mode 2");
        fprintf(stdout,"\n");
	}
 }
/*
 * Calls to GetProvisioningState Host interface command
 * Arguments:
 *	state - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetProvisioningState(AMT_PROVISIONING_STATE *state)
{
	if ( state == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PROVISIONING_STATE_HEADER)];
    memcpy(command ,&GET_PROVISIONING_STATE_HEADER,sizeof(GET_PROVISIONING_STATE_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PROVISIONING_STATE_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PROVISIONING_STATE_RESPONSE *tmp_response =
		(CFG_GET_PROVISIONING_STATE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(PROVISIONING_STATE_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyProvisioningState(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

        *state = tmp_response->ProvisioningState;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GetProvisioningState response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifyProvisioningState(const CFG_GET_PROVISIONING_STATE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_PROVISIONING_STATE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}
/*
 * DisplayProvisioningState print provisioning state
 * Arguments:
 *	*state - pointer to the AMT_PROVISIONING_STATE enum
 * Return values:
 *	NONE
 */
VOID PTHICommand::DisplayProvisioningState(const AMT_PROVISIONING_STATE *state)
{
	fprintf(stdout,"Provisioning State:");
	fprintf(stdout,"\t");
    switch (*state)
	{
    case PROVISIONING_STATE_PRE:
	fprintf(stdout,"Pre");
        break;
    case PROVISIONING_STATE_IN:
	fprintf(stdout,"In");
        break;
    case PROVISIONING_STATE_POST:
	fprintf(stdout,"Post");
        break;
    default:
	fprintf(stdout,"Unknown");
        break;
	}

 }

/*
* Calls to GetEndOfPostState Host interface command
* Arguments:
*	legacy - pointer to the pre-allocated structure
*       which will hold the result
* Return values:
*	PT_STATUS_SUCCESS - on success
*	appropriate error value defined in AMTVersn.h - on failure
*/

AMT_STATUS PTHICommand::GetEndOfPostState(UINT8  *BiosModeState)
{
	if ( BiosModeState == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	PCHAR command = (PCHAR)&GET_ENDOFPOST_STATE_HEADER;
    //memcpy(command ,&GET_ENDOFPOST_STATE_HEADER,sizeof(GET_ENDOFPOST_STATE_HEADER));
	UINT8 *readBuffer;
	ULONG command_size = sizeof(GET_ENDOFPOST_STATE_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage((UCHAR *)command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PROVISIONING_STATE_RESPONSE *tmp_response =
		(CFG_GET_PROVISIONING_STATE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(PROVISIONING_STATE_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyProvisioningState(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

        *BiosModeState = tmp_response->ProvisioningState;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}



/*
 * Calls to GenerateRngKey Host interface command
 * Arguments:
 *	None
 * Return values:
 *	PT_STATUS_SUCCESS - or AMT_STATUS_IN_PROGRESS on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GenerateRngKey()
{

	AMT_STATUS status;
	UCHAR command[sizeof(GENERATE_RNG_SEED_HEADER)];
    memcpy(command ,&GENERATE_RNG_SEED_HEADER,sizeof(GENERATE_RNG_SEED_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GENERATE_RNG_SEED_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GENERATE_RNG_SEED_RESPONSE *tmp_response =
		(CFG_GENERATE_RNG_SEED_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GENERATE_RNG_SEED_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyGenerateRngKey(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GenerateRngKey response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifyGenerateRngKey(const CFG_GENERATE_RNG_SEED_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GENERATE_RNG_SEED_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}

/*
 * Calls to GetRngSeedStatus Host interface command
 * Arguments:
 *	state - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetRngSeedStatus(AMT_RNG_STATUS *rngStatus)
{
	if ( rngStatus == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	UCHAR command[sizeof(GET_RNG_SEED_STATUS_HEADER)];
    memcpy(command ,&GET_RNG_SEED_STATUS_HEADER,sizeof(GET_RNG_SEED_STATUS_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_RNG_SEED_STATUS_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_RNG_SEED_STATUS_RESPONSE *tmp_response =
		(CFG_GET_RNG_SEED_STATUS_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GET_RNG_SEED_STATUS_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetRngSeedStatus(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		*rngStatus = tmp_response->RngStatus;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GetRngSeedStatus response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifyGetRngSeedStatus(const CFG_GET_RNG_SEED_STATUS_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_RNG_SEED_STATUS_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}

/*
 * DisplayRngSeedStatus print RNG seed status
 * Arguments:
 *	*rngStatus - pointer to the AMT_RNG_STATUS enum
 * Return values:
 *	NONE
 */
VOID PTHICommand::DisplayRngSeedStatus(const AMT_RNG_STATUS *rngStatus)
{
//	DisplayMessage(WORD_RNG_SEED_STATUS);
	fprintf(stdout,"\t");
    switch (*rngStatus)
	{
    case RNG_STATUS_EXIST:
    //    DisplayMessage(WORD_EXIST);
        break;
    case RNG_STATUS_IN_PROGRESS:
	//	DisplayMessage(WORD_IN_PROGRESS);
        break;
    case RNG_STATUS_NOT_EXIST:
	  //  DisplayMessage(WORD_NOT_EXIST);
        break;
    default:
	//	DisplayMessage(UNKNOWN);
        break;
	}
    fprintf(stdout,"\n");
 }
/*
 * Calls to ZeroTouchEnabled Host interface command
 * Arguments:
 *	zeroTouchEnabled - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetZeroTouchEnabled(AMT_BOOLEAN *zeroTouchEnabled)
{
	if ( zeroTouchEnabled == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
    UCHAR command[sizeof(GET_ZERO_TOUCH_ENABLED_HEADER)];
    memcpy(command ,&GET_ZERO_TOUCH_ENABLED_HEADER,sizeof(GET_ZERO_TOUCH_ENABLED_HEADER));
	UINT8 *readBuffer;
	UINT32 inBuffSize =0;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;
	ULONG command_size = sizeof(GET_ZERO_TOUCH_ENABLED_HEADER);
	AMT_STATUS status;


	bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);

	do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}
		CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE *tmp_response =
		(CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GET_ZERO_TOUCH_ENABLED_RESPONSE,resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetZeroTouchEnabled(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}


		*zeroTouchEnabled = tmp_response->ZeroTouchEnabled;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
		readBuffer = NULL;
	}
	return status;
}

/*
* Confirms the correctness of the GetZeroTouchEnabled response message
* Arguments:
*	response - pointer to the response message
* Return values:
*	PT_STATUS_SUCCESS - on success
*	PTSDK_STATUS_INTERNAL_ERROR - on failure
*/
AMT_STATUS PTHICommand::_verifyGetZeroTouchEnabled(const CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}

/*
 * DisplayZTCEnabled print if ZTC enabled
 * Arguments:
 *	*ztcEnabled - pointer to the AMT_BOOLEAN enum
 * Return values:
 *	NONE
 */
VOID PTHICommand::DisplayZTCEnabled(const AMT_BOOLEAN *ztcEnabled)
{
   // DisplayMessage(WORD_ZTC);
	fprintf(stdout,"\t\t\t");
    if (*ztcEnabled)
	{
	//	DisplayMessage(WORD_ENABLED);
        fprintf(stdout,"\n");
	}
    else
	{
	//	DisplayMessage(WORD_DISABLED);
        fprintf(stdout,"\n");
	}
 }

/*
 * Calls to GetProvisioningTlsMode Host interface command
 * Arguments:
 *	state - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetProvisioningTlsMode(AMT_PROVISIONING_TLS_MODE *provisioningTlsMode)
{
	if ( provisioningTlsMode == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PROVISIONING_TLS_MODE_HEADER)];
    memcpy(command ,&GET_PROVISIONING_TLS_MODE_HEADER,sizeof(GET_PROVISIONING_TLS_MODE_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PROVISIONING_TLS_MODE_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PROVISIONING_TLS_MODE_RESPONSE *tmp_response =
		(CFG_GET_PROVISIONING_TLS_MODE_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GET_PROVISIONING_TLS_MODE_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetProvisioningTlsMode(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		*provisioningTlsMode = tmp_response->ProvisioningTlsMode;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GetProvisioningTlsMode response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifyGetProvisioningTlsMode(const CFG_GET_PROVISIONING_TLS_MODE_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_PROVISIONING_TLS_MODE_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}
/*
 * DisplayProvisioningTlsMode print provisioning TLS mode
 * Arguments:
 *	*provisioningTlsMode - pointer to the AMT_PROVISIONING_TLS_MODE enum
 * Return values:
 *	NONE
 */
VOID PTHICommand::DisplayProvisioningTlsMode(const AMT_PROVISIONING_TLS_MODE *provisioningTlsMode)
{
//	DisplayMessage(WORD_PROVISIONING_TLS_MODE);
	fprintf(stdout,"\t");
    switch (*provisioningTlsMode)
	{
    case PKI:
	//	DisplayMessage(WORD_PKI);
        break;
    case PSK:
	//	DisplayMessage(WORD_PSK);
        break;
	case NOT_READY:
	//	DisplayMessage(WORD_NOT_READY);
        break;
    default:
	//	DisplayMessage(UNKNOWN);
        break;
	}
    fprintf(stdout,"\n");
 }

/*
 * Calls to StartConfiguration Host interface command
 * Arguments:
 *	None
 * Return values:
 *	PT_STATUS_SUCCESS - or AMT_STATUS_CERTIFICATE_NOT_READY on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::StartConfiguration()
{

	AMT_STATUS status;
	UCHAR command[sizeof(START_CONFIGURATION_HEADER)];
    memcpy(command ,&START_CONFIGURATION_HEADER,sizeof(START_CONFIGURATION_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(START_CONFIGURATION_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_START_CONFIGURATION_RESPONSE *tmp_response =
		(CFG_START_CONFIGURATION_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )//||status!= AMT_STATUS_CERTIFICATE_NOT_READY)
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(START_CONFIGURATION_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyStartConfiguration(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the StartConfiguration response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifyStartConfiguration(const CFG_START_CONFIGURATION_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_START_CONFIGURATION_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}

/*
 * Calls to SetProvisioningServerOTP Host interface command
 * Arguments:
 *	passwordOTP AMT_ANSI_STRING structure of OTP password
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::SetProvisioningServerOTP(AMT_ANSI_STRING passwordOTP)
{
	if ( NULL == passwordOTP.Buffer )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength = sizeof(passwordOTP.Length)  + (passwordOTP.Length*sizeof(CHAR));
	PTHI_MESSAGE_HEADER SET_PROVISIONING_SERVER_OTP_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{SET_PROVISIONING_SERVER_OTP_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(START_CONFIGURATION_HEADER) + msgLength;
	UCHAR *command;
	command = (UCHAR*) malloc(command_size);
		if ( NULL == command )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memcpy(command ,&SET_PROVISIONING_SERVER_OTP_HEADER,sizeof(SET_PROVISIONING_SERVER_OTP_HEADER));
	memcpy(command + sizeof(SET_PROVISIONING_SERVER_OTP_HEADER),&(passwordOTP.Length),sizeof(passwordOTP.Length));
	memcpy(
		command + sizeof(SET_PROVISIONING_SERVER_OTP_HEADER) + sizeof(passwordOTP.Length),
		passwordOTP.Buffer, passwordOTP.Length);

	UINT8 *readBuffer;
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE *tmp_response =
		(CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(SET_PROVISIONING_SERVER_OTP_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifySetProvisioningServerOTP(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the SetProvisioningServerOTP response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifySetProvisioningServerOTP(const CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}
/*
 * Calls to SetDnsSuffix Host interface command
 * Arguments:
 *	passwordOTP AMT_ANSI_STRING structure of DNS suffix
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::SetDnsSuffix(AMT_ANSI_STRING dnsSuffix)
{
	if ( NULL == dnsSuffix.Buffer )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength = sizeof(dnsSuffix.Length)  + (dnsSuffix.Length*sizeof(CHAR));
	PTHI_MESSAGE_HEADER SET_DNS_SUFFIX_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{SET_DNS_SUFFIX_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(SET_DNS_SUFFIX_HEADER) + msgLength;
	UCHAR *command;
	command = (UCHAR*) malloc(command_size);
		if ( NULL == command )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memcpy(command ,&SET_DNS_SUFFIX_HEADER,sizeof(SET_DNS_SUFFIX_HEADER));
	memcpy(command + sizeof(SET_DNS_SUFFIX_HEADER),&(dnsSuffix.Length), sizeof(dnsSuffix.Length));
	memcpy(command + sizeof(SET_DNS_SUFFIX_HEADER) + sizeof(dnsSuffix.Length),dnsSuffix.Buffer, dnsSuffix.Length);

	UINT8 *readBuffer;
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_SET_DNS_SUFFIX_RESPONSE *tmp_response =
		(CFG_SET_DNS_SUFFIX_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(SET_DNS_SUFFIX_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifySetDnsSuffix(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;

	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the SetDnsSuffix response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifySetDnsSuffix(const CFG_SET_DNS_SUFFIX_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_SET_DNS_SUFFIX_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
		//return PT_STATUS_SUCCESS;
	}
	return PT_STATUS_SUCCESS;
}

/*
 * Calls to EnumerateHashHandles Host interface command
 * Arguments:
 *	hashHandles - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::EnumerateHashHandles(AMT_HASH_HANDLES *hashHandles)
{
    if (NULL == hashHandles)
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	UCHAR command[sizeof(ENUMERATE_HASH_HANDLES_HEADER)];
    memcpy(command ,&ENUMERATE_HASH_HANDLES_HEADER,sizeof(ENUMERATE_HASH_HANDLES_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(ENUMERATE_HASH_HANDLES_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_HASH_HANDLES_RESPONSE *tmp_response =
		(CFG_GET_HASH_HANDLES_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(ENUMERATE_HASH_HANDLES_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyHashHandles(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
        hashHandles->Length = tmp_response->HashHandles.Length;
        memset( hashHandles->Handles,0, sizeof(UINT32) * CERT_HASH_MAX_NUMBER);

		if (CERT_HASH_MAX_NUMBER < hashHandles->Length) 
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

        memcpy(hashHandles->Handles,tmp_response->HashHandles.Handles,sizeof(UINT32) * hashHandles->Length);
	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}

/*
 * Confirms the correctness of the EnumerateHashHandles response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifyHashHandles(const CFG_GET_HASH_HANDLES_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		sizeof(AMT_STATUS) + sizeof(response->HashHandles.Length)  +(sizeof(UINT32) * response->HashHandles.Length))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}


/*
 * Calls to GetCertificateHashEntry Host interface command
 * Arguments:
 *	passwordOTP AMT_ANSI_STRING structure of DNS suffix
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetCertificateHashEntry(UINT32 hashHandle, CERTHASH_ENTRY *hashEntry)
{
	if ( NULL == hashEntry )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}

	AMT_STATUS status;
	UINT32 msgLength = sizeof(hashHandle);
	PTHI_MESSAGE_HEADER GET_CERTHASH_ENTRY_HEADER =
	{
		{AMT_MAJOR_VERSION,AMT_MINOR_VERSION},0,{GET_CERTHASH_ENTRY_REQUEST},msgLength
	};

	UINT32 command_size = sizeof(GET_CERTHASH_ENTRY_HEADER) + msgLength;
	UCHAR *command;
	command = (UCHAR*) malloc(command_size);
		if ( NULL == command )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memcpy(command ,&GET_CERTHASH_ENTRY_HEADER,sizeof(GET_CERTHASH_ENTRY_HEADER));
	memcpy(command + sizeof(GET_CERTHASH_ENTRY_HEADER),&(hashHandle), sizeof(hashHandle));
	UINT8 *readBuffer;
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
	if ( command != NULL )
	{
		free(command);
	}
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_CERTHASH_ENTRY_RESPONSE *tmp_response =
            (CFG_GET_CERTHASH_ENTRY_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GET_CERTHASH_ENTRY_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = _verifyGetCertificateHashEntry(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
		hashEntry->IsActive = tmp_response->Hash.IsActive;
        hashEntry->IsDefault = tmp_response->Hash.IsDefault;
		hashEntry->Name.Length = tmp_response->Hash.Name.Length;
		hashEntry->HashAlgorithm = tmp_response->Hash.HashAlgorithm;
        memcpy(hashEntry->CertificateHash,tmp_response->Hash.CertificateHash,sizeof(tmp_response->Hash.CertificateHash));
        hashEntry->Name.Buffer =(CHAR*)malloc(hashEntry->Name.Length*sizeof(CHAR));
				if ( NULL == hashEntry->Name.Buffer )
				{
						status = PT_STATUS_INTERNAL_ERROR;
						break;
				}
        memcpy(hashEntry->Name.Buffer,&(tmp_response->Hash.Name.Buffer) ,hashEntry->Name.Length*sizeof(CHAR));


	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GetCertificateHashEntry response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand::_verifyGetCertificateHashEntry(const CFG_GET_CERTHASH_ENTRY_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount !=
		(sizeof(CFG_GET_CERTHASH_ENTRY_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)
			- sizeof(CHAR*)+ response->Hash.Name.Length))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}

/*
 * DisplayHashEntry print Certificate Hash Entry
 * Arguments:
 *	hashEntry -   the CERTHASH_ENTRY struct
 * Return values:
 *	NONE
 */
VOID PTHICommand::DisplayHashEntry(CERTHASH_ENTRY hashEntry)
{
	char *printString;
	//DisplayMessage(HASH_ENTRY);
	printString =(char*) malloc((hashEntry.Name.Length+1)*sizeof(CHAR));
	if ( NULL == printString )
	{
    fprintf(stdout, "Display Hash Entry internal error!\n");
		return;
	}
	memcpy(printString,hashEntry.Name.Buffer,hashEntry.Name.Length*sizeof(CHAR));
	printString[hashEntry.Name.Length*sizeof(CHAR)] ='\0';
	fprintf(stdout, "%s", printString);
	fprintf(stdout,"\n");
	free(printString);
    fprintf(stdout, "Default = ");
    fprintf(stdout, "%s", hashEntry.IsDefault ? "true" : "false");
    fprintf(stdout, "Active = ");
    fprintf(stdout, "%s", hashEntry.IsActive ? "true" : "false");
    fprintf(stdout, "Hash Algorithm = ");
    int len;
    switch(hashEntry.HashAlgorithm)
    {
    case CERT_HASH_ALGORITHM_MD5:
        fprintf(stdout, "MD5");
        len = 16;
        break;
    case CERT_HASH_ALGORITHM_SHA1:
        fprintf(stdout, "SHA1");
        len = 20;
        break;
    default:
        fprintf(stdout, "UNKNOWN");
        len = 0;
        break;
    }

    for(int i = 0; i < len; i++)
    {
        fprintf(stdout, " %02X", hashEntry.CertificateHash[i]);
    }
 }
/*
 * Calls to GetDnsSuffix Host interface command
 * Arguments:
 *	dnsSuffix - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetDnsSuffix(AMT_ANSI_STRING *dnsSuffix)
{
	if ( dnsSuffix == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PKI_FQDN_SUFFIX_HEADER)];
    memcpy(command ,&GET_PKI_FQDN_SUFFIX_HEADER,sizeof(GET_PKI_FQDN_SUFFIX_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PKI_FQDN_SUFFIX_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PKI_FQDN_SUFFIX_RESPONSE *tmp_response =
		(CFG_GET_PKI_FQDN_SUFFIX_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GET_PKI_FQDN_SUFFIX_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetDnsSuffix(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
        dnsSuffix->Length = tmp_response->Suffix.Length;
        dnsSuffix->Buffer =(CHAR*)malloc(dnsSuffix->Length * sizeof(CHAR));
				if ( NULL == dnsSuffix->Buffer )
				{
						status = PT_STATUS_INTERNAL_ERROR;
						break;
				}
        memcpy(dnsSuffix->Buffer,&(tmp_response->Suffix.Buffer),dnsSuffix->Length * sizeof(CHAR));
	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GetDnsSuffix response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand :: _verifyGetDnsSuffix(const CFG_GET_PKI_FQDN_SUFFIX_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount  !=
		sizeof(AMT_STATUS) + sizeof(response->Suffix.Length)  + response->Suffix.Length*sizeof(CHAR))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}
/*
 * Calls to GetAMTSetupAuditRecord Host interface command
 * Arguments:
 *	auditRecord - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetAMTSetupAuditRecord(AMT_PROV_AUDIT_RECORD *auditRecord)
{
	if ( auditRecord == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	UCHAR command[sizeof(GET_AUDIT_RECORD_HEADER)];
    memcpy(command ,&GET_AUDIT_RECORD_HEADER,sizeof(GET_AUDIT_RECORD_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_AUDIT_RECORD_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_AUDIT_RECORD_RESPONSE *tmp_response =
		(CFG_GET_AUDIT_RECORD_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GET_AUDIT_RECORD_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetAMTSetupAuditRecord(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
        memcpy(auditRecord,&tmp_response->AuditRecord,sizeof(AMT_PROV_AUDIT_RECORD));
	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GetAMTSetupAuditRecord response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand :: _verifyGetAMTSetupAuditRecord(const CFG_GET_AUDIT_RECORD_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount  !=
		(sizeof(CFG_GET_AUDIT_RECORD_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}



/*
 * Calls to GetPID Host interface command
 * Arguments:
 *	pid - pointer to the pre-allocated structure
 *       which will hold the result
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	appropriate error value defined in StatusCodeDefinitions.h - on failure
 */
AMT_STATUS PTHICommand::GetPID(UINT8 *pid)
{
	if ( pid == NULL )
	{
		return PTSDK_STATUS_INVALID_PARAM;
	}
	AMT_STATUS status;
	UCHAR command[sizeof(GET_PID_HEADER)];
    memcpy(command ,&GET_PID_HEADER,sizeof(GET_PID_HEADER));
	UINT8 *readBuffer;
	UINT32 command_size = sizeof(GET_PID_HEADER);
    UINT32 inBuffSize;
    UINT32 outBuffSize=0;
	int	   bytesWritten = 0;

    bytesWritten = PTHIClient.SendMessage(command,command_size);
    if(bytesWritten != command_size)
    {
        return PT_STATUS_INTERNAL_ERROR;
    }
    inBuffSize = PTHIClient.GetBufferSize();
    readBuffer = (UINT8*) malloc (sizeof(UINT8) * inBuffSize);
		if ( NULL == readBuffer )
		{
        return PT_STATUS_INTERNAL_ERROR;
		}
    memset(readBuffer,0,inBuffSize);

    outBuffSize = PTHIClient.ReceiveMessage(readBuffer,inBuffSize);
    do
	{
		if( 0 == outBuffSize)
		{
			status = PT_STATUS_INTERNAL_ERROR;
			break;
		}

		CFG_GET_PID_RESPONSE *tmp_response =
		(CFG_GET_PID_RESPONSE*)readBuffer;

		status = tmp_response->Status;

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		PTHI_MESSAGE_HEADER *resp_header =
		&(tmp_response->Header);

		status = _verifyResponseHeader(GET_PID_RESPONSE,
			resp_header,outBuffSize);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}

		status = _verifyGetPID(tmp_response);

		if ( status != PT_STATUS_SUCCESS )
		{
			break;
		}
		status = tmp_response->Status;
        memcpy(pid,tmp_response->PID,sizeof(tmp_response->PID));
	}while ( 0 );

	if ( readBuffer != NULL )
	{
		free(readBuffer);
	}
	return status;
}
/*
 * Confirms the correctness of the GetPID response message
 * Arguments:
 *	response - pointer to the response message
 * Return values:
 *	PT_STATUS_SUCCESS - on success
 *	PTSDK_STATUS_INTERNAL_ERROR - on failure
 */
AMT_STATUS PTHICommand :: _verifyGetPID(const CFG_GET_PID_RESPONSE *response)
{
	ULONG ByteCount;

	ByteCount = response->Header.Length;

	if (ByteCount  !=
		(sizeof(CFG_GET_PID_RESPONSE) - sizeof(PTHI_MESSAGE_HEADER)))
	{
		return PTSDK_STATUS_INTERNAL_ERROR;
	}
	return PT_STATUS_SUCCESS;
}
