<?php
/**
 * @file
 * Tweeper - a Twitter to RSS web scraper.
 *
 * Copyright (C) 2013-2016  Antonio Ospite <ao2@ao2.it>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once 'autoload.php';

use Tweeper\Tweeper;

date_default_timezone_set('UTC');

/**
 * Check if the script is being run from the command line.
 */
function is_cli() {
  return (php_sapi_name() === "cli");
}

/**
 * Show the script usage.
 */
function usage($argv) {
  if (is_cli()) {
    $usage = "{$argv[0]} [-e|-h|--help] <src_url>\n";
  }
  else {
    $usage = htmlentities("{$_SERVER['SCRIPT_NAME']}?src_url=<src_url>&generate_enclosure=<0|1>");
  }

  return "usage: $usage";
}

/**
 * Parse command line options.
 */
function parse_options_cli($argv, $argc) {
  $options = array(
    'generate_enclosure' => FALSE,
  );

  if ($argc < 2) {
    return $options;
  }

  $cli_options = getopt("eh", array("help"));
  foreach ($cli_options as $opt => $val) {
    switch ($opt) {
      case 'e':
        $options['generate_enclosure'] = TRUE;
        break;

      case 'h':
      case 'help':
        echo usage($argv);
        exit(0);

      default:
        fwrite(STDERR, usage($argv));
        exit(1);
    }
  }

  $options['src_url'] = $argv[count($cli_options) + 1];

  return $options;
}

/**
 * Parse options passed from a query string.
 */
function parse_options_query_string() {
  $options = array(
    'generate_enclosure' => FALSE,
  );

  if (isset($_GET['src_url'])) {
    $options['src_url'] = $_GET['src_url'];
  }

  if (isset($_GET['generate_enclosure'])) {
    $options['generate_enclosure'] = $_GET['generate_enclosure'] == 1;
  }

  return $options;
}

if (is_cli()) {
  $options = parse_options_cli($argv, $argc);
  $error_stream = fopen('php://stderr', 'w');
}
else {
  $options = parse_options_query_string();
  $error_stream = fopen('php://output', 'w');
}

if (!isset($options['src_url'])) {
  fwrite($error_stream, usage(is_cli() ? $argv : NULL));
  exit(1);
}

$tweeper = new Tweeper($options['generate_enclosure']);
$output = $tweeper->tweep($options['src_url']);
if (is_null($output)) {
  exit(1);
}
echo $output;
