use ui;
use graphics;
use core:geometry;
use lang:bs:macro;


/**
 * A data structure instantiated somewhere. Managed by a Workspace.
 */
class Data extends Drawable {
	// Draw this data item at its current location.
	void draw(Graphics g, Bool active) : abstract {
		Rect r = rect;
		g.fill(r, dataBg);
		g.draw(r, dataFg);
	}

	void drawLinks(Graphics g, Bool active) : override {}

	// Update the contents of the visualization from the actual data.
	void update(Workspace work, unsafe:RawPtr object, Nat offset) : abstract;

	// Find a particular offset inside this data. Attempts to stop at a data element that represents
	// "type" if possible, in order to better visualize pointers to different kinds of elements.
	DataOffset findOffset(core:lang:Type? type, Nat offset) { DataOffset(this); }

	// Refine "contained" with a more convenient return type.
	Data contained() : override { this; }

	// Update the read/write state of this data element based on what threads have observed.
	// Called for the root of a data object, thus we assume that 'offset' is zero.
	void updateMemAccess(unsafe:RawPtr object, Trackers trackers) {
		Nat size = object.readSize * object.readCount;

		if (trackers.accessed(object)) {
			checkAccess(object, 0, size, trackers);
		} else {
			clearAccess();
		}
	}

	// Called to clear any access markers for this object.
	void clearAccess() : abstract;

	// Called when this object has been accessed by at least one thread. Recurses through the object
	// hierarchy to find individual elements.
	void checkAccess(unsafe:RawPtr object, Nat offset, Nat size, Trackers trackers) : abstract;
}

/**
 * Data that tracks reads and writes.
 *
 * TODO: There might be better ways of visualizing this.
 */
class RWData extends Data {
	// Read from?
	private Bool read;

	// Written to?
	private Bool write;

	// Concurrency error?
	private Bool error;

	// Custom drawing of the background.
	void draw(Graphics g, Bool active) : override {
		Rect r = rect;

		if (read & write) {
			Size s = r.size;
			s.w /= 2;
			g.fill(Rect(r.p0, s), dataReadBg);
			g.fill(Rect(r.p0 + Point(s.w, 0), s), dataWriteBg);
		} else if (read) {
			g.fill(r, dataReadBg);
		} else if (write) {
			g.fill(r, dataWriteBg);
		} else {
			g.fill(r, dataBg);
		}
		g.draw(r, dataFg);

		if (error) {
			g.line(r.p0, r.p1, dataError);
			g.line(Point(r.p0.x, r.p1.y), Point(r.p1.x, r.p0.y), dataError);
		}
	}

	// Clear access.
	void clearAccess() : override {
		read = false;
		write = false;
		error = false;
	}

	// Update access.
	void checkAccess(unsafe:RawPtr object, Nat offset, Nat size, Trackers trackers) : override {
		var r = trackers.readBy(object, offset, size);
		var w = trackers.writtenBy(object, offset, size);

		read = r.any;
		write = w.any;
		error = false;

		// Writes from multiple threads is not OK.
		if (w.count > 1) {
			trackers.multiWriteError(w);
			error = true;
		} else if (w.any) {
			for (t in r) {
				if (t !is w[0]) {
					trackers.readWriteError(w[0], r);
					error = true;
				}
			}
		}
	}
}

/**
 * Representation of an offset inside some data visualization.
 */
value DataOffset {
	// Inside this element.
	Data data;

	// Y-offset in pixels.
	Float offset;

	// Create.
	init(Data data) {
		init { data = data; offset = 0; }
	}
	init(Data data, Float offset) {
		init { data = data; offset = offset; }
	}
}


/**
 * Data element with a caption.
 */
class DataWithCaption extends Data {
	init(Data wrap, Str caption) {
		Text c(caption, captionFont);

		init() {
			wrap = wrap;
			caption = c;
			captionSz = c.size + compositeBorder*2;
		}
	}

	Size size() : override {
		Size sz = wrap.size();
		sz.w = max(sz.w, captionSz.w);
		sz.h += captionSz.h;
		sz;
	}

	void draw(Graphics g, Bool active) : override {
		Size cSz = captionSz;
		Size wSz = wrap.size;
		cSz.w = max(cSz.w, wSz.w);
		g.fill(Rect(pos, cSz), if (active) { captionActiveBg; } else { captionBg; });
		g.draw(caption, captionFg, pos + compositeBorder);
		g.draw(Rect(pos, cSz), captionFg);

		wrap.size(Size(cSz.w, wSz.h));

		wrap.pos = pos + Size(0, captionSz.h);
		wrap.draw(g, active);
	}

	void drawLinks(Graphics g, Bool active) : override {
		wrap.drawLinks(g, active);
	}

	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {
		wrap.update(work, object, offset);
	}

	DataOffset findOffset(core:lang:Type? type, Nat offset) : override {
		wrap.findOffset(type, offset);
	}

	Bool remove() : override {
		wrap.remove();
	}

	// Return the sub-drawable if the click is there.
	Drawable findAt(Point pt) : override {
		if (wrap.rect.contains(pt))
			wrap.findAt(pt);
		else
			this;
	}

	// Called when the mouse is pressed inside this drawable.
	Bool mouseClicked(Point pt, Bool down) {
		if (down) {
			mouseOrigin = pt - pos;
		} else {
			mouseOrigin = null;
		}
		false;
	}

	// Called when the mouse is moved inside this drawable. "origin" is the location (in world space) where it was pressed.
	Bool mouseMoved(Point pt, Point origin) {
		if (mouseOrigin) {
			pos = pt - mouseOrigin;
			return true;
		}

		false;
	}

	void clearAccess() : override {
		wrap.clearAccess();
	}

	void checkAccess(unsafe:RawPtr object, Nat offset, Nat size, Trackers trackers) : override {
		wrap.checkAccess(object, offset, size, trackers);
	}


	// The drawable we're wrapping.
	Data wrap;

	Data contained() : override { wrap.contained; }

	// Caption text.
	private Text caption;

	// Size of the caption area.
	private Size captionSz;

	// Origin of mouse when pressed.
	private Point? mouseOrigin;
}

/**
 * Unknown data element.
 */
class UnknownData extends RWData {
	private Text text;

	init() {
		init() {
			text = Text("?", dataFont);
		}

		size = text.size + dataBorder*2;
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		g.draw(text, dataFg, pos + dataBorder);
	}

	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {}
}


/**
 * A primitive datatype, such as an integer. Represented as text.
 */
class Primitive extends RWData {
	// The text to draw.
	private Text text;

	// Type of the data we're displaying.
	core:lang:Type type;

	// Size of the data.
	Nat typeSize;

	init(core:lang:Type type, Str initial) {
		init() {
			text = Text(initial, dataFont);
			type = type;
			typeSize = core:lang:Value(type).size.current;
		}
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		g.draw(text, dataFg, pos + dataMargin);
	}

	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {
		if (s = convertPrimitive(type, object, offset)) {
			text = Text(s, dataFont);
			invalidateSize();
		}
	}

	Size computeSize() : override {
		text.size + dataMargin*2;
	}

	DataOffset findOffset(core:lang:Type? type, Nat offset) : override {
		Float o = offset.int.float / typeSize.int.float;
		DataOffset(this, o * size.h);
	}
}

// Arrow for all pointers.
package Path ptrArrow on Render = {
	Path p;
	p.start(Point(0, 0));
	p.line(Point(-6, -4));
	p.line(Point(-6, 4));
	p.close();
	p;
};

/**
 * A pointer type, referring to some other data.
 */
class Pointer extends RWData {
	// Object we're pointing to.
	protected Data? to;

	// Y-offset inside 'to', if present.
	protected Float offset;

	// Our path, re-used for efficiency.
	private Path line;

	init() {
		init() {}

		size = Text("????", dataFont).size + dataMargin*2;
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		if (to) {
		} else {
			// Null.
			Rect r = rect;
			g.line(rect.p0, rect.p1, activeLink);
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		if (to) {
			Point target = to.pos;
			target.y += offset;

			Brush color = if (active) { activeLink; } else { inactiveLink; };
			Rect r = rect;
			// g.line(r.center, to.pos, color);
			Float dy = abs(r.center.y - target.y);
			Float shift = max(0.2*dy, 40);
			line.clear();
			line.start(r.center);
			line.bezier(target - Point(shift, 0), target);
			g.draw(line, color);

			g.push();
			g.transform(translate(target));
			g.fill(ptrArrow, color);
			g.pop();
		}
	}

	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {
		var ptr = object.readPtr(offset);
		if (ptr.any) {
			var off = work.traverse(ptr).findOffset(null, 0);
			this.to = off.data;
			this.offset = off.offset;
		} else {
			to = null;
		}
	}

	DataOffset findOffset(core:lang:Type? type, Nat offset) : override {
		Float o = min(offset.int.float / core:asm:sPtr.current.int.float, 1.0);
		DataOffset(this, o * size.h);
	}
}

/**
 * A pointer on the stack. Examines the pointer passed directly to the object rather than
 * dereferencing it once first.
 */
class StackPointer extends Pointer {
	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {
		if (object.any) {
			var off = work.traverse(object).findOffset(null, 0);
			this.to = off.data;
			this.offset = off.offset;
		} else {
			to = null;
		}
	}
}

/**
 * A composite type, consisting of one or more variables of other types.
 */
class Composite extends Data {
	// Initialize to a particular type, so that we can visualize pointers better.
	init(core:lang:Type type) {
		init() { type = type; }
	}

	void add(Str name, Nat offset, Data data, Bool divider) {
		if (content.empty)
			divider = false;

		content << Var(name, data, offset, divider);
		invalidateSize();
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		Float width = rect.size.w;
		Float xBase = pos.x + dataBorder.w;

		for (v in content) {
			Float y = pos.y + v.yPos;

			if (v.line) {
				g.line(Point(pos.x, y), Point(pos.x + width, y), dataFg);
			}

			y += dataBorder.h;

			// Align the text a bit nicer on the y-axis, to line up the baselines...
			g.draw(v.text, dataFg, Point(xBase, y + dataBorder.h));
			v.value.pos = Point(xBase + namesWidth + dataWidth - v.value.size.w, y);
			v.value.draw(g, false);
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		for (v in content) {
			v.value.drawLinks(g, active);
		}
	}

	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {
		for (v in content) {
			v.value.update(work, object, offset + v.offset);
		}

		invalidateSize();
	}

	DataOffset findOffset(core:lang:Type? type, Nat offset) : override {
		if (content.empty)
			return DataOffset(this);

		Nat found = content.count - 1;
		for (Nat i = 1; i < content.count; i++) {
			if (content[i].offset > offset) {
				found = i - 1;
				break;
			}
		}

		var elem = content[found];

		// Some data member is missing.
		if (offset < elem.offset)
			return DataOffset(this);

		// Stop here?
		if (type) {
			if (core:lang:isA(this.type, type)) {
				Size sz = size(); // Make sure that our size is re-computed, otherwise 'yPos' is not correct.

				// Fast-path.
				if (elem.offset == offset)
					return DataOffset(this, elem.yPos);

				// Ask the element for its offset, and scale that to fit our view of the situation.
				Float start = elem.yPos;
				Float height = sz.h - start;
				if (found + 1 < content.count)
					height = content[found + 1].yPos - start;

				var offset = elem.value.findOffset(null, offset - elem.offset);
				return DataOffset(this, start + height * offset.offset / elem.value.size.h);
			}
		} else if (offset == 0) {
			// If no type, and at zero offset we stop early as well.
			return DataOffset(this);
		}

		return elem.value.findOffset(type, offset - elem.offset);
	}

	// Called to clear any access markers for this object.
	void clearAccess() {
		for (v in content) {
			v.value.clearAccess();
		}
	}

	// Called when this object has been accessed by at least one thread. Recurses through the object
	// hierarchy to find individual elements.
	void checkAccess(unsafe:RawPtr object, Nat offset, Nat size, Trackers trackers) {
		for (i, v in content) {
			Nat end = size;
			if (i + 1 < content.count)
				end = content[i + 1].offset;

			v.value.checkAccess(object, offset + v.offset, end - v.offset, trackers);
		}
	}

private:
	class Var on Render {
		Str name;
		Text text;
		Data value;
		Nat offset;
		// Y-position of this variable. Updated whenever 'computeSize' is called.
		Float yPos;
		// Line before?
		Bool line;

		init(Str name, Data value, Nat offset, Bool divider) {
			init() {
				name = name;
				text = Text(name, variableFont);
				value = value;
				offset = offset;
				line = divider;
			}
		}
	}

	core:lang:Type type;
	Var[] content;
	Float namesWidth;
	Float dataWidth;
	Float height;

	Size computeSize() : override {
		// TODO: It would probably be nice to recurse into
		// the values that are themselves variables and render them
		// more compactly.
		namesWidth = 0;
		dataWidth = 0;
		height = 0;
		for (v in content) {
			var nameSz = v.text.size;
			var dataSz = v.value.size;

			if (v.line)
				height += dataBorder.h;

			v.yPos = height;

			namesWidth = max(namesWidth, nameSz.w + dataBorder.w);
			dataWidth = max(dataWidth, dataSz.w);
			height += max(nameSz.h, dataSz.h) + dataBorder.h;
		}

		height -= dataBorder.h;
		Size(namesWidth + dataWidth, height) + compositeBorder*2;
	}
}

/**
 * A class describing an array.
 */
class ArrayData extends Data {
	init(core:lang:Type type) {
		init() {
			typeInfo = TypeInfo(type, core:lang:Value(type).isValue, false);
		}
	}

	void draw(Graphics g, Bool active) : override {
		super:draw(g, active);
		Float width = rect.size.w;
		Point at = pos + compositeBorder;

		for (v in content) {
			var nameSz = v.text.size;
			var dataSz = v.value.size;

			// Align the text a bit nicer on the y-axis, to line up the baselines...
			g.draw(v.text, dataFg, at + Point(0, dataBorder.h));

			v.value.pos = at + Point(namesWidth + dataWidth - dataSz.w, 0);
			v.value.draw(g, false);

			at.y += max(nameSz.h, dataSz.h) + dataBorder.h;
		}
	}

	void drawLinks(Graphics g, Bool active) : override {
		for (v in content) {
			v.value.drawLinks(g, active);
		}
	}

	void update(Workspace work, unsafe:RawPtr object, Nat offset) : override {
		// TODO: Extract the actual offset!
		Nat dataOffset = object.readSize - core:asm:sPtr.current;
		unsafe:RawPtr data = object.readPtr(offset + dataOffset);
		if (data.empty) {
			// TODO: Write "empty" or something like that.
			return;
		}

		Nat stride = data.readSize;
		Nat count = data.readFilled;

		// Update the elements that are already present.
		for (Nat i = 0; i < content.count; i++) {
			content[i].value.update(work, data, i*stride);
		}

		// Create new elements.
		for (Nat i = content.count; i < count; i++) {
			Var v(i, work.createData(typeInfo, data, i*stride));
			content << v;
			v.value.update(work, data, i*stride);
		}

		// Remove any extra.
		while (content.count > count) {
			content.pop();
		}

		invalidateSize();
	}

	// Called to clear any access markers for this object.
	void clearAccess() {
		for (v in content) {
			v.value.clearAccess();
		}
	}

	// Called when this object has been accessed by at least one thread. Recurses through the object
	// hierarchy to find individual elements.
	void checkAccess(unsafe:RawPtr object, Nat offset, Nat size, Trackers trackers) {
		Nat stride = object.readSize;

		for (i, v in content) {
			v.value.checkAccess(object, offset + i*stride, stride, trackers);
		}
	}

private:
	TypeInfo typeInfo;

	class Var on Render {
		Text text;
		Data value;

		init(Nat id, Data value) {
			init() {
				text = Text(id.toS + ":", variableFont);
				value = value;
			}
		}
	}

	Var[] content;
	Float namesWidth;
	Float dataWidth;
	Float height;

	Size computeSize() : override {
		namesWidth = 0;
		dataWidth = 0;
		height = 0;
		for (v in content) {
			var nameSz = v.text.size;
			var dataSz = v.value.size;

			namesWidth = max(namesWidth, nameSz.w + dataBorder.w);
			dataWidth = max(dataWidth, dataSz.w);
			height += max(nameSz.h, dataSz.h) + dataBorder.h;
		}

		height -= dataBorder.h;
		Size(namesWidth + dataWidth, height) + compositeBorder*2;
	}
}
