/*
 * Extract node coordinates
 */
#include "basil_mysql.h"

int main(int ac, char **av)
{
	/*
	 * Coordinates can be NULL if the blade has been disabled.
	 * In this case, very likely also processor_status != 'up'
	 */
	const char query[] =	"SELECT processor_id, "
				"	cab_position, cab_row, "
				"	cage, slot, cpu, "
				"	x_coord, y_coord, z_coord "
				"FROM  processor "
				"WHERE x_coord IS NOT NULL "
				"AND   y_coord IS NOT NULL "
				"AND   z_coord IS NOT NULL";
	enum query_columns {
			COL_NODE_ID,
			COL_CAB,
			COL_ROW,
			COL_CAGE,
			COL_SLOT,
			COL_CHIP,
			COL_X,
			COL_Y,
			COL_Z,
			COLUMN_COUNT
	};
	MYSQL		*handle;
	MYSQL_BIND	bind[COLUMN_COUNT];
	MYSQL_STMT	*stmt;

	unsigned int	node_id;
	int		col_data[COLUMN_COUNT];
	my_bool		is_null[COLUMN_COUNT];
	my_bool		error[COLUMN_COUNT];

	my_ulonglong	nrows;
	int		rc = -1, i;

	memset(bind, 0, sizeof(bind));
	for (i = 0; i < COLUMN_COUNT; i ++) {
		/*
		 * @buffer:	   address of variable to write into
		 * @buffer_type:   determines which variable type to use
		 * 		   - TINYINT  => MYSQL_TYPE_TINY
		 * 		   - SMALLINT => MYSQL_TYPE_SHORT
		 * 		   - INT      => MYSQL_TYPE_LONG
		 * 		   - BIGINT   => MYSQL_TYPE_LONGLONG
		 * 		   - FLOAT    => MYSQL_TYPE_FLOAT
		 * 		   - DOUBLE   => MYSQL_TYPE_DOUBLE
		 * 		   - VARCHAR
		 * 		     CHAR     => MYSQL_TYPE_STRING
		 * 		     TEXT
		 * @is_unsigned:   whether variable is unsigned
		 * @buffer_length: actual size of the @buffer in bytes
		 * @length:	   actual data length
		 * @is_null:	   indicates that result is NULL
		 * @error:	   to report potential truncation
		 */
		bind[i].buffer_type = MYSQL_TYPE_LONG;
		bind[i].buffer	    = (char *)&col_data[i];
		bind[i].is_null	    = &is_null[i];
		bind[i].error	    = &error[i];
	}

	/* The first column (node_id / processor_id) is the exception */
	bind[COL_NODE_ID].buffer      = (char *)&node_id;
	bind[COL_NODE_ID].is_unsigned = true;

	/* initialize 'MYSQL' connection handler */
	handle = cray_connect_sdb();

	/* launch query as a prepared statement */
	stmt = exec_query(handle, query, bind, COLUMN_COUNT);
	if (stmt == NULL)
		goto err;

	/* Fetch all rows (1) */
	nrows = mysql_stmt_affected_rows(stmt);
	if (nrows == (my_ulonglong)-1) {
		/*
		 * NB: This is _not_ an error if last argument to
		 *     run_query_stmt() is 'false', since then the
		 *     number of rows is only known afterwards.
		 */
		warnx("query '%s' returned an error: %s", query,
			mysql_stmt_error(stmt));
		goto err;
	} else if (nrows == 0) {
		warnx("no results retrieved");
	} else {
		warnx("found %llu rows\n", nrows);
	}

	/* Iterate through the entire result set */
	while (mysql_stmt_fetch(stmt) == 0) {
		printf("%6u c%d-%dc%ds%dn%d  %3d %3d %3d\n", node_id,
			col_data[COL_CAB], col_data[COL_ROW],
			col_data[COL_CAGE], col_data[COL_SLOT], col_data[COL_CHIP],
			col_data[COL_X], col_data[COL_Y], col_data[COL_Z]);
	}

	/* Deallocate handle. Errors can be 'server gone away' or 'unknown' */
	if (mysql_stmt_close(stmt)) {
		warnx("error closing statement: %s",
			mysql_stmt_error(stmt));
		goto err;
	}

	rc = 0;
err:
	mysql_close(handle);

	return rc ? EXIT_FAILURE : EXIT_SUCCESS;
}
