/*
 *  Copyright (C) 2004-2013 Savoir-Faire Linux Inc.
 *  Author: Alexandre Savard <alexandre.savard@savoirfairelinux.com>
 *  Author: Adrien Beraud <adrien.beraud@wisdomvibes.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA.
 *
 *  Additional permission under GNU GPL version 3 section 7:
 *
 *  If you modify this program, or any covered work, by linking or
 *  combining it with the OpenSSL project's OpenSSL library (or a
 *  modified version of that library), containing parts covered by the
 *  terms of the OpenSSL or SSLeay licenses, Savoir-Faire Linux Inc.
 *  grants you additional permission to convey the resulting work.
 *  Corresponding Source for a non-source form of such a combination
 *  shall include the source code for the parts of OpenSSL used as well
 *  as that of the covered work.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "audio_rtp_record_handler.h"
#include <fstream>
#include <algorithm>
#include "logger.h"
#include "sip/sipcall.h"
#include "audio/audiolayer.h"
#include "manager.h"

namespace sfl {

DTMFEvent::DTMFEvent(char digit) : payload(), newevent(true), length(1000)
{
    /*
       From RFC2833:

       Event  encoding (decimal)
       _________________________
       0--9                0--9
       *                     10
       #                     11
       A--D              12--15
       Flash                 16
    */

    switch (digit) {
        case '*':
            digit = 10;
            break;

        case '#':
            digit = 11;
            break;

        case 'A' ... 'D':
            digit = digit - 'A' + 12;
            break;

        case '0' ... '9':
            digit = digit - '0';
            break;

        default:
            ERROR("Unexpected DTMF %c", digit);
    }

    payload.event = digit;
    payload.ebit = false; // end of event bit
    payload.rbit = false; // reserved bit
    payload.duration = 1; // duration for this event
    payload.vol = 10;
}

AudioRtpRecord::AudioRtpRecord() :
    callId_("")
    , codecSampleRate_(0)
    , dtmfQueue_()
    , audioCodecs_()
    , audioCodecMutex_()
    , encoderPayloadType_(0)
    , decoderPayloadType_(0)
    , hasDynamicPayloadType_(false)
    , decData_(DEC_BUFFER_SIZE, 1, 8000)
    , resampledData_(0, 1, 8000)
    , encodedData_()
    , converterEncode_(0)
    , converterDecode_(0)
    , codecFrameSize_(0)
    , codecChannels_(0)
    , converterSamplingRate_(0)
    , fadeFactor_(1.0 / 32000.0)
#if HAVE_SPEEXDSP
    , dspEncode_(0)
    , dspDecode_(0)
    , audioProcessMutex_()
#endif
    , dtmfPayloadType_(101) // same as Asterisk
    , dead_(false)
    , currentCodecIndex_(0)
{}

// Call from processData*
bool AudioRtpRecord::isDead()
{
    return dead_;
}

sfl::AudioCodec *
AudioRtpRecord::getCurrentCodec() const
{
    if (audioCodecs_.empty() or currentCodecIndex_ >= audioCodecs_.size()) {
        ERROR("No codec found");
        return 0;
    }

    return audioCodecs_[currentCodecIndex_];
}

void
AudioRtpRecord::deleteCodecs()
{
    for (auto &i : audioCodecs_)
        delete i;

    audioCodecs_.clear();
}

bool AudioRtpRecord::tryToSwitchPayloadTypes(int newPt)
{
    for (std::vector<AudioCodec *>::iterator i = audioCodecs_.begin(); i != audioCodecs_.end(); ++i)
        if (*i and (*i)->getPayloadType() == newPt) {
            decoderPayloadType_ = (*i)->getPayloadType();
            codecSampleRate_ = (*i)->getClockRate();
            codecFrameSize_ = (*i)->getFrameSize();
            codecChannels_ = (*i)->getChannels();
            hasDynamicPayloadType_ = (*i)->hasDynamicPayload();
            currentCodecIndex_ = std::distance(audioCodecs_.begin(), i);
            DEBUG("Switched payload type to %d", newPt);
            return true;
        }

    ERROR("Could not switch payload types");
    return false;
}

AudioRtpRecord::~AudioRtpRecord()
{
    dead_ = true;

    delete converterEncode_;
    converterEncode_ = 0;
    delete converterDecode_;
    converterDecode_ = 0;
    {
        std::lock_guard<std::mutex> lock(audioCodecMutex_);
        deleteCodecs();
    }
#if HAVE_SPEEXDSP
    {
        std::lock_guard<std::mutex> lock(audioProcessMutex_);
        delete dspDecode_;
        dspDecode_ = 0;
        delete dspEncode_;
        dspEncode_ = 0;
    }
#endif
}

AudioRtpRecordHandler::AudioRtpRecordHandler(SIPCall &call) :
    audioRtpRecord_(),
    id_(call.getCallId()),
    warningInterval_(0)
{}


AudioRtpRecordHandler::~AudioRtpRecordHandler()
{
}

std::string
AudioRtpRecordHandler::getCurrentAudioCodecNames()
{
    std::string result;
    std::lock_guard<std::mutex> lock(audioRtpRecord_.audioCodecMutex_);
    {
        std::string sep = "";

        for (auto &i : audioRtpRecord_.audioCodecs_) {
            if (i)
                result += sep + i->getMimeSubtype();

            sep = " ";
        }
    }

    return result;
}

void AudioRtpRecordHandler::setRtpMedia(const std::vector<AudioCodec*> &audioCodecs)
{
    std::lock_guard<std::mutex> lock(audioRtpRecord_.audioCodecMutex_);

    audioRtpRecord_.deleteCodecs();
    // Set various codec info to reduce indirection
    audioRtpRecord_.audioCodecs_ = audioCodecs;

    if (audioCodecs.empty()) {
        ERROR("Audio codecs empty");
        return;
    }

    audioRtpRecord_.currentCodecIndex_ = 0;
    audioRtpRecord_.encoderPayloadType_ = audioRtpRecord_.decoderPayloadType_ = audioCodecs[0]->getPayloadType();
    audioRtpRecord_.codecSampleRate_ = audioCodecs[0]->getClockRate();
    audioRtpRecord_.codecFrameSize_ = audioCodecs[0]->getFrameSize();
    audioRtpRecord_.codecChannels_ = audioCodecs[0]->getChannels();
    audioRtpRecord_.hasDynamicPayloadType_ = audioCodecs[0]->hasDynamicPayload();
}

void AudioRtpRecordHandler::initBuffers()
{
    // Set sampling rate, main buffer choose the highest one
    Manager::instance().audioSamplingRateChanged(audioRtpRecord_.codecSampleRate_);

    // initialize SampleRate converter using AudioLayer's sampling rate
    // (internal buffers initialized with maximal sampling rate and frame size)
    delete audioRtpRecord_.converterEncode_;
    audioRtpRecord_.converterEncode_ = new SamplerateConverter(getCodecSampleRate());
    delete audioRtpRecord_.converterDecode_;
    audioRtpRecord_.converterDecode_ = new SamplerateConverter(getCodecSampleRate());
}

#if HAVE_SPEEXDSP
void AudioRtpRecordHandler::initDSP()
{
    std::lock_guard<std::mutex> lock(audioRtpRecord_.audioProcessMutex_);
    delete audioRtpRecord_.dspEncode_;
    audioRtpRecord_.dspEncode_ = new DSP(getCodecFrameSize(), getCodecChannels(), getCodecSampleRate());
    delete audioRtpRecord_.dspDecode_;
    audioRtpRecord_.dspDecode_ = new DSP(getCodecFrameSize(), getCodecChannels(), getCodecSampleRate());
}
#endif

void AudioRtpRecordHandler::putDtmfEvent(char digit)
{
    DTMFEvent dtmf(digit);
    audioRtpRecord_.dtmfQueue_.push_back(dtmf);
}

#define RETURN_IF_NULL(A, VAL, M, ...) if (!(A)) { ERROR(M, ##__VA_ARGS__); return VAL; }

int AudioRtpRecordHandler::processDataEncode()
{
    if (audioRtpRecord_.isDead())
        return 0;

    int codecSampleRate = getCodecSampleRate();
    int mainBufferSampleRate = Manager::instance().getMainBuffer().getInternalSamplingRate();

    double resampleFactor = (double) mainBufferSampleRate / codecSampleRate;

    // compute nb of byte to get corresponding to 1 audio frame
    const size_t samplesToGet = resampleFactor * getCodecFrameSize();

    if (Manager::instance().getMainBuffer().availableForGet(id_) < samplesToGet)
        return 0;

    AudioBuffer& micData = audioRtpRecord_.decData_;
    micData.resize(samplesToGet);
    const size_t samps = Manager::instance().getMainBuffer().getData(micData, id_);

    if (samps != samplesToGet) {
        ERROR("Asked for %d samples from mainbuffer, got %d", samplesToGet, samps);
        return 0;
    }

    audioRtpRecord_.fadeInDecodedData();

    AudioBuffer *out = &micData;

    if (codecSampleRate != mainBufferSampleRate) {
        RETURN_IF_NULL(audioRtpRecord_.converterEncode_, 0, "Converter already destroyed");

        micData.setSampleRate(mainBufferSampleRate);
        audioRtpRecord_.resampledData_.setSampleRate(codecSampleRate);
        audioRtpRecord_.converterEncode_->resample(micData, audioRtpRecord_.resampledData_);

        out = &(audioRtpRecord_.resampledData_);
    }

#if HAVE_SPEEXDSP
    const bool denoise = Manager::instance().audioPreference.getNoiseReduce();
    const bool agc = Manager::instance().audioPreference.isAGCEnabled();

    if (denoise or agc) {
        std::lock_guard<std::mutex> lock(audioRtpRecord_.audioProcessMutex_);
        RETURN_IF_NULL(audioRtpRecord_.dspEncode_, 0, "DSP already destroyed");
        if (denoise)
            audioRtpRecord_.dspEncode_->enableDenoise();
        else
            audioRtpRecord_.dspEncode_->disableDenoise();

        if (agc)
            audioRtpRecord_.dspEncode_->enableAGC();
        else
            audioRtpRecord_.dspEncode_->disableAGC();

        audioRtpRecord_.dspEncode_->process(micData, getCodecFrameSize());
    }
#endif

    {
        std::lock_guard<std::mutex> lock(audioRtpRecord_.audioCodecMutex_);
        RETURN_IF_NULL(audioRtpRecord_.getCurrentCodec(), 0, "Audio codec already destroyed");
        unsigned char *micDataEncoded = audioRtpRecord_.encodedData_.data();
        return audioRtpRecord_.getCurrentCodec()->encode(micDataEncoded, out->getData(), getCodecFrameSize());
    }
}
#undef RETURN_IF_NULL

#define RETURN_IF_NULL(A, M, ...) if (!(A)) { ERROR(M, ##__VA_ARGS__); return; }

void AudioRtpRecordHandler::processDataDecode(unsigned char *spkrData, size_t size, int payloadType)
{
    if (audioRtpRecord_.isDead())
        return;

    if (audioRtpRecord_.decoderPayloadType_ != payloadType) {
        const bool switched = audioRtpRecord_.tryToSwitchPayloadTypes(payloadType);

        if (not switched) {
            if (!warningInterval_) {
                warningInterval_ = 250;
                WARN("Invalid payload type %d, expected %d", payloadType, audioRtpRecord_.decoderPayloadType_);
            }

            warningInterval_--;
            return;
        }
    }

    size = std::min(size, audioRtpRecord_.decData_.frames());

    {
        std::lock_guard<std::mutex> lock(audioRtpRecord_.audioCodecMutex_);
        RETURN_IF_NULL(audioRtpRecord_.getCurrentCodec(), "Audio codecs already destroyed");
        // Return the size of data in samples
        audioRtpRecord_.getCurrentCodec()->decode(audioRtpRecord_.decData_.getData(), spkrData, size);
    }

#if HAVE_SPEEXDSP

    const bool denoise = Manager::instance().audioPreference.getNoiseReduce();
    const bool agc = Manager::instance().audioPreference.isAGCEnabled();

    if (denoise or agc) {
        std::lock_guard<std::mutex> lock(audioRtpRecord_.audioProcessMutex_);
        RETURN_IF_NULL(audioRtpRecord_.dspDecode_, "DSP already destroyed");
        if (denoise)
            audioRtpRecord_.dspDecode_->enableDenoise();
        else
            audioRtpRecord_.dspDecode_->disableDenoise();

        if (agc)
            audioRtpRecord_.dspDecode_->enableAGC();
        else
            audioRtpRecord_.dspDecode_->disableAGC();
        audioRtpRecord_.dspDecode_->process(audioRtpRecord_.decData_, getCodecFrameSize());
    }

#endif

    // ensure that decoded buffer's reported sample rate value
    // is consistent with our decoder.
    audioRtpRecord_.decData_.setSampleRate(getCodecSampleRate());
    audioRtpRecord_.fadeInDecodedData();

    AudioBuffer *out = &(audioRtpRecord_.decData_);

    int codecSampleRate = out->getSampleRate();
    int mainBufferSampleRate = Manager::instance().getMainBuffer().getInternalSamplingRate();

    // test if resampling is required
    if (codecSampleRate != mainBufferSampleRate) {
        RETURN_IF_NULL(audioRtpRecord_.converterDecode_, "Converter already destroyed");
        audioRtpRecord_.resampledData_.setSampleRate(mainBufferSampleRate);
        out = &(audioRtpRecord_.resampledData_);
        // Do sample rate conversion
        audioRtpRecord_.converterDecode_->resample(audioRtpRecord_.decData_, audioRtpRecord_.resampledData_);
    }

    Manager::instance().getMainBuffer().putData(*out, id_);
}
#undef RETURN_IF_NULL

void AudioRtpRecord::fadeInDecodedData()
{
    // if factor reaches 1, this function should have no effect
    if (fadeFactor_ >= 1.0)
        return;

    decData_.applyGain(fadeFactor_);

    // Factor used to increase volume in fade in
    const double FADEIN_STEP_SIZE = 4.0;
    fadeFactor_ *= FADEIN_STEP_SIZE;
}

bool
AudioRtpRecordHandler::codecsDiffer(const std::vector<AudioCodec*> &codecs) const
{
    const std::vector<AudioCodec*> &current = audioRtpRecord_.audioCodecs_;

    if (codecs.size() != current.size())
        return true;

    for (const auto &i : codecs) {
        if (!i)
            continue;

        bool matched = false;

        for (const auto &j : current)
            if ((matched = i->getPayloadType() == j->getPayloadType()))
                break;

        if (not matched)
            return true;
    }

    return false;
}

}
