/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file Parameters.cxx
    \author Romain BONDUE
    \date 07/07/2005 */
#include <cctype> // isxdigit(), toupper()
#include <cstring> // memcpy(), memset(), strcmp()
#ifndef NDEBUG
#include <iostream>
#endif // NDEBUG

#include "Parameters.h"



namespace
{
        // Convert 0000 1110 to 'E' for instance (without check).
    inline unsigned char BitsToHexChar (unsigned char Bits) throw()
    {
        return Bits + (Bits < 10 ? '0' : 'A' - 10);

    } // BitsToHexChar()


        // Convert 'E' to 0000 1110 for instance (without check).
    inline unsigned char HexCharToBits (unsigned char HexChar) throw()
    {
        HexChar = toupper (HexChar);
        return HexChar - (HexChar > '9' ? 'A' - 10 : '0');

    } // HexCharToBits()


    const std::string AutoName ("Auto");
    const std::string AdHocName ("Ad-Hoc");
    const std::string ManagedName ("Managed");
    const std::string MasterName ("Master");
    const std::string RepeaterName ("Repeater");
    const std::string SecondaryName ("Secondary");
    const std::string MonitorName ("Monitor");
    const std::string UnknownModeNTCTS ("Unknown mode");

    const std::string NoneName ("NONE");
    const std::string WEPName ("WEP");
    const std::string AESName ("AES");
    const std::string TKIPName ("TKIP");
    const std::string TKIPAESName ("TKIP, AES");
    const std::string UnknownEncryptNTCTS ("UNKNOWN");

    const std::string OpenName ("OPEN");
    const std::string SharedName ("SHARED");
    const std::string WPAPSKName ("WPAPSK");
    const std::string WPANONEName ("WPANONE");
    const std::string WPA2PSKName ("WPA2PSK");
    const std::string WPAName ("WPA");
    const std::string WPA2Name ("WPA2");
    const std::string UnknownAuthNTCTS ("UNKNOWN");

} // anonymous namespace



bool nsWireless::CMacAddress::Empty () const throw()
{
    if (CHexaKey::Empty()) return true;
    const std::string& Str (GetStr());
    std::string::size_type i (0);
    while (i < Str.size() && Str [i] == '0') ++i;
    return i == Str.size();

} // Empty()


const std::string& nsWireless::GetModeName (Mode_e Mode) throw()
{
    switch (Mode)
    {
      case Auto : return AutoName;
      case AdHoc : return AdHocName;
      case Managed : return ManagedName;
      case Master : return MasterName;
      case Repeater : return RepeaterName;
      case Secondary : return SecondaryName;
      case Monitor : return MonitorName;
      default : return UnknownModeNTCTS;
    }

} // GetModeName()


nsWireless::Mode_e nsWireless::GetModeFromName (const std::string& Mode)
                                                                        throw()
{
    if (Mode == AutoName) return Auto;
    if (Mode == AdHocName) return AdHoc;
    if (Mode == ManagedName) return Managed;
    if (Mode == MasterName) return Master;
    if (Mode == RepeaterName) return Repeater;
    if (Mode == SecondaryName) return Secondary;
    if (Mode == MonitorName) return Monitor;
    return Unknown;

} // GetModeFromName()


const std::string& nsWireless::GetEncryptName (EncryptType_e Encrypt) throw()
{
    switch (Encrypt)
    {
      case None : return NoneName;
      case WEP : return WEPName;
      case AES : return AESName;
      case TKIP : return TKIPName;
      case TKIPAES : return TKIPAESName;
      default : return UnknownEncryptNTCTS;
    }

} // GetEncryptName()


nsWireless::EncryptType_e nsWireless::GetEncryptFromName
                                            (const std::string& Name) throw()
{
    if (Name == WEPName) return WEP;
    if (Name == NoneName) return None;
    if (Name == TKIPName) return TKIP;
    if (Name == AESName) return AES;
    if (Name == TKIPAESName) return TKIPAES;
    return UnknownEnc;

} // GetEncryptFromName()


const std::string& nsWireless::GetAuthName (AuthType_e Auth) throw()
{
    switch (Auth)
    {
      case Open : return OpenName;
      case Shared : return SharedName;
      case WPAPSK : return WPAPSKName;
      case WPANONE : return WPANONEName;
      case WPA2PSK : return WPA2PSKName;
      case WPA : return WPAName;
      case WPA2 : return WPA2Name;
      default : return UnknownAuthNTCTS;
    }

} // GetAuthName()


nsWireless::AuthType_e nsWireless::GetAuthFromName (const std::string& Name)
                                                                        throw()
{
    if (Name == OpenName) return Open;
    if (Name == SharedName) return Shared;
    if (Name == WPAPSKName) return WPAPSK;
    if (Name == WPANONEName) return WPANONE;
    if (Name == WPA2PSKName) return WPA2PSK;
    if (Name == WPAName) return WPA;
    if (Name == WPA2Name) return WPA2;
    return UnknownAuth;

} // GetAuthFromName()


nsWireless::CHexaKey& nsWireless::CHexaKey::operator =
                                (const CHexaKey& Key) throw (std::bad_alloc)
{
    if (this == &Key) return *this;
    Set (Key.m_Key, Key.m_Size);
    m_StrKey = Key.m_StrKey;

    return *this;

} // operator = ()


void nsWireless::CHexaKey::Set (const char* Key, unsigned Size)
                                                        throw (std::bad_alloc)
{
    if (Key)
    {
        m_StrKey.erase();
        if (!Size)
        {
            delete[] m_Key;
            m_Key = 0;
        }
        else if (m_Size < Size)
        {
            delete[] m_Key;
            try{m_Key = new char [Size];}
            catch (const std::bad_alloc&)
            {
                m_Key = 0;
                m_Size = 0;
                throw;
            }
                
        }
        memcpy (m_Key, Key, Size);
        m_Size = Size;
    }

} // Set()


void nsWireless::CHexaKey::Set (std::string Key) throw (CBadFormatExc)
{
    for (std::string::size_type i (Key.size()) ; i-- ; )
        if (Key [i] == '-' || Key [i] == ':' || Key [i] == ' ')
            Key.erase (i, 1);
        else if (!isxdigit (static_cast<unsigned char> (Key[i])))
        {
#ifndef NDEBUG
            std::cerr << "Bad format, key : #" << Key << "# byte " << i
                      << " is not an hexadecimal digit." << std::endl;
#endif // NDEBUG
            throw CBadFormatExc();
        }
        else
            Key [i] = toupper (static_cast<unsigned char> (Key[i]));
    if (Key.empty())
    {
#ifndef NDEBUG
        std::cerr << "Bad format, key : #" << Key << "# is empty." << std::endl;
#endif // NDEBUG
        throw CBadFormatExc();
    }
    m_Size = 0;
    delete[] m_Key;
    m_Key = 0;
    m_StrKey = Key;

} // Set()


void nsWireless::CHexaKey::SetFromASCII (const std::string& Key)
                                                        throw (std::bad_alloc)
{
    m_StrKey.erase();
    if (m_Size < Key.size())
    {
        delete[] m_Key;
        try{m_Key = new char [Key.size()];}
        catch (const std::bad_alloc&)
        {
            m_Key = 0;
            m_Size = 0;
            throw;
        }
    }
    m_Size = Key.size();
    Key.copy (m_Key, m_Size);

} // SetFromASCII()


void nsWireless::CHexaKey::Clear () throw()
{
    m_Size = 0;
    delete[] m_Key;
    m_Key = 0;
    m_StrKey.erase();

} // Clear()


const std::string& nsWireless::CHexaKey::GetStr () const throw()
{
    if (m_StrKey.empty() && m_Size)
    {
            /* 4 bits (1/2 byte) are represented with 1 hexadecimal
             * digit (1 byte). */
        m_StrKey.reserve (m_Size * 2);
        for (unsigned i (0) ; i < m_Size ; ++i)
        {
                // This code assume sizeof (char) == 1.
            m_StrKey.push_back (BitsToHexChar (static_cast<unsigned char>
                                                            (m_Key [i]) >> 4));
            m_StrKey.push_back (BitsToHexChar (m_Key [i] & 0x0F));
        }
    }
    return m_StrKey;

} // GetStr()


const char* nsWireless::CHexaKey::Get () const throw (std::bad_alloc)
{
    if (!m_Size && !m_StrKey.empty())
    {
        m_Size = Size();
        try{m_Key = new char [m_Size];}
        catch (const std::bad_alloc&)
        {
            m_Key = 0;
            m_Size = 0;
            throw;
        }
        memset (m_Key, 0, m_Size);
        for (std::string::size_type i (0) ; i < m_StrKey.size() ; ++i)
            m_Key [i / 2] |= HexCharToBits (m_StrKey [i]) << (i & 1 ? 0 : 4);
    }
    return m_Key;

} // Get()


bool nsWireless::CHexaKey::operator == (const CHexaKey& Key) const throw()
{
    if (m_Size && Key.m_Size)
    {
        if (m_Size != Key.m_Size) return false;
        for (unsigned i (0) ; i < m_Size ; ++i)
            if (m_Key [i] != Key.m_Key [i]) return false;
        return true;
    }
    return GetStr() == Key.GetStr();

} // operator == ()


nsWireless::CEncryptionD& nsWireless::CEncryptionD::operator =
                                (const CEncryptionD& D) throw (std::bad_alloc)
{
    if (&D == this) return *this;
    m_Auth = D.m_Auth;
    m_Encrypt = D.m_Encrypt;
    m_DefaultKey = D.m_DefaultKey;
    for (unsigned I (0) ; I < MaxNbKey ; ++I)
        m_KeyTab [I] = D.m_KeyTab [I];

    return *this;

} // operator = ()


bool nsWireless::CEncryptionD::operator == (const CEncryptionD& D) const
                                                                        throw()
{
    if (m_Auth == D.m_Auth && m_Encrypt == D.m_Encrypt ||
                                            m_Encrypt == nsWireless::TKIPAES && 
                                            (D.m_Encrypt == nsWireless::TKIP ||
                                             D.m_Encrypt == nsWireless::AES) ||
                                        D.m_Encrypt == nsWireless::TKIPAES &&
                (m_Encrypt == nsWireless::TKIP || m_Encrypt == nsWireless::AES))
    {
        for (unsigned I (0) ; I < MaxNbKey ; ++I)
            if (m_KeyTab [I] != D.m_KeyTab [I])
                return false;
        return true;
    }
    return false;

} // operator == ()


nsWireless::CCell::CCell (const CMacAddress& Addr, Mode_e Mode,
                          const std::string& SSID,
                          const CEncryptionD& Descriptor, unsigned Channel,
                          const CQuality& Quality, unsigned Rate)
                                                        throw (std::bad_alloc)
    : m_APAddr (Addr), m_Mode (Mode), m_SSID (SSID),
      m_EncDescriptor (Descriptor), m_Channel (Channel), m_Quality (Quality),
      m_Rate (Rate) {}
