#include "codeclib.h"
#include "str.h"
#include "main.h"
#include <assert.h>

struct rtpengine_config rtpe_config;
struct rtpengine_config initial_rtpe_config;

static void hexdump(const unsigned char *buf, size_t len) {
	for (size_t i = 0; i < len; i++)
		printf("%02x", buf[i]);
	printf("\n");
}

struct cb_args {
	const unsigned char *expect_s;
	size_t expect_len;
	const unsigned char *expect2_s;
	size_t expect2_len;
	bool done;
};

static int frame_cb(decoder_t *dec, AVFrame *frame, void *u1, void *u2) {
	struct cb_args *args = u1;
	assert(args);
	assert(args->expect_s);
	assert(args->expect_len);
	if (args->expect_len != frame->linesize[0]
			|| memcmp(frame->data[0], args->expect_s, args->expect_len))
	{
		if (!args->expect2_s || args->expect2_len != frame->linesize[0]
				|| memcmp(frame->data[0], args->expect2_s, args->expect2_len))
		{
			printf(
				"packet content mismatch\n"
				"expected %zu bytes, received %i bytes\n"
				"expected:\n",
				args->expect_len, frame->linesize[0]);
			hexdump(args->expect_s, args->expect_len);
			printf("received:\n");
			hexdump((unsigned char *) frame->data[0], frame->linesize[0]);
			exit(1);
		}
	}
	args->done = true;
	av_frame_free(&frame);
	return 0;
}

static void do_test_amr_xx(const char *file, int line,
		char *fmtp_s, char *data_s, int data_len,
		struct cb_args args,
		char *codec, int clockrate)
{
	printf("running test %s:%i\n", file, line);
	str codec_name = STR_INIT(codec);
	codec_def_t *def = codec_find(&codec_name, MT_AUDIO);
	assert(def);
	if (!def->support_encoding || !def->support_decoding) {
		printf("AMR not fully supported - skipping test\n");
		exit(0);
	}
	const format_t fmt = { .clockrate = clockrate, .channels = 1, .format = AV_SAMPLE_FMT_S16};
	str fmtp_str, *fmtp = NULL;
	if (fmtp_s) {
		str_init(&fmtp_str, fmtp_s);
		fmtp = &fmtp_str;
	}
	decoder_t *d = decoder_new_fmtp(def, clockrate, 1, 0, &fmt, NULL, NULL, fmtp);
	assert(d);
	const str data = { data_s, data_len };
	int ret = decoder_input_data(d, &data, 1, frame_cb, &args, NULL);
	assert(!ret);
	assert(args.done == true);
	decoder_close(d);
	printf("test ok: %s:%i\n", file, line);
}

static void do_test_amr_wb(const char *file, int line,
		char *fmtp_s, char *data_s, size_t data_len,
		struct cb_args args)
{
	do_test_amr_xx(file, line, fmtp_s, data_s, data_len,
			args,
			"AMR-WB", 16000);
}
static void do_test_amr_nb(const char *file, int line,
		char *fmtp_s, char *data_s, int data_len,
		struct cb_args args)
{
	do_test_amr_xx(file, line, fmtp_s, data_s, data_len,
			args,
			"AMR", 8000);
}

#define do_test_wb(in, out, out2, fmt) \
	do_test_amr_wb(__FILE__, __LINE__, fmt, in, sizeof(in)-1, \
			(struct cb_args) { \
				.expect_s = (const unsigned char *) out, .expect_len = sizeof(out)-1, \
				.expect2_s = (const unsigned char *) out2, .expect2_len = sizeof(out2)-1, \
			} )
#define do_test_nb(in, out, out2, fmt) \
	do_test_amr_nb(__FILE__, __LINE__, fmt, in, sizeof(in)-1, \
			(struct cb_args) { \
				.expect_s = (const unsigned char *) out, .expect_len = sizeof(out)-1, \
				.expect2_s = (const unsigned char *) out2, .expect2_len = sizeof(out2)-1, \
			} )

int main(void) {
	rtpe_common_config_ptr = &rtpe_config.common;
	codeclib_init(0);

	do_test_wb(
			"\xf0\x44\xf1\x46\x18\x1d\xd1\x57\x23\x13\x42\xf0\x00\x0c\x50\x33\xdd\xff\x0b\x99\x89\x2c\x68\x52\xf8\xf8\xd9\x59\x16\xd7\x45\xe7\x01\xec\x1f\xfe\x5b\xc6\xf9\x01\xa4\xb5\xe0\x6c\x91\x41\xfe\x52\x2c\xce\x44\xbb\x5a\xdf\x76\x29\xf8\xdb\xca\x18\xd6\x50",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\xff\xff\x02\x00\xff\xff\xff\xff\x02\x00\xfd\xff\x03\x00\x00\x00\xfd\xff\x04\x00\xfc\xff\x02\x00\x01\x00\xfd\xff\x03\x00\xfe\xff\x01\x00\x01\x00\xff\xff\x01\x00\xff\xff\x01\x00\xff\xff\x00\x00\x01\x00\xff\xff\x00\x00\x00\x00\xff\xff\x00\x00\x01\x00\xfe\xff\x02\x00\xff\xff\xff\xff\x03\x00\xfd\xff\x03\x00\xff\xff\xff\xff\x03\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x02\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x02\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x04\x00\xfe\xff\xfd\xff\x00\x00\x02\x00\xfe\xff\xf8\xff\x01\x00\x04\x00\xff\xff\xff\xff\xfc\xff\x06\x00\x00\x00\xf8\xff\x11\x00\x09\x00\x06\x00\x3f\x00\x37\x00\xf9\xff\x11\x00\x4e\x00\x34\x00\xf4\xff\x17\x00\x5d\x00\x31\x00\xe0\xff\x0b\x00\x71\x00\x42\x00\xd3\xff\x09\x00\x74\x00\x3c\x00\xc8\xff\x03\x00\x78\x00\x35\x00\xbc\xff\xff\xff\x7a\x00\x2e\x00\xb0\xff\x00\x00\x77\x00\x26\x00\xaa\xff\xfc\xff\x78\x00\x1c\x00\xa3\xff\xfe\xff\x72\x00\x14\x00\xa0\xff\xfc\xff\x6f\x00\x09\x00\x8e\xff\xfc\xff\x72\x00\xff\xff\x89\xff\xff\xff\x7e\x00\xfe\xff\x7b\xff\x19\x00\xa9\x00\xfa\xff\x62\xff\x14\x00\xae\x00\xf5\xff\x54\xff\x16\x00\xb6\x00\xe8\xff\x3f\xff\x0b\x00\xb9\x00\xee\xff\x34\xff\xfd\xff\xb8\x00\xe9\xff\x2d\xff\x00\x00\xb8\x00\xe4\xff\x2c\xff\xff\xff\xb9\x00\xdf\xff\x25\xff\xff\xff\xb2\x00\xda\xff\x28\xff\xfc\xff\xae\x00\xd6\xff\x2a\xff\xff\xff\xa5\x00\xd8\xff\x30\xff\xfc\xff\xa1\x00\xd5\xff\x35\xff\xf9\xff\x97\x00\xd4\xff\x37\xff\xfa\xff\x92\x00\xcd\xff\x38\xff\xfe\xff\x8e\x00\xcb\xff\x3e\xff\xfe\xff\x88\x00\xcc\xff\x40\xff\xfa\xff\x89\x00\xcf\xff\x41\xff\xfa\xff\x87\x00\xd0\xff\x44\xff\xfa\xff\x89\x00\xd6\xff\x48\xff\xf9\xff\x88\x00\xdd\xff\x4d\xff\xf2\xff\x81\x00\xde\xff\x54\xff\xf4\xff\x7b\x00\xde\xff\x5c\xff\xf6\xff\x73\x00\xe0\xff\x65\xff\xf6\xff\x6d\x00\xe0\xff\x6f\xff\xf7\xff\x63\x00\xe0\xff\x78\xff\xf7\xff\x5d\x00\xde\xff\x76\xff\xf9\xff\x60\x00\xdf\xff\x7f\xff\xfb\xff\x5c\x00\xe8\xff\x85\xff\xfb\xff\x60\x00\xea\xff\x87\xff\xfe\xff\x63\x00\xee\xff\x8b\xff\x00\x00\x64\x00\xf3\xff\x8d\xff\xfe\xff\x66\x00\xf7\xff\x8f\xff\xfd\xff\x68\x00\xf9\xff\x8c\xff\xfd\xff\x6d\x00\xfc\xff\x8c\xff\xfd\xff\x71\x00\xff\xff\x89\xff\xfe\xff\x75\x00\x02\x00\x88\xff\xfc\xff\x78\x00\x03\x00\x87\xff\xfd\xff\x7b\x00\x03\x00\x86\xff\x00\x00\x7e\x00\x03\x00\x84\xff\xfe\xff\x81\x00\x07\x00\x82\xff\x01\x00\x84\x00\x03\x00\x82\xff\x05\x00\x88\x00\x05\x00\x81\xff\x04\x00\x88\x00\x05\x00\x80\xff\x05\x00\x8a\x00\x05\x00",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x04\x00\xfc\xff\x00\x00\x04\x00\xf0\xff\x14\x00\x64\x00\x48\x00\xa0\xff\x84\xff\x14\x00\x58\x00\x18\x00\xa8\xff\x0c\x00\xc4\x00\x38\x00\x34\xff\x7c\xff\x78\x00\x78\x00\xb8\xff\x80\xff\x40\x00\xa4\x00\xc4\xff\x44\xff\x10\x00\xa8\x00\x0c\x00\x60\xff\xcc\xff\x98\x00\x54\x00\x64\xff\x84\xff\x70\x00\x88\x00\xb0\xff\x6c\xff\x30\x00\x88\x00\xf8\xff\x5c\xff\xec\xff\x9c\x00\x1c\x00\x84\xff\xac\xff\x70\x00\x60\x00\x98\xff\x90\xff\x38\x00\x80\x00\xd8\xff\x78\xff\x08\x00\x78\x00\x14\x00\x84\xff\xd0\xff\x6c\x00\x3c\x00\xa0\xff\xa8\xff\x44\x00\x64\x00\xc8\xff\x94\xff\x14\x00\x6c\x00\x00\x00\x8c\xff\xf0\xff\x58\x00\x2c\x00\xa4\xff\xc0\xff\x4c\x00\x44\x00\xc4\xff\xb0\xff\x28\x00\x38\x00\xfc\xff\xb0\xff\x18\x00\xdc\x00\xfc\x00\x20\x00\x98\xfe\x10\xff\xbc\x00\xe8\x00\xe8\xff\x2c\xff\xf4\xff\xe4\x00\x1c\x00\x14\xff\xa8\xff\xf4\x00\x80\x00\xec\xfe\x90\xff\xdc\x00\x9c\x00\x4c\xff\xd0\xfe\xa4\x00\x60\x01\xa4\xff\xcc\xfe\x1c\x00\xd8\x00\xf8\xff\x44\xff\xc0\xff\x40\x01\x60\x01\xbc\xff\x94\xfe\x48\xff\xbc\x00\xa0\x00\x98\xff\x88\xff\x58\x00\x98\x00\xac\xff\x40\xff\x1c\x00\x94\x00\x00\x00\x2c\xff\xf4\xff\xf8\x00\x20\x00\x24\xff\x50\xff\xa8\x00\xdc\x00\x64\xff\x38\xff\x54\x00\xb8\x00\xd0\xff\x34\xff\xf8\xff\xa4\x00\x44\x00\x40\xff\x9c\xff\xac\x00\x30\x00\xac\xff\x88\xff\x0c\x00\x94\x00\xf0\xff\xac\xff\x08\x00\x04\x00\xf4\xff\x0c\x00\xc8\xff\xe0\xff\xa4\x00\xd0\x00\xe4\xff\x20\xff\x78\xff\x7c\x00\x78\x00\x64\xff\x98\xff\xc8\x00\x24\x01\x2c\x00\x48\xff\x8c\xff\xe8\xff\x0c\x00\xf0\xff\xf8\xff\x4c\x00\x40\x00\xc0\xff\xc8\xff\x0c\x00\xe0\xff\xe8\xff\xec\xff\x2c\x00\x44\x00\xe8\xff\xb8\xff\xc4\xff\xf8\xff\x20\x00\x40\x00\xf4\xff\xa4\xff\xec\xff\x38\x00\x1c\x00\xfc\xff\xa0\xff\x80\x00\xcc\x01\x98\x00\xb4\xfe\x74\xfe\x24\x00\x20\x01\x28\x00\x08\xff\xe0\xff\x30\x01\xf4\xff\xf0\xfe\xbc\xff\xcc\x00\x68\x00\xd4\xfe\x38\xff\x00\x01\x58\x01\x78\xff\x28\xfe\xe4\xff\x3c\x01\x5c\x00\x2c\xff\x94\xff\xcc\x00\x34\x00\x0c\xff\x58\xff\xd8\x00\xe0\x00\x5c\xff\xf8\xfe\x1c\x00\x18\x01\xf0\xff\xe0\xfe\x00\x00\x04\x01\xf4\xff\xe0\xfe\xb8\xff\xf4\x00\xa0\x00\x14\xff\x44\xff\x54\x01\x48\x01\xe4\xfe\x7c\xfe\x6c\x00\x0c\x01\xe8\xff\x30\xff\x08\x00\x08\x01\x40\x00\xac\xff\x90\x00\xa4\x00\x1c\xff\x74\xfe\x3c\x00\x5c\x01\x74\x00\x08\xff\x40\xff\xbc\x00\x88\x00\x64\xff\x78\xff\x68\x00\x6c\x00\x78\xff\x6c\xff\x8c\x00\xc4\x00\x78\xff\x20\xff\xbc\x00\x40\x01\x94\xff\x9c\xfe\xa4\xff\x14\x01\x5c\x00\x3c\xff\x94\x00\xb4\x01\x6c\x00\x48\xfe\x70\xfe\xc8\x00\x7c\x01\x7c\xff",
			"octet-align=1");

	do_test_wb(
			"\xf4\x7c\x51\x86\x07\x74\x55\xc8\xc4\xd0\xbc\x00\x03\x14\x0c\xf7\x7f\xc2\xe6\x62\x4b\x1a\x14\xbe\x3e\x36\x56\x45\xb5\xd1\x79\xc0\x7b\x07\xff\x96\xf1\xbe\x40\x69\x2d\x78\x1b\x24\x50\x7f\x94\x8b\x33\x91\x2e\xd6\xb7\xdd\x8a\x7e\x36\xf2\x86\x35\x94",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\xff\xff\x02\x00\xff\xff\xff\xff\x02\x00\xfd\xff\x03\x00\x00\x00\xfd\xff\x04\x00\xfc\xff\x02\x00\x01\x00\xfd\xff\x03\x00\xfe\xff\x01\x00\x01\x00\xff\xff\x01\x00\xff\xff\x01\x00\xff\xff\x00\x00\x01\x00\xff\xff\x00\x00\x00\x00\xff\xff\x00\x00\x01\x00\xfe\xff\x02\x00\xff\xff\xff\xff\x03\x00\xfd\xff\x03\x00\xff\xff\xff\xff\x03\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x02\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x02\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x04\x00\xfe\xff\xfd\xff\x00\x00\x02\x00\xfe\xff\xf8\xff\x01\x00\x04\x00\xff\xff\xff\xff\xfc\xff\x06\x00\x00\x00\xf8\xff\x11\x00\x09\x00\x06\x00\x3f\x00\x37\x00\xf9\xff\x11\x00\x4e\x00\x34\x00\xf4\xff\x17\x00\x5d\x00\x31\x00\xe0\xff\x0b\x00\x71\x00\x42\x00\xd3\xff\x09\x00\x74\x00\x3c\x00\xc8\xff\x03\x00\x78\x00\x35\x00\xbc\xff\xff\xff\x7a\x00\x2e\x00\xb0\xff\x00\x00\x77\x00\x26\x00\xaa\xff\xfc\xff\x78\x00\x1c\x00\xa3\xff\xfe\xff\x72\x00\x14\x00\xa0\xff\xfc\xff\x6f\x00\x09\x00\x8e\xff\xfc\xff\x72\x00\xff\xff\x89\xff\xff\xff\x7e\x00\xfe\xff\x7b\xff\x19\x00\xa9\x00\xfa\xff\x62\xff\x14\x00\xae\x00\xf5\xff\x54\xff\x16\x00\xb6\x00\xe8\xff\x3f\xff\x0b\x00\xb9\x00\xee\xff\x34\xff\xfd\xff\xb8\x00\xe9\xff\x2d\xff\x00\x00\xb8\x00\xe4\xff\x2c\xff\xff\xff\xb9\x00\xdf\xff\x25\xff\xff\xff\xb2\x00\xda\xff\x28\xff\xfc\xff\xae\x00\xd6\xff\x2a\xff\xff\xff\xa5\x00\xd8\xff\x30\xff\xfc\xff\xa1\x00\xd5\xff\x35\xff\xf9\xff\x97\x00\xd4\xff\x37\xff\xfa\xff\x92\x00\xcd\xff\x38\xff\xfe\xff\x8e\x00\xcb\xff\x3e\xff\xfe\xff\x88\x00\xcc\xff\x40\xff\xfa\xff\x89\x00\xcf\xff\x41\xff\xfa\xff\x87\x00\xd0\xff\x44\xff\xfa\xff\x89\x00\xd6\xff\x48\xff\xf9\xff\x88\x00\xdd\xff\x4d\xff\xf2\xff\x81\x00\xde\xff\x54\xff\xf4\xff\x7b\x00\xde\xff\x5c\xff\xf6\xff\x73\x00\xe0\xff\x65\xff\xf6\xff\x6d\x00\xe0\xff\x6f\xff\xf7\xff\x63\x00\xe0\xff\x78\xff\xf7\xff\x5d\x00\xde\xff\x76\xff\xf9\xff\x60\x00\xdf\xff\x7f\xff\xfb\xff\x5c\x00\xe8\xff\x85\xff\xfb\xff\x60\x00\xea\xff\x87\xff\xfe\xff\x63\x00\xee\xff\x8b\xff\x00\x00\x64\x00\xf3\xff\x8d\xff\xfe\xff\x66\x00\xf7\xff\x8f\xff\xfd\xff\x68\x00\xf9\xff\x8c\xff\xfd\xff\x6d\x00\xfc\xff\x8c\xff\xfd\xff\x71\x00\xff\xff\x89\xff\xfe\xff\x75\x00\x02\x00\x88\xff\xfc\xff\x78\x00\x03\x00\x87\xff\xfd\xff\x7b\x00\x03\x00\x86\xff\x00\x00\x7e\x00\x03\x00\x84\xff\xfe\xff\x81\x00\x07\x00\x82\xff\x01\x00\x84\x00\x03\x00\x82\xff\x05\x00\x88\x00\x05\x00\x81\xff\x04\x00\x88\x00\x05\x00\x80\xff\x05\x00\x8a\x00\x05\x00",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xf8\xff\x00\x00\x04\x00\xfc\xff\xf8\xff\x00\x00\x04\x00\x00\x00\xfc\xff\xfc\xff\x04\x00\x00\x00\xf8\xff\x10\x00\x08\x00\x04\x00\x40\x00\x34\x00\xf8\xff\x10\x00\x4c\x00\x34\x00\xf4\xff\x14\x00\x5c\x00\x30\x00\xdc\xff\x0c\x00\x70\x00\x40\x00\xd4\xff\x04\x00\x74\x00\x38\x00\xc4\xff\x00\x00\x74\x00\x34\x00\xb8\xff\xfc\xff\x78\x00\x2c\x00\xb0\xff\xfc\xff\x78\x00\x24\x00\xa4\xff\xfc\xff\x74\x00\x18\x00\xa4\xff\xf8\xff\x70\x00\x10\x00\x9c\xff\xfc\xff\x6c\x00\x08\x00\x8c\xff\xf8\xff\x70\x00\x00\x00\x88\xff\xfc\xff\x7c\x00\xfc\xff\x78\xff\x18\x00\xa4\x00\xfc\xff\x60\xff\x10\x00\xac\x00\xf4\xff\x54\xff\x14\x00\xb4\x00\xe8\xff\x40\xff\x08\x00\xb4\x00\xf0\xff\x34\xff\xfc\xff\xb8\x00\xe8\xff\x30\xff\xfc\xff\xb8\x00\xe4\xff\x28\xff\x00\x00\xb4\x00\xe0\xff\x24\xff\xfc\xff\xb4\x00\xd8\xff\x28\xff\xf8\xff\xac\x00\xd8\xff\x24\xff\xfc\xff\xa4\x00\xd8\xff\x30\xff\xf8\xff\xa0\x00\xd4\xff\x30\xff\xf8\xff\x94\x00\xd4\xff\x38\xff\xf8\xff\x94\x00\xcc\xff\x34\xff\xfc\xff\x8c\x00\xcc\xff\x3c\xff\xfc\xff\x88\x00\xcc\xff\x40\xff\xf8\xff\x8c\x00\xd0\xff\x40\xff\xf8\xff\x88\x00\xd0\xff\x40\xff\xf8\xff\x88\x00\xd8\xff\x48\xff\xf8\xff\x88\x00\xe0\xff\x4c\xff\xf0\xff\x84\x00\xe0\xff\x50\xff\xf0\xff\x7c\x00\xe0\xff\x58\xff\xf4\xff\x74\x00\xe0\xff\x64\xff\xf4\xff\x6c\x00\xe4\xff\x6c\xff\xf4\xff\x64\x00\xe0\xff\x74\xff\xf4\xff\x60\x00\xe0\xff\x74\xff\xf8\xff\x64\x00\xe0\xff\x7c\xff\xf8\xff\x5c\x00\xe8\xff\x80\xff\xf8\xff\x64\x00\xec\xff\x84\xff\xf8\xff\x64\x00\xec\xff\x88\xff\xfc\xff\x68\x00\xf4\xff\x88\xff\xfc\xff\x68\x00\xf8\xff\x88\xff\xf8\xff\x6c\x00\xfc\xff\x88\xff\xf8\xff\x74\x00\xfc\xff\x84\xff\xfc\xff\x74\x00\x00\x00\x84\xff\xf8\xff\x78\x00\x04\x00\x80\xff\xf8\xff\x7c\x00\x04\x00\x80\xff\xf8\xff\x80\x00\x04\x00\x7c\xff\xfc\xff\x84\x00\x04\x00\x7c\xff\xf8\xff\x88\x00\x08\x00\x7c\xff\x00\x00\x88\x00\x04\x00\x78\xff\x04\x00\x8c\x00\x04\x00\x78\xff\x04\x00\x90\x00\x04\x00\x78\xff\x00\x00\x90\x00\x04\x00",
			NULL);

	do_test_wb(
			"\xf4\x7c\x51\x86\x07\x74\x55\xc8\xc4\xd0\xbc\x00\x03\x14\x0c\xf7\x7f\xc2\xe6\x62\x4b\x1a\x14\xbe\x3e\x36\x56\x45\xb5\xd1\x79\xc0\x7b\x07\xff\x96\xf1\xbe\x40\x69\x2d\x78\x1b\x24\x50\x7f\x94\x8b\x33\x91\x2e\xd6\xb7\xdd\x8a\x7e\x36\xf2\x86\x35\x94",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\xff\xff\x02\x00\xff\xff\xff\xff\x02\x00\xfd\xff\x03\x00\x00\x00\xfd\xff\x04\x00\xfc\xff\x02\x00\x01\x00\xfd\xff\x03\x00\xfe\xff\x01\x00\x01\x00\xff\xff\x01\x00\xff\xff\x01\x00\xff\xff\x00\x00\x01\x00\xff\xff\x00\x00\x00\x00\xff\xff\x00\x00\x01\x00\xfe\xff\x02\x00\xff\xff\xff\xff\x03\x00\xfd\xff\x03\x00\xff\xff\xff\xff\x03\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x02\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x02\x00\xfe\xff\x01\x00\x00\x00\xff\xff\x04\x00\xfe\xff\xfd\xff\x00\x00\x02\x00\xfe\xff\xf8\xff\x01\x00\x04\x00\xff\xff\xff\xff\xfc\xff\x06\x00\x00\x00\xf8\xff\x11\x00\x09\x00\x06\x00\x3f\x00\x37\x00\xf9\xff\x11\x00\x4e\x00\x34\x00\xf4\xff\x17\x00\x5d\x00\x31\x00\xe0\xff\x0b\x00\x71\x00\x42\x00\xd3\xff\x09\x00\x74\x00\x3c\x00\xc8\xff\x03\x00\x78\x00\x35\x00\xbc\xff\xff\xff\x7a\x00\x2e\x00\xb0\xff\x00\x00\x77\x00\x26\x00\xaa\xff\xfc\xff\x78\x00\x1c\x00\xa3\xff\xfe\xff\x72\x00\x14\x00\xa0\xff\xfc\xff\x6f\x00\x09\x00\x8e\xff\xfc\xff\x72\x00\xff\xff\x89\xff\xff\xff\x7e\x00\xfe\xff\x7b\xff\x19\x00\xa9\x00\xfa\xff\x62\xff\x14\x00\xae\x00\xf5\xff\x54\xff\x16\x00\xb6\x00\xe8\xff\x3f\xff\x0b\x00\xb9\x00\xee\xff\x34\xff\xfd\xff\xb8\x00\xe9\xff\x2d\xff\x00\x00\xb8\x00\xe4\xff\x2c\xff\xff\xff\xb9\x00\xdf\xff\x25\xff\xff\xff\xb2\x00\xda\xff\x28\xff\xfc\xff\xae\x00\xd6\xff\x2a\xff\xff\xff\xa5\x00\xd8\xff\x30\xff\xfc\xff\xa1\x00\xd5\xff\x35\xff\xf9\xff\x97\x00\xd4\xff\x37\xff\xfa\xff\x92\x00\xcd\xff\x38\xff\xfe\xff\x8e\x00\xcb\xff\x3e\xff\xfe\xff\x88\x00\xcc\xff\x40\xff\xfa\xff\x89\x00\xcf\xff\x41\xff\xfa\xff\x87\x00\xd0\xff\x44\xff\xfa\xff\x89\x00\xd6\xff\x48\xff\xf9\xff\x88\x00\xdd\xff\x4d\xff\xf2\xff\x81\x00\xde\xff\x54\xff\xf4\xff\x7b\x00\xde\xff\x5c\xff\xf6\xff\x73\x00\xe0\xff\x65\xff\xf6\xff\x6d\x00\xe0\xff\x6f\xff\xf7\xff\x63\x00\xe0\xff\x78\xff\xf7\xff\x5d\x00\xde\xff\x76\xff\xf9\xff\x60\x00\xdf\xff\x7f\xff\xfb\xff\x5c\x00\xe8\xff\x85\xff\xfb\xff\x60\x00\xea\xff\x87\xff\xfe\xff\x63\x00\xee\xff\x8b\xff\x00\x00\x64\x00\xf3\xff\x8d\xff\xfe\xff\x66\x00\xf7\xff\x8f\xff\xfd\xff\x68\x00\xf9\xff\x8c\xff\xfd\xff\x6d\x00\xfc\xff\x8c\xff\xfd\xff\x71\x00\xff\xff\x89\xff\xfe\xff\x75\x00\x02\x00\x88\xff\xfc\xff\x78\x00\x03\x00\x87\xff\xfd\xff\x7b\x00\x03\x00\x86\xff\x00\x00\x7e\x00\x03\x00\x84\xff\xfe\xff\x81\x00\x07\x00\x82\xff\x01\x00\x84\x00\x03\x00\x82\xff\x05\x00\x88\x00\x05\x00\x81\xff\x04\x00\x88\x00\x05\x00\x80\xff\x05\x00\x8a\x00\x05\x00",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xfc\xff\x00\x00\x00\x00\xfc\xff\x00\x00\xf8\xff\x00\x00\x04\x00\xfc\xff\xf8\xff\x00\x00\x04\x00\x00\x00\xfc\xff\xfc\xff\x04\x00\x00\x00\xf8\xff\x10\x00\x08\x00\x04\x00\x40\x00\x34\x00\xf8\xff\x10\x00\x4c\x00\x34\x00\xf4\xff\x14\x00\x5c\x00\x30\x00\xdc\xff\x0c\x00\x70\x00\x40\x00\xd4\xff\x04\x00\x74\x00\x38\x00\xc4\xff\x00\x00\x74\x00\x34\x00\xb8\xff\xfc\xff\x78\x00\x2c\x00\xb0\xff\xfc\xff\x78\x00\x24\x00\xa4\xff\xfc\xff\x74\x00\x18\x00\xa4\xff\xf8\xff\x70\x00\x10\x00\x9c\xff\xfc\xff\x6c\x00\x08\x00\x8c\xff\xf8\xff\x70\x00\x00\x00\x88\xff\xfc\xff\x7c\x00\xfc\xff\x78\xff\x18\x00\xa4\x00\xfc\xff\x60\xff\x10\x00\xac\x00\xf4\xff\x54\xff\x14\x00\xb4\x00\xe8\xff\x40\xff\x08\x00\xb4\x00\xf0\xff\x34\xff\xfc\xff\xb8\x00\xe8\xff\x30\xff\xfc\xff\xb8\x00\xe4\xff\x28\xff\x00\x00\xb4\x00\xe0\xff\x24\xff\xfc\xff\xb4\x00\xd8\xff\x28\xff\xf8\xff\xac\x00\xd8\xff\x24\xff\xfc\xff\xa4\x00\xd8\xff\x30\xff\xf8\xff\xa0\x00\xd4\xff\x30\xff\xf8\xff\x94\x00\xd4\xff\x38\xff\xf8\xff\x94\x00\xcc\xff\x34\xff\xfc\xff\x8c\x00\xcc\xff\x3c\xff\xfc\xff\x88\x00\xcc\xff\x40\xff\xf8\xff\x8c\x00\xd0\xff\x40\xff\xf8\xff\x88\x00\xd0\xff\x40\xff\xf8\xff\x88\x00\xd8\xff\x48\xff\xf8\xff\x88\x00\xe0\xff\x4c\xff\xf0\xff\x84\x00\xe0\xff\x50\xff\xf0\xff\x7c\x00\xe0\xff\x58\xff\xf4\xff\x74\x00\xe0\xff\x64\xff\xf4\xff\x6c\x00\xe4\xff\x6c\xff\xf4\xff\x64\x00\xe0\xff\x74\xff\xf4\xff\x60\x00\xe0\xff\x74\xff\xf8\xff\x64\x00\xe0\xff\x7c\xff\xf8\xff\x5c\x00\xe8\xff\x80\xff\xf8\xff\x64\x00\xec\xff\x84\xff\xf8\xff\x64\x00\xec\xff\x88\xff\xfc\xff\x68\x00\xf4\xff\x88\xff\xfc\xff\x68\x00\xf8\xff\x88\xff\xf8\xff\x6c\x00\xfc\xff\x88\xff\xf8\xff\x74\x00\xfc\xff\x84\xff\xfc\xff\x74\x00\x00\x00\x84\xff\xf8\xff\x78\x00\x04\x00\x80\xff\xf8\xff\x7c\x00\x04\x00\x80\xff\xf8\xff\x80\x00\x04\x00\x7c\xff\xfc\xff\x84\x00\x04\x00\x7c\xff\xf8\xff\x88\x00\x08\x00\x7c\xff\x00\x00\x88\x00\x04\x00\x78\xff\x04\x00\x8c\x00\x04\x00\x78\xff\x04\x00\x90\x00\x04\x00\x78\xff\x00\x00\x90\x00\x04\x00",
			"");

	do_test_nb(
			"\xf0\x3c\x53\xff\x3a\xe8\x30\x41\xa5\xa8\xa4\x1d\x2f\xf2\x03\x60\x35\xc0\x00\x07\xc5\x53\xf4\xbc\x98\x00\x01\x14\x2f\xf0\x00\x0f\x70",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf8\xff\xf8\xff\x10\x00\x00\x00\xe8\xff\xf8\xff\x28\x00\x00\x00\xc8\xff\x00\x00\x38\x00\xf8\xff\xb0\xff\x08\x00\x48\x00\xf0\xff\xa8\xff\x08\x00\x60\x00\xf8\xff\x98\xff\x08\x00\x88\x00\xf0\xff\x88\xff\x10\x00\x88\x00\xe8\xff\x70\xff\x18\x00\x80\x00\xd8\xff\x70\xff\x08\x00\x78\x00\xd8\xff\x88\xff\x08\x00\x78\x00\xe8\xff\x90\xff\x10\x00\x70\x00\xe0\xff\x90\xff\x08\x00\x68\x00\xf8\xff\x88\xff\x00\x00\x80\x00\xf8\xff\x88\xff\x00\x00\x80\x00\xf8\xff\x78\xff\x08\x00\x80\x00\xe8\xff\x78\xff\x08\x00\x78\x00\xe0\xff\x88\xff\x10\x00\x70\x00\xe8\xff\x90\xff\x10\x00\x68\x00\xe0\xff\x98\xff\x08\x00\x70\x00\xe8\xff\xa8\xff\xf8\xff\x70\x00\xf0\xff\x98\xff\xf8\xff\x68\x00\x00\x00\x80\xff\xf8\xff\x68\x00\xf8\xff\x90\xff\x00\x00\x70\x00\xf8\xff\x90\xff\x00\x00\x78\x00\xf0\xff\x88\xff\x00\x00\x80\x00\xf8\xff\x80\xff\x00\x00\x88\x00\xf8\xff\x80\xff\xf8\xff\x88\x00\xf8\xff\x70\xff\x00\x00\x80\x00\xf0\xff\x80\xff\xf8\xff\x80\x00\xe8\xff\x80\xff\xf0\xff\x88\x00",
			"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf8\xff\xe8\xff\xe8\xff\x00\x00\xf8\xff\xf0\xff\x08\x00\x10\x00\x08\x00\x00\x00\xf8\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf8\xff\x00\x00\x00\x00\x00\x00\x00\x00\xf8\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf8\xff\xf8\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc8\xff\xb8\xff\xf8\xff\x18\x00\x20\x00\x10\x00\xf8\xff\xc8\xff\xc8\xff\x08\x00\x30\x00\x28\x00\x10\x00\xf0\xff\xf8\xff\x08\x00\x10\x00\x10\x00\x08\x00\xf8\xff\xf8\xff\xf8\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf8\xff\xf8\xff\x60\x00\x98\x00\x48\x00\xf8\xff\xc8\xff\xd0\xff\x10\x00\x18\x00\xe8\xff\xc8\xff\xa8\xff\xb8\xff\xd0\xff\xe0\xff\xe8\xff\xe8\xff\xe8\xff\xe8\xff\xf8\xff\x00\x00\x10\x00\x18\x00\x10\x00\x08\x00\x00\x00\x00\x00\x08\x00\x08\x00\x60\x00\x90\x00\x38\x00\xe8\xff\xb8\xff\xc0\xff\x00\x00\x08\x00\xe0\xff\xc0\xff\xa8\xff\xa8\xff\xc0\xff\xc8\xff\xd0\xff\x10\xff\x78\xfe\xc8\xfe\x40\xff\xd8\xff\x00\x00\x98\xff\x68\xff\x98\xff\xf8\xff\x80\x00\xc0\x00\xc8\x00\xc0\x00\xc0\x00\xd0\x00\xf0\x00\x08\x01\x18\x01\x00\x01\xd0\x00\xa0\x00\x58\x00\x30\x00\x28\x00\x00\x00\xd0\xff\xa0\xff\x70\xff\x58\xff\x58\xff\x48\xff\x10\x00\xa8\x00\x68\x00\x10\x00",
			"octet-align=1");

	return 0;
}

int get_local_log_level(unsigned int u) {
	return -1;
}
