/* ***********************************************************************
   Open SigComp -- Implementation of RFC 3320 Signaling Compression

   Copyright 2005 Estacado Systems, LLC

   Your use of this code is governed by the license under which it
   has been provided to you. Unless you have a written and signed
   document from Estacado Systems, LLC stating otherwise, your license
   is as provided by the GNU General Public License version 2, a copy
   of which is available in this project in the file named "LICENSE."
   Alternately, a copy of the licence is available by writing to
   the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 021110-1307 USA

   Unless your use of this code is goverened by a written and signed
   contract containing provisions to the contrary, this program is
   distributed WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY of FITNESS FOR A PARTICULAR PURPOSE. See the
   license for additional details.

   To discuss alternate licensing terms, contact info@estacado.net
 *********************************************************************** */

#include "Types.h"
#include "torture_tests.h"

osc::u16 cyclesPerBit = 16;

torture_test_t tortureTest[] =
{
  {
    "A.1.1 Instructions: Bit manipulation #1",

    // Code
    "\xf8\x0a\x71\x16\xa0\x7f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x01\xc0\x00\xff\x80\x55\x55\x02\x20\x2a\x03\x21\x04\x20"
    "\x03\x05\x21\xff\x22\x86\x04\x01\x20\xc0\xff\x02\x20\x60\x03\x20"
    "\x04\x21\x60\x05\x20\x61\x22\x86\x04\x23"
    // Input
    "",
    170,

    // Output
    "\x01\x50\x00\x00\xfe\xbf\x00\x00", 
    8,

    // Expected Cycles
    22,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.2 Instructions: Arithmetic #1",

    // Code
    "\xf8\x0a\xa1\x1c\x01\xa0\x45\x9f\x9f\x07\x22\x01\x16\xa0\x76\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x06\xc0\x00\xff\x99\x41\x07\x20\x01\x08\x20\xa3\xe9\x09"
    "\x20\xa0\x65\x0a\x20\x0b\x22\x86\x04\x06\x21\xc0\xff\x07\x21\x62"
    "\x08\x21\x61\x09\x20\x61\x0a\x21\x62\x22\x86\x04\x23"
    // Input
    "\x00",
    174,

    // Output
    "\x00\x00\x00\x00\x00\x00\x00\x04", 
    8,

    // Expected Cycles
    25,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.2 Instructions: Arithmetic #2",

    // Code
    "\xf8\x0a\xa1\x1c\x01\xa0\x45\x9f\x9f\x07\x22\x01\x16\xa0\x76\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x06\xc0\x00\xff\x99\x41\x07\x20\x01\x08\x20\xa3\xe9\x09"
    "\x20\xa0\x65\x0a\x20\x0b\x22\x86\x04\x06\x21\xc0\xff\x07\x21\x62"
    "\x08\x21\x61\x09\x20\x61\x0a\x21\x62\x22\x86\x04\x23"
    // Input
    "\x01",
    174,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::DIV_BY_ZERO,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.2 Instructions: Arithmetic #3",

    // Code
    "\xf8\x0a\xa1\x1c\x01\xa0\x45\x9f\x9f\x07\x22\x01\x16\xa0\x76\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x06\xc0\x00\xff\x99\x41\x07\x20\x01\x08\x20\xa3\xe9\x09"
    "\x20\xa0\x65\x0a\x20\x0b\x22\x86\x04\x06\x21\xc0\xff\x07\x21\x62"
    "\x08\x21\x61\x09\x20\x61\x0a\x21\x62\x22\x86\x04\x23"
    // Input
    "\x02",
    174,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::DIV_BY_ZERO,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.3 Instructions: Sorting #1",

    // Code
    "\xf8\x0d\xc1\x0c\x88\x02\x17\x0b\x88\x02\x17\x22\xa1\x2e\x2d\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x0a\x00\x0a\x00\x11\x00\x07\x00\x16\x00\x03\x00"
    "\x03\x00\x03\x00\x13\x00\x01\x00\x10\x00\x0e\x00\x08\x00\x02\x00"
    "\x0d\x00\x14\x00\x12\x00\x17\x00\x0f\x00\x15\x00\x0c\x00\x06\x00"
    "\x09\x6e\x67\x20\x69\x75\x69\x20\x74\x69\x74\x2c\x20\x79\x6f\x75"
    "\x27\x20\x53\x46\x6f\x6e\x67\x61\x20\x75\x72\x72\x64\x6f\x20\x74"
    "\x6f\x6e\x2e\x2e\x00\x70\x65\x70\x20\x6e\x74\x72\x65\x6e\x69"
    // Input
    "",
    223,

    // Output
    "\x46\x6f\x72\x64\x2c\x20\x79\x6f\x75\x27\x72\x65\x20\x74\x75\x72"
    "\x6e\x69\x6e\x67\x20\x69\x6e\x74\x6f\x20\x61\x20\x70\x65\x6e\x67"
    "\x75\x69\x6e\x2e\x20\x53\x74\x6f\x70\x20\x69\x74\x2e", 
    45,

    // Expected Cycles
    371,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.4 Instructions: SHA-1 #1",

    // Code
    "\xf8\x08\x71\x0d\xa0\xc3\x03\xa0\x44\x22\xa0\x44\x14\x0d\xa0\xc6"
    "\x38\xa0\x44\x22\xa0\x44\x14\x0e\x86\xa0\xfe\x0e\xa0\x42\xa0\xff"
    "\x0d\xa0\xfe\x8e\xa0\x44\x22\xa0\x44\x14\x0e\x86\xa0\xff\x0e\xa0"
    "\x42\xa1\x07\x0d\xa0\xff\xa2\x80\xa0\xff\x22\xa0\xff\x14\x23\x00"
    "\x00\x00\x00\x00\x00\x00\x61\x62\x63\x61\x62\x63\x64\x62\x63\x64"
    "\x65\x63\x64\x65\x66\x64\x65\x66\x67\x65\x66\x67\x68\x66\x67\x68"
    "\x69\x67\x68\x69\x6a\x68\x69\x6a\x6b\x69\x6a\x6b\x6c\x6a\x6b\x6c"
    "\x6d\x6b\x6c\x6d\x6e\x6c\x6d\x6e\x6f\x6d\x6e\x6f\x70\x6e\x6f\x70"
    "\x71\x61\x30\x31\x32\x33\x34\x35\x36\x37"
    // Input
    "",
    138,

    // Output
    "\xa9\x99\x3e\x36\x47\x06\x81\x6a\xba\x3e\x25\x71\x78\x50\xc2\x6c"
    "\x9c\xd0\xd8\x9d\x84\x98\x3e\x44\x1c\x3b\xd2\x6e\xba\xae\x4a\xa1"
    "\xf9\x51\x29\xe5\xe5\x46\x70\xf1\x12\xff\x34\x7b\x4f\x27\xd6\x9e"
    "\x1f\x32\x8e\x6f\x4b\x55\x73\xe3\x66\x6e\x12\x2f\x4f\x46\x04\x52"
    "\xeb\xb5\x63\x93\x4f\x46\x04\x52\xeb\xb5\x63\x93\x4f\x46\x04\x52"
    "",
    80,

    // Expected Cycles
    17176,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.5 Instructions: LOAD and MULTILOAD #1",

    // Code
    "\xf8\x03\x71\x0e\x87\xa0\x84\x0e\xa0\x82\xc0\x80\x0e\xc0\x80\xa0"
    "\x86\x0e\xc0\x84\xc0\x84\x22\x87\x08\x1c\x01\xa0\x41\x9f\x89\x08"
    "\x20\x02\x06\x20\x3c\x0f\x60\x03\xa0\xa3\xa0\xb2\x87\x0f\x60\x04"
    "\x2a\x87\xc0\x80\xc0\x84\x22\x87\x08\x23"
    // Input
    "\x00",
    59,

    // Output
    "\x00\x84\x00\x84\x00\x86\x00\x86\x00\x2a\x00\x80\x00\x2a\x00\x2a"
    "", 
    16,

    // Expected Cycles
    36,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.5 Instructions: LOAD and MULTILOAD #2",

    // Code
    "\xf8\x03\x71\x0e\x87\xa0\x84\x0e\xa0\x82\xc0\x80\x0e\xc0\x80\xa0"
    "\x86\x0e\xc0\x84\xc0\x84\x22\x87\x08\x1c\x01\xa0\x41\x9f\x89\x08"
    "\x20\x02\x06\x20\x3c\x0f\x60\x03\xa0\xa3\xa0\xb2\x87\x0f\x60\x04"
    "\x2a\x87\xc0\x80\xc0\x84\x22\x87\x08\x23"
    // Input
    "\x01",
    59,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MULTILOAD_OVERWRITTEN,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.5 Instructions: LOAD and MULTILOAD #3",

    // Code
    "\xf8\x03\x71\x0e\x87\xa0\x84\x0e\xa0\x82\xc0\x80\x0e\xc0\x80\xa0"
    "\x86\x0e\xc0\x84\xc0\x84\x22\x87\x08\x1c\x01\xa0\x41\x9f\x89\x08"
    "\x20\x02\x06\x20\x3c\x0f\x60\x03\xa0\xa3\xa0\xb2\x87\x0f\x60\x04"
    "\x2a\x87\xc0\x80\xc0\x84\x22\x87\x08\x23"
    // Input
    "\x02",
    59,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MULTILOAD_OVERWRITTEN,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.6 Instructions: COPY #1",

    // Code
    "\xf8\x03\x91\x0e\x20\x8e\x0e\x86\x86\x0e\xa0\x42\x87\x12\x20\x87"
    "\x21\x0e\x86\x80\x41\x00\x12\x86\xa0\x55\xa0\x41\x22\x20\xa0\x77"
    "\x0e\x86\x20\x0e\xa0\x42\x30\x15\x20\x04\xa0\x41\x01\x12\x20\x04"
    "\x30\x22\x30\x04\x12\x30\x04\x2e\x22\x20\x02\x23"
    // Input
    "",
    60,

    // Output
    "\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40"
    "\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40\x40"
    "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41"
    "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41"
    "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41"
    "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41"
    "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41"
    "\x41\x41\x41\x41\x41\x41\x55\x41\x42\x43\x44\x43\x44", 
    125,

    // Expected Cycles
    365,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.7 Instructions: COPY-LITERAL and COPY-OFFSET #1",

    // Code
    "\xf8\x06\x11\x0e\x20\x80\x41\x00\x0e\x86\x86\x0e\xa0\x42\x87\x0e"
    "\xa0\x44\x21\x13\x20\x87\x22\x22\x86\x08\x0e\xa0\x44\xa0\x9c\x13"
    "\x20\x02\x22\x22\xa0\x9c\x02\x0e\x86\xa0\x48\x0e\xa0\x42\xa0\x52"
    "\x0e\xa0\x44\xa0\x52\x15\xa0\x48\x0a\xa0\x41\x01\x14\x02\x06\x22"
    "\x0e\xa0\x46\x06\x14\x63\x04\x22\x22\x61\x0a\x0e\xa0\x44\xa0\x50"
    "\x14\x04\x04\x22\x22\xa0\x44\x02\x14\x05\x04\x22\x22\xa0\x44\x02"
    "\x22\x60\x0a\x23"
    // Input
    "",
    100,

    // Output
    "\x41\x41\x41\x41\x00\x61\x41\x41\x41\x41\x49\x4A\x41\x42\x43\x44"
    "\x49\x4A\x41\x42\x00\x4A\x00\x4E\x47\x48\x48\x45\x46\x47\x47\x48"
    "\x45\x46", 
    34,

    // Expected Cycles
    216,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.8 Instructions: MEMSET #1",

    // Code
    "\xf8\x01\x81\x0e\x86\x87\x0e\xa0\x42\xa0\x81\x15\x86\xa0\x81\x00"
    "\x01\x15\xa0\x81\x0f\x86\x0f\x22\x87\x10\x23"
    // Input
    "",
    27,

    // Output
    "\x80\x40\x4f\x5e\x6d\x7c\x8b\x9a\xa9\xb8\xc7\xd6\xe5\xf4\x03\x12"
    "", 
    16,

    // Expected Cycles
    166,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.9 Instructions: CRC #1",

    // Code
    "\xf8\x01\xa1\x15\xa0\x46\x18\x01\x01\x15\xa0\x5e\x14\x87\x01\x1c"
    "\x02\xa0\x44\x9f\x93\x1b\x62\xa0\x46\x2c\x9f\x8d\x23"
    // Input
    "\x62\xcb",
    31,

    // Output
    "", 
    0,

    // Expected Cycles
    95,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.9 Instructions: CRC #2",

    // Code
    "\xf8\x01\xa1\x15\xa0\x46\x18\x01\x01\x15\xa0\x5e\x14\x87\x01\x1c"
    "\x02\xa0\x44\x9f\x93\x1b\x62\xa0\x46\x2c\x9f\x8d\x23"
    // Input
    "\xab\xcd",
    31,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::USER_REQUESTED,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.10 Instructions: INPUT-BITS #1",

    // Code
    "\xf8\x01\x51\x1d\x62\xa0\x46\x14\x22\xa0\x46\x02\x06\x22\x01\x0a"
    "\x22\x07\x06\x22\x01\x16\xee\x23"
    // Input
    "\x93\x2e\xac\x71",
    28,

    // Output
    "\x00\x00\x00\x02\x00\x02\x00\x13\x00\x00\x00\x03\x00\x1a\x00\x38"
    "", 
    16,

    // Expected Cycles
    66,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.11 Instructions: INPUT-HUFFMAN #1",

    // Code
    "\xf8\x01\xd1\x1e\xa0\x46\x1c\x02\x62\x00\x62\x62\x62\x00\xff\x00"
    "\x22\xa0\x46\x02\x06\x22\x01\x0a\x22\x07\x06\x22\x01\x16\xe6\x23"
    ""
    // Input
    "\x93\x2e\xac\x71\x66\xd8\x6f",
    39,

    // Output
    "\x00\x00\x00\x03\x00\x08\x04\xd7\x00\x02\x00\x03\x03\x99\x30\xfe"
    "", 
    16,

    // Expected Cycles
    84,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.12 Instructions: INPUT-BYTES #1",

    // Code
    "\xf8\x02\x71\x0e\x86\xa0\x48\x0e\xa0\x42\xa0\x4c\x1d\x62\xa0\x46"
    "\x1d\x22\xa0\x46\x02\x06\x22\x02\x0a\x22\x07\x1c\x62\xa0\x48\x0e"
    "\x22\xa0\x48\x62\x06\x22\x01\x16\xe5\x23"
    // Input
    "\x93\x2e\xac\x71\x66\xd8\x6f\xb1\x59\x2b\xdc\x9a\x97\x34\xd8\x47"
    "\xa7\x33\x87\x4e\x1b\xcb\xcd\x51\xb5\xdc\x96\x59\x9d\x6a",
    72,

    // Output
    "\x00\x00\x93\x2e\x00\x01\xb1\x66\xd8\x6f\xb1\x00\x1a\x2b\x00\x03"
    "\x9a\x97\x34\xd8\x00\x07\x00\x01\x33\x87\x4e\x00\x08\xdc\x96\x51"
    "\xb5\xdc\x96\x00\x59\x9d\x6a", 
    39,

    // Expected Cycles
    130,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.13 Instructions: Stack Manipulation #1",

    // Code
    "\xf8\x14\x11\x0e\xa0\x46\x86\x10\x02\x10\x60\x10\xa0\x42\x22\x86"
    "\x08\x11\x86\x11\x63\x11\xa0\x46\x22\x86\x08\x16\x28\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x0e\xa0\x46\x20\x0e\xa0\x48\xa1\x40\x07\x24\x88\x18"
    "\x34\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x18\x64\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x0e\xa0\x46\xa1\x7f\x0e\xa1\x7f\x1a\x0f\xa1\xb0\x03"
    "\x01\x80\xc0\x01\x8f\x19\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x23"
    // Input
    "",
    324,

    // Output
    "\x00\x03\x00\x02\x00\x01\x00\x42\x00\x42\x00\x00\x00\x01\x00\x01"
    "", 
    16,

    // Expected Cycles
    40,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.14 Instructions: Program Flow #1",

    // Code
    "\xf8\x03\xf1\x0e\xa0\x44\x04\x0e\x86\xa0\x92\x07\x20\xa0\x90\x22"
    "\xa0\x43\x01\x16\x60\x06\x21\x01\x0e\x86\xa0\x84\x07\x20\xa0\xa1"
    "\x22\xa0\x43\x01\x17\x61\x06\x60\xf1\x06\x07\x22\x01\x0e\x86\xa0"
    "\x84\x07\x20\xa0\xb6\x22\xa0\x43\x01\x1a\x04\x62\x08\x60\x9f\xdc"
    "\xf1\x23"
    // Input
    "",
    66,

    // Output
    "\x00\x01\x01\x02\x02\x03\x03\x04\x04\x05\x05\x06\x07\x07\x07\x08"
    "\x08\x08\x09\x09", 
    20,

    // Expected Cycles
    131,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #1",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x01",
    152,

    // Output
    "", 
    0,

    // Expected Cycles
    23,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #2",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x02",
    152,

    // Output
    "", 
    0,

    // Expected Cycles
    14,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #3",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x03",
    152,

    // Output
    "", 
    0,

    // Expected Cycles
    24,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #4",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x04\x05",
    153,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::INVALID_STATE_ID_LENGTH,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #5",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x04\x15",
    153,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::INVALID_STATE_ID_LENGTH,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #6",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x04\x06",
    153,

    // Output
    "", 
    0,

    // Expected Cycles
    23,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #7",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x09",
    152,

    // Output
    "", 
    0,

    // Expected Cycles
    34,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #8",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x1e\x06",
    153,

    // Output
    "", 
    0,

    // Expected Cycles
    46,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #9",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x1e\x07",
    153,

    // Output
    "", 
    0,

    // Expected Cycles
    47,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.15 Instructions: State creation #10",

    // Code
    "\xf8\x09\x41\x1c\x01\xa0\x45\x9f\x7f\x04\x22\x0b\x17\x62\x8f\x0d"
    "\x06\x06\x20\x0a\xa1\x0a\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x0c"
    "\x06\x06\x20\x0a\x88\x00\x14\x00\x04\x22\x01\x17\x62\x8f\x16\x06"
    "\x06\x1c\x01\xa0\x47\x9f\x51\x21\xa0\x48\x63\x12\xa0\xe4\x63\xa0"
    "\x48\x04\x22\x01\x17\x62\x8f\x0a\x06\x06\x21\xa0\xe4\x06\x04\x22"
    "\x01\x17\x62\x8f\x0e\x06\x06\x23\x00\x00\x0a\x88\x00\x14\x00\x23"
    "\x00\x00\x00\x00\x00\x00\x00\x43\x7a\xe8\x0a\x0f\xdc\x1e\x6a\x87"
    "\xc1\xb6\x2a\x76\x76\xb9\x73\x31\x8c\x0e\xf5\x00\x00\x00\x00\x00"
    "\x00\x00\x00\xc0\xcc\x3f\xee\x79\xbc\xfc\x8f\xd1\x08\x65\xe8\x03"
    "\x52\xee\x29\x77\x17\xdf\x57"
    // Input
    "\x1e\x14",
    153,

    // Output
    "", 
    0,

    // Expected Cycles
    60,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.16 Instructions: [1: setup] STATE-ACCESS #1",

    // Code
    "\xf8\x19\x01\x23\x00\x00\x10\x89\x00\x14\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x22\xa2\x0c\x04\x23\x00\x00\x00\x00\x00\x00\x00\x74"
    "\x65\x73\x74"
    // Input
    "",
    403,

    // Output
    "", 
    0,

    // Expected Cycles
    17,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.16 Instructions: [2: test] STATE-ACCESS #1",

    // Code
    "\xf8\x19\x41\x1c\x01\xa0\x45\x9f\x9f\x17\x62\x01\x06\x0d\x1c\x1f"
    "\x89\x14\x00\x00\x00\x89\x1f\x89\x14\x0c\x04\xa0\x46\x00\x22\xa0"
    "\x46\x04\x16\xa1\x45\x17\x62\x03\x06\x10\x1b\x1f\x87\x14\x00\x00"
    "\x00\x00\x16\xa1\x35\x1f\x89\x13\x06\x04\xa0\x46\x00\x16\xa1\x2a"
    "\x1f\x89\x14\x0c\x05\xa0\x46\x00\x16\xa1\x1f\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x5d\xf8\xbc\x3e\x20\x93\xb5\xab\xe1\xf1\x70\x13\x42"
    "\x4c\xe7\xfe\x05\xe0\x69\x39"
    // Input
    "\x00",
    408,

    // Output
    "\x74\x65\x73\x74", 
    4,

    // Expected Cycles
    26,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.16 Instructions: [2: test] STATE-ACCESS #2",

    // Code
    "\xf8\x19\x41\x1c\x01\xa0\x45\x9f\x9f\x17\x62\x01\x06\x0d\x1c\x1f"
    "\x89\x14\x00\x00\x00\x89\x1f\x89\x14\x0c\x04\xa0\x46\x00\x22\xa0"
    "\x46\x04\x16\xa1\x45\x17\x62\x03\x06\x10\x1b\x1f\x87\x14\x00\x00"
    "\x00\x00\x16\xa1\x35\x1f\x89\x13\x06\x04\xa0\x46\x00\x16\xa1\x2a"
    "\x1f\x89\x14\x0c\x05\xa0\x46\x00\x16\xa1\x1f\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x5d\xf8\xbc\x3e\x20\x93\xb5\xab\xe1\xf1\x70\x13\x42"
    "\x4c\xe7\xfe\x05\xe0\x69\x39"
    // Input
    "\x01",
    408,

    // Output
    "\x74\x65\x73\x74", 
    4,

    // Expected Cycles
    15,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.16 Instructions: [2: test] STATE-ACCESS #3",

    // Code
    "\xf8\x19\x41\x1c\x01\xa0\x45\x9f\x9f\x17\x62\x01\x06\x0d\x1c\x1f"
    "\x89\x14\x00\x00\x00\x89\x1f\x89\x14\x0c\x04\xa0\x46\x00\x22\xa0"
    "\x46\x04\x16\xa1\x45\x17\x62\x03\x06\x10\x1b\x1f\x87\x14\x00\x00"
    "\x00\x00\x16\xa1\x35\x1f\x89\x13\x06\x04\xa0\x46\x00\x16\xa1\x2a"
    "\x1f\x89\x14\x0c\x05\xa0\x46\x00\x16\xa1\x1f\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x5d\xf8\xbc\x3e\x20\x93\xb5\xab\xe1\xf1\x70\x13\x42"
    "\x4c\xe7\xfe\x05\xe0\x69\x39"
    // Input
    "\x02",
    408,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_NOT_FOUND,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.16 Instructions: [2: test] STATE-ACCESS #4",

    // Code
    "\xf8\x19\x41\x1c\x01\xa0\x45\x9f\x9f\x17\x62\x01\x06\x0d\x1c\x1f"
    "\x89\x14\x00\x00\x00\x89\x1f\x89\x14\x0c\x04\xa0\x46\x00\x22\xa0"
    "\x46\x04\x16\xa1\x45\x17\x62\x03\x06\x10\x1b\x1f\x87\x14\x00\x00"
    "\x00\x00\x16\xa1\x35\x1f\x89\x13\x06\x04\xa0\x46\x00\x16\xa1\x2a"
    "\x1f\x89\x14\x0c\x05\xa0\x46\x00\x16\xa1\x1f\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x5d\xf8\xbc\x3e\x20\x93\xb5\xab\xe1\xf1\x70\x13\x42"
    "\x4c\xe7\xfe\x05\xe0\x69\x39"
    // Input
    "\x03",
    408,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_NOT_FOUND,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.1.16 Instructions: [2: test] STATE-ACCESS #5",

    // Code
    "\xf8\x19\x41\x1c\x01\xa0\x45\x9f\x9f\x17\x62\x01\x06\x0d\x1c\x1f"
    "\x89\x14\x00\x00\x00\x89\x1f\x89\x14\x0c\x04\xa0\x46\x00\x22\xa0"
    "\x46\x04\x16\xa1\x45\x17\x62\x03\x06\x10\x1b\x1f\x87\x14\x00\x00"
    "\x00\x00\x16\xa1\x35\x1f\x89\x13\x06\x04\xa0\x46\x00\x16\xa1\x2a"
    "\x1f\x89\x14\x0c\x05\xa0\x46\x00\x16\xa1\x1f\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x5d\xf8\xbc\x3e\x20\x93\xb5\xab\xe1\xf1\x70\x13\x42"
    "\x4c\xe7\xfe\x05\xe0\x69\x39"
    // Input
    "\x04",
    408,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_TOO_SHORT,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.1 Dispatcher tests: Useful Values #1",
    // Code
    "\xf8\x05\xa1\x0e\x86\x20\x0e\xa0\x42\x21\x17\x42\x02"
    "\xf8\x06\xf8\x17\x43\x00\xf2\x06\x0c\x17\x44\x00\xec"
    "\x3f\xec\x17\x44\xa3\xc0\xe6\x07\xe6\x1c\x01\xa0\x47"
    "\x9f\xdf\x06\x23\x40\x07\x23\x01\x12\x20\x01\x63\x1c"
    "\x01\xa0\x49\x9f\xcf\x0e\xa0\x44\x43\x06\x22\x03\x08"
    "\x22\x08\x06\x22\xa3\xe8\x08\x22\x41\x06\x22\x64\x07"
    "\x22\xa3\xd6\x12\x20\x62\x20\x23\x00\x00\xa3\xc0\x86"
    "\x87\x06"
    // Input
    "",
    93,

    // Output
    "", 
    0,

    // Expected Cycles
    966,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.1 Dispatcher tests: Useful Values #2",

    // Code
    "\xf9\x46\x84\x75\x55\xD9\x2D"
    // Input
    "\x00\x00",
    9,

    // Output
    "", 
    0,

    // Expected Cycles
    cyclesPerBit * 1072,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.1 Dispatcher tests: Useful Values #3",

    // Code
    "\xf9\x46\x84\x75\x55\xD9\x2D"
    // Input
    "\x00\x01",
    9,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::CYCLES_EXHAUSTED,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.1 Dispatcher tests: Useful Values #4",

    // Code
    "\xf9\x46\x84\x75\x55\xD9\x2D"
    // Input
    "\x01\x00",
    9,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::SEGFAULT,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.2 Dispatcher tests: Cycles Checking #1",

    // Code
    "\xf8\x01\xa1\x0f\x86\x04\x20\x29\x00\x22\x12\xa0\x44\x02\x60\x14"
    "\x02\xa0\x64\x23\x22\xa0\x44\x02\x06\x22\x01\x16\xef"
    // Input
    "",
    29,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::CYCLES_EXHAUSTED,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.3 Dispatcher tests: Message-based transport #1",

    // Code
    "\xf8"
    // Input
    "",
    1,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MESSAGE_TOO_SHORT,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.3 Dispatcher tests: Message-based transport #2",

    // Code
    "\xf8\x00"
    // Input
    "",
    2,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MESSAGE_TOO_SHORT,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.3 Dispatcher tests: Message-based transport #3",

    // Code
    "\xf8\x00\xe1\x06\x00\x11\x22\x00\x02\x23\x00\x00\x00\x00\x00\x00\x01"
    // Input
    "",
    17,

    // Output
    "\x40\x00", 
    2,

    // Expected Cycles
    5,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.3 Dispatcher tests: Message-based transport #4",

    // Code
    "\xf8\x00\xf1\x06\x00\x11\x22\x00\x02\x23\x00\x00\x00\x00\x00\x00\x01"
    // Input
    "",
    17,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MESSAGE_TOO_SHORT,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.3 Dispatcher tests: Message-based transport #5",

    // Code
    "\xf8\x00\xe0\x06\x00\x11\x22\x00\x02\x23\x00\x00\x00\x00\x00\x00\x01"
    // Input
    "",
    17,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::INVALID_CODE_LOCATION,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.3 Dispatcher tests: Message-based transport #6",

    // Code
    "\xf8\x00\xee\x06\x00\x11\x22\x00\x02\x23\x00\x00\x00\x00\x00\x00\x01"
    // Input
    "",
    17,

    // Output
    "\x40\x00", 
    2,

    // Expected Cycles
    5,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.4 Dispatcher tests: Stream-based transport #1",

    // Two Messages
    "\xff\xff\xf8\x01\x71\x08\x00\x02\x22\x00\x02\x22\xa0\x92\x05\x23"
    "\x00\x00\x00\x00\x00\x00\x00\xff\x00\xff\x03\xff\xff\xff\xff\xff"
    "\xff\xff\xf8\x01\x7e\x08\x00\x02\x22\x00\x02\x22\xa3\xd2\x05\x23"
    "\x00\x00\x00\x00\x00\x00\x00\xff\x04\xff\xff\xff\xff\xff\xff\xff"
    "\xff\xff\xff",
    67,

    // Output
    "\x40\x00\xff\xff\xff\xff\xff", 7, // 0x4000 = Decompression memory size

    // Expected Cycles
    11,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  {
    "A.2.4 Dispatcher tests: Stream-based transport #2",

    // Use input from previous stream
    "", 0,

    // Output
    "\x40\x00\xff\xff\xff\xff\xff", 7, // 0x4000 = Decompression memory size

    // Expected Cycles
    11,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  {
    "A.2.4 Dispatcher tests: Stream-based transport #3",

    // Code
    "\xf8\xff\xff", 3,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MESSAGE_TOO_SHORT,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  {
    "A.2.4 Dispatcher tests: Stream-based transport #4",

    // Code
    "\xf8\x00\xff\xff", 4,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MESSAGE_TOO_SHORT,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  {
    "A.2.4 Dispatcher tests: Stream-based transport #5",

    // Input
    "\xf8\x01\x81\x08\x00\x02\x22\x00\x02\x22\xa0\x92\x05\x23\xff\xff"
    "\x00\x00\x00\x00\x00\x00\x00\xff\x00\xff\x03\xff\xff\xff",
    30,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::MESSAGE_TOO_SHORT,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  // This test case is inserted to clear out the junk left in the TCP
  // stream by the previous test case.
  {
    "A.2.4 Dispatcher tests: Stream-based transport #5a (cleanup)",

    // Input
    "\xff\xff", 2,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::INTERNAL_ERROR,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  {
    "A.2.4 Dispatcher tests: Stream-based transport #6",

    // Input
    "\xf8\x01\x70\x08\x00\x02\x22\x00\x02\x22\xa0\x92\x05\x23\xff\xff"
    "\x00\x00\x00\x00\x00\x00\x00\xff\x04\xff\xff\xff\xff",
    29,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::INVALID_CODE_LOCATION,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  // This test case is inserted to clear out the junk left in the TCP
  // stream by the previous test case.
  {
    "A.2.4 Dispatcher tests: Stream-based transport #6a (cleanup)",

    // Input
    "\xff\xff", 2,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::INTERNAL_ERROR,

    // Compartment ID
    0,

    // Input stream based?
    true
  },
  {
    "A.2.5 Dispatcher tests: Input past the end of a message #1",

    // Code
    "\xf8\x03\x81\x0e\x86\xa0\x46\x0e\xa0\x42\xa0\x4d\x1d\x09\xa0\x46"
    "\x9f\x96\x1c\x07\xa0\x46\x07\x16\x23\x1d\x07\xa0\x46\x9f\x89\x1c"
    "\x02\xa0\x46\x9f\x83\x22\xa0\x46\x02\x1d\x10\xa0\x46\x07\x16\x0c"
    "\x1d\x08\xa0\x46\x9f\x72\x22\xa0\x47\x01\x23"
    // Input
    "\xff\xfa\x00\x68\x69\x21",
    65,

    // Output
    "\x68\x69\x21", 
    3,

    // Expected Cycles
    23,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.2.5 Dispatcher tests: Input past the end of a message #2",

    // Code
    "\xf8\x03\x81\x0e\x86\xa0\x46\x0e\xa0\x42\xa0\x4d\x1d\x09\xa0\x46"
    "\x9f\x96\x1c\x07\xa0\x46\x07\x16\x23\x1d\x07\xa0\x46\x9f\x89\x1c"
    "\x02\xa0\x46\x9f\x83\x22\xa0\x46\x02\x1d\x10\xa0\x46\x07\x16\x0c"
    "\x1d\x08\xa0\x46\x9f\x72\x22\xa0\x47\x01\x23"
    // Input
    "\xff\xfa\x00\x68\x69",
    64,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::USER_REQUESTED,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.1 State handler tests: SigComp feedback mechanism #1",

    // Code
    "\xf8\x05\x03\x1c\x01\xa0\x41\x9f\x1f\x17\x60\x01\x07\x0e\x9f\x19"
    "\x0e\xa0\x42\xa4\x7f\x16\x0e\x0e\xa0\x42\xa4\xff\x15\xa0\x44\xa0"
    "\x7f\x01\x01\x0e\xa0\xc3\xa8\x01\x0e\xa0\xc5\xa6\x00\x0e\xa0\xcc"
    "\xac\x00\x0e\xa0\xd9\xb4\x00\x0e\xa0\xee\xb5\x00\x15\xa0\xc6\x06"
    "\x00\x01\x15\xa0\xcd\x0c\x00\x01\x15\xa0\xda\x14\x00\x01\x23\xa0"
    "\x42\xa0\xc3"
    // Input
    "\x00",
    84,

    // Output
    "", 
    0,

    // Expected Cycles
    52,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.1 State handler tests: SigComp feedback mechanism #2",

    // Code
    "\xf8\x05\x03\x1c\x01\xa0\x41\x9f\x1f\x17\x60\x01\x07\x0e\x9f\x19"
    "\x0e\xa0\x42\xa4\x7f\x16\x0e\x0e\xa0\x42\xa4\xff\x15\xa0\x44\xa0"
    "\x7f\x01\x01\x0e\xa0\xc3\xa8\x01\x0e\xa0\xc5\xa6\x00\x0e\xa0\xcc"
    "\xac\x00\x0e\xa0\xd9\xb4\x00\x0e\xa0\xee\xb5\x00\x15\xa0\xc6\x06"
    "\x00\x01\x15\xa0\xcd\x0c\x00\x01\x15\xa0\xda\x14\x00\x01\x23\xa0"
    "\x42\xa0\xc3"
    // Input
    "\x01",
    84,

    // Output
    "", 
    0,

    // Expected Cycles
    179,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.2 State handler tests: State memory management #1",

    // Code
    "\xf8\x1b\xa1\x0f\x86\x02\x89\xa2\x04\x1c\x01\xa0\x47\x9f\x99\x17"
    "\x63\x05\x08\xa0\x68\xa0\x70\x17\x63\x03\x07\x34\xa0\x56\x08\x23"
    "\x06\x06\x23\xa2\x04\x0e\xa0\x44\x63\x06\x23\x06\x12\x62\x02\xa0"
    "\x4a\x17\x62\x63\x08\xa0\x58\x9f\x71\x0e\xa0\x48\x65\x08\x24\x88"
    "\x20\x64\x89\x00\x06\x65\x06\x22\x02\x16\xe3\x1f\xa2\x16\x06\x00"
    "\x00\x00\x00\x1f\xa2\x1c\x06\x00\x00\x00\x00\x1f\xa2\x22\x06\x00"
    "\x00\x00\x00\x1f\xa2\x2e\x06\x00\x00\x00\x00\x16\x1e\x1f\xa2\x28"
    "\x06\x00\x00\x00\x00\x16\x14\x20\x8b\x89\x00\x06\x00\x16\x0c\x1f"
    "\xa2\x34\x06\x00\x00\x00\x00\x16\x02\x23\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x74\x65\x73\x74\x00\x00\x00\x01\x00\x02\x00\x03\x00"
    "\x04\x00\x03\x00\x02\x00\x01\x00\x00\x8e\xea\x4b\x43\xa7\x87\xf9"
    "\x01\x0e\xef\x56\x7b\x23\x9a\x34\x6b\x15\xa6\xb4\x0f\xc0\xe4\x4d"
    "\x2c\xd4\xa2\x21\x47\xe6\x0a\xef\xf2\xbc\x0f\xb6\xaf"
    // Input
    "\x00",
    446,

    // Output
    "", 
    0,

    // Expected Cycles
    811,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.2 State handler tests: State memory management #2",

    // Code
    "\xf8\x1b\xa1\x0f\x86\x02\x89\xa2\x04\x1c\x01\xa0\x47\x9f\x99\x17"
    "\x63\x05\x08\xa0\x68\xa0\x70\x17\x63\x03\x07\x34\xa0\x56\x08\x23"
    "\x06\x06\x23\xa2\x04\x0e\xa0\x44\x63\x06\x23\x06\x12\x62\x02\xa0"
    "\x4a\x17\x62\x63\x08\xa0\x58\x9f\x71\x0e\xa0\x48\x65\x08\x24\x88"
    "\x20\x64\x89\x00\x06\x65\x06\x22\x02\x16\xe3\x1f\xa2\x16\x06\x00"
    "\x00\x00\x00\x1f\xa2\x1c\x06\x00\x00\x00\x00\x1f\xa2\x22\x06\x00"
    "\x00\x00\x00\x1f\xa2\x2e\x06\x00\x00\x00\x00\x16\x1e\x1f\xa2\x28"
    "\x06\x00\x00\x00\x00\x16\x14\x20\x8b\x89\x00\x06\x00\x16\x0c\x1f"
    "\xa2\x34\x06\x00\x00\x00\x00\x16\x02\x23\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x74\x65\x73\x74\x00\x00\x00\x01\x00\x02\x00\x03\x00"
    "\x04\x00\x03\x00\x02\x00\x01\x00\x00\x8e\xea\x4b\x43\xa7\x87\xf9"
    "\x01\x0e\xef\x56\x7b\x23\x9a\x34\x6b\x15\xa6\xb4\x0f\xc0\xe4\x4d"
    "\x2c\xd4\xa2\x21\x47\xe6\x0a\xef\xf2\xbc\x0f\xb6\xaf"
    // Input
    "\x01",
    446,

    // Output
    "", 
    0,

    // Expected Cycles
    2603,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.2 State handler tests: State memory management #3",

    // Code
    "\xf8\x1b\xa1\x0f\x86\x02\x89\xa2\x04\x1c\x01\xa0\x47\x9f\x99\x17"
    "\x63\x05\x08\xa0\x68\xa0\x70\x17\x63\x03\x07\x34\xa0\x56\x08\x23"
    "\x06\x06\x23\xa2\x04\x0e\xa0\x44\x63\x06\x23\x06\x12\x62\x02\xa0"
    "\x4a\x17\x62\x63\x08\xa0\x58\x9f\x71\x0e\xa0\x48\x65\x08\x24\x88"
    "\x20\x64\x89\x00\x06\x65\x06\x22\x02\x16\xe3\x1f\xa2\x16\x06\x00"
    "\x00\x00\x00\x1f\xa2\x1c\x06\x00\x00\x00\x00\x1f\xa2\x22\x06\x00"
    "\x00\x00\x00\x1f\xa2\x2e\x06\x00\x00\x00\x00\x16\x1e\x1f\xa2\x28"
    "\x06\x00\x00\x00\x00\x16\x14\x20\x8b\x89\x00\x06\x00\x16\x0c\x1f"
    "\xa2\x34\x06\x00\x00\x00\x00\x16\x02\x23\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x74\x65\x73\x74\x00\x00\x00\x01\x00\x02\x00\x03\x00"
    "\x04\x00\x03\x00\x02\x00\x01\x00\x00\x8e\xea\x4b\x43\xa7\x87\xf9"
    "\x01\x0e\xef\x56\x7b\x23\x9a\x34\x6b\x15\xa6\xb4\x0f\xc0\xe4\x4d"
    "\x2c\xd4\xa2\x21\x47\xe6\x0a\xef\xf2\xbc\x0f\xb6\xaf"
    // Input
    "\x02",
    446,

    // Output
    "", 
    0,

    // Expected Cycles
    811,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.2 State handler tests: State memory management #4",

    // Code
    "\xf8\x1b\xa1\x0f\x86\x02\x89\xa2\x04\x1c\x01\xa0\x47\x9f\x99\x17"
    "\x63\x05\x08\xa0\x68\xa0\x70\x17\x63\x03\x07\x34\xa0\x56\x08\x23"
    "\x06\x06\x23\xa2\x04\x0e\xa0\x44\x63\x06\x23\x06\x12\x62\x02\xa0"
    "\x4a\x17\x62\x63\x08\xa0\x58\x9f\x71\x0e\xa0\x48\x65\x08\x24\x88"
    "\x20\x64\x89\x00\x06\x65\x06\x22\x02\x16\xe3\x1f\xa2\x16\x06\x00"
    "\x00\x00\x00\x1f\xa2\x1c\x06\x00\x00\x00\x00\x1f\xa2\x22\x06\x00"
    "\x00\x00\x00\x1f\xa2\x2e\x06\x00\x00\x00\x00\x16\x1e\x1f\xa2\x28"
    "\x06\x00\x00\x00\x00\x16\x14\x20\x8b\x89\x00\x06\x00\x16\x0c\x1f"
    "\xa2\x34\x06\x00\x00\x00\x00\x16\x02\x23\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x74\x65\x73\x74\x00\x00\x00\x01\x00\x02\x00\x03\x00"
    "\x04\x00\x03\x00\x02\x00\x01\x00\x00\x8e\xea\x4b\x43\xa7\x87\xf9"
    "\x01\x0e\xef\x56\x7b\x23\x9a\x34\x6b\x15\xa6\xb4\x0f\xc0\xe4\x4d"
    "\x2c\xd4\xa2\x21\x47\xe6\x0a\xef\xf2\xbc\x0f\xb6\xaf"
    // Input
    "\x03",
    446,

    // Output
    "", 
    0,

    // Expected Cycles
    1805,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.2 State handler tests: State memory management #5",

    // Code
    "\xf8\x1b\xa1\x0f\x86\x02\x89\xa2\x04\x1c\x01\xa0\x47\x9f\x99\x17"
    "\x63\x05\x08\xa0\x68\xa0\x70\x17\x63\x03\x07\x34\xa0\x56\x08\x23"
    "\x06\x06\x23\xa2\x04\x0e\xa0\x44\x63\x06\x23\x06\x12\x62\x02\xa0"
    "\x4a\x17\x62\x63\x08\xa0\x58\x9f\x71\x0e\xa0\x48\x65\x08\x24\x88"
    "\x20\x64\x89\x00\x06\x65\x06\x22\x02\x16\xe3\x1f\xa2\x16\x06\x00"
    "\x00\x00\x00\x1f\xa2\x1c\x06\x00\x00\x00\x00\x1f\xa2\x22\x06\x00"
    "\x00\x00\x00\x1f\xa2\x2e\x06\x00\x00\x00\x00\x16\x1e\x1f\xa2\x28"
    "\x06\x00\x00\x00\x00\x16\x14\x20\x8b\x89\x00\x06\x00\x16\x0c\x1f"
    "\xa2\x34\x06\x00\x00\x00\x00\x16\x02\x23\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x74\x65\x73\x74\x00\x00\x00\x01\x00\x02\x00\x03\x00"
    "\x04\x00\x03\x00\x02\x00\x01\x00\x00\x8e\xea\x4b\x43\xa7\x87\xf9"
    "\x01\x0e\xef\x56\x7b\x23\x9a\x34\x6b\x15\xa6\xb4\x0f\xc0\xe4\x4d"
    "\x2c\xd4\xa2\x21\x47\xe6\x0a\xef\xf2\xbc\x0f\xb6\xaf"
    // Input
    "\x04",
    446,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_NOT_FOUND,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.2 State handler tests: State memory management #6",

    // Code
    "\xf8\x1b\xa1\x0f\x86\x02\x89\xa2\x04\x1c\x01\xa0\x47\x9f\x99\x17"
    "\x63\x05\x08\xa0\x68\xa0\x70\x17\x63\x03\x07\x34\xa0\x56\x08\x23"
    "\x06\x06\x23\xa2\x04\x0e\xa0\x44\x63\x06\x23\x06\x12\x62\x02\xa0"
    "\x4a\x17\x62\x63\x08\xa0\x58\x9f\x71\x0e\xa0\x48\x65\x08\x24\x88"
    "\x20\x64\x89\x00\x06\x65\x06\x22\x02\x16\xe3\x1f\xa2\x16\x06\x00"
    "\x00\x00\x00\x1f\xa2\x1c\x06\x00\x00\x00\x00\x1f\xa2\x22\x06\x00"
    "\x00\x00\x00\x1f\xa2\x2e\x06\x00\x00\x00\x00\x16\x1e\x1f\xa2\x28"
    "\x06\x00\x00\x00\x00\x16\x14\x20\x8b\x89\x00\x06\x00\x16\x0c\x1f"
    "\xa2\x34\x06\x00\x00\x00\x00\x16\x02\x23\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x74\x65\x73\x74\x00\x00\x00\x01\x00\x02\x00\x03\x00"
    "\x04\x00\x03\x00\x02\x00\x01\x00\x00\x8e\xea\x4b\x43\xa7\x87\xf9"
    "\x01\x0e\xef\x56\x7b\x23\x9a\x34\x6b\x15\xa6\xb4\x0f\xc0\xe4\x4d"
    "\x2c\xd4\xa2\x21\x47\xe6\x0a\xef\xf2\xbc\x0f\xb6\xaf"
    // Input
    "\x05",
    446,

    // Output
    "", 
    0,

    // Expected Cycles
    2057,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.2 State handler tests: State memory management #7",

    // Code
    "\xf8\x1b\xa1\x0f\x86\x02\x89\xa2\x04\x1c\x01\xa0\x47\x9f\x99\x17"
    "\x63\x05\x08\xa0\x68\xa0\x70\x17\x63\x03\x07\x34\xa0\x56\x08\x23"
    "\x06\x06\x23\xa2\x04\x0e\xa0\x44\x63\x06\x23\x06\x12\x62\x02\xa0"
    "\x4a\x17\x62\x63\x08\xa0\x58\x9f\x71\x0e\xa0\x48\x65\x08\x24\x88"
    "\x20\x64\x89\x00\x06\x65\x06\x22\x02\x16\xe3\x1f\xa2\x16\x06\x00"
    "\x00\x00\x00\x1f\xa2\x1c\x06\x00\x00\x00\x00\x1f\xa2\x22\x06\x00"
    "\x00\x00\x00\x1f\xa2\x2e\x06\x00\x00\x00\x00\x16\x1e\x1f\xa2\x28"
    "\x06\x00\x00\x00\x00\x16\x14\x20\x8b\x89\x00\x06\x00\x16\x0c\x1f"
    "\xa2\x34\x06\x00\x00\x00\x00\x16\x02\x23\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x74\x65\x73\x74\x00\x00\x00\x01\x00\x02\x00\x03\x00"
    "\x04\x00\x03\x00\x02\x00\x01\x00\x00\x8e\xea\x4b\x43\xa7\x87\xf9"
    "\x01\x0e\xef\x56\x7b\x23\x9a\x34\x6b\x15\xa6\xb4\x0f\xc0\xe4\x4d"
    "\x2c\xd4\xa2\x21\x47\xe6\x0a\xef\xf2\xbc\x0f\xb6\xaf"
    // Input
    "\x06",
    446,

    // Output
    "", 
    0,

    // Expected Cycles
    1993,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #1",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x00",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    1809,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #2",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x01",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    1809,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    1,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #3",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x02",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    1809,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    2,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #4",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x03",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    1993,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #5",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x04",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    1994,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    1,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #6",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x05",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    1804,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    2,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #7",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x06",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_NOT_FOUND,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #8",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x07",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_NOT_FOUND,

    // Compartment ID
    1,

    // Input stream based?
    false
  },
  {
    "A.3.3 State handler tests: Multiple compartments #9",

    // Code
    "\xf8\x1b\x11\x0f\x86\x02\x89\xa2\x07\x1c\x01\xa0\x45\x9f\x99\x17"
    "\x62\x03\x0d\x3d\x06\x17\x62\x05\x37\x86\xa0\x68\x06\x22\x89\x20"
    "\xa1\xc0\x62\x00\x06\x00\x06\x22\x03\x20\xa1\xc0\x62\x00\x06\x00"
    "\x07\x22\xa2\x02\x0a\x22\x03\x06\x22\xa2\x03\x20\xa1\xc0\x62\x00"
    "\x06\x00\x20\xa1\xc0\xa2\x06\x00\x06\x62\x16\x2b\x20\xa7\xc0\x20"
    "\x00\x06\x00\x16\x22\x1f\xa2\x13\x06\x00\x00\x00\x00\x1f\xa2\x19"
    "\x06\x00\x00\x00\x00\x1f\xa2\x25\x06\x00\x00\x00\x00\x1f\xa2\x2b"
    "\x06\x00\x00\x00\x00\x23\x00\x00\x00\x00\x00\x00\x00\x17\x62\x07"
    "\x06\x10\x1a\x1f\xa2\x07\x06\x00\x00\x00\x00\x16\xea\x1f\xa2\x0d"
    "\x06\x00\x00\x00\x00\x16\xe0\x1f\xa2\x1f\x06\x00\x00\x00\x00\x16"
    "\x9f\xd6\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    "\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\xac\xa6\x0b\x8e\xb2\x83"
    "\x9d\xbf\xaf\xc6\x3d\xd2\x34\xc5\xd9\x1d\x53\x61\xbd\xd6\xba\x2a"
    "\xc6\x5a\x47\xc2\x18\x14\xee\x07\xc2\x75\x94\x1d\xd7\xa1\x48\x87"
    "\x9c\x8d\xe9\x0e"
    // Input
    "\x08",
    437,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_NOT_FOUND,

    // Compartment ID
    2,

    // Input stream based?
    false
  },
  {
    "A.3.4 State handler tests: Accessing RFC 3485 State #1",

    // Code
    "\xf8\x03\xa1\x1f\xa0\xa6\x14\xac\xfe\x01\x20\x00\x1f\xa0\xa6\x06"
    "\xac\xff\x01\x21\x00\x1f\xa0\xa6\x0c\xad\x00\x01\x22\x00\x22\x20"
    "\x03\x23\x00\x00\x00\x00\x00\x00\x00\xfb\xe5\x07\xdf\xe5\xe6\xaa"
    "\x5a\xf2\xab\xb9\x14\xce\xaa\x05\xf9\x9c\xe6\x1b\xa5"
    // Input
    "",
    61,

    // Output
    "\x53\x49\x50", 
    3,

    // Expected Cycles
    11,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.5 State handler tests: Bytecode state creation #1",

    // Code
    "\xf8\x04\x61\x12\xa0\xbe\x08\x1e\x20\x08\x1e\x21\x06\x0a\x0e\x23"
    "\xbe\x03\x12\xa0\xbe\x08\xa0\x50\x20\x08\xa0\x50\xa0\x53\x14\x0a"
    "\x20\x08\xa0\x50\xa0\x53\x0c\x0a\x16\x05\x4f\x4b\x31\x1c\x01\xa0"
    "\xb3\x9f\x55\x22\xa0\xa7\x03\x23\x00\x00\x0e\xa0\xa7\xa0\xaa\x06"
    "\x0a\x4f\x4b\x32\x22\xa0\x50\x03\x23\x02"
    // Input
    "",
    74,

    // Output
    "\x4f\x4b", 
    2,

    // Expected Cycles
    67,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.5 State handler tests: Bytecode state creation #2",

    // Code
    "\xf9\x53\xe9\x7f\x4a\xfd\xd2\x03"
    // Input
    "",
    8,

    // Output
    "\x4f\x4b\x31", 
    3,

    // Expected Cycles
    7,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.5 State handler tests: Bytecode state creation #3",

    // Code
    "\xfb\x24\x63\xcd\xff\x5c\xf8\xc7\x6d\xf6\xa2\x89\xff"
    // Input
    "",
    13,

    // Output
    "\x4f\x4b\x32", 
    3,

    // Expected Cycles
    5,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.5 State handler tests: Bytecode state creation #4",

    // Code
    "\xf9\x5b\x4b\x43\xd5\x67\x83"
    // Input
    "",
    7,

    // Output
    "\x00\x00\x32", 
    3,

    // Expected Cycles
    5,

    // Decompression Failure Expected?
    osc::OK,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {
    "A.3.5 State handler tests: Bytecode state creation #5",

    // Code
    "\xf9\xde\x81\x26\x11\x99\x1f"
    // Input
    "",
    7,

    // Output
    "", 
    0,

    // Expected Cycles
    0,

    // Decompression Failure Expected?
    osc::STATE_NOT_FOUND,

    // Compartment ID
    0,

    // Input stream based?
    false
  },
  {0,0,0,0,0,0,osc::OK,0,false}
};
