#
# SyncQMate conduit for PilotManager
#  Alan Harder, 3/98
#  Alan.Harder@Sun.COM
#
# Downloads records from QMate (by Steve Dakin, sdakin@netcom.com) and
# writes them to QIF files for importing into Quicken.  A separate QIF
# file is created for each account in QMate in the ~/.pilotmgr/SyncQMate
# directory.  Existing QIF files are renamed and appended with the create
# date when new QIF files are written.
# After importing data into quicken you need to manually delete the QIF
# files yourself..
#
# Also, if you're interested here is a cron command you can use to retrieve
# stock and mutual fund quotes daily and write them to a datafile for
# importing into quicken.  Change the symbols list, output file, path to perl
# and path to copyurl for your setup.  copyurl is available from
# www.moshpit.org/pilotmgr/binaries/misc.  Here it is:
#  00 15  *  *  1-5	cat | /usr/bin/tcsh%touch ~/quicken/quotes.txt%/usr/local/bin/perl /export/disk1/WWW/copyurl -s "http://fast.quote.com/fq/quotecom/quote?symbols=SUNW,INTC" | /usr/local/bin/perl -e 'chomp($d=`date +\045m/\045d/\045y`); do { $_=<> } until (/<TH.*Symbol/); $x=0; $c=7; while (<>) { if (/Mutual Fund Quotes/) { $c=6; next } last if /Fundamental Data/; if (/<TD A/) { $x++; if ($x==1 or $x==$c) { s/<.*?>//g; s/^\s+//; chomp; print "$_  "; print "$d \n" if ($x==$c) } } $x=0 if /<\/TR>/ }' >> ~/quicken/quotes.txt
#
#
package SyncQMate;

use Tk;
use FileHandle;
use strict;

my ($gConfigDlg);
my ($VERSION) = ('0.1');

sub conduitInit
{
}

sub conduitQuit
{
}

sub conduitInfo
{
    return { 'database' =>
		{
		    'name' => 'QMateTransactionsDB',
		    'creator' => 'QMat',
		    'type' => 'QTrn',
		    'flags' => 0,
		    'version' => 0,
		},
	     'version' => $VERSION,
	     'author' => 'Alan Harder',
	     'email' => 'Alan.Harder@Sun.COM' };
}

sub conduitConfigure
{
    my ($this, $wm) = @_;

    &createConfigDlg($wm) unless (defined $gConfigDlg);

    $gConfigDlg->Popup(-popanchor => 'c', -overanchor => 'c',
		       -popover => $wm);
}

sub createConfigDlg
{
    my ($wm) = @_;

    $gConfigDlg = $wm->Toplevel(-title => 'SyncFood');
    $gConfigDlg->transient($wm);

    $gConfigDlg->Label(-text => "SyncQMate v$VERSION\nAlan Harder\nAlan.Harder\@Sun.COM")->pack(-side => 'top');

    $gConfigDlg->Button(-text => 'Dismiss', -command => ['withdraw', $gConfigDlg])->pack;

    PilotMgr::setColors($gConfigDlg);
}

sub conduitSync
{
    my ($this, $dlp, $info) = @_;
    my ($db, $i, $rec, @recs, $fd, $fn, $fn2);
    my ($acctNames, $acctTypes, $files, $counter, $typeList) =
	({}, {}, {}, {}, ['Bank', 'Cash', 'CCard']);

    $db = $dlp->open('QMateAccountsDB');
    $dlp->getStatus();
    $i=0;
    while (1)
    {
	$rec = $db->getRecord($i++);
	last unless defined $rec;
	next if $rec->{'deleted'};

	&translateQMateAcct($rec);
	$acctNames->{ $rec->{'acctIndex'} } = $rec->{'acctName'};
	$acctTypes->{ $rec->{'acctIndex'} } = $typeList->[$rec->{'acctType'}];

	$rec->{'acctName'} =~ tr| '"?*<>|_______|;
	$fn = "SyncQMate/$rec->{acctName}.qif";
	if (-f $fn)
	{
	    @_ = localtime((stat($fn))[9]);
	    $_[4]++;
	    $fn2 = sprintf "SyncQMate/$rec->{acctName}-%02d%02d%02d.qif",
			   @_[4,3,5];
	    rename $fn,$fn2;
	}

	$fd = new FileHandle ">SyncQMate/$rec->{acctName}.qif" or
	    croak("error opening qif file");
	print $fd "!Type:", $typeList->[$rec->{'acctType'}], "\n";
	$files->{ $rec->{'acctIndex'} } = $fd;
	$counter->{ $rec->{'acctIndex'} } = 0;
    }
    $db->close();

    $db = $dlp->open('QMateTransactionsDB');
    $dlp->getStatus();
    $i = 0;
    while (1)
    {
	$rec = $db->getRecord($i++);
	last unless defined $rec;
	next if $rec->{'deleted'};

	&translateQMateRec($rec);
	next if $rec->{'synced'};	# skip if already been synced

	push(@recs, $rec);
    }

    foreach $rec (@recs)
    {
	$fd = $files->{ $rec->{'account'} };
	$counter->{ $rec->{'account'} }++;

	print $fd "D$rec->{date} \nT$rec->{amount} \n";
	print $fd "N$rec->{number} \n" if defined $rec->{'number'};
	print $fd "P$rec->{payee} \nM$rec->{note} \nL$rec->{categ} \n^ \n";

#	$rec->{'synced'} = 1;		# seems to automatically set
#	&translateQMateRec($rec, 1);	# this flag when you write the rec
	$db->setRecord($rec);
    }
    $db->close;

    foreach $i (keys %$files)
    {
	undef $files->{$i};
	PilotMgr::msg("Exported ", $counter->{$i}, " records to account ",
		      $acctNames->{$i});
	unlink("SyncQMate/$acctNames->{$i}.qif") if ($counter->{$i} == 0);
    }
}

sub translateQMateAcct
{
    my ($rec) = @_;
    my ($txt, $idx);

    return if $rec->{'deleted'};
    $txt = $rec->{'raw'};

    $rec->{'acctIndex'} = 256 * ord($txt) + ord(substr($txt, 1, 1));
    $txt = substr($txt, 2);
    $rec->{'acctType'} = 256 * ord($txt) + ord(substr($txt, 1, 1));
    $txt = substr($txt, 2);
    $idx = index($txt, "\000");
    $rec->{'acctName'} = substr($txt, 0, $idx);
}

sub translateQMateRec
{
    my ($rec, $dir) = @_;
    my ($txt, $val);

    unless ($dir)
    {
	# raw -> expanded
	return if ($rec->{'deleted'});
	$txt = $rec->{'raw'};

	# 0-1: account (index)
	#
	$rec->{'account'} = 256 * ord($txt) + ord(substr($txt, 1, 1));
	$txt = substr($txt, 2);

	# 2-3: date
	#  bits 15-9 = year - 1904
	#  bits  8-5 = month
	#  bits  4-0 = day
	#
	$val = 256 * ord($txt) + ord(substr($txt, 1, 1));
	$rec->{'date'} = (($val & 0x01E0) >> 5) . '/' .
			 ($val & 0x001F) . '/';
	$val = (($val & 0xFE00) >> 9) + 4;
	$val -= 100 if ($val >= 100);
	$rec->{'date'} .= sprintf("%02d", $val);
	$txt = substr($txt, 2);

	# 4-5: check number/transaction type
	#
	$val = 256 * ord($txt) + ord(substr($txt, 1, 1));
	$val -= 65536 if ($val >= 32768);
	if ($val == 0)		{}
	elsif ($val > 0)	{ $rec->{'number'} = $val; }
	elsif ($val == -1)	{ $rec->{'number'} = 'PRINT'; }
	elsif ($val == -2)	{ $rec->{'number'} = 'ATM'; }
	elsif ($val == -3)	{ $rec->{'number'} = 'CCARD'; }
	elsif ($val == -4)	{ $rec->{'number'} = 'DEP'; }
	elsif ($val == -5)	{ $rec->{'number'} = 'EFT'; }
	elsif ($val == -6)	{ $rec->{'number'} = 'TXFR'; }
	elsif ($val == -7)	{ $rec->{'number'} = 'WITHD'; }
	elsif ($val == -8)	{ $rec->{'number'} = 'XMIT'; }
	elsif ($val == -9)	{ $rec->{'number'} = 'SEND'; }
	$txt = substr($txt, 2);

	# 6-22: amount
	#
	$val = index($txt, "\000");
	$rec->{'amount'} = substr($txt, 0, $val);
	$txt = substr($txt, 17);

	# 23: cleared(bit0) and synced(bit1) flags
	#
	$val = ord($txt);
	$rec->{'cleared'} = ($val & 1) ? 1 : 0;
	$rec->{'synced'} = ($val & 2) ? 1 : 0;
	$txt = substr($txt, 2);

	# 24-end: payee, note, category strings
	#  (prepended with one byte lengths, but also null terminated)
	#
	$val = index($txt, "\000");
	$rec->{'payee'} = substr($txt, 0, $val);
	$txt = substr($txt, $val+2);
	$val = index($txt, "\000");
	$rec->{'note'} = substr($txt, 0, $val);
	$txt = substr($txt, $val+2);
	$val = index($txt, "\000");
	$rec->{'categ'} = substr($txt, 0, $val);
    }
    else
    {
	# expanded -> raw
	$rec->{'raw'} = '';

	$val = $rec->{'account'};
	$rec->{'raw'} .= chr($val/256) . chr($val%256);

	my ($mo,$da,$yr) = split('/', $rec->{'date'});
	$yr += 100 if ($yr < 32);
	$val = $da | ($mo << 5) | (($yr-4) << 9);
	$rec->{'raw'} .= chr($val/256) . chr($val%256);

	$val = $rec->{'number'};
	if (!defined $val)		{ $val = 0; }
	elsif ($val =~ /^\d+$/)		{}
	elsif ($val =~ /^print$/i)	{ $val = -1; }
	elsif ($val =~ /^atm$/i)	{ $val = -2; }
	elsif ($val =~ /^ccard$/i)	{ $val = -3; }
	elsif ($val =~ /^dep$/i)	{ $val = -4; }
	elsif ($val =~ /^eft$/i)	{ $val = -5; }
	elsif ($val =~ /^txfr$/i)	{ $val = -6; }
	elsif ($val =~ /^withd$/i)	{ $val = -7; }
	elsif ($val =~ /^xmit$/i)	{ $val = -8; }
	elsif ($val =~ /^send$/i)	{ $val = -9; }
	else				{ $val = 0; }
	$val += 65536 if ($val < 0);
	$rec->{'raw'} .= chr($val/256) . chr($val%256);

	$rec->{'raw'} .= $rec->{'amount'} .
			 ("\000" x (17-length($rec->{'amount'})));

	$val = ($rec->{'cleared'} ? 1 : 0) + ($rec->{'synced'} ? 2 : 0);
	$rec->{'raw'} .= chr($val);

	$rec->{'raw'} .= chr( length($rec->{'payee'}) ) .
			 $rec->{'payee'} .
			 "\000" .
			 chr( length($rec->{'note'}) ) .
			 $rec->{'note'} .
			 "\000" .
			 chr( length($rec->{'categ'}) ) .
			 $rec->{'categ'} .
			 "\000";
    }
}

1;

