/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/triangular/cdf' );
import Triangular = require( './../../../../../base/dists/triangular/ctor' );
import entropy = require( './../../../../../base/dists/triangular/entropy' );
import kurtosis = require( './../../../../../base/dists/triangular/kurtosis' );
import logcdf = require( './../../../../../base/dists/triangular/logcdf' );
import logpdf = require( './../../../../../base/dists/triangular/logpdf' );
import mean = require( './../../../../../base/dists/triangular/mean' );
import median = require( './../../../../../base/dists/triangular/median' );
import mgf = require( './../../../../../base/dists/triangular/mgf' );
import mode = require( './../../../../../base/dists/triangular/mode' );
import pdf = require( './../../../../../base/dists/triangular/pdf' );
import quantile = require( './../../../../../base/dists/triangular/quantile' );
import skewness = require( './../../../../../base/dists/triangular/skewness' );
import stdev = require( './../../../../../base/dists/triangular/stdev' );
import variance = require( './../../../../../base/dists/triangular/variance' );

/**
* Interface describing the `triangular` namespace.
*/
interface Namespace {
	/**
	* Triangular distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - lower limit
	* @param b - upper limit
	* @param c - mode
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 0.5, -1.0, 1.0, 0.0 );
	* // returns 0.875
	*
	* y = ns.cdf( 0.5, -1.0, 1.0, 0.5 );
	* // returns 0.75
	*
	* y = ns.cdf( -10.0, -20.0, 0.0, -2.0 );
	* // returns ~0.278
	*
	* y = ns.cdf( -2.0, -1.0, 1.0, 0.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 0.0, 10.0, 2.0 );
	* y = mycdf( 0.5 );
	* // returns 0.0125
	*
	* y = mycdf( 8.0 );
	* // returns 0.95
	*/
	cdf: typeof cdf;

	/**
	* Triangular Distribution.
	*/
	Triangular: typeof Triangular;

	/**
	* Returns the differential entropy of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 0.0, 1.0, 0.5 );
	* // returns ~-0.193
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0, 9.0 );
	* // returns ~1.886
	*
	* @example
	* var v = ns.entropy( -4.0, 4.0, -1.0 );
	* // returns ~1.886
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 0.0, 1.0, 0.5 );
	* // returns -0.6
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0, 9.0 );
	* // returns -0.6
	*
	* @example
	* var v = ns.kurtosis( -4.0, 4.0, -1.0 );
	* // returns -0.6
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Triangular distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - lower limit
	* @param b - upper limit
	* @param c - mode
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 0.5, -1.0, 1.0, 0.0 );
	* // returns ~-0.134
	*
	* y = ns.logcdf( 0.5, -1.0, 1.0, 0.5 );
	* // returns ~-0.288
	*
	* y = ns.logcdf( -10.0, -20.0, 0.0, -2.0 );
	* // returns ~-1.281
	*
	* y = ns.logcdf( -2.0, -1.0, 1.0, 0.0 );
	* // returns -Infinity
	*
	* var mylogcdf = ns.logcdf.factory( 0.0, 10.0, 2.0 );
	* y = mylogcdf( 0.5 );
	* // returns ~-4.382
	*
	* y = mylogcdf( 8.0 );
	* // returns ~-0.051
	*/
	logcdf: typeof logcdf;

	/**
	* Triangular distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param a - lower limit
	* @param b - upper limit
	* @param c - mode
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 0.5, -1.0, 1.0, 0.0 );
	* // returns ~-0.693
	*
	* y = ns.logpdf( 0.5, -1.0, 1.0, 0.5 );
	* // returns 0.0
	*
	* y = ns.logpdf( -10.0, -20.0, 0.0, -2.0 );
	* // returns ~-2.89
	*
	* var mylogpdf = ns.logpdf.factory( 0.0, 10.0, 5.0 );
	* y = mylogpdf( 2.0 );
	* // returns ~-2.526
	*
	* y = mylogpdf( 12.0 );
	* // returns -Infinity
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 0.0, 1.0, 0.5 );
	* // returns ~0.5
	*
	* @example
	* var v = ns.mean( 4.0, 12.0, 9.0 );
	* // returns ~8.333
	*
	* @example
	* var v = ns.mean( -4.0, 4.0, -1.0 );
	* // returns ~-0.333
	*
	* @example
	* var v = ns.mean( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns median
	*
	* @example
	* var v = ns.median( 0.0, 1.0, 0.5 );
	* // returns 0.5
	*
	* @example
	* var v = ns.median( 4.0, 12.0, 9.0 );
	* // returns ~8.472
	*
	* @example
	* var v = ns.median( -4.0, 4.0, -1.0 );
	* // returns ~-0.472
	*
	* @example
	* var v = ns.median( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Triangular distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param a - lower limit
	* @param b - upper limit
	* @param c - mode
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.5, -1.0, 1.0, 0.0 );
	* // returns ~1.021
	*
	* y = ns.mgf( 0.5, -1.0, 1.0, 0.5 );
	* // returns ~1.111
	*
	* y = ns.mgf( -0.3, -20.0, 0.0, -2.0 );
	* // returns ~24.334
	*
	* y = ns.mgf( -2.0, -1.0, 1.0, 0.0 );
	* // returns ~1.381
	*
	* var mymgf = ns.mgf.factory( 0.0, 2.0, 1.0 );
	* y = mymgf( -1.0 );
	* // returns ~0.3996
	*
	* y = mymgf( 2.0 );
	* // returns ~10.205
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns mode
	*
	* @example
	* var v = ns.mode( 0.0, 1.0, 0.5 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mode( 4.0, 12.0, 9.0 );
	* // returns 9.0
	*
	* @example
	* var v = ns.mode( -4.0, 4.0, -1.0 );
	* // returns -1.0
	*
	* @example
	* var v = ns.mode( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Triangular distribution probability density function (PDF).
	*
	* @param x - input value
	* @param a - lower limit
	* @param b - upper limit
	* @param c - mode
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 0.5, -1.0, 1.0, 0.0 );
	* // returns 0.5
	*
	* y = ns.pdf( 0.5, -1.0, 1.0, 0.5 );
	* // returns 1.0
	*
	* y = ns.pdf( -10.0, -20.0, 0.0, -2.0 );
	* // returns ~0.056
	*
	* var mypdf = ns.pdf.factory( 0.0, 10.0, 5.0 );
	* y = mypdf( 2.0 );
	* // returns 0.08
	*
	* y = mypdf( 12.0 );
	* // returns 0.0
	*/
	pdf: typeof pdf;

	/**
	* Triangular distribution quantile function.
	*
	* @param p - input value
	* @param a - lower limit
	* @param b - upper limit
	* @param c - mode
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.9, -1.0, 1.0, 0.0 );
	* // returns ~0.553
	*
	* y = ns.quantile( 0.1, -1.0, 1.0, 0.5 );
	* // returns ~-0.452
	*
	* y = ns.quantile( 0.1, -20.0, 0.0, -2.0 );
	* // returns -14.0
	*
	* y = ns.quantile( 0.8, 0.0, 20.0, 0.0 );
	* // returns ~11.056
	*
	* var myquantile = ns.quantile.factory( 2.0, 4.0, 2.5 );
	* y = myquantile( 0.4 );
	* // returns ~2.658
	*
	* y = myquantile( 0.8 );
	* // returns ~3.225
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 0.0, 1.0, 0.5 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0, 9.0 );
	* // returns ~-0.236
	*
	* @example
	* var v = ns.skewness( -4.0, 4.0, -1.0 );
	* // returns ~0.236
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0.0, 1.0, 0.5 );
	* // returns ~0.204
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0, 9.0 );
	* // returns ~1.65
	*
	* @example
	* var v = ns.stdev( -4.0, 4.0, -1.0 );
	* // returns ~1.65
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a triangular distribution.
	*
	* ## Notes
	*
	* -   If the condition `a <= c <= b` is not satisfied, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @param c - mode
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0.0, 1.0, 0.5 );
	* // returns ~0.042
	*
	* @example
	* var v = ns.variance( 4.0, 12.0, 9.0 );
	* // returns ~2.722
	*
	* @example
	* var v = ns.variance( -4.0, 4.0, -1.0 );
	* // returns ~2.722
	*
	* @example
	* var v = ns.variance( 1.0, -0.1, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 0.0, 1.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 4.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 0.0, NaN, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 0.0, 4.0, NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Triangular distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
