/*
 * Main API entry point
 *
 * Copyright (c) 2000-2001 Johannes Erdfelt <johannes@erdfelt.com>
 *
 * This library is covered by the LGPL, read LICENSE for details.
 */

#include <stdlib.h>	/* getenv */
#include <stdio.h>	/* stderr */

#include "usbi.h"

int usb_debug = 0;
struct usb_bus *usb_busses = NULL;

int usb_find_devices(void)
{
  struct usb_bus *bus;
  int ret;

  for (bus = usb_busses; bus; bus = bus->next) {
    struct usb_device *devices, *dev, *ndev, *tdev;

    /* Find all of the devices and put them into a temporary list */
    ret = usb_find_devices_on_bus(bus, &devices);
    if (ret < 0)
      return ret;

    /*
     *  Now walk through all of the devices we know about and compare
     * against this new list. Any duplicates will be removed from the new
     * list. If we don't find it in the new list, the device was removed.
     * Any devices still in the new list, are new to us.
     */
    dev = bus->devices;
    while (dev) {
      int found = 0;
      struct usb_device *tdev = dev->next;

      ndev = devices;
      while (ndev) {
        struct usb_device *tndev = ndev->next;

        if (!strcmp(dev->filename, ndev->filename)) {
          /* Remove it from the new devices list */
          LIST_DEL(devices, ndev);

          usb_free_dev(ndev);
          found = 1;
          break;
        }

        ndev = tndev;
      }

      if (!found)
        /* The device was removed from the system */
        LIST_DEL(bus->devices, dev);

      dev = tdev;
    }

    /* Anything on the *devices list is new. So add them to bus->devices */
    /*  and process them like the new device they are */
    dev = devices;
    while (dev) {
      struct usb_device *tdev = dev->next;

      /* Remove it from the temporary list first and add it to the real */
      /*  bus->devices list */
      LIST_DEL(devices, dev);

      LIST_ADD(bus->devices, dev);

      /* Some ports fetch the descriptors on scanning (like Linux) so we */
      /*  don't need to fetch them again */
      if (!dev->config) {
        usb_dev_handle *udev;

        udev = usb_open(dev);
        if (udev) {
          usb_fetch_and_parse_descriptors(udev);

          usb_close(udev);
        }
      }

      dev = tdev;
    }
  }

  return 0;
}

void usb_set_debug(int level)
{
  if (usb_debug || level)
    fprintf(stderr, "usb_set_debug: Setting debugging level to %d (%s)\n",
	level, level ? "on" : "off");

  usb_debug = level;
}

void usb_init(void)
{
  if (getenv("USB_DEBUG"))
    usb_set_debug(atoi(getenv("USB_DEBUG")));

  usb_os_init();
}

usb_dev_handle *usb_open(struct usb_device *dev)
{
  usb_dev_handle *udev;

  udev = malloc(sizeof(*udev));
  if (!udev)
    return NULL;

  udev->fd = -1;
  udev->device = dev;
  udev->bus = dev->bus;
  udev->config = udev->interface = udev->altsetting = -1;

  if (usb_os_open(udev) < 0) {
    free(udev);
    return NULL;
  }

  return udev;
}

int usb_close(usb_dev_handle *dev)
{
  int ret;

  ret = usb_os_close(dev);
  free(dev);

  return ret;
}

struct usb_device *usb_device(usb_dev_handle *dev)
{
  return dev->device;
}

void usb_free_dev(struct usb_device *dev)
{
  usb_destroy_configuration(dev);
  free(dev);
}

struct usb_bus *usb_get_busses(void)
{
  return usb_busses;
}

