/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     write_mesh.c                                                   */
/*                                                                          */
/* description:  functions for writing meshes and DOF vectors in binary     */
/*               machine independent and native formats                     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Univesitaet Bremen                                           */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "alberta_intern.h"
 
static XDR *xdrp = nil;
static FILE *file = nil;

/*
   WARNING:
   XDR routines to read/write ALBERTA types must be changed if the
   ALBERTA types change!

   current state:  REAL   = double
                   U_CHAR = unsigned char
                   S_CHAR = signed char
                   DOF    = int

   Another WARNING! (D.K.)
   XDR routines are not well documented in the "xdr" man page.
   Do not change anything unless you know what you are doing!
*/

/*--------------------------------------------------------------------------*/

static void write_int(int val)
{
  if(xdrp)
    xdr_int(xdrp, &val);
  else
    fwrite(&val, sizeof(int), 1, file);
}

#if 0
/* NEVER use this. If you need 64 bits, then use xdr_int64_t() */
static void write_long(long int val)
{
  if(xdrp)
    xdr_long(xdrp, &val);
  else
    fwrite(&val, sizeof(long int), 1, file);
}
#endif

static void write_REAL(REAL val)
{
  if(xdrp)
    AI_xdr_REAL(xdrp, &val);
  else
    fwrite(&val, sizeof(REAL), 1, file);
}

static void write_string(const char *string, int write_length)
{
  int strileng = 0;

  if(string)
    strileng = strlen(string);

  if(write_length)
    write_int(strileng);

  if(strileng) {
    if(xdrp)
      xdr_string(xdrp, (char **)&string, strileng+1);
    else
      fwrite(string, sizeof(char), strileng+1, file);
  }

  return;
}

static void write_vector(void *start, int n, size_t size, xdrproc_t xdrproc)
{
  if(xdrp)
    xdr_vector(xdrp, (char *)start, n, size, xdrproc);
  else
    fwrite(start, size, n, file);
}

static void write_U_CHAR(U_CHAR val)
{
  if(xdrp)
    AI_xdr_U_CHAR(xdrp, &val);
  else
    fwrite(&val, sizeof(U_CHAR), 1, file);
}

/*--------------------------------------------------------------------------*/

static int  *n_dof, *node;

static int         n_vert_ptrs;
static DOF         *vert_dofs, **vert_ptrs;
static DOF_ADMIN   *vert_admin;

static int         n_edge_ptrs;
static DOF         *edge_dofs, **edge_ptrs;
static DOF_ADMIN   *edge_admin;

static int         n_face_ptrs;
static DOF         *face_dofs, **face_ptrs;
static DOF_ADMIN   *face_admin;

static int         *mesh_n_dof, *mesh_node;

/*--------------------------------------------------------------------------*/

static void write_el_recursive(int dim, EL *el)
{
  FUNCNAME("write_el_recursive");
  int    i, n, node0, n1;
  static U_CHAR uc_no = 0, uc_yes = 1;

  DEBUG_TEST_EXIT(el, "el == nil\n");

  if (el->child[0]) {
    DEBUG_TEST_EXIT(el->child[1], "child 0 but no child 1\n");
    
    write_U_CHAR(uc_yes);    
  }
  else
    write_U_CHAR(uc_no);   


  if(dim > 1) {
    if (el->new_coord) {
      write_U_CHAR(uc_yes);   
      write_vector(el->new_coord, DIM_OF_WORLD, sizeof(REAL), 
		   (xdrproc_t)AI_xdr_REAL);
    }
    else
      write_U_CHAR(uc_no); 
  }

  if (n_dof[VERTEX] > 0) {
    node0 = node[VERTEX];
    n1 = vert_admin->n0_dof[VERTEX];
   
    for (i = 0; i < N_VERTICES(dim); i++) 
      write_int(vert_dofs[el->dof[node0 + i][n1]]);
  }

  if (dim > 1 && n_dof[EDGE] > 0) {
    node0 = node[EDGE];
    n1 = edge_admin->n0_dof[EDGE];
     
    for (i = 0; i < N_EDGES(dim); i++) 
      if(el->dof[node0 + i][n1] >= 0)
	write_int(edge_dofs[el->dof[node0 + i][n1]]);
      else
	write_int(-1);
  }

  if (dim == 3 && n_dof[FACE] > 0) {
    node0 = node[FACE];
    n1 = face_admin->n0_dof[FACE];
    
    for (i = 0; i < N_FACES_3D; i++) 
      if(el->dof[node0 + i][n1] >= 0)
	write_int(face_dofs[el->dof[node0 + i][n1]]);
      else
	write_int(-1);
  }
  
  if ((n = n_dof[CENTER]) > 0) {
    node0 = node[CENTER];
    
    write_vector(el->dof[node0], n, sizeof(DOF), (xdrproc_t)AI_xdr_DOF);
  }


  if (el->child[0])
    for (i = 0; i < 2; i++)
      write_el_recursive(dim, el->child[i]);

  return;
}

/*--------------------------------------------------------------------------*/

#define OFFSET 10      /* must be > MAX(VERTEX,EDGE,FACE,CENTER) */

static void fill_dofs_fct(const EL_INFO *elinfo, void *data)
{
  EL   *el = elinfo->el;
  int  i, n0, n1, dim = elinfo->mesh->dim;
  DOF  *p;
  
  if (vert_admin) {
    n0 = mesh_node[VERTEX];
    n1 = vert_admin->n0_dof[VERTEX];
    for (i = 0; i < N_VERTICES(dim); i++) {
      p = el->dof[n0+i];
      vert_dofs[p[n1]] = VERTEX-OFFSET;
      vert_ptrs[p[n1]] = p;
    }
  }

  if (dim > 1 && edge_admin) {
    n0 = mesh_node[EDGE];
    n1 = edge_admin->n0_dof[EDGE];
    for (i = 0; i < N_EDGES(dim); i++) {
      p = el->dof[n0+i];
      if(p[n1] >= 0) {
	edge_dofs[p[n1]] = EDGE-OFFSET;
	edge_ptrs[p[n1]] = p;
      }
    }
  }
  
  if (dim == 3 && face_admin) {
    n0 = mesh_node[FACE];
    n1 = face_admin->n0_dof[FACE];
    for (i = 0; i < N_FACES_3D; i++) {
      p = el->dof[n0+i];
      if(p[n1] >= 0) {
	face_dofs[p[n1]] = FACE-OFFSET;
	face_ptrs[p[n1]] = p;
      }
    }
  }

  return;
}

/*--------------------------------------------------------------------------*/

static int write_mesh_master(const int write_xdr,
			     MESH *mesh, const char *filename, REAL time)
{
  FUNCNAME("write_mesh_master");
  MACRO_EL   *mel, *meln;
  DOF_ADMIN  *admin;
  int        i, j, n, m, iadmin, dim;
  int        neigh_i[N_NEIGH_MAX];
  
  if (!mesh)
  {
    ERROR("no mesh - no file created\n");
    return(1);
  }

  n_edge_ptrs = 0;
  n_face_ptrs = 0;
  n_vert_ptrs = 0;

  dim = mesh->dim;

  dof_compress(mesh);

  n_dof = mesh->n_dof;
  node  = mesh->node;


  if(write_xdr) {
    if (!(xdrp = AI_xdr_open_file(filename, XDR_ENCODE))) {
      ERROR("Cannot open XDR file '%s' for writing.\n",filename);
      return(1);
    }
  }
  else {
    if (!(file=fopen(filename,"wb"))) {
      ERROR("Cannot open file '%s' for writing.\n",filename);
      return(1);
    }
  }

  write_string(ALBERTA_VERSION, false); /* file marker */

  write_int(dim);
  
  write_int(DIM_OF_WORLD);
  
  write_REAL(time);

  write_string(mesh->name, true);

  write_int(mesh->n_vertices);

  if(dim > 1)
    write_int(mesh->n_edges);

  write_int(mesh->n_elements);
  write_int(mesh->n_hier_elements);


  if(dim == 3) {
    write_int(mesh->n_faces);
    write_int(mesh->max_edge_neigh);
  }

  write_vector(mesh->diam, DIM_OF_WORLD, sizeof(REAL), (xdrproc_t)AI_xdr_REAL);

  write_int(mesh->n_dof_el);
  write_vector(mesh->n_dof, N_NODE_TYPES, sizeof(int), (xdrproc_t)xdr_int);
  write_int(mesh->n_node_el);
  write_vector(mesh->node, N_NODE_TYPES, sizeof(int), (xdrproc_t)xdr_int);



  write_int(mesh->n_dof_admin);
  for (iadmin = 0; iadmin < mesh->n_dof_admin; iadmin++) { 
    admin = mesh->dof_admin[iadmin];

    DEBUG_TEST_EXIT(admin, "dof admin no. %d not found!\n", iadmin);

    write_vector(admin->n_dof, N_NODE_TYPES, sizeof(int), (xdrproc_t)xdr_int); 
    write_int(admin->used_count);

    /* after dof_compress(), no more information is required */

    write_string(admin->name, true);

    write_U_CHAR(admin->preserve_coarse_dofs);
  }


  if(dim == 3) {
    face_admin = nil;
    if (mesh->n_dof[FACE] > 0) 
    {
      m = 99999;
      for (iadmin = 0; iadmin < mesh->n_dof_admin; iadmin++) 
      {
	if ((admin = mesh->dof_admin[iadmin])) 
	{
	  if (admin->n_dof[FACE] > 0 && 
	      (admin->n_dof[FACE] < m || admin->preserve_coarse_dofs)) 
	  {
	    face_admin = admin;
	    m = admin->n_dof[FACE];
	  } 
	} 
      } 
      DEBUG_TEST_EXIT(face_admin, "no admin with face dofs?\n"); 
      face_dofs = MEM_ALLOC(face_admin->used_count, DOF); 
      for (i=0; i<face_admin->used_count; i++) face_dofs[i] = -1; 
      face_ptrs = MEM_ALLOC(face_admin->used_count, DOF*); 
      for (i=0; i<face_admin->used_count; i++) face_ptrs[i] = nil; 
    } 
  }

  if(dim > 1) {
    edge_admin = nil;
    if (mesh->n_dof[EDGE] > 0) 
    { 
      if (dim == 3 && face_admin && (face_admin->n_dof[EDGE] > 0)
	  && face_admin->preserve_coarse_dofs) 
      {
	edge_admin = face_admin; 
	edge_dofs = face_dofs; 
	edge_ptrs = face_ptrs; 
      } 
      else 
      {
	m = 99999;
	for (iadmin = 0; iadmin < mesh->n_dof_admin; iadmin++) 
	{ 
	  if ((admin = mesh->dof_admin[iadmin])) 
	  {
	    if (admin->n_dof[EDGE] > 0  && 
		(admin->n_dof[EDGE] < m || admin->preserve_coarse_dofs)) 
	    {
	      edge_admin = admin;
	      m = admin->n_dof[EDGE];
	    } 
	  } 
	} 
	DEBUG_TEST_EXIT(edge_admin, "no admin with edge dofs?\n");
	edge_dofs = MEM_ALLOC(edge_admin->used_count, DOF);
	for (i=0; i<edge_admin->used_count; i++) edge_dofs[i] = -1;
	edge_ptrs = MEM_ALLOC(edge_admin->used_count, DOF*);
	for (i=0; i<edge_admin->used_count; i++) edge_ptrs[i] = nil;
      } 
    } 
  }

  vert_admin = nil; 
  if (mesh->n_dof[VERTEX] > 0) 
  {  
    if (dim == 3 && face_admin && (face_admin->n_dof[VERTEX] > 0)) 
    {
      vert_admin = face_admin; 
      vert_dofs = face_dofs; 
      vert_ptrs = face_ptrs; 
    } 
    else
    {
      if (dim > 1 && edge_admin && (edge_admin->n_dof[VERTEX] > 0)) 
      {
	vert_admin = edge_admin;
	vert_dofs = edge_dofs;
	vert_ptrs = edge_ptrs;
      } 
      else
      {
	m = 99999;
	for (iadmin = 0; iadmin < mesh->n_dof_admin; iadmin++) 
	{
	  if ((admin = mesh->dof_admin[iadmin])) 
	  {
	    if (admin->n_dof[VERTEX] > 0  && admin->n_dof[VERTEX] < m) 
	    {
	      vert_admin = admin;
	    m = admin->n_dof[VERTEX];
	    } 
	  } 
	} 
	DEBUG_TEST_EXIT(vert_admin, "no admin with vertex dofs?\n");
	vert_dofs = MEM_ALLOC(vert_admin->used_count, DOF);
	for (i=0; i<vert_admin->used_count; i++) vert_dofs[i] = -1;
	vert_ptrs = MEM_ALLOC(vert_admin->used_count, DOF*);
	for (i=0; i<vert_admin->used_count; i++) vert_ptrs[i] = nil;
      } 
    }
  }

  mesh_n_dof = mesh->n_dof;
  mesh_node  = mesh->node;
  mesh_traverse(mesh, 0, CALL_EVERY_EL_PREORDER|FILL_NOTHING, 
		fill_dofs_fct, nil);

  if (vert_admin) 
  {
    for (i = 0; i < vert_admin->used_count; i++)
      if (vert_dofs[i] == VERTEX-OFFSET)
	n_vert_ptrs++;
  } 

  if (dim > 1 && edge_admin) 
  {
    for (i = 0; i < edge_admin->used_count; i++)
      if (edge_dofs[i] == EDGE-OFFSET)
 	n_edge_ptrs++;
  }

  if (dim == 3 && face_admin) 
  {
    for (i = 0; i < face_admin->used_count; i++)
      if (face_dofs[i] == FACE-OFFSET)
	n_face_ptrs++; 
  } 

  write_int(n_vert_ptrs);   

  if (n_vert_ptrs) 
  {
    j = 0;
    n = mesh_n_dof[VERTEX];
    for (i = 0; i < vert_admin->used_count; i++) 
    {
      if (vert_dofs[i] == VERTEX-OFFSET) 
      {
	vert_dofs[i] = j++; 
        
        write_vector(vert_ptrs[i], n, sizeof(DOF), (xdrproc_t)AI_xdr_DOF); 
      } 
    } 
    DEBUG_TEST_EXIT(j==n_vert_ptrs, "j != n_vert_ptrs\n"); 
  } 
 
  if(dim > 1) {
    write_int(n_edge_ptrs); 

    if (n_edge_ptrs) {
      j = 0;
      n = mesh_n_dof[EDGE];
      for (i = 0; i < edge_admin->used_count; i++) {
	if (edge_dofs[i] == EDGE-OFFSET) { 
	  edge_dofs[i] = j++; 
	  
	  write_vector(edge_ptrs[i], n, sizeof(DOF), (xdrproc_t)AI_xdr_DOF);
	}
      }
      DEBUG_TEST_EXIT(j==n_edge_ptrs, "j != n_edge_ptrs\n");
    } 
  }

  if(dim == 3) {
    write_int(n_face_ptrs);
    
    if (n_face_ptrs) {
      j = 0; 
      n = mesh_n_dof[FACE]; 
      for (i = 0; i < face_admin->used_count; i++) {
	if (face_dofs[i] == FACE-OFFSET) {
	  face_dofs[i] = j++; 
	  
	  write_vector(face_ptrs[i], n, sizeof(DOF), (xdrproc_t)AI_xdr_DOF);
	} 
      } 
      DEBUG_TEST_EXIT(j==n_face_ptrs, "j != n_face_ptrs\n"); 
    } 
  }
    

/*--------------------------------------------------------------------------*/
/* gather info about macro elements (vertices, ...)                         */
/*--------------------------------------------------------------------------*/
  {
    typedef int  intNV[N_VERTICES(dim)];
    int    (*mcindex)[N_VERTICES(dim)] = MEM_ALLOC(mesh->n_macro_el, intNV);
    REAL_D   *mccoord;
    int    mccount, m;

    mccount = ((MESH_MEM_INFO *)(mesh->mem_info))->count;
    mccoord  = ((MESH_MEM_INFO *)(mesh->mem_info))->coords;

    for (m = 0, mel = mesh->macro_els;
	 m < mesh->n_macro_el;
	 m++, mel = mesh->macro_els + m) { 
      for (i = 0; i < N_VERTICES(dim); i++)
	mcindex[m][i] = 
((char *)(mel->coord[i]) - (char *)mccoord)/(sizeof(char)*sizeof(REAL_D));

      if (mel->index != m) mel->index = m;
    }

    write_int(mesh->n_macro_el);
    write_int(mccount);                           /* number of macro coords */

    for (i = 0; i < mccount; i++) 
       write_vector(mccoord[i], DIM_OF_WORLD, sizeof(REAL),
		    (xdrproc_t)AI_xdr_REAL);
 
    for (m = 0, mel = mesh->macro_els;
	 m < mesh->n_macro_el;
	 m++, mel = mesh->macro_els + m) { 
      write_vector(mcindex[m], N_VERTICES(dim), sizeof(int), 
		   (xdrproc_t)xdr_int);
      write_vector(mel->vertex_bound, 
		   N_VERTICES(dim), sizeof(S_CHAR), (xdrproc_t)AI_xdr_S_CHAR);

      if(dim == 2)
	write_vector(mel->edge_bound, N_EDGES_2D, sizeof(S_CHAR), 
		     (xdrproc_t)AI_xdr_S_CHAR);

      if(dim == 3) {
	write_vector(mel->face_bound, N_FACES_3D, sizeof(S_CHAR), 
		     (xdrproc_t)AI_xdr_S_CHAR);
	write_vector(mel->edge_bound, N_EDGES_3D, sizeof(S_CHAR), 
		     (xdrproc_t)AI_xdr_S_CHAR);
      }

      for (i = 0; i < N_NEIGH(dim); i++) 
      { 
	if ((meln = mel->neigh[i]))
	  neigh_i[i] = meln->index; 
	else 
	  neigh_i[i] = -1; 
      } 

      write_vector(neigh_i, N_NEIGH(dim), sizeof(int), (xdrproc_t)xdr_int);
      write_vector(mel->opp_vertex, N_NEIGH(dim), sizeof(U_CHAR), 
		   (xdrproc_t)AI_xdr_U_CHAR);


      if(dim == 3)
	write_U_CHAR(mel->el_type);    

      write_el_recursive(dim, mel->el); 
    } 

    MEM_FREE(mcindex, mesh->n_macro_el, intNV);
  } 

  /* Write the magic cookie. */
  write_int(mesh->cookie);

  write_string("EOF.", false);                       /* file end marker */

    
  if (dim == 3 && face_admin) 
  {
    MEM_FREE(face_ptrs, face_admin->used_count, DOF*); 
    MEM_FREE(face_dofs, face_admin->used_count, DOF); 
    if (edge_admin == face_admin)  edge_admin = nil; 
    if (vert_admin == face_admin)  vert_admin = nil; 
    face_admin = nil; 
  } 

  if (dim > 1 && edge_admin) 
  {
    MEM_FREE(edge_ptrs, edge_admin->used_count, DOF*); 
    MEM_FREE(edge_dofs, edge_admin->used_count, DOF); 
    if (vert_admin == edge_admin)  vert_admin = nil; 
    edge_admin = nil; 
  } 

  if (vert_admin) 
  {
    MEM_FREE(vert_ptrs, vert_admin->used_count, DOF*); 
    MEM_FREE(vert_dofs, vert_admin->used_count, DOF); 
    vert_admin = nil; 
  }

  if(write_xdr) {
    AI_xdr_close_file(xdrp);
    xdrp = nil;
  }
  else
    fclose(file);

  file = nil;

  return(0);
}


extern int write_mesh(MESH *mesh, const char *filename, REAL time)
{
  return write_mesh_master(false, mesh, filename, time);
}

extern int write_mesh_xdr(MESH *mesh, const char *filename, REAL time)
{
  return write_mesh_master(true, mesh, filename, time);
}


/*--------------------------------------------------------------------------*/
/* write DOF vectors of various types                                       */
/*--------------------------------------------------------------------------*/
typedef DOF_REAL_VEC DOF_VEC;

static int write_dof_vec_master(const int write_xdr, const DOF_VEC *dv,
				const char *filename, const char *dofvectype)
{
  FUNCNAME("write_dof_vec_master");
  const FE_SPACE  *fe_space;
  const DOF_ADMIN *admin;
  MESH            *mesh;
  int              i, iadmin, last;


  if (!dv || !(fe_space = dv->fe_space))
  {
    ERROR("no %s or fe_space - no file created\n", dofvectype);
    return(1);
  }
  if (!(admin = fe_space->admin) || !(mesh = admin->mesh))
  {
    ERROR("no dof_admin or dof_admin->mesh - no file created\n");
    return(1);
  }

  dof_compress(mesh);

  iadmin = -1;
  for (i = 0; i < mesh->n_dof_admin; i++)
    if (mesh->dof_admin[i] == admin) 
    {
      iadmin = i;
      break;
    }
  if (iadmin < 0) 
  {
    ERROR("vec->admin not in mesh->dof_admin[] - no file created\n");
    return(1);
  }

  last = admin->used_count;
  DEBUG_TEST_EXIT(last <= dv->size,
	      "dof_vec->size %d < admin->size_used %d\n", dv->size, last);

 
  if(write_xdr) {
    if (!(xdrp = AI_xdr_open_file(filename, XDR_ENCODE))) {
      ERROR("Cannot open XDR file '%s' for writing.\n",filename);
      return(1);
    }
  }
  else {
    if (!(file=fopen(filename,"wb"))) {
      ERROR("Cannot open file '%s' for writing.\n",filename);
      return(1);
    }
  }

  write_string(dofvectype, false);  

  write_string(dv->name, true);
  
  write_U_CHAR(admin->preserve_coarse_dofs);

  write_vector((void *)admin->n_dof, N_NODE_TYPES, sizeof(int), 
	       (xdrproc_t)xdr_int);  

  if (fe_space->bas_fcts)
    write_string(fe_space->bas_fcts->name, true);
  else
    write_int(0);


  write_int(last);

  if (last) {
    if (!strncmp(dofvectype, "DOF_REAL_VEC    ", 12))
      write_vector(dv->vec, last, sizeof(REAL), (xdrproc_t)AI_xdr_REAL);  
    else if (!strncmp(dofvectype, "DOF_REAL_D_VEC  ", 12))
      write_vector(dv->vec, last*DIM_OF_WORLD, sizeof(REAL), 
		   (xdrproc_t)AI_xdr_REAL);
    else if (!strncmp(dofvectype, "DOF_INT_VEC     ", 12))
      write_vector(dv->vec, last, sizeof(int), (xdrproc_t)xdr_int);
    else if (!strncmp(dofvectype, "DOF_SCHAR_VEC   ", 12))
      write_vector(dv->vec, last, sizeof(S_CHAR), (xdrproc_t)AI_xdr_S_CHAR);  
    else if (!strncmp(dofvectype, "DOF_UCHAR_VEC   ", 12))
      write_vector(dv->vec, last, sizeof(U_CHAR), (xdrproc_t)AI_xdr_U_CHAR);  
    else
      ERROR("Invalid file id '%s'.\n",dofvectype);
  }      

  /* Write the magic cookie. */
  write_int(mesh->cookie);
      
  write_string("EOF.", false);                         /* file end marker */

  if(write_xdr) {
    AI_xdr_close_file(xdrp); 
    xdrp = nil;
  }
  else
    fclose(file);
  file = nil;

  return(0);
}

/*--------------------------------------------------------------------------*/

int write_dof_real_vec_xdr(const DOF_REAL_VEC *dv, const char *fn)
{ 
  return(write_dof_vec_master(true, (const DOF_VEC *)dv,
			      fn, "DOF_REAL_VEC    "));
}

int write_dof_real_vec(const DOF_REAL_VEC *dv, const char *fn)
{ 
  return(write_dof_vec_master(false, (const DOF_VEC *)dv,
			      fn, "DOF_REAL_VEC    "));
}

int write_dof_real_d_vec_xdr(const DOF_REAL_D_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(true, (const DOF_VEC *)dv,
			      fn, "DOF_REAL_D_VEC  "));
}

int write_dof_real_d_vec(const DOF_REAL_D_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(false, (const DOF_VEC *)dv,
			      fn, "DOF_REAL_D_VEC  "));
}

int write_dof_int_vec_xdr(const DOF_INT_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(true, (const DOF_VEC *)dv,
			      fn, "DOF_INT_VEC     "));
}

int write_dof_int_vec(const DOF_INT_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(false, (const DOF_VEC *)dv,
			      fn, "DOF_INT_VEC     "));
}

int write_dof_schar_vec_xdr(const DOF_SCHAR_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(true, (const DOF_VEC *)dv,
			      fn, "DOF_SCHAR_VEC   "));
}

int write_dof_schar_vec(const DOF_SCHAR_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(false, (const DOF_VEC *)dv,
			      fn, "DOF_SCHAR_VEC   "));
}

int write_dof_uchar_vec_xdr(const DOF_UCHAR_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(true, (const DOF_VEC *)dv,
			      fn, "DOF_UCHAR_VEC   "));
}

int write_dof_uchar_vec(const DOF_UCHAR_VEC *dv, const char *fn)
{
  return(write_dof_vec_master(false, (const DOF_VEC *)dv,
			      fn, "DOF_UCHAR_VEC   "));
}

/*--------------------------------------------------------------------------*/
/*  write_dof_matrix_pbm: print matrix structure as portable bitmap         */
/*--------------------------------------------------------------------------*/

int write_dof_matrix_pbm(const DOF_MATRIX *matrix,
			  const char *filename)
{
  FUNCNAME("write_dof_matrix_pbm");
  int  i, j, jcol, size;
  MATRIX_ROW *row;
  char *pbm_row;
  FILE *file;

  if (!(file=fopen(filename,"w")))
  {
    ERROR("cannot open file %s\n",filename);
    return(1);
  }

  size = matrix->size + 1;
  pbm_row = MEM_CALLOC(size, char);

  fprintf(file, "P1\n");
  fprintf(file, "# ALBERTA output of DOF_MATRIX %s\n", matrix->name);
  fprintf(file, "%d %d\n", matrix->size, matrix->size);

  for (i=0; i<matrix->size; i++) {
    memset(pbm_row, '0', matrix->size);
    for (row = matrix->matrix_row[i]; row; row = row->next) {
      
      for (j=0; j<ROW_LENGTH; j++) {
	jcol = row->col[j];

	if (ENTRY_USED(jcol) && row->entry[j])
	  pbm_row[jcol] = '1';

      }
    }
    fprintf(file, "%s\n", pbm_row);
  }

  MEM_FREE(pbm_row, size, char);
  fclose(file);

  return 0;
}

/*--------------------------------------------------------------------------*/
