package tests::FieldTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::Field;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();
    $self->init_i18n();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_locale();
    $self->set_up_test_schema();

    $self->{'i18n_field'} =
      new Lire::Field( 'name' => 'File',
                       'type' => 'string',
                       'i18n_domain' => 'lire-test',
                       'description' => '<para>JPEG Files</para>',
                       'label' => 'Downloads' );

    return $self;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    $self->tear_down_locale();

    return $self;
}

sub test_new {
    my $self = $_[0];

    $self->assert_dies( qr/missing 'name' parameter/,
                        sub { new Lire::Field() } );

    $self->assert_dies( qr/missing 'type' parameter/,
                        sub { new Lire::Field( 'name' => 'name' ) } );

    $self->assert_dies( qr/'name' parameter should be a valid XML name: 'bad name'/,
                        sub { new Lire::Field( 'name' => 'bad name',
                                               'type' => 'int') } );

    $self->assert_dies( qr/'type' parameter should be a valid Lire type: 'unknown'/,
                        sub { new Lire::Field( 'name' => 'name',
                                               'type' => 'unknown' ) } );

    my $field = new Lire::Field( 'name' => 'field', type => 'int',
                                 'description' => '<para>Description.</para>',
                                 'label' => 'Field Label',
                               );
    $self->assert_isa( 'Lire::Field', $field );

    $self->assert_str_equals( "field", $field->name() );
    $self->assert_str_equals( "int", $field->type() );
    $self->assert_null( $field->pos(), "pos attribute should be undef"  );
    $self->assert_str_equals( 'Field Label', $field->label() );
    $self->assert_str_equals( '<para>Description.</para>', 
                              $field->description() );
    $self->assert_str_equals( "lire", $field->{'i18n_domain'} );

    $field = new Lire::Field( 'name' => 'field', type => 'int',
                              'description' => '<para>Description.</para>',
                              'i18n_domain' => 'lire-test',
                              'label' => 'Field Label',
                               );
    $self->assert_str_equals( "lire-test", $field->{'i18n_domain'} );
}

sub test_sql_type {
    my $self = $_[0];

    my @specials = ([ "id", "INTEGER PRIMARY KEY" ],
                    [ "bool", "NUMBER(1,0)" ],
                    [ "int", "NUMBER(10,0)" ],
                    [ "number", "NUMBER(10,5)" ],
                    [ "timestamp", "TIMESTAMP" ],
                    [ "date", "TIMESTAMP" ],
                    [ "time", "TIMESTAMP" ],

                    [ "duration", "NUMBER(10,1)" ],
                    [ "bytes", "NUMBER(10,1)" ],
                  );
    foreach my $p ( @specials ) {
        my $field = new Lire::Field( 'name' => "name", type => $p->[0],
                                     'i18n_domain' => 'lire' );

        $self->annotate( "checking " . $field->type(). "\n" );
        $self->assert_str_equals( $p->[1], $field->sql_type() );
    }

    my @strings = qw/string hostname ip port string url email filename /;
    foreach my $t ( @strings ) {
        my $field = new Lire::Field( 'name' => "name", type => $t,
                                     'i18n_domain' => 'lire' );
        $self->annotate( "checking " . $field->type(). "\n" );
        $self->assert_str_equals( "VARCHAR(512)", $field->sql_type() );
    }
}

sub test_label_i18n {
    my $self = $_[0];

    my $field = $self->{'i18n_field'};
    $self->assert_str_equals( 'Downloads', $field->label() );
    $self->set_locale( 'fr_CA.iso8859-1' );
    $self->assert_str_equals( 'Tlchargements', $field->label() );
}

sub test_label_no_label_i18n {
    my $self = $_[0];

    my $field = $self->{'i18n_field'};
    delete $field->{'label'};
    $self->assert_str_equals( 'File', $field->label() );
    $self->set_locale( 'fr_CA.iso8859-1' );
    $self->assert_str_equals( 'Fichier', $field->label() );
}

sub test_description_i18n {
    my $self = $_[0];

    my $field = $self->{'i18n_field'};
    $self->assert_str_equals( '<para>JPEG Files</para>', $field->description() );
    $self->set_locale( 'fr_CA.iso8859-1' );
    $self->assert_str_equals( '<para>Fichiers JPEG</para>', $field->description() );
}

1;
