/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


#include "log.h"
#include "_verify_x509.h"



int grid_X509_empty_callback(char *buf, int buf_size, int verify, void *cb_tmp)
{
         if ( buf_size > 0 ) buf = '\0';
              return 0;
}


size_t u_strlen (const unsigned char * s)
{
    size_t i = 0;

    for (i = 0; s[i] != '\0'; i++) ;

    return i;
}


unsigned char * u_strcpy (unsigned char * dest, const unsigned char * src)
{
    size_t i = 0;
    bzero (dest, u_strlen (src) + 1);

    for (i = 0; i < u_strlen (src); i++)
    {
        dest[i] = src[i];
    }

    return dest;
}


/* Check if certificate can be used as a CA to sign standard X509 certs */
/*
 *  Return 1 if true; 0 if not.
 */

int grid_x509IsCA(X509 *cert)
{
    int idret;

    /* final argument to X509_check_purpose() is whether to check for CAness */
    idret = X509_check_purpose(cert, X509_PURPOSE_SSL_CLIENT, 1);
    if (idret == 1)
        return 1;
    else if (idret == 0)
        return 0;
    else
    {
        Log( L_WARN, "Purpose warning code = %d\n", idret );
        return 1;
    }

#if 0
    BASIC_CONSTRAINTS *                 x509v3_bc = NULL;
    int                                 index = -1;
    int                                 critical;
    if((x509v3_bc = X509_get_ext_d2i(cert,
                    NID_basic_constraints,
                    &critical,
                    &index)) && x509v3_bc->ca)
    {
        *type = GLOBUS_GSI_CERT_UTILS_TYPE_CA;
        goto exit;
    }
#endif
}

/**
 * Note that timegm() is non-standard. Linux manpage advices the following
 * substition instead.
 */
time_t my_timegm(struct tm *tm)
{
   time_t ret;
   char *tz;

   tz = getenv("TZ");
   setenv("TZ", "", 1);
   tzset();
   ret = mktime(tm);
   if (tz)
       setenv("TZ", tz, 1);
   else
       unsetenv("TZ");
   tzset();

   return ret;
}


/* ASN1 time string (in a char *) to time_t */
/**
 *  (Use ASN1_STRING_data() to convert ASN1_GENERALIZEDTIME to char * if
 *   necessary)
 */

time_t grid_asn1TimeToTimeT(unsigned char *asn1time, size_t len)
{
   char   zone;
   struct tm time_tm;

   memset(&time_tm,0,sizeof(struct tm));

   if (len == 0) len = u_strlen(asn1time);

   if ((len != 13) && (len != 15)) return 0; /* dont understand */

   if ((len == 13) &&
       ((sscanf((char *) asn1time, "%02d%02d%02d%02d%02d%02d%c",
         &(time_tm.tm_year),
         &(time_tm.tm_mon),
         &(time_tm.tm_mday),
         &(time_tm.tm_hour),
         &(time_tm.tm_min),
         &(time_tm.tm_sec),
         &zone) != 7) || (zone != 'Z'))) return 0; /* dont understand */

   if ((len == 15) &&
       ((sscanf((char *) asn1time, "20%02d%02d%02d%02d%02d%02d%c",
         &(time_tm.tm_year),
         &(time_tm.tm_mon),
         &(time_tm.tm_mday),
         &(time_tm.tm_hour),
         &(time_tm.tm_min),
         &(time_tm.tm_sec),
         &zone) != 7) || (zone != 'Z'))) return 0; /* dont understand */

   /* time format fixups */

   if (time_tm.tm_year < 90) time_tm.tm_year += 100;
   --(time_tm.tm_mon);

   return my_timegm(&time_tm);
}



unsigned long verify_x509_readPrivateKeyFromFile (char *filename, EVP_PKEY **pkey)
{
    BIO                 *certbio = NULL;
    Log( L_DEBUG, "--- Reading the Private Key From File ---");

    if ( (certbio = BIO_new(BIO_s_file())) == NULL ) return ERR_get_error();

    Log( L_INFO, "Reading file %s", filename );
    if ( BIO_read_filename(certbio, filename) <= 0 ) return ERR_get_error();

    Log( L_DEBUG, "Reading Private key" );
    *pkey = PEM_read_bio_PrivateKey( certbio, NULL, grid_X509_empty_callback, NULL );

    if ( *pkey == NULL ) Log( L_WARN, "No private key found." );

    BIO_free(certbio);
    return X509_V_OK;
}

unsigned long verify_x509_readPrivateKeyFromPEM  (unsigned char * pem, EVP_PKEY **pkey)
{
    BIO                 *certbio = NULL;
    Log( L_DEBUG, "--- Reading the Private Key From PEM ---");

    /* if ( (certbio = BIO_new(BIO_s_mem())) == NULL ) return ERR_get_error(); */

    Log( L_INFO, "Reading PEM string", pem);
    if ( (certbio = BIO_new_mem_buf (pem, -1)) == NULL ) return ERR_get_error();

    Log( L_DEBUG, "Reading Private key" );
    *pkey = PEM_read_bio_PrivateKey( certbio, NULL, grid_X509_empty_callback, NULL );

    if ( *pkey == NULL ) Log( L_WARN, "No private key found." );

    BIO_free(certbio);
    return X509_V_OK;
}



unsigned long verify_x509_readPublicCertChain ( char *filename, STACK_OF(X509) **certstack)
{
    char                *oper = "Reading proxy";

    STACK_OF(X509_INFO) *sk      = NULL;
    BIO                 *certbio = NULL;
    X509_INFO           *xi;
    unsigned long        err;

    Log( L_DEBUG, "--- Welcome to the grid_readProxy function ---");

    *certstack = sk_X509_new_null();
    if (*certstack == NULL) return ERR_get_error();

    if ( (certbio = BIO_new(BIO_s_file())) == NULL ) return ERR_get_error();

    Log( L_INFO, "Reading file %s", filename );
    if ( BIO_read_filename(certbio, filename) <= 0 ) return ERR_get_error();

    Log( L_DEBUG, "Reading X509_INFO records" );
    if ( !(sk=PEM_X509_INFO_read_bio(certbio, NULL, NULL, NULL)) )
    {
        err = ERR_get_error();
        Error( oper, "No X509 records found" );
        BIO_free(certbio);
        sk_X509_INFO_free(sk);
        sk_X509_free(*certstack);
        return err;
    }

    while (sk_X509_INFO_num(sk))
    {
        xi=sk_X509_INFO_shift(sk);
        if (xi->x509 != NULL)
        {
            sk_X509_push(*certstack, xi->x509);
            xi->x509=NULL;
        }
        X509_INFO_free(xi);
    }

    if (!sk_X509_num(*certstack))
    {
        err = ERR_get_error();
        Error( oper, "No certificates found" );
        BIO_free(certbio);
        sk_X509_INFO_free(sk);
        sk_X509_free(*certstack);
        return err;
    }

    BIO_free(certbio);
    sk_X509_INFO_free(sk);

    return X509_V_OK;
}




int startVerifyProcess (char * proxyfilename, char * CA_dir)
{
    unsigned long     result = -1;
    STACK_OF(X509)   *certStack;
    EVP_PKEY         *pkey = NULL;

    /* read the file and build the certificate stack, plus the proxy private key */
    result = grid_readProxy( proxyfilename, &certStack, &pkey );
    if ( result == X509_V_OK )
    {
        /* first, verify the certificate chain */
        result = grid_verifyCert( CA_dir, certStack );
        if ( result == X509_V_OK )
        {
            /* still OK? then match the proxy public and private keys */
            result = grid_verifyPrivateKey( certStack, pkey );
            if ( result == X509_V_OK )
            {
                /* aaah, nirwana */
                printf( "OK\n" );
            }
            else
                Error( "Verifying private key", "%s\n",
                        ERR_reason_error_string( result ) );
        }
        else
            Error( "Verifying certificate chain", "%s\n",
                   X509_verify_cert_error_string( result ) );
    }
    else
        Error( "Reading proxy", "%s\n",
               ERR_reason_error_string( result ) );

    return result;
}
