/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  A base class for all classes for reading data files. Each iFileLoader loads a single kind of data,
//  but may provide data for multiple data types.
//
#ifndef IFILELOADER_H
#define IFILELOADER_H


#include <vtkObjectBase.h>


#include "iarray.h"
#include "ipointermacro.h"
#include "istring.h"

#include <vtkSetGet.h>

class iDataReader;
class iDataReaderExtension;
class iDataSubject;
class iDataType;
class iErrorStatus;
class iFile;
class iProgressEventObserver;
class iViewModule;

class vtkDataSet;


namespace iParameter
{
	//
    // Data parameters
	//
	namespace BoundaryConditions
	{
		const int Periodic = 0;
		const int Wall = 1;

		inline bool IsValid(int m){ return m>=0 && m<=1; }
	};
};


//
//  Helper macros
//
#define	IDATAFILELOADER_DECLARE_SWAP_BYTES_FUNCTIONS(_type_) \
	static void SwapBytes(_type_ &p); \
	static void SwapBytesRange(_type_ *data, vtkIdType count)

#define	IDATAFILELOADER_DECLARE_READ_BLOCK_FUNCTIONS(_type_) \
	bool ReadBlock(iFile& F, _type_ *p, vtkIdType len, float updateStart, float updateDuration); \
	ReadingBuffer Buffer(_type_ &d) const; \
	ReadingBuffer Buffer(_type_ *d, vtkIdType l) const


class iFileLoader : public vtkObjectBase
{

	IPOINTER_AS_PART(ViewModule);
	IPOINTER_AS_USER(ErrorStatus);

	friend class iDataReader;
	friend class iDataSubject;
	friend class iFieldFileLoader;
	friend class iParticleFileLoader;

public:

	vtkTypeMacro(iFileLoader,vtkObjectBase);

	IDATAFILELOADER_DECLARE_SWAP_BYTES_FUNCTIONS(vtkTypeInt32);
	IDATAFILELOADER_DECLARE_SWAP_BYTES_FUNCTIONS(vtkTypeInt64);
	IDATAFILELOADER_DECLARE_SWAP_BYTES_FUNCTIONS(vtkTypeFloat32);
	IDATAFILELOADER_DECLARE_SWAP_BYTES_FUNCTIONS(vtkTypeFloat64);

	//
	//  Operation on data
	//
	void ReadFile(const iString &fname);
	void Finalize();
	void EraseData();
	void Reset();
	void ShiftData(double dr[3]);

	inline bool IsTwoCopies() const { return mTwoCopies; }
	void SetTwoCopies(bool s);

	virtual bool IsAborted() const;
	inline bool IsAnimatable() const { return (mRecord >= 0); }

	inline const iString& GetLastFileName() const { return mLastFileName; }
	inline const iString GetFileRoot() const { return mFileRoot; }
	virtual const iString GetFileName(int rec) const;
	iString GetRecordAsString(int rec) const;
	bool IsSeriesFileName(const iString &fname) const;

	virtual float GetMemorySize() const;

	//
	//  Boundary conditions
	//
	void SetBoundaryConditions(int s);
	void SetDirectionPeriodic(int d, bool s);
	inline int GetBoundaryConditions() const {	return mBoundaryConditions; }
	inline bool IsBoxPeriodic() const {	return (mBoundaryConditions == iParameter::BoundaryConditions::Periodic); }
	inline bool IsDirectionPeriodic(int d) const { if(d>=0 && d<3) return this->IsBoxPeriodic() && mPeriodic[d]; else return false; }

	//
	//  Access to components
	//
	inline iDataReader* GetReader() const { return mReader; }
	inline iDataReaderExtension* GetReaderExtension() const { return mReaderExtension; }
	bool IsThereData() const;
	virtual bool IsThereData(int n) const;
	virtual iDataSubject* GetSubject(int n) const;
	virtual vtkDataSet* GetData(int n) const;
	inline iProgressEventObserver* GetObserver() const { return mObserver; }

	bool IsUsingData(const iDataType &type) const;
	const iDataType& GetDataType(int n) const;
	virtual int DissectFileName(const iString &fname, iString &root, iString &suffix) const;
	inline int NumStreams() const { return mStreams.Size(); }

	bool IsThereData(const iDataType &type) const;
	iDataSubject* GetSubject(const iDataType &type) const;
	vtkDataSet* GetData(const iDataType &type) const;

	inline int GetPriority() const { return mPriority; }

	//
	//  Helper struct
	//
	struct ReadingBuffer
	{
		void *Data;
		vtkIdType Length;
		int Type, Size;
		ReadingBuffer(void *d, vtkIdType l, int t, int s)
		{
			Data = d;
			Length = l;
			Type = t;
			Size = s;
		}
	};

protected:

	iFileLoader(iDataReaderExtension *ext, int priority);
	virtual ~iFileLoader();

	virtual void ReadFileBody(const iString &suffix, const iString &fname) = 0;
	virtual void FinalizeBody() = 0;
	virtual void ShiftDataBody(vtkDataSet *data, double dx[3]) = 0;
	virtual void Polish(vtkDataSet * ){} // optional post-finalize

	//
	//  File reading helpers
	//
	IDATAFILELOADER_DECLARE_READ_BLOCK_FUNCTIONS(vtkTypeInt32);
	IDATAFILELOADER_DECLARE_READ_BLOCK_FUNCTIONS(vtkTypeInt64);
	IDATAFILELOADER_DECLARE_READ_BLOCK_FUNCTIONS(vtkTypeFloat32);
	IDATAFILELOADER_DECLARE_READ_BLOCK_FUNCTIONS(vtkTypeFloat64);

	bool DetectFortranFileStructure(iFile &F, vtkIdType len);
	bool ReadFortranHeaderFooter(iFile& F, vtkIdType &lrec);

	bool SkipFortranRecord(iFile& F, vtkIdType len);
	bool ReadFortranRecord(iFile& F, const ReadingBuffer &b, float updateStart, float updateDuration, bool autoswap = true);
	bool ReadFortranRecord(iFile& F, const ReadingBuffer &b1, const ReadingBuffer &b2, float updateStart, float updateDuration, bool autoswap = true);
	bool ReadFortranRecord(iFile& F, const ReadingBuffer &b1, const ReadingBuffer &b2, const ReadingBuffer &b3, float updateStart, float updateDuration, bool autoswap = true);
	bool ReadFortranRecord(iFile& F, int nbufs, const ReadingBuffer *b, float updateStart, float updateDuration, bool autoswap = true); // read any number of buffers

	//
	//  Data management
	//
	struct Stream
	{
		iDataSubject *Subject;
		vtkDataSet *ReleasedData;

		Stream();
		virtual ~Stream();
	};

	virtual Stream* CreateNewStream() const;
	inline Stream* GetStream(int i) const { return mStreams[i]; } // no bound checking for speed
	void AttachDataToStream(int i, vtkDataSet *ds);

	void NotifyDependencies();

	//
	//  Components
	//
	int mBoundaryConditions;
	bool mPeriodic[3];

	iString mFileRoot, mFileSuffix, mLastFileName;
	bool mTwoCopies, mIsBigEndian, mOverflow;
	iProgressEventObserver *mObserver;
	double mShift[3];
	int mRecord, mFortranHeaderFooterLength;

	iDataReader *mReader;
	iDataReaderExtension *mReaderExtension;

private:

	iFileLoader(iDataReader *r, int priority);

	void Define();

	iArray<Stream*> mStreams;

	const int mPriority;
	bool mUpdated; // should not be used by children
};

#endif  // IFILELOADER_H

