#! /usr/bin/env python

# Genetic
# Copyright (C) 2001 Jean-Baptiste LAMY
#
# This program is free software. See README or LICENSE for the license terms.

import random, string, math
from genetic import organism, lifecycle


# Set the function to minimize. We choose the cosinus function, between 0.0 and 6.0 -- the
# minimum is PI !

def FUNC(x):
  if x < 0.0 or x > 6.0: return None # Organism with "None" phenotype cannot live !
  return math.cos(x)


class FUNCMinimizer(organism.Organism):
  characteristics = [
    organism.Characteristic("FUNC", FUNC, organism.RECESSIVE_PHENOTYPE)
    ]
  
  def __cmp__(self, other):
    return cmp(self.FUNC, other.FUNC)


# Create the initial set of organisms.
#
# As usual in Python, __<something>__ genes are some "magic" value that
# influence the chromosom replication or stability :
#
# __mutation__ is the pobability that each gene mutate while reproducting,
# ranging from 0.0 to 1.0.
#
# __mutampl__ is the amplitude of the mutation. A gene that mutate is modified
# by + or - a random number ranging from 0.0 to __mutampl__. Notice that
# __mutampl__ doesn't apply to magic genes, for them the old value of the gene
# is used instead.
#
# __mutsign__ is the sign of the mutation. If __mutsign__ is 0.0 (default) or
# if 0.5 < __mutsign__ < 2.0, the mutation can be + or - (it can increase or
# decrease the gene value). If __mutsign__ < 0.5, the mutation can only
# decrease the value; if > 2.0, it can only increase.
#
# __crossover__ is the probability that a cross-over occurs -- a cross-over
# will mix 2 chromosom.
#
# __deletion__ is the probability to delete a random gene -- organism doesn't
# like that...
#
# __loss__ is the probability for the chromosom to be lost -- organism doesn't
# like it too !
#
# __swap__ is the probability to swap the values of 2 random genes.
#
# __duplicate__ is the probability to get a random gene from the other
# chromosom.
#
# __break__ is the probability for the chromosom to be broken in 2 parts. If
# you use the recessive PHENOTYPE, organism often like this one, as it allows
# them to have more genetic stuff -- more chance to survive !
#
# __dominancy__ is the dominancy of the chromosom. This is only use by the
# PHENOTYPE function organism.perchromosom_dominancy_phenotype, which choose
# the genes of the most dominant chromosom.
#
# What is really MAGIC is that the are genes... so they can mutate ! Yes, the
# probability of mutation can change ! This is what make the organisms
# adaptate very well.
#
# Those magic genes are set per-chromosom, so if you have genes with different
# scales or units, it can be a good idea to put them on different chromosom
# (so they can have different magic values).
#
# Notice that, if __mutation__ and/or __mutampl__ fall down, the organisms
# won't continue their evolution, and they will refuse to go on, even if they
# are not at the "end" of the problem ! So yes, they are free to "fix" them
# where they are, and stop evoluting ! This is also a kind of adaptation... ;-)

organismA = FUNCMinimizer([
  organism.Chromosom(__mutation__ = 0.5, __mutampl__ = 4.0, x = 5.0),
  ])

organismB = FUNCMinimizer([
  organism.Chromosom(__mutation__ = 1.0, __mutampl__ = 2.0, x = 1.0),
  ])

organisms = [organismA, organismB]


# Define some parameters

# True for elitism (= always keep the best organism)
ELITISM = 1

# Number of generation to run
NB_GENERATION = 8

# Number of children each generation made
NB_CHILDREN  = 75

# Number of organisms kept per generation (other children doesn't survive)
NB_ORGANISMS = 10


# Ready !
#
# This will make live a lot of organism, and finally print the last and "best"
# one -- the one whose phenotype minimizes FUNC !

lifecycle.run(organisms, ELITISM, NB_GENERATION, NB_CHILDREN, NB_ORGANISMS)

# You should approach the PI value.
#
# Obviously, mathematicians will say that the cos function already include the
# PI value, so we do nothing... let them speak and let our organisms live !
