/******************************************************************************
 * $Id: head.c.txt,v 1.3 2007/01/13 22:36:23 gareuselesinge Exp $
 * This file is part of FreePOPs (http://www.freepops.org)                    *
 * This file is distributed under the terms of GNU GPL license.               *
 ******************************************************************************/

/******************************************************************************
 * File description:
 *	Event logging for windows and lua
 * Notes:
 *	windows 2000 or newer
 * Authors:
 * 	Name <gareuselesinge@users.sourceforge.net>
 ******************************************************************************/

#include <lua.h>
#include <stdio.h>
#include <lauxlib.h>
#include <string.h>

#include "luay.h"

static int cmp_level(lua_State* L, const char * f, int n){
	return !strcmp(lua_tostring(L,n),f);
}

#ifndef WIN32

#define LOG_ZONE "lwel"
#include "log.h"

static int lwel_log(lua_State* L){
	const char * msg = lua_tostring(L,2);

	if (cmp_level(L,"DEBUG",1)) {
		DBG("wel: DEBUG: %s",msg);
	} else if (cmp_level(L,"INFO",1)) {
		SAY("wel: INFO: %s",msg);
	} else if (cmp_level(L,"WARN",1)) {
		SAY("wel: WARN: %s",msg);
	} else if (cmp_level(L,"ERROR",1)) {
		ERROR_PRINT("wel: ERROR: %s",msg);
	} else if (cmp_level(L,"FATAL",1)) {
		ERROR_PRINT("wel: FATAL: %s",msg);
	} else {
		return luaL_error(L,"logging with an unknown level");
	}

	lua_pushboolean(L,1);
	return 1;
}

static int new_lwel(lua_State* L) {
	lua_pushcfunction(L,lwel_log);
	return 1;
}

static const struct luaL_Reg log_f [] = {
  {"new",new_lwel},	
  {NULL,NULL}
};

int luaopen_wel_core(lua_State* L){
	luaL_register(L,"wel.core",log_f);
	return 1;
}
	
#else

#include <windows.h>

#include "manifest.h"

#define ES_METATABLE "lwel.EventSource"

static int lwel_reg(lua_State* L, const char *logName,  const char *sourceName, const char *dllName){
	// This number of categories for the event source, hardcoded in the manifest
	DWORD dwCategoryNum = 5;
	
	HKEY hk; 
	DWORD dwData, dwDisp; 
	TCHAR szBuf[MAX_PATH]; 
	size_t cchSize = MAX_PATH;
	size_t rc;
	
	dwData = EVENTLOG_ERROR_TYPE | EVENTLOG_WARNING_TYPE | EVENTLOG_INFORMATION_TYPE; 

	// Create the event source as a subkey of the log. 
	rc = snprintf(szBuf,cchSize,"SYSTEM\\CurrentControlSet\\Services\\EventLog\\%s\\%s",
	  logName, sourceName); 
	if (rc >= cchSize) 
		return luaL_error(L,"Log name too long (max is %d): "
		"'SYSTEM\\CurrentControlSet\\Services\\EventLog\\%s\\%s' needs truncation"
		" up to '%s'", MAX_PATH,logName,sourceName,szBuf);
 
	if (RegCreateKeyEx(HKEY_LOCAL_MACHINE, szBuf, 0, NULL, REG_OPTION_NON_VOLATILE,
	    KEY_WRITE, NULL, &hk, &dwDisp)) {
		return luaL_error(L,"Could not create the registry key '%s'", szBuf); 
	}
	
	// Set the name of the message file (contains the mesage manifest)
	if (RegSetValueEx(hk, "EventMessageFile", 0, REG_EXPAND_SZ, 
	    (LPBYTE) dllName, (DWORD) (lstrlen(dllName)+1)*sizeof(TCHAR))) {
		RegCloseKey(hk); 
		return luaL_error(L,"Could not set the event msg file to '%s'",dllName); 
	}
	
	// Set the supported event types. 
	if (RegSetValueEx(hk,"TypesSupported",0,REG_DWORD,(LPBYTE)&dwData,sizeof(DWORD))){
		RegCloseKey(hk); 
		return luaL_error(L,"Could not set the supported msg types."); 
	}
 
	
	// Set the category message file and number of categories.
	if (RegSetValueEx(hk, "CategoryMessageFile", 0, REG_EXPAND_SZ,
	    (LPBYTE) dllName, (DWORD) (lstrlen(dllName)+1)*sizeof(TCHAR))) {
		RegCloseKey(hk); 
		return luaL_error(L,"Couldn't set the category msg file to '%s'",dllName); 
	}
	if (RegSetValueEx(hk,"CategoryCount",0,REG_DWORD,
	    (LPBYTE)&dwCategoryNum,sizeof(DWORD))) {
		RegCloseKey(hk); 
		return luaL_error(L,"Could not set the category count to %d",dwCategoryNum);
	}
	
	RegCloseKey(hk);

	return 0;
}

static int lwel_log(lua_State * L) {
	HANDLE h = *(HANDLE*)luaL_checkudata(L,lua_upvalueindex(1),ES_METATABLE);
	const char * msg = lua_tostring(L,2);
	DWORD dwEventID;
	WORD wType;	
	
	if (cmp_level(L,"DEBUG",1)) {
		wType = EVENTLOG_INFORMATION_TYPE;
		dwEventID = MSG_DEBUG;
	} else if (cmp_level(L,"INFO",1)) {
		wType = EVENTLOG_INFORMATION_TYPE;
		dwEventID = MSG_INFO;
	} else if (cmp_level(L,"WARN",1)) {
		wType = EVENTLOG_WARNING_TYPE;
		dwEventID = MSG_WARN;
	} else if (cmp_level(L,"ERROR",1)) {
		wType = EVENTLOG_ERROR_TYPE;
		dwEventID = MSG_ERROR;
	} else if (cmp_level(L,"FATAL",1)) {
		wType = EVENTLOG_ERROR_TYPE;
		dwEventID = MSG_FATAL;
	} else {
		return luaL_error(L,"logging with an unknown level");
	}
 
	if (!ReportEvent(h,wType,0,dwEventID, NULL,1,0,&msg,NULL)) {
		return luaL_error(L,"Cannot report the event"); 
	}
 
	lua_pushboolean(L,1);
	return 1;
}

static int lwel_gc(lua_State* L){
	DeregisterEventSource(*(HANDLE*)luaL_checkudata(L,1,ES_METATABLE));
	return 0;
}

static int new_lwel(lua_State* L) {

	HANDLE h,*p;

	const char *logName = luaL_optstring(L,1,"FreePOPs");
	const char *sourceName = luaL_optstring(L,2,"UnknownModule");
	const char *dllName = luaL_optstring(L,3,"%SystemRoot%\\System32\\lwel-manifest.dll"); 

	
	lwel_reg(L, logName, sourceName, dllName);
	lua_pop(L,3);
	

	// Get a handle to the event log.
	h = RegisterEventSource(NULL, sourceName);
	if (h == NULL) 
		return luaL_error(L,"Cannot register the event source."); 

	p = lua_newuserdata(L,sizeof(HANDLE));
	*p = h;
	luaL_getmetatable(L,ES_METATABLE);
	lua_setmetatable(L,-2);
	lua_pushcclosure(L,lwel_log,1);

	return 1;
}

static const struct luaL_Reg log_f [] = {
  {"new",new_lwel},	
  {NULL,NULL}
};

int luaopen_wel_core(lua_State* L){

	luaL_newmetatable(L,ES_METATABLE);
	lua_pushcfunction(L,lwel_gc);
	lua_setfield(L,-2,"__gc");
	lua_pop(L,1);
	
	luaL_register(L,"wel.core",log_f);

	return 1;
}

#endif
