{
  Copyright 2002-2018 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{$ifdef read_interface}

  TPixelTextureNode = class;

  { Cache for resources not specific to renderer (OpenGL).
    Includes all TTexturesVideosCache resources (texture, movie
    data) and adds cache for 3D models. }
  TX3DFontTexturesCache = class(TX3DNodesCache)
  strict private
    type
      { @exclude Internal for TX3DFontTexturesCache. }
      TCachedFontTexture = class
      private
        Font: TTextureFontData;
        Blending: Boolean;
        References: Cardinal;
        FontTexture: TPixelTextureNode;
      end;

      TCachedFontTextureList = specialize TObjectList<TCachedFontTexture>;

    var
      CachedFontTextures: TCachedFontTextureList;
  public
    constructor Create;
    destructor Destroy; override;

    { Return TPixelTextureNode to draw given font configuration. }
    function LoadFontTexture(const Font: TTextureFontData;
      const Blending: boolean): TPixelTextureNode;

    procedure FreeFontTexture(var FontTexture: TPixelTextureNode);

    function Empty: boolean; override;
  end;

{$endif read_interface}

{$ifdef read_implementation}

{ TX3DFontTexturesCache ------------------------------------------------------------ }

{ $define DEBUG_FONT_TEXTURE_CACHE}

constructor TX3DFontTexturesCache.Create;
begin
  inherited;
  CachedFontTextures := TCachedFontTextureList.Create;
end;

destructor TX3DFontTexturesCache.Destroy;
begin
  if CachedFontTextures <> nil then
  begin
    Assert(CachedFontTextures.Count = 0, ' Some references to font textures still exist when freeing TX3DFontTexturesCache');
    FreeAndNil(CachedFontTextures);
  end;
  inherited;
end;

function TX3DFontTexturesCache.LoadFontTexture(const Font: TTextureFontData;
  const Blending: boolean): TPixelTextureNode;
var
  I: Integer;
  C: TCachedFontTexture;
begin
  for I := 0 to CachedFontTextures.Count - 1 do
  begin
    C := CachedFontTextures[I];
    if (C.Font = Font) and
       (C.Blending = Blending) then
    begin
      Inc(C.References);

      {$ifdef DEBUG_FONT_TEXTURE_CACHE}
      Writeln('++ : font texture ', Font.Image.URL, ', blending ', Blending, ' : ', C.References);
      {$endif}

      Exit(C.FontTexture);
    end;
  end;

  Result := TPixelTextureNode.Create;
  Result.KeepExistingBegin; // never free it using ref-counting
  { although repeat=TRUE seems counter-intuitive, in fact:
    - We *can* use it, since CastleTextureFontData surrounds each glyph from
      the right and top with GlyphPadding, so we will not accidentally pick
      other glyph by slightly going outside the texture area.
    - We *want* to use it actually, for EXTRA_GLYPH_SPACE. This way glyphs
      on the very left and bottom border of the font texture will also
      have a transparent pixel row to the left / bottom. }
  Result.RepeatS := true;
  Result.RepeatT := true;
  { convert grayscale image with TreatAsAlpha to TGrayscaleAlphaImage }
  Result.FdImage.Value := Font.Image.ToGrayscaleAlphaImage;
  if Blending then
    Result.AlphaChannelField := acBlending
  else
    Result.AlphaChannelField := acTest;

  C := TCachedFontTexture.Create;
  CachedFontTextures.Add(C);
  C.References := 1;
  C.Font := Font;
  C.Blending := Blending;
  C.FontTexture := Result;

  {$ifdef DEBUG_FONT_TEXTURE_CACHE}
  Writeln('++ : font texture ', Font.Image.URL, ', blending ', Blending, ' : ', 1);
  {$endif}
end;

procedure TX3DFontTexturesCache.FreeFontTexture(var FontTexture: TPixelTextureNode);
var
  I: Integer;
  C: TCachedFontTexture;
begin
  if FontTexture = nil then Exit;

  for I := 0 to CachedFontTextures.Count - 1 do
  begin
    C := CachedFontTextures[I];
    if C.FontTexture = FontTexture then
    begin
      {$ifdef DEBUG_FONT_TEXTURE_CACHE}
      Writeln('-- : font texture ', C.Font.Image.URL, ', blending ', C.Blending, ' : ', C.References - 1);
      {$endif}

      FontTexture := nil;

      if C.References = 1 then
      begin
        { Simple FreeAndNil would also be OK, as this node should never
          be a child of another node in X3D graph.
          (Being a TTextNode.FontTextureNode value doesn't make it a normal child,
          i.e. TX3DNode internal child-parent references are not set.)
          But for safety we call FreeIfUnusedAndNil. }
        //FreeIfUnusedAndNil(C.FontTexture);
        { TODO: Contrary to above comment, we need to call FreeAndNil now,
          otherwise it is leaking (which means that FreeIfUnusedAndNil doesn't
          free it). Testcase: "The Unholy Society". Why? }
        FreeAndNil(C.FontTexture);
        CachedFontTextures.Delete(I);
        CheckEmpty;
      end else
        Dec(C.References);

      Exit;
    end;
  end;

  raise EInternalError.CreateFmt('FreeFontTexture: no reference found for font texture %s',
    [PointerToStr(FontTexture)]);
end;

function TX3DFontTexturesCache.Empty: boolean;
begin
  Result := (inherited Empty) and (CachedFontTextures.Count = 0);
end;

{$endif read_implementation}
