/*
 * Copyright (c) 2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>

namespace arm_gemm {

void a64_hybrid_s8s32_mmla_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int32_t> output_arg,
    const int32_t *, Activation, bool accumulate
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 186f\n"
      "cmp %x[M], #0x4\n"
      "bgt 149f\n"
      "beq 112f\n"
      "cmp %x[M], #0x2\n"
      "bgt 75f\n"
      "beq 38f\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "tbz %x[flags], #0, 13f\n"
      "cmp x11, #0x10\n"
      "bge 11f\n"
      "tbz x11, #3, 6f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "tbz x11, #2, 4f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "tbz x11, #1, 3f\n"
      "ldr d16, [x9], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "b 10f\n"
      "3:"  // Height 1: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 10f\n"
      "ldr s16, [x9, #0x0]\n"
      "b 10f\n"
      "4:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x11, #1, 5f\n"
      "ldr d11, [x9], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "b 10f\n"
      "5:"  // Height 1: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 10f\n"
      "ldr s11, [x9, #0x0]\n"
      "b 10f\n"
      "6:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x11, #2, 8f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "tbz x11, #1, 7f\n"
      "ldr d10, [x9], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "b 10f\n"
      "7:"  // Height 1: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 10f\n"
      "ldr s10, [x9, #0x0]\n"
      "b 10f\n"
      "8:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x11, #1, 9f\n"
      "ldr d9, [x9], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x11, #0, 10f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "b 10f\n"
      "9:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "mov x25, #0x0\n"
      "10:"  // Height 1: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 12f\n"
      "11:"  // Height 1: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "12:"  // Height 1: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "b 14f\n"
      "13:"  // Height 1: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "14:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "15:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 16f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 17f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 17f\n"
      "16:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "17:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "blt 20f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q7, [x10, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 19f\n"
      "18:"  // Height 1: Multiply loop: Main loop head
      "trn1 v19.2d, v1.2d, v20.2d\n"
      ".inst 0x4e87a668  // smmla v8.4s, v19.16b, v7.16b\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x4e86a66c  // smmla v12.4s, v19.16b, v6.16b\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x4e92a669  // smmla v9.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x4e91a66d  // smmla v13.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e92a66a  // smmla v10.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x4e91a66e  // smmla v14.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x70]\n"
      "trn2 v1.2d, v1.2d, v20.2d\n"
      ".inst 0x4e92a66b  // smmla v11.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x80]\n"
      ".inst 0x4e91a66f  // smmla v15.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x90]\n"
      ".inst 0x4e92a428  // smmla v8.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xa0]\n"
      ".inst 0x4e91a42c  // smmla v12.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xb0]\n"
      ".inst 0x4e92a429  // smmla v9.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xc0]\n"
      ".inst 0x4e91a42d  // smmla v13.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xd0]\n"
      ".inst 0x4e92a42a  // smmla v10.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xe0]\n"
      ".inst 0x4e91a42e  // smmla v14.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xf0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e92a42b  // smmla v11.4s, v1.16b, v18.16b\n"
      ".inst 0x4e91a42f  // smmla v15.4s, v1.16b, v17.16b\n"
      "ldr q1, [x26, #0x0]\n"
      "add x10, x10, #0x100\n"
      "ldr q7, [x10, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "bge 18b\n"
      "19:"  // Height 1: Multiply loop: Single iteration only
      "trn1 v20.2d, v1.2d, v21.2d\n"
      ".inst 0x4e87a688  // smmla v8.4s, v20.16b, v7.16b\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x4e86a68c  // smmla v12.4s, v20.16b, v6.16b\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x4e92a689  // smmla v9.4s, v20.16b, v18.16b\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x4e91a68d  // smmla v13.4s, v20.16b, v17.16b\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e92a68a  // smmla v10.4s, v20.16b, v18.16b\n"
      "ldr q19, [x10, #0x60]\n"
      ".inst 0x4e91a68e  // smmla v14.4s, v20.16b, v17.16b\n"
      "ldr q18, [x10, #0x70]\n"
      "trn2 v1.2d, v1.2d, v21.2d\n"
      ".inst 0x4e93a68b  // smmla v11.4s, v20.16b, v19.16b\n"
      "ldr q17, [x10, #0x80]\n"
      ".inst 0x4e92a68f  // smmla v15.4s, v20.16b, v18.16b\n"
      "ldr q19, [x10, #0x90]\n"
      ".inst 0x4e91a428  // smmla v8.4s, v1.16b, v17.16b\n"
      "ldr q18, [x10, #0xa0]\n"
      ".inst 0x4e93a42c  // smmla v12.4s, v1.16b, v19.16b\n"
      "ldr q17, [x10, #0xb0]\n"
      ".inst 0x4e92a429  // smmla v9.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xc0]\n"
      ".inst 0x4e91a42d  // smmla v13.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xd0]\n"
      ".inst 0x4e92a42a  // smmla v10.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xe0]\n"
      ".inst 0x4e91a42e  // smmla v14.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xf0]\n"
      "add x26, x26, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e92a42b  // smmla v11.4s, v1.16b, v18.16b\n"
      ".inst 0x4e91a42f  // smmla v15.4s, v1.16b, v17.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "add x10, x10, #0x100\n"
      "20:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 27f\n"
      "cmp x27, #0x8\n"
      "blt 22f\n"
      "21:"  // Height 1: Multiply loop: Odd block loop
      "ldr d19, [x26], #0x8\n"
      "ldr q18, [x10, #0x0]\n"
      "trn1 v19.2d, v19.2d, v17.2d\n"
      "ldr q17, [x10, #0x10]\n"
      ".inst 0x4e92a668  // smmla v8.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x4e91a66c  // smmla v12.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x4e92a669  // smmla v9.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x4e91a66d  // smmla v13.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e92a66a  // smmla v10.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x4e91a66e  // smmla v14.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x70]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e92a66b  // smmla v11.4s, v19.16b, v18.16b\n"
      ".inst 0x4e91a66f  // smmla v15.4s, v19.16b, v17.16b\n"
      "add x10, x10, #0x80\n"
      "bge 21b\n"
      "22:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x27, 27f\n"
      "tbz x27, #2, 24f\n"
      "ldr s1, [x26], #0x4\n"
      "tbz x27, #1, 23f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "tbz x27, #0, 26f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "b 26f\n"
      "23:"  // Height 1: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 26f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "b 26f\n"
      "24:"  // Height 1: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 25f\n"
      "ldr h1, [x26], #0x2\n"
      "tbz x27, #0, 26f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "b 26f\n"
      "25:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "26:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q23, [x10, #0x0]\n"
      "ldr q18, [x10, #0x10]\n"
      "trn1 v19.2d, v1.2d, v17.2d\n"
      ".inst 0x4e97a668  // smmla v8.4s, v19.16b, v23.16b\n"
      "ldr q17, [x10, #0x20]\n"
      ".inst 0x4e92a66c  // smmla v12.4s, v19.16b, v18.16b\n"
      "ldr q31, [x10, #0x30]\n"
      ".inst 0x4e91a669  // smmla v9.4s, v19.16b, v17.16b\n"
      "ldr q20, [x10, #0x40]\n"
      ".inst 0x4e9fa66d  // smmla v13.4s, v19.16b, v31.16b\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e94a66a  // smmla v10.4s, v19.16b, v20.16b\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x4e91a66e  // smmla v14.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x70]\n"
      ".inst 0x4e92a66b  // smmla v11.4s, v19.16b, v18.16b\n"
      ".inst 0x4e91a66f  // smmla v15.4s, v19.16b, v17.16b\n"
      "add x10, x10, #0x80\n"
      "27:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 15b\n"
      "cmp x11, #0x10\n"
      "uzp1 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp1 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v11.2d, v11.2d, v15.2d\n"
      "bge 36f\n"
      "tbz x11, #3, 31f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "tbz x11, #2, 29f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "tbz x11, #1, 28f\n"
      "str d11, [x9], #0x8\n"
      "tbz x11, #0, 35f\n"
      "st1 { v11.s }[2], [x9]\n"
      "b 35f\n"
      "28:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x11, #0, 35f\n"
      "str s11, [x9, #0x0]\n"
      "b 35f\n"
      "29:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x11, #1, 30f\n"
      "str d10, [x9], #0x8\n"
      "tbz x11, #0, 35f\n"
      "st1 { v10.s }[2], [x9]\n"
      "b 35f\n"
      "30:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x11, #0, 35f\n"
      "str s10, [x9, #0x0]\n"
      "b 35f\n"
      "31:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x11, #2, 33f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "tbz x11, #1, 32f\n"
      "str d9, [x9], #0x8\n"
      "tbz x11, #0, 35f\n"
      "st1 { v9.s }[2], [x9]\n"
      "b 35f\n"
      "32:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x11, #0, 35f\n"
      "str s9, [x9, #0x0]\n"
      "b 35f\n"
      "33:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x11, #1, 34f\n"
      "str d8, [x9], #0x8\n"
      "tbz x11, #0, 35f\n"
      "st1 { v8.s }[2], [x9]\n"
      "b 35f\n"
      "34:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "35:"  // Height 1: Partial direct writeback: Done
      "b 37f\n"
      "36:"  // Height 1: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "37:"  // Height 1: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 2b\n"
      "b 224f\n"
      "38:"  // Height 2
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "39:"  // Height 2: Column loop
      "tbz %x[flags], #0, 50f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "bge 48f\n"
      "tbz x11, #3, 43f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "tbz x11, #2, 41f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "tbz x11, #1, 40f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x11, #0, 47f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "b 47f\n"
      "40:"  // Height 2: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 47f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "b 47f\n"
      "41:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x11, #1, 42f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x11, #0, 47f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "b 47f\n"
      "42:"  // Height 2: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 47f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "b 47f\n"
      "43:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x11, #2, 45f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "tbz x11, #1, 44f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x11, #0, 47f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "b 47f\n"
      "44:"  // Height 2: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 47f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "b 47f\n"
      "45:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x11, #1, 46f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x11, #0, 47f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "b 47f\n"
      "46:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "47:"  // Height 2: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 49f\n"
      "48:"  // Height 2: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "49:"  // Height 2: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "b 51f\n"
      "50:"  // Height 2: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "51:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "52:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 53f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 54f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 54f\n"
      "53:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "54:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "blt 57f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q7, [x10, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 56f\n"
      "55:"  // Height 2: Multiply loop: Main loop head
      "trn1 v19.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a668  // smmla v8.4s, v19.16b, v7.16b\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x4e86a66c  // smmla v12.4s, v19.16b, v6.16b\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x4e92a669  // smmla v9.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x4e91a66d  // smmla v13.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e92a66a  // smmla v10.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x4e91a66e  // smmla v14.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x70]\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e92a66b  // smmla v11.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x80]\n"
      ".inst 0x4e91a66f  // smmla v15.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x90]\n"
      ".inst 0x4e92a428  // smmla v8.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xa0]\n"
      ".inst 0x4e91a42c  // smmla v12.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xb0]\n"
      ".inst 0x4e92a429  // smmla v9.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xc0]\n"
      ".inst 0x4e91a42d  // smmla v13.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xd0]\n"
      ".inst 0x4e92a42a  // smmla v10.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xe0]\n"
      ".inst 0x4e91a42e  // smmla v14.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xf0]\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e92a42b  // smmla v11.4s, v1.16b, v18.16b\n"
      "add x10, x10, #0x100\n"
      "ldr q7, [x10, #0x0]\n"
      ".inst 0x4e91a42f  // smmla v15.4s, v1.16b, v17.16b\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "bge 55b\n"
      "56:"  // Height 2: Multiply loop: Single iteration only
      "trn1 v19.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a668  // smmla v8.4s, v19.16b, v7.16b\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x4e86a66c  // smmla v12.4s, v19.16b, v6.16b\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x4e92a669  // smmla v9.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x4e91a66d  // smmla v13.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e92a66a  // smmla v10.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x4e91a66e  // smmla v14.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x70]\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e92a66b  // smmla v11.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x80]\n"
      ".inst 0x4e91a66f  // smmla v15.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x90]\n"
      ".inst 0x4e92a428  // smmla v8.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xa0]\n"
      ".inst 0x4e91a42c  // smmla v12.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xb0]\n"
      ".inst 0x4e92a429  // smmla v9.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xc0]\n"
      ".inst 0x4e91a42d  // smmla v13.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xd0]\n"
      ".inst 0x4e92a42a  // smmla v10.4s, v1.16b, v18.16b\n"
      "ldr q18, [x10, #0xe0]\n"
      ".inst 0x4e91a42e  // smmla v14.4s, v1.16b, v17.16b\n"
      "ldr q17, [x10, #0xf0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e92a42b  // smmla v11.4s, v1.16b, v18.16b\n"
      ".inst 0x4e91a42f  // smmla v15.4s, v1.16b, v17.16b\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "add x10, x10, #0x100\n"
      "57:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 64f\n"
      "cmp x27, #0x8\n"
      "blt 59f\n"
      "58:"  // Height 2: Multiply loop: Odd block loop
      "ldr d18, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "trn1 v19.2d, v18.2d, v17.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr q17, [x10, #0x0]\n"
      "ldr q22, [x10, #0x10]\n"
      ".inst 0x4e91a668  // smmla v8.4s, v19.16b, v17.16b\n"
      ".inst 0x4e96a66c  // smmla v12.4s, v19.16b, v22.16b\n"
      "ldr q1, [x10, #0x20]\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x4e81a669  // smmla v9.4s, v19.16b, v1.16b\n"
      ".inst 0x4e91a66d  // smmla v13.4s, v19.16b, v17.16b\n"
      "ldr q18, [x10, #0x40]\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e92a66a  // smmla v10.4s, v19.16b, v18.16b\n"
      ".inst 0x4e91a66e  // smmla v14.4s, v19.16b, v17.16b\n"
      "ldr q18, [x10, #0x60]\n"
      "ldr q17, [x10, #0x70]\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e92a66b  // smmla v11.4s, v19.16b, v18.16b\n"
      ".inst 0x4e91a66f  // smmla v15.4s, v19.16b, v17.16b\n"
      "add x10, x10, #0x80\n"
      "bge 58b\n"
      "59:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x27, 64f\n"
      "tbz x27, #2, 61f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "tbz x27, #1, 60f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "tbz x27, #0, 63f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "b 63f\n"
      "60:"  // Height 2: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 63f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "b 63f\n"
      "61:"  // Height 2: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 62f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "tbz x27, #0, 63f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "b 63f\n"
      "62:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "63:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q18, [x10, #0x0]\n"
      "ldr q17, [x10, #0x10]\n"
      "trn1 v19.2d, v1.2d, v2.2d\n"
      ".inst 0x4e92a668  // smmla v8.4s, v19.16b, v18.16b\n"
      "ldr q5, [x10, #0x20]\n"
      ".inst 0x4e91a66c  // smmla v12.4s, v19.16b, v17.16b\n"
      "ldr q21, [x10, #0x30]\n"
      ".inst 0x4e85a669  // smmla v9.4s, v19.16b, v5.16b\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x4e95a66d  // smmla v13.4s, v19.16b, v21.16b\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x4e92a66a  // smmla v10.4s, v19.16b, v18.16b\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x4e91a66e  // smmla v14.4s, v19.16b, v17.16b\n"
      "ldr q17, [x10, #0x70]\n"
      ".inst 0x4e92a66b  // smmla v11.4s, v19.16b, v18.16b\n"
      ".inst 0x4e91a66f  // smmla v15.4s, v19.16b, v17.16b\n"
      "add x10, x10, #0x80\n"
      "64:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 52b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "cmp x11, #0x10\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "bge 73f\n"
      "tbz x11, #3, 68f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "tbz x11, #2, 66f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "tbz x11, #1, 65f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "tbz x11, #0, 72f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "b 72f\n"
      "65:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x11, #0, 72f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x24, #0x0]\n"
      "b 72f\n"
      "66:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x11, #1, 67f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "tbz x11, #0, 72f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "b 72f\n"
      "67:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x11, #0, 72f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x24, #0x0]\n"
      "b 72f\n"
      "68:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x11, #2, 70f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "tbz x11, #1, 69f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "tbz x11, #0, 72f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "b 72f\n"
      "69:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x11, #0, 72f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x24, #0x0]\n"
      "b 72f\n"
      "70:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x11, #1, 71f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "tbz x11, #0, 72f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "b 72f\n"
      "71:"  // Height 2: Partial direct writeback: partial_1_0
      "str s7, [x9, #0x0]\n"
      "str s8, [x24, #0x0]\n"
      "72:"  // Height 2: Partial direct writeback: Done
      "b 74f\n"
      "73:"  // Height 2: Full writeback
      "str q7, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0x0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "74:"  // Height 2: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 39b\n"
      "b 224f\n"
      "75:"  // Height 3
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "76:"  // Height 3: Column loop
      "tbz %x[flags], #0, 87f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "cmp x11, #0x10\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 85f\n"
      "tbz x11, #3, 80f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "tbz x11, #2, 78f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "tbz x11, #1, 77f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "tbz x11, #0, 84f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "b 84f\n"
      "77:"  // Height 3: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 84f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s24, [x23, #0x0]\n"
      "b 84f\n"
      "78:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x11, #1, 79f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "tbz x11, #0, 84f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "b 84f\n"
      "79:"  // Height 3: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 84f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "b 84f\n"
      "80:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x11, #2, 82f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "tbz x11, #1, 81f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "tbz x11, #0, 84f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "b 84f\n"
      "81:"  // Height 3: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 84f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "b 84f\n"
      "82:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x11, #1, 83f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "tbz x11, #0, 84f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "b 84f\n"
      "83:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s17, [x23, #0x0]\n"
      "84:"  // Height 3: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 86f\n"
      "85:"  // Height 3: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0x0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "86:"  // Height 3: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "b 88f\n"
      "87:"  // Height 3: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "88:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "89:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 90f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 91f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 91f\n"
      "90:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "91:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "blt 94f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q7, [x10, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 93f\n"
      "92:"  // Height 3: Multiply loop: Main loop head
      "trn1 v28.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a788  // smmla v8.4s, v28.16b, v7.16b\n"
      "trn1 v27.2d, v3.2d, v29.2d\n"
      ".inst 0x4e87a770  // smmla v16.4s, v27.16b, v7.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e86a78c  // smmla v12.4s, v28.16b, v6.16b\n"
      ".inst 0x4e86a774  // smmla v20.4s, v27.16b, v6.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      "trn2 v3.2d, v3.2d, v29.2d\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x80]\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x90]\n"
      "ldr q2, [x25, #0x0]\n"
      ".inst 0x4e9aa428  // smmla v8.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa470  // smmla v16.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xa0]\n"
      ".inst 0x4e99a42c  // smmla v12.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e99a474  // smmla v20.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xb0]\n"
      ".inst 0x4e9aa429  // smmla v9.4s, v1.16b, v26.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e9aa471  // smmla v17.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xc0]\n"
      ".inst 0x4e99a42d  // smmla v13.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e99a475  // smmla v21.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xd0]\n"
      ".inst 0x4e9aa42a  // smmla v10.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa472  // smmla v18.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xe0]\n"
      ".inst 0x4e99a42e  // smmla v14.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a476  // smmla v22.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e9aa42b  // smmla v11.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa473  // smmla v19.4s, v3.16b, v26.16b\n"
      "ldr q7, [x10, #0x0]\n"
      ".inst 0x4e99a42f  // smmla v15.4s, v1.16b, v25.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e99a477  // smmla v23.4s, v3.16b, v25.16b\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      "bge 92b\n"
      "93:"  // Height 3: Multiply loop: Single iteration only
      "trn1 v28.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a788  // smmla v8.4s, v28.16b, v7.16b\n"
      "trn1 v27.2d, v3.2d, v29.2d\n"
      ".inst 0x4e87a770  // smmla v16.4s, v27.16b, v7.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e86a78c  // smmla v12.4s, v28.16b, v6.16b\n"
      ".inst 0x4e86a774  // smmla v20.4s, v27.16b, v6.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      "trn2 v3.2d, v3.2d, v29.2d\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x80]\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x90]\n"
      ".inst 0x4e9aa428  // smmla v8.4s, v1.16b, v26.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e9aa470  // smmla v16.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xa0]\n"
      ".inst 0x4e99a42c  // smmla v12.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e99a474  // smmla v20.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xb0]\n"
      ".inst 0x4e9aa429  // smmla v9.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa471  // smmla v17.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xc0]\n"
      ".inst 0x4e99a42d  // smmla v13.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a475  // smmla v21.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xd0]\n"
      ".inst 0x4e9aa42a  // smmla v10.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa472  // smmla v18.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xe0]\n"
      ".inst 0x4e99a42e  // smmla v14.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a476  // smmla v22.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e9aa42b  // smmla v11.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa473  // smmla v19.4s, v3.16b, v26.16b\n"
      ".inst 0x4e99a42f  // smmla v15.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a477  // smmla v23.4s, v3.16b, v25.16b\n"
      "94:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 101f\n"
      "cmp x27, #0x8\n"
      "blt 96f\n"
      "95:"  // Height 3: Multiply loop: Odd block loop
      "ldr d26, [x26], #0x8\n"
      "ldr d25, [x25], #0x8\n"
      "trn1 v28.2d, v26.2d, v25.2d\n"
      "ldr d25, [x24], #0x8\n"
      "ldr q26, [x10, #0x0]\n"
      "trn1 v27.2d, v25.2d, v27.2d\n"
      ".inst 0x4e9aa788  // smmla v8.4s, v28.16b, v26.16b\n"
      "ldr q25, [x10, #0x10]\n"
      ".inst 0x4e9aa770  // smmla v16.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e99a78c  // smmla v12.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a774  // smmla v20.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "bge 95b\n"
      "96:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x27, 101f\n"
      "tbz x27, #2, 98f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "tbz x27, #1, 97f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "tbz x27, #0, 100f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "b 100f\n"
      "97:"  // Height 3: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 100f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "b 100f\n"
      "98:"  // Height 3: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 99f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "tbz x27, #0, 100f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "b 100f\n"
      "99:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "100:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q26, [x10, #0x0]\n"
      "ldr q29, [x10, #0x10]\n"
      "trn1 v28.2d, v1.2d, v2.2d\n"
      "trn1 v27.2d, v3.2d, v25.2d\n"
      ".inst 0x4e9aa788  // smmla v8.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa770  // smmla v16.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e9da78c  // smmla v12.4s, v28.16b, v29.16b\n"
      ".inst 0x4e9da774  // smmla v20.4s, v27.16b, v29.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "101:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 89b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "cmp x11, #0x10\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "uzp1 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v19.2d, v19.2d, v23.2d\n"
      "bge 110f\n"
      "tbz x11, #3, 105f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "tbz x11, #2, 103f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "tbz x11, #1, 102f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "tbz x11, #0, 109f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "b 109f\n"
      "102:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x11, #0, 109f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x24, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "b 109f\n"
      "103:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x11, #1, 104f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "tbz x11, #0, 109f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "b 109f\n"
      "104:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x11, #0, 109f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x24, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "b 109f\n"
      "105:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x11, #2, 107f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "tbz x11, #1, 106f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "tbz x11, #0, 109f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "b 109f\n"
      "106:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x11, #0, 109f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x24, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "b 109f\n"
      "107:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x11, #1, 108f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "tbz x11, #0, 109f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "b 109f\n"
      "108:"  // Height 3: Partial direct writeback: partial_1_0
      "str s7, [x9, #0x0]\n"
      "str s8, [x24, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "109:"  // Height 3: Partial direct writeback: Done
      "b 111f\n"
      "110:"  // Height 3: Full writeback
      "str q7, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0x0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q16, [x23, #0x0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "111:"  // Height 3: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 76b\n"
      "b 224f\n"
      "112:"  // Height 4
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "113:"  // Height 4: Column loop
      "tbz %x[flags], #0, 124f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "cmp x11, #0x10\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 122f\n"
      "tbz x11, #3, 117f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "tbz x11, #2, 115f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "tbz x11, #1, 114f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "tbz x11, #0, 121f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "b 121f\n"
      "114:"  // Height 4: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 121f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s24, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "b 121f\n"
      "115:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x11, #1, 116f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "tbz x11, #0, 121f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "b 121f\n"
      "116:"  // Height 4: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 121f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "b 121f\n"
      "117:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x11, #2, 119f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "tbz x11, #1, 118f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "tbz x11, #0, 121f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "b 121f\n"
      "118:"  // Height 4: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 121f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "b 121f\n"
      "119:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x11, #1, 120f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "tbz x11, #0, 121f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "b 121f\n"
      "120:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "121:"  // Height 4: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 123f\n"
      "122:"  // Height 4: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0x0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "123:"  // Height 4: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "b 125f\n"
      "124:"  // Height 4: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "125:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "126:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 127f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 128f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 128f\n"
      "127:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "128:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "blt 131f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q4, [x23, #0x0]\n"
      "ldr q7, [x10, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 130f\n"
      "129:"  // Height 4: Multiply loop: Main loop head
      "trn1 v28.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a788  // smmla v8.4s, v28.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "trn1 v27.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a770  // smmla v16.4s, v27.16b, v7.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e86a78c  // smmla v12.4s, v28.16b, v6.16b\n"
      ".inst 0x4e86a774  // smmla v20.4s, v27.16b, v6.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      "add x23, x23, #0x10\n"
      "ldr q4, [x23, #0x0]\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x80]\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x90]\n"
      "ldr q2, [x25, #0x0]\n"
      ".inst 0x4e9aa428  // smmla v8.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa470  // smmla v16.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xa0]\n"
      ".inst 0x4e99a42c  // smmla v12.4s, v1.16b, v25.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e99a474  // smmla v20.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xb0]\n"
      ".inst 0x4e9aa429  // smmla v9.4s, v1.16b, v26.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e9aa471  // smmla v17.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xc0]\n"
      ".inst 0x4e99a42d  // smmla v13.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e99a475  // smmla v21.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xd0]\n"
      ".inst 0x4e9aa42a  // smmla v10.4s, v1.16b, v26.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e9aa472  // smmla v18.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xe0]\n"
      ".inst 0x4e99a42e  // smmla v14.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e99a476  // smmla v22.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e9aa42b  // smmla v11.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa473  // smmla v19.4s, v3.16b, v26.16b\n"
      "ldr q7, [x10, #0x0]\n"
      ".inst 0x4e99a42f  // smmla v15.4s, v1.16b, v25.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e99a477  // smmla v23.4s, v3.16b, v25.16b\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      "bge 129b\n"
      "130:"  // Height 4: Multiply loop: Single iteration only
      "trn1 v28.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a788  // smmla v8.4s, v28.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v27.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a770  // smmla v16.4s, v27.16b, v7.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e86a78c  // smmla v12.4s, v28.16b, v6.16b\n"
      ".inst 0x4e86a774  // smmla v20.4s, v27.16b, v6.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x80]\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x90]\n"
      ".inst 0x4e9aa428  // smmla v8.4s, v1.16b, v26.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e9aa470  // smmla v16.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xa0]\n"
      ".inst 0x4e99a42c  // smmla v12.4s, v1.16b, v25.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e99a474  // smmla v20.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xb0]\n"
      ".inst 0x4e9aa429  // smmla v9.4s, v1.16b, v26.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e9aa471  // smmla v17.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xc0]\n"
      ".inst 0x4e99a42d  // smmla v13.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a475  // smmla v21.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xd0]\n"
      ".inst 0x4e9aa42a  // smmla v10.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa472  // smmla v18.4s, v3.16b, v26.16b\n"
      "ldr q26, [x10, #0xe0]\n"
      ".inst 0x4e99a42e  // smmla v14.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a476  // smmla v22.4s, v3.16b, v25.16b\n"
      "ldr q25, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e9aa42b  // smmla v11.4s, v1.16b, v26.16b\n"
      ".inst 0x4e9aa473  // smmla v19.4s, v3.16b, v26.16b\n"
      ".inst 0x4e99a42f  // smmla v15.4s, v1.16b, v25.16b\n"
      ".inst 0x4e99a477  // smmla v23.4s, v3.16b, v25.16b\n"
      "131:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 138f\n"
      "cmp x27, #0x8\n"
      "blt 133f\n"
      "132:"  // Height 4: Multiply loop: Odd block loop
      "ldr d26, [x26], #0x8\n"
      "ldr d25, [x25], #0x8\n"
      "trn1 v28.2d, v26.2d, v25.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr d26, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "trn1 v27.2d, v26.2d, v25.2d\n"
      "cmp x27, #0x8\n"
      "ldr q26, [x10, #0x0]\n"
      "ldr q25, [x10, #0x10]\n"
      ".inst 0x4e9aa788  // smmla v8.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa770  // smmla v16.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e99a78c  // smmla v12.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a774  // smmla v20.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "bge 132b\n"
      "133:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x27, 138f\n"
      "tbz x27, #2, 135f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "tbz x27, #1, 134f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "tbz x27, #0, 137f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "b 137f\n"
      "134:"  // Height 4: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 137f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "b 137f\n"
      "135:"  // Height 4: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 136f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "tbz x27, #0, 137f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "b 137f\n"
      "136:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "ldr b4, [x23, #0x0]\n"
      "137:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q26, [x10, #0x0]\n"
      "ldr q25, [x10, #0x10]\n"
      "trn1 v28.2d, v1.2d, v2.2d\n"
      "trn1 v27.2d, v3.2d, v4.2d\n"
      ".inst 0x4e9aa788  // smmla v8.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa770  // smmla v16.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x4e99a78c  // smmla v12.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a774  // smmla v20.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x4e9aa789  // smmla v9.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa771  // smmla v17.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x4e99a78d  // smmla v13.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a775  // smmla v21.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x4e9aa78a  // smmla v10.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa772  // smmla v18.4s, v27.16b, v26.16b\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x4e99a78e  // smmla v14.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a776  // smmla v22.4s, v27.16b, v25.16b\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e9aa78b  // smmla v11.4s, v28.16b, v26.16b\n"
      ".inst 0x4e9aa773  // smmla v19.4s, v27.16b, v26.16b\n"
      ".inst 0x4e99a78f  // smmla v15.4s, v28.16b, v25.16b\n"
      ".inst 0x4e99a777  // smmla v23.4s, v27.16b, v25.16b\n"
      "138:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 126b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "add x22, x23, x20, LSL #2\n"
      "cmp x11, #0x10\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "bge 147f\n"
      "tbz x11, #3, 142f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v17.4s }, [x22], #0x10\n"
      "tbz x11, #2, 140f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x23], #0x10\n"
      "st1 { v18.4s }, [x22], #0x10\n"
      "tbz x11, #1, 139f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d22, [x23], #0x8\n"
      "str d19, [x22], #0x8\n"
      "tbz x11, #0, 146f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v22.s }[2], [x23]\n"
      "st1 { v19.s }[2], [x22]\n"
      "b 146f\n"
      "139:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x11, #0, 146f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x24, #0x0]\n"
      "str s22, [x23, #0x0]\n"
      "str s19, [x22, #0x0]\n"
      "b 146f\n"
      "140:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x11, #1, 141f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d21, [x23], #0x8\n"
      "str d18, [x22], #0x8\n"
      "tbz x11, #0, 146f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v21.s }[2], [x23]\n"
      "st1 { v18.s }[2], [x22]\n"
      "b 146f\n"
      "141:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x11, #0, 146f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x24, #0x0]\n"
      "str s21, [x23, #0x0]\n"
      "str s18, [x22, #0x0]\n"
      "b 146f\n"
      "142:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x11, #2, 144f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "tbz x11, #1, 143f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d17, [x22], #0x8\n"
      "tbz x11, #0, 146f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v20.s }[2], [x23]\n"
      "st1 { v17.s }[2], [x22]\n"
      "b 146f\n"
      "143:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x11, #0, 146f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x24, #0x0]\n"
      "str s20, [x23, #0x0]\n"
      "str s17, [x22, #0x0]\n"
      "b 146f\n"
      "144:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x11, #1, 145f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d16, [x22], #0x8\n"
      "tbz x11, #0, 146f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v16.s }[2], [x22]\n"
      "b 146f\n"
      "145:"  // Height 4: Partial direct writeback: partial_1_0
      "str s7, [x9, #0x0]\n"
      "str s8, [x24, #0x0]\n"
      "str s15, [x23, #0x0]\n"
      "str s16, [x22, #0x0]\n"
      "146:"  // Height 4: Partial direct writeback: Done
      "b 148f\n"
      "147:"  // Height 4: Full writeback
      "str q7, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0x0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q15, [x23, #0x0]\n"
      "str q20, [x23, #0x10]\n"
      "str q21, [x23, #0x20]\n"
      "str q22, [x23, #0x30]\n"
      "str q16, [x22, #0x0]\n"
      "str q17, [x22, #0x10]\n"
      "str q18, [x22, #0x20]\n"
      "str q19, [x22, #0x30]\n"
      "148:"  // Height 4: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 113b\n"
      "b 224f\n"
      "149:"  // Height 5
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "150:"  // Height 5: Column loop
      "tbz %x[flags], #0, 161f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "cmp x11, #0x10\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 159f\n"
      "tbz x11, #3, 154f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "tbz x11, #2, 152f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "tbz x11, #1, 151f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d6, [x21], #0x8\n"
      "tbz x11, #0, 158f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v6.s }[2], [x21]\n"
      "b 158f\n"
      "151:"  // Height 5: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 158f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s24, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s6, [x21, #0x0]\n"
      "b 158f\n"
      "152:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x11, #1, 153f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "tbz x11, #0, 158f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "b 158f\n"
      "153:"  // Height 5: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 158f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "b 158f\n"
      "154:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x11, #2, 156f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "tbz x11, #1, 155f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "tbz x11, #0, 158f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "b 158f\n"
      "155:"  // Height 5: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 158f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "b 158f\n"
      "156:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x11, #1, 157f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "tbz x11, #0, 158f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "b 158f\n"
      "157:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "ldr s25, [x21, #0x0]\n"
      "158:"  // Height 5: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 160f\n"
      "159:"  // Height 5: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0x0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q25, [x21, #0x0]\n"
      "ldr q26, [x21, #0x10]\n"
      "ldr q27, [x21, #0x20]\n"
      "ldr q6, [x21, #0x30]\n"
      "160:"  // Height 5: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "zip1 v24.2d, v25.2d, v28.2d\n"
      "zip2 v28.2d, v25.2d, v28.2d\n"
      "zip1 v25.2d, v26.2d, v29.2d\n"
      "zip2 v29.2d, v26.2d, v29.2d\n"
      "zip1 v26.2d, v27.2d, v30.2d\n"
      "zip2 v30.2d, v27.2d, v30.2d\n"
      "zip1 v27.2d, v6.2d, v31.2d\n"
      "zip2 v31.2d, v6.2d, v31.2d\n"
      "b 162f\n"
      "161:"  // Height 5: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "162:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "163:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 164f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 165f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 165f\n"
      "164:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "165:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "blt 168f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q4, [x23, #0x0]\n"
      "ldr q5, [x22, #0x0]\n"
      "ldr q7, [x10, #0x0]\n"
      "blt 167f\n"
      "166:"  // Height 5: Multiply loop: Main loop head
      "trn1 v6.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a4c8  // smmla v8.4s, v6.16b, v7.16b\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "trn1 v4.2d, v5.2d, v0.2d\n"
      "trn2 v5.2d, v5.2d, v0.2d\n"
      "ldr q0, [x10, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e80a4cc  // smmla v12.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a454  // smmla v20.4s, v2.16b, v0.16b\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4e80a49c  // smmla v28.4s, v4.16b, v0.16b\n"
      "ldr q0, [x10, #0x30]\n"
      ".inst 0x4e87a4c9  // smmla v9.4s, v6.16b, v7.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e80a4cd  // smmla v13.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a455  // smmla v21.4s, v2.16b, v0.16b\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4e80a49d  // smmla v29.4s, v4.16b, v0.16b\n"
      "ldr q0, [x10, #0x50]\n"
      ".inst 0x4e87a4ca  // smmla v10.4s, v6.16b, v7.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e80a4ce  // smmla v14.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a456  // smmla v22.4s, v2.16b, v0.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e80a49e  // smmla v30.4s, v4.16b, v0.16b\n"
      "ldr q0, [x10, #0x70]\n"
      ".inst 0x4e87a4cb  // smmla v11.4s, v6.16b, v7.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e80a4cf  // smmla v15.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a457  // smmla v23.4s, v2.16b, v0.16b\n"
      "ldr q2, [x25, #0x0]\n"
      ".inst 0x4e80a49f  // smmla v31.4s, v4.16b, v0.16b\n"
      "ldr q0, [x10, #0x90]\n"
      "ldr q4, [x23, #0x0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4e80a42c  // smmla v12.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bc  // smmla v28.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xb0]\n"
      ".inst 0x4e86a429  // smmla v9.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a471  // smmla v17.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4b9  // smmla v25.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4e80a42d  // smmla v13.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bd  // smmla v29.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xd0]\n"
      ".inst 0x4e86a42a  // smmla v10.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a472  // smmla v18.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4ba  // smmla v26.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4e80a42e  // smmla v14.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4be  // smmla v30.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e86a42b  // smmla v11.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a473  // smmla v19.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bb  // smmla v27.4s, v5.16b, v6.16b\n"
      "ldr q7, [x10, #0x0]\n"
      ".inst 0x4e80a42f  // smmla v15.4s, v1.16b, v0.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      "ldr q3, [x24, #0x0]\n"
      ".inst 0x4e80a4bf  // smmla v31.4s, v5.16b, v0.16b\n"
      "ldr q5, [x22, #0x0]\n"
      "bge 166b\n"
      "167:"  // Height 5: Multiply loop: Single iteration only
      "trn1 v6.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a4c8  // smmla v8.4s, v6.16b, v7.16b\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v4.2d, v5.2d, v0.2d\n"
      "trn2 v5.2d, v5.2d, v0.2d\n"
      "ldr q0, [x10, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e80a4cc  // smmla v12.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a454  // smmla v20.4s, v2.16b, v0.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e80a49c  // smmla v28.4s, v4.16b, v0.16b\n"
      "ldr q0, [x10, #0x30]\n"
      ".inst 0x4e87a4c9  // smmla v9.4s, v6.16b, v7.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e80a4cd  // smmla v13.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a455  // smmla v21.4s, v2.16b, v0.16b\n"
      "add x22, x22, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e80a49d  // smmla v29.4s, v4.16b, v0.16b\n"
      "ldr q0, [x10, #0x50]\n"
      ".inst 0x4e87a4ca  // smmla v10.4s, v6.16b, v7.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e80a4ce  // smmla v14.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a456  // smmla v22.4s, v2.16b, v0.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e80a49e  // smmla v30.4s, v4.16b, v0.16b\n"
      "ldr q0, [x10, #0x70]\n"
      ".inst 0x4e87a4cb  // smmla v11.4s, v6.16b, v7.16b\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e80a4cf  // smmla v15.4s, v6.16b, v0.16b\n"
      ".inst 0x4e80a457  // smmla v23.4s, v2.16b, v0.16b\n"
      ".inst 0x4e80a49f  // smmla v31.4s, v4.16b, v0.16b\n"
      "ldr q2, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q0, [x10, #0xa0]\n"
      ".inst 0x4e82a42c  // smmla v12.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a474  // smmla v20.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4bc  // smmla v28.4s, v5.16b, v2.16b\n"
      "ldr q2, [x10, #0xb0]\n"
      ".inst 0x4e80a429  // smmla v9.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a471  // smmla v17.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4b9  // smmla v25.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xc0]\n"
      ".inst 0x4e82a42d  // smmla v13.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a475  // smmla v21.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4bd  // smmla v29.4s, v5.16b, v2.16b\n"
      "ldr q2, [x10, #0xd0]\n"
      ".inst 0x4e80a42a  // smmla v10.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a472  // smmla v18.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4ba  // smmla v26.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xe0]\n"
      ".inst 0x4e82a42e  // smmla v14.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a476  // smmla v22.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4be  // smmla v30.4s, v5.16b, v2.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e80a42b  // smmla v11.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a473  // smmla v19.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bb  // smmla v27.4s, v5.16b, v0.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bf  // smmla v31.4s, v5.16b, v6.16b\n"
      "168:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 175f\n"
      "cmp x27, #0x8\n"
      "blt 170f\n"
      "169:"  // Height 5: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d0, [x25], #0x8\n"
      "trn1 v4.2d, v1.2d, v0.2d\n"
      "ldr d1, [x24], #0x8\n"
      "ldr d0, [x23], #0x8\n"
      "trn1 v3.2d, v1.2d, v0.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr d0, [x22], #0x8\n"
      "ldr q1, [x10, #0x0]\n"
      "trn1 v2.2d, v0.2d, v2.2d\n"
      ".inst 0x4e81a488  // smmla v8.4s, v4.16b, v1.16b\n"
      "ldr q0, [x10, #0x10]\n"
      ".inst 0x4e81a470  // smmla v16.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x20]\n"
      ".inst 0x4e80a48c  // smmla v12.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      "cmp x27, #0x8\n"
      ".inst 0x4e80a45c  // smmla v28.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x30]\n"
      ".inst 0x4e81a489  // smmla v9.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a471  // smmla v17.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a459  // smmla v25.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x40]\n"
      ".inst 0x4e80a48d  // smmla v13.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45d  // smmla v29.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x50]\n"
      ".inst 0x4e81a48a  // smmla v10.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a472  // smmla v18.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45a  // smmla v26.4s, v2.16b, v1.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e80a48e  // smmla v14.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45e  // smmla v30.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x70]\n"
      ".inst 0x4e86a48b  // smmla v11.4s, v4.16b, v6.16b\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a473  // smmla v19.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a45b  // smmla v27.4s, v2.16b, v6.16b\n"
      ".inst 0x4e80a48f  // smmla v15.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45f  // smmla v31.4s, v2.16b, v0.16b\n"
      "bge 169b\n"
      "170:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x27, 175f\n"
      "tbz x27, #2, 172f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "tbz x27, #1, 171f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "tbz x27, #0, 174f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "b 174f\n"
      "171:"  // Height 5: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 174f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "b 174f\n"
      "172:"  // Height 5: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 173f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "tbz x27, #0, 174f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "b 174f\n"
      "173:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "ldr b4, [x23, #0x0]\n"
      "ldr b5, [x22, #0x0]\n"
      "174:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "trn1 v7.2d, v1.2d, v2.2d\n"
      "trn1 v3.2d, v3.2d, v4.2d\n"
      "trn1 v2.2d, v5.2d, v0.2d\n"
      "ldr q1, [x10, #0x10]\n"
      ".inst 0x4e86a4e8  // smmla v8.4s, v7.16b, v6.16b\n"
      ".inst 0x4e86a470  // smmla v16.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a458  // smmla v24.4s, v2.16b, v6.16b\n"
      "ldr q0, [x10, #0x20]\n"
      ".inst 0x4e81a4ec  // smmla v12.4s, v7.16b, v1.16b\n"
      ".inst 0x4e81a474  // smmla v20.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45c  // smmla v28.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x30]\n"
      ".inst 0x4e80a4e9  // smmla v9.4s, v7.16b, v0.16b\n"
      ".inst 0x4e80a471  // smmla v17.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a459  // smmla v25.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x40]\n"
      ".inst 0x4e81a4ed  // smmla v13.4s, v7.16b, v1.16b\n"
      ".inst 0x4e81a475  // smmla v21.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45d  // smmla v29.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x50]\n"
      ".inst 0x4e80a4ea  // smmla v10.4s, v7.16b, v0.16b\n"
      ".inst 0x4e80a472  // smmla v18.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45a  // smmla v26.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x60]\n"
      ".inst 0x4e81a4ee  // smmla v14.4s, v7.16b, v1.16b\n"
      ".inst 0x4e81a476  // smmla v22.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45e  // smmla v30.4s, v2.16b, v1.16b\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e80a4eb  // smmla v11.4s, v7.16b, v0.16b\n"
      ".inst 0x4e80a473  // smmla v19.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45b  // smmla v27.4s, v2.16b, v0.16b\n"
      ".inst 0x4e86a4ef  // smmla v15.4s, v7.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a45f  // smmla v31.4s, v2.16b, v6.16b\n"
      "175:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 163b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "cmp x11, #0x10\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "uzp1 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v27.2d, v27.2d, v31.2d\n"
      "bge 184f\n"
      "tbz x11, #3, 179f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v17.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "tbz x11, #2, 177f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x23], #0x10\n"
      "st1 { v18.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "tbz x11, #1, 176f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d22, [x23], #0x8\n"
      "str d19, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "tbz x11, #0, 183f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v22.s }[2], [x23]\n"
      "st1 { v19.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "b 183f\n"
      "176:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x11, #0, 183f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x24, #0x0]\n"
      "str s22, [x23, #0x0]\n"
      "str s19, [x22, #0x0]\n"
      "str s27, [x21, #0x0]\n"
      "b 183f\n"
      "177:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x11, #1, 178f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d21, [x23], #0x8\n"
      "str d18, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "tbz x11, #0, 183f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v21.s }[2], [x23]\n"
      "st1 { v18.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "b 183f\n"
      "178:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x11, #0, 183f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x24, #0x0]\n"
      "str s21, [x23, #0x0]\n"
      "str s18, [x22, #0x0]\n"
      "str s26, [x21, #0x0]\n"
      "b 183f\n"
      "179:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x11, #2, 181f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "tbz x11, #1, 180f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d17, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "tbz x11, #0, 183f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v20.s }[2], [x23]\n"
      "st1 { v17.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "b 183f\n"
      "180:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x11, #0, 183f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x24, #0x0]\n"
      "str s20, [x23, #0x0]\n"
      "str s17, [x22, #0x0]\n"
      "str s25, [x21, #0x0]\n"
      "b 183f\n"
      "181:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x11, #1, 182f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d16, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "tbz x11, #0, 183f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v16.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "b 183f\n"
      "182:"  // Height 5: Partial direct writeback: partial_1_0
      "str s7, [x9, #0x0]\n"
      "str s8, [x24, #0x0]\n"
      "str s15, [x23, #0x0]\n"
      "str s16, [x22, #0x0]\n"
      "str s24, [x21, #0x0]\n"
      "183:"  // Height 5: Partial direct writeback: Done
      "b 185f\n"
      "184:"  // Height 5: Full writeback
      "str q7, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0x0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q15, [x23, #0x0]\n"
      "str q20, [x23, #0x10]\n"
      "str q21, [x23, #0x20]\n"
      "str q22, [x23, #0x30]\n"
      "str q16, [x22, #0x0]\n"
      "str q17, [x22, #0x10]\n"
      "str q18, [x22, #0x20]\n"
      "str q19, [x22, #0x30]\n"
      "str q24, [x21, #0x0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "185:"  // Height 5: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 150b\n"
      "b 224f\n"
      "186:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x18\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x9, %x[output_ptr]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "187:"  // Height 6: Column loop
      "tbz %x[flags], #0, 198f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "cmp x11, #0x10\n"
      "add x20, x21, x20, LSL #2\n"
      "bge 196f\n"
      "tbz x11, #3, 191f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v29.4s }, [x20], #0x10\n"
      "tbz x11, #2, 189f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "ld1 { v30.4s }, [x20], #0x10\n"
      "tbz x11, #1, 188f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d6, [x21], #0x8\n"
      "ldr d31, [x20], #0x8\n"
      "tbz x11, #0, 195f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v6.s }[2], [x21]\n"
      "ld1 { v31.s }[2], [x20]\n"
      "b 195f\n"
      "188:"  // Height 6: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x11, #0, 195f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x24, #0x0]\n"
      "ldr s24, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s6, [x21, #0x0]\n"
      "ldr s31, [x20, #0x0]\n"
      "b 195f\n"
      "189:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x11, #1, 190f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "ldr d30, [x20], #0x8\n"
      "tbz x11, #0, 195f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "ld1 { v30.s }[2], [x20]\n"
      "b 195f\n"
      "190:"  // Height 6: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x11, #0, 195f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x24, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "ldr s30, [x20, #0x0]\n"
      "b 195f\n"
      "191:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x11, #2, 193f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "tbz x11, #1, 192f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "ldr d29, [x20], #0x8\n"
      "tbz x11, #0, 195f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "ld1 { v29.s }[2], [x20]\n"
      "b 195f\n"
      "192:"  // Height 6: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x11, #0, 195f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x24, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "ldr s29, [x20, #0x0]\n"
      "b 195f\n"
      "193:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x11, #1, 194f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "ldr d28, [x20], #0x8\n"
      "tbz x11, #0, 195f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "ld1 { v28.s }[2], [x20]\n"
      "b 195f\n"
      "194:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x24, #0x0]\n"
      "mov x25, #0x0\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "ldr s25, [x21, #0x0]\n"
      "ldr s28, [x20, #0x0]\n"
      "195:"  // Height 6: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 197f\n"
      "196:"  // Height 6: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0x0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0x0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q25, [x21, #0x0]\n"
      "ldr q26, [x21, #0x10]\n"
      "ldr q27, [x21, #0x20]\n"
      "ldr q6, [x21, #0x30]\n"
      "ldr q28, [x20, #0x0]\n"
      "ldr q29, [x20, #0x10]\n"
      "ldr q30, [x20, #0x20]\n"
      "ldr q31, [x20, #0x30]\n"
      "197:"  // Height 6: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "zip1 v24.2d, v25.2d, v28.2d\n"
      "zip2 v28.2d, v25.2d, v28.2d\n"
      "zip1 v25.2d, v26.2d, v29.2d\n"
      "zip2 v29.2d, v26.2d, v29.2d\n"
      "zip1 v26.2d, v27.2d, v30.2d\n"
      "zip2 v30.2d, v27.2d, v30.2d\n"
      "zip1 v27.2d, v6.2d, v31.2d\n"
      "zip2 v31.2d, v6.2d, v31.2d\n"
      "b 199f\n"
      "198:"  // Height 6: no accumulate
      "movi v8.4s, #0x0\n"
      "movi v9.4s, #0x0\n"
      "movi v10.4s, #0x0\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "movi v15.4s, #0x0\n"
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "199:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "200:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 201f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 202f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 202f\n"
      "201:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "202:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "blt 205f\n"
      "ldr q1, [x26, #0x0]\n"
      "ldr q2, [x25, #0x0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0x0]\n"
      "ldr q4, [x23, #0x0]\n"
      "ldr q5, [x22, #0x0]\n"
      "ldr q6, [x21, #0x0]\n"
      "ldr q7, [x10, #0x0]\n"
      "blt 204f\n"
      "203:"  // Height 6: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "cmp x27, #0x20\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q2, [x25, #0x0]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q0, [x10, #0x90]\n"
      "ldr q4, [x23, #0x0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q6, [x10, #0xa0]\n"
      ".inst 0x4e80a42c  // smmla v12.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bc  // smmla v28.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xb0]\n"
      ".inst 0x4e86a429  // smmla v9.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a471  // smmla v17.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4b9  // smmla v25.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xc0]\n"
      ".inst 0x4e80a42d  // smmla v13.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bd  // smmla v29.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xd0]\n"
      ".inst 0x4e86a42a  // smmla v10.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a472  // smmla v18.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4ba  // smmla v26.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xe0]\n"
      ".inst 0x4e80a42e  // smmla v14.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4be  // smmla v30.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e86a42b  // smmla v11.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a473  // smmla v19.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bb  // smmla v27.4s, v5.16b, v6.16b\n"
      "ldr q7, [x10, #0x0]\n"
      ".inst 0x4e80a42f  // smmla v15.4s, v1.16b, v0.16b\n"
      "ldr q1, [x26, #0x0]\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      "ldr q3, [x24, #0x0]\n"
      ".inst 0x4e80a4bf  // smmla v31.4s, v5.16b, v0.16b\n"
      "ldr q5, [x22, #0x0]\n"
      "ldr q6, [x21, #0x0]\n"
      "bge 203b\n"
      "204:"  // Height 6: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "add x25, x25, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "add x21, x21, #0x10\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q2, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q0, [x10, #0xa0]\n"
      ".inst 0x4e82a42c  // smmla v12.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a474  // smmla v20.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4bc  // smmla v28.4s, v5.16b, v2.16b\n"
      "ldr q2, [x10, #0xb0]\n"
      ".inst 0x4e80a429  // smmla v9.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a471  // smmla v17.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4b9  // smmla v25.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xc0]\n"
      ".inst 0x4e82a42d  // smmla v13.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a475  // smmla v21.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4bd  // smmla v29.4s, v5.16b, v2.16b\n"
      "ldr q2, [x10, #0xd0]\n"
      ".inst 0x4e80a42a  // smmla v10.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a472  // smmla v18.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4ba  // smmla v26.4s, v5.16b, v0.16b\n"
      "ldr q0, [x10, #0xe0]\n"
      ".inst 0x4e82a42e  // smmla v14.4s, v1.16b, v2.16b\n"
      ".inst 0x4e82a476  // smmla v22.4s, v3.16b, v2.16b\n"
      ".inst 0x4e82a4be  // smmla v30.4s, v5.16b, v2.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e80a42b  // smmla v11.4s, v1.16b, v0.16b\n"
      ".inst 0x4e80a473  // smmla v19.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a4bb  // smmla v27.4s, v5.16b, v0.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bf  // smmla v31.4s, v5.16b, v6.16b\n"
      "205:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 212f\n"
      "cmp x27, #0x8\n"
      "blt 207f\n"
      "206:"  // Height 6: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d0, [x25], #0x8\n"
      "trn1 v4.2d, v1.2d, v0.2d\n"
      "sub x27, x27, #0x8\n"
      "ldr d1, [x24], #0x8\n"
      "ldr d0, [x23], #0x8\n"
      "trn1 v3.2d, v1.2d, v0.2d\n"
      "cmp x27, #0x8\n"
      "ldr d1, [x22], #0x8\n"
      "ldr d0, [x21], #0x8\n"
      "trn1 v2.2d, v1.2d, v0.2d\n"
      "ldr q1, [x10, #0x0]\n"
      "ldr q0, [x10, #0x10]\n"
      ".inst 0x4e81a488  // smmla v8.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a470  // smmla v16.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a458  // smmla v24.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x20]\n"
      ".inst 0x4e80a48c  // smmla v12.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a474  // smmla v20.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45c  // smmla v28.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x30]\n"
      ".inst 0x4e81a489  // smmla v9.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a471  // smmla v17.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a459  // smmla v25.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x40]\n"
      ".inst 0x4e80a48d  // smmla v13.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a475  // smmla v21.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45d  // smmla v29.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x50]\n"
      ".inst 0x4e81a48a  // smmla v10.4s, v4.16b, v1.16b\n"
      ".inst 0x4e81a472  // smmla v18.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45a  // smmla v26.4s, v2.16b, v1.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e80a48e  // smmla v14.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a476  // smmla v22.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45e  // smmla v30.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a48b  // smmla v11.4s, v4.16b, v6.16b\n"
      ".inst 0x4e86a473  // smmla v19.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a45b  // smmla v27.4s, v2.16b, v6.16b\n"
      ".inst 0x4e80a48f  // smmla v15.4s, v4.16b, v0.16b\n"
      ".inst 0x4e80a477  // smmla v23.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45f  // smmla v31.4s, v2.16b, v0.16b\n"
      "bge 206b\n"
      "207:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x27, 212f\n"
      "tbz x27, #2, 209f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "ldr s6, [x21], #0x4\n"
      "tbz x27, #1, 208f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "ld1 { v6.h }[2], [x21], #0x2\n"
      "tbz x27, #0, 211f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "ld1 { v6.b }[6], [x21]\n"
      "b 211f\n"
      "208:"  // Height 6: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 211f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "ld1 { v6.b }[4], [x21]\n"
      "b 211f\n"
      "209:"  // Height 6: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 210f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "ldr h6, [x21], #0x2\n"
      "tbz x27, #0, 211f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "ld1 { v6.b }[2], [x21]\n"
      "b 211f\n"
      "210:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0x0]\n"
      "ldr b2, [x25, #0x0]\n"
      "ldr b3, [x24, #0x0]\n"
      "ldr b4, [x23, #0x0]\n"
      "ldr b5, [x22, #0x0]\n"
      "ldr b6, [x21, #0x0]\n"
      "211:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q0, [x10, #0x0]\n"
      "trn1 v7.2d, v1.2d, v2.2d\n"
      "trn1 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e80a4e8  // smmla v8.4s, v7.16b, v0.16b\n"
      "trn1 v2.2d, v5.2d, v6.2d\n"
      "ldr q1, [x10, #0x10]\n"
      ".inst 0x4e80a470  // smmla v16.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a458  // smmla v24.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x20]\n"
      ".inst 0x4e81a4ec  // smmla v12.4s, v7.16b, v1.16b\n"
      ".inst 0x4e81a474  // smmla v20.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45c  // smmla v28.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x30]\n"
      ".inst 0x4e80a4e9  // smmla v9.4s, v7.16b, v0.16b\n"
      ".inst 0x4e80a471  // smmla v17.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a459  // smmla v25.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x40]\n"
      ".inst 0x4e81a4ed  // smmla v13.4s, v7.16b, v1.16b\n"
      ".inst 0x4e81a475  // smmla v21.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45d  // smmla v29.4s, v2.16b, v1.16b\n"
      "ldr q1, [x10, #0x50]\n"
      ".inst 0x4e80a4ea  // smmla v10.4s, v7.16b, v0.16b\n"
      ".inst 0x4e80a472  // smmla v18.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45a  // smmla v26.4s, v2.16b, v0.16b\n"
      "ldr q0, [x10, #0x60]\n"
      ".inst 0x4e81a4ee  // smmla v14.4s, v7.16b, v1.16b\n"
      ".inst 0x4e81a476  // smmla v22.4s, v3.16b, v1.16b\n"
      ".inst 0x4e81a45e  // smmla v30.4s, v2.16b, v1.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e80a4eb  // smmla v11.4s, v7.16b, v0.16b\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e80a473  // smmla v19.4s, v3.16b, v0.16b\n"
      ".inst 0x4e80a45b  // smmla v27.4s, v2.16b, v0.16b\n"
      ".inst 0x4e86a4ef  // smmla v15.4s, v7.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a45f  // smmla v31.4s, v2.16b, v6.16b\n"
      "212:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 200b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "add x20, x21, x20, LSL #2\n"
      "cmp x11, #0x10\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "prfm pstl1keep, [x20, #0x0]\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "uzp1 v23.2d, v24.2d, v28.2d\n"
      "uzp2 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v28.2d, v25.2d, v29.2d\n"
      "uzp2 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v29.2d, v26.2d, v30.2d\n"
      "uzp2 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v30.2d, v27.2d, v31.2d\n"
      "uzp2 v27.2d, v27.2d, v31.2d\n"
      "bge 221f\n"
      "tbz x11, #3, 216f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v17.4s }, [x22], #0x10\n"
      "st1 { v23.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x21], #0x10\n"
      "st1 { v24.4s }, [x20], #0x10\n"
      "st1 { v25.4s }, [x20], #0x10\n"
      "tbz x11, #2, 214f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x23], #0x10\n"
      "st1 { v18.4s }, [x22], #0x10\n"
      "st1 { v29.4s }, [x21], #0x10\n"
      "st1 { v26.4s }, [x20], #0x10\n"
      "tbz x11, #1, 213f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d22, [x23], #0x8\n"
      "str d19, [x22], #0x8\n"
      "str d30, [x21], #0x8\n"
      "str d27, [x20], #0x8\n"
      "tbz x11, #0, 220f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v22.s }[2], [x23]\n"
      "st1 { v19.s }[2], [x22]\n"
      "st1 { v30.s }[2], [x21]\n"
      "st1 { v27.s }[2], [x20]\n"
      "b 220f\n"
      "213:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x11, #0, 220f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x24, #0x0]\n"
      "str s22, [x23, #0x0]\n"
      "str s19, [x22, #0x0]\n"
      "str s30, [x21, #0x0]\n"
      "str s27, [x20, #0x0]\n"
      "b 220f\n"
      "214:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x11, #1, 215f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d21, [x23], #0x8\n"
      "str d18, [x22], #0x8\n"
      "str d29, [x21], #0x8\n"
      "str d26, [x20], #0x8\n"
      "tbz x11, #0, 220f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v21.s }[2], [x23]\n"
      "st1 { v18.s }[2], [x22]\n"
      "st1 { v29.s }[2], [x21]\n"
      "st1 { v26.s }[2], [x20]\n"
      "b 220f\n"
      "215:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x11, #0, 220f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x24, #0x0]\n"
      "str s21, [x23, #0x0]\n"
      "str s18, [x22, #0x0]\n"
      "str s29, [x21, #0x0]\n"
      "str s26, [x20, #0x0]\n"
      "b 220f\n"
      "216:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x11, #2, 218f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v23.4s }, [x21], #0x10\n"
      "st1 { v24.4s }, [x20], #0x10\n"
      "tbz x11, #1, 217f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d17, [x22], #0x8\n"
      "str d28, [x21], #0x8\n"
      "str d25, [x20], #0x8\n"
      "tbz x11, #0, 220f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v20.s }[2], [x23]\n"
      "st1 { v17.s }[2], [x22]\n"
      "st1 { v28.s }[2], [x21]\n"
      "st1 { v25.s }[2], [x20]\n"
      "b 220f\n"
      "217:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x11, #0, 220f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x24, #0x0]\n"
      "str s20, [x23, #0x0]\n"
      "str s17, [x22, #0x0]\n"
      "str s28, [x21, #0x0]\n"
      "str s25, [x20, #0x0]\n"
      "b 220f\n"
      "218:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x11, #1, 219f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d16, [x22], #0x8\n"
      "str d23, [x21], #0x8\n"
      "str d24, [x20], #0x8\n"
      "tbz x11, #0, 220f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v16.s }[2], [x22]\n"
      "st1 { v23.s }[2], [x21]\n"
      "st1 { v24.s }[2], [x20]\n"
      "b 220f\n"
      "219:"  // Height 6: Partial direct writeback: partial_1_0
      "str s7, [x9, #0x0]\n"
      "str s8, [x24, #0x0]\n"
      "str s15, [x23, #0x0]\n"
      "str s16, [x22, #0x0]\n"
      "str s23, [x21, #0x0]\n"
      "str s24, [x20, #0x0]\n"
      "220:"  // Height 6: Partial direct writeback: Done
      "b 222f\n"
      "221:"  // Height 6: Full writeback
      "str q7, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0x0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q15, [x23, #0x0]\n"
      "str q20, [x23, #0x10]\n"
      "str q21, [x23, #0x20]\n"
      "str q22, [x23, #0x30]\n"
      "str q16, [x22, #0x0]\n"
      "str q17, [x22, #0x10]\n"
      "str q18, [x22, #0x20]\n"
      "str q19, [x22, #0x30]\n"
      "str q23, [x21, #0x0]\n"
      "str q28, [x21, #0x10]\n"
      "str q29, [x21, #0x20]\n"
      "str q30, [x21, #0x30]\n"
      "str q24, [x20, #0x0]\n"
      "str q25, [x20, #0x10]\n"
      "str q26, [x20, #0x20]\n"
      "str q27, [x20, #0x30]\n"
      "222:"  // Height 6: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 187b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 224f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 223f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "223:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "224:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
