// Copyright 2005, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Author: wan@google.com (Zhanyong Wan)
//
// Tests for Google Test itself.  This verifies that the basic constructs of
// Google Test work.

#include <gtest/gtest.h>

// Verifies that the command line flag variables can be accessed
// in code once <gtest/gtest.h> has been #included.
// Do not move it after other #includes.
TEST(CommandLineFlagsTest, CanBeAccessedInCodeOnceGTestHIsIncluded) {
  bool dummy = testing::GTEST_FLAG(also_run_disabled_tests)
      || testing::GTEST_FLAG(break_on_failure)
      || testing::GTEST_FLAG(catch_exceptions)
      || testing::GTEST_FLAG(color) != "unknown"
      || testing::GTEST_FLAG(filter) != "unknown"
      || testing::GTEST_FLAG(list_tests)
      || testing::GTEST_FLAG(output) != "unknown"
      || testing::GTEST_FLAG(print_time)
      || testing::GTEST_FLAG(random_seed)
      || testing::GTEST_FLAG(repeat) > 0
      || testing::GTEST_FLAG(show_internal_stack_frames)
      || testing::GTEST_FLAG(shuffle)
      || testing::GTEST_FLAG(stack_trace_depth) > 0
      || testing::GTEST_FLAG(throw_on_failure);
  EXPECT_TRUE(dummy || !dummy);  // Suppresses warning that dummy is unused.
}

#include <gtest/gtest-spi.h>

// Indicates that this translation unit is part of Google Test's
// implementation.  It must come before gtest-internal-inl.h is
// included, or there will be a compiler error.  This trick is to
// prevent a user from accidentally including gtest-internal-inl.h in
// his code.
#define GTEST_IMPLEMENTATION_ 1
#include "src/gtest-internal-inl.h"
#undef GTEST_IMPLEMENTATION_

#include <limits.h>  // For INT_MAX.
#include <stdlib.h>
#include <time.h>

#if GTEST_HAS_PTHREAD
#include <pthread.h>
#endif  // GTEST_HAS_PTHREAD

#ifdef __BORLANDC__
#include <map>
#endif

namespace testing {
namespace internal {

bool ShouldUseColor(bool stdout_is_tty);
const char* FormatTimeInMillisAsSeconds(TimeInMillis ms);
bool ParseInt32Flag(const char* str, const char* flag, Int32* value);

// Provides access to otherwise private parts of the TestEventListeners class
// that are needed to test it.
class TestEventListenersAccessor {
 public:
  static TestEventListener* GetRepeater(TestEventListeners* listeners) {
    return listeners->repeater();
  }

  static void SetDefaultResultPrinter(TestEventListeners* listeners,
                                      TestEventListener* listener) {
    listeners->SetDefaultResultPrinter(listener);
  }
  static void SetDefaultXmlGenerator(TestEventListeners* listeners,
                                     TestEventListener* listener) {
    listeners->SetDefaultXmlGenerator(listener);
  }

  static bool EventForwardingEnabled(const TestEventListeners& listeners) {
    return listeners.EventForwardingEnabled();
  }

  static void SuppressEventForwarding(TestEventListeners* listeners) {
    listeners->SuppressEventForwarding();
  }
};

}  // namespace internal
}  // namespace testing

using testing::AssertionFailure;
using testing::AssertionResult;
using testing::AssertionSuccess;
using testing::DoubleLE;
using testing::EmptyTestEventListener;
using testing::FloatLE;
using testing::GTEST_FLAG(also_run_disabled_tests);
using testing::GTEST_FLAG(break_on_failure);
using testing::GTEST_FLAG(catch_exceptions);
using testing::GTEST_FLAG(color);
using testing::GTEST_FLAG(death_test_use_fork);
using testing::GTEST_FLAG(filter);
using testing::GTEST_FLAG(list_tests);
using testing::GTEST_FLAG(output);
using testing::GTEST_FLAG(print_time);
using testing::GTEST_FLAG(random_seed);
using testing::GTEST_FLAG(repeat);
using testing::GTEST_FLAG(show_internal_stack_frames);
using testing::GTEST_FLAG(shuffle);
using testing::GTEST_FLAG(stack_trace_depth);
using testing::GTEST_FLAG(throw_on_failure);
using testing::IsNotSubstring;
using testing::IsSubstring;
using testing::Message;
using testing::ScopedFakeTestPartResultReporter;
using testing::StaticAssertTypeEq;
using testing::Test;
using testing::TestEventListeners;
using testing::TestCase;
using testing::TestPartResult;
using testing::TestPartResultArray;
using testing::TestProperty;
using testing::TestResult;
using testing::UnitTest;
using testing::internal::AlwaysFalse;
using testing::internal::AlwaysTrue;
using testing::internal::AppendUserMessage;
using testing::internal::CodePointToUtf8;
using testing::internal::EqFailure;
using testing::internal::FloatingPoint;
using testing::internal::FormatTimeInMillisAsSeconds;
using testing::internal::GTestFlagSaver;
using testing::internal::GetCurrentOsStackTraceExceptTop;
using testing::internal::GetNextRandomSeed;
using testing::internal::GetRandomSeedFromFlag;
using testing::internal::GetTestTypeId;
using testing::internal::GetTypeId;
using testing::internal::GetUnitTestImpl;
using testing::internal::Int32;
using testing::internal::Int32FromEnvOrDie;
using testing::internal::ParseInt32Flag;
using testing::internal::ShouldRunTestOnShard;
using testing::internal::ShouldShard;
using testing::internal::ShouldUseColor;
using testing::internal::StreamableToString;
using testing::internal::String;
using testing::internal::TestEventListenersAccessor;
using testing::internal::TestResultAccessor;
using testing::internal::ThreadLocal;
using testing::internal::UInt32;
using testing::internal::Vector;
using testing::internal::WideStringToUtf8;
using testing::internal::kMaxRandomSeed;
using testing::internal::kTestTypeIdInGoogleTest;
using testing::internal::scoped_ptr;

class TestingVector : public Vector<int> {
};

::std::ostream& operator<<(::std::ostream& os,
                           const TestingVector& vector) {
  os << "{ ";
  for (int i = 0; i < vector.size(); i++) {
    os << vector.GetElement(i) << " ";
  }
  os << "}";
  return os;
}

// This line tests that we can define tests in an unnamed namespace.
namespace {

TEST(GetRandomSeedFromFlagTest, HandlesZero) {
  const int seed = GetRandomSeedFromFlag(0);
  EXPECT_LE(1, seed);
  EXPECT_LE(seed, static_cast<int>(kMaxRandomSeed));
}

TEST(GetRandomSeedFromFlagTest, PreservesValidSeed) {
  EXPECT_EQ(1, GetRandomSeedFromFlag(1));
  EXPECT_EQ(2, GetRandomSeedFromFlag(2));
  EXPECT_EQ(kMaxRandomSeed - 1, GetRandomSeedFromFlag(kMaxRandomSeed - 1));
  EXPECT_EQ(static_cast<int>(kMaxRandomSeed),
            GetRandomSeedFromFlag(kMaxRandomSeed));
}

TEST(GetRandomSeedFromFlagTest, NormalizesInvalidSeed) {
  const int seed1 = GetRandomSeedFromFlag(-1);
  EXPECT_LE(1, seed1);
  EXPECT_LE(seed1, static_cast<int>(kMaxRandomSeed));

  const int seed2 = GetRandomSeedFromFlag(kMaxRandomSeed + 1);
  EXPECT_LE(1, seed2);
  EXPECT_LE(seed2, static_cast<int>(kMaxRandomSeed));
}

TEST(GetNextRandomSeedTest, WorksForValidInput) {
  EXPECT_EQ(2, GetNextRandomSeed(1));
  EXPECT_EQ(3, GetNextRandomSeed(2));
  EXPECT_EQ(static_cast<int>(kMaxRandomSeed),
            GetNextRandomSeed(kMaxRandomSeed - 1));
  EXPECT_EQ(1, GetNextRandomSeed(kMaxRandomSeed));

  // We deliberately don't test GetNextRandomSeed() with invalid
  // inputs, as that requires death tests, which are expensive.  This
  // is fine as GetNextRandomSeed() is internal and has a
  // straightforward definition.
}

static void ClearCurrentTestPartResults() {
  TestResultAccessor::ClearTestPartResults(
      GetUnitTestImpl()->current_test_result());
}

// Tests GetTypeId.

TEST(GetTypeIdTest, ReturnsSameValueForSameType) {
  EXPECT_EQ(GetTypeId<int>(), GetTypeId<int>());
  EXPECT_EQ(GetTypeId<Test>(), GetTypeId<Test>());
}

class SubClassOfTest : public Test {};
class AnotherSubClassOfTest : public Test {};

TEST(GetTypeIdTest, ReturnsDifferentValuesForDifferentTypes) {
  EXPECT_NE(GetTypeId<int>(), GetTypeId<const int>());
  EXPECT_NE(GetTypeId<int>(), GetTypeId<char>());
  EXPECT_NE(GetTypeId<int>(), GetTestTypeId());
  EXPECT_NE(GetTypeId<SubClassOfTest>(), GetTestTypeId());
  EXPECT_NE(GetTypeId<AnotherSubClassOfTest>(), GetTestTypeId());
  EXPECT_NE(GetTypeId<AnotherSubClassOfTest>(), GetTypeId<SubClassOfTest>());
}

// Verifies that GetTestTypeId() returns the same value, no matter it
// is called from inside Google Test or outside of it.
TEST(GetTestTypeIdTest, ReturnsTheSameValueInsideOrOutsideOfGoogleTest) {
  EXPECT_EQ(kTestTypeIdInGoogleTest, GetTestTypeId());
}

// Tests FormatTimeInMillisAsSeconds().

TEST(FormatTimeInMillisAsSecondsTest, FormatsZero) {
  EXPECT_STREQ("0", FormatTimeInMillisAsSeconds(0));
}

TEST(FormatTimeInMillisAsSecondsTest, FormatsPositiveNumber) {
  EXPECT_STREQ("0.003", FormatTimeInMillisAsSeconds(3));
  EXPECT_STREQ("0.01", FormatTimeInMillisAsSeconds(10));
  EXPECT_STREQ("0.2", FormatTimeInMillisAsSeconds(200));
  EXPECT_STREQ("1.2", FormatTimeInMillisAsSeconds(1200));
  EXPECT_STREQ("3", FormatTimeInMillisAsSeconds(3000));
}

TEST(FormatTimeInMillisAsSecondsTest, FormatsNegativeNumber) {
  EXPECT_STREQ("-0.003", FormatTimeInMillisAsSeconds(-3));
  EXPECT_STREQ("-0.01", FormatTimeInMillisAsSeconds(-10));
  EXPECT_STREQ("-0.2", FormatTimeInMillisAsSeconds(-200));
  EXPECT_STREQ("-1.2", FormatTimeInMillisAsSeconds(-1200));
  EXPECT_STREQ("-3", FormatTimeInMillisAsSeconds(-3000));
}

#if !GTEST_OS_SYMBIAN
// NULL testing does not work with Symbian compilers.

#ifdef __BORLANDC__
// Silences warnings: "Condition is always true", "Unreachable code"
#pragma option push -w-ccc -w-rch
#endif

// Tests that GTEST_IS_NULL_LITERAL_(x) is true when x is a null
// pointer literal.
TEST(NullLiteralTest, IsTrueForNullLiterals) {
  EXPECT_TRUE(GTEST_IS_NULL_LITERAL_(NULL));
  EXPECT_TRUE(GTEST_IS_NULL_LITERAL_(0));
  EXPECT_TRUE(GTEST_IS_NULL_LITERAL_(0U));
  EXPECT_TRUE(GTEST_IS_NULL_LITERAL_(0L));
  EXPECT_TRUE(GTEST_IS_NULL_LITERAL_(false));
#ifndef __BORLANDC__
  // Some compilers may fail to detect some null pointer literals;
  // as long as users of the framework don't use such literals, this
  // is harmless.
  EXPECT_TRUE(GTEST_IS_NULL_LITERAL_(1 - 1));
  EXPECT_TRUE(GTEST_IS_NULL_LITERAL_(true && false));
#endif
}

// Tests that GTEST_IS_NULL_LITERAL_(x) is false when x is not a null
// pointer literal.
TEST(NullLiteralTest, IsFalseForNonNullLiterals) {
  EXPECT_FALSE(GTEST_IS_NULL_LITERAL_(1));
  EXPECT_FALSE(GTEST_IS_NULL_LITERAL_(0.0));
  EXPECT_FALSE(GTEST_IS_NULL_LITERAL_('a'));
  EXPECT_FALSE(GTEST_IS_NULL_LITERAL_(static_cast<void*>(NULL)));
}

#ifdef __BORLANDC__
// Restores warnings after previous "#pragma option push" supressed them
#pragma option pop
#endif

#endif  // !GTEST_OS_SYMBIAN
//
// Tests CodePointToUtf8().

// Tests that the NUL character L'\0' is encoded correctly.
TEST(CodePointToUtf8Test, CanEncodeNul) {
  char buffer[32];
  EXPECT_STREQ("", CodePointToUtf8(L'\0', buffer));
}

// Tests that ASCII characters are encoded correctly.
TEST(CodePointToUtf8Test, CanEncodeAscii) {
  char buffer[32];
  EXPECT_STREQ("a", CodePointToUtf8(L'a', buffer));
  EXPECT_STREQ("Z", CodePointToUtf8(L'Z', buffer));
  EXPECT_STREQ("&", CodePointToUtf8(L'&', buffer));
  EXPECT_STREQ("\x7F", CodePointToUtf8(L'\x7F', buffer));
}

// Tests that Unicode code-points that have 8 to 11 bits are encoded
// as 110xxxxx 10xxxxxx.
TEST(CodePointToUtf8Test, CanEncode8To11Bits) {
  char buffer[32];
  // 000 1101 0011 => 110-00011 10-010011
  EXPECT_STREQ("\xC3\x93", CodePointToUtf8(L'\xD3', buffer));

  // 101 0111 0110 => 110-10101 10-110110
  EXPECT_STREQ("\xD5\xB6", CodePointToUtf8(L'\x576', buffer));
}

// Tests that Unicode code-points that have 12 to 16 bits are encoded
// as 1110xxxx 10xxxxxx 10xxxxxx.
TEST(CodePointToUtf8Test, CanEncode12To16Bits) {
  char buffer[32];
  // 0000 1000 1101 0011 => 1110-0000 10-100011 10-010011
  EXPECT_STREQ("\xE0\xA3\x93", CodePointToUtf8(L'\x8D3', buffer));

  // 1100 0111 0100 1101 => 1110-1100 10-011101 10-001101
  EXPECT_STREQ("\xEC\x9D\x8D", CodePointToUtf8(L'\xC74D', buffer));
}

#if !GTEST_WIDE_STRING_USES_UTF16_
// Tests in this group require a wchar_t to hold > 16 bits, and thus
// are skipped on Windows, Cygwin, and Symbian, where a wchar_t is
// 16-bit wide. This code may not compile on those systems.

// Tests that Unicode code-points that have 17 to 21 bits are encoded
// as 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx.
TEST(CodePointToUtf8Test, CanEncode17To21Bits) {
  char buffer[32];
  // 0 0001 0000 1000 1101 0011 => 11110-000 10-010000 10-100011 10-010011
  EXPECT_STREQ("\xF0\x90\xA3\x93", CodePointToUtf8(L'\x108D3', buffer));

  // 0 0001 0000 0100 0000 0000 => 11110-000 10-010000 10-010000 10-000000
  EXPECT_STREQ("\xF0\x90\x90\x80", CodePointToUtf8(L'\x10400', buffer));

  // 1 0000 1000 0110 0011 0100 => 11110-100 10-001000 10-011000 10-110100
  EXPECT_STREQ("\xF4\x88\x98\xB4", CodePointToUtf8(L'\x108634', buffer));
}

// Tests that encoding an invalid code-point generates the expected result.
TEST(CodePointToUtf8Test, CanEncodeInvalidCodePoint) {
  char buffer[32];
  EXPECT_STREQ("(Invalid Unicode 0x1234ABCD)",
               CodePointToUtf8(L'\x1234ABCD', buffer));
}

#endif  // !GTEST_WIDE_STRING_USES_UTF16_

// Tests WideStringToUtf8().

// Tests that the NUL character L'\0' is encoded correctly.
TEST(WideStringToUtf8Test, CanEncodeNul) {
  EXPECT_STREQ("", WideStringToUtf8(L"", 0).c_str());
  EXPECT_STREQ("", WideStringToUtf8(L"", -1).c_str());
}

// Tests that ASCII strings are encoded correctly.
TEST(WideStringToUtf8Test, CanEncodeAscii) {
  EXPECT_STREQ("a", WideStringToUtf8(L"a", 1).c_str());
  EXPECT_STREQ("ab", WideStringToUtf8(L"ab", 2).c_str());
  EXPECT_STREQ("a", WideStringToUtf8(L"a", -1).c_str());
  EXPECT_STREQ("ab", WideStringToUtf8(L"ab", -1).c_str());
}

// Tests that Unicode code-points that have 8 to 11 bits are encoded
// as 110xxxxx 10xxxxxx.
TEST(WideStringToUtf8Test, CanEncode8To11Bits) {
  // 000 1101 0011 => 110-00011 10-010011
  EXPECT_STREQ("\xC3\x93", WideStringToUtf8(L"\xD3", 1).c_str());
  EXPECT_STREQ("\xC3\x93", WideStringToUtf8(L"\xD3", -1).c_str());

  // 101 0111 0110 => 110-10101 10-110110
  EXPECT_STREQ("\xD5\xB6", WideStringToUtf8(L"\x576", 1).c_str());
  EXPECT_STREQ("\xD5\xB6", WideStringToUtf8(L"\x576", -1).c_str());
}

// Tests that Unicode code-points that have 12 to 16 bits are encoded
// as 1110xxxx 10xxxxxx 10xxxxxx.
TEST(WideStringToUtf8Test, CanEncode12To16Bits) {
  // 0000 1000 1101 0011 => 1110-0000 10-100011 10-010011
  EXPECT_STREQ("\xE0\xA3\x93", WideStringToUtf8(L"\x8D3", 1).c_str());
  EXPECT_STREQ("\xE0\xA3\x93", WideStringToUtf8(L"\x8D3", -1).c_str());

  // 1100 0111 0100 1101 => 1110-1100 10-011101 10-001101
  EXPECT_STREQ("\xEC\x9D\x8D", WideStringToUtf8(L"\xC74D", 1).c_str());
  EXPECT_STREQ("\xEC\x9D\x8D", WideStringToUtf8(L"\xC74D", -1).c_str());
}

// Tests that the conversion stops when the function encounters \0 character.
TEST(WideStringToUtf8Test, StopsOnNulCharacter) {
  EXPECT_STREQ("ABC", WideStringToUtf8(L"ABC\0XYZ", 100).c_str());
}

// Tests that the conversion stops when the function reaches the limit
// specified by the 'length' parameter.
TEST(WideStringToUtf8Test, StopsWhenLengthLimitReached) {
  EXPECT_STREQ("ABC", WideStringToUtf8(L"ABCDEF", 3).c_str());
}


#if !GTEST_WIDE_STRING_USES_UTF16_
// Tests that Unicode code-points that have 17 to 21 bits are encoded
// as 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx. This code may not compile
// on the systems using UTF-16 encoding.
TEST(WideStringToUtf8Test, CanEncode17To21Bits) {
  // 0 0001 0000 1000 1101 0011 => 11110-000 10-010000 10-100011 10-010011
  EXPECT_STREQ("\xF0\x90\xA3\x93", WideStringToUtf8(L"\x108D3", 1).c_str());
  EXPECT_STREQ("\xF0\x90\xA3\x93", WideStringToUtf8(L"\x108D3", -1).c_str());

  // 1 0000 1000 0110 0011 0100 => 11110-100 10-001000 10-011000 10-110100
  EXPECT_STREQ("\xF4\x88\x98\xB4", WideStringToUtf8(L"\x108634", 1).c_str());
  EXPECT_STREQ("\xF4\x88\x98\xB4", WideStringToUtf8(L"\x108634", -1).c_str());
}

// Tests that encoding an invalid code-point generates the expected result.
TEST(WideStringToUtf8Test, CanEncodeInvalidCodePoint) {
  EXPECT_STREQ("(Invalid Unicode 0xABCDFF)",
               WideStringToUtf8(L"\xABCDFF", -1).c_str());
}
#else  // !GTEST_WIDE_STRING_USES_UTF16_
// Tests that surrogate pairs are encoded correctly on the systems using
// UTF-16 encoding in the wide strings.
TEST(WideStringToUtf8Test, CanEncodeValidUtf16SUrrogatePairs) {
  EXPECT_STREQ("\xF0\x90\x90\x80",
               WideStringToUtf8(L"\xD801\xDC00", -1).c_str());
}

// Tests that encoding an invalid UTF-16 surrogate pair
// generates the expected result.
TEST(WideStringToUtf8Test, CanEncodeInvalidUtf16SurrogatePair) {
  // Leading surrogate is at the end of the string.
  EXPECT_STREQ("\xED\xA0\x80", WideStringToUtf8(L"\xD800", -1).c_str());
  // Leading surrogate is not followed by the trailing surrogate.
  EXPECT_STREQ("\xED\xA0\x80$", WideStringToUtf8(L"\xD800$", -1).c_str());
  // Trailing surrogate appearas without a leading surrogate.
  EXPECT_STREQ("\xED\xB0\x80PQR", WideStringToUtf8(L"\xDC00PQR", -1).c_str());
}
#endif  // !GTEST_WIDE_STRING_USES_UTF16_

// Tests that codepoint concatenation works correctly.
#if !GTEST_WIDE_STRING_USES_UTF16_
TEST(WideStringToUtf8Test, ConcatenatesCodepointsCorrectly) {
  EXPECT_STREQ(
      "\xF4\x88\x98\xB4"
          "\xEC\x9D\x8D"
          "\n"
          "\xD5\xB6"
          "\xE0\xA3\x93"
          "\xF4\x88\x98\xB4",
      WideStringToUtf8(L"\x108634\xC74D\n\x576\x8D3\x108634", -1).c_str());
}
#else
TEST(WideStringToUtf8Test, ConcatenatesCodepointsCorrectly) {
  EXPECT_STREQ(
      "\xEC\x9D\x8D" "\n" "\xD5\xB6" "\xE0\xA3\x93",
      WideStringToUtf8(L"\xC74D\n\x576\x8D3", -1).c_str());
}
#endif  // !GTEST_WIDE_STRING_USES_UTF16_

// Tests the Random class.

TEST(RandomDeathTest, GeneratesCrashesOnInvalidRange) {
  testing::internal::Random random(42);
  EXPECT_DEATH_IF_SUPPORTED(
      random.Generate(0),
      "Cannot generate a number in the range \\[0, 0\\)");
  EXPECT_DEATH_IF_SUPPORTED(
      random.Generate(testing::internal::Random::kMaxRange + 1),
      "Generation of a number in \\[0, 2147483649\\) was requested, "
      "but this can only generate numbers in \\[0, 2147483648\\)");
}

TEST(RandomTest, GeneratesNumbersWithinRange) {
  const UInt32 kRange = 10000;
  testing::internal::Random random(12345);
  for (int i = 0; i < 10; i++) {
    EXPECT_LT(random.Generate(kRange), kRange) << " for iteration " << i;
  }

  testing::internal::Random random2(testing::internal::Random::kMaxRange);
  for (int i = 0; i < 10; i++) {
    EXPECT_LT(random2.Generate(kRange), kRange) << " for iteration " << i;
  }
}

TEST(RandomTest, RepeatsWhenReseeded) {
  const int kSeed = 123;
  const int kArraySize = 10;
  const UInt32 kRange = 10000;
  UInt32 values[kArraySize];

  testing::internal::Random random(kSeed);
  for (int i = 0; i < kArraySize; i++) {
    values[i] = random.Generate(kRange);
  }

  random.Reseed(kSeed);
  for (int i = 0; i < kArraySize; i++) {
    EXPECT_EQ(values[i], random.Generate(kRange)) << " for iteration " << i;
  }
}

// Tests the Vector class template.

// Tests Vector::Clear().
TEST(VectorTest, Clear) {
  Vector<int> a;
  a.PushBack(1);
  a.Clear();
  EXPECT_EQ(0, a.size());

  a.PushBack(2);
  a.PushBack(3);
  a.Clear();
  EXPECT_EQ(0, a.size());
}

// Tests Vector::PushBack().
TEST(VectorTest, PushBack) {
  Vector<char> a;
  a.PushBack('a');
  ASSERT_EQ(1, a.size());
  EXPECT_EQ('a', a.GetElement(0));

  a.PushBack('b');
  ASSERT_EQ(2, a.size());
  EXPECT_EQ('a', a.GetElement(0));
  EXPECT_EQ('b', a.GetElement(1));
}

// Tests Vector::PushFront().
TEST(VectorTest, PushFront) {
  Vector<int> a;
  ASSERT_EQ(0, a.size());

  // Calls PushFront() on an empty Vector.
  a.PushFront(1);
  ASSERT_EQ(1, a.size());
  EXPECT_EQ(1, a.GetElement(0));

  // Calls PushFront() on a singleton Vector.
  a.PushFront(2);
  ASSERT_EQ(2, a.size());
  EXPECT_EQ(2, a.GetElement(0));
  EXPECT_EQ(1, a.GetElement(1));

  // Calls PushFront() on a Vector with more than one elements.
  a.PushFront(3);
  ASSERT_EQ(3, a.size());
  EXPECT_EQ(3, a.GetElement(0));
  EXPECT_EQ(2, a.GetElement(1));
  EXPECT_EQ(1, a.GetElement(2));
}

// Tests Vector::PopFront().
TEST(VectorTest, PopFront) {
  Vector<int> a;

  // Popping on an empty Vector should fail.
  EXPECT_FALSE(a.PopFront(NULL));

  // Popping again on an empty Vector should fail, and the result element
  // shouldn't be overwritten.
  int element = 1;
  EXPECT_FALSE(a.PopFront(&element));
  EXPECT_EQ(1, element);

  a.PushFront(2);
  a.PushFront(3);

  // PopFront() should pop the element in the front of the Vector.
  EXPECT_TRUE(a.PopFront(&element));
  EXPECT_EQ(3, element);

  // After popping the last element, the Vector should be empty.
  EXPECT_TRUE(a.PopFront(NULL));
  EXPECT_EQ(0, a.size());
}

// Tests inserting at the beginning using Vector::Insert().
TEST(VectorTest, InsertAtBeginning) {
  Vector<int> a;
  ASSERT_EQ(0, a.size());

  // Inserts into an empty Vector.
  a.Insert(1, 0);
  ASSERT_EQ(1, a.size());
  EXPECT_EQ(1, a.GetElement(0));

  // Inserts at the beginning of a singleton Vector.
  a.Insert(2, 0);
  ASSERT_EQ(2, a.size());
  EXPECT_EQ(2, a.GetElement(0));
  EXPECT_EQ(1, a.GetElement(1));

  // Inserts at the beginning of a Vector with more than one elements.
  a.Insert(3, 0);
  ASSERT_EQ(3, a.size());
  EXPECT_EQ(3, a.GetElement(0));
  EXPECT_EQ(2, a.GetElement(1));
  EXPECT_EQ(1, a.GetElement(2));
}

// Tests inserting at a location other than the beginning using
// Vector::Insert().
TEST(VectorTest, InsertNotAtBeginning) {
  // Prepares a singleton Vector.
  Vector<int> a;
  a.PushBack(1);

  // Inserts at the end of a singleton Vector.
  a.Insert(2, a.size());
  ASSERT_EQ(2, a.size());
  EXPECT_EQ(1, a.GetElement(0));
  EXPECT_EQ(2, a.GetElement(1));

  // Inserts at the end of a Vector with more than one elements.
  a.Insert(3, a.size());
  ASSERT_EQ(3, a.size());
  EXPECT_EQ(1, a.GetElement(0));
  EXPECT_EQ(2, a.GetElement(1));
  EXPECT_EQ(3, a.GetElement(2));

  // Inserts in the middle of a Vector.
  a.Insert(4, 1);
  ASSERT_EQ(4, a.size());
  EXPECT_EQ(1, a.GetElement(0));
  EXPECT_EQ(4, a.GetElement(1));
  EXPECT_EQ(2, a.GetElement(2));
  EXPECT_EQ(3, a.GetElement(3));
}

// Tests Vector::GetElementOr().
TEST(VectorTest, GetElementOr) {
  Vector<char> a;
  EXPECT_EQ('x', a.GetElementOr(0, 'x'));

  a.PushBack('a');
  a.PushBack('b');
  EXPECT_EQ('a', a.GetElementOr(0, 'x'));
  EXPECT_EQ('b', a.GetElementOr(1, 'x'));
  EXPECT_EQ('x', a.GetElementOr(-2, 'x'));
  EXPECT_EQ('x', a.GetElementOr(2, 'x'));
}

TEST(VectorTest, Swap) {
  Vector<int> a;
  a.PushBack(0);
  a.PushBack(1);
  a.PushBack(2);

  // Swaps an element with itself.
  a.Swap(0, 0);
  ASSERT_EQ(0, a.GetElement(0));
  ASSERT_EQ(1, a.GetElement(1));
  ASSERT_EQ(2, a.GetElement(2));

  // Swaps two different elements where the indices go up.
  a.Swap(0, 1);
  ASSERT_EQ(1, a.GetElement(0));
  ASSERT_EQ(0, a.GetElement(1));
  ASSERT_EQ(2, a.GetElement(2));

  // Swaps two different elements where the indices go down.
  a.Swap(2, 0);
  ASSERT_EQ(2, a.GetElement(0));
  ASSERT_EQ(0, a.GetElement(1));
  ASSERT_EQ(1, a.GetElement(2));
}

TEST(VectorTest, Clone) {
  // Clones an empty Vector.
  Vector<int> a;
  scoped_ptr<Vector<int> > empty(a.Clone());
  EXPECT_EQ(0, empty->size());

  // Clones a singleton.
  a.PushBack(42);
  scoped_ptr<Vector<int> > singleton(a.Clone());
  ASSERT_EQ(1, singleton->size());
  EXPECT_EQ(42, singleton->GetElement(0));

  // Clones a Vector with more elements.
  a.PushBack(43);
  a.PushBack(44);
  scoped_ptr<Vector<int> > big(a.Clone());
  ASSERT_EQ(3, big->size());
  EXPECT_EQ(42, big->GetElement(0));
  EXPECT_EQ(43, big->GetElement(1));
  EXPECT_EQ(44, big->GetElement(2));
}

// Tests Vector::Erase().
TEST(VectorDeathTest, Erase) {
  Vector<int> a;

  // Tests erasing from an empty vector.
  EXPECT_DEATH_IF_SUPPORTED(
      a.Erase(0),
      "Invalid Vector index 0: must be in range \\[0, -1\\]\\.");

  // Tests erasing from a singleton vector.
  a.PushBack(0);

  a.Erase(0);
  EXPECT_EQ(0, a.size());

  // Tests Erase parameters beyond the bounds of the vector.
  Vector<int> a1;
  a1.PushBack(0);
  a1.PushBack(1);
  a1.PushBack(2);

  EXPECT_DEATH_IF_SUPPORTED(
      a1.Erase(3),
      "Invalid Vector index 3: must be in range \\[0, 2\\]\\.");
  EXPECT_DEATH_IF_SUPPORTED(
      a1.Erase(-1),
      "Invalid Vector index -1: must be in range \\[0, 2\\]\\.");

  // Tests erasing at the end of the vector.
  Vector<int> a2;
  a2.PushBack(0);
  a2.PushBack(1);
  a2.PushBack(2);

  a2.Erase(2);
  ASSERT_EQ(2, a2.size());
  EXPECT_EQ(0, a2.GetElement(0));
  EXPECT_EQ(1, a2.GetElement(1));

  // Tests erasing in the middle of the vector.
  Vector<int> a3;
  a3.PushBack(0);
  a3.PushBack(1);
  a3.PushBack(2);

  a3.Erase(1);
  ASSERT_EQ(2, a3.size());
  EXPECT_EQ(0, a3.GetElement(0));
  EXPECT_EQ(2, a3.GetElement(1));

  // Tests erasing at the beginning of the vector.
  Vector<int> a4;
  a4.PushBack(0);
  a4.PushBack(1);
  a4.PushBack(2);

  a4.Erase(0);
  ASSERT_EQ(2, a4.size());
  EXPECT_EQ(1, a4.GetElement(0));
  EXPECT_EQ(2, a4.GetElement(1));
}

// Tests the GetElement accessor.
TEST(VectorDeathTest, GetElement) {
  Vector<int> a;
  a.PushBack(0);
  a.PushBack(1);
  a.PushBack(2);
  const Vector<int>& b = a;

  EXPECT_EQ(0, b.GetElement(0));
  EXPECT_EQ(1, b.GetElement(1));
  EXPECT_EQ(2, b.GetElement(2));
  EXPECT_DEATH_IF_SUPPORTED(
      b.GetElement(3),
      "Invalid Vector index 3: must be in range \\[0, 2\\]\\.");
  EXPECT_DEATH_IF_SUPPORTED(
      b.GetElement(-1),
      "Invalid Vector index -1: must be in range \\[0, 2\\]\\.");
}

// Tests the GetMutableElement accessor.
TEST(VectorDeathTest, GetMutableElement) {
  Vector<int> a;
  a.PushBack(0);
  a.PushBack(1);
  a.PushBack(2);

  EXPECT_EQ(0, a.GetMutableElement(0));
  EXPECT_EQ(1, a.GetMutableElement(1));
  EXPECT_EQ(2, a.GetMutableElement(2));

  a.GetMutableElement(0) = 42;
  EXPECT_EQ(42, a.GetMutableElement(0));
  EXPECT_EQ(1, a.GetMutableElement(1));
  EXPECT_EQ(2, a.GetMutableElement(2));

  EXPECT_DEATH_IF_SUPPORTED(
      a.GetMutableElement(3),
      "Invalid Vector index 3: must be in range \\[0, 2\\]\\.");
  EXPECT_DEATH_IF_SUPPORTED(
      a.GetMutableElement(-1),
      "Invalid Vector index -1: must be in range \\[0, 2\\]\\.");
}

TEST(VectorDeathTest, Swap) {
  Vector<int> a;
  a.PushBack(0);
  a.PushBack(1);
  a.PushBack(2);

  EXPECT_DEATH_IF_SUPPORTED(
      a.Swap(-1, 1),
      "Invalid first swap element -1: must be in range \\[0, 2\\]");
  EXPECT_DEATH_IF_SUPPORTED(
      a.Swap(3, 1),
      "Invalid first swap element 3: must be in range \\[0, 2\\]");
  EXPECT_DEATH_IF_SUPPORTED(
      a.Swap(1, -1),
      "Invalid second swap element -1: must be in range \\[0, 2\\]");
  EXPECT_DEATH_IF_SUPPORTED(
      a.Swap(1, 3),
      "Invalid second swap element 3: must be in range \\[0, 2\\]");
}

TEST(VectorDeathTest, ShuffleRange) {
  Vector<int> a;
  a.PushBack(0);
  a.PushBack(1);
  a.PushBack(2);
  testing::internal::Random random(1);

  EXPECT_DEATH_IF_SUPPORTED(
      a.ShuffleRange(&random, -1, 1),
      "Invalid shuffle range start -1: must be in range \\[0, 3\\]");
  EXPECT_DEATH_IF_SUPPORTED(
      a.ShuffleRange(&random, 4, 4),
      "Invalid shuffle range start 4: must be in range \\[0, 3\\]");
  EXPECT_DEATH_IF_SUPPORTED(
      a.ShuffleRange(&random, 3, 2),
      "Invalid shuffle range finish 2: must be in range \\[3, 3\\]");
  EXPECT_DEATH_IF_SUPPORTED(
      a.ShuffleRange(&random, 3, 4),
      "Invalid shuffle range finish 4: must be in range \\[3, 3\\]");
}

class VectorShuffleTest : public Test {
 protected:
  static const int kVectorSize = 20;

  VectorShuffleTest() : random_(1) {
    for (int i = 0; i < kVectorSize; i++) {
      vector_.PushBack(i);
    }
  }

  static bool VectorIsCorrupt(const TestingVector& vector) {
    if (kVectorSize != vector.size()) {
      return true;
    }

    bool found_in_vector[kVectorSize] = { false };
    for (int i = 0; i < vector.size(); i++) {
      const int e = vector.GetElement(i);
      if (e < 0 || e >= kVectorSize || found_in_vector[e]) {
        return true;
      }
      found_in_vector[e] = true;
    }

    // Vector size is correct, elements' range is correct, no
    // duplicate elements.  Therefore no corruption has occurred.
    return false;
  }

  static bool VectorIsNotCorrupt(const TestingVector& vector) {
    return !VectorIsCorrupt(vector);
  }

  static bool RangeIsShuffled(const TestingVector& vector, int begin, int end) {
    for (int i = begin; i < end; i++) {
      if (i != vector.GetElement(i)) {
        return true;
      }
    }
    return false;
  }

  static bool RangeIsUnshuffled(
      const TestingVector& vector, int begin, int end) {
    return !RangeIsShuffled(vector, begin, end);
  }

  static bool VectorIsShuffled(const TestingVector& vector) {
    return RangeIsShuffled(vector, 0, vector.size());
  }

  static bool VectorIsUnshuffled(const TestingVector& vector) {
    return !VectorIsShuffled(vector);
  }

  testing::internal::Random random_;
  TestingVector vector_;
};  // class VectorShuffleTest

const int VectorShuffleTest::kVectorSize;

TEST_F(VectorShuffleTest, HandlesEmptyRange) {
  // Tests an empty range at the beginning...
  vector_.ShuffleRange(&random_, 0, 0);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsUnshuffled, vector_);

  // ...in the middle...
  vector_.ShuffleRange(&random_, kVectorSize/2, kVectorSize/2);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsUnshuffled, vector_);

  // ...at the end...
  vector_.ShuffleRange(&random_, kVectorSize - 1, kVectorSize - 1);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsUnshuffled, vector_);

  // ...and past the end.
  vector_.ShuffleRange(&random_, kVectorSize, kVectorSize);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsUnshuffled, vector_);
}

TEST_F(VectorShuffleTest, HandlesRangeOfSizeOne) {
  // Tests a size one range at the beginning...
  vector_.ShuffleRange(&random_, 0, 1);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsUnshuffled, vector_);

  // ...in the middle...
  vector_.ShuffleRange(&random_, kVectorSize/2, kVectorSize/2 + 1);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsUnshuffled, vector_);

  // ...and at the end.
  vector_.ShuffleRange(&random_, kVectorSize - 1, kVectorSize);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsUnshuffled, vector_);
}

// Because we use our own random number generator and a fixed seed,
// we can guarantee that the following "random" tests will succeed.

TEST_F(VectorShuffleTest, ShufflesEntireVector) {
  vector_.Shuffle(&random_);
  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  EXPECT_FALSE(VectorIsUnshuffled(vector_)) << vector_;

  // Tests the first and last elements in particular to ensure that
  // there are no off-by-one problems in our shuffle algorithm.
  EXPECT_NE(0, vector_.GetElement(0));
  EXPECT_NE(kVectorSize - 1, vector_.GetElement(kVectorSize - 1));
}

TEST_F(VectorShuffleTest, ShufflesStartOfVector) {
  const int kRangeSize = kVectorSize/2;

  vector_.ShuffleRange(&random_, 0, kRangeSize);

  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  EXPECT_PRED3(RangeIsShuffled, vector_, 0, kRangeSize);
  EXPECT_PRED3(RangeIsUnshuffled, vector_, kRangeSize, kVectorSize);
}

TEST_F(VectorShuffleTest, ShufflesEndOfVector) {
  const int kRangeSize = kVectorSize / 2;
  vector_.ShuffleRange(&random_, kRangeSize, kVectorSize);

  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  EXPECT_PRED3(RangeIsUnshuffled, vector_, 0, kRangeSize);
  EXPECT_PRED3(RangeIsShuffled, vector_, kRangeSize, kVectorSize);
}

TEST_F(VectorShuffleTest, ShufflesMiddleOfVector) {
  int kRangeSize = kVectorSize/3;
  vector_.ShuffleRange(&random_, kRangeSize, 2*kRangeSize);

  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  EXPECT_PRED3(RangeIsUnshuffled, vector_, 0, kRangeSize);
  EXPECT_PRED3(RangeIsShuffled, vector_, kRangeSize, 2*kRangeSize);
  EXPECT_PRED3(RangeIsUnshuffled, vector_, 2*kRangeSize, kVectorSize);
}

TEST_F(VectorShuffleTest, ShufflesRepeatably) {
  TestingVector vector2;
  for (int i = 0; i < kVectorSize; i++) {
    vector2.PushBack(i);
  }

  random_.Reseed(1234);
  vector_.Shuffle(&random_);
  random_.Reseed(1234);
  vector2.Shuffle(&random_);

  ASSERT_PRED1(VectorIsNotCorrupt, vector_);
  ASSERT_PRED1(VectorIsNotCorrupt, vector2);

  for (int i = 0; i < kVectorSize; i++) {
    EXPECT_EQ(vector_.GetElement(i), vector2.GetElement(i))
        << " where i is " << i;
  }
}

// Tests the size of the AssertHelper class.

TEST(AssertHelperTest, AssertHelperIsSmall) {
  // To avoid breaking clients that use lots of assertions in one
  // function, we cannot grow the size of AssertHelper.
  EXPECT_LE(sizeof(testing::internal::AssertHelper), sizeof(void*));
}

// Tests the String class.

// Tests String's constructors.
TEST(StringTest, Constructors) {
  // Default ctor.
  String s1;
  // We aren't using EXPECT_EQ(NULL, s1.c_str()) because comparing
  // pointers with NULL isn't supported on all platforms.
  EXPECT_EQ(0U, s1.length());
  EXPECT_TRUE(NULL == s1.c_str());

  // Implicitly constructs from a C-string.
  String s2 = "Hi";
  EXPECT_EQ(2U, s2.length());
  EXPECT_STREQ("Hi", s2.c_str());

  // Constructs from a C-string and a length.
  String s3("hello", 3);
  EXPECT_EQ(3U, s3.length());
  EXPECT_STREQ("hel", s3.c_str());

  // The empty String should be created when String is constructed with
  // a NULL pointer and length 0.
  EXPECT_EQ(0U, String(NULL, 0).length());
  EXPECT_FALSE(String(NULL, 0).c_str() == NULL);

  // Constructs a String that contains '\0'.
  String s4("a\0bcd", 4);
  EXPECT_EQ(4U, s4.length());
  EXPECT_EQ('a', s4.c_str()[0]);
  EXPECT_EQ('\0', s4.c_str()[1]);
  EXPECT_EQ('b', s4.c_str()[2]);
  EXPECT_EQ('c', s4.c_str()[3]);

  // Copy ctor where the source is NULL.
  const String null_str;
  String s5 = null_str;
  EXPECT_TRUE(s5.c_str() == NULL);

  // Copy ctor where the source isn't NULL.
  String s6 = s3;
  EXPECT_EQ(3U, s6.length());
  EXPECT_STREQ("hel", s6.c_str());

  // Copy ctor where the source contains '\0'.
  String s7 = s4;
  EXPECT_EQ(4U, s7.length());
  EXPECT_EQ('a', s7.c_str()[0]);
  EXPECT_EQ('\0', s7.c_str()[1]);
  EXPECT_EQ('b', s7.c_str()[2]);
  EXPECT_EQ('c', s7.c_str()[3]);
}

#if GTEST_HAS_STD_STRING

TEST(StringTest, ConvertsFromStdString) {
  // An empty std::string.
  const std::string src1("");
  const String dest1 = src1;
  EXPECT_EQ(0U, dest1.length());
  EXPECT_STREQ("", dest1.c_str());

  // A normal std::string.
  const std::string src2("Hi");
  const String dest2 = src2;
  EXPECT_EQ(2U, dest2.length());
  EXPECT_STREQ("Hi", dest2.c_str());

  // An std::string with an embedded NUL character.
  const char src3[] = "a\0b";
  const String dest3 = std::string(src3, sizeof(src3));
  EXPECT_EQ(sizeof(src3), dest3.length());
  EXPECT_EQ('a', dest3.c_str()[0]);
  EXPECT_EQ('\0', dest3.c_str()[1]);
  EXPECT_EQ('b', dest3.c_str()[2]);
}

TEST(StringTest, ConvertsToStdString) {
  // An empty String.
  const String src1("");
  const std::string dest1 = src1;
  EXPECT_EQ("", dest1);

  // A normal String.
  const String src2("Hi");
  const std::string dest2 = src2;
  EXPECT_EQ("Hi", dest2);

  // A String containing a '\0'.
  const String src3("x\0y", 3);
  const std::string dest3 = src3;
  EXPECT_EQ(std::string("x\0y", 3), dest3);
}

#endif  // GTEST_HAS_STD_STRING

#if GTEST_HAS_GLOBAL_STRING

TEST(StringTest, ConvertsFromGlobalString) {
  // An empty ::string.
  const ::string src1("");
  const String dest1 = src1;
  EXPECT_EQ(0U, dest1.length());
  EXPECT_STREQ("", dest1.c_str());

  // A normal ::string.
  const ::string src2("Hi");
  const String dest2 = src2;
  EXPECT_EQ(2U, dest2.length());
  EXPECT_STREQ("Hi", dest2.c_str());

  // An ::string with an embedded NUL character.
  const char src3[] = "x\0y";
  const String dest3 = ::string(src3, sizeof(src3));
  EXPECT_EQ(sizeof(src3), dest3.length());
  EXPECT_EQ('x', dest3.c_str()[0]);
  EXPECT_EQ('\0', dest3.c_str()[1]);
  EXPECT_EQ('y', dest3.c_str()[2]);
}

TEST(StringTest, ConvertsToGlobalString) {
  // An empty String.
  const String src1("");
  const ::string dest1 = src1;
  EXPECT_EQ("", dest1);

  // A normal String.
  const String src2("Hi");
  const ::string dest2 = src2;
  EXPECT_EQ("Hi", dest2);

  const String src3("x\0y", 3);
  const ::string dest3 = src3;
  EXPECT_EQ(::string("x\0y", 3), dest3);
}

#endif  // GTEST_HAS_GLOBAL_STRING

// Tests String::ShowCStringQuoted().
TEST(StringTest, ShowCStringQuoted) {
  EXPECT_STREQ("(null)",
               String::ShowCStringQuoted(NULL).c_str());
  EXPECT_STREQ("\"\"",
               String::ShowCStringQuoted("").c_str());
  EXPECT_STREQ("\"foo\"",
               String::ShowCStringQuoted("foo").c_str());
}

// Tests String::empty().
TEST(StringTest, Empty) {
  EXPECT_TRUE(String("").empty());
  EXPECT_FALSE(String().empty());
  EXPECT_FALSE(String(NULL).empty());
  EXPECT_FALSE(String("a").empty());
  EXPECT_FALSE(String("\0", 1).empty());
}

// Tests String::Compare().
TEST(StringTest, Compare) {
  // NULL vs NULL.
  EXPECT_EQ(0, String().Compare(String()));

  // NULL vs non-NULL.
  EXPECT_EQ(-1, String().Compare(String("")));

  // Non-NULL vs NULL.
  EXPECT_EQ(1, String("").Compare(String()));

  // The following covers non-NULL vs non-NULL.

  // "" vs "".
  EXPECT_EQ(0, String("").Compare(String("")));

  // "" vs non-"".
  EXPECT_EQ(-1, String("").Compare(String("\0", 1)));
  EXPECT_EQ(-1, String("").Compare(" "));

  // Non-"" vs "".
  EXPECT_EQ(1, String("a").Compare(String("")));

  // The following covers non-"" vs non-"".

  // Same length and equal.
  EXPECT_EQ(0, String("a").Compare(String("a")));

  // Same length and different.
  EXPECT_EQ(-1, String("a\0b", 3).Compare(String("a\0c", 3)));
  EXPECT_EQ(1, String("b").Compare(String("a")));

  // Different lengths.
  EXPECT_EQ(-1, String("a").Compare(String("ab")));
  EXPECT_EQ(-1, String("a").Compare(String("a\0", 2)));
  EXPECT_EQ(1, String("abc").Compare(String("aacd")));
}

// Tests String::operator==().
TEST(StringTest, Equals) {
  const String null(NULL);
  EXPECT_TRUE(null == NULL);  // NOLINT
  EXPECT_FALSE(null == "");  // NOLINT
  EXPECT_FALSE(null == "bar");  // NOLINT

  const String empty("");
  EXPECT_FALSE(empty == NULL);  // NOLINT
  EXPECT_TRUE(empty == "");  // NOLINT
  EXPECT_FALSE(empty == "bar");  // NOLINT

  const String foo("foo");
  EXPECT_FALSE(foo == NULL);  // NOLINT
  EXPECT_FALSE(foo == "");  // NOLINT
  EXPECT_FALSE(foo == "bar");  // NOLINT
  EXPECT_TRUE(foo == "foo");  // NOLINT

  const String bar("x\0y", 3);
  EXPECT_FALSE(bar == "x");
}

// Tests String::operator!=().
TEST(StringTest, NotEquals) {
  const String null(NULL);
  EXPECT_FALSE(null != NULL);  // NOLINT
  EXPECT_TRUE(null != "");  // NOLINT
  EXPECT_TRUE(null != "bar");  // NOLINT

  const String empty("");
  EXPECT_TRUE(empty != NULL);  // NOLINT
  EXPECT_FALSE(empty != "");  // NOLINT
  EXPECT_TRUE(empty != "bar");  // NOLINT

  const String foo("foo");
  EXPECT_TRUE(foo != NULL);  // NOLINT
  EXPECT_TRUE(foo != "");  // NOLINT
  EXPECT_TRUE(foo != "bar");  // NOLINT
  EXPECT_FALSE(foo != "foo");  // NOLINT

  const String bar("x\0y", 3);
  EXPECT_TRUE(bar != "x");
}

// Tests String::length().
TEST(StringTest, Length) {
  EXPECT_EQ(0U, String().length());
  EXPECT_EQ(0U, String("").length());
  EXPECT_EQ(2U, String("ab").length());
  EXPECT_EQ(3U, String("a\0b", 3).length());
}

// Tests String::EndsWith().
TEST(StringTest, EndsWith) {
  EXPECT_TRUE(String("foobar").EndsWith("bar"));
  EXPECT_TRUE(String("foobar").EndsWith(""));
  EXPECT_TRUE(String("").EndsWith(""));

  EXPECT_FALSE(String("foobar").EndsWith("foo"));
  EXPECT_FALSE(String("").EndsWith("foo"));
}

// Tests String::EndsWithCaseInsensitive().
TEST(StringTest, EndsWithCaseInsensitive) {
  EXPECT_TRUE(String("foobar").EndsWithCaseInsensitive("BAR"));
  EXPECT_TRUE(String("foobaR").EndsWithCaseInsensitive("bar"));
  EXPECT_TRUE(String("foobar").EndsWithCaseInsensitive(""));
  EXPECT_TRUE(String("").EndsWithCaseInsensitive(""));

  EXPECT_FALSE(String("Foobar").EndsWithCaseInsensitive("foo"));
  EXPECT_FALSE(String("foobar").EndsWithCaseInsensitive("Foo"));
  EXPECT_FALSE(String("").EndsWithCaseInsensitive("foo"));
}

// C++Builder's preprocessor is buggy; it fails to expand macros that
// appear in macro parameters after wide char literals.  Provide an alias
// for NULL as a workaround.
static const wchar_t* const kNull = NULL;

// Tests String::CaseInsensitiveWideCStringEquals
TEST(StringTest, CaseInsensitiveWideCStringEquals) {
  EXPECT_TRUE(String::CaseInsensitiveWideCStringEquals(NULL, NULL));
  EXPECT_FALSE(String::CaseInsensitiveWideCStringEquals(kNull, L""));
  EXPECT_FALSE(String::CaseInsensitiveWideCStringEquals(L"", kNull));
  EXPECT_FALSE(String::CaseInsensitiveWideCStringEquals(kNull, L"foobar"));
  EXPECT_FALSE(String::CaseInsensitiveWideCStringEquals(L"foobar", kNull));
  EXPECT_TRUE(String::CaseInsensitiveWideCStringEquals(L"foobar", L"foobar"));
  EXPECT_TRUE(String::CaseInsensitiveWideCStringEquals(L"foobar", L"FOOBAR"));
  EXPECT_TRUE(String::CaseInsensitiveWideCStringEquals(L"FOOBAR", L"foobar"));
}

// Tests that NULL can be assigned to a String.
TEST(StringTest, CanBeAssignedNULL) {
  const String src(NULL);
  String dest;

  dest = src;
  EXPECT_STREQ(NULL, dest.c_str());
}

// Tests that the empty string "" can be assigned to a String.
TEST(StringTest, CanBeAssignedEmpty) {
  const String src("");
  String dest;

  dest = src;
  EXPECT_STREQ("", dest.c_str());
}

// Tests that a non-empty string can be assigned to a String.
TEST(StringTest, CanBeAssignedNonEmpty) {
  const String src("hello");
  String dest;
  dest = src;
  EXPECT_EQ(5U, dest.length());
  EXPECT_STREQ("hello", dest.c_str());

  const String src2("x\0y", 3);
  String dest2;
  dest2 = src2;
  EXPECT_EQ(3U, dest2.length());
  EXPECT_EQ('x', dest2.c_str()[0]);
  EXPECT_EQ('\0', dest2.c_str()[1]);
  EXPECT_EQ('y', dest2.c_str()[2]);
}

// Tests that a String can be assigned to itself.
TEST(StringTest, CanBeAssignedSelf) {
  String dest("hello");

  dest = dest;
  EXPECT_STREQ("hello", dest.c_str());
}

// Tests streaming a String.
TEST(StringTest, Streams) {
  EXPECT_EQ(StreamableToString(String()), "(null)");
  EXPECT_EQ(StreamableToString(String("")), "");
  EXPECT_EQ(StreamableToString(String("a\0b", 3)), "a\\0b");
}

// Tests that String::Format() works.
TEST(StringTest, FormatWorks) {
  // Normal case: the format spec is valid, the arguments match the
  // spec, and the result is < 4095 characters.
  EXPECT_STREQ("Hello, 42", String::Format("%s, %d", "Hello", 42).c_str());

  // Edge case: the result is 4095 characters.
  char buffer[4096];
  const size_t kSize = sizeof(buffer);
  memset(buffer, 'a', kSize - 1);
  buffer[kSize - 1] = '\0';
  EXPECT_STREQ(buffer, String::Format("%s", buffer).c_str());

  // The result needs to be 4096 characters, exceeding Format()'s limit.
  EXPECT_STREQ("<formatting error or buffer exceeded>",
               String::Format("x%s", buffer).c_str());

#if GTEST_OS_LINUX
  // On Linux, invalid format spec should lead to an error message.
  // In other environment (e.g. MSVC on Windows), String::Format() may
  // simply ignore a bad format spec, so this assertion is run on
  // Linux only.
  EXPECT_STREQ("<formatting error or buffer exceeded>",
               String::Format("%").c_str());
#endif
}

#if GTEST_OS_WINDOWS

// Tests String::ShowWideCString().
TEST(StringTest, ShowWideCString) {
  EXPECT_STREQ("(null)",
               String::ShowWideCString(NULL).c_str());
  EXPECT_STREQ("", String::ShowWideCString(L"").c_str());
  EXPECT_STREQ("foo", String::ShowWideCString(L"foo").c_str());
}

// Tests String::ShowWideCStringQuoted().
TEST(StringTest, ShowWideCStringQuoted) {
  EXPECT_STREQ("(null)",
               String::ShowWideCStringQuoted(NULL).c_str());
  EXPECT_STREQ("L\"\"",
               String::ShowWideCStringQuoted(L"").c_str());
  EXPECT_STREQ("L\"foo\"",
               String::ShowWideCStringQuoted(L"foo").c_str());
}

#if GTEST_OS_WINDOWS_MOBILE
TEST(StringTest, AnsiAndUtf16Null) {
  EXPECT_EQ(NULL, String::AnsiToUtf16(NULL));
  EXPECT_EQ(NULL, String::Utf16ToAnsi(NULL));
}

TEST(StringTest, AnsiAndUtf16ConvertBasic) {
  const char* ansi = String::Utf16ToAnsi(L"str");
  EXPECT_STREQ("str", ansi);
  delete [] ansi;
  const WCHAR* utf16 = String::AnsiToUtf16("str");
  EXPECT_EQ(0, wcsncmp(L"str", utf16, 3));
  delete [] utf16;
}

TEST(StringTest, AnsiAndUtf16ConvertPathChars) {
  const char* ansi = String::Utf16ToAnsi(L".:\\ \"*?");
  EXPECT_STREQ(".:\\ \"*?", ansi);
  delete [] ansi;
  const WCHAR* utf16 = String::AnsiToUtf16(".:\\ \"*?");
  EXPECT_EQ(0, wcsncmp(L".:\\ \"*?", utf16, 3));
  delete [] utf16;
}
#endif  // GTEST_OS_WINDOWS_MOBILE

#endif  // GTEST_OS_WINDOWS

// Tests TestProperty construction.
TEST(TestPropertyTest, StringValue) {
  TestProperty property("key", "1");
  EXPECT_STREQ("key", property.key());
  EXPECT_STREQ("1", property.value());
}

// Tests TestProperty replacing a value.
TEST(TestPropertyTest, ReplaceStringValue) {
  TestProperty property("key", "1");
  EXPECT_STREQ("1", property.value());
  property.SetValue("2");
  EXPECT_STREQ("2", property.value());
}

// AddFatalFailure() and AddNonfatalFailure() must be stand-alone
// functions (i.e. their definitions cannot be inlined at the call
// sites), or C++Builder won't compile the code.
static void AddFatalFailure() {
  FAIL() << "Expected fatal failure.";
}

static void AddNonfatalFailure() {
  ADD_FAILURE() << "Expected non-fatal failure.";
}

class ScopedFakeTestPartResultReporterTest : public Test {
 public:  // Must be public and not protected due to a bug in g++ 3.4.2.
  enum FailureMode {
    FATAL_FAILURE,
    NONFATAL_FAILURE
  };
  static void AddFailure(FailureMode failure) {
    if (failure == FATAL_FAILURE) {
      AddFatalFailure();
    } else {
      AddNonfatalFailure();
    }
  }
};

// Tests that ScopedFakeTestPartResultReporter intercepts test
// failures.
TEST_F(ScopedFakeTestPartResultReporterTest, InterceptsTestFailures) {
  TestPartResultArray results;
  {
    ScopedFakeTestPartResultReporter reporter(
        ScopedFakeTestPartResultReporter::INTERCEPT_ONLY_CURRENT_THREAD,
        &results);
    AddFailure(NONFATAL_FAILURE);
    AddFailure(FATAL_FAILURE);
  }

  EXPECT_EQ(2, results.size());
  EXPECT_TRUE(results.GetTestPartResult(0).nonfatally_failed());
  EXPECT_TRUE(results.GetTestPartResult(1).fatally_failed());
}

TEST_F(ScopedFakeTestPartResultReporterTest, DeprecatedConstructor) {
  TestPartResultArray results;
  {
    // Tests, that the deprecated constructor still works.
    ScopedFakeTestPartResultReporter reporter(&results);
    AddFailure(NONFATAL_FAILURE);
  }
  EXPECT_EQ(1, results.size());
}

#if GTEST_IS_THREADSAFE && GTEST_HAS_PTHREAD

class ScopedFakeTestPartResultReporterWithThreadsTest
  : public ScopedFakeTestPartResultReporterTest {
 protected:
  static void AddFailureInOtherThread(FailureMode failure) {
    pthread_t tid;
    pthread_create(&tid,
                   NULL,
                   ScopedFakeTestPartResultReporterWithThreadsTest::
                       FailureThread,
                   &failure);
    pthread_join(tid, NULL);
  }
 private:
  static void* FailureThread(void* attr) {
    FailureMode* failure = static_cast<FailureMode*>(attr);
    AddFailure(*failure);
    return NULL;
  }
};

TEST_F(ScopedFakeTestPartResultReporterWithThreadsTest,
       InterceptsTestFailuresInAllThreads) {
  TestPartResultArray results;
  {
    ScopedFakeTestPartResultReporter reporter(
        ScopedFakeTestPartResultReporter::INTERCEPT_ALL_THREADS, &results);
    AddFailure(NONFATAL_FAILURE);
    AddFailure(FATAL_FAILURE);
    AddFailureInOtherThread(NONFATAL_FAILURE);
    AddFailureInOtherThread(FATAL_FAILURE);
  }

  EXPECT_EQ(4, results.size());
  EXPECT_TRUE(results.GetTestPartResult(0).nonfatally_failed());
  EXPECT_TRUE(results.GetTestPartResult(1).fatally_failed());
  EXPECT_TRUE(results.GetTestPartResult(2).nonfatally_failed());
  EXPECT_TRUE(results.GetTestPartResult(3).fatally_failed());
}

#endif  // GTEST_IS_THREADSAFE && GTEST_HAS_PTHREAD

// Tests EXPECT_FATAL_FAILURE{,ON_ALL_THREADS}.  Makes sure that they
// work even if the failure is generated in a called function rather than
// the current context.

typedef ScopedFakeTestPartResultReporterTest ExpectFatalFailureTest;

TEST_F(ExpectFatalFailureTest, CatchesFatalFaliure) {
  EXPECT_FATAL_FAILURE(AddFatalFailure(), "Expected fatal failure.");
}

TEST_F(ExpectFatalFailureTest, CatchesFatalFailureOnAllThreads) {
  // We have another test below to verify that the macro catches fatal
  // failures generated on another thread.
  EXPECT_FATAL_FAILURE_ON_ALL_THREADS(AddFatalFailure(),
                                      "Expected fatal failure.");
}

#ifdef __BORLANDC__
// Silences warnings: "Condition is always true"
#pragma option push -w-ccc
#endif

// Tests that EXPECT_FATAL_FAILURE() can be used in a non-void
// function even when the statement in it contains ASSERT_*.

int NonVoidFunction() {
  EXPECT_FATAL_FAILURE(ASSERT_TRUE(false), "");
  EXPECT_FATAL_FAILURE_ON_ALL_THREADS(FAIL(), "");
  return 0;
}

TEST_F(ExpectFatalFailureTest, CanBeUsedInNonVoidFunction) {
  NonVoidFunction();
}

// Tests that EXPECT_FATAL_FAILURE(statement, ...) doesn't abort the
// current function even though 'statement' generates a fatal failure.

void DoesNotAbortHelper(bool* aborted) {
  EXPECT_FATAL_FAILURE(ASSERT_TRUE(false), "");
  EXPECT_FATAL_FAILURE_ON_ALL_THREADS(FAIL(), "");

  *aborted = false;
}

#ifdef __BORLANDC__
// Restores warnings after previous "#pragma option push" supressed them
#pragma option pop
#endif

TEST_F(ExpectFatalFailureTest, DoesNotAbort) {
  bool aborted = true;
  DoesNotAbortHelper(&aborted);
  EXPECT_FALSE(aborted);
}

// Tests that the EXPECT_FATAL_FAILURE{,_ON_ALL_THREADS} accepts a
// statement that contains a macro which expands to code containing an
// unprotected comma.

static int global_var = 0;
#define GTEST_USE_UNPROTECTED_COMMA_ global_var++, global_var++

TEST_F(ExpectFatalFailureTest, AcceptsMacroThatExpandsToUnprotectedComma) {
#ifndef __BORLANDC__
  // ICE's in C++Builder 2007.
  EXPECT_FATAL_FAILURE({
    GTEST_USE_UNPROTECTED_COMMA_;
    AddFatalFailure();
  }, "");
#endif

  EXPECT_FATAL_FAILURE_ON_ALL_THREADS({
    GTEST_USE_UNPROTECTED_COMMA_;
    AddFatalFailure();
  }, "");
}

// Tests EXPECT_NONFATAL_FAILURE{,ON_ALL_THREADS}.

typedef ScopedFakeTestPartResultReporterTest ExpectNonfatalFailureTest;

TEST_F(ExpectNonfatalFailureTest, CatchesNonfatalFailure) {
  EXPECT_NONFATAL_FAILURE(AddNonfatalFailure(),
                          "Expected non-fatal failure.");
}

TEST_F(ExpectNonfatalFailureTest, CatchesNonfatalFailureOnAllThreads) {
  // We have another test below to verify that the macro catches
  // non-fatal failures generated on another thread.
  EXPECT_NONFATAL_FAILURE_ON_ALL_THREADS(AddNonfatalFailure(),
                                         "Expected non-fatal failure.");
}

// Tests that the EXPECT_NONFATAL_FAILURE{,_ON_ALL_THREADS} accepts a
// statement that contains a macro which expands to code containing an
// unprotected comma.
TEST_F(ExpectNonfatalFailureTest, AcceptsMacroThatExpandsToUnprotectedComma) {
  EXPECT_NONFATAL_FAILURE({
    GTEST_USE_UNPROTECTED_COMMA_;
    AddNonfatalFailure();
  }, "");

  EXPECT_NONFATAL_FAILURE_ON_ALL_THREADS({
    GTEST_USE_UNPROTECTED_COMMA_;
    AddNonfatalFailure();
  }, "");
}

#if GTEST_IS_THREADSAFE && GTEST_HAS_PTHREAD

typedef ScopedFakeTestPartResultReporterWithThreadsTest
    ExpectFailureWithThreadsTest;

TEST_F(ExpectFailureWithThreadsTest, ExpectFatalFailureOnAllThreads) {
  EXPECT_FATAL_FAILURE_ON_ALL_THREADS(AddFailureInOtherThread(FATAL_FAILURE),
                                      "Expected fatal failure.");
}

TEST_F(ExpectFailureWithThreadsTest, ExpectNonFatalFailureOnAllThreads) {
  EXPECT_NONFATAL_FAILURE_ON_ALL_THREADS(
      AddFailureInOtherThread(NONFATAL_FAILURE), "Expected non-fatal failure.");
}

#endif  // GTEST_IS_THREADSAFE && GTEST_HAS_PTHREAD

// Tests the TestProperty class.

TEST(TestPropertyTest, ConstructorWorks) {
  const TestProperty property("key", "value");
  EXPECT_STREQ("key", property.key());
  EXPECT_STREQ("value", property.value());
}

TEST(TestPropertyTest, SetValue) {
  TestProperty property("key", "value_1");
  EXPECT_STREQ("key", property.key());
  property.SetValue("value_2");
  EXPECT_STREQ("key", property.key());
  EXPECT_STREQ("value_2", property.value());
}

// Tests the TestPartResult class.

TEST(TestPartResultTest, ConstructorWorks) {
  Message message;
  message << "something is terribly wrong";
  message << static_cast<const char*>(testing::internal::kStackTraceMarker);
  message << "some unimportant stack trace";

  const TestPartResult result(TestPartResult::kNonFatalFailure,
                              "some_file.cc",
                              42,
                              message.GetString().c_str());

  EXPECT_EQ(TestPartResult::kNonFatalFailure, result.type());
  EXPECT_STREQ("some_file.cc", result.file_name());
  EXPECT_EQ(42, result.line_number());
  EXPECT_STREQ(message.GetString().c_str(), result.message());
  EXPECT_STREQ("something is terribly wrong", result.summary());
}

TEST(TestPartResultTest, ResultAccessorsWork) {
  const TestPartResult success(TestPartResult::kSuccess,
                               "file.cc",
                               42,
                               "message");
  EXPECT_TRUE(success.passed());
  EXPECT_FALSE(success.failed());
  EXPECT_FALSE(success.nonfatally_failed());
  EXPECT_FALSE(success.fatally_failed());

  const TestPartResult nonfatal_failure(TestPartResult::kNonFatalFailure,
                                        "file.cc",
                                        42,
                                        "message");
  EXPECT_FALSE(nonfatal_failure.passed());
  EXPECT_TRUE(nonfatal_failure.failed());
  EXPECT_TRUE(nonfatal_failure.nonfatally_failed());
  EXPECT_FALSE(nonfatal_failure.fatally_failed());

  const TestPartResult fatal_failure(TestPartResult::kFatalFailure,
                                     "file.cc",
                                     42,
                                     "message");
  EXPECT_FALSE(fatal_failure.passed());
  EXPECT_TRUE(fatal_failure.failed());
  EXPECT_FALSE(fatal_failure.nonfatally_failed());
  EXPECT_TRUE(fatal_failure.fatally_failed());
}

// Tests the TestResult class

// The test fixture for testing TestResult.
class TestResultTest : public Test {
 protected:
  typedef Vector<TestPartResult> TPRVector;

  // We make use of 2 TestPartResult objects,
  TestPartResult * pr1, * pr2;

  // ... and 3 TestResult objects.
  TestResult * r0, * r1, * r2;

  virtual void SetUp() {
    // pr1 is for success.
    pr1 = new TestPartResult(TestPartResult::kSuccess,
                             "foo/bar.cc",
                             10,
                             "Success!");

    // pr2 is for fatal failure.
    pr2 = new TestPartResult(TestPartResult::kFatalFailure,
                             "foo/bar.cc",
                             -1,  // This line number means "unknown"
                             "Failure!");

    // Creates the TestResult objects.
    r0 = new TestResult();
    r1 = new TestResult();
    r2 = new TestResult();

    // In order to test TestResult, we need to modify its internal
    // state, in particular the TestPartResult Vector it holds.
    // test_part_results() returns a const reference to this Vector.
    // We cast it to a non-const object s.t. it can be modified (yes,
    // this is a hack).
    TPRVector* results1 = const_cast<Vector<TestPartResult> *>(
        &TestResultAccessor::test_part_results(*r1));
    TPRVector* results2 = const_cast<Vector<TestPartResult> *>(
        &TestResultAccessor::test_part_results(*r2));

    // r0 is an empty TestResult.

    // r1 contains a single SUCCESS TestPartResult.
    results1->PushBack(*pr1);

    // r2 contains a SUCCESS, and a FAILURE.
    results2->PushBack(*pr1);
    results2->PushBack(*pr2);
  }

  virtual void TearDown() {
    delete pr1;
    delete pr2;

    delete r0;
    delete r1;
    delete r2;
  }

  // Helper that compares two two TestPartResults.
  static void CompareTestPartResult(const TestPartResult& expected,
                                    const TestPartResult& actual) {
    EXPECT_EQ(expected.type(), actual.type());
    EXPECT_STREQ(expected.file_name(), actual.file_name());
    EXPECT_EQ(expected.line_number(), actual.line_number());
    EXPECT_STREQ(expected.summary(), actual.summary());
    EXPECT_STREQ(expected.message(), actual.message());
    EXPECT_EQ(expected.passed(), actual.passed());
    EXPECT_EQ(expected.failed(), actual.failed());
    EXPECT_EQ(expected.nonfatally_failed(), actual.nonfatally_failed());
    EXPECT_EQ(expected.fatally_failed(), actual.fatally_failed());
  }
};

// Tests TestResult::total_part_count().
TEST_F(TestResultTest, total_part_count) {
  ASSERT_EQ(0, r0->total_part_count());
  ASSERT_EQ(1, r1->total_part_count());
  ASSERT_EQ(2, r2->total_part_count());
}

// Tests TestResult::Passed().
TEST_F(TestResultTest, Passed) {
  ASSERT_TRUE(r0->Passed());
  ASSERT_TRUE(r1->Passed());
  ASSERT_FALSE(r2->Passed());
}

// Tests TestResult::Failed().
TEST_F(TestResultTest, Failed) {
  ASSERT_FALSE(r0->Failed());
  ASSERT_FALSE(r1->Failed());
  ASSERT_TRUE(r2->Failed());
}

// Tests TestResult::GetTestPartResult().

typedef TestResultTest TestResultDeathTest;

TEST_F(TestResultDeathTest, GetTestPartResult) {
  CompareTestPartResult(*pr1, r2->GetTestPartResult(0));
  CompareTestPartResult(*pr2, r2->GetTestPartResult(1));
  EXPECT_DEATH_IF_SUPPORTED(
      r2->GetTestPartResult(2),
      "Invalid Vector index 2: must be in range \\[0, 1\\]\\.");
  EXPECT_DEATH_IF_SUPPORTED(
      r2->GetTestPartResult(-1),
      "Invalid Vector index -1: must be in range \\[0, 1\\]\\.");
}

// Tests TestResult has no properties when none are added.
TEST(TestResultPropertyTest, NoPropertiesFoundWhenNoneAreAdded) {
  TestResult test_result;
  ASSERT_EQ(0, test_result.test_property_count());
}

// Tests TestResult has the expected property when added.
TEST(TestResultPropertyTest, OnePropertyFoundWhenAdded) {
  TestResult test_result;
  TestProperty property("key_1", "1");
  TestResultAccessor::RecordProperty(&test_result, property);
  ASSERT_EQ(1, test_result.test_property_count());
  const TestProperty& actual_property = test_result.GetTestProperty(0);
  EXPECT_STREQ("key_1", actual_property.key());
  EXPECT_STREQ("1", actual_property.value());
}

// Tests TestResult has multiple properties when added.
TEST(TestResultPropertyTest, MultiplePropertiesFoundWhenAdded) {
  TestResult test_result;
  TestProperty property_1("key_1", "1");
  TestProperty property_2("key_2", "2");
  TestResultAccessor::RecordProperty(&test_result, property_1);
  TestResultAccessor::RecordProperty(&test_result, property_2);
  ASSERT_EQ(2, test_result.test_property_count());
  const TestProperty& actual_property_1 = test_result.GetTestProperty(0);
  EXPECT_STREQ("key_1", actual_property_1.key());
  EXPECT_STREQ("1", actual_property_1.value());

  const TestProperty& actual_property_2 = test_result.GetTestProperty(1);
  EXPECT_STREQ("key_2", actual_property_2.key());
  EXPECT_STREQ("2", actual_property_2.value());
}

// Tests TestResult::RecordProperty() overrides values for duplicate keys.
TEST(TestResultPropertyTest, OverridesValuesForDuplicateKeys) {
  TestResult test_result;
  TestProperty property_1_1("key_1", "1");
  TestProperty property_2_1("key_2", "2");
  TestProperty property_1_2("key_1", "12");
  TestProperty property_2_2("key_2", "22");
  TestResultAccessor::RecordProperty(&test_result, property_1_1);
  TestResultAccessor::RecordProperty(&test_result, property_2_1);
  TestResultAccessor::RecordProperty(&test_result, property_1_2);
  TestResultAccessor::RecordProperty(&test_result, property_2_2);

  ASSERT_EQ(2, test_result.test_property_count());
  const TestProperty& actual_property_1 = test_result.GetTestProperty(0);
  EXPECT_STREQ("key_1", actual_property_1.key());
  EXPECT_STREQ("12", actual_property_1.value());

  const TestProperty& actual_property_2 = test_result.GetTestProperty(1);
  EXPECT_STREQ("key_2", actual_property_2.key());
  EXPECT_STREQ("22", actual_property_2.value());
}

// Tests TestResult::GetTestProperty().
TEST(TestResultPropertyDeathTest, GetTestProperty) {
  TestResult test_result;
  TestProperty property_1("key_1", "1");
  TestProperty property_2("key_2", "2");
  TestProperty property_3("key_3", "3");
  TestResultAccessor::RecordProperty(&test_result, property_1);
  TestResultAccessor::RecordProperty(&test_result, property_2);
  TestResultAccessor::RecordProperty(&test_result, property_3);

  const TestProperty& fetched_property_1 = test_result.GetTestProperty(0);
  const TestProperty& fetched_property_2 = test_result.GetTestProperty(1);
  const TestProperty& fetched_property_3 = test_result.GetTestProperty(2);

  EXPECT_STREQ("key_1", fetched_property_1.key());
  EXPECT_STREQ("1", fetched_property_1.value());

  EXPECT_STREQ("key_2", fetched_property_2.key());
  EXPECT_STREQ("2", fetched_property_2.value());

  EXPECT_STREQ("key_3", fetched_property_3.key());
  EXPECT_STREQ("3", fetched_property_3.value());

  EXPECT_DEATH_IF_SUPPORTED(
      test_result.GetTestProperty(3),
      "Invalid Vector index 3: must be in range \\[0, 2\\]\\.");
  EXPECT_DEATH_IF_SUPPORTED(
      test_result.GetTestProperty(-1),
      "Invalid Vector index -1: must be in range \\[0, 2\\]\\.");
}

// When a property using a reserved key is supplied to this function, it tests
// that a non-fatal failure is added, a fatal failure is not added, and that the
// property is not recorded.
void ExpectNonFatalFailureRecordingPropertyWithReservedKey(const char* key) {
  TestResult test_result;
  TestProperty property(key, "1");
  EXPECT_NONFATAL_FAILURE(
      TestResultAccessor::RecordProperty(&test_result, property),
      "Reserved key");
  ASSERT_EQ(0, test_result.test_property_count()) << "Not recorded";
}

// Attempting to recording a property with the Reserved literal "name"
// should add a non-fatal failure and the property should not be recorded.
TEST(TestResultPropertyTest, AddFailureWhenUsingReservedKeyCalledName) {
  ExpectNonFatalFailureRecordingPropertyWithReservedKey("name");
}

// Attempting to recording a property with the Reserved literal "status"
// should add a non-fatal failure and the property should not be recorded.
TEST(TestResultPropertyTest, AddFailureWhenUsingReservedKeyCalledStatus) {
  ExpectNonFatalFailureRecordingPropertyWithReservedKey("status");
}

// Attempting to recording a property with the Reserved literal "time"
// should add a non-fatal failure and the property should not be recorded.
TEST(TestResultPropertyTest, AddFailureWhenUsingReservedKeyCalledTime) {
  ExpectNonFatalFailureRecordingPropertyWithReservedKey("time");
}

// Attempting to recording a property with the Reserved literal "classname"
// should add a non-fatal failure and the property should not be recorded.
TEST(TestResultPropertyTest, AddFailureWhenUsingReservedKeyCalledClassname) {
  ExpectNonFatalFailureRecordingPropertyWithReservedKey("classname");
}

// Tests that GTestFlagSaver works on Windows and Mac.

class GTestFlagSaverTest : public Test {
 protected:
  // Saves the Google Test flags such that we can restore them later, and
  // then sets them to their default values.  This will be called
  // before the first test in this test case is run.
  static void SetUpTestCase() {
    saver_ = new GTestFlagSaver;

    GTEST_FLAG(also_run_disabled_tests) = false;
    GTEST_FLAG(break_on_failure) = false;
    GTEST_FLAG(catch_exceptions) = false;
    GTEST_FLAG(death_test_use_fork) = false;
    GTEST_FLAG(color) = "auto";
    GTEST_FLAG(filter) = "";
    GTEST_FLAG(list_tests) = false;
    GTEST_FLAG(output) = "";
    GTEST_FLAG(print_time) = true;
    GTEST_FLAG(random_seed) = 0;
    GTEST_FLAG(repeat) = 1;
    GTEST_FLAG(shuffle) = false;
    GTEST_FLAG(throw_on_failure) = false;
  }

  // Restores the Google Test flags that the tests have modified.  This will
  // be called after the last test in this test case is run.
  static void TearDownTestCase() {
    delete saver_;
    saver_ = NULL;
  }

  // Verifies that the Google Test flags have their default values, and then
  // modifies each of them.
  void VerifyAndModifyFlags() {
    EXPECT_FALSE(GTEST_FLAG(also_run_disabled_tests));
    EXPECT_FALSE(GTEST_FLAG(break_on_failure));
    EXPECT_FALSE(GTEST_FLAG(catch_exceptions));
    EXPECT_STREQ("auto", GTEST_FLAG(color).c_str());
    EXPECT_FALSE(GTEST_FLAG(death_test_use_fork));
    EXPECT_STREQ("", GTEST_FLAG(filter).c_str());
    EXPECT_FALSE(GTEST_FLAG(list_tests));
    EXPECT_STREQ("", GTEST_FLAG(output).c_str());
    EXPECT_TRUE(GTEST_FLAG(print_time));
    EXPECT_EQ(0, GTEST_FLAG(random_seed));
    EXPECT_EQ(1, GTEST_FLAG(repeat));
    EXPECT_FALSE(GTEST_FLAG(shuffle));
    EXPECT_FALSE(GTEST_FLAG(throw_on_failure));

    GTEST_FLAG(also_run_disabled_tests) = true;
    GTEST_FLAG(break_on_failure) = true;
    GTEST_FLAG(catch_exceptions) = true;
    GTEST_FLAG(color) = "no";
    GTEST_FLAG(death_test_use_fork) = true;
    GTEST_FLAG(filter) = "abc";
    GTEST_FLAG(list_tests) = true;
    GTEST_FLAG(output) = "xml:foo.xml";
    GTEST_FLAG(print_time) = false;
    GTEST_FLAG(random_seed) = 1;
    GTEST_FLAG(repeat) = 100;
    GTEST_FLAG(shuffle) = true;
    GTEST_FLAG(throw_on_failure) = true;
  }
 private:
  // For saving Google Test flags during this test case.
  static GTestFlagSaver* saver_;
};

GTestFlagSaver* GTestFlagSaverTest::saver_ = NULL;

// Google Test doesn't guarantee the order of tests.  The following two
// tests are designed to work regardless of their order.

// Modifies the Google Test flags in the test body.
TEST_F(GTestFlagSaverTest, ModifyGTestFlags) {
  VerifyAndModifyFlags();
}

// Verifies that the Google Test flags in the body of the previous test were
// restored to their original values.
TEST_F(GTestFlagSaverTest, VerifyGTestFlags) {
  VerifyAndModifyFlags();
}

// Sets an environment variable with the given name to the given
// value.  If the value argument is "", unsets the environment
// variable.  The caller must ensure that both arguments are not NULL.
static void SetEnv(const char* name, const char* value) {
#if GTEST_OS_WINDOWS_MOBILE
  // Environment variables are not supported on Windows CE.
  return;
#elif defined(__BORLANDC__)
  // C++Builder's putenv only stores a pointer to its parameter; we have to
  // ensure that the string remains valid as long as it might be needed.
  // We use an std::map to do so.
  static std::map<String, String*> added_env;

  // Because putenv stores a pointer to the string buffer, we can't delete the
  // previous string (if present) until after it's replaced.
  String *prev_env = NULL;
  if (added_env.find(name) != added_env.end()) {
    prev_env = added_env[name];
  }
  added_env[name] = new String((Message() << name << "=" << value).GetString());
  putenv(added_env[name]->c_str());
  delete prev_env;
#elif GTEST_OS_WINDOWS  // If we are on Windows proper.
  _putenv((Message() << name << "=" << value).GetString().c_str());
#else
  if (*value == '\0') {
    unsetenv(name);
  } else {
    setenv(name, value, 1);
  }
#endif  // GTEST_OS_WINDOWS_MOBILE
}

#if !GTEST_OS_WINDOWS_MOBILE
// Environment variables are not supported on Windows CE.

using testing::internal::Int32FromGTestEnv;

// Tests Int32FromGTestEnv().

// Tests that Int32FromGTestEnv() returns the default value when the
// environment variable is not set.
TEST(Int32FromGTestEnvTest, ReturnsDefaultWhenVariableIsNotSet) {
  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "TEMP", "");
  EXPECT_EQ(10, Int32FromGTestEnv("temp", 10));
}

// Tests that Int32FromGTestEnv() returns the default value when the
// environment variable overflows as an Int32.
TEST(Int32FromGTestEnvTest, ReturnsDefaultWhenValueOverflows) {
  printf("(expecting 2 warnings)\n");

  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "TEMP", "12345678987654321");
  EXPECT_EQ(20, Int32FromGTestEnv("temp", 20));

  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "TEMP", "-12345678987654321");
  EXPECT_EQ(30, Int32FromGTestEnv("temp", 30));
}

// Tests that Int32FromGTestEnv() returns the default value when the
// environment variable does not represent a valid decimal integer.
TEST(Int32FromGTestEnvTest, ReturnsDefaultWhenValueIsInvalid) {
  printf("(expecting 2 warnings)\n");

  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "TEMP", "A1");
  EXPECT_EQ(40, Int32FromGTestEnv("temp", 40));

  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "TEMP", "12X");
  EXPECT_EQ(50, Int32FromGTestEnv("temp", 50));
}

// Tests that Int32FromGTestEnv() parses and returns the value of the
// environment variable when it represents a valid decimal integer in
// the range of an Int32.
TEST(Int32FromGTestEnvTest, ParsesAndReturnsValidValue) {
  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "TEMP", "123");
  EXPECT_EQ(123, Int32FromGTestEnv("temp", 0));

  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "TEMP", "-321");
  EXPECT_EQ(-321, Int32FromGTestEnv("temp", 0));
}
#endif  // !GTEST_OS_WINDOWS_MOBILE

// Tests ParseInt32Flag().

// Tests that ParseInt32Flag() returns false and doesn't change the
// output value when the flag has wrong format
TEST(ParseInt32FlagTest, ReturnsFalseForInvalidFlag) {
  Int32 value = 123;
  EXPECT_FALSE(ParseInt32Flag("--a=100", "b", &value));
  EXPECT_EQ(123, value);

  EXPECT_FALSE(ParseInt32Flag("a=100", "a", &value));
  EXPECT_EQ(123, value);
}

// Tests that ParseInt32Flag() returns false and doesn't change the
// output value when the flag overflows as an Int32.
TEST(ParseInt32FlagTest, ReturnsDefaultWhenValueOverflows) {
  printf("(expecting 2 warnings)\n");

  Int32 value = 123;
  EXPECT_FALSE(ParseInt32Flag("--abc=12345678987654321", "abc", &value));
  EXPECT_EQ(123, value);

  EXPECT_FALSE(ParseInt32Flag("--abc=-12345678987654321", "abc", &value));
  EXPECT_EQ(123, value);
}

// Tests that ParseInt32Flag() returns false and doesn't change the
// output value when the flag does not represent a valid decimal
// integer.
TEST(ParseInt32FlagTest, ReturnsDefaultWhenValueIsInvalid) {
  printf("(expecting 2 warnings)\n");

  Int32 value = 123;
  EXPECT_FALSE(ParseInt32Flag("--abc=A1", "abc", &value));
  EXPECT_EQ(123, value);

  EXPECT_FALSE(ParseInt32Flag("--abc=12X", "abc", &value));
  EXPECT_EQ(123, value);
}

// Tests that ParseInt32Flag() parses the value of the flag and
// returns true when the flag represents a valid decimal integer in
// the range of an Int32.
TEST(ParseInt32FlagTest, ParsesAndReturnsValidValue) {
  Int32 value = 123;
  EXPECT_TRUE(ParseInt32Flag("--" GTEST_FLAG_PREFIX_ "abc=456", "abc", &value));
  EXPECT_EQ(456, value);

  EXPECT_TRUE(ParseInt32Flag("--" GTEST_FLAG_PREFIX_ "abc=-789",
                             "abc", &value));
  EXPECT_EQ(-789, value);
}

// Tests that Int32FromEnvOrDie() parses the value of the var or
// returns the correct default.
// Environment variables are not supported on Windows CE.
#if !GTEST_OS_WINDOWS_MOBILE
TEST(Int32FromEnvOrDieTest, ParsesAndReturnsValidValue) {
  EXPECT_EQ(333, Int32FromEnvOrDie(GTEST_FLAG_PREFIX_UPPER_ "UnsetVar", 333));
  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "UnsetVar", "123");
  EXPECT_EQ(123, Int32FromEnvOrDie(GTEST_FLAG_PREFIX_UPPER_ "UnsetVar", 333));
  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "UnsetVar", "-123");
  EXPECT_EQ(-123, Int32FromEnvOrDie(GTEST_FLAG_PREFIX_UPPER_ "UnsetVar", 333));
}
#endif  // !GTEST_OS_WINDOWS_MOBILE

// Tests that Int32FromEnvOrDie() aborts with an error message
// if the variable is not an Int32.
TEST(Int32FromEnvOrDieDeathTest, AbortsOnFailure) {
  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "VAR", "xxx");
  EXPECT_DEATH_IF_SUPPORTED(
      Int32FromEnvOrDie(GTEST_FLAG_PREFIX_UPPER_ "VAR", 123),
      ".*");
}

// Tests that Int32FromEnvOrDie() aborts with an error message
// if the variable cannot be represnted by an Int32.
TEST(Int32FromEnvOrDieDeathTest, AbortsOnInt32Overflow) {
  SetEnv(GTEST_FLAG_PREFIX_UPPER_ "VAR", "1234567891234567891234");
  EXPECT_DEATH_IF_SUPPORTED(
      Int32FromEnvOrDie(GTEST_FLAG_PREFIX_UPPER_ "VAR", 123),
      ".*");
}

// Tests that ShouldRunTestOnShard() selects all tests
// where there is 1 shard.
TEST(ShouldRunTestOnShardTest, IsPartitionWhenThereIsOneShard) {
  EXPECT_TRUE(ShouldRunTestOnShard(1, 0, 0));
  EXPECT_TRUE(ShouldRunTestOnShard(1, 0, 1));
  EXPECT_TRUE(ShouldRunTestOnShard(1, 0, 2));
  EXPECT_TRUE(ShouldRunTestOnShard(1, 0, 3));
  EXPECT_TRUE(ShouldRunTestOnShard(1, 0, 4));
}

class ShouldShardTest : public testing::Test {
 protected:
  virtual void SetUp() {
    index_var_ = GTEST_FLAG_PREFIX_UPPER_ "INDEX";
    total_var_ = GTEST_FLAG_PREFIX_UPPER_ "TOTAL";
  }

  virtual void TearDown() {
    SetEnv(index_var_, "");
    SetEnv(total_var_, "");
  }

  const char* index_var_;
  const char* total_var_;
};

// Tests that sharding is disabled if neither of the environment variables
// are set.
TEST_F(ShouldShardTest, ReturnsFalseWhenNeitherEnvVarIsSet) {
  SetEnv(index_var_, "");
  SetEnv(total_var_, "");

  EXPECT_FALSE(ShouldShard(total_var_, index_var_, false));
  EXPECT_FALSE(ShouldShard(total_var_, index_var_, true));
}

// Tests that sharding is not enabled if total_shards  == 1.
TEST_F(ShouldShardTest, ReturnsFalseWhenTotalShardIsOne) {
  SetEnv(index_var_, "0");
  SetEnv(total_var_, "1");
  EXPECT_FALSE(ShouldShard(total_var_, index_var_, false));
  EXPECT_FALSE(ShouldShard(total_var_, index_var_, true));
}

// Tests that sharding is enabled if total_shards > 1 and
// we are not in a death test subprocess.
// Environment variables are not supported on Windows CE.
#if !GTEST_OS_WINDOWS_MOBILE
TEST_F(ShouldShardTest, WorksWhenShardEnvVarsAreValid) {
  SetEnv(index_var_, "4");
  SetEnv(total_var_, "22");
  EXPECT_TRUE(ShouldShard(total_var_, index_var_, false));
  EXPECT_FALSE(ShouldShard(total_var_, index_var_, true));

  SetEnv(index_var_, "8");
  SetEnv(total_var_, "9");
  EXPECT_TRUE(ShouldShard(total_var_, index_var_, false));
  EXPECT_FALSE(ShouldShard(total_var_, index_var_, true));

  SetEnv(index_var_, "0");
  SetEnv(total_var_, "9");
  EXPECT_TRUE(ShouldShard(total_var_, index_var_, false));
  EXPECT_FALSE(ShouldShard(total_var_, index_var_, true));
}
#endif  // !GTEST_OS_WINDOWS_MOBILE

// Tests that we exit in error if the sharding values are not valid.

typedef ShouldShardTest ShouldShardDeathTest;

TEST_F(ShouldShardDeathTest, AbortsWhenShardingEnvVarsAreInvalid) {
  SetEnv(index_var_, "4");
  SetEnv(total_var_, "4");
  EXPECT_DEATH_IF_SUPPORTED(ShouldShard(total_var_, index_var_, false), ".*");

  SetEnv(index_var_, "4");
  SetEnv(total_var_, "-2");
  EXPECT_DEATH_IF_SUPPORTED(ShouldShard(total_var_, index_var_, false), ".*");

  SetEnv(index_var_, "5");
  SetEnv(total_var_, "");
  EXPECT_DEATH_IF_SUPPORTED(ShouldShard(total_var_, index_var_, false), ".*");

  SetEnv(index_var_, "");
  SetEnv(total_var_, "5");
  EXPECT_DEATH_IF_SUPPORTED(ShouldShard(total_var_, index_var_, false), ".*");
}

// Tests that ShouldRunTestOnShard is a partition when 5
// shards are used.
TEST(ShouldRunTestOnShardTest, IsPartitionWhenThereAreFiveShards) {
  // Choose an arbitrary number of tests and shards.
  const int num_tests = 17;
  const int num_shards = 5;

  // Check partitioning: each test should be on exactly 1 shard.
  for (int test_id = 0; test_id < num_tests; test_id++) {
    int prev_selected_shard_index = -1;
    for (int shard_index = 0; shard_index < num_shards; shard_index++) {
      if (ShouldRunTestOnShard(num_shards, shard_index, test_id)) {
        if (prev_selected_shard_index < 0) {
          prev_selected_shard_index = shard_index;
        } else {
          ADD_FAILURE() << "Shard " << prev_selected_shard_index << " and "
            << shard_index << " are both selected to run test " << test_id;
        }
      }
    }
  }

  // Check balance: This is not required by the sharding protocol, but is a
  // desirable property for performance.
  for (int shard_index = 0; shard_index < num_shards; shard_index++) {
    int num_tests_on_shard = 0;
    for (int test_id = 0; test_id < num_tests; test_id++) {
      num_tests_on_shard +=
        ShouldRunTestOnShard(num_shards, shard_index, test_id);
    }
    EXPECT_GE(num_tests_on_shard, num_tests / num_shards);
  }
}

// For the same reason we are not explicitly testing everything in the
// Test class, there are no separate tests for the following classes
// (except for some trivial cases):
//
//   TestCase, UnitTest, UnitTestResultPrinter.
//
// Similarly, there are no separate tests for the following macros:
//
//   TEST, TEST_F, RUN_ALL_TESTS

TEST(UnitTestTest, CanGetOriginalWorkingDir) {
  ASSERT_TRUE(UnitTest::GetInstance()->original_working_dir() != NULL);
  EXPECT_STRNE(UnitTest::GetInstance()->original_working_dir(), "");
}

// This group of tests is for predicate assertions (ASSERT_PRED*, etc)
// of various arities.  They do not attempt to be exhaustive.  Rather,
// view them as smoke tests that can be easily reviewed and verified.
// A more complete set of tests for predicate assertions can be found
// in gtest_pred_impl_unittest.cc.

// First, some predicates and predicate-formatters needed by the tests.

// Returns true iff the argument is an even number.
bool IsEven(int n) {
  return (n % 2) == 0;
}

// A functor that returns true iff the argument is an even number.
struct IsEvenFunctor {
  bool operator()(int n) { return IsEven(n); }
};

// A predicate-formatter function that asserts the argument is an even
// number.
AssertionResult AssertIsEven(const char* expr, int n) {
  if (IsEven(n)) {
    return AssertionSuccess();
  }

  Message msg;
  msg << expr << " evaluates to " << n << ", which is not even.";
  return AssertionFailure(msg);
}

// A predicate-formatter functor that asserts the argument is an even
// number.
struct AssertIsEvenFunctor {
  AssertionResult operator()(const char* expr, int n) {
    return AssertIsEven(expr, n);
  }
};

// Returns true iff the sum of the arguments is an even number.
bool SumIsEven2(int n1, int n2) {
  return IsEven(n1 + n2);
}

// A functor that returns true iff the sum of the arguments is an even
// number.
struct SumIsEven3Functor {
  bool operator()(int n1, int n2, int n3) {
    return IsEven(n1 + n2 + n3);
  }
};

// A predicate-formatter function that asserts the sum of the
// arguments is an even number.
AssertionResult AssertSumIsEven4(
    const char* e1, const char* e2, const char* e3, const char* e4,
    int n1, int n2, int n3, int n4) {
  const int sum = n1 + n2 + n3 + n4;
  if (IsEven(sum)) {
    return AssertionSuccess();
  }

  Message msg;
  msg << e1 << " + " << e2 << " + " << e3 << " + " << e4
      << " (" << n1 << " + " << n2 << " + " << n3 << " + " << n4
      << ") evaluates to " << sum << ", which is not even.";
  return AssertionFailure(msg);
}

// A predicate-formatter functor that asserts the sum of the arguments
// is an even number.
struct AssertSumIsEven5Functor {
  AssertionResult operator()(
      const char* e1, const char* e2, const char* e3, const char* e4,
      const char* e5, int n1, int n2, int n3, int n4, int n5) {
    const int sum = n1 + n2 + n3 + n4 + n5;
    if (IsEven(sum)) {
      return AssertionSuccess();
    }

    Message msg;
    msg << e1 << " + " << e2 << " + " << e3 << " + " << e4 << " + " << e5
        << " ("
        << n1 << " + " << n2 << " + " << n3 << " + " << n4 << " + " << n5
        << ") evaluates to " << sum << ", which is not even.";
    return AssertionFailure(msg);
  }
};


// Tests unary predicate assertions.

// Tests unary predicate assertions that don't use a custom formatter.
TEST(Pred1Test, WithoutFormat) {
  // Success cases.
  EXPECT_PRED1(IsEvenFunctor(), 2) << "This failure is UNEXPECTED!";
  ASSERT_PRED1(IsEven, 4);

  // Failure cases.
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED1(IsEven, 5) << "This failure is expected.";
  }, "This failure is expected.");
  EXPECT_FATAL_FAILURE(ASSERT_PRED1(IsEvenFunctor(), 5),
                       "evaluates to false");
}

// Tests unary predicate assertions that use a custom formatter.
TEST(Pred1Test, WithFormat) {
  // Success cases.
  EXPECT_PRED_FORMAT1(AssertIsEven, 2);
  ASSERT_PRED_FORMAT1(AssertIsEvenFunctor(), 4)
    << "This failure is UNEXPECTED!";

  // Failure cases.
  const int n = 5;
  EXPECT_NONFATAL_FAILURE(EXPECT_PRED_FORMAT1(AssertIsEvenFunctor(), n),
                          "n evaluates to 5, which is not even.");
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_PRED_FORMAT1(AssertIsEven, 5) << "This failure is expected.";
  }, "This failure is expected.");
}

// Tests that unary predicate assertions evaluates their arguments
// exactly once.
TEST(Pred1Test, SingleEvaluationOnFailure) {
  // A success case.
  static int n = 0;
  EXPECT_PRED1(IsEven, n++);
  EXPECT_EQ(1, n) << "The argument is not evaluated exactly once.";

  // A failure case.
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_PRED_FORMAT1(AssertIsEvenFunctor(), n++)
        << "This failure is expected.";
  }, "This failure is expected.");
  EXPECT_EQ(2, n) << "The argument is not evaluated exactly once.";
}


// Tests predicate assertions whose arity is >= 2.

// Tests predicate assertions that don't use a custom formatter.
TEST(PredTest, WithoutFormat) {
  // Success cases.
  ASSERT_PRED2(SumIsEven2, 2, 4) << "This failure is UNEXPECTED!";
  EXPECT_PRED3(SumIsEven3Functor(), 4, 6, 8);

  // Failure cases.
  const int n1 = 1;
  const int n2 = 2;
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED2(SumIsEven2, n1, n2) << "This failure is expected.";
  }, "This failure is expected.");
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_PRED3(SumIsEven3Functor(), 1, 2, 4);
  }, "evaluates to false");
}

// Tests predicate assertions that use a custom formatter.
TEST(PredTest, WithFormat) {
  // Success cases.
  ASSERT_PRED_FORMAT4(AssertSumIsEven4, 4, 6, 8, 10) <<
    "This failure is UNEXPECTED!";
  EXPECT_PRED_FORMAT5(AssertSumIsEven5Functor(), 2, 4, 6, 8, 10);

  // Failure cases.
  const int n1 = 1;
  const int n2 = 2;
  const int n3 = 4;
  const int n4 = 6;
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT4(AssertSumIsEven4, n1, n2, n3, n4);
  }, "evaluates to 13, which is not even.");
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_PRED_FORMAT5(AssertSumIsEven5Functor(), 1, 2, 4, 6, 8)
        << "This failure is expected.";
  }, "This failure is expected.");
}

// Tests that predicate assertions evaluates their arguments
// exactly once.
TEST(PredTest, SingleEvaluationOnFailure) {
  // A success case.
  int n1 = 0;
  int n2 = 0;
  EXPECT_PRED2(SumIsEven2, n1++, n2++);
  EXPECT_EQ(1, n1) << "Argument 1 is not evaluated exactly once.";
  EXPECT_EQ(1, n2) << "Argument 2 is not evaluated exactly once.";

  // Another success case.
  n1 = n2 = 0;
  int n3 = 0;
  int n4 = 0;
  int n5 = 0;
  ASSERT_PRED_FORMAT5(AssertSumIsEven5Functor(),
                      n1++, n2++, n3++, n4++, n5++)
                        << "This failure is UNEXPECTED!";
  EXPECT_EQ(1, n1) << "Argument 1 is not evaluated exactly once.";
  EXPECT_EQ(1, n2) << "Argument 2 is not evaluated exactly once.";
  EXPECT_EQ(1, n3) << "Argument 3 is not evaluated exactly once.";
  EXPECT_EQ(1, n4) << "Argument 4 is not evaluated exactly once.";
  EXPECT_EQ(1, n5) << "Argument 5 is not evaluated exactly once.";

  // A failure case.
  n1 = n2 = n3 = 0;
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED3(SumIsEven3Functor(), ++n1, n2++, n3++)
        << "This failure is expected.";
  }, "This failure is expected.");
  EXPECT_EQ(1, n1) << "Argument 1 is not evaluated exactly once.";
  EXPECT_EQ(1, n2) << "Argument 2 is not evaluated exactly once.";
  EXPECT_EQ(1, n3) << "Argument 3 is not evaluated exactly once.";

  // Another failure case.
  n1 = n2 = n3 = n4 = 0;
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT4(AssertSumIsEven4, ++n1, n2++, n3++, n4++);
  }, "evaluates to 1, which is not even.");
  EXPECT_EQ(1, n1) << "Argument 1 is not evaluated exactly once.";
  EXPECT_EQ(1, n2) << "Argument 2 is not evaluated exactly once.";
  EXPECT_EQ(1, n3) << "Argument 3 is not evaluated exactly once.";
  EXPECT_EQ(1, n4) << "Argument 4 is not evaluated exactly once.";
}


// Some helper functions for testing using overloaded/template
// functions with ASSERT_PREDn and EXPECT_PREDn.

bool IsPositive(int n) {
  return n > 0;
}

bool IsPositive(double x) {
  return x > 0;
}

template <typename T>
bool IsNegative(T x) {
  return x < 0;
}

template <typename T1, typename T2>
bool GreaterThan(T1 x1, T2 x2) {
  return x1 > x2;
}

// Tests that overloaded functions can be used in *_PRED* as long as
// their types are explicitly specified.
TEST(PredicateAssertionTest, AcceptsOverloadedFunction) {
  // C++Builder requires C-style casts rather than static_cast.
  EXPECT_PRED1((bool (*)(int))(IsPositive), 5);  // NOLINT
  ASSERT_PRED1((bool (*)(double))(IsPositive), 6.0);  // NOLINT
}

// Tests that template functions can be used in *_PRED* as long as
// their types are explicitly specified.
TEST(PredicateAssertionTest, AcceptsTemplateFunction) {
  EXPECT_PRED1(IsNegative<int>, -5);
  // Makes sure that we can handle templates with more than one
  // parameter.
  ASSERT_PRED2((GreaterThan<int, int>), 5, 0);
}


// Some helper functions for testing using overloaded/template
// functions with ASSERT_PRED_FORMATn and EXPECT_PRED_FORMATn.

AssertionResult IsPositiveFormat(const char* /* expr */, int n) {
  return n > 0 ? AssertionSuccess() :
      AssertionFailure(Message() << "Failure");
}

AssertionResult IsPositiveFormat(const char* /* expr */, double x) {
  return x > 0 ? AssertionSuccess() :
      AssertionFailure(Message() << "Failure");
}

template <typename T>
AssertionResult IsNegativeFormat(const char* /* expr */, T x) {
  return x < 0 ? AssertionSuccess() :
      AssertionFailure(Message() << "Failure");
}

template <typename T1, typename T2>
AssertionResult EqualsFormat(const char* /* expr1 */, const char* /* expr2 */,
                             const T1& x1, const T2& x2) {
  return x1 == x2 ? AssertionSuccess() :
      AssertionFailure(Message() << "Failure");
}

// Tests that overloaded functions can be used in *_PRED_FORMAT*
// without explicitly specifying their types.
TEST(PredicateFormatAssertionTest, AcceptsOverloadedFunction) {
  EXPECT_PRED_FORMAT1(IsPositiveFormat, 5);
  ASSERT_PRED_FORMAT1(IsPositiveFormat, 6.0);
}

// Tests that template functions can be used in *_PRED_FORMAT* without
// explicitly specifying their types.
TEST(PredicateFormatAssertionTest, AcceptsTemplateFunction) {
  EXPECT_PRED_FORMAT1(IsNegativeFormat, -5);
  ASSERT_PRED_FORMAT2(EqualsFormat, 3, 3);
}


// Tests string assertions.

// Tests ASSERT_STREQ with non-NULL arguments.
TEST(StringAssertionTest, ASSERT_STREQ) {
  const char * const p1 = "good";
  ASSERT_STREQ(p1, p1);

  // Let p2 have the same content as p1, but be at a different address.
  const char p2[] = "good";
  ASSERT_STREQ(p1, p2);

  EXPECT_FATAL_FAILURE(ASSERT_STREQ("bad", "good"),
                       "Expected: \"bad\"");
}

// Tests ASSERT_STREQ with NULL arguments.
TEST(StringAssertionTest, ASSERT_STREQ_Null) {
  ASSERT_STREQ(static_cast<const char *>(NULL), NULL);
  EXPECT_FATAL_FAILURE(ASSERT_STREQ(NULL, "non-null"),
                       "non-null");
}

// Tests ASSERT_STREQ with NULL arguments.
TEST(StringAssertionTest, ASSERT_STREQ_Null2) {
  EXPECT_FATAL_FAILURE(ASSERT_STREQ("non-null", NULL),
                       "non-null");
}

// Tests ASSERT_STRNE.
TEST(StringAssertionTest, ASSERT_STRNE) {
  ASSERT_STRNE("hi", "Hi");
  ASSERT_STRNE("Hi", NULL);
  ASSERT_STRNE(NULL, "Hi");
  ASSERT_STRNE("", NULL);
  ASSERT_STRNE(NULL, "");
  ASSERT_STRNE("", "Hi");
  ASSERT_STRNE("Hi", "");
  EXPECT_FATAL_FAILURE(ASSERT_STRNE("Hi", "Hi"),
                       "\"Hi\" vs \"Hi\"");
}

// Tests ASSERT_STRCASEEQ.
TEST(StringAssertionTest, ASSERT_STRCASEEQ) {
  ASSERT_STRCASEEQ("hi", "Hi");
  ASSERT_STRCASEEQ(static_cast<const char *>(NULL), NULL);

  ASSERT_STRCASEEQ("", "");
  EXPECT_FATAL_FAILURE(ASSERT_STRCASEEQ("Hi", "hi2"),
                       "(ignoring case)");
}

// Tests ASSERT_STRCASENE.
TEST(StringAssertionTest, ASSERT_STRCASENE) {
  ASSERT_STRCASENE("hi1", "Hi2");
  ASSERT_STRCASENE("Hi", NULL);
  ASSERT_STRCASENE(NULL, "Hi");
  ASSERT_STRCASENE("", NULL);
  ASSERT_STRCASENE(NULL, "");
  ASSERT_STRCASENE("", "Hi");
  ASSERT_STRCASENE("Hi", "");
  EXPECT_FATAL_FAILURE(ASSERT_STRCASENE("Hi", "hi"),
                       "(ignoring case)");
}

// Tests *_STREQ on wide strings.
TEST(StringAssertionTest, STREQ_Wide) {
  // NULL strings.
  ASSERT_STREQ(static_cast<const wchar_t *>(NULL), NULL);

  // Empty strings.
  ASSERT_STREQ(L"", L"");

  // Non-null vs NULL.
  EXPECT_NONFATAL_FAILURE(EXPECT_STREQ(L"non-null", NULL),
                          "non-null");

  // Equal strings.
  EXPECT_STREQ(L"Hi", L"Hi");

  // Unequal strings.
  EXPECT_NONFATAL_FAILURE(EXPECT_STREQ(L"abc", L"Abc"),
                          "Abc");

  // Strings containing wide characters.
  EXPECT_NONFATAL_FAILURE(EXPECT_STREQ(L"abc\x8119", L"abc\x8120"),
                          "abc");
}

// Tests *_STRNE on wide strings.
TEST(StringAssertionTest, STRNE_Wide) {
  // NULL strings.
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_STRNE(static_cast<const wchar_t *>(NULL), NULL);
  }, "");

  // Empty strings.
  EXPECT_NONFATAL_FAILURE(EXPECT_STRNE(L"", L""),
                          "L\"\"");

  // Non-null vs NULL.
  ASSERT_STRNE(L"non-null", NULL);

  // Equal strings.
  EXPECT_NONFATAL_FAILURE(EXPECT_STRNE(L"Hi", L"Hi"),
                          "L\"Hi\"");

  // Unequal strings.
  EXPECT_STRNE(L"abc", L"Abc");

  // Strings containing wide characters.
  EXPECT_NONFATAL_FAILURE(EXPECT_STRNE(L"abc\x8119", L"abc\x8119"),
                          "abc");
}

// Tests for ::testing::IsSubstring().

// Tests that IsSubstring() returns the correct result when the input
// argument type is const char*.
TEST(IsSubstringTest, ReturnsCorrectResultForCString) {
  EXPECT_FALSE(IsSubstring("", "", NULL, "a"));
  EXPECT_FALSE(IsSubstring("", "", "b", NULL));
  EXPECT_FALSE(IsSubstring("", "", "needle", "haystack"));

  EXPECT_TRUE(IsSubstring("", "", static_cast<const char*>(NULL), NULL));
  EXPECT_TRUE(IsSubstring("", "", "needle", "two needles"));
}

// Tests that IsSubstring() returns the correct result when the input
// argument type is const wchar_t*.
TEST(IsSubstringTest, ReturnsCorrectResultForWideCString) {
  EXPECT_FALSE(IsSubstring("", "", kNull, L"a"));
  EXPECT_FALSE(IsSubstring("", "", L"b", kNull));
  EXPECT_FALSE(IsSubstring("", "", L"needle", L"haystack"));

  EXPECT_TRUE(IsSubstring("", "", static_cast<const wchar_t*>(NULL), NULL));
  EXPECT_TRUE(IsSubstring("", "", L"needle", L"two needles"));
}

// Tests that IsSubstring() generates the correct message when the input
// argument type is const char*.
TEST(IsSubstringTest, GeneratesCorrectMessageForCString) {
  EXPECT_STREQ("Value of: needle_expr\n"
               "  Actual: \"needle\"\n"
               "Expected: a substring of haystack_expr\n"
               "Which is: \"haystack\"",
               IsSubstring("needle_expr", "haystack_expr",
                           "needle", "haystack").failure_message());
}

#if GTEST_HAS_STD_STRING

// Tests that IsSubstring returns the correct result when the input
// argument type is ::std::string.
TEST(IsSubstringTest, ReturnsCorrectResultsForStdString) {
  EXPECT_TRUE(IsSubstring("", "", std::string("hello"), "ahellob"));
  EXPECT_FALSE(IsSubstring("", "", "hello", std::string("world")));
}

#endif  // GTEST_HAS_STD_STRING

#if GTEST_HAS_STD_WSTRING
// Tests that IsSubstring returns the correct result when the input
// argument type is ::std::wstring.
TEST(IsSubstringTest, ReturnsCorrectResultForStdWstring) {
  EXPECT_TRUE(IsSubstring("", "", ::std::wstring(L"needle"), L"two needles"));
  EXPECT_FALSE(IsSubstring("", "", L"needle", ::std::wstring(L"haystack")));
}

// Tests that IsSubstring() generates the correct message when the input
// argument type is ::std::wstring.
TEST(IsSubstringTest, GeneratesCorrectMessageForWstring) {
  EXPECT_STREQ("Value of: needle_expr\n"
               "  Actual: L\"needle\"\n"
               "Expected: a substring of haystack_expr\n"
               "Which is: L\"haystack\"",
               IsSubstring(
                   "needle_expr", "haystack_expr",
                   ::std::wstring(L"needle"), L"haystack").failure_message());
}

#endif  // GTEST_HAS_STD_WSTRING

// Tests for ::testing::IsNotSubstring().

// Tests that IsNotSubstring() returns the correct result when the input
// argument type is const char*.
TEST(IsNotSubstringTest, ReturnsCorrectResultForCString) {
  EXPECT_TRUE(IsNotSubstring("", "", "needle", "haystack"));
  EXPECT_FALSE(IsNotSubstring("", "", "needle", "two needles"));
}

// Tests that IsNotSubstring() returns the correct result when the input
// argument type is const wchar_t*.
TEST(IsNotSubstringTest, ReturnsCorrectResultForWideCString) {
  EXPECT_TRUE(IsNotSubstring("", "", L"needle", L"haystack"));
  EXPECT_FALSE(IsNotSubstring("", "", L"needle", L"two needles"));
}

// Tests that IsNotSubstring() generates the correct message when the input
// argument type is const wchar_t*.
TEST(IsNotSubstringTest, GeneratesCorrectMessageForWideCString) {
  EXPECT_STREQ("Value of: needle_expr\n"
               "  Actual: L\"needle\"\n"
               "Expected: not a substring of haystack_expr\n"
               "Which is: L\"two needles\"",
               IsNotSubstring(
                   "needle_expr", "haystack_expr",
                   L"needle", L"two needles").failure_message());
}

#if GTEST_HAS_STD_STRING

// Tests that IsNotSubstring returns the correct result when the input
// argument type is ::std::string.
TEST(IsNotSubstringTest, ReturnsCorrectResultsForStdString) {
  EXPECT_FALSE(IsNotSubstring("", "", std::string("hello"), "ahellob"));
  EXPECT_TRUE(IsNotSubstring("", "", "hello", std::string("world")));
}

// Tests that IsNotSubstring() generates the correct message when the input
// argument type is ::std::string.
TEST(IsNotSubstringTest, GeneratesCorrectMessageForStdString) {
  EXPECT_STREQ("Value of: needle_expr\n"
               "  Actual: \"needle\"\n"
               "Expected: not a substring of haystack_expr\n"
               "Which is: \"two needles\"",
               IsNotSubstring(
                   "needle_expr", "haystack_expr",
                   ::std::string("needle"), "two needles").failure_message());
}

#endif  // GTEST_HAS_STD_STRING

#if GTEST_HAS_STD_WSTRING

// Tests that IsNotSubstring returns the correct result when the input
// argument type is ::std::wstring.
TEST(IsNotSubstringTest, ReturnsCorrectResultForStdWstring) {
  EXPECT_FALSE(
      IsNotSubstring("", "", ::std::wstring(L"needle"), L"two needles"));
  EXPECT_TRUE(IsNotSubstring("", "", L"needle", ::std::wstring(L"haystack")));
}

#endif  // GTEST_HAS_STD_WSTRING

// Tests floating-point assertions.

template <typename RawType>
class FloatingPointTest : public Test {
 protected:

  // Pre-calculated numbers to be used by the tests.
  struct TestValues {
    RawType close_to_positive_zero;
    RawType close_to_negative_zero;
    RawType further_from_negative_zero;

    RawType close_to_one;
    RawType further_from_one;

    RawType infinity;
    RawType close_to_infinity;
    RawType further_from_infinity;

    RawType nan1;
    RawType nan2;
  };

  typedef typename testing::internal::FloatingPoint<RawType> Floating;
  typedef typename Floating::Bits Bits;

  virtual void SetUp() {
    const size_t max_ulps = Floating::kMaxUlps;

    // The bits that represent 0.0.
    const Bits zero_bits = Floating(0).bits();

    // Makes some numbers close to 0.0.
    values_.close_to_positive_zero = Floating::ReinterpretBits(
        zero_bits + max_ulps/2);
    values_.close_to_negative_zero = -Floating::ReinterpretBits(
        zero_bits + max_ulps - max_ulps/2);
    values_.further_from_negative_zero = -Floating::ReinterpretBits(
        zero_bits + max_ulps + 1 - max_ulps/2);

    // The bits that represent 1.0.
    const Bits one_bits = Floating(1).bits();

    // Makes some numbers close to 1.0.
    values_.close_to_one = Floating::ReinterpretBits(one_bits + max_ulps);
    values_.further_from_one = Floating::ReinterpretBits(
        one_bits + max_ulps + 1);

    // +infinity.
    values_.infinity = Floating::Infinity();

    // The bits that represent +infinity.
    const Bits infinity_bits = Floating(values_.infinity).bits();

    // Makes some numbers close to infinity.
    values_.close_to_infinity = Floating::ReinterpretBits(
        infinity_bits - max_ulps);
    values_.further_from_infinity = Floating::ReinterpretBits(
        infinity_bits - max_ulps - 1);

    // Makes some NAN's.  Sets the most significant bit of the fraction so that
    // our NaN's are quiet; trying to process a signaling NaN would raise an
    // exception if our environment enables floating point exceptions.
    values_.nan1 = Floating::ReinterpretBits(Floating::kExponentBitMask
        | (static_cast<Bits>(1) << (Floating::kFractionBitCount - 1)) | 1);
    values_.nan2 = Floating::ReinterpretBits(Floating::kExponentBitMask
        | (static_cast<Bits>(1) << (Floating::kFractionBitCount - 1)) | 200);
  }

  void TestSize() {
    EXPECT_EQ(sizeof(RawType), sizeof(Bits));
  }

  static TestValues values_;
};

template <typename RawType>
typename FloatingPointTest<RawType>::TestValues
    FloatingPointTest<RawType>::values_;

// Instantiates FloatingPointTest for testing *_FLOAT_EQ.
typedef FloatingPointTest<float> FloatTest;

// Tests that the size of Float::Bits matches the size of float.
TEST_F(FloatTest, Size) {
  TestSize();
}

// Tests comparing with +0 and -0.
TEST_F(FloatTest, Zeros) {
  EXPECT_FLOAT_EQ(0.0, -0.0);
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(-0.0, 1.0),
                          "1.0");
  EXPECT_FATAL_FAILURE(ASSERT_FLOAT_EQ(0.0, 1.5),
                       "1.5");
}

// Tests comparing numbers close to 0.
//
// This ensures that *_FLOAT_EQ handles the sign correctly and no
// overflow occurs when comparing numbers whose absolute value is very
// small.
TEST_F(FloatTest, AlmostZeros) {
  // In C++Builder, names within local classes (such as used by
  // EXPECT_FATAL_FAILURE) cannot be resolved against static members of the
  // scoping class.  Use a static local alias as a workaround.
  static const FloatTest::TestValues& v(this->values_);

  EXPECT_FLOAT_EQ(0.0, v.close_to_positive_zero);
  EXPECT_FLOAT_EQ(-0.0, v.close_to_negative_zero);
  EXPECT_FLOAT_EQ(v.close_to_positive_zero, v.close_to_negative_zero);

  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_FLOAT_EQ(v.close_to_positive_zero,
                    v.further_from_negative_zero);
  }, "v.further_from_negative_zero");
}

// Tests comparing numbers close to each other.
TEST_F(FloatTest, SmallDiff) {
  EXPECT_FLOAT_EQ(1.0, values_.close_to_one);
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(1.0, values_.further_from_one),
                          "values_.further_from_one");
}

// Tests comparing numbers far apart.
TEST_F(FloatTest, LargeDiff) {
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(2.5, 3.0),
                          "3.0");
}

// Tests comparing with infinity.
//
// This ensures that no overflow occurs when comparing numbers whose
// absolute value is very large.
TEST_F(FloatTest, Infinity) {
  EXPECT_FLOAT_EQ(values_.infinity, values_.close_to_infinity);
  EXPECT_FLOAT_EQ(-values_.infinity, -values_.close_to_infinity);
#if !GTEST_OS_SYMBIAN
  // Nokia's STLport crashes if we try to output infinity or NaN.
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(values_.infinity, -values_.infinity),
                          "-values_.infinity");

  // This is interesting as the representations of infinity and nan1
  // are only 1 DLP apart.
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(values_.infinity, values_.nan1),
                          "values_.nan1");
#endif  // !GTEST_OS_SYMBIAN
}

// Tests that comparing with NAN always returns false.
TEST_F(FloatTest, NaN) {
#if !GTEST_OS_SYMBIAN
// Nokia's STLport crashes if we try to output infinity or NaN.

  // In C++Builder, names within local classes (such as used by
  // EXPECT_FATAL_FAILURE) cannot be resolved against static members of the
  // scoping class.  Use a static local alias as a workaround.
  static const FloatTest::TestValues& v(this->values_);

  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(v.nan1, v.nan1),
                          "v.nan1");
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(v.nan1, v.nan2),
                          "v.nan2");
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(1.0, v.nan1),
                          "v.nan1");

  EXPECT_FATAL_FAILURE(ASSERT_FLOAT_EQ(v.nan1, v.infinity),
                       "v.infinity");
#endif  // !GTEST_OS_SYMBIAN
}

// Tests that *_FLOAT_EQ are reflexive.
TEST_F(FloatTest, Reflexive) {
  EXPECT_FLOAT_EQ(0.0, 0.0);
  EXPECT_FLOAT_EQ(1.0, 1.0);
  ASSERT_FLOAT_EQ(values_.infinity, values_.infinity);
}

// Tests that *_FLOAT_EQ are commutative.
TEST_F(FloatTest, Commutative) {
  // We already tested EXPECT_FLOAT_EQ(1.0, values_.close_to_one).
  EXPECT_FLOAT_EQ(values_.close_to_one, 1.0);

  // We already tested EXPECT_FLOAT_EQ(1.0, values_.further_from_one).
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(values_.further_from_one, 1.0),
                          "1.0");
}

// Tests EXPECT_NEAR.
TEST_F(FloatTest, EXPECT_NEAR) {
  EXPECT_NEAR(-1.0f, -1.1f, 0.2f);
  EXPECT_NEAR(2.0f, 3.0f, 1.0f);
  EXPECT_NONFATAL_FAILURE(EXPECT_NEAR(1.0f,1.2f, 0.1f),  // NOLINT
                          "The difference between 1.0f and 1.2f is 0.2, "
                          "which exceeds 0.1f");
  // To work around a bug in gcc 2.95.0, there is intentionally no
  // space after the first comma in the previous line.
}

// Tests ASSERT_NEAR.
TEST_F(FloatTest, ASSERT_NEAR) {
  ASSERT_NEAR(-1.0f, -1.1f, 0.2f);
  ASSERT_NEAR(2.0f, 3.0f, 1.0f);
  EXPECT_FATAL_FAILURE(ASSERT_NEAR(1.0f,1.2f, 0.1f),  // NOLINT
                       "The difference between 1.0f and 1.2f is 0.2, "
                       "which exceeds 0.1f");
  // To work around a bug in gcc 2.95.0, there is intentionally no
  // space after the first comma in the previous line.
}

// Tests the cases where FloatLE() should succeed.
TEST_F(FloatTest, FloatLESucceeds) {
  EXPECT_PRED_FORMAT2(FloatLE, 1.0f, 2.0f);  // When val1 < val2,
  ASSERT_PRED_FORMAT2(FloatLE, 1.0f, 1.0f);  // val1 == val2,

  // or when val1 is greater than, but almost equals to, val2.
  EXPECT_PRED_FORMAT2(FloatLE, values_.close_to_positive_zero, 0.0f);
}

// Tests the cases where FloatLE() should fail.
TEST_F(FloatTest, FloatLEFails) {
  // When val1 is greater than val2 by a large margin,
  EXPECT_NONFATAL_FAILURE(EXPECT_PRED_FORMAT2(FloatLE, 2.0f, 1.0f),
                          "(2.0f) <= (1.0f)");

  // or by a small yet non-negligible margin,
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT2(FloatLE, values_.further_from_one, 1.0f);
  }, "(values_.further_from_one) <= (1.0f)");

#if !GTEST_OS_SYMBIAN && !defined(__BORLANDC__)
  // Nokia's STLport crashes if we try to output infinity or NaN.
  // C++Builder gives bad results for ordered comparisons involving NaNs
  // due to compiler bugs.
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT2(FloatLE, values_.nan1, values_.infinity);
  }, "(values_.nan1) <= (values_.infinity)");
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT2(FloatLE, -values_.infinity, values_.nan1);
  }, "(-values_.infinity) <= (values_.nan1)");
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_PRED_FORMAT2(FloatLE, values_.nan1, values_.nan1);
  }, "(values_.nan1) <= (values_.nan1)");
#endif  // !GTEST_OS_SYMBIAN && !defined(__BORLANDC__)
}

// Instantiates FloatingPointTest for testing *_DOUBLE_EQ.
typedef FloatingPointTest<double> DoubleTest;

// Tests that the size of Double::Bits matches the size of double.
TEST_F(DoubleTest, Size) {
  TestSize();
}

// Tests comparing with +0 and -0.
TEST_F(DoubleTest, Zeros) {
  EXPECT_DOUBLE_EQ(0.0, -0.0);
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(-0.0, 1.0),
                          "1.0");
  EXPECT_FATAL_FAILURE(ASSERT_DOUBLE_EQ(0.0, 1.0),
                       "1.0");
}

// Tests comparing numbers close to 0.
//
// This ensures that *_DOUBLE_EQ handles the sign correctly and no
// overflow occurs when comparing numbers whose absolute value is very
// small.
TEST_F(DoubleTest, AlmostZeros) {
  // In C++Builder, names within local classes (such as used by
  // EXPECT_FATAL_FAILURE) cannot be resolved against static members of the
  // scoping class.  Use a static local alias as a workaround.
  static const DoubleTest::TestValues& v(this->values_);

  EXPECT_DOUBLE_EQ(0.0, v.close_to_positive_zero);
  EXPECT_DOUBLE_EQ(-0.0, v.close_to_negative_zero);
  EXPECT_DOUBLE_EQ(v.close_to_positive_zero, v.close_to_negative_zero);

  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_DOUBLE_EQ(v.close_to_positive_zero,
                     v.further_from_negative_zero);
  }, "v.further_from_negative_zero");
}

// Tests comparing numbers close to each other.
TEST_F(DoubleTest, SmallDiff) {
  EXPECT_DOUBLE_EQ(1.0, values_.close_to_one);
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(1.0, values_.further_from_one),
                          "values_.further_from_one");
}

// Tests comparing numbers far apart.
TEST_F(DoubleTest, LargeDiff) {
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(2.0, 3.0),
                          "3.0");
}

// Tests comparing with infinity.
//
// This ensures that no overflow occurs when comparing numbers whose
// absolute value is very large.
TEST_F(DoubleTest, Infinity) {
  EXPECT_DOUBLE_EQ(values_.infinity, values_.close_to_infinity);
  EXPECT_DOUBLE_EQ(-values_.infinity, -values_.close_to_infinity);
#if !GTEST_OS_SYMBIAN
  // Nokia's STLport crashes if we try to output infinity or NaN.
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(values_.infinity, -values_.infinity),
                          "-values_.infinity");

  // This is interesting as the representations of infinity_ and nan1_
  // are only 1 DLP apart.
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(values_.infinity, values_.nan1),
                          "values_.nan1");
#endif  // !GTEST_OS_SYMBIAN
}

// Tests that comparing with NAN always returns false.
TEST_F(DoubleTest, NaN) {
#if !GTEST_OS_SYMBIAN
  // In C++Builder, names within local classes (such as used by
  // EXPECT_FATAL_FAILURE) cannot be resolved against static members of the
  // scoping class.  Use a static local alias as a workaround.
  static const DoubleTest::TestValues& v(this->values_);

  // Nokia's STLport crashes if we try to output infinity or NaN.
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(v.nan1, v.nan1),
                          "v.nan1");
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(v.nan1, v.nan2), "v.nan2");
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(1.0, v.nan1), "v.nan1");
  EXPECT_FATAL_FAILURE(ASSERT_DOUBLE_EQ(v.nan1, v.infinity),
                       "v.infinity");
#endif  // !GTEST_OS_SYMBIAN
}

// Tests that *_DOUBLE_EQ are reflexive.
TEST_F(DoubleTest, Reflexive) {
  EXPECT_DOUBLE_EQ(0.0, 0.0);
  EXPECT_DOUBLE_EQ(1.0, 1.0);
#if !GTEST_OS_SYMBIAN
  // Nokia's STLport crashes if we try to output infinity or NaN.
  ASSERT_DOUBLE_EQ(values_.infinity, values_.infinity);
#endif  // !GTEST_OS_SYMBIAN
}

// Tests that *_DOUBLE_EQ are commutative.
TEST_F(DoubleTest, Commutative) {
  // We already tested EXPECT_DOUBLE_EQ(1.0, values_.close_to_one).
  EXPECT_DOUBLE_EQ(values_.close_to_one, 1.0);

  // We already tested EXPECT_DOUBLE_EQ(1.0, values_.further_from_one).
  EXPECT_NONFATAL_FAILURE(EXPECT_DOUBLE_EQ(values_.further_from_one, 1.0),
                          "1.0");
}

// Tests EXPECT_NEAR.
TEST_F(DoubleTest, EXPECT_NEAR) {
  EXPECT_NEAR(-1.0, -1.1, 0.2);
  EXPECT_NEAR(2.0, 3.0, 1.0);
  EXPECT_NONFATAL_FAILURE(EXPECT_NEAR(1.0, 1.2, 0.1),  // NOLINT
                          "The difference between 1.0 and 1.2 is 0.2, "
                          "which exceeds 0.1");
  // To work around a bug in gcc 2.95.0, there is intentionally no
  // space after the first comma in the previous statement.
}

// Tests ASSERT_NEAR.
TEST_F(DoubleTest, ASSERT_NEAR) {
  ASSERT_NEAR(-1.0, -1.1, 0.2);
  ASSERT_NEAR(2.0, 3.0, 1.0);
  EXPECT_FATAL_FAILURE(ASSERT_NEAR(1.0, 1.2, 0.1),  // NOLINT
                       "The difference between 1.0 and 1.2 is 0.2, "
                       "which exceeds 0.1");
  // To work around a bug in gcc 2.95.0, there is intentionally no
  // space after the first comma in the previous statement.
}

// Tests the cases where DoubleLE() should succeed.
TEST_F(DoubleTest, DoubleLESucceeds) {
  EXPECT_PRED_FORMAT2(DoubleLE, 1.0, 2.0);  // When val1 < val2,
  ASSERT_PRED_FORMAT2(DoubleLE, 1.0, 1.0);  // val1 == val2,

  // or when val1 is greater than, but almost equals to, val2.
  EXPECT_PRED_FORMAT2(DoubleLE, values_.close_to_positive_zero, 0.0);
}

// Tests the cases where DoubleLE() should fail.
TEST_F(DoubleTest, DoubleLEFails) {
  // When val1 is greater than val2 by a large margin,
  EXPECT_NONFATAL_FAILURE(EXPECT_PRED_FORMAT2(DoubleLE, 2.0, 1.0),
                          "(2.0) <= (1.0)");

  // or by a small yet non-negligible margin,
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT2(DoubleLE, values_.further_from_one, 1.0);
  }, "(values_.further_from_one) <= (1.0)");

#if !GTEST_OS_SYMBIAN && !defined(__BORLANDC__)
  // Nokia's STLport crashes if we try to output infinity or NaN.
  // C++Builder gives bad results for ordered comparisons involving NaNs
  // due to compiler bugs.
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT2(DoubleLE, values_.nan1, values_.infinity);
  }, "(values_.nan1) <= (values_.infinity)");
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_PRED_FORMAT2(DoubleLE, -values_.infinity, values_.nan1);
  }, " (-values_.infinity) <= (values_.nan1)");
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_PRED_FORMAT2(DoubleLE, values_.nan1, values_.nan1);
  }, "(values_.nan1) <= (values_.nan1)");
#endif  // !GTEST_OS_SYMBIAN && !defined(__BORLANDC__)
}


// Verifies that a test or test case whose name starts with DISABLED_ is
// not run.

// A test whose name starts with DISABLED_.
// Should not run.
TEST(DisabledTest, DISABLED_TestShouldNotRun) {
  FAIL() << "Unexpected failure: Disabled test should not be run.";
}

// A test whose name does not start with DISABLED_.
// Should run.
TEST(DisabledTest, NotDISABLED_TestShouldRun) {
  EXPECT_EQ(1, 1);
}

// A test case whose name starts with DISABLED_.
// Should not run.
TEST(DISABLED_TestCase, TestShouldNotRun) {
  FAIL() << "Unexpected failure: Test in disabled test case should not be run.";
}

// A test case and test whose names start with DISABLED_.
// Should not run.
TEST(DISABLED_TestCase, DISABLED_TestShouldNotRun) {
  FAIL() << "Unexpected failure: Test in disabled test case should not be run.";
}

// Check that when all tests in a test case are disabled, SetupTestCase() and
// TearDownTestCase() are not called.
class DisabledTestsTest : public Test {
 protected:
  static void SetUpTestCase() {
    FAIL() << "Unexpected failure: All tests disabled in test case. "
              "SetupTestCase() should not be called.";
  }

  static void TearDownTestCase() {
    FAIL() << "Unexpected failure: All tests disabled in test case. "
              "TearDownTestCase() should not be called.";
  }
};

TEST_F(DisabledTestsTest, DISABLED_TestShouldNotRun_1) {
  FAIL() << "Unexpected failure: Disabled test should not be run.";
}

TEST_F(DisabledTestsTest, DISABLED_TestShouldNotRun_2) {
  FAIL() << "Unexpected failure: Disabled test should not be run.";
}

// Tests that disabled typed tests aren't run.

#if GTEST_HAS_TYPED_TEST

template <typename T>
class TypedTest : public Test {
};

typedef testing::Types<int, double> NumericTypes;
TYPED_TEST_CASE(TypedTest, NumericTypes);

TYPED_TEST(TypedTest, DISABLED_ShouldNotRun) {
  FAIL() << "Unexpected failure: Disabled typed test should not run.";
}

template <typename T>
class DISABLED_TypedTest : public Test {
};

TYPED_TEST_CASE(DISABLED_TypedTest, NumericTypes);

TYPED_TEST(DISABLED_TypedTest, ShouldNotRun) {
  FAIL() << "Unexpected failure: Disabled typed test should not run.";
}

#endif  // GTEST_HAS_TYPED_TEST

// Tests that disabled type-parameterized tests aren't run.

#if GTEST_HAS_TYPED_TEST_P

template <typename T>
class TypedTestP : public Test {
};

TYPED_TEST_CASE_P(TypedTestP);

TYPED_TEST_P(TypedTestP, DISABLED_ShouldNotRun) {
  FAIL() << "Unexpected failure: "
         << "Disabled type-parameterized test should not run.";
}

REGISTER_TYPED_TEST_CASE_P(TypedTestP, DISABLED_ShouldNotRun);

INSTANTIATE_TYPED_TEST_CASE_P(My, TypedTestP, NumericTypes);

template <typename T>
class DISABLED_TypedTestP : public Test {
};

TYPED_TEST_CASE_P(DISABLED_TypedTestP);

TYPED_TEST_P(DISABLED_TypedTestP, ShouldNotRun) {
  FAIL() << "Unexpected failure: "
         << "Disabled type-parameterized test should not run.";
}

REGISTER_TYPED_TEST_CASE_P(DISABLED_TypedTestP, ShouldNotRun);

INSTANTIATE_TYPED_TEST_CASE_P(My, DISABLED_TypedTestP, NumericTypes);

#endif  // GTEST_HAS_TYPED_TEST_P

// Tests that assertion macros evaluate their arguments exactly once.

class SingleEvaluationTest : public Test {
 public:  // Must be public and not protected due to a bug in g++ 3.4.2.
  // This helper function is needed by the FailedASSERT_STREQ test
  // below.  It's public to work around C++Builder's bug with scoping local
  // classes.
  static void CompareAndIncrementCharPtrs() {
    ASSERT_STREQ(p1_++, p2_++);
  }

  // This helper function is needed by the FailedASSERT_NE test below.  It's
  // public to work around C++Builder's bug with scoping local classes.
  static void CompareAndIncrementInts() {
    ASSERT_NE(a_++, b_++);
  }

 protected:
  SingleEvaluationTest() {
    p1_ = s1_;
    p2_ = s2_;
    a_ = 0;
    b_ = 0;
  }

  static const char* const s1_;
  static const char* const s2_;
  static const char* p1_;
  static const char* p2_;

  static int a_;
  static int b_;
};

const char* const SingleEvaluationTest::s1_ = "01234";
const char* const SingleEvaluationTest::s2_ = "abcde";
const char* SingleEvaluationTest::p1_;
const char* SingleEvaluationTest::p2_;
int SingleEvaluationTest::a_;
int SingleEvaluationTest::b_;

// Tests that when ASSERT_STREQ fails, it evaluates its arguments
// exactly once.
TEST_F(SingleEvaluationTest, FailedASSERT_STREQ) {
  EXPECT_FATAL_FAILURE(SingleEvaluationTest::CompareAndIncrementCharPtrs(),
                       "p2_++");
  EXPECT_EQ(s1_ + 1, p1_);
  EXPECT_EQ(s2_ + 1, p2_);
}

// Tests that string assertion arguments are evaluated exactly once.
TEST_F(SingleEvaluationTest, ASSERT_STR) {
  // successful EXPECT_STRNE
  EXPECT_STRNE(p1_++, p2_++);
  EXPECT_EQ(s1_ + 1, p1_);
  EXPECT_EQ(s2_ + 1, p2_);

  // failed EXPECT_STRCASEEQ
  EXPECT_NONFATAL_FAILURE(EXPECT_STRCASEEQ(p1_++, p2_++),
                          "ignoring case");
  EXPECT_EQ(s1_ + 2, p1_);
  EXPECT_EQ(s2_ + 2, p2_);
}

// Tests that when ASSERT_NE fails, it evaluates its arguments exactly
// once.
TEST_F(SingleEvaluationTest, FailedASSERT_NE) {
  EXPECT_FATAL_FAILURE(SingleEvaluationTest::CompareAndIncrementInts(),
                       "(a_++) != (b_++)");
  EXPECT_EQ(1, a_);
  EXPECT_EQ(1, b_);
}

// Tests that assertion arguments are evaluated exactly once.
TEST_F(SingleEvaluationTest, OtherCases) {
  // successful EXPECT_TRUE
  EXPECT_TRUE(0 == a_++);  // NOLINT
  EXPECT_EQ(1, a_);

  // failed EXPECT_TRUE
  EXPECT_NONFATAL_FAILURE(EXPECT_TRUE(-1 == a_++), "-1 == a_++");
  EXPECT_EQ(2, a_);

  // successful EXPECT_GT
  EXPECT_GT(a_++, b_++);
  EXPECT_EQ(3, a_);
  EXPECT_EQ(1, b_);

  // failed EXPECT_LT
  EXPECT_NONFATAL_FAILURE(EXPECT_LT(a_++, b_++), "(a_++) < (b_++)");
  EXPECT_EQ(4, a_);
  EXPECT_EQ(2, b_);

  // successful ASSERT_TRUE
  ASSERT_TRUE(0 < a_++);  // NOLINT
  EXPECT_EQ(5, a_);

  // successful ASSERT_GT
  ASSERT_GT(a_++, b_++);
  EXPECT_EQ(6, a_);
  EXPECT_EQ(3, b_);
}

#if GTEST_HAS_EXCEPTIONS

void ThrowAnInteger() {
  throw 1;
}

// Tests that assertion arguments are evaluated exactly once.
TEST_F(SingleEvaluationTest, ExceptionTests) {
  // successful EXPECT_THROW
  EXPECT_THROW({  // NOLINT
    a_++;
    ThrowAnInteger();
  }, int);
  EXPECT_EQ(1, a_);

  // failed EXPECT_THROW, throws different
  EXPECT_NONFATAL_FAILURE(EXPECT_THROW({  // NOLINT
    a_++;
    ThrowAnInteger();
  }, bool), "throws a different type");
  EXPECT_EQ(2, a_);

  // failed EXPECT_THROW, throws nothing
  EXPECT_NONFATAL_FAILURE(EXPECT_THROW(a_++, bool), "throws nothing");
  EXPECT_EQ(3, a_);

  // successful EXPECT_NO_THROW
  EXPECT_NO_THROW(a_++);
  EXPECT_EQ(4, a_);

  // failed EXPECT_NO_THROW
  EXPECT_NONFATAL_FAILURE(EXPECT_NO_THROW({  // NOLINT
    a_++;
    ThrowAnInteger();
  }), "it throws");
  EXPECT_EQ(5, a_);

  // successful EXPECT_ANY_THROW
  EXPECT_ANY_THROW({  // NOLINT
    a_++;
    ThrowAnInteger();
  });
  EXPECT_EQ(6, a_);

  // failed EXPECT_ANY_THROW
  EXPECT_NONFATAL_FAILURE(EXPECT_ANY_THROW(a_++), "it doesn't");
  EXPECT_EQ(7, a_);
}

#endif  // GTEST_HAS_EXCEPTIONS

// Tests {ASSERT|EXPECT}_NO_FATAL_FAILURE.
class NoFatalFailureTest : public Test {
 protected:
  void Succeeds() {}
  void FailsNonFatal() {
    ADD_FAILURE() << "some non-fatal failure";
  }
  void Fails() {
    FAIL() << "some fatal failure";
  }

  void DoAssertNoFatalFailureOnFails() {
    ASSERT_NO_FATAL_FAILURE(Fails());
    ADD_FAILURE() << "shold not reach here.";
  }

  void DoExpectNoFatalFailureOnFails() {
    EXPECT_NO_FATAL_FAILURE(Fails());
    ADD_FAILURE() << "other failure";
  }
};

TEST_F(NoFatalFailureTest, NoFailure) {
  EXPECT_NO_FATAL_FAILURE(Succeeds());
  ASSERT_NO_FATAL_FAILURE(Succeeds());
}

TEST_F(NoFatalFailureTest, NonFatalIsNoFailure) {
  EXPECT_NONFATAL_FAILURE(
      EXPECT_NO_FATAL_FAILURE(FailsNonFatal()),
      "some non-fatal failure");
  EXPECT_NONFATAL_FAILURE(
      ASSERT_NO_FATAL_FAILURE(FailsNonFatal()),
      "some non-fatal failure");
}

TEST_F(NoFatalFailureTest, AssertNoFatalFailureOnFatalFailure) {
  TestPartResultArray gtest_failures;
  {
    ScopedFakeTestPartResultReporter gtest_reporter(&gtest_failures);
    DoAssertNoFatalFailureOnFails();
  }
  ASSERT_EQ(2, gtest_failures.size());
  EXPECT_EQ(TestPartResult::kFatalFailure,
            gtest_failures.GetTestPartResult(0).type());
  EXPECT_EQ(TestPartResult::kFatalFailure,
            gtest_failures.GetTestPartResult(1).type());
  EXPECT_PRED_FORMAT2(testing::IsSubstring, "some fatal failure",
                      gtest_failures.GetTestPartResult(0).message());
  EXPECT_PRED_FORMAT2(testing::IsSubstring, "it does",
                      gtest_failures.GetTestPartResult(1).message());
}

TEST_F(NoFatalFailureTest, ExpectNoFatalFailureOnFatalFailure) {
  TestPartResultArray gtest_failures;
  {
    ScopedFakeTestPartResultReporter gtest_reporter(&gtest_failures);
    DoExpectNoFatalFailureOnFails();
  }
  ASSERT_EQ(3, gtest_failures.size());
  EXPECT_EQ(TestPartResult::kFatalFailure,
            gtest_failures.GetTestPartResult(0).type());
  EXPECT_EQ(TestPartResult::kNonFatalFailure,
            gtest_failures.GetTestPartResult(1).type());
  EXPECT_EQ(TestPartResult::kNonFatalFailure,
            gtest_failures.GetTestPartResult(2).type());
  EXPECT_PRED_FORMAT2(testing::IsSubstring, "some fatal failure",
                      gtest_failures.GetTestPartResult(0).message());
  EXPECT_PRED_FORMAT2(testing::IsSubstring, "it does",
                      gtest_failures.GetTestPartResult(1).message());
  EXPECT_PRED_FORMAT2(testing::IsSubstring, "other failure",
                      gtest_failures.GetTestPartResult(2).message());
}

TEST_F(NoFatalFailureTest, MessageIsStreamable) {
  TestPartResultArray gtest_failures;
  {
    ScopedFakeTestPartResultReporter gtest_reporter(&gtest_failures);
    EXPECT_NO_FATAL_FAILURE(FAIL() << "foo") << "my message";
  }
  ASSERT_EQ(2, gtest_failures.size());
  EXPECT_EQ(TestPartResult::kNonFatalFailure,
            gtest_failures.GetTestPartResult(0).type());
  EXPECT_EQ(TestPartResult::kNonFatalFailure,
            gtest_failures.GetTestPartResult(1).type());
  EXPECT_PRED_FORMAT2(testing::IsSubstring, "foo",
                      gtest_failures.GetTestPartResult(0).message());
  EXPECT_PRED_FORMAT2(testing::IsSubstring, "my message",
                      gtest_failures.GetTestPartResult(1).message());
}

// Tests non-string assertions.

// Tests EqFailure(), used for implementing *EQ* assertions.
TEST(AssertionTest, EqFailure) {
  const String foo_val("5"), bar_val("6");
  const String msg1(
      EqFailure("foo", "bar", foo_val, bar_val, false)
      .failure_message());
  EXPECT_STREQ(
      "Value of: bar\n"
      "  Actual: 6\n"
      "Expected: foo\n"
      "Which is: 5",
      msg1.c_str());

  const String msg2(
      EqFailure("foo", "6", foo_val, bar_val, false)
      .failure_message());
  EXPECT_STREQ(
      "Value of: 6\n"
      "Expected: foo\n"
      "Which is: 5",
      msg2.c_str());

  const String msg3(
      EqFailure("5", "bar", foo_val, bar_val, false)
      .failure_message());
  EXPECT_STREQ(
      "Value of: bar\n"
      "  Actual: 6\n"
      "Expected: 5",
      msg3.c_str());

  const String msg4(
      EqFailure("5", "6", foo_val, bar_val, false).failure_message());
  EXPECT_STREQ(
      "Value of: 6\n"
      "Expected: 5",
      msg4.c_str());

  const String msg5(
      EqFailure("foo", "bar",
                String("\"x\""), String("\"y\""),
                true).failure_message());
  EXPECT_STREQ(
      "Value of: bar\n"
      "  Actual: \"y\"\n"
      "Expected: foo (ignoring case)\n"
      "Which is: \"x\"",
      msg5.c_str());
}

// Tests AppendUserMessage(), used for implementing the *EQ* macros.
TEST(AssertionTest, AppendUserMessage) {
  const String foo("foo");

  Message msg;
  EXPECT_STREQ("foo",
               AppendUserMessage(foo, msg).c_str());

  msg << "bar";
  EXPECT_STREQ("foo\nbar",
               AppendUserMessage(foo, msg).c_str());
}

#ifdef __BORLANDC__
// Silences warnings: "Condition is always true", "Unreachable code"
#pragma option push -w-ccc -w-rch
#endif

// Tests ASSERT_TRUE.
TEST(AssertionTest, ASSERT_TRUE) {
  ASSERT_TRUE(2 > 1);  // NOLINT
  EXPECT_FATAL_FAILURE(ASSERT_TRUE(2 < 1),
                       "2 < 1");
}

// Tests ASSERT_FALSE.
TEST(AssertionTest, ASSERT_FALSE) {
  ASSERT_FALSE(2 < 1);  // NOLINT
  EXPECT_FATAL_FAILURE(ASSERT_FALSE(2 > 1),
                       "Value of: 2 > 1\n"
                       "  Actual: true\n"
                       "Expected: false");
}

#ifdef __BORLANDC__
// Restores warnings after previous "#pragma option push" supressed them
#pragma option pop
#endif

// Tests using ASSERT_EQ on double values.  The purpose is to make
// sure that the specialization we did for integer and anonymous enums
// isn't used for double arguments.
TEST(ExpectTest, ASSERT_EQ_Double) {
  // A success.
  ASSERT_EQ(5.6, 5.6);

  // A failure.
  EXPECT_FATAL_FAILURE(ASSERT_EQ(5.1, 5.2),
                       "5.1");
}

// Tests ASSERT_EQ.
TEST(AssertionTest, ASSERT_EQ) {
  ASSERT_EQ(5, 2 + 3);
  EXPECT_FATAL_FAILURE(ASSERT_EQ(5, 2*3),
                       "Value of: 2*3\n"
                       "  Actual: 6\n"
                       "Expected: 5");
}

// Tests ASSERT_EQ(NULL, pointer).
#if !GTEST_OS_SYMBIAN
// The NULL-detection template magic fails to compile with
// the Nokia compiler and crashes the ARM compiler, hence
// not testing on Symbian.
TEST(AssertionTest, ASSERT_EQ_NULL) {
  // A success.
  const char* p = NULL;
  // Some older GCC versions may issue a spurious waring in this or the next
  // assertion statement. This warning should not be suppressed with
  // static_cast since the test verifies the ability to use bare NULL as the
  // expected parameter to the macro.
  ASSERT_EQ(NULL, p);

  // A failure.
  static int n = 0;
  EXPECT_FATAL_FAILURE(ASSERT_EQ(NULL, &n),
                       "Value of: &n\n");
}
#endif  // !GTEST_OS_SYMBIAN

// Tests ASSERT_EQ(0, non_pointer).  Since the literal 0 can be
// treated as a null pointer by the compiler, we need to make sure
// that ASSERT_EQ(0, non_pointer) isn't interpreted by Google Test as
// ASSERT_EQ(static_cast<void*>(NULL), non_pointer).
TEST(ExpectTest, ASSERT_EQ_0) {
  int n = 0;

  // A success.
  ASSERT_EQ(0, n);

  // A failure.
  EXPECT_FATAL_FAILURE(ASSERT_EQ(0, 5.6),
                       "Expected: 0");
}

// Tests ASSERT_NE.
TEST(AssertionTest, ASSERT_NE) {
  ASSERT_NE(6, 7);
  EXPECT_FATAL_FAILURE(ASSERT_NE('a', 'a'),
                       "Expected: ('a') != ('a'), "
                       "actual: 'a' (97, 0x61) vs 'a' (97, 0x61)");
}

// Tests ASSERT_LE.
TEST(AssertionTest, ASSERT_LE) {
  ASSERT_LE(2, 3);
  ASSERT_LE(2, 2);
  EXPECT_FATAL_FAILURE(ASSERT_LE(2, 0),
                       "Expected: (2) <= (0), actual: 2 vs 0");
}

// Tests ASSERT_LT.
TEST(AssertionTest, ASSERT_LT) {
  ASSERT_LT(2, 3);
  EXPECT_FATAL_FAILURE(ASSERT_LT(2, 2),
                       "Expected: (2) < (2), actual: 2 vs 2");
}

// Tests ASSERT_GE.
TEST(AssertionTest, ASSERT_GE) {
  ASSERT_GE(2, 1);
  ASSERT_GE(2, 2);
  EXPECT_FATAL_FAILURE(ASSERT_GE(2, 3),
                       "Expected: (2) >= (3), actual: 2 vs 3");
}

// Tests ASSERT_GT.
TEST(AssertionTest, ASSERT_GT) {
  ASSERT_GT(2, 1);
  EXPECT_FATAL_FAILURE(ASSERT_GT(2, 2),
                       "Expected: (2) > (2), actual: 2 vs 2");
}

#if GTEST_HAS_EXCEPTIONS

void ThrowNothing() {}

// Tests ASSERT_THROW.
TEST(AssertionTest, ASSERT_THROW) {
  ASSERT_THROW(ThrowAnInteger(), int);
#if !defined(__BORLANDC__) || __BORLANDC__ >= 0x600 || defined(_DEBUG)
  // ICE's in C++Builder 2007 (Release build).
  EXPECT_FATAL_FAILURE(
      ASSERT_THROW(ThrowAnInteger(), bool),
      "Expected: ThrowAnInteger() throws an exception of type bool.\n"
      "  Actual: it throws a different type.");
#endif
  EXPECT_FATAL_FAILURE(
      ASSERT_THROW(ThrowNothing(), bool),
      "Expected: ThrowNothing() throws an exception of type bool.\n"
      "  Actual: it throws nothing.");
}

// Tests ASSERT_NO_THROW.
TEST(AssertionTest, ASSERT_NO_THROW) {
  ASSERT_NO_THROW(ThrowNothing());
  EXPECT_FATAL_FAILURE(ASSERT_NO_THROW(ThrowAnInteger()),
                       "Expected: ThrowAnInteger() doesn't throw an exception."
                       "\n  Actual: it throws.");
}

// Tests ASSERT_ANY_THROW.
TEST(AssertionTest, ASSERT_ANY_THROW) {
  ASSERT_ANY_THROW(ThrowAnInteger());
  EXPECT_FATAL_FAILURE(
      ASSERT_ANY_THROW(ThrowNothing()),
      "Expected: ThrowNothing() throws an exception.\n"
      "  Actual: it doesn't.");
}

#endif  // GTEST_HAS_EXCEPTIONS

// Makes sure we deal with the precedence of <<.  This test should
// compile.
TEST(AssertionTest, AssertPrecedence) {
  ASSERT_EQ(1 < 2, true);
  ASSERT_EQ(true && false, false);
}

// A subroutine used by the following test.
void TestEq1(int x) {
  ASSERT_EQ(1, x);
}

// Tests calling a test subroutine that's not part of a fixture.
TEST(AssertionTest, NonFixtureSubroutine) {
  EXPECT_FATAL_FAILURE(TestEq1(2),
                       "Value of: x");
}

// An uncopyable class.
class Uncopyable {
 public:
  explicit Uncopyable(int value) : value_(value) {}

  int value() const { return value_; }
  bool operator==(const Uncopyable& rhs) const {
    return value() == rhs.value();
  }
 private:
  // This constructor deliberately has no implementation, as we don't
  // want this class to be copyable.
  Uncopyable(const Uncopyable&);  // NOLINT

  int value_;
};

::std::ostream& operator<<(::std::ostream& os, const Uncopyable& value) {
  return os << value.value();
}


bool IsPositiveUncopyable(const Uncopyable& x) {
  return x.value() > 0;
}

// A subroutine used by the following test.
void TestAssertNonPositive() {
  Uncopyable y(-1);
  ASSERT_PRED1(IsPositiveUncopyable, y);
}
// A subroutine used by the following test.
void TestAssertEqualsUncopyable() {
  Uncopyable x(5);
  Uncopyable y(-1);
  ASSERT_EQ(x, y);
}

// Tests that uncopyable objects can be used in assertions.
TEST(AssertionTest, AssertWorksWithUncopyableObject) {
  Uncopyable x(5);
  ASSERT_PRED1(IsPositiveUncopyable, x);
  ASSERT_EQ(x, x);
  EXPECT_FATAL_FAILURE(TestAssertNonPositive(),
    "IsPositiveUncopyable(y) evaluates to false, where\ny evaluates to -1");
  EXPECT_FATAL_FAILURE(TestAssertEqualsUncopyable(),
    "Value of: y\n  Actual: -1\nExpected: x\nWhich is: 5");
}

// Tests that uncopyable objects can be used in expects.
TEST(AssertionTest, ExpectWorksWithUncopyableObject) {
  Uncopyable x(5);
  EXPECT_PRED1(IsPositiveUncopyable, x);
  Uncopyable y(-1);
  EXPECT_NONFATAL_FAILURE(EXPECT_PRED1(IsPositiveUncopyable, y),
    "IsPositiveUncopyable(y) evaluates to false, where\ny evaluates to -1");
  EXPECT_EQ(x, x);
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(x, y),
    "Value of: y\n  Actual: -1\nExpected: x\nWhich is: 5");
}


// The version of gcc used in XCode 2.2 has a bug and doesn't allow
// anonymous enums in assertions.  Therefore the following test is not
// done on Mac.
#if !GTEST_OS_MAC

// Tests using assertions with anonymous enums.
enum {
  CASE_A = -1,
#if GTEST_OS_LINUX
  // We want to test the case where the size of the anonymous enum is
  // larger than sizeof(int), to make sure our implementation of the
  // assertions doesn't truncate the enums.  However, MSVC
  // (incorrectly) doesn't allow an enum value to exceed the range of
  // an int, so this has to be conditionally compiled.
  //
  // On Linux, CASE_B and CASE_A have the same value when truncated to
  // int size.  We want to test whether this will confuse the
  // assertions.
  CASE_B = testing::internal::kMaxBiggestInt,
#else
  CASE_B = INT_MAX,
#endif  // GTEST_OS_LINUX
};

TEST(AssertionTest, AnonymousEnum) {
#if GTEST_OS_LINUX
  EXPECT_EQ(static_cast<int>(CASE_A), static_cast<int>(CASE_B));
#endif  // GTEST_OS_LINUX

  EXPECT_EQ(CASE_A, CASE_A);
  EXPECT_NE(CASE_A, CASE_B);
  EXPECT_LT(CASE_A, CASE_B);
  EXPECT_LE(CASE_A, CASE_B);
  EXPECT_GT(CASE_B, CASE_A);
  EXPECT_GE(CASE_A, CASE_A);
  EXPECT_NONFATAL_FAILURE(EXPECT_GE(CASE_A, CASE_B),
                          "(CASE_A) >= (CASE_B)");

  ASSERT_EQ(CASE_A, CASE_A);
  ASSERT_NE(CASE_A, CASE_B);
  ASSERT_LT(CASE_A, CASE_B);
  ASSERT_LE(CASE_A, CASE_B);
  ASSERT_GT(CASE_B, CASE_A);
  ASSERT_GE(CASE_A, CASE_A);
  EXPECT_FATAL_FAILURE(ASSERT_EQ(CASE_A, CASE_B),
                       "Value of: CASE_B");
}

#endif  // !GTEST_OS_MAC

#if GTEST_OS_WINDOWS

static HRESULT UnexpectedHRESULTFailure() {
  return E_UNEXPECTED;
}

static HRESULT OkHRESULTSuccess() {
  return S_OK;
}

static HRESULT FalseHRESULTSuccess() {
  return S_FALSE;
}

// HRESULT assertion tests test both zero and non-zero
// success codes as well as failure message for each.
//
// Windows CE doesn't support message texts.
TEST(HRESULTAssertionTest, EXPECT_HRESULT_SUCCEEDED) {
  EXPECT_HRESULT_SUCCEEDED(S_OK);
  EXPECT_HRESULT_SUCCEEDED(S_FALSE);

  EXPECT_NONFATAL_FAILURE(EXPECT_HRESULT_SUCCEEDED(UnexpectedHRESULTFailure()),
    "Expected: (UnexpectedHRESULTFailure()) succeeds.\n"
    "  Actual: 0x8000FFFF");
}

TEST(HRESULTAssertionTest, ASSERT_HRESULT_SUCCEEDED) {
  ASSERT_HRESULT_SUCCEEDED(S_OK);
  ASSERT_HRESULT_SUCCEEDED(S_FALSE);

  EXPECT_FATAL_FAILURE(ASSERT_HRESULT_SUCCEEDED(UnexpectedHRESULTFailure()),
    "Expected: (UnexpectedHRESULTFailure()) succeeds.\n"
    "  Actual: 0x8000FFFF");
}

TEST(HRESULTAssertionTest, EXPECT_HRESULT_FAILED) {
  EXPECT_HRESULT_FAILED(E_UNEXPECTED);

  EXPECT_NONFATAL_FAILURE(EXPECT_HRESULT_FAILED(OkHRESULTSuccess()),
    "Expected: (OkHRESULTSuccess()) fails.\n"
    "  Actual: 0x00000000");
  EXPECT_NONFATAL_FAILURE(EXPECT_HRESULT_FAILED(FalseHRESULTSuccess()),
    "Expected: (FalseHRESULTSuccess()) fails.\n"
    "  Actual: 0x00000001");
}

TEST(HRESULTAssertionTest, ASSERT_HRESULT_FAILED) {
  ASSERT_HRESULT_FAILED(E_UNEXPECTED);

#ifndef __BORLANDC__
  // ICE's in C++Builder 2007 and 2009.
  EXPECT_FATAL_FAILURE(ASSERT_HRESULT_FAILED(OkHRESULTSuccess()),
    "Expected: (OkHRESULTSuccess()) fails.\n"
    "  Actual: 0x00000000");
#endif
  EXPECT_FATAL_FAILURE(ASSERT_HRESULT_FAILED(FalseHRESULTSuccess()),
    "Expected: (FalseHRESULTSuccess()) fails.\n"
    "  Actual: 0x00000001");
}

// Tests that streaming to the HRESULT macros works.
TEST(HRESULTAssertionTest, Streaming) {
  EXPECT_HRESULT_SUCCEEDED(S_OK) << "unexpected failure";
  ASSERT_HRESULT_SUCCEEDED(S_OK) << "unexpected failure";
  EXPECT_HRESULT_FAILED(E_UNEXPECTED) << "unexpected failure";
  ASSERT_HRESULT_FAILED(E_UNEXPECTED) << "unexpected failure";

  EXPECT_NONFATAL_FAILURE(
      EXPECT_HRESULT_SUCCEEDED(E_UNEXPECTED) << "expected failure",
      "expected failure");

#ifndef __BORLANDC__
  // ICE's in C++Builder 2007 and 2009.
  EXPECT_FATAL_FAILURE(
      ASSERT_HRESULT_SUCCEEDED(E_UNEXPECTED) << "expected failure",
      "expected failure");
#endif

  EXPECT_NONFATAL_FAILURE(
      EXPECT_HRESULT_FAILED(S_OK) << "expected failure",
      "expected failure");

  EXPECT_FATAL_FAILURE(
      ASSERT_HRESULT_FAILED(S_OK) << "expected failure",
      "expected failure");
}

#endif  // GTEST_OS_WINDOWS

#ifdef __BORLANDC__
// Silences warnings: "Condition is always true", "Unreachable code"
#pragma option push -w-ccc -w-rch
#endif

// Tests that the assertion macros behave like single statements.
TEST(AssertionSyntaxTest, BasicAssertionsBehavesLikeSingleStatement) {
  if (AlwaysFalse())
    ASSERT_TRUE(false) << "This should never be executed; "
                          "It's a compilation test only.";

  if (AlwaysTrue())
    EXPECT_FALSE(false);
  else
    ;  // NOLINT

  if (AlwaysFalse())
    ASSERT_LT(1, 3);

  if (AlwaysFalse())
    ;  // NOLINT
  else
    EXPECT_GT(3, 2) << "";
}

#if GTEST_HAS_EXCEPTIONS
// Tests that the compiler will not complain about unreachable code in the
// EXPECT_THROW/EXPECT_ANY_THROW/EXPECT_NO_THROW macros.
TEST(ExpectThrowTest, DoesNotGenerateUnreachableCodeWarning) {
  int n = 0;

  EXPECT_THROW(throw 1, int);
  EXPECT_NONFATAL_FAILURE(EXPECT_THROW(n++, int), "");
  EXPECT_NONFATAL_FAILURE(EXPECT_THROW(throw 1, const char*), "");
  EXPECT_NO_THROW(n++);
  EXPECT_NONFATAL_FAILURE(EXPECT_NO_THROW(throw 1), "");
  EXPECT_ANY_THROW(throw 1);
  EXPECT_NONFATAL_FAILURE(EXPECT_ANY_THROW(n++), "");
}

TEST(AssertionSyntaxTest, ExceptionAssertionsBehavesLikeSingleStatement) {
  if (AlwaysFalse())
    EXPECT_THROW(ThrowNothing(), bool);

  if (AlwaysTrue())
    EXPECT_THROW(ThrowAnInteger(), int);
  else
    ;  // NOLINT

  if (AlwaysFalse())
    EXPECT_NO_THROW(ThrowAnInteger());

  if (AlwaysTrue())
    EXPECT_NO_THROW(ThrowNothing());
  else
    ;  // NOLINT

  if (AlwaysFalse())
    EXPECT_ANY_THROW(ThrowNothing());

  if (AlwaysTrue())
    EXPECT_ANY_THROW(ThrowAnInteger());
  else
    ;  // NOLINT
}
#endif  // GTEST_HAS_EXCEPTIONS

TEST(AssertionSyntaxTest, NoFatalFailureAssertionsBehavesLikeSingleStatement) {
  if (AlwaysFalse())
    EXPECT_NO_FATAL_FAILURE(FAIL()) << "This should never be executed. "
                                    << "It's a compilation test only.";
  else
    ;  // NOLINT

  if (AlwaysFalse())
    ASSERT_NO_FATAL_FAILURE(FAIL()) << "";
  else
    ;  // NOLINT

  if (AlwaysTrue())
    EXPECT_NO_FATAL_FAILURE(SUCCEED());
  else
    ;  // NOLINT

  if (AlwaysFalse())
    ;  // NOLINT
  else
    ASSERT_NO_FATAL_FAILURE(SUCCEED());
}

// Tests that the assertion macros work well with switch statements.
TEST(AssertionSyntaxTest, WorksWithSwitch) {
  switch (0) {
    case 1:
      break;
    default:
      ASSERT_TRUE(true);
  }

  switch (0)
    case 0:
      EXPECT_FALSE(false) << "EXPECT_FALSE failed in switch case";

  // Binary assertions are implemented using a different code path
  // than the Boolean assertions.  Hence we test them separately.
  switch (0) {
    case 1:
    default:
      ASSERT_EQ(1, 1) << "ASSERT_EQ failed in default switch handler";
  }

  switch (0)
    case 0:
      EXPECT_NE(1, 2);
}

#if GTEST_HAS_EXCEPTIONS

void ThrowAString() {
    throw "String";
}

// Test that the exception assertion macros compile and work with const
// type qualifier.
TEST(AssertionSyntaxTest, WorksWithConst) {
    ASSERT_THROW(ThrowAString(), const char*);

    EXPECT_THROW(ThrowAString(), const char*);
}

#endif  // GTEST_HAS_EXCEPTIONS

}  // namespace

namespace testing {

// Tests that Google Test tracks SUCCEED*.
TEST(SuccessfulAssertionTest, SUCCEED) {
  SUCCEED();
  SUCCEED() << "OK";
  EXPECT_EQ(2, GetUnitTestImpl()->current_test_result()->total_part_count());
}

// Tests that Google Test doesn't track successful EXPECT_*.
TEST(SuccessfulAssertionTest, EXPECT) {
  EXPECT_TRUE(true);
  EXPECT_EQ(0, GetUnitTestImpl()->current_test_result()->total_part_count());
}

// Tests that Google Test doesn't track successful EXPECT_STR*.
TEST(SuccessfulAssertionTest, EXPECT_STR) {
  EXPECT_STREQ("", "");
  EXPECT_EQ(0, GetUnitTestImpl()->current_test_result()->total_part_count());
}

// Tests that Google Test doesn't track successful ASSERT_*.
TEST(SuccessfulAssertionTest, ASSERT) {
  ASSERT_TRUE(true);
  EXPECT_EQ(0, GetUnitTestImpl()->current_test_result()->total_part_count());
}

// Tests that Google Test doesn't track successful ASSERT_STR*.
TEST(SuccessfulAssertionTest, ASSERT_STR) {
  ASSERT_STREQ("", "");
  EXPECT_EQ(0, GetUnitTestImpl()->current_test_result()->total_part_count());
}

}  // namespace testing

namespace {

// Tests EXPECT_TRUE.
TEST(ExpectTest, EXPECT_TRUE) {
  EXPECT_TRUE(2 > 1);  // NOLINT
  EXPECT_NONFATAL_FAILURE(EXPECT_TRUE(2 < 1),
                          "Value of: 2 < 1\n"
                          "  Actual: false\n"
                          "Expected: true");
  EXPECT_NONFATAL_FAILURE(EXPECT_TRUE(2 > 3),
                          "2 > 3");
}

// Tests EXPECT_FALSE.
TEST(ExpectTest, EXPECT_FALSE) {
  EXPECT_FALSE(2 < 1);  // NOLINT
  EXPECT_NONFATAL_FAILURE(EXPECT_FALSE(2 > 1),
                          "Value of: 2 > 1\n"
                          "  Actual: true\n"
                          "Expected: false");
  EXPECT_NONFATAL_FAILURE(EXPECT_FALSE(2 < 3),
                          "2 < 3");
}

#ifdef __BORLANDC__
// Restores warnings after previous "#pragma option push" supressed them
#pragma option pop
#endif

// Tests EXPECT_EQ.
TEST(ExpectTest, EXPECT_EQ) {
  EXPECT_EQ(5, 2 + 3);
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(5, 2*3),
                          "Value of: 2*3\n"
                          "  Actual: 6\n"
                          "Expected: 5");
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(5, 2 - 3),
                          "2 - 3");
}

// Tests using EXPECT_EQ on double values.  The purpose is to make
// sure that the specialization we did for integer and anonymous enums
// isn't used for double arguments.
TEST(ExpectTest, EXPECT_EQ_Double) {
  // A success.
  EXPECT_EQ(5.6, 5.6);

  // A failure.
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(5.1, 5.2),
                          "5.1");
}

#if !GTEST_OS_SYMBIAN
// Tests EXPECT_EQ(NULL, pointer).
TEST(ExpectTest, EXPECT_EQ_NULL) {
  // A success.
  const char* p = NULL;
  // Some older GCC versions may issue a spurious waring in this or the next
  // assertion statement. This warning should not be suppressed with
  // static_cast since the test verifies the ability to use bare NULL as the
  // expected parameter to the macro.
  EXPECT_EQ(NULL, p);

  // A failure.
  int n = 0;
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(NULL, &n),
                          "Value of: &n\n");
}
#endif  // !GTEST_OS_SYMBIAN

// Tests EXPECT_EQ(0, non_pointer).  Since the literal 0 can be
// treated as a null pointer by the compiler, we need to make sure
// that EXPECT_EQ(0, non_pointer) isn't interpreted by Google Test as
// EXPECT_EQ(static_cast<void*>(NULL), non_pointer).
TEST(ExpectTest, EXPECT_EQ_0) {
  int n = 0;

  // A success.
  EXPECT_EQ(0, n);

  // A failure.
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(0, 5.6),
                          "Expected: 0");
}

// Tests EXPECT_NE.
TEST(ExpectTest, EXPECT_NE) {
  EXPECT_NE(6, 7);

  EXPECT_NONFATAL_FAILURE(EXPECT_NE('a', 'a'),
                          "Expected: ('a') != ('a'), "
                          "actual: 'a' (97, 0x61) vs 'a' (97, 0x61)");
  EXPECT_NONFATAL_FAILURE(EXPECT_NE(2, 2),
                          "2");
  char* const p0 = NULL;
  EXPECT_NONFATAL_FAILURE(EXPECT_NE(p0, p0),
                          "p0");
  // Only way to get the Nokia compiler to compile the cast
  // is to have a separate void* variable first. Putting
  // the two casts on the same line doesn't work, neither does
  // a direct C-style to char*.
  void* pv1 = (void*)0x1234;  // NOLINT
  char* const p1 = reinterpret_cast<char*>(pv1);
  EXPECT_NONFATAL_FAILURE(EXPECT_NE(p1, p1),
                          "p1");
}

// Tests EXPECT_LE.
TEST(ExpectTest, EXPECT_LE) {
  EXPECT_LE(2, 3);
  EXPECT_LE(2, 2);
  EXPECT_NONFATAL_FAILURE(EXPECT_LE(2, 0),
                          "Expected: (2) <= (0), actual: 2 vs 0");
  EXPECT_NONFATAL_FAILURE(EXPECT_LE(1.1, 0.9),
                          "(1.1) <= (0.9)");
}

// Tests EXPECT_LT.
TEST(ExpectTest, EXPECT_LT) {
  EXPECT_LT(2, 3);
  EXPECT_NONFATAL_FAILURE(EXPECT_LT(2, 2),
                          "Expected: (2) < (2), actual: 2 vs 2");
  EXPECT_NONFATAL_FAILURE(EXPECT_LT(2, 1),
                          "(2) < (1)");
}

// Tests EXPECT_GE.
TEST(ExpectTest, EXPECT_GE) {
  EXPECT_GE(2, 1);
  EXPECT_GE(2, 2);
  EXPECT_NONFATAL_FAILURE(EXPECT_GE(2, 3),
                          "Expected: (2) >= (3), actual: 2 vs 3");
  EXPECT_NONFATAL_FAILURE(EXPECT_GE(0.9, 1.1),
                          "(0.9) >= (1.1)");
}

// Tests EXPECT_GT.
TEST(ExpectTest, EXPECT_GT) {
  EXPECT_GT(2, 1);
  EXPECT_NONFATAL_FAILURE(EXPECT_GT(2, 2),
                          "Expected: (2) > (2), actual: 2 vs 2");
  EXPECT_NONFATAL_FAILURE(EXPECT_GT(2, 3),
                          "(2) > (3)");
}

#if GTEST_HAS_EXCEPTIONS

// Tests EXPECT_THROW.
TEST(ExpectTest, EXPECT_THROW) {
  EXPECT_THROW(ThrowAnInteger(), int);
  EXPECT_NONFATAL_FAILURE(EXPECT_THROW(ThrowAnInteger(), bool),
                          "Expected: ThrowAnInteger() throws an exception of "
                          "type bool.\n  Actual: it throws a different type.");
  EXPECT_NONFATAL_FAILURE(
      EXPECT_THROW(ThrowNothing(), bool),
      "Expected: ThrowNothing() throws an exception of type bool.\n"
      "  Actual: it throws nothing.");
}

// Tests EXPECT_NO_THROW.
TEST(ExpectTest, EXPECT_NO_THROW) {
  EXPECT_NO_THROW(ThrowNothing());
  EXPECT_NONFATAL_FAILURE(EXPECT_NO_THROW(ThrowAnInteger()),
                          "Expected: ThrowAnInteger() doesn't throw an "
                          "exception.\n  Actual: it throws.");
}

// Tests EXPECT_ANY_THROW.
TEST(ExpectTest, EXPECT_ANY_THROW) {
  EXPECT_ANY_THROW(ThrowAnInteger());
  EXPECT_NONFATAL_FAILURE(
      EXPECT_ANY_THROW(ThrowNothing()),
      "Expected: ThrowNothing() throws an exception.\n"
      "  Actual: it doesn't.");
}

#endif  // GTEST_HAS_EXCEPTIONS

// Make sure we deal with the precedence of <<.
TEST(ExpectTest, ExpectPrecedence) {
  EXPECT_EQ(1 < 2, true);
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(true, true && false),
                          "Value of: true && false");
}


// Tests the StreamableToString() function.

// Tests using StreamableToString() on a scalar.
TEST(StreamableToStringTest, Scalar) {
  EXPECT_STREQ("5", StreamableToString(5).c_str());
}

// Tests using StreamableToString() on a non-char pointer.
TEST(StreamableToStringTest, Pointer) {
  int n = 0;
  int* p = &n;
  EXPECT_STRNE("(null)", StreamableToString(p).c_str());
}

// Tests using StreamableToString() on a NULL non-char pointer.
TEST(StreamableToStringTest, NullPointer) {
  int* p = NULL;
  EXPECT_STREQ("(null)", StreamableToString(p).c_str());
}

// Tests using StreamableToString() on a C string.
TEST(StreamableToStringTest, CString) {
  EXPECT_STREQ("Foo", StreamableToString("Foo").c_str());
}

// Tests using StreamableToString() on a NULL C string.
TEST(StreamableToStringTest, NullCString) {
  char* p = NULL;
  EXPECT_STREQ("(null)", StreamableToString(p).c_str());
}

// Tests using streamable values as assertion messages.

#if GTEST_HAS_STD_STRING
// Tests using std::string as an assertion message.
TEST(StreamableTest, string) {
  static const std::string str(
      "This failure message is a std::string, and is expected.");
  EXPECT_FATAL_FAILURE(FAIL() << str,
                       str.c_str());
}

// Tests that we can output strings containing embedded NULs.
// Limited to Linux because we can only do this with std::string's.
TEST(StreamableTest, stringWithEmbeddedNUL) {
  static const char char_array_with_nul[] =
      "Here's a NUL\0 and some more string";
  static const std::string string_with_nul(char_array_with_nul,
                                           sizeof(char_array_with_nul)
                                           - 1);  // drops the trailing NUL
  EXPECT_FATAL_FAILURE(FAIL() << string_with_nul,
                       "Here's a NUL\\0 and some more string");
}

#endif  // GTEST_HAS_STD_STRING

// Tests that we can output a NUL char.
TEST(StreamableTest, NULChar) {
  EXPECT_FATAL_FAILURE({  // NOLINT
    FAIL() << "A NUL" << '\0' << " and some more string";
  }, "A NUL\\0 and some more string");
}

// Tests using int as an assertion message.
TEST(StreamableTest, int) {
  EXPECT_FATAL_FAILURE(FAIL() << 900913,
                       "900913");
}

// Tests using NULL char pointer as an assertion message.
//
// In MSVC, streaming a NULL char * causes access violation.  Google Test
// implemented a workaround (substituting "(null)" for NULL).  This
// tests whether the workaround works.
TEST(StreamableTest, NullCharPtr) {
  EXPECT_FATAL_FAILURE(FAIL() << static_cast<const char*>(NULL),
                       "(null)");
}

// Tests that basic IO manipulators (endl, ends, and flush) can be
// streamed to testing::Message.
TEST(StreamableTest, BasicIoManip) {
  EXPECT_FATAL_FAILURE({  // NOLINT
    FAIL() << "Line 1." << std::endl
           << "A NUL char " << std::ends << std::flush << " in line 2.";
  }, "Line 1.\nA NUL char \\0 in line 2.");
}

// Tests the macros that haven't been covered so far.

void AddFailureHelper(bool* aborted) {
  *aborted = true;
  ADD_FAILURE() << "Failure";
  *aborted = false;
}

// Tests ADD_FAILURE.
TEST(MacroTest, ADD_FAILURE) {
  bool aborted = true;
  EXPECT_NONFATAL_FAILURE(AddFailureHelper(&aborted),
                          "Failure");
  EXPECT_FALSE(aborted);
}

// Tests FAIL.
TEST(MacroTest, FAIL) {
  EXPECT_FATAL_FAILURE(FAIL(),
                       "Failed");
  EXPECT_FATAL_FAILURE(FAIL() << "Intentional failure.",
                       "Intentional failure.");
}

// Tests SUCCEED
TEST(MacroTest, SUCCEED) {
  SUCCEED();
  SUCCEED() << "Explicit success.";
}


// Tests for EXPECT_EQ() and ASSERT_EQ().
//
// These tests fail *intentionally*, s.t. the failure messages can be
// generated and tested.
//
// We have different tests for different argument types.

// Tests using bool values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, Bool) {
  EXPECT_EQ(true,  true);
  EXPECT_FATAL_FAILURE(ASSERT_EQ(false, true),
                       "Value of: true");
}

// Tests using int values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, Int) {
  ASSERT_EQ(32, 32);
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(32, 33),
                          "33");
}

// Tests using time_t values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, Time_T) {
  EXPECT_EQ(static_cast<time_t>(0),
            static_cast<time_t>(0));
  EXPECT_FATAL_FAILURE(ASSERT_EQ(static_cast<time_t>(0),
                                 static_cast<time_t>(1234)),
                       "1234");
}

// Tests using char values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, Char) {
  ASSERT_EQ('z', 'z');
  const char ch = 'b';
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ('\0', ch),
                          "ch");
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ('a', ch),
                          "ch");
}

// Tests using wchar_t values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, WideChar) {
  EXPECT_EQ(L'b', L'b');

  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(L'\0', L'x'),
                          "Value of: L'x'\n"
                          "  Actual: L'x' (120, 0x78)\n"
                          "Expected: L'\0'\n"
                          "Which is: L'\0' (0, 0x0)");

  static wchar_t wchar;
  wchar = L'b';
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(L'a', wchar),
                          "wchar");
  wchar = L'\x8119';
  EXPECT_FATAL_FAILURE(ASSERT_EQ(L'\x8120', wchar),
                       "Value of: wchar");
}

#if GTEST_HAS_STD_STRING
// Tests using ::std::string values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, StdString) {
  // Compares a const char* to an std::string that has identical
  // content.
  ASSERT_EQ("Test", ::std::string("Test"));

  // Compares two identical std::strings.
  static const ::std::string str1("A * in the middle");
  static const ::std::string str2(str1);
  EXPECT_EQ(str1, str2);

  // Compares a const char* to an std::string that has different
  // content
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ("Test", ::std::string("test")),
                          "::std::string(\"test\")");

  // Compares an std::string to a char* that has different content.
  char* const p1 = const_cast<char*>("foo");
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(::std::string("bar"), p1),
                          "p1");

  // Compares two std::strings that have different contents, one of
  // which having a NUL character in the middle.  This should fail.
  static ::std::string str3(str1);
  str3.at(2) = '\0';
  EXPECT_FATAL_FAILURE(ASSERT_EQ(str1, str3),
                       "Value of: str3\n"
                       "  Actual: \"A \\0 in the middle\"");
}

#endif  // GTEST_HAS_STD_STRING

#if GTEST_HAS_STD_WSTRING

// Tests using ::std::wstring values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, StdWideString) {
  // Compares an std::wstring to a const wchar_t* that has identical
  // content.
  EXPECT_EQ(::std::wstring(L"Test\x8119"), L"Test\x8119");

  // Compares two identical std::wstrings.
  const ::std::wstring wstr1(L"A * in the middle");
  const ::std::wstring wstr2(wstr1);
  ASSERT_EQ(wstr1, wstr2);

  // Compares an std::wstring to a const wchar_t* that has different
  // content.
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_EQ(::std::wstring(L"Test\x8119"), L"Test\x8120");
  }, "L\"Test\\x8120\"");

  // Compares two std::wstrings that have different contents, one of
  // which having a NUL character in the middle.
  ::std::wstring wstr3(wstr1);
  wstr3.at(2) = L'\0';
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(wstr1, wstr3),
                          "wstr3");

  // Compares a wchar_t* to an std::wstring that has different
  // content.
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_EQ(const_cast<wchar_t*>(L"foo"), ::std::wstring(L"bar"));
  }, "");
}

#endif  // GTEST_HAS_STD_WSTRING

#if GTEST_HAS_GLOBAL_STRING
// Tests using ::string values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, GlobalString) {
  // Compares a const char* to a ::string that has identical content.
  EXPECT_EQ("Test", ::string("Test"));

  // Compares two identical ::strings.
  const ::string str1("A * in the middle");
  const ::string str2(str1);
  ASSERT_EQ(str1, str2);

  // Compares a ::string to a const char* that has different content.
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(::string("Test"), "test"),
                          "test");

  // Compares two ::strings that have different contents, one of which
  // having a NUL character in the middle.
  ::string str3(str1);
  str3.at(2) = '\0';
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(str1, str3),
                          "str3");

  // Compares a ::string to a char* that has different content.
  EXPECT_FATAL_FAILURE({  // NOLINT
    ASSERT_EQ(::string("bar"), const_cast<char*>("foo"));
  }, "");
}

#endif  // GTEST_HAS_GLOBAL_STRING

#if GTEST_HAS_GLOBAL_WSTRING

// Tests using ::wstring values in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, GlobalWideString) {
  // Compares a const wchar_t* to a ::wstring that has identical content.
  ASSERT_EQ(L"Test\x8119", ::wstring(L"Test\x8119"));

  // Compares two identical ::wstrings.
  static const ::wstring wstr1(L"A * in the middle");
  static const ::wstring wstr2(wstr1);
  EXPECT_EQ(wstr1, wstr2);

  // Compares a const wchar_t* to a ::wstring that has different
  // content.
  EXPECT_NONFATAL_FAILURE({  // NOLINT
    EXPECT_EQ(L"Test\x8120", ::wstring(L"Test\x8119"));
  }, "Test\\x8119");

  // Compares a wchar_t* to a ::wstring that has different content.
  wchar_t* const p1 = const_cast<wchar_t*>(L"foo");
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(p1, ::wstring(L"bar")),
                          "bar");

  // Compares two ::wstrings that have different contents, one of which
  // having a NUL character in the middle.
  static ::wstring wstr3;
  wstr3 = wstr1;
  wstr3.at(2) = L'\0';
  EXPECT_FATAL_FAILURE(ASSERT_EQ(wstr1, wstr3),
                       "wstr3");
}

#endif  // GTEST_HAS_GLOBAL_WSTRING

// Tests using char pointers in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, CharPointer) {
  char* const p0 = NULL;
  // Only way to get the Nokia compiler to compile the cast
  // is to have a separate void* variable first. Putting
  // the two casts on the same line doesn't work, neither does
  // a direct C-style to char*.
  void* pv1 = (void*)0x1234;  // NOLINT
  void* pv2 = (void*)0xABC0;  // NOLINT
  char* const p1 = reinterpret_cast<char*>(pv1);
  char* const p2 = reinterpret_cast<char*>(pv2);
  ASSERT_EQ(p1, p1);

  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(p0, p2),
                          "Value of: p2");
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(p1, p2),
                          "p2");
  EXPECT_FATAL_FAILURE(ASSERT_EQ(reinterpret_cast<char*>(0x1234),
                                 reinterpret_cast<char*>(0xABC0)),
                       "ABC0");
}

// Tests using wchar_t pointers in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, WideCharPointer) {
  wchar_t* const p0 = NULL;
  // Only way to get the Nokia compiler to compile the cast
  // is to have a separate void* variable first. Putting
  // the two casts on the same line doesn't work, neither does
  // a direct C-style to char*.
  void* pv1 = (void*)0x1234;  // NOLINT
  void* pv2 = (void*)0xABC0;  // NOLINT
  wchar_t* const p1 = reinterpret_cast<wchar_t*>(pv1);
  wchar_t* const p2 = reinterpret_cast<wchar_t*>(pv2);
  EXPECT_EQ(p0, p0);

  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(p0, p2),
                          "Value of: p2");
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(p1, p2),
                          "p2");
  void* pv3 = (void*)0x1234;  // NOLINT
  void* pv4 = (void*)0xABC0;  // NOLINT
  const wchar_t* p3 = reinterpret_cast<const wchar_t*>(pv3);
  const wchar_t* p4 = reinterpret_cast<const wchar_t*>(pv4);
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(p3, p4),
                          "p4");
}

// Tests using other types of pointers in {EXPECT|ASSERT}_EQ.
TEST(EqAssertionTest, OtherPointer) {
  ASSERT_EQ(static_cast<const int*>(NULL),
            static_cast<const int*>(NULL));
  EXPECT_FATAL_FAILURE(ASSERT_EQ(static_cast<const int*>(NULL),
                                 reinterpret_cast<const int*>(0x1234)),
                       "0x1234");
}

// Tests the FRIEND_TEST macro.

// This class has a private member we want to test.  We will test it
// both in a TEST and in a TEST_F.
class Foo {
 public:
  Foo() {}

 private:
  int Bar() const { return 1; }

  // Declares the friend tests that can access the private member
  // Bar().
  FRIEND_TEST(FRIEND_TEST_Test, TEST);
  FRIEND_TEST(FRIEND_TEST_Test2, TEST_F);
};

// Tests that the FRIEND_TEST declaration allows a TEST to access a
// class's private members.  This should compile.
TEST(FRIEND_TEST_Test, TEST) {
  ASSERT_EQ(1, Foo().Bar());
}

// The fixture needed to test using FRIEND_TEST with TEST_F.
class FRIEND_TEST_Test2 : public Test {
 protected:
  Foo foo;
};

// Tests that the FRIEND_TEST declaration allows a TEST_F to access a
// class's private members.  This should compile.
TEST_F(FRIEND_TEST_Test2, TEST_F) {
  ASSERT_EQ(1, foo.Bar());
}

// Tests the life cycle of Test objects.

// The test fixture for testing the life cycle of Test objects.
//
// This class counts the number of live test objects that uses this
// fixture.
class TestLifeCycleTest : public Test {
 protected:
  // Constructor.  Increments the number of test objects that uses
  // this fixture.
  TestLifeCycleTest() { count_++; }

  // Destructor.  Decrements the number of test objects that uses this
  // fixture.
  ~TestLifeCycleTest() { count_--; }

  // Returns the number of live test objects that uses this fixture.
  int count() const { return count_; }

 private:
  static int count_;
};

int TestLifeCycleTest::count_ = 0;

// Tests the life cycle of test objects.
TEST_F(TestLifeCycleTest, Test1) {
  // There should be only one test object in this test case that's
  // currently alive.
  ASSERT_EQ(1, count());
}

// Tests the life cycle of test objects.
TEST_F(TestLifeCycleTest, Test2) {
  // After Test1 is done and Test2 is started, there should still be
  // only one live test object, as the object for Test1 should've been
  // deleted.
  ASSERT_EQ(1, count());
}

}  // namespace

// Tests streaming a user type whose definition and operator << are
// both in the global namespace.
class Base {
 public:
  explicit Base(int x) : x_(x) {}
  int x() const { return x_; }
 private:
  int x_;
};
std::ostream& operator<<(std::ostream& os,
                         const Base& val) {
  return os << val.x();
}
std::ostream& operator<<(std::ostream& os,
                         const Base* pointer) {
  return os << "(" << pointer->x() << ")";
}

TEST(MessageTest, CanStreamUserTypeInGlobalNameSpace) {
  Message msg;
  Base a(1);

  msg << a << &a;  // Uses ::operator<<.
  EXPECT_STREQ("1(1)", msg.GetString().c_str());
}

// Tests streaming a user type whose definition and operator<< are
// both in an unnamed namespace.
namespace {
class MyTypeInUnnamedNameSpace : public Base {
 public:
  explicit MyTypeInUnnamedNameSpace(int x): Base(x) {}
};
std::ostream& operator<<(std::ostream& os,
                         const MyTypeInUnnamedNameSpace& val) {
  return os << val.x();
}
std::ostream& operator<<(std::ostream& os,
                         const MyTypeInUnnamedNameSpace* pointer) {
  return os << "(" << pointer->x() << ")";
}
}  // namespace

TEST(MessageTest, CanStreamUserTypeInUnnamedNameSpace) {
  Message msg;
  MyTypeInUnnamedNameSpace a(1);

  msg << a << &a;  // Uses <unnamed_namespace>::operator<<.
  EXPECT_STREQ("1(1)", msg.GetString().c_str());
}

// Tests streaming a user type whose definition and operator<< are
// both in a user namespace.
namespace namespace1 {
class MyTypeInNameSpace1 : public Base {
 public:
  explicit MyTypeInNameSpace1(int x): Base(x) {}
};
std::ostream& operator<<(std::ostream& os,
                         const MyTypeInNameSpace1& val) {
  return os << val.x();
}
std::ostream& operator<<(std::ostream& os,
                         const MyTypeInNameSpace1* pointer) {
  return os << "(" << pointer->x() << ")";
}
}  // namespace namespace1

TEST(MessageTest, CanStreamUserTypeInUserNameSpace) {
  Message msg;
  namespace1::MyTypeInNameSpace1 a(1);

  msg << a << &a;  // Uses namespace1::operator<<.
  EXPECT_STREQ("1(1)", msg.GetString().c_str());
}

// Tests streaming a user type whose definition is in a user namespace
// but whose operator<< is in the global namespace.
namespace namespace2 {
class MyTypeInNameSpace2 : public ::Base {
 public:
  explicit MyTypeInNameSpace2(int x): Base(x) {}
};
}  // namespace namespace2
std::ostream& operator<<(std::ostream& os,
                         const namespace2::MyTypeInNameSpace2& val) {
  return os << val.x();
}
std::ostream& operator<<(std::ostream& os,
                         const namespace2::MyTypeInNameSpace2* pointer) {
  return os << "(" << pointer->x() << ")";
}

TEST(MessageTest, CanStreamUserTypeInUserNameSpaceWithStreamOperatorInGlobal) {
  Message msg;
  namespace2::MyTypeInNameSpace2 a(1);

  msg << a << &a;  // Uses ::operator<<.
  EXPECT_STREQ("1(1)", msg.GetString().c_str());
}

// Tests streaming NULL pointers to testing::Message.
TEST(MessageTest, NullPointers) {
  Message msg;
  char* const p1 = NULL;
  unsigned char* const p2 = NULL;
  int* p3 = NULL;
  double* p4 = NULL;
  bool* p5 = NULL;
  Message* p6 = NULL;

  msg << p1 << p2 << p3 << p4 << p5 << p6;
  ASSERT_STREQ("(null)(null)(null)(null)(null)(null)",
               msg.GetString().c_str());
}

// Tests streaming wide strings to testing::Message.
TEST(MessageTest, WideStrings) {
  // Streams a NULL of type const wchar_t*.
  const wchar_t* const_wstr = NULL;
  EXPECT_STREQ("(null)",
               (Message() << const_wstr).GetString().c_str());

  // Streams a NULL of type wchar_t*.
  wchar_t* wstr = NULL;
  EXPECT_STREQ("(null)",
               (Message() << wstr).GetString().c_str());

  // Streams a non-NULL of type const wchar_t*.
  const_wstr = L"abc\x8119";
  EXPECT_STREQ("abc\xe8\x84\x99",
               (Message() << const_wstr).GetString().c_str());

  // Streams a non-NULL of type wchar_t*.
  wstr = const_cast<wchar_t*>(const_wstr);
  EXPECT_STREQ("abc\xe8\x84\x99",
               (Message() << wstr).GetString().c_str());
}


// This line tests that we can define tests in the testing namespace.
namespace testing {

// Tests the TestInfo class.

class TestInfoTest : public Test {
 protected:
  static const TestInfo* GetTestInfo(const char* test_name) {
    const TestCase* const test_case = GetUnitTestImpl()->
        GetTestCase("TestInfoTest", "", NULL, NULL);

    for (int i = 0; i < test_case->total_test_count(); ++i) {
      const TestInfo* const test_info = test_case->GetTestInfo(i);
      if (strcmp(test_name, test_info->name()) == 0)
        return test_info;
    }
    return NULL;
  }

  static const TestResult* GetTestResult(
      const TestInfo* test_info) {
    return test_info->result();
  }
};

// Tests TestInfo::test_case_name() and TestInfo::name().
TEST_F(TestInfoTest, Names) {
  const TestInfo* const test_info = GetTestInfo("Names");

  ASSERT_STREQ("TestInfoTest", test_info->test_case_name());
  ASSERT_STREQ("Names", test_info->name());
}

// Tests TestInfo::result().
TEST_F(TestInfoTest, result) {
  const TestInfo* const test_info = GetTestInfo("result");

  // Initially, there is no TestPartResult for this test.
  ASSERT_EQ(0, GetTestResult(test_info)->total_part_count());

  // After the previous assertion, there is still none.
  ASSERT_EQ(0, GetTestResult(test_info)->total_part_count());
}

// Tests setting up and tearing down a test case.

class SetUpTestCaseTest : public Test {
 protected:
  // This will be called once before the first test in this test case
  // is run.
  static void SetUpTestCase() {
    printf("Setting up the test case . . .\n");

    // Initializes some shared resource.  In this simple example, we
    // just create a C string.  More complex stuff can be done if
    // desired.
    shared_resource_ = "123";

    // Increments the number of test cases that have been set up.
    counter_++;

    // SetUpTestCase() should be called only once.
    EXPECT_EQ(1, counter_);
  }

  // This will be called once after the last test in this test case is
  // run.
  static void TearDownTestCase() {
    printf("Tearing down the test case . . .\n");

    // Decrements the number of test cases that have been set up.
    counter_--;

    // TearDownTestCase() should be called only once.
    EXPECT_EQ(0, counter_);

    // Cleans up the shared resource.
    shared_resource_ = NULL;
  }

  // This will be called before each test in this test case.
  virtual void SetUp() {
    // SetUpTestCase() should be called only once, so counter_ should
    // always be 1.
    EXPECT_EQ(1, counter_);
  }

  // Number of test cases that have been set up.
  static int counter_;

  // Some resource to be shared by all tests in this test case.
  static const char* shared_resource_;
};

int SetUpTestCaseTest::counter_ = 0;
const char* SetUpTestCaseTest::shared_resource_ = NULL;

// A test that uses the shared resource.
TEST_F(SetUpTestCaseTest, Test1) {
  EXPECT_STRNE(NULL, shared_resource_);
}

// Another test that uses the shared resource.
TEST_F(SetUpTestCaseTest, Test2) {
  EXPECT_STREQ("123", shared_resource_);
}

// The InitGoogleTestTest test case tests testing::InitGoogleTest().

// The Flags struct stores a copy of all Google Test flags.
struct Flags {
  // Constructs a Flags struct where each flag has its default value.
  Flags() : also_run_disabled_tests(false),
            break_on_failure(false),
            catch_exceptions(false),
            death_test_use_fork(false),
            filter(""),
            list_tests(false),
            output(""),
            print_time(true),
            random_seed(0),
            repeat(1),
            shuffle(false),
            throw_on_failure(false) {}

  // Factory methods.

  // Creates a Flags struct where the gtest_also_run_disabled_tests flag has
  // the given value.
  static Flags AlsoRunDisabledTests(bool also_run_disabled_tests) {
    Flags flags;
    flags.also_run_disabled_tests = also_run_disabled_tests;
    return flags;
  }

  // Creates a Flags struct where the gtest_break_on_failure flag has
  // the given value.
  static Flags BreakOnFailure(bool break_on_failure) {
    Flags flags;
    flags.break_on_failure = break_on_failure;
    return flags;
  }

  // Creates a Flags struct where the gtest_catch_exceptions flag has
  // the given value.
  static Flags CatchExceptions(bool catch_exceptions) {
    Flags flags;
    flags.catch_exceptions = catch_exceptions;
    return flags;
  }

  // Creates a Flags struct where the gtest_death_test_use_fork flag has
  // the given value.
  static Flags DeathTestUseFork(bool death_test_use_fork) {
    Flags flags;
    flags.death_test_use_fork = death_test_use_fork;
    return flags;
  }

  // Creates a Flags struct where the gtest_filter flag has the given
  // value.
  static Flags Filter(const char* filter) {
    Flags flags;
    flags.filter = filter;
    return flags;
  }

  // Creates a Flags struct where the gtest_list_tests flag has the
  // given value.
  static Flags ListTests(bool list_tests) {
    Flags flags;
    flags.list_tests = list_tests;
    return flags;
  }

  // Creates a Flags struct where the gtest_output flag has the given
  // value.
  static Flags Output(const char* output) {
    Flags flags;
    flags.output = output;
    return flags;
  }

  // Creates a Flags struct where the gtest_print_time flag has the given
  // value.
  static Flags PrintTime(bool print_time) {
    Flags flags;
    flags.print_time = print_time;
    return flags;
  }

  // Creates a Flags struct where the gtest_random_seed flag has
  // the given value.
  static Flags RandomSeed(Int32 random_seed) {
    Flags flags;
    flags.random_seed = random_seed;
    return flags;
  }

  // Creates a Flags struct where the gtest_repeat flag has the given
  // value.
  static Flags Repeat(Int32 repeat) {
    Flags flags;
    flags.repeat = repeat;
    return flags;
  }

  // Creates a Flags struct where the gtest_shuffle flag has
  // the given value.
  static Flags Shuffle(bool shuffle) {
    Flags flags;
    flags.shuffle = shuffle;
    return flags;
  }

  // Creates a Flags struct where the gtest_throw_on_failure flag has
  // the given value.
  static Flags ThrowOnFailure(bool throw_on_failure) {
    Flags flags;
    flags.throw_on_failure = throw_on_failure;
    return flags;
  }

  // These fields store the flag values.
  bool also_run_disabled_tests;
  bool break_on_failure;
  bool catch_exceptions;
  bool death_test_use_fork;
  const char* filter;
  bool list_tests;
  const char* output;
  bool print_time;
  Int32 random_seed;
  Int32 repeat;
  bool shuffle;
  bool throw_on_failure;
};

// Fixture for testing InitGoogleTest().
class InitGoogleTestTest : public Test {
 protected:
  // Clears the flags before each test.
  virtual void SetUp() {
    GTEST_FLAG(also_run_disabled_tests) = false;
    GTEST_FLAG(break_on_failure) = false;
    GTEST_FLAG(catch_exceptions) = false;
    GTEST_FLAG(death_test_use_fork) = false;
    GTEST_FLAG(filter) = "";
    GTEST_FLAG(list_tests) = false;
    GTEST_FLAG(output) = "";
    GTEST_FLAG(print_time) = true;
    GTEST_FLAG(random_seed) = 0;
    GTEST_FLAG(repeat) = 1;
    GTEST_FLAG(shuffle) = false;
    GTEST_FLAG(throw_on_failure) = false;
  }

  // Asserts that two narrow or wide string arrays are equal.
  template <typename CharType>
  static void AssertStringArrayEq(size_t size1, CharType** array1,
                                  size_t size2, CharType** array2) {
    ASSERT_EQ(size1, size2) << " Array sizes different.";

    for (size_t i = 0; i != size1; i++) {
      ASSERT_STREQ(array1[i], array2[i]) << " where i == " << i;
    }
  }

  // Verifies that the flag values match the expected values.
  static void CheckFlags(const Flags& expected) {
    EXPECT_EQ(expected.also_run_disabled_tests,
              GTEST_FLAG(also_run_disabled_tests));
    EXPECT_EQ(expected.break_on_failure, GTEST_FLAG(break_on_failure));
    EXPECT_EQ(expected.catch_exceptions, GTEST_FLAG(catch_exceptions));
    EXPECT_EQ(expected.death_test_use_fork, GTEST_FLAG(death_test_use_fork));
    EXPECT_STREQ(expected.filter, GTEST_FLAG(filter).c_str());
    EXPECT_EQ(expected.list_tests, GTEST_FLAG(list_tests));
    EXPECT_STREQ(expected.output, GTEST_FLAG(output).c_str());
    EXPECT_EQ(expected.print_time, GTEST_FLAG(print_time));
    EXPECT_EQ(expected.random_seed, GTEST_FLAG(random_seed));
    EXPECT_EQ(expected.repeat, GTEST_FLAG(repeat));
    EXPECT_EQ(expected.shuffle, GTEST_FLAG(shuffle));
    EXPECT_EQ(expected.throw_on_failure, GTEST_FLAG(throw_on_failure));
  }

  // Parses a command line (specified by argc1 and argv1), then
  // verifies that the flag values are expected and that the
  // recognized flags are removed from the command line.
  template <typename CharType>
  static void TestParsingFlags(int argc1, const CharType** argv1,
                               int argc2, const CharType** argv2,
                               const Flags& expected) {
    // Parses the command line.
    internal::ParseGoogleTestFlagsOnly(&argc1, const_cast<CharType**>(argv1));

    // Verifies the flag values.
    CheckFlags(expected);

    // Verifies that the recognized flags are removed from the command
    // line.
    AssertStringArrayEq(argc1 + 1, argv1, argc2 + 1, argv2);
  }

  // This macro wraps TestParsingFlags s.t. the user doesn't need
  // to specify the array sizes.
#define GTEST_TEST_PARSING_FLAGS_(argv1, argv2, expected) \
  TestParsingFlags(sizeof(argv1)/sizeof(*argv1) - 1, argv1, \
                   sizeof(argv2)/sizeof(*argv2) - 1, argv2, expected)
};

// Tests parsing an empty command line.
TEST_F(InitGoogleTestTest, Empty) {
  const char* argv[] = {
    NULL
  };

  const char* argv2[] = {
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags());
}

// Tests parsing a command line that has no flag.
TEST_F(InitGoogleTestTest, NoFlag) {
  const char* argv[] = {
    "foo.exe",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags());
}

// Tests parsing a bad --gtest_filter flag.
TEST_F(InitGoogleTestTest, FilterBad) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_filter",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    "--gtest_filter",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Filter(""));
}

// Tests parsing an empty --gtest_filter flag.
TEST_F(InitGoogleTestTest, FilterEmpty) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_filter=",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Filter(""));
}

// Tests parsing a non-empty --gtest_filter flag.
TEST_F(InitGoogleTestTest, FilterNonEmpty) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_filter=abc",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Filter("abc"));
}

// Tests parsing --gtest_break_on_failure.
TEST_F(InitGoogleTestTest, BreakOnFailureWithoutValue) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_break_on_failure",
    NULL
};

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::BreakOnFailure(true));
}

// Tests parsing --gtest_break_on_failure=0.
TEST_F(InitGoogleTestTest, BreakOnFailureFalse_0) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_break_on_failure=0",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::BreakOnFailure(false));
}

// Tests parsing --gtest_break_on_failure=f.
TEST_F(InitGoogleTestTest, BreakOnFailureFalse_f) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_break_on_failure=f",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::BreakOnFailure(false));
}

// Tests parsing --gtest_break_on_failure=F.
TEST_F(InitGoogleTestTest, BreakOnFailureFalse_F) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_break_on_failure=F",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::BreakOnFailure(false));
}

// Tests parsing a --gtest_break_on_failure flag that has a "true"
// definition.
TEST_F(InitGoogleTestTest, BreakOnFailureTrue) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_break_on_failure=1",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::BreakOnFailure(true));
}

// Tests parsing --gtest_catch_exceptions.
TEST_F(InitGoogleTestTest, CatchExceptions) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_catch_exceptions",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::CatchExceptions(true));
}

// Tests parsing --gtest_death_test_use_fork.
TEST_F(InitGoogleTestTest, DeathTestUseFork) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_death_test_use_fork",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::DeathTestUseFork(true));
}

// Tests having the same flag twice with different values.  The
// expected behavior is that the one coming last takes precedence.
TEST_F(InitGoogleTestTest, DuplicatedFlags) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_filter=a",
    "--gtest_filter=b",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Filter("b"));
}

// Tests having an unrecognized flag on the command line.
TEST_F(InitGoogleTestTest, UnrecognizedFlag) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_break_on_failure",
    "bar",  // Unrecognized by Google Test.
    "--gtest_filter=b",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    "bar",
    NULL
  };

  Flags flags;
  flags.break_on_failure = true;
  flags.filter = "b";
  GTEST_TEST_PARSING_FLAGS_(argv, argv2, flags);
}

// Tests having a --gtest_list_tests flag
TEST_F(InitGoogleTestTest, ListTestsFlag) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_list_tests",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ListTests(true));
}

// Tests having a --gtest_list_tests flag with a "true" value
TEST_F(InitGoogleTestTest, ListTestsTrue) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_list_tests=1",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ListTests(true));
}

// Tests having a --gtest_list_tests flag with a "false" value
TEST_F(InitGoogleTestTest, ListTestsFalse) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_list_tests=0",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ListTests(false));
}

// Tests parsing --gtest_list_tests=f.
TEST_F(InitGoogleTestTest, ListTestsFalse_f) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_list_tests=f",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ListTests(false));
}

// Tests parsing --gtest_list_tests=F.
TEST_F(InitGoogleTestTest, ListTestsFalse_F) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_list_tests=F",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ListTests(false));
}

// Tests parsing --gtest_output (invalid).
TEST_F(InitGoogleTestTest, OutputEmpty) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_output",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    "--gtest_output",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags());
}

// Tests parsing --gtest_output=xml
TEST_F(InitGoogleTestTest, OutputXml) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_output=xml",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Output("xml"));
}

// Tests parsing --gtest_output=xml:file
TEST_F(InitGoogleTestTest, OutputXmlFile) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_output=xml:file",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Output("xml:file"));
}

// Tests parsing --gtest_output=xml:directory/path/
TEST_F(InitGoogleTestTest, OutputXmlDirectory) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_output=xml:directory/path/",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Output("xml:directory/path/"));
}

// Tests having a --gtest_print_time flag
TEST_F(InitGoogleTestTest, PrintTimeFlag) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_print_time",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::PrintTime(true));
}

// Tests having a --gtest_print_time flag with a "true" value
TEST_F(InitGoogleTestTest, PrintTimeTrue) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_print_time=1",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::PrintTime(true));
}

// Tests having a --gtest_print_time flag with a "false" value
TEST_F(InitGoogleTestTest, PrintTimeFalse) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_print_time=0",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::PrintTime(false));
}

// Tests parsing --gtest_print_time=f.
TEST_F(InitGoogleTestTest, PrintTimeFalse_f) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_print_time=f",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::PrintTime(false));
}

// Tests parsing --gtest_print_time=F.
TEST_F(InitGoogleTestTest, PrintTimeFalse_F) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_print_time=F",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::PrintTime(false));
}

// Tests parsing --gtest_random_seed=number
TEST_F(InitGoogleTestTest, RandomSeed) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_random_seed=1000",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::RandomSeed(1000));
}

// Tests parsing --gtest_repeat=number
TEST_F(InitGoogleTestTest, Repeat) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_repeat=1000",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Repeat(1000));
}

// Tests having a --gtest_also_run_disabled_tests flag
TEST_F(InitGoogleTestTest, AlsoRunDisabledTestsFlag) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_also_run_disabled_tests",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::AlsoRunDisabledTests(true));
}

// Tests having a --gtest_also_run_disabled_tests flag with a "true" value
TEST_F(InitGoogleTestTest, AlsoRunDisabledTestsTrue) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_also_run_disabled_tests=1",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::AlsoRunDisabledTests(true));
}

// Tests having a --gtest_also_run_disabled_tests flag with a "false" value
TEST_F(InitGoogleTestTest, AlsoRunDisabledTestsFalse) {
    const char* argv[] = {
      "foo.exe",
      "--gtest_also_run_disabled_tests=0",
      NULL
    };

    const char* argv2[] = {
      "foo.exe",
      NULL
    };

    GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::AlsoRunDisabledTests(false));
}

// Tests parsing --gtest_shuffle.
TEST_F(InitGoogleTestTest, ShuffleWithoutValue) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_shuffle",
    NULL
};

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Shuffle(true));
}

// Tests parsing --gtest_shuffle=0.
TEST_F(InitGoogleTestTest, ShuffleFalse_0) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_shuffle=0",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Shuffle(false));
}

// Tests parsing a --gtest_shuffle flag that has a "true"
// definition.
TEST_F(InitGoogleTestTest, ShuffleTrue) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_shuffle=1",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::Shuffle(true));
}

// Tests parsing --gtest_throw_on_failure.
TEST_F(InitGoogleTestTest, ThrowOnFailureWithoutValue) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_throw_on_failure",
    NULL
};

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ThrowOnFailure(true));
}

// Tests parsing --gtest_throw_on_failure=0.
TEST_F(InitGoogleTestTest, ThrowOnFailureFalse_0) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_throw_on_failure=0",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ThrowOnFailure(false));
}

// Tests parsing a --gtest_throw_on_failure flag that has a "true"
// definition.
TEST_F(InitGoogleTestTest, ThrowOnFailureTrue) {
  const char* argv[] = {
    "foo.exe",
    "--gtest_throw_on_failure=1",
    NULL
  };

  const char* argv2[] = {
    "foo.exe",
    NULL
  };

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, Flags::ThrowOnFailure(true));
}

#if GTEST_OS_WINDOWS
// Tests parsing wide strings.
TEST_F(InitGoogleTestTest, WideStrings) {
  const wchar_t* argv[] = {
    L"foo.exe",
    L"--gtest_filter=Foo*",
    L"--gtest_list_tests=1",
    L"--gtest_break_on_failure",
    L"--non_gtest_flag",
    NULL
  };

  const wchar_t* argv2[] = {
    L"foo.exe",
    L"--non_gtest_flag",
    NULL
  };

  Flags expected_flags;
  expected_flags.break_on_failure = true;
  expected_flags.filter = "Foo*";
  expected_flags.list_tests = true;

  GTEST_TEST_PARSING_FLAGS_(argv, argv2, expected_flags);
}
#endif  // GTEST_OS_WINDOWS

// Tests current_test_info() in UnitTest.
class CurrentTestInfoTest : public Test {
 protected:
  // Tests that current_test_info() returns NULL before the first test in
  // the test case is run.
  static void SetUpTestCase() {
    // There should be no tests running at this point.
    const TestInfo* test_info =
      UnitTest::GetInstance()->current_test_info();
    EXPECT_TRUE(test_info == NULL)
        << "There should be no tests running at this point.";
  }

  // Tests that current_test_info() returns NULL after the last test in
  // the test case has run.
  static void TearDownTestCase() {
    const TestInfo* test_info =
      UnitTest::GetInstance()->current_test_info();
    EXPECT_TRUE(test_info == NULL)
        << "There should be no tests running at this point.";
  }
};

// Tests that current_test_info() returns TestInfo for currently running
// test by checking the expected test name against the actual one.
TEST_F(CurrentTestInfoTest, WorksForFirstTestInATestCase) {
  const TestInfo* test_info =
    UnitTest::GetInstance()->current_test_info();
  ASSERT_TRUE(NULL != test_info)
      << "There is a test running so we should have a valid TestInfo.";
  EXPECT_STREQ("CurrentTestInfoTest", test_info->test_case_name())
      << "Expected the name of the currently running test case.";
  EXPECT_STREQ("WorksForFirstTestInATestCase", test_info->name())
      << "Expected the name of the currently running test.";
}

// Tests that current_test_info() returns TestInfo for currently running
// test by checking the expected test name against the actual one.  We
// use this test to see that the TestInfo object actually changed from
// the previous invocation.
TEST_F(CurrentTestInfoTest, WorksForSecondTestInATestCase) {
  const TestInfo* test_info =
    UnitTest::GetInstance()->current_test_info();
  ASSERT_TRUE(NULL != test_info)
      << "There is a test running so we should have a valid TestInfo.";
  EXPECT_STREQ("CurrentTestInfoTest", test_info->test_case_name())
      << "Expected the name of the currently running test case.";
  EXPECT_STREQ("WorksForSecondTestInATestCase", test_info->name())
      << "Expected the name of the currently running test.";
}

}  // namespace testing

// These two lines test that we can define tests in a namespace that
// has the name "testing" and is nested in another namespace.
namespace my_namespace {
namespace testing {

// Makes sure that TEST knows to use ::testing::Test instead of
// ::my_namespace::testing::Test.
class Test {};

// Makes sure that an assertion knows to use ::testing::Message instead of
// ::my_namespace::testing::Message.
class Message {};

// Makes sure that an assertion knows to use
// ::testing::AssertionResult instead of
// ::my_namespace::testing::AssertionResult.
class AssertionResult {};

// Tests that an assertion that should succeed works as expected.
TEST(NestedTestingNamespaceTest, Success) {
  EXPECT_EQ(1, 1) << "This shouldn't fail.";
}

// Tests that an assertion that should fail works as expected.
TEST(NestedTestingNamespaceTest, Failure) {
  EXPECT_FATAL_FAILURE(FAIL() << "This failure is expected.",
                       "This failure is expected.");
}

}  // namespace testing
}  // namespace my_namespace

// Tests that one can call superclass SetUp and TearDown methods--
// that is, that they are not private.
// No tests are based on this fixture; the test "passes" if it compiles
// successfully.
class ProtectedFixtureMethodsTest : public Test {
 protected:
  virtual void SetUp() {
    Test::SetUp();
  }
  virtual void TearDown() {
    Test::TearDown();
  }
};

// StreamingAssertionsTest tests the streaming versions of a representative
// sample of assertions.
TEST(StreamingAssertionsTest, Unconditional) {
  SUCCEED() << "expected success";
  EXPECT_NONFATAL_FAILURE(ADD_FAILURE() << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(FAIL() << "expected failure",
                       "expected failure");
}

#ifdef __BORLANDC__
// Silences warnings: "Condition is always true", "Unreachable code"
#pragma option push -w-ccc -w-rch
#endif

TEST(StreamingAssertionsTest, Truth) {
  EXPECT_TRUE(true) << "unexpected failure";
  ASSERT_TRUE(true) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_TRUE(false) << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_TRUE(false) << "expected failure",
                       "expected failure");
}

TEST(StreamingAssertionsTest, Truth2) {
  EXPECT_FALSE(false) << "unexpected failure";
  ASSERT_FALSE(false) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_FALSE(true) << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_FALSE(true) << "expected failure",
                       "expected failure");
}

#ifdef __BORLANDC__
// Restores warnings after previous "#pragma option push" supressed them
#pragma option pop
#endif

TEST(StreamingAssertionsTest, IntegerEquals) {
  EXPECT_EQ(1, 1) << "unexpected failure";
  ASSERT_EQ(1, 1) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_EQ(1, 2) << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_EQ(1, 2) << "expected failure",
                       "expected failure");
}

TEST(StreamingAssertionsTest, IntegerLessThan) {
  EXPECT_LT(1, 2) << "unexpected failure";
  ASSERT_LT(1, 2) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_LT(2, 1) << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_LT(2, 1) << "expected failure",
                       "expected failure");
}

TEST(StreamingAssertionsTest, StringsEqual) {
  EXPECT_STREQ("foo", "foo") << "unexpected failure";
  ASSERT_STREQ("foo", "foo") << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_STREQ("foo", "bar") << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_STREQ("foo", "bar") << "expected failure",
                       "expected failure");
}

TEST(StreamingAssertionsTest, StringsNotEqual) {
  EXPECT_STRNE("foo", "bar") << "unexpected failure";
  ASSERT_STRNE("foo", "bar") << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_STRNE("foo", "foo") << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_STRNE("foo", "foo") << "expected failure",
                       "expected failure");
}

TEST(StreamingAssertionsTest, StringsEqualIgnoringCase) {
  EXPECT_STRCASEEQ("foo", "FOO") << "unexpected failure";
  ASSERT_STRCASEEQ("foo", "FOO") << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_STRCASEEQ("foo", "bar") << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_STRCASEEQ("foo", "bar") << "expected failure",
                       "expected failure");
}

TEST(StreamingAssertionsTest, StringNotEqualIgnoringCase) {
  EXPECT_STRCASENE("foo", "bar") << "unexpected failure";
  ASSERT_STRCASENE("foo", "bar") << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_STRCASENE("foo", "FOO") << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_STRCASENE("bar", "BAR") << "expected failure",
                       "expected failure");
}

TEST(StreamingAssertionsTest, FloatingPointEquals) {
  EXPECT_FLOAT_EQ(1.0, 1.0) << "unexpected failure";
  ASSERT_FLOAT_EQ(1.0, 1.0) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_FLOAT_EQ(0.0, 1.0) << "expected failure",
                          "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_FLOAT_EQ(0.0, 1.0) << "expected failure",
                       "expected failure");
}

#if GTEST_HAS_EXCEPTIONS

TEST(StreamingAssertionsTest, Throw) {
  EXPECT_THROW(ThrowAnInteger(), int) << "unexpected failure";
  ASSERT_THROW(ThrowAnInteger(), int) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_THROW(ThrowAnInteger(), bool) <<
                          "expected failure", "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_THROW(ThrowAnInteger(), bool) <<
                       "expected failure", "expected failure");
}

TEST(StreamingAssertionsTest, NoThrow) {
  EXPECT_NO_THROW(ThrowNothing()) << "unexpected failure";
  ASSERT_NO_THROW(ThrowNothing()) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_NO_THROW(ThrowAnInteger()) <<
                          "expected failure", "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_NO_THROW(ThrowAnInteger()) <<
                       "expected failure", "expected failure");
}

TEST(StreamingAssertionsTest, AnyThrow) {
  EXPECT_ANY_THROW(ThrowAnInteger()) << "unexpected failure";
  ASSERT_ANY_THROW(ThrowAnInteger()) << "unexpected failure";
  EXPECT_NONFATAL_FAILURE(EXPECT_ANY_THROW(ThrowNothing()) <<
                          "expected failure", "expected failure");
  EXPECT_FATAL_FAILURE(ASSERT_ANY_THROW(ThrowNothing()) <<
                       "expected failure", "expected failure");
}

#endif  // GTEST_HAS_EXCEPTIONS

// Tests that Google Test correctly decides whether to use colors in the output.

TEST(ColoredOutputTest, UsesColorsWhenGTestColorFlagIsYes) {
  GTEST_FLAG(color) = "yes";

  SetEnv("TERM", "xterm");  // TERM supports colors.
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.
  EXPECT_TRUE(ShouldUseColor(false));  // Stdout is not a TTY.

  SetEnv("TERM", "dumb");  // TERM doesn't support colors.
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.
  EXPECT_TRUE(ShouldUseColor(false));  // Stdout is not a TTY.
}

TEST(ColoredOutputTest, UsesColorsWhenGTestColorFlagIsAliasOfYes) {
  SetEnv("TERM", "dumb");  // TERM doesn't support colors.

  GTEST_FLAG(color) = "True";
  EXPECT_TRUE(ShouldUseColor(false));  // Stdout is not a TTY.

  GTEST_FLAG(color) = "t";
  EXPECT_TRUE(ShouldUseColor(false));  // Stdout is not a TTY.

  GTEST_FLAG(color) = "1";
  EXPECT_TRUE(ShouldUseColor(false));  // Stdout is not a TTY.
}

TEST(ColoredOutputTest, UsesNoColorWhenGTestColorFlagIsNo) {
  GTEST_FLAG(color) = "no";

  SetEnv("TERM", "xterm");  // TERM supports colors.
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.
  EXPECT_FALSE(ShouldUseColor(false));  // Stdout is not a TTY.

  SetEnv("TERM", "dumb");  // TERM doesn't support colors.
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.
  EXPECT_FALSE(ShouldUseColor(false));  // Stdout is not a TTY.
}

TEST(ColoredOutputTest, UsesNoColorWhenGTestColorFlagIsInvalid) {
  SetEnv("TERM", "xterm");  // TERM supports colors.

  GTEST_FLAG(color) = "F";
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.

  GTEST_FLAG(color) = "0";
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.

  GTEST_FLAG(color) = "unknown";
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.
}

TEST(ColoredOutputTest, UsesColorsWhenStdoutIsTty) {
  GTEST_FLAG(color) = "auto";

  SetEnv("TERM", "xterm");  // TERM supports colors.
  EXPECT_FALSE(ShouldUseColor(false));  // Stdout is not a TTY.
  EXPECT_TRUE(ShouldUseColor(true));    // Stdout is a TTY.
}

TEST(ColoredOutputTest, UsesColorsWhenTermSupportsColors) {
  GTEST_FLAG(color) = "auto";

#if GTEST_OS_WINDOWS
  // On Windows, we ignore the TERM variable as it's usually not set.

  SetEnv("TERM", "dumb");
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "");
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "xterm");
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.
#else
  // On non-Windows platforms, we rely on TERM to determine if the
  // terminal supports colors.

  SetEnv("TERM", "dumb");  // TERM doesn't support colors.
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "emacs");  // TERM doesn't support colors.
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "vt100");  // TERM doesn't support colors.
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "xterm-mono");  // TERM doesn't support colors.
  EXPECT_FALSE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "xterm");  // TERM supports colors.
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "xterm-color");  // TERM supports colors.
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.

  SetEnv("TERM", "linux");  // TERM supports colors.
  EXPECT_TRUE(ShouldUseColor(true));  // Stdout is a TTY.
#endif  // GTEST_OS_WINDOWS
}

// Verifies that StaticAssertTypeEq works in a namespace scope.

static bool dummy1 = StaticAssertTypeEq<bool, bool>();
static bool dummy2 = StaticAssertTypeEq<const int, const int>();

// Verifies that StaticAssertTypeEq works in a class.

template <typename T>
class StaticAssertTypeEqTestHelper {
 public:
  StaticAssertTypeEqTestHelper() { StaticAssertTypeEq<bool, T>(); }
};

TEST(StaticAssertTypeEqTest, WorksInClass) {
  StaticAssertTypeEqTestHelper<bool>();
}

// Verifies that StaticAssertTypeEq works inside a function.

typedef int IntAlias;

TEST(StaticAssertTypeEqTest, CompilesForEqualTypes) {
  StaticAssertTypeEq<int, IntAlias>();
  StaticAssertTypeEq<int*, IntAlias*>();
}

TEST(ThreadLocalTest, DefaultConstructor) {
  ThreadLocal<int> t1;
  EXPECT_EQ(0, t1.get());

  ThreadLocal<void*> t2;
  EXPECT_TRUE(t2.get() == NULL);
}

TEST(ThreadLocalTest, Init) {
  ThreadLocal<int> t1(123);
  EXPECT_EQ(123, t1.get());

  int i = 0;
  ThreadLocal<int*> t2(&i);
  EXPECT_EQ(&i, t2.get());
}

TEST(GetCurrentOsStackTraceExceptTopTest, ReturnsTheStackTrace) {
  testing::UnitTest* const unit_test = testing::UnitTest::GetInstance();

  // We don't have a stack walker in Google Test yet.
  EXPECT_STREQ("", GetCurrentOsStackTraceExceptTop(unit_test, 0).c_str());
  EXPECT_STREQ("", GetCurrentOsStackTraceExceptTop(unit_test, 1).c_str());
}

TEST(HasNonfatalFailureTest, ReturnsFalseWhenThereIsNoFailure) {
  EXPECT_FALSE(HasNonfatalFailure());
}

static void FailFatally() { FAIL(); }

TEST(HasNonfatalFailureTest, ReturnsFalseWhenThereIsOnlyFatalFailure) {
  FailFatally();
  const bool has_nonfatal_failure = HasNonfatalFailure();
  ClearCurrentTestPartResults();
  EXPECT_FALSE(has_nonfatal_failure);
}

TEST(HasNonfatalFailureTest, ReturnsTrueWhenThereIsNonfatalFailure) {
  ADD_FAILURE();
  const bool has_nonfatal_failure = HasNonfatalFailure();
  ClearCurrentTestPartResults();
  EXPECT_TRUE(has_nonfatal_failure);
}

TEST(HasNonfatalFailureTest, ReturnsTrueWhenThereAreFatalAndNonfatalFailures) {
  FailFatally();
  ADD_FAILURE();
  const bool has_nonfatal_failure = HasNonfatalFailure();
  ClearCurrentTestPartResults();
  EXPECT_TRUE(has_nonfatal_failure);
}

// A wrapper for calling HasNonfatalFailure outside of a test body.
static bool HasNonfatalFailureHelper() {
  return testing::Test::HasNonfatalFailure();
}

TEST(HasNonfatalFailureTest, WorksOutsideOfTestBody) {
  EXPECT_FALSE(HasNonfatalFailureHelper());
}

TEST(HasNonfatalFailureTest, WorksOutsideOfTestBody2) {
  ADD_FAILURE();
  const bool has_nonfatal_failure = HasNonfatalFailureHelper();
  ClearCurrentTestPartResults();
  EXPECT_TRUE(has_nonfatal_failure);
}

TEST(HasFailureTest, ReturnsFalseWhenThereIsNoFailure) {
  EXPECT_FALSE(HasFailure());
}

TEST(HasFailureTest, ReturnsTrueWhenThereIsFatalFailure) {
  FailFatally();
  const bool has_failure = HasFailure();
  ClearCurrentTestPartResults();
  EXPECT_TRUE(has_failure);
}

TEST(HasFailureTest, ReturnsTrueWhenThereIsNonfatalFailure) {
  ADD_FAILURE();
  const bool has_failure = HasFailure();
  ClearCurrentTestPartResults();
  EXPECT_TRUE(has_failure);
}

TEST(HasFailureTest, ReturnsTrueWhenThereAreFatalAndNonfatalFailures) {
  FailFatally();
  ADD_FAILURE();
  const bool has_failure = HasFailure();
  ClearCurrentTestPartResults();
  EXPECT_TRUE(has_failure);
}

// A wrapper for calling HasFailure outside of a test body.
static bool HasFailureHelper() { return testing::Test::HasFailure(); }

TEST(HasFailureTest, WorksOutsideOfTestBody) {
  EXPECT_FALSE(HasFailureHelper());
}

TEST(HasFailureTest, WorksOutsideOfTestBody2) {
  ADD_FAILURE();
  const bool has_failure = HasFailureHelper();
  ClearCurrentTestPartResults();
  EXPECT_TRUE(has_failure);
}

class TestListener : public EmptyTestEventListener {
 public:
  TestListener() : on_start_counter_(NULL), is_destroyed_(NULL) {}
  TestListener(int* on_start_counter, bool* is_destroyed)
      : on_start_counter_(on_start_counter),
        is_destroyed_(is_destroyed) {}

  virtual ~TestListener() {
    if (is_destroyed_)
      *is_destroyed_ = true;
  }

 protected:
  virtual void OnTestProgramStart(const UnitTest& /*unit_test*/) {
    if (on_start_counter_ != NULL)
      (*on_start_counter_)++;
  }

 private:
  int* on_start_counter_;
  bool* is_destroyed_;
};

// Tests the constructor.
TEST(TestEventListenersTest, ConstructionWorks) {
  TestEventListeners listeners;

  EXPECT_TRUE(TestEventListenersAccessor::GetRepeater(&listeners) != NULL);
  EXPECT_TRUE(listeners.default_result_printer() == NULL);
  EXPECT_TRUE(listeners.default_xml_generator() == NULL);
}

// Tests that the TestEventListeners destructor deletes all the listeners it
// owns.
TEST(TestEventListenersTest, DestructionWorks) {
  bool default_result_printer_is_destroyed = false;
  bool default_xml_printer_is_destroyed = false;
  bool extra_listener_is_destroyed = false;
  TestListener* default_result_printer = new TestListener(
      NULL, &default_result_printer_is_destroyed);
  TestListener* default_xml_printer = new TestListener(
      NULL, &default_xml_printer_is_destroyed);
  TestListener* extra_listener = new TestListener(
      NULL, &extra_listener_is_destroyed);

  {
    TestEventListeners listeners;
    TestEventListenersAccessor::SetDefaultResultPrinter(&listeners,
                                                        default_result_printer);
    TestEventListenersAccessor::SetDefaultXmlGenerator(&listeners,
                                                       default_xml_printer);
    listeners.Append(extra_listener);
  }
  EXPECT_TRUE(default_result_printer_is_destroyed);
  EXPECT_TRUE(default_xml_printer_is_destroyed);
  EXPECT_TRUE(extra_listener_is_destroyed);
}

// Tests that a listener Append'ed to a TestEventListeners list starts
// receiving events.
TEST(TestEventListenersTest, Append) {
  int on_start_counter = 0;
  bool is_destroyed = false;
  TestListener* listener = new TestListener(&on_start_counter, &is_destroyed);
  {
    TestEventListeners listeners;
    listeners.Append(listener);
    TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
        *UnitTest::GetInstance());
    EXPECT_EQ(1, on_start_counter);
  }
  EXPECT_TRUE(is_destroyed);
}

// Tests that listeners receive events in the order they were appended to
// the list, except for *End requests, which must be received in the reverse
// order.
class SequenceTestingListener : public EmptyTestEventListener {
 public:
  SequenceTestingListener(Vector<String>* vector, const char* id)
      : vector_(vector), id_(id) {}

 protected:
  virtual void OnTestProgramStart(const UnitTest& /*unit_test*/) {
    vector_->PushBack(GetEventDescription("OnTestProgramStart"));
  }

  virtual void OnTestProgramEnd(const UnitTest& /*unit_test*/) {
    vector_->PushBack(GetEventDescription("OnTestProgramEnd"));
  }

  virtual void OnTestIterationStart(const UnitTest& /*unit_test*/,
                                    int /*iteration*/) {
    vector_->PushBack(GetEventDescription("OnTestIterationStart"));
  }

  virtual void OnTestIterationEnd(const UnitTest& /*unit_test*/,
                                  int /*iteration*/) {
    vector_->PushBack(GetEventDescription("OnTestIterationEnd"));
  }

 private:
  String GetEventDescription(const char* method) {
    Message message;
    message << id_ << "." << method;
    return message.GetString();
  }

  Vector<String>* vector_;
  const char* const id_;

  GTEST_DISALLOW_COPY_AND_ASSIGN_(SequenceTestingListener);
};

TEST(EventListenerTest, AppendKeepsOrder) {
  Vector<String> vec;
  TestEventListeners listeners;
  listeners.Append(new SequenceTestingListener(&vec, "1st"));
  listeners.Append(new SequenceTestingListener(&vec, "2nd"));
  listeners.Append(new SequenceTestingListener(&vec, "3rd"));

  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
      *UnitTest::GetInstance());
  ASSERT_EQ(3, vec.size());
  EXPECT_STREQ("1st.OnTestProgramStart", vec.GetElement(0).c_str());
  EXPECT_STREQ("2nd.OnTestProgramStart", vec.GetElement(1).c_str());
  EXPECT_STREQ("3rd.OnTestProgramStart", vec.GetElement(2).c_str());

  vec.Clear();
  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramEnd(
      *UnitTest::GetInstance());
  ASSERT_EQ(3, vec.size());
  EXPECT_STREQ("3rd.OnTestProgramEnd", vec.GetElement(0).c_str());
  EXPECT_STREQ("2nd.OnTestProgramEnd", vec.GetElement(1).c_str());
  EXPECT_STREQ("1st.OnTestProgramEnd", vec.GetElement(2).c_str());

  vec.Clear();
  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestIterationStart(
      *UnitTest::GetInstance(), 0);
  ASSERT_EQ(3, vec.size());
  EXPECT_STREQ("1st.OnTestIterationStart", vec.GetElement(0).c_str());
  EXPECT_STREQ("2nd.OnTestIterationStart", vec.GetElement(1).c_str());
  EXPECT_STREQ("3rd.OnTestIterationStart", vec.GetElement(2).c_str());

  vec.Clear();
  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestIterationEnd(
      *UnitTest::GetInstance(), 0);
  ASSERT_EQ(3, vec.size());
  EXPECT_STREQ("3rd.OnTestIterationEnd", vec.GetElement(0).c_str());
  EXPECT_STREQ("2nd.OnTestIterationEnd", vec.GetElement(1).c_str());
  EXPECT_STREQ("1st.OnTestIterationEnd", vec.GetElement(2).c_str());
}

// Tests that a listener removed from a TestEventListeners list stops receiving
// events and is not deleted when the list is destroyed.
TEST(TestEventListenersTest, Release) {
  int on_start_counter = 0;
  bool is_destroyed = false;
  // Although Append passes the ownership of this object to the list,
  // the following calls release it, and we need to delete it before the
  // test ends.
  TestListener* listener = new TestListener(&on_start_counter, &is_destroyed);
  {
    TestEventListeners listeners;
    listeners.Append(listener);
    EXPECT_EQ(listener, listeners.Release(listener));
    TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
        *UnitTest::GetInstance());
    EXPECT_TRUE(listeners.Release(listener) == NULL);
  }
  EXPECT_EQ(0, on_start_counter);
  EXPECT_FALSE(is_destroyed);
  delete listener;
}

// Tests that no events are forwarded when event forwarding is disabled.
TEST(EventListenerTest, SuppressEventForwarding) {
  int on_start_counter = 0;
  TestListener* listener = new TestListener(&on_start_counter, NULL);

  TestEventListeners listeners;
  listeners.Append(listener);
  ASSERT_TRUE(TestEventListenersAccessor::EventForwardingEnabled(listeners));
  TestEventListenersAccessor::SuppressEventForwarding(&listeners);
  ASSERT_FALSE(TestEventListenersAccessor::EventForwardingEnabled(listeners));
  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
      *UnitTest::GetInstance());
  EXPECT_EQ(0, on_start_counter);
}

// Tests that events generated by Google Test are not forwarded in
// death test subprocesses.
TEST(EventListenerDeathTest, EventsNotForwardedInDeathTestSubprecesses) {
  EXPECT_DEATH_IF_SUPPORTED({
      GTEST_CHECK_(TestEventListenersAccessor::EventForwardingEnabled(
          *GetUnitTestImpl()->listeners())) << "expected failure";},
      "expected failure");
}

// Tests that a listener installed via SetDefaultResultPrinter() starts
// receiving events and is returned via default_result_printer() and that
// the previous default_result_printer is removed from the list and deleted.
TEST(EventListenerTest, default_result_printer) {
  int on_start_counter = 0;
  bool is_destroyed = false;
  TestListener* listener = new TestListener(&on_start_counter, &is_destroyed);

  TestEventListeners listeners;
  TestEventListenersAccessor::SetDefaultResultPrinter(&listeners, listener);

  EXPECT_EQ(listener, listeners.default_result_printer());

  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
      *UnitTest::GetInstance());

  EXPECT_EQ(1, on_start_counter);

  // Replacing default_result_printer with something else should remove it
  // from the list and destroy it.
  TestEventListenersAccessor::SetDefaultResultPrinter(&listeners, NULL);

  EXPECT_TRUE(listeners.default_result_printer() == NULL);
  EXPECT_TRUE(is_destroyed);

  // After broadcasting an event the counter is still the same, indicating
  // the listener is not in the list anymore.
  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
      *UnitTest::GetInstance());
  EXPECT_EQ(1, on_start_counter);
}

// Tests that the default_result_printer listener stops receiving events
// when removed via Release and that is not owned by the list anymore.
TEST(EventListenerTest, RemovingDefaultResultPrinterWorks) {
  int on_start_counter = 0;
  bool is_destroyed = false;
  // Although Append passes the ownership of this object to the list,
  // the following calls release it, and we need to delete it before the
  // test ends.
  TestListener* listener = new TestListener(&on_start_counter, &is_destroyed);
  {
    TestEventListeners listeners;
    TestEventListenersAccessor::SetDefaultResultPrinter(&listeners, listener);

    EXPECT_EQ(listener, listeners.Release(listener));
    EXPECT_TRUE(listeners.default_result_printer() == NULL);
    EXPECT_FALSE(is_destroyed);

    // Broadcasting events now should not affect default_result_printer.
    TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
        *UnitTest::GetInstance());
    EXPECT_EQ(0, on_start_counter);
  }
  // Destroying the list should not affect the listener now, too.
  EXPECT_FALSE(is_destroyed);
  delete listener;
}

// Tests that a listener installed via SetDefaultXmlGenerator() starts
// receiving events and is returned via default_xml_generator() and that
// the previous default_xml_generator is removed from the list and deleted.
TEST(EventListenerTest, default_xml_generator) {
  int on_start_counter = 0;
  bool is_destroyed = false;
  TestListener* listener = new TestListener(&on_start_counter, &is_destroyed);

  TestEventListeners listeners;
  TestEventListenersAccessor::SetDefaultXmlGenerator(&listeners, listener);

  EXPECT_EQ(listener, listeners.default_xml_generator());

  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
      *UnitTest::GetInstance());

  EXPECT_EQ(1, on_start_counter);

  // Replacing default_xml_generator with something else should remove it
  // from the list and destroy it.
  TestEventListenersAccessor::SetDefaultXmlGenerator(&listeners, NULL);

  EXPECT_TRUE(listeners.default_xml_generator() == NULL);
  EXPECT_TRUE(is_destroyed);

  // After broadcasting an event the counter is still the same, indicating
  // the listener is not in the list anymore.
  TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
      *UnitTest::GetInstance());
  EXPECT_EQ(1, on_start_counter);
}

// Tests that the default_xml_generator listener stops receiving events
// when removed via Release and that is not owned by the list anymore.
TEST(EventListenerTest, RemovingDefaultXmlGeneratorWorks) {
  int on_start_counter = 0;
  bool is_destroyed = false;
  // Although Append passes the ownership of this object to the list,
  // the following calls release it, and we need to delete it before the
  // test ends.
  TestListener* listener = new TestListener(&on_start_counter, &is_destroyed);
  {
    TestEventListeners listeners;
    TestEventListenersAccessor::SetDefaultXmlGenerator(&listeners, listener);

    EXPECT_EQ(listener, listeners.Release(listener));
    EXPECT_TRUE(listeners.default_xml_generator() == NULL);
    EXPECT_FALSE(is_destroyed);

    // Broadcasting events now should not affect default_xml_generator.
    TestEventListenersAccessor::GetRepeater(&listeners)->OnTestProgramStart(
        *UnitTest::GetInstance());
    EXPECT_EQ(0, on_start_counter);
  }
  // Destroying the list should not affect the listener now, too.
  EXPECT_FALSE(is_destroyed);
  delete listener;
}
