// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/gl/init/gl_initializer.h"

#include "base/files/file_path.h"
#include "base/logging.h"
#include "base/native_library.h"
#include "ui/gfx/gfx_utils_oxide.h"
#include "ui/gl/gl_egl_api_implementation.h"
#include "ui/gl/gl_gl_api_implementation.h"
#include "ui/gl/gl_glx_api_implementation.h"
#include "ui/gl/gl_implementation.h"
#include "ui/gl/gl_implementation_osmesa.h"
#include "ui/gl/gl_osmesa_api_implementation.h"
#include "ui/gl/gl_surface_egl.h"
#include "ui/gl/gl_surface_glx.h"
#include "ui/ozone/public/ozone_platform.h"

namespace gl {
namespace init {

namespace {

void GL_BINDING_CALL MarshalClearDepthToClearDepthf(GLclampd depth) {
  glClearDepthf(static_cast<GLclampf>(depth));
}

void GL_BINDING_CALL MarshalDepthRangeToDepthRangef(GLclampd z_near,
                                                    GLclampd z_far) {
  glDepthRangef(static_cast<GLclampf>(z_near), static_cast<GLclampf>(z_far));
}

} // namespace

bool InitializeStaticGLBindings(GLImplementation implementation) {
  DCHECK_EQ(kGLImplementationNone, GetGLImplementation());
  ui::OzonePlatform::InitializeForGPU();

  switch (implementation) {
    case kGLImplementationDesktopGL: {
      base::NativeLibrary library =
          base::LoadNativeLibrary(base::FilePath("libGL.so.1"), nullptr);
      if (!library) {
        return false;
      }

      GLGetProcAddressProc get_proc_address =
          reinterpret_cast<GLGetProcAddressProc>(
            base::GetFunctionPointerFromNativeLibrary(library,
                                                      "glXGetProcAddress"));
      if (!get_proc_address) {
        LOG(ERROR) << "glxGetProcAddress not found.";
        base::UnloadNativeLibrary(library);
        return false;
      }

      SetGLGetProcAddressProc(get_proc_address);
      AddGLNativeLibrary(library);
      SetGLImplementation(kGLImplementationDesktopGL);

      InitializeStaticGLBindingsGL();
      InitializeStaticGLBindingsGLX();
      break;
    }

    case kGLImplementationEGLGLES2: {
      base::NativeLibrary gles_library =
          base::LoadNativeLibrary(base::FilePath("libGLESv2.so.2"), nullptr);
      if (!gles_library) {
        return false;
      }

      base::NativeLibrary egl_library =
          base::LoadNativeLibrary(base::FilePath("libEGL.so.1"), nullptr);
      if (!egl_library) {
        base::UnloadNativeLibrary(gles_library);
        return false;
      }

      GLGetProcAddressProc get_proc_address =
          reinterpret_cast<GLGetProcAddressProc>(
            base::GetFunctionPointerFromNativeLibrary(egl_library,
                                                      "eglGetProcAddress"));
      if (!get_proc_address) {
        LOG(ERROR) << "eglGetProcAddress not found.";
        base::UnloadNativeLibrary(egl_library);
        base::UnloadNativeLibrary(gles_library);
        return false;
      }

      SetGLGetProcAddressProc(get_proc_address);
      AddGLNativeLibrary(egl_library);
      AddGLNativeLibrary(gles_library);
      SetGLImplementation(kGLImplementationEGLGLES2);

      InitializeStaticGLBindingsGL();
      InitializeStaticGLBindingsEGL();

      // These two functions take single precision float rather than double
      // precision float parameters in GLES.
      ::gl::g_driver_gl.fn.glClearDepthFn = MarshalClearDepthToClearDepthf;
      ::gl::g_driver_gl.fn.glDepthRangeFn = MarshalDepthRangeToDepthRangef;
      break;
    }

    case kGLImplementationOSMesaGL:
      return InitializeStaticGLBindingsOSMesaGL();

    case kGLImplementationMockGL:
      SetGLImplementation(kGLImplementationMockGL);
      InitializeStaticGLBindingsGL();
      return true;

    default:
      NOTIMPLEMENTED();
      return false;
  }

  return true;
}
bool InitializeGLOneOffPlatform() {
  switch (GetGLImplementation()) {
    case kGLImplementationDesktopGL: {
      if (!GLSurfaceGLX::InitializeOneOff()) {
        LOG(ERROR) << "GLSurfaceGLX::InitializeOneOff failed.";
        return false;
      }
      break;
    }

    case kGLImplementationEGLGLES2: {
      if (!GLSurfaceEGL::InitializeOneOff(gfx::GetOxideNativeDisplay())) {
        LOG(ERROR) << "GLSurfaceEGL::InitializeOneOff failed.";
        return false;
      }
      break;
    }

    case kGLImplementationOSMesaGL:
    case kGLImplementationMockGL:
      return true;

    default:
      return false;
  }

  return true;
}

void ShutdownGLPlatform() {
  ClearBindingsGL();
  ClearBindingsEGL();
  ClearBindingsGLX();
  ClearBindingsOSMESA();
}

void InitializeDebugGLBindings() {
  InitializeDebugGLBindingsGL();
  InitializeDebugGLBindingsEGL();
  InitializeDebugGLBindingsGLX();
  InitializeDebugGLBindingsOSMESA();
}

} // namespace init
} // namespace gl
