// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

'use strict';

tvcm.require('tcmalloc.heap_instance_track');
tvcm.require('tracing.analysis.object_snapshot_view');
tvcm.require('tracing.analysis.object_instance_view');
tvcm.require('tracing.tracks.container_track');
tvcm.require('tracing.tracks.counter_track');
tvcm.require('tracing.tracks.object_instance_track');
tvcm.require('tracing.tracks.spacing_track');
tvcm.require('tracing.tracks.thread_track');
tvcm.require('tracing.trace_model_settings');
tvcm.require('tracing.filter');
tvcm.require('tvcm.ui');
tvcm.require('tvcm.ui.dom_helpers');

tvcm.requireStylesheet('tracing.tracks.process_track_base');

tvcm.exportTo('tracing.tracks', function() {

  var ObjectSnapshotView = tracing.analysis.ObjectSnapshotView;
  var ObjectInstanceView = tracing.analysis.ObjectInstanceView;
  var TraceModelSettings = tracing.TraceModelSettings;
  var SpacingTrack = tracing.tracks.SpacingTrack;

  /**
   * Visualizes a Process by building ThreadTracks and CounterTracks.
   * @constructor
   */
  var ProcessTrackBase =
      tvcm.ui.define('process-track-base', tracing.tracks.ContainerTrack);

  ProcessTrackBase.prototype = {

    __proto__: tracing.tracks.ContainerTrack.prototype,

    decorate: function(viewport) {
      tracing.tracks.ContainerTrack.prototype.decorate.call(this, viewport);

      this.processBase_ = undefined;

      this.classList.add('process-track-base');
      this.classList.add('expanded');

      this.processNameEl_ = tvcm.ui.createSpan();
      this.processNameEl_.classList.add('process-track-name');

      this.headerEl_ = tvcm.ui.createDiv({className: 'process-track-header'});
      this.headerEl_.appendChild(this.processNameEl_);
      this.headerEl_.addEventListener('click', this.onHeaderClick_.bind(this));

      this.appendChild(this.headerEl_);
    },

    get processBase() {
      return this.processBase_;
    },

    set processBase(processBase) {
      this.processBase_ = processBase;

      if (this.processBase_) {
        var modelSettings = new TraceModelSettings(this.processBase_.model);
        var defaultValue;
        if (this.processBase_.labels !== undefined &&
            this.processBase_.labels.length == 1 &&
            this.processBase_.labels[0] == 'chrome://tracing') {
          defaultValue = false;
        } else {
          defaultValue = true;
        }
        this.expanded = modelSettings.getSettingFor(
            this.processBase_, 'expanded', defaultValue);
      }

      this.updateContents_();
    },

    get expanded() {
      return this.classList.contains('expanded');
    },

    set expanded(expanded) {
      expanded = !!expanded;

      if (this.expanded === expanded)
        return;

      this.classList.toggle('expanded');

      // Expanding and collapsing tracks is, essentially, growing and shrinking
      // the viewport. We dispatch a change event to trigger any processing
      // to happen.
      this.viewport_.dispatchChangeEvent();

      if (!this.processBase_)
        return;

      var modelSettings = new TraceModelSettings(this.processBase_.model);
      modelSettings.setSettingFor(this.processBase_, 'expanded', expanded);
    },

    get hasVisibleContent() {
      if (this.expanded)
        return this.children.length > 1;
      return true;
    },

    onHeaderClick_: function(e) {
      e.stopPropagation();
      e.preventDefault();
      this.expanded = !this.expanded;
    },

    updateContents_: function() {
      this.tracks_.forEach(function(track) {
        this.removeChild(track);
      }, this);

      if (!this.processBase_)
        return;

      this.processNameEl_.textContent = this.processBase_.userFriendlyName;
      this.headerEl_.title = this.processBase_.userFriendlyDetails;

      // Create the object instance tracks for this process.
      this.willAppendTracks_();
      this.appendObjectInstanceTracks_();
      this.appendCounterTracks_();
      this.appendThreadTracks_();
      this.didAppendTracks_();
    },

    addEventsToTrackMap: function(eventToTrackMap) {
      this.tracks_.forEach(function(track) {
        track.addEventsToTrackMap(eventToTrackMap);
      });
    },

    willAppendTracks_: function() {
    },

    didAppendTracks_: function() {
    },

    appendObjectInstanceTracks_: function() {
      var instancesByTypeName =
          this.processBase_.objects.getAllInstancesByTypeName();
      var instanceTypeNames = tvcm.dictionaryKeys(instancesByTypeName);
      instanceTypeNames.sort();

      var didAppendAtLeastOneTrack = false;
      instanceTypeNames.forEach(function(typeName) {
        var allInstances = instancesByTypeName[typeName];

        // If a object snapshot has a view it will be shown,
        // unless the view asked for it to not be shown.
        var instanceViewInfo = ObjectInstanceView.getViewInfo(typeName);
        var snapshotViewInfo = ObjectSnapshotView.getViewInfo(typeName);
        if (instanceViewInfo && !instanceViewInfo.options.showInTrackView)
          instanceViewInfo = undefined;
        if (snapshotViewInfo && !snapshotViewInfo.options.showInTrackView)
          snapshotViewInfo = undefined;
        var hasViewInfo = instanceViewInfo || snapshotViewInfo;

        // There are some instances that don't merit their own track in
        // the UI. Filter them out.
        var visibleInstances = [];
        for (var i = 0; i < allInstances.length; i++) {
          var instance = allInstances[i];

          // Do not create tracks for instances that have no snapshots.
          if (instance.snapshots.length === 0)
            continue;

          // Do not create tracks for instances that have implicit snapshots
          // and don't have a view.
          if (instance.hasImplicitSnapshots && !hasViewInfo)
            continue;

          visibleInstances.push(instance);
        }
        if (visibleInstances.length === 0)
          return;

        // Look up the constructor for this track, or use the default
        // constructor if none exists.
        var trackConstructor =
            tracing.tracks.ObjectInstanceTrack.getTrackConstructor(typeName);
        if (!trackConstructor)
          trackConstructor = tracing.tracks.ObjectInstanceTrack;
        var track = new trackConstructor(this.viewport);
        track.objectInstances = visibleInstances;
        this.appendChild(track);
        didAppendAtLeastOneTrack = true;
      }, this);
      if (didAppendAtLeastOneTrack)
        this.appendChild(new SpacingTrack(this.viewport));
    },

    appendCounterTracks_: function() {
      // Add counter tracks for this process.
      var counters = tvcm.dictionaryValues(this.processBase.counters);
      counters.sort(tracing.trace_model.Counter.compare);

      // Create the counters for this process.
      counters.forEach(function(counter) {
        var track = new tracing.tracks.CounterTrack(this.viewport);
        track.counter = counter;
        this.appendChild(track);
        this.appendChild(new SpacingTrack(this.viewport));
      }.bind(this));
    },

    appendThreadTracks_: function() {
      // Get a sorted list of threads.
      var threads = tvcm.dictionaryValues(this.processBase.threads);
      threads.sort(tracing.trace_model.Thread.compare);

      // Create the threads.
      threads.forEach(function(thread) {
        var track = new tracing.tracks.ThreadTrack(this.viewport);
        track.thread = thread;
        if (!track.hasVisibleContent)
          return;
        this.appendChild(track);
        this.appendChild(new SpacingTrack(this.viewport));
      }.bind(this));
    }
  };

  return {
    ProcessTrackBase: ProcessTrackBase
  };
});
