/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU Affero General Public License as
  published by the Free Software Foundation; either version 3,
  or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not,
  see <http://www.gnu.org/licenses/>
*/

/**
 * @file taler-merchant-httpd_mfa.h
 * @brief internal APIs for multi-factor authentication (MFA)
 * @author Christian Grothoff
 */
#ifndef TALER_MERCHANT_HTTPD_MFA_H
#define TALER_MERCHANT_HTTPD_MFA_H

#include "taler-merchant-httpd.h"
#include "taler_merchant_util.h"


/**
 * Parse the given @a challenge_id into its parts.
 *
 * @param[in,out] hc handler context with the connection to the client
 * @param challenge_id challenge of "$NUMBER-$H_BODY"
 * @param[out] challenge_serial set to $NUMBER
 * @param[out] h_body set to $H_BODY (decoded)
 * @return #GNUNET_OK on success,
 *         #GNUNET_NO if an error message was returned to the client
 *         #GNUNET_SYSERR to just close the connection
 */
enum GNUNET_GenericReturnValue
TMH_mfa_parse_challenge_id (struct TMH_HandlerContext *hc,
                            const char *challenge_id,
                            uint64_t *challenge_serial,
                            struct TALER_MERCHANT_MFA_BodyHash *h_body);


/**
 * Check that a set of MFA challenges has been satisfied by the
 * client for the request in @a hc.
 *
 * @param[in,out] hc handler context with the connection to the client
 * @param op operation for which we should check challenges for
 * @param combi_and true to tell the client to solve all challenges (AND),
 *       false means that any of the challenges will do (OR)
 * @param ... pairs of channel and address, terminated by
 *        #TALER_MERCHANT_MFA_CHANNEL_NONE
 * @return #GNUNET_OK on success (challenges satisfied)
 *         #GNUNET_NO if an error message was returned to the client
 *         #GNUNET_SYSERR to just close the connection
 */
enum GNUNET_GenericReturnValue
TMH_mfa_challenges_do (
  struct TMH_HandlerContext *hc,
  enum TALER_MERCHANT_MFA_CriticalOperation op,
  bool combi_and,
  ...);


/**
 * Check MFA for a simple operation that simply requires
 * a single additional factor (if any are configured).
 *
 * @param[in,out] hc handler context with the connection to the client
 * @param op operation for which we should check challenges for
 * @param mi instance to check authentication for
 * @return #GNUNET_OK on success (challenges satisfied)
 *         #GNUNET_NO if an error message was returned to the client
 *         #GNUNET_SYSERR to just close the connection
 */
enum GNUNET_GenericReturnValue
TMH_mfa_check_simple (
  struct TMH_HandlerContext *hc,
  enum TALER_MERCHANT_MFA_CriticalOperation op,
  struct TMH_MerchantInstance *mi);


#endif
