/*
 * SPDX-FileCopyrightText: Copyright (c) 2017-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifndef _IFRNVL_H_
#define _IFRNVL_H_

//
// NVL - NVLINK Configuration and Error Data
//

// Link Remote Endpoint Type
typedef enum
{
    INFOROM_NVLINK_REMOTE_TYPE_UNSUPPORTED = -1,
    INFOROM_NVLINK_REMOTE_TYPE_NONE        =  0,
    INFOROM_NVLINK_REMOTE_TYPE_PEER        =  1,
    INFOROM_NVLINK_REMOTE_TYPE_SYSTEM      =  2,
    INFOROM_NVLINK_REMOTE_TYPE_SWITCH      =  3,
    INFOROM_NVLINK_REMOTE_TYPE_MAX         =  3,
    INFOROM_NVLINK_NUM_REMOTE_TYPES        =  4
} INFOROM_NVLINK_REMOTE_TYPE;

// Fatal Errors
typedef enum
{
    INFOROM_NVLINK_FATAL_ERR_TYPE_UNSUPPORTED = -1,

    //
    // NVLink 1.0
    //
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_DL_DATA_PARITY       =  0,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_DL_CTRL_PARITY       =  1,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_PROTOCOL             =  2,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_OVERFLOW             =  3,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_RAM_DATA_PARITY      =  4,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_RAM_HDR_PARITY       =  5,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_RESP                 =  6,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_POISON               =  7,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_TX_RAM_DATA_PARITY      =  8,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_TX_RAM_HDR_PARITY       =  9,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_TX_CREDIT               = 10,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_DL_FLOW_CTRL_PARITY     = 11,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TL_DL_HDR_PARITY           = 12,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_RECOVERY_LONG        = 13,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_RAM            = 14,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_INTERFACE      = 15,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_SUBLINK_CHANGE = 16,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_SUBLINK_CHANGE = 17,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_DL_PROTOCOL    = 18,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_LTSSM_FAULT             = 19,
    INFOROM_NVLINK_FATAL_ERR_TYPE_MAX_V1                     = 19,
    INFOROM_NVLINK_NUM_V1_FATAL_ERR_TYPES                    = 20,

    //
    // NVLink 2.0
    //
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_DATA_PARITY                    =  0,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_CTRL_PARITY                    =  1,
    // No direct equivalent to:  _TL_RX_PROTOCOL                           =  2,
    // No direct equivalent to:  _TL_RX_OVERFLOW                           =  3,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RAM_DATA_PARITY                   =  4,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RAM_HDR_PARITY                    =  5,
    // No direct equivalent to:  _TL_RX_RESP                               =  6,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DATA_POISONED_PKT_RCVD            =  7,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RAM_DATA_PARITY                   =  8,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RAM_HDR_PARITY                    =  9,
    // No direct equivalent to:  _TL_TX_CREDIT                             = 10,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DL_FLOW_CONTROL_PARITY            = 11,
    // No direct equivalent to:  _TL_DL_HDR_PARITY                         = 12,
    // Identical to NVLink 1.0:  _DL_TX_RECOVERY_LONG                      = 13,
    // Identical to NVLink 1.0:  _DL_TX_FAULT_RAM                          = 14,
    // Identical to NVLink 1.0:  _DL_TX_FAULT_INTERFACE                    = 15,
    // Identical to NVLink 1.0:  _DL_TX_FAULT_SUBLINK_CHANGE               = 16,
    // Identical to NVLink 1.0:  _DL_RX_FAULT_SUBLINK_CHANGE               = 17,
    // Identical to NVLink 1.0:  _DL_RX_FAULT_DL_PROTOCOL                  = 18,
    // Identical to NVLink 1.0:  _DL_LTSSM_FAULT                           = 19,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_HDR_PARITY                     = 20,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_AE_FLIT_RCVD              = 21,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_BE_FLIT_RCVD              = 22,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_ADDR_ALIGN                = 23,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_PKT_LEN                           = 24,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CMD_ENC                      = 25,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_DAT_LEN_ENC                  = 26,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_ADDR_TYPE                    = 27,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_RSP_STATUS                   = 28,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_PKT_STATUS                   = 29,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_REQ  = 30,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_RESP = 31,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_GT_ATOMIC_REQ_MAX_SIZE    = 32,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_GT_RMW_REQ_MAX_SIZE       = 33,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_LT_ATR_RESP_MIN_SIZE      = 34,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_PO_FOR_CACHE_ATTR         = 35,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_COMPRESSED_RESP           = 36,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RESP_STATUS_TARGET                = 37,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RESP_STATUS_UNSUPPORTED_REQUEST   = 38,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_HDR_OVERFLOW                      = 39,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DATA_OVERFLOW                     = 40,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_STOMPED_PKT_RCVD                  = 41,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_CORRECTABLE_INTERNAL              = 42,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_UNSUPPORTED_VC_OVERFLOW           = 43,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_UNSUPPORTED_NVLINK_CREDIT_RELEASE = 44,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_UNSUPPORTED_NCISOC_CREDIT_RELEASE = 45,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_HDR_CREDIT_OVERFLOW               = 46,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DATA_CREDIT_OVERFLOW              = 47,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DL_REPLAY_CREDIT_OVERFLOW         = 48,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_UNSUPPORTED_VC_OVERFLOW           = 49,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_STOMPED_PKT_SENT                  = 50,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DATA_POISONED_PKT_SENT            = 51,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RESP_STATUS_TARGET                = 52,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RESP_STATUS_UNSUPPORTED_REQUEST   = 53,
    INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_RX_RAM_DATA_PARITY                   = 54,
    INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_RX_RAM_HDR_PARITY                    = 55,
    INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_TX_RAM_DATA_PARITY                   = 56,
    INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_TX_RAM_HDR_PARITY                    = 57,
    INFOROM_NVLINK_FATAL_ERR_TYPE_MAX_V2                                   = 57,
    INFOROM_NVLINK_NUM_V2_FATAL_ERR_TYPES                                  = 58,

    // NVLink 3.0

    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_COLLAPSED_RESPONSE       = 58,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_NCISOC_HDR_ECC_DBE               = 59,
    INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_NCISOC_PARITY                    = 60,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_LTSSM_FAULT_UP                       = 61,
    INFOROM_NVLINK_FATAL_ERR_TYPE_DL_LTSSM_FAULT_DOWN                     = 62,
    INFOROM_NVLINK_FATAL_ERR_TYPE_MAX_V3                                  = 62,
    INFOROM_NVLINK_NUM_V3_FATAL_ERR_TYPES                                 = 63,

    INFOROM_NVLINK_FATAL_ERR_TYPE_MAX  = 62,
    INFOROM_NVLINK_NUM_FATAL_ERR_TYPES = 63,
} INFOROM_NVLINK_FATAL_ERR_TYPE;

typedef enum
{
    // DL RX -----------------------------------------
    // Fatal Counts
    INFOROM_NVLINK_DL_RX_FAULT_DL_PROTOCOL_FATAL = 0,
    INFOROM_NVLINK_DL_RX_FAULT_SUBLINK_CHANGE_FATAL,

    // Correctable Accumulated Counts
    INFOROM_NVLINK_DL_RX_FLIT_CRC_CORR,
    INFOROM_NVLINK_DL_RX_LANE0_CRC_CORR,
    INFOROM_NVLINK_DL_RX_LANE1_CRC_CORR,
    INFOROM_NVLINK_DL_RX_LANE2_CRC_CORR,
    INFOROM_NVLINK_DL_RX_LANE3_CRC_CORR,
    INFOROM_NVLINK_DL_RX_LINK_REPLAY_EVENTS_CORR,

    // DL TX -----------------------------------------
    // Fatal Counts
    INFOROM_NVLINK_DL_TX_FAULT_RAM_FATAL,
    INFOROM_NVLINK_DL_TX_FAULT_INTERFACE_FATAL,
    INFOROM_NVLINK_DL_TX_FAULT_SUBLINK_CHANGE_FATAL,

    // Correctable Accumulated Counts
    INFOROM_NVLINK_DL_TX_LINK_REPLAY_EVENTS_CORR,

    // DL Disassociated ------------------------------
    // Fatal Counts
    INFOROM_NVLINK_DL_LTSSM_FAULT_UP_FATAL,
    INFOROM_NVLINK_DL_LTSSM_FAULT_DOWN_FATAL,

    // Correctable Accumulated Counts
    INFOROM_NVLINK_DL_LINK_RECOVERY_EVENTS_CORR,

    // TLC RX ----------------------------------------
    // Fatal Counts
    INFOROM_NVLINK_TLC_RX_DL_HDR_PARITY_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_DL_DATA_PARITY_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_DL_CTRL_PARITY_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_INVALID_AE_FATAL,
    INFOROM_NVLINK_TLC_RX_INVALID_BE_FATAL,
    INFOROM_NVLINK_TLC_RX_INVALID_ADDR_ALIGN_FATAL,
    INFOROM_NVLINK_TLC_RX_PKTLEN_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_RSVD_PACKET_STATUS_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_RSVD_CACHE_ATTR_PROBE_REQ_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_RSVD_CACHE_ATTR_PROBE_RSP_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_DATLEN_GT_RMW_REQ_MAX_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_DATLEN_LT_ATR_RSP_MIN_ERR_FATAL,
    INFOROM_NVLINK_TLC_RX_INVALID_CR_FATAL,
    INFOROM_NVLINK_TLC_RX_INVALID_COLLAPSED_RESPONSE_FATAL,
    INFOROM_NVLINK_TLC_RX_HDR_OVERFLOW_FATAL,
    INFOROM_NVLINK_TLC_RX_DATA_OVERFLOW_FATAL,
    INFOROM_NVLINK_TLC_RX_STOMP_DETECTED_FATAL,
    INFOROM_NVLINK_TLC_RX_RSVD_CMD_ENC_FATAL,
    INFOROM_NVLINK_TLC_RX_RSVD_DAT_LEN_ENC_FATAL,
    INFOROM_NVLINK_TLC_RX_INVALID_PO_FOR_CACHE_ATTR_FATAL,
    // This is only needed for NCISOC support (not used outside SoC?)
    // TLC_RX_NCISOC_CREDIT_PARITY_ERR_FATAL_COUNT,

    // Non-Fatal Counts
    INFOROM_NVLINK_TLC_RX_RSP_STATUS_HW_ERR_NONFATAL,
    INFOROM_NVLINK_TLC_RX_RSP_STATUS_UR_ERR_NONFATAL,
    INFOROM_NVLINK_TLC_RX_RSP_STATUS_PRIV_ERR_NONFATAL,
    INFOROM_NVLINK_TLC_RX_POISON_NONFATAL,
    INFOROM_NVLINK_TLC_RX_AN1_HEARTBEAT_TIMEOUT_NONFATAL,
    INFOROM_NVLINK_TLC_RX_ILLEGAL_PRI_WRITE_NONFATAL,

    // Correctable Accumulated Counts
    // These should be logged in the ECC object instead
    // TLC_RX_HDR_RAM_ECC_SBE_ACCUM,
    // TLC_RX_DAT0_RAM_ECC_SBE_ACCUM,
    // TLC_RX_DAT1_RAM_ECC_SBE_ACCUM,

    // TLC TX ----------------------------------------
    // Fatal Counts
    INFOROM_NVLINK_TLC_TX_DL_CREDIT_PARITY_ERR_FATAL,
    // These should be logged in the ECC object instead
    // TLC_TX_CREQ_HDR_RAM_ECC_DBE_FATAL_COUNT,
    // TLC_TX_RSP_HDR_RAM_ECC_DBE_FATAL_COUNT,
    // TLC_TX_RSP1_HDR_RAM_ECC_DBE_FATAL_COUNT,
    // TLC_TX_COM_HDR_RAM_ECC_DBE_FATAL_COUNT,
    INFOROM_NVLINK_TLC_TX_NCISOC_HDR_ECC_DBE_FATAL,
    INFOROM_NVLINK_TLC_TX_NCISOC_PARITY_ERR_FATAL,

    // Non-Fatal Counts
    // This is only needed for NCISOC support (not used outside SoC?)
    // TLC_TX_NCISOC_DAT_ECC_DBE_NONFATAL_COUNT,
    INFOROM_NVLINK_TLC_TX_ILLEGAL_PRI_WRITE_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC0_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC1_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC2_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC3_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC4_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC5_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC6_NONFATAL,
    INFOROM_NVLINK_TLC_TX_AN1_TIMEOUT_VC7_NONFATAL,
    INFOROM_NVLINK_TLC_TX_POISON_NONFATAL,
    INFOROM_NVLINK_TLC_TX_RSP_STATUS_HW_ERR_NONFATAL,
    INFOROM_NVLINK_TLC_TX_RSP_STATUS_UR_ERR_NONFATAL,
    INFOROM_NVLINK_TLC_TX_RSP_STATUS_PRIV_ERR_NONFATAL,

    // Correctable Counts
    // These should be logged in the ECC object instead
    // TLC_TX_CREQ_RAMS_ECC_SBE_CORR_ACCUM,
    // TLC_TX_RSP_RAMS_ECC_SBE_CORR_ACCUM,
    // TLC_TX_RSP1_RAMS_ECC_SBE_CORR_ACCUM,
    // TLC_TX_COM_RAMS_ECC_SBE_CORR_ACCUM,
    // This is only needed for NCISOC support (not used outside SoC?)
    // TLC_TX_NCISOC_ECC_LIMIT_ERR_CORR_ACCUM,

    // NVLIPT Disassociated --------------------------
    // Fatal Counts
    INFOROM_NVLINK_NVLIPT_SLEEP_WHILE_ACTIVE_LINK_FATAL,
    INFOROM_NVLINK_NVLIPT_RSTSEQ_PHYCTL_TIMEOUT_FATAL,
    INFOROM_NVLINK_NVLIPT_RSTSEQ_CLKCTL_TIMEOUT_FATAL,
    INFOROM_NVLINK_NVLIPT_CLKCTL_ILLEGAL_REQUEST_FATAL,
    INFOROM_NVLINK_NVLIPT_RSTSEQ_PLL_TIMEOUT_FATAL,
    INFOROM_NVLINK_NVLIPT_RSTSEQ_PHYARB_TIMEOUT_FATAL,

    // Non-Fatal Counts
    INFOROM_NVLINK_NVLIPT_ILLEGAL_LINK_STATE_REQUEST_NONFATAL,
    INFOROM_NVLINK_NVLIPT_FAILED_MINION_REQUEST_NONFATAL,
    INFOROM_NVLINK_NVLIPT_RESERVED_REQUEST_VALUE_NONFATAL,
    INFOROM_NVLINK_NVLIPT_LINK_STATE_WRITE_WHILE_BUSY_NONFATAL,
    INFOROM_NVLINK_NVLIPT_WRITE_TO_LOCKED_SYSTEM_REG_NONFATAL,
    INFOROM_NVLINK_NVLIPT_LINK_STATE_REQUEST_TIMEOUT_NONFATAL,

    // TODO 3014908 log these in the NVL object until we have ECC object support
    INFOROM_NVLINK_TLC_RX_HDR_RAM_ECC_DBE_FATAL,
    INFOROM_NVLINK_TLC_RX_DAT0_RAM_ECC_DBE_FATAL,
    INFOROM_NVLINK_TLC_RX_DAT1_RAM_ECC_DBE_FATAL,
    INFOROM_NVLINK_TLC_TX_CREQ_DAT_RAM_ECC_DBE_NONFATAL,
    INFOROM_NVLINK_TLC_TX_RSP_DAT_RAM_ECC_DBE_NONFATAL,
    INFOROM_NVLINK_TLC_TX_COM_DAT_RAM_ECC_DBE_NONFATAL,
    INFOROM_NVLINK_TLC_TX_RSP1_DAT_RAM_ECC_DBE_FATAL,

    // DL Interrupts -----------------------------------------
    // Fatal Counts
    INFOROM_NVLINK_DL_PHY_A_FATAL,
    INFOROM_NVLINK_DL_RX_CRC_COUNTER_FATAL,
    INFOROM_NVLINK_DL_TX_PL_ERROR_FATAL,
    INFOROM_NVLINK_DL_RX_PL_ERROR_FATAL,

    INFOROM_NVLINK_MAX_ERROR_TYPE
} INFOROM_NVLINK_ERROR_TYPES;

#define INFOROM_NVLINK_V1_FATAL_ERR_TYPES   \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_DL_DATA_PARITY)       | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_DL_CTRL_PARITY)       | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_PROTOCOL)             | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_OVERFLOW)             | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_RAM_DATA_PARITY)      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_RAM_HDR_PARITY)       | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_RESP)                 | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_RX_POISON)               | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_TX_RAM_DATA_PARITY)      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_TX_RAM_HDR_PARITY)       | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_TX_CREDIT)               | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_DL_FLOW_CTRL_PARITY)     | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TL_DL_HDR_PARITY)           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_RECOVERY_LONG)        | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_RAM)            | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_INTERFACE)      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_SUBLINK_CHANGE) | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_SUBLINK_CHANGE) | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_DL_PROTOCOL)    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_LTSSM_FAULT)

#define INFOROM_NVLINK_V2_FATAL_ERR_TYPES   \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_DATA_PARITY)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_CTRL_PARITY)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RAM_DATA_PARITY)                   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RAM_HDR_PARITY)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DATA_POISONED_PKT_RCVD)            | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RAM_DATA_PARITY)                   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RAM_HDR_PARITY)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DL_FLOW_CONTROL_PARITY)            | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_RECOVERY_LONG)                      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_RAM)                          | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_INTERFACE)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_SUBLINK_CHANGE)               | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_SUBLINK_CHANGE)               | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_DL_PROTOCOL)                  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_LTSSM_FAULT)                           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_HDR_PARITY)                     | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_AE_FLIT_RCVD)              | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_BE_FLIT_RCVD)              | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_ADDR_ALIGN)                | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_PKT_LEN)                           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CMD_ENC)                      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_DAT_LEN_ENC)                  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_ADDR_TYPE)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_RSP_STATUS)                   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_PKT_STATUS)                   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_REQ)  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_RESP) | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_GT_ATOMIC_REQ_MAX_SIZE)    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_GT_RMW_REQ_MAX_SIZE)       | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_LT_ATR_RESP_MIN_SIZE)      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_PO_FOR_CACHE_ATTR)         | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_COMPRESSED_RESP)           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RESP_STATUS_TARGET)                | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RESP_STATUS_UNSUPPORTED_REQUEST)   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_HDR_OVERFLOW)                      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DATA_OVERFLOW)                     | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_STOMPED_PKT_RCVD)                  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_CORRECTABLE_INTERNAL)              | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_UNSUPPORTED_VC_OVERFLOW)           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_UNSUPPORTED_NVLINK_CREDIT_RELEASE) | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_UNSUPPORTED_NCISOC_CREDIT_RELEASE) | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_HDR_CREDIT_OVERFLOW)               | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DATA_CREDIT_OVERFLOW)              | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DL_REPLAY_CREDIT_OVERFLOW)         | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_UNSUPPORTED_VC_OVERFLOW)           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_STOMPED_PKT_SENT)                  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DATA_POISONED_PKT_SENT)            | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RESP_STATUS_TARGET)                | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_RESP_STATUS_UNSUPPORTED_REQUEST)   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_RX_RAM_DATA_PARITY)                   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_RX_RAM_HDR_PARITY)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_TX_RAM_DATA_PARITY)                   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_MIF_TX_RAM_HDR_PARITY)

#define INFOROM_NVLINK_V3_FATAL_ERR_TYPES \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_DATA_PARITY)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_CTRL_PARITY)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_DL_FLOW_CONTROL_PARITY)            | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_RAM)                          | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_INTERFACE)                    | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_FAULT_SUBLINK_CHANGE)               | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_SUBLINK_CHANGE)               | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_FAULT_DL_PROTOCOL)                  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DL_HDR_PARITY)                     | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_AE_FLIT_RCVD)              | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_BE_FLIT_RCVD)              | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_ADDR_ALIGN)                | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_PKT_LEN)                           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CMD_ENC)                      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_DAT_LEN_ENC)                  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_PKT_STATUS)                   | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_REQ)  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_RESP) | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_GT_RMW_REQ_MAX_SIZE)       | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DAT_LEN_LT_ATR_RESP_MIN_SIZE)      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_PO_FOR_CACHE_ATTR)         | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_COMPRESSED_RESP)           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_HDR_OVERFLOW)                      | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_DATA_OVERFLOW)                     | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_STOMPED_PKT_RCVD)                  | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_RX_INVALID_COLLAPSED_RESPONSE)        | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_NCISOC_HDR_ECC_DBE)                | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_TLC_TX_NCISOC_PARITY)                     | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_LTSSM_FAULT_UP)                        | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_LTSSM_FAULT_DOWN)

#define INFOROM_NVLINK_V4_FATAL_ERR_TYPES \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_PHY_A)                                 | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_CRC_COUNTER)                        | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_TX_PL_ERROR)                           | \
    NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_DL_RX_PL_ERROR)

#define INFOROM_ADD_NVLINK_FATAL_ERR_TYPE(mask, type) \
    mask |= NVBIT64(INFOROM_NVLINK_FATAL_ERR_TYPE_##type)

typedef struct
{
    NvU32 errorsPerMinute;
    NvU32 timestamp;
} INFOROM_NVLINK_ERROR_RATE;

#define INFOROM_NVLINK_NUM_LINKS    6

typedef union INFOROM_NVL_OBJECT INFOROM_NVL_OBJECT, *PINFOROM_NVL_OBJECT;

// Version 1.0 ----------------------------------------------------------------

// Link Remote Endpoint Type
typedef enum
{
    INFOROM_NVL_OBJECT_V1_REMOTE_TYPE_UNSUPPORTED = -1,
    INFOROM_NVL_OBJECT_V1_REMOTE_TYPE_NONE        =  0,
    INFOROM_NVL_OBJECT_V1_REMOTE_TYPE_PEER        =  1,
    INFOROM_NVL_OBJECT_V1_REMOTE_TYPE_SYSTEM      =  2,
    INFOROM_NVL_OBJECT_V1_REMOTE_TYPE_SWITCH      =  3,
    INFOROM_NVL_OBJECT_V1_REMOTE_TYPE_MAX         =  3,
    INFOROM_NVL_OBJECT_V1_NUM_REMOTE_TYPES        =  4,
} INFOROM_NVL_OBJECT_V1_REMOTE_TYPE;

// Fatal Errors
typedef enum
{
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_UNSUPPORTED                = -1,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_DL_DATA_PARITY       =  0,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_DL_CTRL_PARITY       =  1,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_PROTOCOL             =  2,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_OVERFLOW             =  3,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_RAM_DATA_PARITY      =  4,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_RAM_HDR_PARITY       =  5,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_RESP                 =  6,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_RX_POISON               =  7,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_TX_RAM_DATA_PARITY      =  8,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_TX_RAM_HDR_PARITY       =  9,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_TX_CREDIT               = 10,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_DL_FLOW_CTRL_PARITY     = 11,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_TL_DL_HDR_PARITY           = 12,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_DL_TX_RECOVERY_LONG        = 13,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_DL_TX_FAULT_RAM            = 14,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_DL_TX_FAULT_INTERFACE      = 15,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_DL_TX_FAULT_SUBLINK_CHANGE = 16,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_DL_RX_FAULT_SUBLINK_CHANGE = 17,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_DL_RX_FAULT_DL_PROTOCOL    = 18,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_DL_LTSSM_FAULT             = 19,
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX_MAX                        = 19,
    INFOROM_NVL_OBJECT_V1_NUM_FATAL_ERR_INDICES                      = 20,
} INFOROM_NVL_OBJECT_V1_FATAL_ERR_INDEX;

typedef struct INFOROM_NVL_OBJECT_V1_FATAL_ERR_DATA
{
    inforom_U008 counts[INFOROM_NVL_OBJECT_V1_NUM_FATAL_ERR_INDICES];
    inforom_U008 reserved[25];
} INFOROM_NVL_OBJECT_V1_FATAL_ERR_DATA;

#define INFOROM_NVL_OBJECT_V1_FATAL_ERR_DATA_FMT                          "45b"

// Non-Fatal Errors
typedef struct INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_ENTRY
{
    inforom_U024 errRate;
    inforom_U032 timestamp;
} INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_ENTRY;

#define INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_FMT                         "td"

#define INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_DAY_ENTRIES                    5
#define INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_DAY_INTERVAL          (60*60*24)
#define INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_MONTH_ENTRIES                  3
#define INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_MONTH_INTERVAL     (60*60*24*30)

typedef struct INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_DATA
{
    INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_ENTRY maxNDaysAgo[
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_DAY_ENTRIES];
    INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_ENTRY maxNMonthsAgo[
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_MONTH_ENTRIES];
} INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_DATA;

#define INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_DATA_FMT                            \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_FMT)

typedef struct INFOROM_NVL_OBJECT_V1_ERR_DATA
{
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_DATA    fatal;
    INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_DATA nonfatal;
} INFOROM_NVL_OBJECT_V1_ERR_DATA;

#define INFOROM_NVL_OBJECT_V1_ERR_DATA_FMT         \
    INFOROM_NVL_OBJECT_V1_FATAL_ERR_DATA_FMT       \
    INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_DATA_FMT

#define INFOROM_NVL_OBJECT_V1_NUM_LINKS    6

typedef struct INFOROM_NVL_OBJECT_V1
{
    INFOROM_OBJECT_HEADER_V1_00 header;
    inforom_U004                lastErrorRemoteType[
        INFOROM_NVL_OBJECT_V1_NUM_LINKS];
    inforom_U008                reserved1[1];
    INFOROM_NVL_OBJECT_V1_ERR_DATA errData[INFOROM_NVL_OBJECT_V1_NUM_LINKS];
    inforom_U016                padding[1];
    inforom_U032                reserved2[13];
} INFOROM_NVL_OBJECT_V1;

#define INFOROM_NVL_OBJECT_V1_FMT                               \
    INFOROM_OBJECT_HEADER_V1_00_FMT                             \
    "6n"    /* lastRemoteType */                                \
    "b"     /* reserved1 */                                     \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V1_ERR_DATA_FMT)       \
    INFOROM_FMT_REP02(INFOROM_NVL_OBJECT_V1_ERR_DATA_FMT)       \
    "w"     /* padding */                                       \
    "13d"   /* reserved2 */

#define INFOROM_NVL_OBJECT_V1_PACKED_SIZE  672

// Version 2.0 ----------------------------------------------------------------

// Link Remote Endpoint Type
typedef enum
{
    INFOROM_NVL_OBJECT_V2_REMOTE_TYPE_UNSUPPORTED = -1,
    INFOROM_NVL_OBJECT_V2_REMOTE_TYPE_NONE        =  0,
    INFOROM_NVL_OBJECT_V2_REMOTE_TYPE_PEER        =  1,
    INFOROM_NVL_OBJECT_V2_REMOTE_TYPE_SYSTEM      =  2,
    INFOROM_NVL_OBJECT_V2_REMOTE_TYPE_SWITCH      =  3,
    INFOROM_NVL_OBJECT_V2_REMOTE_TYPE_MAX         =  3,
    INFOROM_NVL_OBJECT_V2_NUM_REMOTE_TYPES        =  4,
} INFOROM_NVL_OBJECT_V2_REMOTE_TYPE;

// Fatal Errors
typedef enum
{
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_UNSUPPORTED                              = -1,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DL_HDR_PARITY                     =  0,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DL_DATA_PARITY                    =  1,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DL_CTRL_PARITY                    =  2,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RAM_DATA_PARITY                   =  3,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RAM_HDR_PARITY                    =  4,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_INVALID_AE_FLIT_RCVD              =  5,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_INVALID_BE_FLIT_RCVD              =  6,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_INVALID_ADDR_ALIGN                =  7,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_PKT_LEN                           =  8,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RSVD_CMD_ENC                      =  9,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RSVD_DAT_LEN_ENC                  = 10,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RSVD_ADDR_TYPE                    = 11,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RSVD_RSP_STATUS                   = 12,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RSVD_PKT_STATUS                   = 13,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_REQ  = 14,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RSVD_CACHE_ATTR_ENC_IN_PROBE_RESP = 15,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DAT_LEN_GT_ATOMIC_REQ_MAX_SIZE    = 16,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DAT_LEN_GT_RMW_REQ_MAX_SIZE       = 17,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DAT_LEN_LT_ATR_RESP_MIN_SIZE      = 18,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_INVALID_PO_FOR_CACHE_ATTR         = 19,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_INVALID_COMPRESSED_RESP           = 20,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RESP_STATUS_TARGET                = 21,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_RESP_STATUS_UNSUPPORTED_REQUEST   = 22,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_HDR_OVERFLOW                      = 23,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DATA_OVERFLOW                     = 24,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_STOMPED_PKT_RCVD                  = 25,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_DATA_POISONED_PKT_RCVD            = 26,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_CORRECTABLE_INTERNAL              = 27,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_UNSUPPORTED_VC_OVERFLOW           = 28,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_UNSUPPORTED_NVLINK_CREDIT_RELEASE = 29,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_RX_UNSUPPORTED_NCISOC_CREDIT_RELEASE = 30,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_HDR_CREDIT_OVERFLOW               = 31,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_DATA_CREDIT_OVERFLOW              = 32,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_DL_REPLAY_CREDIT_OVERFLOW         = 33,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_DL_FLOW_CONTROL_PARITY            = 34,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_RAM_HDR_PARITY                    = 35,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_RAM_DATA_PARITY                   = 36,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_UNSUPPORTED_VC_OVERFLOW           = 37,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_STOMPED_PKT_SENT                  = 38,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_DATA_POISONED_PKT_SENT            = 39,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_RESP_STATUS_TARGET                = 40,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_TLC_TX_RESP_STATUS_UNSUPPORTED_REQUEST   = 41,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_MIF_RX_RAM_DATA_PARITY                   = 42,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_MIF_RX_RAM_HDR_PARITY                    = 43,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_MIF_TX_RAM_DATA_PARITY                   = 44,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_MIF_TX_RAM_HDR_PARITY                    = 45,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_DL_TX_RECOVERY_LONG                      = 46,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_DL_TX_FAULT_RAM                          = 47,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_DL_TX_FAULT_INTERFACE                    = 48,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_DL_TX_FAULT_SUBLINK_CHANGE               = 49,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_DL_RX_FAULT_SUBLINK_CHANGE               = 50,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_DL_RX_FAULT_DL_PROTOCOL                  = 51,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_DL_LTSSM_FAULT                           = 52,
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX_MAX                                      = 52,
    INFOROM_NVL_OBJECT_V2_NUM_FATAL_ERR_INDICES                                    = 53,
} INFOROM_NVL_OBJECT_V2_FATAL_ERR_INDEX;

typedef struct INFOROM_NVL_OBJECT_V2_FATAL_ERR_DATA
{
    inforom_U008 counts[INFOROM_NVL_OBJECT_V2_NUM_FATAL_ERR_INDICES];
} INFOROM_NVL_OBJECT_V2_FATAL_ERR_DATA;

#define INFOROM_NVL_OBJECT_V2_FATAL_ERR_DATA_FMT                          "53b"

// Non-Fatal Errors
typedef INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_ENTRY
        INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_RATE_ENTRY;

#define INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_RATE_FMT \
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_FMT

#define INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_RATE_DAY_ENTRIES \
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_DAY_ENTRIES
#define INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_RATE_DAY_INTERVAL \
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_DAY_INTERVAL
#define INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_RATE_MONTH_ENTRIES \
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_MONTH_ENTRIES
#define INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_RATE_MONTH_INTERVAL \
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_RATE_MONTH_INTERVAL

typedef INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_DATA
        INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_DATA;

#define INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_DATA_FMT \
        INFOROM_NVL_OBJECT_V1_NONFATAL_ERR_DATA_FMT

typedef struct INFOROM_NVL_OBJECT_V2_ERR_DATA
{
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_DATA    fatal;
    INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_DATA nonfatal;
} INFOROM_NVL_OBJECT_V2_ERR_DATA;

#define INFOROM_NVL_OBJECT_V2_ERR_DATA_FMT         \
    INFOROM_NVL_OBJECT_V2_FATAL_ERR_DATA_FMT       \
    INFOROM_NVL_OBJECT_V2_NONFATAL_ERR_DATA_FMT

#define INFOROM_NVL_OBJECT_V2_NUM_LINKS     INFOROM_NVL_OBJECT_V1_NUM_LINKS

typedef struct INFOROM_NVL_OBJECT_V2
{
    INFOROM_OBJECT_HEADER_V1_00 header;
    inforom_U004                lastErrorRemoteType[
        INFOROM_NVL_OBJECT_V2_NUM_LINKS];
    inforom_U008                reserved1[1];
    INFOROM_NVL_OBJECT_V2_ERR_DATA errData[INFOROM_NVL_OBJECT_V2_NUM_LINKS];
    inforom_U016                padding[1];
    inforom_U032                reserved2[1];
} INFOROM_NVL_OBJECT_V2;

#define INFOROM_NVL_OBJECT_V2_FMT                               \
    INFOROM_OBJECT_HEADER_V1_00_FMT                             \
    "6n"    /* lastRemoteType */                                \
    "b"     /* reserved1 */                                     \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V2_ERR_DATA_FMT)       \
    INFOROM_FMT_REP02(INFOROM_NVL_OBJECT_V2_ERR_DATA_FMT)       \
    "w"     /* padding */                                       \
    "d"     /* reserved2 */

#define INFOROM_NVL_OBJECT_V2_PACKED_SIZE  672

// Version 3.G0 ---------------------------------------------------------------

#define INFOROM_NVL_OBJECT_V3G_NUM_LINKS 12
#define INFOROM_NVL_OBJECT_V3G_MAX_SEED_BUFFER_SIZE 7

typedef INFOROM_NVL_OBJECT_V2_REMOTE_TYPE INFOROM_NVL_OBJECT_V3_REMOTE_TYPE;

typedef struct INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO
{
    inforom_U032  lastUpdated;
    NvU64_ALIGN32 remoteDeviceSID;
    inforom_U008  remoteDeviceType;
    inforom_U008  remoteDeviceLink;
} INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO;

#define INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT "dq2b"

typedef struct INFOROM_NVL_OBJECT_V3_MINION_DATA
{
    inforom_U032 lastUpdated;
    inforom_U032 data[INFOROM_NVL_OBJECT_V3G_MAX_SEED_BUFFER_SIZE];
} INFOROM_NVL_OBJECT_V3_MINION_DATA;

#define INFOROM_NVL_OBJECT_V3_MINION_DATA_FMT "8d"

typedef struct INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE
{
    inforom_U032 lastUpdated;
    inforom_U032 flitCrcErrorsPerMinute;
    inforom_U016 laneCrcErrorsPerMinute[4];
} INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE;

#define INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT "2d4w"

typedef INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE
        INFOROM_NVL_OBJECT_V3G_CORRECTABLE_ERROR_RATE_ARRAY[INFOROM_NVL_OBJECT_V3G_NUM_LINKS];

#define INFOROM_NVL_OBJECT_V3G_CORRECTABLE_ERROR_RATE_ARRAY_FMT         \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT) \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT)

typedef struct INFOROM_NVL_OBJECT_V3G_MAX_CORRECTABLE_ERROR_RATES
{
    INFOROM_NVL_OBJECT_V3G_CORRECTABLE_ERROR_RATE_ARRAY dailyMaxCorrectableErrorRates[5];
    INFOROM_NVL_OBJECT_V3G_CORRECTABLE_ERROR_RATE_ARRAY monthlyMaxCorrectableErrorRates[3];
} INFOROM_NVL_OBJECT_V3G_MAX_CORRECTABLE_ERROR_RATES;

#define INFOROM_NVL_OBJECT_V3G_MAX_CORRECTABLE_ERROR_RATES_FMT                  \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V3G_CORRECTABLE_ERROR_RATE_ARRAY_FMT)

#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_NVLIPT_INSTANCE_ID              3:0
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID                        7:4
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL0                      0
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL1                      1
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL2                      2
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL3                      3
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL4                      4
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL5                      5
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL(n)     \
    (NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_DL0+(n))
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC0                     6
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC1                     7
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC2                     8
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC3                     9
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC4                    10
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC5                    11
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC(n)    \
    (NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_TLC0+(n))
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_MINION                  12
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_BLOCK_ID_NVLIPT                  13
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_DIRECTION                       9:8
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_DIRECTION_NA                      1
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_DIRECTION_RX                      2
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_DIRECTION_TX                      3
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_SEVERITY                      11:10
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_SEVERITY_CORRECTABLE              1
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_SEVERITY_UNCORRECTABLE_NONFATAL   2
#define NV_INFOROM_NVL_OBJECT_V3_ERROR_METADATA_SEVERITY_UNCORRECTABLE_FATAL      3

#define INFOROM_NVL_OBJECT_V3_NVLIPT_INSTANCE_MAX 15
#define INFOROM_NVL_OBJECT_V3_BLOCK_ID_MAX        15

#define NV_INFOROM_NVL_OBJECT_V3_NVLIPT_ERROR_SUBTYPE        4:0
#define NV_INFOROM_NVL_OBJECT_V3_NVLIPT_ERROR_LINK_ID        7:5
#define NV_INFOROM_NVL_OBJECT_V3_NVLIPT_ERROR_LINK_ID_COMMON   7

// Format is <BLOCK>_<DIRECTION>_<SUBTYPE>_<SEVERITY>_<TYPE>
// Each <BLOCK>_<DIRECTION> has its own subtype space.
typedef enum
{
    // DL RX -----------------------------------------
    // Fatal Counts
    DL_RX_FAULT_DL_PROTOCOL_FATAL_COUNT = 0,
    DL_RX_FAULT_SUBLINK_CHANGE_FATAL_COUNT,

    // Correctable Accumulated Counts
    DL_RX_FLIT_CRC_CORR_ACCUM,
    DL_RX_LANE0_CRC_CORR_ACCUM,
    DL_RX_LANE1_CRC_CORR_ACCUM,
    DL_RX_LANE2_CRC_CORR_ACCUM,
    DL_RX_LANE3_CRC_CORR_ACCUM,
    DL_RX_LINK_REPLAY_EVENTS_CORR_ACCUM,

    // DL TX -----------------------------------------
    // Fatal Counts
    DL_TX_FAULT_RAM_FATAL_COUNT = 0,
    DL_TX_FAULT_INTERFACE_FATAL_COUNT,
    DL_TX_FAULT_SUBLINK_CHANGE_FATAL_COUNT,

    // Correctable Accumulated Counts
    DL_TX_LINK_REPLAY_EVENTS_CORR_ACCUM,

    // DL Disassociated ------------------------------
    // Fatal Counts
    DL_NA_LTSSM_FAULT_UP_FATAL_COUNT = 0,
    DL_NA_LTSSM_FAULT_DOWN_FATAL_COUNT,

    // Correctable Accumulated Counts
    DL_NA_LINK_RECOVERY_EVENTS_CORR_ACCUM,

    DL_NA_PHY_A_FATAL_COUNT,
    DL_RX_CRC_COUNTER_FATAL_COUNT,
    DL_TX_PL_ERROR_FATAL_COUNT,
    DL_RX_PL_ERROR_FATAL_COUNT,

    // TLC RX ----------------------------------------
    // Fatal Counts
    TLC_RX_DL_HDR_PARITY_ERR_FATAL_COUNT = 0,
    TLC_RX_DL_DATA_PARITY_ERR_FATAL_COUNT,
    TLC_RX_DL_CTRL_PARITY_ERR_FATAL_COUNT,
    TLC_RX_INVALID_AE_FATAL_COUNT,
    TLC_RX_INVALID_BE_FATAL_COUNT,
    TLC_RX_INVALID_ADDR_ALIGN_FATAL_COUNT,
    TLC_RX_PKTLEN_ERR_FATAL_COUNT,
    TLC_RX_RSVD_PACKET_STATUS_ERR_FATAL_COUNT,
    TLC_RX_RSVD_CACHE_ATTR_PROBE_REQ_ERR_FATAL_COUNT,
    TLC_RX_RSVD_CACHE_ATTR_PROBE_RSP_ERR_FATAL_COUNT,
    TLC_RX_DATLEN_GT_RMW_REQ_MAX_ERR_FATAL_COUNT,
    TLC_RX_DATLEN_LT_ATR_RSP_MIN_ERR_FATAL_COUNT,
    TLC_RX_INVALID_CR_FATAL_COUNT,
    TLC_RX_INVALID_COLLAPSED_RESPONSE_FATAL_COUNT,
    TLC_RX_HDR_OVERFLOW_FATAL_COUNT,
    TLC_RX_DATA_OVERFLOW_FATAL_COUNT,
    TLC_RX_STOMP_DETECTED_FATAL_COUNT,
    TLC_RX_RSVD_CMD_ENC_FATAL_COUNT,
    TLC_RX_RSVD_DAT_LEN_ENC_FATAL_COUNT,
    TLC_RX_INVALID_PO_FOR_CACHE_ATTR_FATAL_COUNT,
    // This is only needed for NCISOC support (not used outside SoC?)
    // TLC_RX_NCISOC_CREDIT_PARITY_ERR_FATAL_COUNT,

    // Non-Fatal Counts
    TLC_RX_RSP_STATUS_HW_ERR_NONFATAL_COUNT,
    TLC_RX_RSP_STATUS_UR_ERR_NONFATAL_COUNT,
    TLC_RX_RSP_STATUS_PRIV_ERR_NONFATAL_COUNT,
    TLC_RX_POISON_NONFATAL_COUNT,
    TLC_RX_AN1_HEARTBEAT_TIMEOUT_NONFATAL_COUNT,
    TLC_RX_ILLEGAL_PRI_WRITE_NONFATAL_COUNT,

    // Fatal Counts addendum
    TLC_RX_HDR_RAM_ECC_DBE_FATAL_COUNT,
    TLC_RX_DAT0_RAM_ECC_DBE_FATAL_COUNT,
    TLC_RX_DAT1_RAM_ECC_DBE_FATAL_COUNT,

    // Correctable Accumulated Counts
    // These should be logged in the ECC object instead
    // TLC_RX_HDR_RAM_ECC_SBE_ACCUM,
    // TLC_RX_DAT0_RAM_ECC_SBE_ACCUM,
    // TLC_RX_DAT1_RAM_ECC_SBE_ACCUM,

    // TLC TX ----------------------------------------
    // Fatal Counts
    TLC_TX_DL_CREDIT_PARITY_ERR_FATAL_COUNT = 0,
    // These should be logged in the ECC object instead
    // TLC_TX_CREQ_HDR_RAM_ECC_DBE_FATAL_COUNT,
    // TLC_TX_RSP_HDR_RAM_ECC_DBE_FATAL_COUNT,
    // TLC_TX_RSP1_HDR_RAM_ECC_DBE_FATAL_COUNT,
    // TLC_TX_COM_HDR_RAM_ECC_DBE_FATAL_COUNT,
    TLC_TX_NCISOC_HDR_ECC_DBE_FATAL_COUNT,
    TLC_TX_NCISOC_PARITY_ERR_FATAL_COUNT,

    // Non-Fatal Counts
    // This is only needed for NCISOC support (not used outside SoC?)
    // TLC_TX_NCISOC_DAT_ECC_DBE_NONFATAL_COUNT,
    TLC_TX_ILLEGAL_PRI_WRITE_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC0_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC1_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC2_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC3_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC4_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC5_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC6_NONFATAL_COUNT,
    TLC_TX_AN1_TIMEOUT_VC7_NONFATAL_COUNT,
    TLC_TX_POISON_NONFATAL_COUNT,
    TLC_TX_RSP_STATUS_HW_ERR_NONFATAL_COUNT,
    TLC_TX_RSP_STATUS_UR_ERR_NONFATAL_COUNT,
    TLC_TX_RSP_STATUS_PRIV_ERR_NONFATAL_COUNT,
    TLC_TX_CREQ_DAT_RAM_ECC_DBE_NONFATAL_COUNT,
    TLC_TX_RSP_DAT_RAM_ECC_DBE_NONFATAL_COUNT,
    TLC_TX_COM_DAT_RAM_ECC_DBE_NONFATAL_COUNT,

    // Correctable Counts
    // These should be logged in the ECC object instead
    // TLC_TX_CREQ_RAMS_ECC_SBE_CORR_ACCUM,
    // TLC_TX_RSP_RAMS_ECC_SBE_CORR_ACCUM,
    // TLC_TX_RSP1_RAMS_ECC_SBE_CORR_ACCUM,
    // TLC_TX_COM_RAMS_ECC_SBE_CORR_ACCUM,
    // This is only needed for NCISOC support (not used outside SoC?)
    // TLC_TX_NCISOC_ECC_LIMIT_ERR_CORR_ACCUM,

    // TLC TX Fatal addendum
    TLC_TX_RSP1_DAT_RAM_ECC_DBE_FATAL_COUNT,

    // NVLIPT Disassociated --------------------------
    // Fatal Counts
    NVLIPT_NA_SLEEP_WHILE_ACTIVE_LINK_FATAL_COUNT = 0,
    NVLIPT_NA_RSTSEQ_PHYCTL_TIMEOUT_FATAL_COUNT,
    NVLIPT_NA_RSTSEQ_CLKCTL_TIMEOUT_FATAL_COUNT,
    NVLIPT_NA_CLKCTL_ILLEGAL_REQUEST_FATAL_COUNT,
    NVLIPT_NA_RSTSEQ_PLL_TIMEOUT_FATAL_COUNT,
    NVLIPT_NA_RSTSEQ_PHYARB_TIMEOUT_FATAL_COUNT,

    // Non-Fatal Counts
    NVLIPT_NA_ILLEGAL_LINK_STATE_REQUEST_NONFATAL_COUNT,
    NVLIPT_NA_FAILED_MINION_REQUEST_NONFATAL_COUNT,
    NVLIPT_NA_RESERVED_REQUEST_VALUE_NONFATAL_COUNT,
    NVLIPT_NA_LINK_STATE_WRITE_WHILE_BUSY_NONFATAL_COUNT,
    NVLIPT_NA_WRITE_TO_LOCKED_SYSTEM_REG_NONFATAL_COUNT,
    NVLIPT_NA_LINK_STATE_REQUEST_TIMEOUT_NONFATAL_COUNT
} INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_SUBTYPE;

#define INFOROM_NVL_OBJECT_V3_FATAL_ERR_INDEX_UNSUPPORTED ((NvU8) NV_U8_MAX)

typedef enum
{
    INFOROM_NVL_ERROR_TYPE_INVALID = 0,
    INFOROM_NVL_ERROR_TYPE_COUNT,
    INFOROM_NVL_ERROR_TYPE_ACCUM
} INFOROM_NVL_OBJECT_V3_ERROR_TYPE;

typedef enum
{
    INFOROM_NVL_ERROR_BLOCK_TYPE_DL = 0,
    INFOROM_NVL_ERROR_BLOCK_TYPE_TLC,
    INFOROM_NVL_ERROR_BLOCK_TYPE_NVLIPT
} INFOROM_NVL_ERROR_BLOCK_TYPE;

typedef struct
{
    NvU8                       nvliptInstance;
    NvU8                       localLinkIdx;
    INFOROM_NVLINK_ERROR_TYPES error;
    NvU32                      count; // count is used only for correctable errors
} INFOROM_NVLINK_ERROR_EVENT;

#define INFOROM_NVL_OBJECT_MAX_SUBLINK_WIDTH 8
typedef struct
{
    NvU16 flitCrc;
    NvU8  laneCrc[INFOROM_NVL_OBJECT_MAX_SUBLINK_WIDTH];
    NvU16 rxLinkReplay;
    NvU16 txLinkReplay;
    NvU16 linkRecovery;
} INFOROM_NVLINK_CORRECTABLE_ERROR_COUNTS;

typedef struct INFOROM_NVL_OBJECT_V3_ERROR_ENTRY
{
    inforom_U008 header;
    inforom_U016 metadata;
    inforom_U008 errorSubtype;
    union
    {
        struct
        {
            inforom_U032 lastError;
            inforom_U032 avgEventDelta;
            inforom_U032 totalCount;
        } event;
        struct
        {
            inforom_U032  lastUpdated;
            NvU64_ALIGN32 totalCount;
        } accum;
    } data;
} INFOROM_NVL_OBJECT_V3_ERROR_ENTRY;

#define INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT "bwb3d"

#define INFOROM_NVL_OBJECT_V3G_NUM_ERROR_ENTRIES 240

typedef struct INFOROM_NVL_OBJECT_V3G
{
    INFOROM_OBJECT_HEADER_V1_00 header;
    INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO remoteDeviceInfo[INFOROM_NVL_OBJECT_V3G_NUM_LINKS];
    INFOROM_NVL_OBJECT_V3_MINION_DATA minionData[INFOROM_NVL_OBJECT_V3G_NUM_LINKS];
    INFOROM_NVL_OBJECT_V3G_MAX_CORRECTABLE_ERROR_RATES maxCorrectableErrorRates;
    INFOROM_NVL_OBJECT_V3_ERROR_ENTRY errorLog[INFOROM_NVL_OBJECT_V3G_NUM_ERROR_ENTRIES];
} INFOROM_NVL_OBJECT_V3G;

#define INFOROM_NVL_OBJECT_V3G_REMOTE_DEVICE_INFO_ARRAY_FMT         \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT) \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT)

#define INFOROM_NVL_OBJECT_V3G_MINION_DATA_ARRAY_FMT                \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V3_MINION_DATA_FMT)        \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_MINION_DATA_FMT)

#define INFOROM_NVL_OBJECT_V3G_ERROR_LOG_FMT                        \
    INFOROM_FMT_REP128(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP64(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)        \
    INFOROM_FMT_REP32(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)        \
    INFOROM_FMT_REP16(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)

#define INFOROM_NVL_OBJECT_V3G_FMT                                  \
    INFOROM_OBJECT_HEADER_V1_00_FMT                                 \
    INFOROM_NVL_OBJECT_V3G_REMOTE_DEVICE_INFO_ARRAY_FMT             \
    INFOROM_NVL_OBJECT_V3G_MINION_DATA_ARRAY_FMT                    \
    INFOROM_NVL_OBJECT_V3G_MAX_CORRECTABLE_ERROR_RATES_FMT          \
    INFOROM_NVL_OBJECT_V3G_ERROR_LOG_FMT

#define INFOROM_NVL_OBJECT_V3G_PACKED_SIZE 5936

// Version 3.S0 ---------------------------------------------------------------
#define INFOROM_NVL_OBJECT_V3S_NUM_LINKS 36
#define INFOROM_NVL_OBJECT_V3S_NUM_ERROR_ENTRIES 721

typedef INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE
        INFOROM_NVL_OBJECT_V3S_CORRECTABLE_ERROR_RATE_ARRAY[INFOROM_NVL_OBJECT_V3S_NUM_LINKS];

#define INFOROM_NVL_OBJECT_V3S_CORRECTABLE_ERROR_RATE_ARRAY_FMT         \
    INFOROM_FMT_REP32(INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT) \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT)

typedef struct INFOROM_NVL_OBJECT_V3S_MAX_CORRECTABLE_ERROR_RATES
{
    INFOROM_NVL_OBJECT_V3S_CORRECTABLE_ERROR_RATE_ARRAY dailyMaxCorrectableErrorRates[5];
    INFOROM_NVL_OBJECT_V3S_CORRECTABLE_ERROR_RATE_ARRAY monthlyMaxCorrectableErrorRates[3];
} INFOROM_NVL_OBJECT_V3S_MAX_CORRECTABLE_ERROR_RATES;

#define INFOROM_NVL_OBJECT_V3S_MAX_CORRECTABLE_ERROR_RATES_FMT                  \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V3S_CORRECTABLE_ERROR_RATE_ARRAY_FMT)

typedef struct INFOROM_NVL_OBJECT_V3S
{
    INFOROM_OBJECT_HEADER_V1_00 header;
    INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO remoteDeviceInfo[INFOROM_NVL_OBJECT_V3S_NUM_LINKS];
    INFOROM_NVL_OBJECT_V3_MINION_DATA minionData[INFOROM_NVL_OBJECT_V3S_NUM_LINKS];
    INFOROM_NVL_OBJECT_V3S_MAX_CORRECTABLE_ERROR_RATES maxCorrectableErrorRates;
    INFOROM_NVL_OBJECT_V3_ERROR_ENTRY errorLog[INFOROM_NVL_OBJECT_V3S_NUM_ERROR_ENTRIES];
} INFOROM_NVL_OBJECT_V3S;

#define INFOROM_NVL_OBJECT_V3S_REMOTE_DEVICE_INFO_ARRAY_FMT         \
    INFOROM_FMT_REP32(INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT) \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT)

#define INFOROM_NVL_OBJECT_V3S_MINION_DATA_ARRAY_FMT                \
    INFOROM_FMT_REP32(INFOROM_NVL_OBJECT_V3_MINION_DATA_FMT)        \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_MINION_DATA_FMT)

#define INFOROM_NVL_OBJECT_V3S_ERROR_LOG_FMT                        \
    INFOROM_FMT_REP512(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP128(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP64(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)        \
    INFOROM_FMT_REP16(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)        \
    INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT

#define INFOROM_NVL_OBJECT_V3S_FMT                                  \
    INFOROM_OBJECT_HEADER_V1_00_FMT                                 \
    INFOROM_NVL_OBJECT_V3S_REMOTE_DEVICE_INFO_ARRAY_FMT             \
    INFOROM_NVL_OBJECT_V3S_MINION_DATA_ARRAY_FMT                    \
    INFOROM_NVL_OBJECT_V3S_MAX_CORRECTABLE_ERROR_RATES_FMT          \
    INFOROM_NVL_OBJECT_V3S_ERROR_LOG_FMT

#define INFOROM_NVL_OBJECT_V3S_PACKED_SIZE 17808

// Version 4.G0 ---------------------------------------------------------------

#define INFOROM_NVL_OBJECT_V4G_NUM_LINKS 18

typedef struct INFOROM_NVL_OBJECT_V4_L1_THRESHOLD_DATA
{
    inforom_U032 word1;
    inforom_U032 word2;
} INFOROM_NVL_OBJECT_V4_L1_THRESHOLD_DATA;

#define INFOROM_NVL_OBJECT_V4_L1_THRESHOLD_DATA_FMT "2d"

typedef INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE
        INFOROM_NVL_OBJECT_V4G_CORRECTABLE_ERROR_RATE_ARRAY[INFOROM_NVL_OBJECT_V4G_NUM_LINKS];

#define INFOROM_NVL_OBJECT_V4G_CORRECTABLE_ERROR_RATE_ARRAY_FMT         \
    INFOROM_FMT_REP16(INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT) \
    INFOROM_FMT_REP02(INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT)

typedef struct INFOROM_NVL_OBJECT_V4G_MAX_CORRECTABLE_ERROR_RATES
{
    INFOROM_NVL_OBJECT_V4G_CORRECTABLE_ERROR_RATE_ARRAY dailyMaxCorrectableErrorRates[5];
    INFOROM_NVL_OBJECT_V4G_CORRECTABLE_ERROR_RATE_ARRAY monthlyMaxCorrectableErrorRates[3];
} INFOROM_NVL_OBJECT_V4G_MAX_CORRECTABLE_ERROR_RATES;

#define INFOROM_NVL_OBJECT_V4G_MAX_CORRECTABLE_ERROR_RATES_FMT                  \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V4G_CORRECTABLE_ERROR_RATE_ARRAY_FMT)

typedef struct INFOROM_NVL_OBJECT_V4G_RESERVED
{
    inforom_U032 reserved;
} INFOROM_NVL_OBJECT_V4G_RESERVED;

#define INFOROM_NVL_OBJECT_V4G_RESERVED_FMT     "d"

#define INFOROM_NVL_OBJECT_V4G_NUM_ERROR_ENTRIES 364

typedef struct INFOROM_NVL_OBJECT_V4G
{
    INFOROM_OBJECT_HEADER_V1_00 header;
    INFOROM_NVL_OBJECT_V4_L1_THRESHOLD_DATA l1ThresholdData;
    INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO remoteDeviceInfo[INFOROM_NVL_OBJECT_V4G_NUM_LINKS];
    INFOROM_NVL_OBJECT_V4G_MAX_CORRECTABLE_ERROR_RATES maxCorrectableErrorRates;
    INFOROM_NVL_OBJECT_V3_ERROR_ENTRY errorLog[INFOROM_NVL_OBJECT_V4G_NUM_ERROR_ENTRIES];
    INFOROM_NVL_OBJECT_V4G_RESERVED reserved;
} INFOROM_NVL_OBJECT_V4G;

#define INFOROM_NVL_OBJECT_V4G_REMOTE_DEVICE_INFO_ARRAY_FMT         \
    INFOROM_FMT_REP16(INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT) \
    INFOROM_FMT_REP02(INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT)

#define INFOROM_NVL_OBJECT_V4G_ERROR_LOG_FMT                        \
    INFOROM_FMT_REP128(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP128(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP64(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)        \
    INFOROM_FMT_REP32(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)        \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)         \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)

#define INFOROM_NVL_OBJECT_V4G_FMT                                  \
    INFOROM_OBJECT_HEADER_V1_00_FMT                                 \
    INFOROM_NVL_OBJECT_V4G_REMOTE_DEVICE_INFO_ARRAY_FMT             \
    INFOROM_NVL_OBJECT_V4_L1_THRESHOLD_DATA_FMT                     \
    INFOROM_NVL_OBJECT_V4G_MAX_CORRECTABLE_ERROR_RATES_FMT          \
    INFOROM_NVL_OBJECT_V4G_ERROR_LOG_FMT                            \
    INFOROM_NVL_OBJECT_V4G_RESERVED_FMT

#define INFOROM_NVL_OBJECT_V4G_PACKED_SIZE 8400

// Version 4.S0 ---------------------------------------------------------------
#define INFOROM_NVL_OBJECT_V4S_NUM_LINKS 64
#define INFOROM_NVL_OBJECT_V4S_NUM_ERROR_ENTRIES 1286

typedef INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE
        INFOROM_NVL_OBJECT_V4S_CORRECTABLE_ERROR_RATE_ARRAY[INFOROM_NVL_OBJECT_V4S_NUM_LINKS];

#define INFOROM_NVL_OBJECT_V4S_CORRECTABLE_ERROR_RATE_ARRAY_FMT         \
    INFOROM_FMT_REP64(INFOROM_NVL_OBJECT_V3_CORRECTABLE_ERROR_RATE_FMT)

typedef struct INFOROM_NVL_OBJECT_V4S_MAX_CORRECTABLE_ERROR_RATES
{
    INFOROM_NVL_OBJECT_V4S_CORRECTABLE_ERROR_RATE_ARRAY dailyMaxCorrectableErrorRates[5];
    INFOROM_NVL_OBJECT_V4S_CORRECTABLE_ERROR_RATE_ARRAY monthlyMaxCorrectableErrorRates[3];
} INFOROM_NVL_OBJECT_V4S_MAX_CORRECTABLE_ERROR_RATES;

#define INFOROM_NVL_OBJECT_V4S_MAX_CORRECTABLE_ERROR_RATES_FMT                  \
    INFOROM_FMT_REP08(INFOROM_NVL_OBJECT_V4S_CORRECTABLE_ERROR_RATE_ARRAY_FMT)

typedef struct INFOROM_NVL_OBJECT_V4S
{
    INFOROM_OBJECT_HEADER_V1_00 header;
    INFOROM_NVL_OBJECT_V4_L1_THRESHOLD_DATA l1ThresholdData;
    INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO remoteDeviceInfo[INFOROM_NVL_OBJECT_V4S_NUM_LINKS];
    INFOROM_NVL_OBJECT_V4S_MAX_CORRECTABLE_ERROR_RATES maxCorrectableErrorRates;
    INFOROM_NVL_OBJECT_V3_ERROR_ENTRY errorLog[INFOROM_NVL_OBJECT_V4S_NUM_ERROR_ENTRIES];
} INFOROM_NVL_OBJECT_V4S;

#define INFOROM_NVL_OBJECT_V4S_REMOTE_DEVICE_INFO_ARRAY_FMT         \
    INFOROM_FMT_REP64(INFOROM_NVL_OBJECT_V3_REMOTE_DEVICE_INFO_FMT)

#define INFOROM_NVL_OBJECT_V4S_ERROR_LOG_FMT                        \
    INFOROM_FMT_REP512(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP512(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP256(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)       \
    INFOROM_FMT_REP04(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)        \
    INFOROM_FMT_REP02(INFOROM_NVL_OBJECT_V3_ERROR_ENTRY_FMT)

#define INFOROM_NVL_OBJECT_V4S_FMT                                  \
    INFOROM_OBJECT_HEADER_V1_00_FMT                                 \
    INFOROM_NVL_OBJECT_V4S_REMOTE_DEVICE_INFO_ARRAY_FMT             \
    INFOROM_NVL_OBJECT_V4_L1_THRESHOLD_DATA_FMT                     \
    INFOROM_NVL_OBJECT_V4S_MAX_CORRECTABLE_ERROR_RATES_FMT          \
    INFOROM_NVL_OBJECT_V4S_ERROR_LOG_FMT

#define INFOROM_NVL_OBJECT_V4S_PACKED_SIZE 29680

#define INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS 18
#define INFOROM_NVL_OBJECT_V5_MAX_NUM_LANES_PER_LINK 2

#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_UPTIME_FMT "d"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_LOCAL_MAX_RAW_BER_FMT "b"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_LOCAL_MAX_EFF_BER_FMT "b"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_CURR_EFF_MONITOR_FMT "w"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_PLR_XMIT_RETRY_FMT "d"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_TOTAL_RECOVERY_FMT "d"

#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_FMT                   \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_UPTIME_FMT                \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_LOCAL_MAX_RAW_BER_FMT     \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_LOCAL_MAX_EFF_BER_FMT     \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_CURR_EFF_MONITOR_FMT      \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_PLR_XMIT_RETRY_FMT        \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_TOTAL_RECOVERY_FMT

typedef struct INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA // u8[16]
{
    inforom_U032 uptime; // u8[4]
    inforom_U008 localMaxRawBERExp; // u8[1]
    inforom_U008 localMaxEffBERExp; // u8[1]
    inforom_U016 currEffBERMonitor; // u8[2]
    inforom_U032 plrXmitRetryWithinTSecMaxLo; // u8[4]
    inforom_U032 totalSuccessfulRecoveryEvents; // u8[4]
} INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA;

#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_MFDE_FMT "43d"

typedef struct INFOROM_NVL_OBJECT_V5_RECORD_METRIC_MFDE // u8[172]
{
    inforom_U032 data[43]; // u8[172]
} INFOROM_NVL_OBJECT_V5_RECORD_METRIC_MFDE;

#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_INITIAL_RAW_BER_FMT "18b"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_INITIAL_EFF_BER_FMT "18b"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_LAST_DOWN_REASON_FMT "18b"
#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_PADDING_FMT "82b"

#define INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_FMT                        \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_MFDE_FMT                            \
    INFOROM_FMT_REP16(INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_FMT)    \
    INFOROM_FMT_REP02(INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA_FMT)    \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_INITIAL_RAW_BER_FMT            \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_INITIAL_EFF_BER_FMT            \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_LAST_DOWN_REASON_FMT           \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_PADDING_FMT

typedef struct INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA // u8[596]
{
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_MFDE mfdeEvent; // u8[172]
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_LINK_DATA link[INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS]; // u8[288]
    inforom_U008 initialRawBERExp[INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS]; // u8[18]
    inforom_U008 initialEffBERExp[INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS]; // u8[18]
    inforom_U008 lastDownReason[INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS]; // u8[18]
    inforom_U008 padding[82]; // u8[82]
} INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA;

#define INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_HISTOGRAM_FMT "12b"

typedef struct INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_HISTOGRAM // u8[12]
{
    inforom_U008 fixedPoint[12]; // u8[12]
} INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_HISTOGRAM;

#define INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA_LINK_DOWN_COUNT_FMT "d"

#define INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA_FMT                             \
    "4b"                                                                                    \
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA_LINK_DOWN_COUNT_FMT                 \
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_HISTOGRAM_FMT

typedef struct INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA // u8[20]
{
    inforom_U008 reserved[4]; // u8[4]
    inforom_U032 linkDownCount; // u8[4]
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_HISTOGRAM downReasonHistogram; // u8[12]
} INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA;

#define INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA_GLOBAL_MAX_RAW_BER_FMT "18b"
#define INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA_GLOBAL_MAX_EFF_BER_FMT "18b"

#define INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA_FMT                                  \
    INFOROM_FMT_REP16(INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA_FMT)              \
    INFOROM_FMT_REP02(INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA_FMT)              \
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA_GLOBAL_MAX_RAW_BER_FMT                   \
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA_GLOBAL_MAX_EFF_BER_FMT

typedef struct INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA // u8[396]
{
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_LINK_DATA link[INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS]; // u8[360]
    inforom_U008 globalMaxRawBERExp[INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS]; // u8[18]
    inforom_U008 globalMaxEffBERExp[INFOROM_NVL_OBJECT_V5_MAX_NUM_LINKS]; // u8[18]
} INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA;

#define INFOROM_NVL_OBJECT_V5_RESERVED_FMT "8b"

#define INFOROM_NVL_OBJECT_V5_FMT                       \
    INFOROM_OBJECT_HEADER_V1_00_FMT                     \
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA_FMT  \
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA_FMT        \
    INFOROM_NVL_OBJECT_V5_RESERVED_FMT

#define INFOROM_NVL_OBJECT_V5_PACKED_SIZE 1008

typedef struct INFOROM_NVL_OBJECT_V5 // u8[1008]
{
    INFOROM_OBJECT_HEADER_V1_00 header; // u8[8]
    INFOROM_NVL_OBJECT_V5_ACCUMULATIVE_METRIC_DATA accumulative; // u8[396]
    INFOROM_NVL_OBJECT_V5_RECORD_METRIC_DATA record; // u8[596]
    inforom_U008 reserved[8]; // u8[8]
} INFOROM_NVL_OBJECT_V5;

union INFOROM_NVL_OBJECT
{
    INFOROM_OBJECT_HEADER_V1_00 header;
    INFOROM_NVL_OBJECT_V1       v1;
    INFOROM_NVL_OBJECT_V2       v2;
    INFOROM_NVL_OBJECT_V3G      v3g;
    INFOROM_NVL_OBJECT_V3S      v3s;
    INFOROM_NVL_OBJECT_V4G      v4g;
    INFOROM_NVL_OBJECT_V4S      v4s;
    INFOROM_NVL_OBJECT_V5       v5;
};

#define INFOROM_NVL_OBJECT_FLUSH_INTERVAL (60 * 10)

typedef struct
{
    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
    } errorsPerMinute[INFOROM_NVL_OBJECT_V3G_NUM_LINKS];

    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
        NvU32 rxLinkReplay;
        NvU32 txLinkReplay;
        NvU32 linkRecovery;
    } lastRead[INFOROM_NVL_OBJECT_V3G_NUM_LINKS];
} INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V3G;

typedef struct
{
    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
    } errorsPerMinute[INFOROM_NVL_OBJECT_V3S_NUM_LINKS];

    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
        NvU32 rxLinkReplay;
        NvU32 txLinkReplay;
        NvU32 linkRecovery;
    } lastRead[INFOROM_NVL_OBJECT_V3S_NUM_LINKS];
} INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V3S;

typedef struct
{
    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
    } errorsPerMinute[INFOROM_NVL_OBJECT_V4G_NUM_LINKS];

    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
        NvU32 rxLinkReplay;
        NvU32 txLinkReplay;
        NvU32 linkRecovery;
    } lastRead[INFOROM_NVL_OBJECT_V4G_NUM_LINKS];
} INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V4G;

typedef struct
{
    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
    } errorsPerMinute[INFOROM_NVL_OBJECT_V4S_NUM_LINKS];

    struct
    {
        NvU32 flitCrc;
        NvU32 laneCrc[4];
        NvU32 rxLinkReplay;
        NvU32 txLinkReplay;
        NvU32 linkRecovery;
    } lastRead[INFOROM_NVL_OBJECT_V4S_NUM_LINKS];
} INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V4S;

typedef struct INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V5
{
    struct
    {
        void *pParams;
        void *pAsyncHandle;
    } request[2];
    NvU8 head;
} INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V5;

typedef union
{
    INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V3G v3g;
    INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V3S v3s;
    INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V4G v4g;
    INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V4S v4s;
    INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE_V5  v5;
} INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE;

typedef struct
{
    const char         *pFmt;
    NvU8               *pPackedObject;
    INFOROM_NVL_OBJECT *pNvl;

    NvBool bDirty;
    NvBool bDisableFatalErrorLogging;
    NvBool bDisableCorrectableErrorLogging;
    NvBool bCallbackPending;

    NvU32 timeSinceLastFlush;

    INFOROM_NVL_CORRECTABLE_ERROR_RATE_STATE correctableErrorRateState;
} INFOROM_NVLINK_STATE, *PINFOROM_NVLINK_STATE;

#endif /* _IFRNVL_H_ */
