#' @include stls-class.R summary.stls-class.R bootconfint.R
NULL

#'Print function for objects of class \code{"stls"}
#'
#'@param x object of class \code{"stls"}
#'@param digits number of digits to be printed.
#'@param ... additional arguments
#'@export

print.stls <- function (x, digits = max(3, getOption("digits") - 3), ...)
{
  cat("\nCall:\n", deparse(x$call), "\n\n", sep = "")
  if (length(x$coefficients))
  {
    cat("Coefficients:\n")
    print.default(format(x$coefficients, digits = digits), print.gap = 2,
                  quote = FALSE)
  }
  else cat("No coefficients\n")
  cat("\n")
  cat("Iterations:\n")
  print.default(format(x$counts, digits = digits), print.gap = 2,
                quote = FALSE)
  cat("\n")
  invisible(x)
}

#'Summary function for objects of class \code{"stls"}
#'
#'@param object object of class \code{"stls"}
#'@param level confidence level. A number between 0 and 1. The default value is \code{0.95}.
#'@param ... additional arguments
#'@return A list with values:
#'  \item{coefficients }{the named vector of coefficients}
#'  \item{counts }{number of iterations used by \code{\link{optim}}. See the documentation for \code{\link{optim}} for further details}
#'  \item{call }{the matched call}
#'  \item{covariance }{if \code{covar = TRUE}, the estimated covariance matrix}
#'  \item{level}{confidence level}
#'  \item{confint}{confidence intervals, based on normal distribution}
#'  \item{bootconfint}{bootstrap confidence intervals, percentile method}
#'@export
#'@importFrom stats pt qt

summary.stls <- function(object,level=0.95,...)
{
  x <- object
  if(length(x$covariance))
  {
    R <- x$covariance
    se <- sqrt(diag(R))
    est <- t(x$coefficients)
    tval <- est/se
    rdf <- x$df.residual
    ans <- list()
    ans$call <- x$call
    ans$counts <- x$counts
    ans$level <- level
    ans$coefficients <- cbind(est, se, tval, 2 * pt(abs(tval), rdf, lower.tail = FALSE))
    dimnames(ans$coefficients) <- list(rownames(t(x$coefficients)), c("Estimate", "Std. Error", "t value", "Pr(>|t|)"))
    ans$covariance <- R
    error <- qt((1-(1-level)/2),df=rdf)*se
    left <- t(x$coefficients)-error
    right <- t(x$coefficients)+error
    ans$confint	<- cbind(left,right)
    dimnames(ans$confint) <- list(rownames(t(x$coefficients)), c("Lower","Upper"))
    ans$bootconfint <- bootconfint(x$bootrepl,level)
    dimnames(ans$bootconfint) <- list(rownames(t(x$coefficients)), c("Lower","Upper"))
  }
  else
  {
    est <- t(x$coefficients)
    ans <- list()
    ans$call <- x$call
    ans$counts <- x$counts
    se <- NA
    tval <- NA
    pval <- NA
    ans$coefficients <- cbind(est, se, tval, pval)
    dimnames(ans$coefficients) <- list(rownames(t(x$coefficients)), c("Estimate", "Std. Error", "t value", "Pr(>|t|)"))
  }
  class(ans) <- c("summary.stls","stls")
  ans

}

#'Print function for objects of class \code{"summary.stls"}
#'
#'@param x object of class \code{"summary.stls"}
#'@param digits number of digits to be printed.
#'@param ... additional arguments
#'@export
#'@importFrom stats printCoefmat
#'
print.summary.stls <- function(x, digits= max(3, getOption("digits") - 3), ...)
{
	cat("\nCall:\n", deparse(x$call), "\n\n", sep = "")
	if (length(x$coefficients))
	{
 		cat("Coefficients:\n")
		printCoefmat(x$coefficients, digits = digits, print.gap=2.5)
 	}
	else cat("No coefficients\n")
	cat("\n")
	if(length(x$counts))
	{
		cat("Number of iterations:\n")
		print(x$counts)
	}
	cat("\n")
	if(length(x$covariance))
	{
		cat("\n")
		if(length(x$confint))
		{
			cat(gettextf("%d%% Confidence Intervals:",(100*x$level)),"\n")
			print.default(format(x$confint, digits = digits), print.gap = 3,
 			quote = FALSE)
		}
		else cat("No confidence intervals\n")
		cat("\n")
		if(length(x$bootconfint))
		{
			cat(gettextf("%d%% Confidence Intervals (Percentile):", (100*x$level)),"\n")
			print.default(format(x$bootconfint, digits = digits), print.gap = 3,
 			quote = FALSE)
		}
	}
	else
		cat("No covariance matrix has been estimated, hence no t-tests or confidence intervals are returned.\n To get these, choose covar=TRUE in the function call for stls().")
	cat("\n")
	invisible(x)

}


#'Function to extract model coefficients for objects of class \code{"stls"}
#'
#'@param object object of class \code{"stls"}
#'@param ... additional arguments
#'@export


coef.stls <- function(object,...)
{
	object$coefficients
}

#'Function to obtain the variance-covariance matrix for objects of class \code{"stls"}
#'
#'@param object object of class \code{"stls"}
#'@param ... additional arguments
#'@export

vcov.stls <- function(object,...)
{
	if(length(object$covariance))
		object$covariance
	else
		cat("No covariance matrix has been estimated. To do this, choose covar=TRUE in the function call for stls(). \n")

}

#'Function to extract model residuals from objects of class \code{"stls"}
#'
#'@param object object of class \code{"stls"}
#'@param ... additional arguments
#'@export

residuals.stls <- function(object,...)
{
	object$residuals
}

#'Function to obtain fitted values from objects of class \code{"stls"}
#'
#'@param object object of class \code{"stls"}
#'@param ... additional arguments
#'@export

fitted.stls <- function(object,...)
{
	object$fitted.values
}




