% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/histDecomp.R, R/plotMethods.R, R/printMethods.R
\name{hist_decomp}
\alias{hist_decomp}
\alias{plot.histdecomp}
\alias{print.histdecomp}
\title{Compute historical decompositions for structural STVAR models.}
\usage{
hist_decomp(stvar)

\method{plot}{histdecomp}(x, ..., plot_by_shock = FALSE, which_to_plot)

\method{print}{histdecomp}(x, ..., digits = 3, which_vars, which_indices)
}
\arguments{
\item{stvar}{an object of class \code{'stvar'} defining a structural or reduced form
STVAR model. For a reduced form model (that is not readily identified statiscally),
the shocks are automatically identified by the lower triangular Cholesky decomposition.}

\item{x}{object of class \code{'histdecomp'} created by the function \code{hist_decomp}.}

\item{...}{currently not in use.}

\item{plot_by_shock}{should the historical decompositions by plotted so that there
is one figure for each shock (rather than one figure for each variable)?}

\item{which_to_plot}{a numeric vector with the indices of the variables or shocks
(depending on the argument \code{plot_by_shock}) to be plotted. The default is to plot
all of them.}

\item{digits}{how many significant digits to print?}

\item{which_vars}{a numeric vector specifying the variables to print. The default is that
all the variables are printed.}

\item{which_indices}{a numeric vector specifying the time period indices to print. The default
is that all the time periods are printed.}
}
\value{
Returns a class \code{'histdecomp'} list with the following elements:
  \describe{
    \item{init_cond_comp}{A matrix of size \eqn{(T \times d)} containing the contributions of the initial conditions to the movements of
     the variables at each time point; the element \code{t, i} giving the contribution at the time \code{t} on the variable \code{i}.}
    \item{steady_state_comp}{A matrix of size \eqn{(T \times d)} containing the contributions of the steady state component to the movements of
     the variables at each time point; the element \code{t, i} giving the contribution at the time \code{t} on the variable \code{i}.}
    \item{shock_comp}{A matrix of size \eqn{(T \times d)} containing the contributions of the shocks to the movements of the variables at each
     time point; the element \code{t, i} giving the contribution at the time \code{t} on the variable \code{i}.}
    \item{contributions_of_shocks}{A 3D array of size \eqn{(T \times d \times d)} containing the cumulative contributions of the shocks to the
     movements of the variables at each time point; the element \code{t, i1, i2} giving the contribution of the shock \code{i1} to the variable
     \code{i2} at the time \code{t}.}
    \item{stvar}{The original STVAR model object.}
 }

Returns the input object \code{x} invisibly.
}
\description{
\code{hist_decomp} computes historical decompositions for structural STVAR models.
}
\details{
The historical decomposition quantifies the cumulative effects the shocks to the movements of
  the variables (see, e.g., Kilian and Lütkepohl, 2017, Section~4.3) The historical decompositions are
  computed as described in Wong (2018). Note that due to the effect of the "initial conditions" and the
  "steady state component", which are not attributed to the shocks, the cumulative effects of the shocks
  do not sum to the observed time series.
}
\section{Functions}{
\itemize{
\item \code{plot(histdecomp)}: plot method

\item \code{print(histdecomp)}: print method

}}
\examples{
# Recursively identified logistic Student's t STVAR(p=3, M=2) model with the first
# lag of the second variable as the switching variable:
params32logt <- c(0.5959, 0.0447, 2.6279, 0.2897, 0.2837, 0.0504, -0.2188, 0.4008,
  0.3128, 0.0271, -0.1194, 0.1559, -0.0972, 0.0082, -0.1118, 0.2391, 0.164, -0.0363,
  -1.073, 0.6759, 3e-04, 0.0069, 0.4271, 0.0533, -0.0498, 0.0355, -0.4686, 0.0812,
  0.3368, 0.0035, 0.0325, 1.2289, -0.047, 0.1666, 1.2067, 7.2392, 11.6091)
mod32logt <- STVAR(gdpdef, p=3, M=2, params=params32logt, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", identification="recursive")

# Calculate the historical decomposition:
histdec <- hist_decomp(mod32logt)

# Print historical decomposition for Variable 1 for the first ten time periods:
print(histdec, which_vars=1, which_indices=1:10)

# Plot the historical decomposition for all variables:
plot(histdec)

# Plot the contributions of Shock 1 on the movements of all the variables:
plot(histdec, plot_by_shock=TRUE, which_to_plot=1)
}
\references{
\itemize{
   \item Kilian L., Lütkepohl H. 2017. Structural Vector Autoregressive Analysis. 1st edition.
     \emph{Cambridge University Press}, Cambridge.
   \item Wong H. 2018. Historical decomposition for nonlinear vector autoregressive models.
     \emph{CAMA Working Paper No. 62/2017, available as SSRN:3057759}.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{GFEVD}}, \code{\link{linear_IRF}}, \code{\link{fitSSTVAR}}, \code{\link{cfact_hist}},
 \code{\link{cfact_fore}}, \code{\link{cfact_girf}}
}
