/*
   igraph library.
   Copyright (C) 2025  The igraph development team <igraph@igraph.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/

#include "igraph_setup.h"
#include "igraph_random.h"

#include "random/random_internal.h"

/**
 * \ingroup setup
 * \function setup_rng
 * \brief Initializes the random number generator.
 *
 * This function initializes the igraph library by setting up a random seed
 * for its internal random number generator. It should be called before
 * using any igraph functions that may use random numbers.
 *
 * \return Error code; currently always \c IGRAPH_SUCCESS.
 */
static igraph_error_t setup_rng(void) {
    igraph_rng_t *rng = igraph_rng_default();

    if (!rng->is_seeded) {
        igraph_rng_seed(rng, igraph_i_get_random_seed());
        rng->is_seeded = true;
    };

    return IGRAPH_SUCCESS;
}

/**
 * \ingroup setup
 * \function igraph_setup
 * \brief Initializes the igraph library.
 *
 * This function is a convenience function to call all setup functions that are
 * provided by the igraph library.
 *
 * </para><para>
 * Most of the library functions will work even if this function is not called,
 * but it is recommended to call it before using any igraph functions that may
 * use random numbers, such as graph generators or random sampling functions.
 * This function initializes the random number generator with a seed based on
 * the current time, ensuring that the random numbers generated by igraph are
 * different each time the program is run.
 *
 * \return Error code; currently always \c IGRAPH_SUCCESS.
 */
igraph_error_t igraph_setup(void) {
    IGRAPH_CHECK(setup_rng());
    return IGRAPH_SUCCESS;
}
