/*
   igraph library.
   Copyright (C) 2009-2025  The igraph development team <igraph@igraph.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/

#ifndef IGRAPH_GAMES_H
#define IGRAPH_GAMES_H

#include "igraph_decls.h"
#include "igraph_constants.h"
#include "igraph_datatype.h"
#include "igraph_error.h"
#include "igraph_graphicality.h"
#include "igraph_matrix.h"
#include "igraph_matrix_list.h"
#include "igraph_types.h"
#include "igraph_vector.h"
#include "igraph_vector_list.h"

IGRAPH_BEGIN_C_DECLS

/* -------------------------------------------------- */
/* Constructors, games (=stochastic)                  */
/* -------------------------------------------------- */

IGRAPH_EXPORT igraph_error_t igraph_barabasi_game(igraph_t *graph, igraph_int_t n,
                                       igraph_real_t power,
                                       igraph_int_t m,
                                       const igraph_vector_int_t *outseq,
                                       igraph_bool_t outpref,
                                       igraph_real_t A,
                                       igraph_bool_t directed,
                                       igraph_barabasi_algorithm_t algo,
                                       const igraph_t *start_from);

IGRAPH_EXPORT igraph_error_t igraph_erdos_renyi_game_gnp(
        igraph_t *graph,
        igraph_int_t n, igraph_real_t p,
        igraph_bool_t directed,
        igraph_edge_type_sw_t allowed_edge_types,
        igraph_bool_t edge_labeled);

IGRAPH_EXPORT igraph_error_t igraph_erdos_renyi_game_gnm(
        igraph_t *graph,
        igraph_int_t n, igraph_int_t m,
        igraph_bool_t directed,
        igraph_edge_type_sw_t allowed_edge_types,
        igraph_bool_t edge_labeled);

IGRAPH_EXPERIMENTAL IGRAPH_EXPORT igraph_error_t igraph_iea_game(
        igraph_t *graph,
        igraph_int_t n, igraph_int_t m,
        igraph_bool_t directed, igraph_bool_t loops);

IGRAPH_EXPORT igraph_error_t igraph_degree_sequence_game(igraph_t *graph, const igraph_vector_int_t *out_deg,
                                              const igraph_vector_int_t *in_deg,
                                              igraph_degseq_t method);
IGRAPH_EXPORT igraph_error_t igraph_growing_random_game(igraph_t *graph, igraph_int_t n,
                                             igraph_int_t m, igraph_bool_t directed, igraph_bool_t citation);
IGRAPH_EXPORT igraph_error_t igraph_barabasi_aging_game(igraph_t *graph,
                                             igraph_int_t nodes,
                                             igraph_int_t m,
                                             const igraph_vector_int_t *outseq,
                                             igraph_bool_t outpref,
                                             igraph_real_t pa_exp,
                                             igraph_real_t aging_exp,
                                             igraph_int_t aging_bin,
                                             igraph_real_t zero_deg_appeal,
                                             igraph_real_t zero_age_appeal,
                                             igraph_real_t deg_coef,
                                             igraph_real_t age_coef,
                                             igraph_bool_t directed);
IGRAPH_EXPORT igraph_error_t igraph_recent_degree_game(igraph_t *graph, igraph_int_t n,
                                            igraph_real_t power,
                                            igraph_int_t window,
                                            igraph_int_t m,
                                            const igraph_vector_int_t *outseq,
                                            igraph_bool_t outpref,
                                            igraph_real_t zero_appeal,
                                            igraph_bool_t directed);
IGRAPH_EXPORT igraph_error_t igraph_recent_degree_aging_game(igraph_t *graph,
                                                  igraph_int_t nodes,
                                                  igraph_int_t m,
                                                  const igraph_vector_int_t *outseq,
                                                  igraph_bool_t outpref,
                                                  igraph_real_t pa_exp,
                                                  igraph_real_t aging_exp,
                                                  igraph_int_t aging_bin,
                                                  igraph_int_t window,
                                                  igraph_real_t zero_appeal,
                                                  igraph_bool_t directed);
IGRAPH_EXPORT igraph_error_t igraph_callaway_traits_game(igraph_t *graph, igraph_int_t nodes,
                                              igraph_int_t types, igraph_int_t edges_per_step,
                                              const igraph_vector_t *type_dist,
                                              const igraph_matrix_t *pref_matrix,
                                              igraph_bool_t directed,
                                              igraph_vector_int_t *node_type_vec);
IGRAPH_EXPORT igraph_error_t igraph_establishment_game(igraph_t *graph, igraph_int_t nodes,
                                            igraph_int_t types, igraph_int_t k,
                                            const igraph_vector_t *type_dist,
                                            const igraph_matrix_t *pref_matrix,
                                            igraph_bool_t directed,
                                            igraph_vector_int_t *node_type_vec);
IGRAPH_EXPORT igraph_error_t igraph_grg_game(igraph_t *graph, igraph_int_t nodes,
                                  igraph_real_t radius, igraph_bool_t torus,
                                  igraph_vector_t *x, igraph_vector_t *y);
IGRAPH_EXPORT igraph_error_t igraph_preference_game(igraph_t *graph, igraph_int_t nodes,
                                         igraph_int_t types,
                                         const igraph_vector_t *type_dist,
                                         igraph_bool_t fixed_sizes,
                                         const igraph_matrix_t *pref_matrix,
                                         igraph_vector_int_t *node_type_vec,
                                         igraph_bool_t directed, igraph_bool_t loops);
IGRAPH_EXPORT igraph_error_t igraph_asymmetric_preference_game(igraph_t *graph, igraph_int_t nodes,
                                                    igraph_int_t out_types,
                                                    igraph_int_t in_types,
                                                    const igraph_matrix_t *type_dist_matrix,
                                                    const igraph_matrix_t *pref_matrix,
                                                    igraph_vector_int_t *node_type_out_vec,
                                                    igraph_vector_int_t *node_type_in_vec,
                                                    igraph_bool_t loops);

IGRAPH_EXPORT igraph_error_t igraph_rewire_edges(igraph_t *graph, igraph_real_t prob,
                                                 igraph_edge_type_sw_t allowed_edge_types);
IGRAPH_EXPORT igraph_error_t igraph_rewire_directed_edges(igraph_t *graph, igraph_real_t prob,
                                               igraph_bool_t loops, igraph_neimode_t mode);

IGRAPH_EXPORT igraph_error_t igraph_watts_strogatz_game(igraph_t *graph, igraph_int_t dim,
                                             igraph_int_t size, igraph_int_t nei,
                                             igraph_real_t p,
                                             igraph_edge_type_sw_t allowed_edge_types);

IGRAPH_EXPORT igraph_error_t igraph_lastcit_game(igraph_t *graph,
                                      igraph_int_t nodes, igraph_int_t edges_per_node,
                                      igraph_int_t agebins,
                                      const igraph_vector_t *preference, igraph_bool_t directed);

IGRAPH_EXPORT igraph_error_t igraph_cited_type_game(igraph_t *graph, igraph_int_t nodes,
                                         const igraph_vector_int_t *types,
                                         const igraph_vector_t *pref,
                                         igraph_int_t edges_per_step,
                                         igraph_bool_t directed);

IGRAPH_EXPORT igraph_error_t igraph_citing_cited_type_game(igraph_t *graph, igraph_int_t nodes,
                                                const igraph_vector_int_t *types,
                                                const igraph_matrix_t *pref,
                                                igraph_int_t edges_per_step,
                                                igraph_bool_t directed);

IGRAPH_EXPORT igraph_error_t igraph_forest_fire_game(igraph_t *graph, igraph_int_t nodes,
                                          igraph_real_t fw_prob, igraph_real_t bw_factor,
                                          igraph_int_t ambs, igraph_bool_t directed);


IGRAPH_EXPORT igraph_error_t igraph_simple_interconnected_islands_game(
    igraph_t *graph,
    igraph_int_t islands_n,
    igraph_int_t islands_size,
    igraph_real_t islands_pin,
    igraph_int_t n_inter);

IGRAPH_EXPORT igraph_error_t igraph_static_fitness_game(igraph_t *graph, igraph_int_t no_of_edges,
                                             const igraph_vector_t *fitness_out, const igraph_vector_t *fitness_in,
                                             igraph_edge_type_sw_t allowed_edge_types);

IGRAPH_EXPORT igraph_error_t igraph_static_power_law_game(igraph_t *graph,
                                               igraph_int_t no_of_nodes, igraph_int_t no_of_edges,
                                               igraph_real_t exponent_out, igraph_real_t exponent_in,
                                               igraph_edge_type_sw_t allowed_edge_types,
                                               igraph_bool_t finite_size_correction);

IGRAPH_EXPERIMENTAL IGRAPH_EXPORT igraph_error_t igraph_chung_lu_game(igraph_t *graph,
                                                  const igraph_vector_t *expected_out_deg,
                                                  const igraph_vector_t *expected_in_deg,
                                                  igraph_bool_t loops,
                                                  igraph_chung_lu_t variant);

IGRAPH_EXPORT igraph_error_t igraph_k_regular_game(igraph_t *graph,
                                        igraph_int_t no_of_nodes, igraph_int_t k,
                                        igraph_bool_t directed, igraph_bool_t multiple);

IGRAPH_EXPORT igraph_error_t igraph_sbm_game(
        igraph_t *graph,
        const igraph_matrix_t *pref_matrix,
        const igraph_vector_int_t *block_sizes,
        igraph_bool_t directed,
        igraph_edge_type_sw_t allowed_edge_types);

IGRAPH_EXPORT igraph_error_t igraph_hsbm_game(igraph_t *graph, igraph_int_t n,
                                   igraph_int_t m, const igraph_vector_t *rho,
                                   const igraph_matrix_t *C, igraph_real_t p);

IGRAPH_EXPORT igraph_error_t igraph_hsbm_list_game(igraph_t *graph, igraph_int_t n,
                                        const igraph_vector_int_t *mlist,
                                        const igraph_vector_list_t *rholist,
                                        const igraph_matrix_list_t *Clist,
                                        igraph_real_t p);

IGRAPH_EXPORT igraph_error_t igraph_correlated_game(igraph_t *new_graph, const igraph_t *old_graph,
                                         igraph_real_t corr, igraph_real_t p,
                                         const igraph_vector_int_t *permutation);

IGRAPH_EXPORT igraph_error_t igraph_correlated_pair_game(igraph_t *graph1, igraph_t *graph2,
                                              igraph_int_t n, igraph_real_t corr, igraph_real_t p,
                                              igraph_bool_t directed,
                                              const igraph_vector_int_t *permutation);

IGRAPH_EXPORT igraph_error_t igraph_tree_game(igraph_t *graph, igraph_int_t n, igraph_bool_t directed,
                                   igraph_random_tree_t method);

IGRAPH_EXPORT igraph_error_t igraph_dot_product_game(igraph_t *graph, const igraph_matrix_t *vecs,
                                          igraph_bool_t directed);

IGRAPH_END_C_DECLS

#endif
