% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimization_function.R
\name{optimization_function}
\alias{optimization_function}
\title{Compute optimization scores for spatial predictor selection}
\usage{
optimization_function(
  x = NULL,
  weight.r.squared = NULL,
  weight.penalization.n.predictors = NULL,
  optimization.method = "moran.i"
)
}
\arguments{
\item{x}{Data frame containing optimization metrics for candidate spatial predictor sets. Generated internally by \code{\link[=select_spatial_predictors_sequential]{select_spatial_predictors_sequential()}} or \code{\link[=select_spatial_predictors_recursive]{select_spatial_predictors_recursive()}}. Must include columns: \code{moran.i}, \code{r.squared}, \code{penalization.per.variable}, and \code{p.value.binary} (for "p.value" method).}

\item{weight.r.squared}{Numeric value between 0 and 1 specifying the weight for R-squared in the optimization score. Higher values prioritize model performance.}

\item{weight.penalization.n.predictors}{Numeric value between 0 and 1 specifying the weight for penalizing the number of spatial predictors. Higher values favor more parsimonious models.}

\item{optimization.method}{Character string specifying the optimization method: \code{"moran.i"} (default) or \code{"p.value"}. Default: \code{"moran.i"}.}
}
\value{
Numeric vector of optimization scores, one per row in \code{x}. Higher scores indicate better solutions. All values are rescaled between 0 and 1 for comparability.
}
\description{
Computes optimization scores for candidate spatial predictor sets using either the "moran.i" or "p.value" method. Higher scores indicate better trade-offs between spatial autocorrelation reduction, model performance, and parsimony.
}
\details{
This function balances three objectives when selecting spatial predictors:
\enumerate{
\item \strong{Reduce spatial autocorrelation}: Maximize \verb{1 - Moran's I} to minimize residual spatial autocorrelation
\item \strong{Maintain model performance}: Account for model R-squared
\item \strong{Favor parsimony}: Penalize models with many spatial predictors
}

\strong{Optimization methods:}

The \strong{"moran.i"} method computes:

\verb{score = (1 - Moran's I) + w1 × R² - w2 × penalization}

where all components are rescaled to the range 0 to 1, \code{w1} = \code{weight.r.squared}, and \code{w2} = \code{weight.penalization.n.predictors}.

The \strong{"p.value"} method computes:

\verb{score = max(1 - Moran's I, binary p-value) + w1 × R² - w2 × penalization}

where the binary p-value is 1 if p equal or lower than 0.05 (no significant spatial autocorrelation), and 0 otherwise.

\strong{Practical differences:}
\itemize{
\item The "moran.i" method uses continuous Moran's I values and typically selects more spatial predictors to achieve lower spatial autocorrelation
\item The "p.value" method uses binary significance thresholds and typically selects fewer predictors, stopping once spatial autocorrelation becomes non-significant
}

The optimal model is the one with the highest optimization score.
}
\examples{
\dontrun{
# This function is typically called internally during spatial predictor selection
# Example showing the structure of input data:

# Simulated optimization data frame
opt_data <- data.frame(
  moran.i = c(0.5, 0.3, 0.2, 0.15),
  r.squared = c(0.6, 0.65, 0.68, 0.69),
  penalization.per.variable = c(0.1, 0.2, 0.3, 0.4),
  p.value.binary = c(0, 0, 1, 1)
)

# Compute optimization scores
scores_moran <- optimization_function(
  x = opt_data,
  weight.r.squared = 0.5,
  weight.penalization.n.predictors = 0.5,
  optimization.method = "moran.i"
)

# Compare methods
scores_pvalue <- optimization_function(
  x = opt_data,
  weight.r.squared = 0.5,
  weight.penalization.n.predictors = 0.5,
  optimization.method = "p.value"
)

# Higher score indicates better solution
which.max(scores_moran)
which.max(scores_pvalue)
}
}
\seealso{
\code{\link[=select_spatial_predictors_recursive]{select_spatial_predictors_recursive()}}, \code{\link[=select_spatial_predictors_sequential]{select_spatial_predictors_sequential()}}, \code{\link[=moran]{moran()}}

Other utilities: 
\code{\link{.vif_to_df}()},
\code{\link{auc}()},
\code{\link{beowulf_cluster}()},
\code{\link{objects_size}()},
\code{\link{prepare_importance_spatial}()},
\code{\link{rescale_vector}()},
\code{\link{root_mean_squared_error}()},
\code{\link{setup_parallel_execution}()},
\code{\link{standard_error}()},
\code{\link{statistical_mode}()},
\code{\link{thinning}()},
\code{\link{thinning_til_n}()}
}
\concept{utilities}
