#' Check alternative start codons for a start codon SNV
#'
#' @param roi.aln.df A data frame of the detection window alignment
#' @param minus.strand A logical value indicating whether the pathogenic mutation on the minus strand
#' @param snv.flank.bp A non-negative integer specifying the length of flanking regions to be added to both ends of the start codon for detecting SNV reversions
#' 
#' @return A list including original start codon positions and alternative start codon positions 
#' 
#' @examples
#' \dontrun{
#'     checkStartCodonSNV()
#' }
#'
#' @noRd
checkStartCodonSNV <- function(roi.aln.df, minus.strand, snv.flank.bp=10){
    
    ref.bases <- roi.aln.df$ref_pos[ !is.na(roi.aln.df$ref_pos) ]
    roi.start <- min(ref.bases)
    roi.end <- max(ref.bases)
    
    snv.base.ref <- roi.aln.df$ref_seq[ roi.aln.df$pathog_mut==1 ]
    snv.base <- roi.aln.df$ref_pos[ roi.aln.df$pathog_mut==1 ]
    ref.cds.bases <- roi.aln.df$ref_pos[ roi.aln.df$ref_cds==1 ]
    ref.noncds.bases <- setdiff( ref.bases, ref.cds.bases )
    
    cryptic.start.codon <- NULL
    nWindow <- floor(snv.flank.bp/3)
    
    if(!minus.strand){
        
        if(snv.base.ref=="A"){
            snv.window.left <- snv.base 
        }else if(snv.base.ref=="T"){
            snv.window.left <- snv.base - 1 
        }else if(snv.base.ref=="G"){
            snv.window.left <- snv.base - 2
        }
        snv.window.mid <- snv.window.left + 1
        snv.window.right <- snv.window.left + 2
        
        real.start.codon <- c( snv.window.left, snv.window.mid, snv.window.right )
        
        if( roi.start<=snv.window.left & roi.end>=snv.window.right & nWindow>0 ){
            
            for(i in 1:nWindow){
                
                right.window <- snv.window.left:snv.window.right + 3*i  
                check.bases <- snv.window.left:right.window[3]
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% right.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% right.window], collapse="")=="ATG" ){
                            cryptic.start.codon <- right.window
                        }
                    }
                }
                if( !is.null(cryptic.start.codon) ){ break }
                
                left.window <- snv.window.left:snv.window.right - 3*i  
                check.bases <- left.window[1]:snv.window.right
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% left.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% left.window], collapse="")=="ATG" ){
                            cryptic.start.codon <- left.window
                        }
                    }
                }
                if( !is.null(cryptic.start.codon) ){ break }
                
            }
        }
        
    }else{
        
        if(snv.base.ref=="C"){
            snv.window.right <- snv.base + 2
        }else if(snv.base.ref=="A"){
            snv.window.right <- snv.base + 1
        }else if(snv.base.ref=="T"){
            snv.window.right <- snv.base 
        }
        snv.window.mid <- snv.window.right - 1
        snv.window.left <- snv.window.right - 2
        
        real.start.codon <- c( snv.window.left, snv.window.mid, snv.window.right )
        
        if( roi.start<=snv.window.left & roi.end>=snv.window.right & nWindow>0 ){
            
            for(i in 1:nWindow){
                
                left.window <- snv.window.left:snv.window.right - 3*i  
                check.bases <- left.window[1]:snv.window.right
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% left.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% left.window], collapse="")=="CAT" ){
                            cryptic.start.codon <- left.window
                        }
                    }
                }
                if( !is.null(cryptic.start.codon) ){ break }
                
                right.window <- snv.window.left:snv.window.right + 3*i  
                check.bases <- snv.window.left:right.window[3]
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% right.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% right.window], collapse="")=="CAT" ){
                            cryptic.start.codon <- right.window
                        }
                    }
                }
                if( !is.null(cryptic.start.codon) ){ break }
                
            }
        }
        
    }
    
    start.codon <- list( real.start.codon=real.start.codon, cryptic.start.codon=cryptic.start.codon )
    return(start.codon)
}


