% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analysis.R
\name{predict_cell_types}
\alias{predict_cell_types}
\title{Predict Average Cell Type Proportions for a Sample}
\usage{
predict_cell_types(
  spatial_data,
  reference,
  sample_name = NULL,
  nfolds = 5,
  transform_input = TRUE,
  normalize_reference = TRUE,
  lambda_selection_rule = "auto",
  alpha = 1,
  lambda_min = 0.001,
  lambda_max = 1,
  lambda_n = 100,
  min_nonzero = 3,
  max_nonzero = 14,
  keep_top_n = 14,
  nonzero_threshold = 0.001,
  generate_plots = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{spatial_data}{A data.frame or matrix containing spatial gene expression data.
Genes should be in row names, and columns should represent spots/barcodes.
Assumes expression values are log-transformed (e.g., log(CPM+1) or log(TPM+1)).}

\item{reference}{A data.frame or matrix containing reference expression data.
Genes should be in row names, cell types should be in column names.
Alternatively, a character string specifying a built-in reference ("human" or "mouse").}

\item{sample_name}{Optional name for the sample (used in plot titles). If NULL, uses "Sample".}

\item{nfolds}{Number of folds for cross-validation in \code{cv.glmnet}. (Default: 5)}

\item{transform_input}{Logical, whether to apply \code{exp(data) - 1} transformation to the
input spatial data. Set to \code{FALSE} if data is already in linear scale (e.g., counts, CPM). (Default: TRUE)}

\item{normalize_reference}{Logical, whether to normalize each cell type in the reference
to have the same total expression. (Default: TRUE)}

\item{lambda_selection_rule}{Character, method for lambda selection. Options are:
"auto" (use glmnet's default lambda sequence) or "custom" (use custom lambda range). (Default: "auto")}

\item{alpha}{The elasticnet mixing parameter, where alpha=1 is the lasso (default) and \code{alpha=0} is ridge.}

\item{lambda_min}{Minimum lambda value for custom lambda sequence (only used when lambda_selection_rule="custom"). (Default: 0.001)}

\item{lambda_max}{Maximum lambda value for custom lambda sequence (only used when lambda_selection_rule="custom"). (Default: 1.0)}

\item{lambda_n}{Number of lambda values in custom sequence (only used when lambda_selection_rule="custom"). (Default: 100)}

\item{min_nonzero}{Minimum number of desired non-zero coefficients for lambda selection. (Default: 3)}

\item{max_nonzero}{Maximum number of desired non-zero coefficients for lambda selection. (Default: 14)}

\item{keep_top_n}{Maximum number of positive coefficients to retain after filtering. If more
coefficients are positive, only the top \code{keep_top_n} are kept. Set to \code{Inf} to disable. (Default: 14)}

\item{nonzero_threshold}{Threshold below which coefficients are considered zero during lambda
selection and final filtering. (Default: 1e-3)}

\item{generate_plots}{Logical, whether to generate CV and coefficient path plots. (Default: TRUE)}

\item{verbose}{Logical, whether to print progress messages. (Default: TRUE)}
}
\value{
A list containing:
\itemize{
\item \code{proportions}: Data frame with columns 'Cell_Type' and 'Proportion'
\item \code{nonzero_celltypes}: Vector of cell type names with non-zero proportions
\item \code{selected_lambda}: The lambda value selected by the algorithm
\item \code{selection_rule}: Whether lambda was selected by "3-14_rule_glmnet", "3-14_rule_custom", or "fallback"
\item \code{common_genes}: Vector of genes used in the analysis
\item \code{cv_plot}: Function to generate cross-validation ggplot (if generate_plots=TRUE)
\item \code{coef_plot}: Function to generate coefficient path ggplot (if generate_plots=TRUE)
}
Returns \code{NULL} if processing fails.
}
\description{
This function takes spatial transcriptomics data for a single sample (potentially
across multiple spots), calculates the average expression, and predicts
average cell type proportions using Lasso regression against a reference dataset.
It applies an exponential transformation to input data, uses a specific rule for
lambda selection (seeking 3-14 non-zero coefficients), filters coefficients,
and normalizes the final proportions to sum to 1.
}
\examples{
# Example 1: Using built-in human reference with glmnet lambda sequence
# Load example human average expression data
load(system.file("extdata", "human_avg_expression.rda", package = "oCELLoc"))

# Run with built-in human reference and glmnet lambda sequence
results_human <- predict_cell_types(
  spatial_data = human_avg_expression,
  reference = "human",
  sample_name = "Human_Example",
  lambda_selection_rule = "auto"
)

# View top results
print(head(results_human$proportions, 10))
print(results_human$nonzero_celltypes)


# Example 2: Using built-in mouse reference with custom lambda sequence
# Load example mouse average expression data  
load(system.file("extdata", "mouse_avg_expression.rda", package = "oCELLoc"))

# Run with built-in mouse reference and custom lambda sequence
results_mouse <- predict_cell_types(
  spatial_data = mouse_avg_expression,
  reference = "mouse",
  sample_name = "Mouse_Example",
  lambda_selection_rule = "custom",
  lambda_min = 0.001,
  lambda_max = 0.5,
  lambda_n = 50
)

# View top results
print(head(results_mouse$proportions, 10))
print(results_mouse$nonzero_celltypes)

}
