rand.test.lm0 <-
  function(x, y, weights = NULL, offset = NULL, 
           anova = FALSE, assign = 1:ncol(x), 
           term.labels = paste("x", 1:ncol(x)),
           method = "perm", homosced = FALSE, lambda = 0, 
           R = 9999, parallel = FALSE, cl = NULL,
           perm.dist = TRUE, na.rm = TRUE){
    # Randomization Test for Regression (lm interface)
    # Nathaniel E. Helwig (helwig@umn.edu)
    # last updated: 2026-01-15
    
    
    #########   INITIAL CHECKS   #########
    
    ### check x and y
    x <- as.matrix(x)
    y <- as.matrix(y)
    nvar <- ncol(y)
    n <- nrow(x)
    p <- ncol(x)
    if(nrow(y) != n) stop("Inputs 'x' and 'y' are incompatible.")
    
    ### multivariate y?
    if(nvar > 1L) stop("Multivariate use of np.lm.test() is not supported.\nUse the np.reg.test() function for multivariate linear model tests.")
      
    ### x names
    xnames <- colnames(x)
    if(is.null(xnames)) xnames <- paste0("x", 1:p)
    
    ### weights and offset
    if(is.null(weights)) {
      weights <- w <- rep(1.0, n)
    } else {
      weights <- as.numeric(weights)
      if(length(weights) != n) stop("Invalid 'weights' input")
      w <- sqrt(weights)
    }
    if(is.null(offset)){
      offset <- o <- rep(0.0, n)
    } else{
      offset <- o <- as.numeric(offset)
      if(length(offset) != n) stop("Invalid 'offset' input")
    }
    
    ### check method
    method <- as.character(method[1])
    method.options <- c("perm", "flip", "both")
    method <- method.options[pmatch(method, method.options)]
    if(is.na(method)) stop("Invalid 'method' input.")
    
    ### check homosced
    homosced <- as.logical(homosced[1])
    
    ### check lambda
    nlambda <- length(lambda)
    if(nlambda == 1L){
      lambda <- rep(lambda, p)
    } else if(nlambda != p){
      warning("length(lambda) != ncol(x)\nUsing lambda <- rep(lambda[1], ncol(x))")
      lambda <- rep(lambda[1], p)
    }
    if(any(lambda < 0)) stop("Input lambda must contain non-negative penalty weights.")
    use.lambda <- ifelse(max(lambda) > 0, TRUE, FALSE)
    
    ### check R
    R <- as.integer(R)
    if(R < 0) {
      stop("Input 'R' must be a non-negative integer.")
    } else if(R == 0 && nvar > 1L){
      stop("Input 'R' must be a positive integer for multivariate tests.")
    }
    
    ### check parallel
    parallel <- as.logical(parallel[1])
    
    ### check 'cl'
    make.cl <- FALSE
    if(parallel){
      if(is.null(cl)){
        make.cl <- TRUE
        cl <- parallel::makeCluster(2L)
      } else {
        if(!any(class(cl) == "cluster")) stop("Input 'cl' must be an object of class 'cluster'.")
      }
    }
    
    ### exact or approximate?
    if(method == "perm"){
      suppressWarnings( nperm <- factorial(n) )
    } else if(method == "sign"){
      suppressWarnings( nperm <- 2^n )
    } else {
      suppressWarnings( np <- factorial(n) )
      suppressWarnings( ns <- 2^n )
      nperm <- np * ns
    }
    exact <- ifelse(nperm <= R + 1L, TRUE, FALSE)
    if(exact){
      if(method == "perm"){
        ix <- permn(n)
        nperm <- ncol(ix)
      } else if(method == "flip"){
        ix <- flipn(n)
        nperm <- ncol(ix)
      } else {
        ixp <- permn(n)
        ixs <- flipn(n)
        igrid <- expand.grid(perm = 1:ncol(ixp), sign = 1:ncol(ixs))
        ix <- apply(igrid, 1, function(u) c(ixp[,u[1]], ixs[,u[2]]))
      }
    }
    
    
    #########   REGRESSION TEST   #########
    
    ### subtract offset from y
    y <- y - o
    
    ### weight data
    xw <- w * x
    yw <- w * y
    
    ### center data
    xwbar <- colMeans(xw)
    ywbar <- colMeans(yw)
    xw <- scale(xw, center = xwbar, scale = FALSE)
    yw <- scale(yw, center = ywbar, scale = FALSE)
    
    ### calculate crossproducts and inverses
    xtx <- crossprod(xw) + diag(n * lambda, nrow = p, ncol = p)
    xtxi <- psdinv(xtx)
    xinv <- tcrossprod(xtxi, xw)
    coefs <- as.numeric(xinv %*% yw)
    names(coefs) <- xnames
    
    ### univariate or multivariate
    if(nvar == 1L){
      
      ## UNIVARIATE TEST
      
      yw <- as.numeric(yw)
      fit <- as.numeric(xw %*% coefs)
      res <- yw - fit
      if(homosced){
        sigsq <- sum(res^2) / (n - p - 1)
        se.coefs <- sqrt(diag(xtxi) / sigsq)
      } else {
        omega <- crossprod(abs(res) * xw)
        siosi <- xtxi %*% omega %*% xtxi
        se.coefs <- sqrt(diag(siosi))
      }
      
      ## observed test statistic
      Tstat <- Tstat.lm0(x = xw, y = yw, homosced = homosced,
                         xtx = xtx, xtxi = xtxi, xinv = xinv,
                         anova = anova, assign = assign,
                         term.labels = term.labels)
      ncoefs <- length(Tstat)
      
      ## permutation distribution
      if(exact){
        
        # parallel or sequential computation?
        if(parallel){
          permdist <- parallel::parCapply(cl = cl, x = ix, 
                                          FUN = Tperm.lm0, 
                                          xmat = xw, yvec = yw, 
                                          method = method, 
                                          homosced = homosced,
                                          exact = exact, xtx = xtx, 
                                          xtxi = xtxi, xinv = xinv,
                                          anova = anova, assign = assign,
                                          term.labels = term.labels)
        } else {
          permdist <- apply(X = ix, MARGIN = 2, 
                            FUN = Tperm.lm0, 
                            xmat = xw, yvec = yw, 
                            method = method, 
                            homosced = homosced,
                            exact = exact, xtx = xtx, 
                            xtxi = xtxi, xinv = xinv,
                            anova = anova, assign = assign,
                            term.labels = term.labels)
        } # end if(parallel)
        permdist <- t(permdist)
        colnames(permdist) <- if(anova) term.labels else xnames
        
      } else {
        
        # approximate permutation test (given input R)
        nperm <- R + 1L
        permdist <- matrix(0.0, nperm, ncoefs)
        colnames(permdist) <- if(anova) term.labels else xnames
        permdist[1,] <- Tstat
        
        # parallel or sequential computation?
        if(parallel){
          permdist[2:nperm,] <- t(parallel::parSapply(cl = cl, X = integer(R), 
                                                      FUN = Tperm.lm0, 
                                                      xmat = xw, yvec = yw, 
                                                      method = method, 
                                                      homosced = homosced,
                                                      exact = exact, xtx = xtx, 
                                                      xtxi = xtxi, xinv = xinv,
                                                      anova = anova, assign = assign,
                                                      term.labels = term.labels))
        } else {
          permdist[2:nperm,] <- t(sapply(X = integer(R),
                                         FUN = Tperm.lm0,
                                         xmat = xw, yvec = yw, 
                                         method = method, 
                                         homosced = homosced,
                                         exact = exact, xtx = xtx, 
                                         xtxi = xtxi, xinv = xinv,
                                         anova = anova, assign = assign,
                                         term.labels = term.labels))
        } # end if(parallel)
        
      } # end if(exact)
      
      ## permutation p-value
      p.value <- rep(0.0, ncoefs)
      names(p.value) <- if(anova) term.labels else xnames
      for(j in 1:ncoefs) p.value[j] <- mean(permdist[,j] >= Tstat[j])
      
      ## intercept
      alpha <- ywbar - sum(xwbar * coefs)
      coefs <- c(alpha, coefs)
      
      ## name coefficients
      names(coefs) <- c("(Intercept)", xnames)
      
    } else {
      
      ## MULTIVARIATE TEST
      
      # TBD...
      
    } # end if(nvar == 1L)
    
    ### return results
    if(make.cl) parallel::stopCluster(cl)
    if(!perm.dist) permdist <- NULL
    if(anova){
      tab <- data.frame(term = term.labels,
                        df = as.integer(table(assign)),
                        statistic = Tstat,
                        p.value = p.value)
    } else {
      tab <- data.frame(term = xnames, 
                        df = rep(1, p),
                        statistic = Tstat,
                        p.value = p.value)
    }
    rownames(tab) <- 1:nrow(tab)
    res <- list(statistic = Tstat, p.value = p.value,
                perm.dist = permdist, method = method, 
                homosced = homosced, lambda = lambda,
                R = nperm - ifelse(exact, 0, 1), exact = exact, 
                coefficients = coefs, se.coef = se.coefs, 
                signif.table = tab, resid = res)
    #if(nvar > 1L) {
    #  res$univariate <- Tuni
    #  res$adj.p.values <- uni.p.value
    #}
    class(res) <- "rand.test.lm0"
    return(res)
    
  } # end rand.test.lm0


### permutation replication (univariate)
Tperm.lm0 <-
  function(i, xmat, yvec, method = "perm", 
           homosced = FALSE, exact = FALSE,
           xtx = NULL, xtxi = NULL, xinv = NULL,
           anova = FALSE, assign = 1:ncol(xmat), 
           term.labels = paste("x", 1:ncol(xmat))){
    n <- nrow(xmat)
    if(method == "perm"){
      if(!exact) i <- sample.int(n)
      yvec <- yvec[i]
    } else if(method == "flip"){
      if(!exact) i <- sample(c(-1, 1), size = n, replace = TRUE)
      yvec <- yvec * i
    } else {
      if(!exact) i <- c(sample.int(n), sample(c(-1, 1), size = n, replace = TRUE))
      yvec <- yvec * i[(n+1):(2*n)]   # flip sign
      yvec <- yvec[i[1:n]]            # permute
    }
    Tstat <- Tstat.lm0(x = xmat, y = yvec, homosced = homosced,
                       xtx = xtx, xtxi = xtxi, xinv = xinv,
                       anova = anova, assign = assign,
                       term.labels = term.labels)
    return(Tstat)
  } # end Tperm.lm0.R

Tstat.lm0 <-
  function(x, y, homosced = FALSE, 
           xtx = NULL, xtxi = NULL, xinv = NULL,
           anova = FALSE, assign = 1:ncol(x), 
           term.labels = paste("x", 1:ncol(x))){
    
    # dimensions
    n <- nrow(x)
    p <- ncol(x)
    
    # coefficients
    if(is.null(xtx)) xtx <- crossprod(x)
    if(is.null(xtxi)) xtxi <- psdinv(xtx)
    if(is.null(xinv)) xinv <- tcrossprod(xtxi, x)
    beta <- xinv %*% y
    
    # fitted values and residuals
    fit <- as.numeric(x %*% beta)
    res <- y - fit
    
    # test statistic
    if(anova){
      nterms <- length(term.labels)
      Tstat <- rep(0.0, nterms)
      names(Tstat) <- term.labels
      if(homosced) {
        sigsq <- sum(res^2) / (n - p - 1)
        for(j in 1:nterms){
          id <- which(assign == j)
          if(length(id) == 1L){
            top <- beta[id]^2 / xtxi[id, id]
            Tstat[j] <- top / sigsq
          } else {
            top <- t(beta[id]) %*% psdinv(xtxi[id, id, drop = FALSE]) %*% beta[id]
            Tstat[j] <- top / (length(id) * sigsq)
          }
        }
      } else {
        omega <- crossprod(abs(res) * x)
        siosi <- xtxi %*% omega %*% xtxi
        for(j in 1:nterms){
          id <- which(assign == j)
          if(length(id) == 1L){
            Tstat[j] <- beta[id]^2 / siosi[id, id]
          } else {
            Tstat[j] <- t(beta[id]) %*% psdinv(siosi[id, id, drop = FALSE]) %*% beta[id]
          }
          
        }
      } # end if(homosced)
    } else {
      if(homosced) {
        sigsq <- sum(res^2) / (n - p - 1)
        Tstat <- (beta^2 / diag(xtxi)) / sigsq
      } else {
        omega <- crossprod(abs(res) * x)
        siosi <- xtxi %*% omega %*% xtxi
        Tstat <- (beta^2 / diag(siosi))
      }
    }
    
    # return
    as.numeric(Tstat)
    
  } # end Tstat.lm0