#' Fit Nominal Data to Theoretical Shapes Using AIC (Safe Exponential)
#'
#' Computes the multinomial log-likelihood of observed counts against
#' four theoretical distributions (uniform, triangular, normal-like,
#' and exponential/Pareto-like) and returns AIC and DeltaAIC values.
#'
#' @importFrom stats lm coef dnorm
#' @param df A data.frame or tibble containing the nominal variable.
#' @param var Character string giving the name of the nominal variable in `df`.
#' @param rate_exp Numeric. Default exponential rate. Only used if tail not clearly exponential.
#' @param eps Small numeric value added to probabilities to avoid log(0). Default is 1e-12.
#'
#' @return A data.frame with columns: Shape, AIC, DeltaAIC.
#' @name shape_aic
#' @export

shape_aic <- function(df, var, rate_exp = 0.7, eps = 1e-12) {

  # ---- Helper: centered positions (IDENTICAL to plotting) ----
  .center_positions <- function(k) {
    center <- ceiling(k / 2)
    pos <- integer(k)
    pos[1] <- center
    r <- l <- center
    side <- 1
    for (i in 2:k) {
      if (side == 1) { r <- r + 1; pos[i] <- r }
      else { l <- l - 1; pos[i] <- l }
      side <- -side
    }
    pos
  }

  # ---- Observed frequencies ----
  obs_table <- as.data.frame(table(df[[var]]), stringsAsFactors = FALSE)
  names(obs_table) <- c("Category", "Freq")

  # Sort descending (most frequent first)
  obs_table <- obs_table[order(-obs_table$Freq), ]

  k <- nrow(obs_table)
  n <- sum(obs_table$Freq)

  # ---- CENTER THE OBSERVED COUNTS ----
  centered_pos <- .center_positions(k)
  obs_centered <- numeric(k)
  obs_centered[centered_pos] <- obs_table$Freq

  # ---- Theoretical distributions (IDENTICAL to plot) ----

  # Uniform
  theo_uniform <- rep(n / k, k)

  # Triangular
  mid <- ceiling(k / 2)
  theo_triangular <- c(seq_len(mid), rev(seq_len(k - mid)))[1:k]
  theo_triangular <- theo_triangular / sum(theo_triangular) * n

  # Exponential (Pareto-like)
  raw_exp <- exp(-rate_exp * seq_len(k))
  raw_exp <- raw_exp / sum(raw_exp) * n
  theo_exponential <- numeric(k)
  theo_exponential[centered_pos] <- sort(raw_exp, decreasing = TRUE)

  # Normal-like
  d <- abs(seq_len(k) - ceiling(k / 2))
  theo_normal <- dnorm(d, mean = 0, sd = k / 6)
  theo_normal <- theo_normal / sum(theo_normal) * n

  theo_list <- list(
    Uniform = theo_uniform,
    Triangular = theo_triangular,
    Exponential = theo_exponential,
    Normal = theo_normal
  )

  # ---- Multinomial log-likelihood ----
  logLik_multinom <- function(obs, p) {
    sum(obs * log(p + eps))
  }

  # ---- AIC (no free parameters) ----
  AIC_vals <- sapply(names(theo_list), function(name) {
    p <- theo_list[[name]] / sum(theo_list[[name]])
    -2 * logLik_multinom(obs_centered, p)
  })

  out <- data.frame(
    Shape = names(AIC_vals),
    AIC = as.numeric(AIC_vals),
    stringsAsFactors = FALSE
  )

  out <- out[order(out$AIC), ]
  out$DeltaAIC <- out$AIC - min(out$AIC)
  rownames(out) <- NULL
  out
}
