% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pmodel.R
\name{pmodel}
\alias{pmodel}
\title{Personalised model}
\usage{
pmodel(
  x = NULL,
  model = NULL,
  newdata = NULL,
  OOB = TRUE,
  fun = coef,
  return_attr = c("modelcall", "data", "similarity")
)
}
\arguments{
\item{x}{cforest object or matrix of weights.}

\item{model}{model object. If NULL the model in \code{x$info$model} is used.}

\item{newdata}{new data. If NULL cforest learning data is used. Ignored if \code{x} is a matrix.}

\item{OOB}{In case of using the learning data, should patient similarities be
computed out of bag?}

\item{fun}{function to apply on the personalised model before returning. The
default \code{coef} returns a matrix of personalised coefficients. For returning
the model objects use \code{identity}.}

\item{return_attr}{which attributes to add to the object returned. If it contains
\code{"modelcall"} the call of the base model is returned, if it contains 
\code{"data"} the data, and if it contains \code{"similarity"} the matrix of 
similarity weights is added.}
}
\value{
depends on fun.
}
\description{
Compute personalised models from cforest object.
}
\examples{
library("model4you")

if(require("mvtnorm") & require("survival")) {
  
  ## function to simulate the data
  sim_data <- function(n = 500, p = 10, beta = 3, sd = 1){
    
    ## treatment
    lev <- c("C", "A")
    a <- rep(factor(lev, labels = lev, levels = lev), length = n)
    
    ## correlated z variables
    sigma <- diag(p) 
    sigma[sigma == 0] <- 0.2
    ztemp <- rmvnorm(n, sigma = sigma)
    z <- (pnorm(ztemp) * 2 * pi) - pi  
    colnames(z) <- paste0("z", 1:ncol(z))
    z1 <- z[,1]
    
    ## outcome
    y <- 7 + 0.2 * (a \%in\% "A") + beta * cos(z1) * (a \%in\% "A") + rnorm(n, 0, sd)
    
    data.frame(y = y, a = a, z)
  }
  
  ## simulate data
  set.seed(123)
  beta <- 3
  ntrain <- 500
  ntest <- 50
  simdata <- simdata_s <- sim_data(p = 5, beta = beta, n = ntrain)
  tsimdata <- tsimdata_s <- sim_data(p = 5, beta = beta, n = ntest)
  simdata_s$cens <- rep(1, ntrain)
  tsimdata_s$cens <- rep(1, ntest)
  
  ## base model
  basemodel_lm <- lm(y ~ a, data = simdata)
  
  ## forest
  frst_lm <- pmforest(basemodel_lm, ntree = 20, 
                      perturb = list(replace = FALSE, fraction = 0.632),
                      control = ctree_control(mincriterion = 0))
  
  ## personalised models
  # (1) return the model objects
  pmodels_lm <- pmodel(x = frst_lm, newdata = tsimdata, fun = identity)
  class(pmodels_lm)
  # (2) return coefficients only (default)
  coefs_lm <- pmodel(x = frst_lm, newdata = tsimdata)
  
  # compare predictive objective functions of personalised models versus
  # base model
  sum(objfun(pmodels_lm)) # -RSS personalised models
  sum(objfun(basemodel_lm, newdata = tsimdata)) # -RSS base model
  
  
  if(require("ggplot2")) {
    ## dependence plot
    dp_lm <- cbind(coefs_lm, tsimdata)
    ggplot(tsimdata) +
      stat_function(fun = function(z1) 0.2 + beta * cos(z1), 
                    aes(color = "true treatment\neffect")) +
      geom_point(data = dp_lm, 
                 aes(y = aA, x = z1, color = "estimates lm"), 
                 alpha = 0.5)  +
      ylab("treatment effect") + 
      xlab("patient characteristic z1")
  }
}

}
