% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/define_case.R
\name{define_case}
\alias{define_case}
\title{Identify diseases/events from administrative records}
\usage{
define_case(
  data,
  vars,
  match = "in",
  vals,
  clnt_id,
  n_per_clnt = 1,
  date_var = NULL,
  apart = NULL,
  within = NULL,
  uid = NULL,
  excl_vals = NULL,
  excl_args = NULL,
  keep = c("all", "first", "last"),
  if_all = FALSE,
  mode = c("flag", "filter"),
  force_collect = FALSE,
  verbose = getOption("healthdb.verbose"),
  ...
)
}
\arguments{
\item{data}{Data.frames or remote tables (e.g., from \code{\link[dbplyr:tbl_sql]{dbplyr::tbl_sql()}})}

\item{vars}{An expression passing to \code{\link[dplyr:select]{dplyr::select()}}. It can be quoted/unquoted column names, or helper functions, such as \code{\link[dplyr:reexports]{dplyr::starts_with()}}.}

\item{match}{One of "in", "start", "regex", "like", "between", and "glue_sql". It determines how values would be matched. See \code{\link[=identify_row]{identify_row()}} for detail.}

\item{vals}{Depending on \code{match}, it takes different input. See \code{\link[=identify_row]{identify_row()}}.}

\item{clnt_id}{Grouping variable (quoted/unquoted).}

\item{n_per_clnt}{A single number specifying the minimum number of group size.}

\item{date_var}{Variable name (quoted/unquoted) for the dates to be interpreted.}

\item{apart}{An integer specifying the minimum gap (in days) between adjacent dates in a draw.}

\item{within}{An integer specifying the maximum time span (in days) of a draw.}

\item{uid}{Variable name for a unique row identifier. It is necessary for SQL to produce consistent result based on sorting.}

\item{excl_vals}{Same as \code{vals} but clients/groups with these values are going to be removed from the result. This is intended for exclusion criteria of a case definition.}

\item{excl_args}{A named list of arguments passing to the second \code{\link[=identify_row]{identify_row()}} call for \code{excl_vals}. If not supplied, \code{var}, \code{match} and \code{if_all} of the first call will be re-used.}

\item{keep}{One of:
\itemize{
\item "first" (keeping each client's earliest record),
\item "last" (keeping the latest),
\item and "all" (keeping all relevant records, default).
\item Note that "first"/"last" should not be used with "flag" mode.
}}

\item{if_all}{A logical for whether combining the predicates (if multiple columns were selected by vars) with AND instead of OR. Default is FALSE, e.g., var1 in vals OR var2 in vals.}

\item{mode}{Either:
\itemize{
\item "flag" - add new columns starting with "flag_" indicating if the client met the condition,
\item or "filter" - remove clients that did not meet the condition from the data.
\item This will be passed to both \code{\link[=restrict_n]{restrict_n()}} AND \code{\link[=restrict_date]{restrict_date()}}. Default is "flag".
}}

\item{force_collect}{A logical for whether force downloading the result table if it is not a local data.frame. Downloading data could be slow, so the user has to opt in; default is FALSE.}

\item{verbose}{A logical for whether printing explanation for the operation. Default is fetching from options. Use \code{options(healthdb.verbose = FALSE)} to suppress once and for all.}

\item{...}{Additional arguments, e.g., \code{flag_at}, passing to \code{\link[=restrict_date]{restrict_date()}}.}
}
\value{
A subset of input data satisfied the specified case definition.
}
\description{
This function is a composite of \code{\link[=identify_row]{identify_row()}}, \code{\link[=exclude]{exclude()}}, \code{\link[=restrict_n]{restrict_n()}}, and \code{\link[=restrict_date]{restrict_date()}}. It is aimed to implement common case definitions in epidemiological studies using administrative database as a one-shot big query. The intended use case is for definitions in the form of, e.g., two or more physician visits with some diagnostic code at least 30 days apart within two years. The component functions mentioned above are chained in the following order if all arguments were supplied: \code{identify_row(vals) \%>\% exclude(identify_row(excl_vals), by = clnt_id) \%>\% restrict_n() \%>\% restrict_date()}. Only necessary steps in the chain will be ran if some arguments are missing, see the verbose output for what was done. Note that if \code{date_var} is supplied, \code{n_per_clnt} will be counted by distinct dates instead of number of records.
}
\examples{
sample_size <- 30
df <- data.frame(
  clnt_id = rep(1:3, each = 10),
  service_dt = sample(seq(as.Date("2020-01-01"), as.Date("2020-01-31"), by = 1),
    size = sample_size, replace = TRUE
  ),
  diagx = sample(letters, size = sample_size, replace = TRUE),
  diagx_1 = sample(c(NA, letters), size = sample_size, replace = TRUE),
  diagx_2 = sample(c(NA, letters), size = sample_size, replace = TRUE)
)

# define from one source
define_case(df,
  vars = starts_with("diagx"), "in", vals = letters[1:4],
  clnt_id = clnt_id, date_var = service_dt,
  excl_args = list(if_all = TRUE),
  # remove non-case
  mode = "filter",
  # keeping the first record
  keep = "first"
)

# multiple sources with purrr::pmap
# arguments with length = 1 will be recycle to match the number of sources
# wrap expressions/unquoted variables with bquote(),
# or rlang:exprs() to prevent immediate evaluation,
# or just use quoted variable names
purrr::pmap(
  list(
    data = list(df, df),
    vars = rlang::exprs(starts_with("diagx")),
    match = c("in", "start"),
    vals = list(letters[1:4], letters[5:10]),
    clnt_id = list(bquote(clnt_id)), n_per_clnt = c(2, 3),
    date_var = "service_dt",
    excl_vals = list(letters[11:13], letters[14:16]),
    excl_args = list(list(if_all = TRUE), list(if_all = FALSE))
  ),
  define_case
)
}
