/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5WBprivate.h" 

struct H5WB_t {
    void  *wrapped_buf;  
    size_t wrapped_size; 
    void  *actual_buf;   
    size_t actual_size;  
    size_t alloc_size;   
};

H5FL_DEFINE_STATIC(H5WB_t);

H5FL_BLK_DEFINE_STATIC(extra_buf);

H5WB_t *
H5WB_wrap(void *buf, size_t buf_size)
{
    H5WB_t *wb = NULL; 
    H5WB_t *ret_value; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(buf);
    assert(buf_size);

    
    if (NULL == (wb = H5FL_MALLOC(H5WB_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for wrapped buffer info");

    
    wb->wrapped_buf  = buf;
    wb->wrapped_size = buf_size;

    
    wb->actual_buf  = NULL;
    wb->actual_size = 0;
    wb->alloc_size  = 0;

    
    ret_value = wb;

done:
    
    if (!ret_value && wb)
        wb = H5FL_FREE(H5WB_t, wb);

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5WB_actual(H5WB_t *wb, size_t need)
{
    void *ret_value; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(wb);
    assert(wb->wrapped_buf);

    
    if (wb->actual_buf && wb->actual_buf != wb->wrapped_buf) {
        
        assert(wb->actual_size > wb->wrapped_size);

        
        if (need <= wb->alloc_size)
            HGOTO_DONE(wb->actual_buf);
        
        else
            wb->actual_buf = H5FL_BLK_FREE(extra_buf, wb->actual_buf);
    } 

    
    if (need > wb->wrapped_size) {
        
        if (NULL == (wb->actual_buf = H5FL_BLK_MALLOC(extra_buf, need)))
            HGOTO_ERROR(H5E_ATTR, H5E_NOSPACE, NULL, "memory allocation failed");

        
        wb->alloc_size = need;
    } 
    else {
        
        wb->actual_buf = wb->wrapped_buf;
        wb->alloc_size = 0;
    } 

    
    ret_value = wb->actual_buf;

done:
    
    if (ret_value)
        wb->actual_size = need;

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5WB_actual_clear(H5WB_t *wb, size_t need)
{
    void *ret_value; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(wb);
    assert(wb->wrapped_buf);

    
    if (NULL == (ret_value = H5WB_actual(wb, need)))
        HGOTO_ERROR(H5E_ATTR, H5E_NOSPACE, NULL, "memory allocation failed");

    
    memset(ret_value, 0, need);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5WB_unwrap(H5WB_t *wb)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(wb);
    assert(wb->wrapped_buf);

    
    if (wb->actual_buf && wb->actual_buf != wb->wrapped_buf) {
        
        assert(wb->actual_size > wb->wrapped_size);

        wb->actual_buf = H5FL_BLK_FREE(extra_buf, wb->actual_buf);
    } 

    
    wb = H5FL_FREE(H5WB_t, wb);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
