/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5TSpkg.h"    

#ifdef H5_HAVE_THREADS

typedef int64_t H5TS_rec_entry_count_t;

#if H5TS_ENABLE_REC_RWLOCK_STATS

static void
H5TS__update_stats_rdlock(H5TS_rec_rwlock_t *lock, const H5TS_rec_entry_count_t *count)
{
    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    assert(lock);
    assert(H5TS_REC_RWLOCK_READ == lock->lock_type);
    assert(count);
    assert(*count >= 1);

    lock->stats.read_locks_granted++;

    if (*count == 1) {
        lock->stats.real_read_locks_granted++;
        if (lock->reader_thread_count > lock->stats.max_read_locks)
            lock->stats.max_read_locks = lock->reader_thread_count;
    }

    if (*count > lock->stats.max_read_lock_recursion_depth)
        lock->stats.max_read_lock_recursion_depth = *count;

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 

static void
H5TS__update_stats_rd_lock_delay(H5TS_rec_rwlock_t *lock)
{
    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    assert(lock);

    lock->stats.read_locks_delayed++;

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 

static void
H5TS__update_stats_rd_unlock(H5TS_rec_rwlock_t *lock, const H5TS_rec_entry_count_t *count)
{
    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    assert(lock);
    assert(H5TS_REC_RWLOCK_READ == lock->lock_type);
    assert(count);
    assert(*count >= 0);

    lock->stats.read_locks_released++;

    if (*count == 0)
        lock->stats.real_read_locks_released++;

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 

static void
H5TS__update_stats_wr_lock(H5TS_rec_rwlock_t *lock)
{
    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    assert(lock);
    assert(H5TS_REC_RWLOCK_WRITE == lock->lock_type);
    assert(lock->rec_write_lock_count >= 1);

    lock->stats.write_locks_granted++;

    if (lock->rec_write_lock_count == 1) {
        lock->stats.real_write_locks_granted++;
        if (lock->rec_write_lock_count > lock->stats.max_write_locks)
            lock->stats.max_write_locks = lock->rec_write_lock_count;
    }

    if (lock->rec_write_lock_count > lock->stats.max_write_lock_recursion_depth)
        lock->stats.max_write_lock_recursion_depth = lock->rec_write_lock_count;

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 

static void
H5TS__update_stats_wr_lock_delay(H5TS_rec_rwlock_t *lock)
{
    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    assert(lock);

    lock->stats.write_locks_delayed++;

    if (lock->stats.max_write_locks_pending <= lock->waiting_writers_count)
        lock->stats.max_write_locks_pending = lock->waiting_writers_count + 1;

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 

static void
H5TS__update_stats_wr_unlock(H5TS_rec_rwlock_t *lock)
{
    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    assert(lock);
    assert(H5TS_REC_RWLOCK_WRITE == lock->lock_type);
    assert(lock->rec_write_lock_count >= 0);

    lock->stats.write_locks_released++;

    if (lock->rec_write_lock_count == 0)
        lock->stats.real_write_locks_released++;

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 

herr_t
H5TS__rec_rwlock_get_stats(H5TS_rec_rwlock_t *lock, H5TS_rec_rwlock_stats_t *stats)
{
    bool   have_mutex = false;
    herr_t ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock || NULL == stats))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&lock->mutex)))
        HGOTO_DONE(FAIL);
    have_mutex = true;

    
    *stats = lock->stats;

done:
    if (H5_LIKELY(have_mutex))
        if (H5_UNLIKELY(H5TS_mutex_unlock(&lock->mutex) < 0))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__rec_rwlock_reset_stats(H5TS_rec_rwlock_t *lock)
{
    bool   have_mutex = false;
    herr_t ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&lock->mutex) < 0))
        HGOTO_DONE(FAIL);
    have_mutex = true;

    
    memset(&lock->stats, 0, sizeof(lock->stats));

done:
    if (H5_LIKELY(have_mutex))
        if (H5_UNLIKELY(H5TS_mutex_unlock(&lock->mutex) < 0))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__rec_rwlock_print_stats(const char *header_str, H5TS_rec_rwlock_stats_t *stats)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == header_str || NULL == stats))
        HGOTO_DONE(FAIL);

    Rfprintf(Rstdout, "\n\n%s\n\n", header_str);
    Rfprintf(Rstdout, "  read_locks_granted             = %" PRId64 "\n", stats->read_locks_granted);
    Rfprintf(Rstdout, "  read_locks_released            = %" PRId64 "\n", stats->read_locks_released);
    Rfprintf(Rstdout, "  real_read_locks_granted        = %" PRId64 "\n", stats->real_read_locks_granted);
    Rfprintf(Rstdout, "  real_read_locks_released       = %" PRId64 "\n", stats->real_read_locks_released);
    Rfprintf(Rstdout, "  max_read_locks                 = %" PRId64 "\n", stats->max_read_locks);
    Rfprintf(Rstdout, "  max_read_lock_recursion_depth  = %" PRId64 "\n", stats->max_read_lock_recursion_depth);
    Rfprintf(Rstdout, "  read_locks_delayed             = %" PRId64 "\n", stats->read_locks_delayed);
    Rfprintf(Rstdout, "  write_locks_granted            = %" PRId64 "\n", stats->write_locks_granted);
    Rfprintf(Rstdout, "  write_locks_released           = %" PRId64 "\n", stats->write_locks_released);
    Rfprintf(Rstdout, "  real_write_locks_granted       = %" PRId64 "\n", stats->real_write_locks_granted);
    Rfprintf(Rstdout, "  real_write_locks_released      = %" PRId64 "\n", stats->real_write_locks_released);
    Rfprintf(Rstdout, "  max_write_locks                = %" PRId64 "\n", stats->max_write_locks);
    Rfprintf(Rstdout, "  max_write_lock_recursion_depth = %" PRId64 "\n",
            stats->max_write_lock_recursion_depth);
    Rfprintf(Rstdout, "  write_locks_delayed            = %" PRId64 "\n", stats->write_locks_delayed);
    Rfprintf(Rstdout, "  max_write_locks_pending        = %" PRId64 "\n\n", stats->max_write_locks_pending);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#endif 

herr_t
H5TS__rec_rwlock_init(H5TS_rec_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

#ifdef H5_HAVE_WIN_THREADS
    
    HGOTO_DONE(FAIL);
#else
    
    memset(lock, 0, sizeof(*lock));
    HDcompile_assert(H5TS_REC_RWLOCK_UNUSED == 0);
    if (H5_UNLIKELY(H5TS_mutex_init(&lock->mutex, H5TS_MUTEX_TYPE_PLAIN) < 0))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(H5TS_cond_init(&lock->writers_cv) < 0))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(H5TS_cond_init(&lock->readers_cv) < 0))
        HGOTO_DONE(FAIL);
#endif

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__rec_rwlock_destroy(H5TS_rec_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_destroy(&lock->mutex) < 0))
        ret_value = FAIL;
    if (H5_UNLIKELY(H5TS_cond_destroy(&lock->readers_cv) < 0))
        ret_value = FAIL;
    if (H5_UNLIKELY(H5TS_cond_destroy(&lock->writers_cv) < 0))
        ret_value = FAIL;
    if (lock->is_key_registered)
        if (H5_UNLIKELY(H5TS_key_delete(lock->rec_read_lock_count_key) < 0))
            ret_value = FAIL;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__rec_rwlock_rdlock(H5TS_rec_rwlock_t *lock)
{
    H5TS_rec_entry_count_t *count;
    H5TS_thread_t           my_thread  = H5TS_thread_self();
    bool                    have_mutex = false;
    herr_t                  ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&lock->mutex) < 0))
        HGOTO_DONE(FAIL);
    have_mutex = true;

    
    if (H5_UNLIKELY(H5TS_REC_RWLOCK_WRITE == lock->lock_type &&
                    H5TS_thread_equal(my_thread, lock->write_thread)))
        HGOTO_DONE(FAIL);

    
    if (!lock->is_key_registered) {
        if (H5_UNLIKELY(H5TS_key_create(&lock->rec_read_lock_count_key, free) < 0))
            HGOTO_DONE(FAIL);
        lock->is_key_registered = true;
        count                   = NULL;
    }
    else if (H5_UNLIKELY(H5TS_key_get_value(lock->rec_read_lock_count_key, (void **)&count) < 0))
        HGOTO_DONE(FAIL);
    if (NULL == count) {
        if (H5_UNLIKELY(NULL == (count = calloc(1, sizeof(*count)))))
            HGOTO_DONE(FAIL);
        if (H5_UNLIKELY(H5TS_key_set_value(lock->rec_read_lock_count_key, (void *)count) < 0))
            HGOTO_DONE(FAIL);
    }

    if (*count > 0) { 
        assert(H5TS_REC_RWLOCK_READ == lock->lock_type);
        assert(lock->reader_thread_count > 0 && lock->rec_write_lock_count == 0);
    }
    else { 
        
        if (H5TS_REC_RWLOCK_WRITE == lock->lock_type) {
#if H5TS_ENABLE_REC_RWLOCK_STATS
            H5TS__update_stats_rd_lock_delay(lock);
#endif

            do {
                if (H5_UNLIKELY(H5TS_cond_wait(&lock->readers_cv, &lock->mutex) < 0))
                    HGOTO_DONE(FAIL);
            } while (H5TS_REC_RWLOCK_WRITE == lock->lock_type);
        }

        
        lock->lock_type = H5TS_REC_RWLOCK_READ;
        lock->reader_thread_count++;
    }

    
    (*count)++;
#if H5TS_ENABLE_REC_RWLOCK_STATS
    H5TS__update_stats_rdlock(lock, count);
#endif

done:
    if (H5_LIKELY(have_mutex))
        if (H5_UNLIKELY(H5TS_mutex_unlock(&lock->mutex) < 0))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__rec_rwlock_wrlock(H5TS_rec_rwlock_t *lock)
{
    H5TS_thread_t my_thread  = H5TS_thread_self();
    bool          have_mutex = false;
    herr_t        ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&lock->mutex) < 0))
        HGOTO_DONE(FAIL);
    have_mutex = true;

    
    if (H5TS_REC_RWLOCK_WRITE != lock->lock_type || !H5TS_thread_equal(my_thread, lock->write_thread)) {
        
        if (H5TS_REC_RWLOCK_READ == lock->lock_type) {
            H5TS_rec_entry_count_t *count;

            
            assert(lock->is_key_registered);

            
            if (H5_UNLIKELY(H5TS_key_get_value(lock->rec_read_lock_count_key, (void **)&count) < 0))
                HGOTO_DONE(FAIL);
            if (H5_UNLIKELY(NULL != count && *count > 0))
                HGOTO_DONE(FAIL);
        }

        
        if (H5TS_REC_RWLOCK_UNUSED != lock->lock_type) {
#if H5TS_ENABLE_REC_RWLOCK_STATS
            H5TS__update_stats_wr_lock_delay(lock);
#endif

            do {
                int result;

                lock->waiting_writers_count++;
                result = H5TS_cond_wait(&lock->writers_cv, &lock->mutex);
                lock->waiting_writers_count--;
                if (H5_UNLIKELY(result != 0))
                    HGOTO_DONE(FAIL);
            } while (H5TS_REC_RWLOCK_UNUSED != lock->lock_type);
        }

        
        lock->lock_type    = H5TS_REC_RWLOCK_WRITE;
        lock->write_thread = my_thread;
    }

    
    lock->rec_write_lock_count++;
#if H5TS_ENABLE_REC_RWLOCK_STATS
    H5TS__update_stats_wr_lock(lock);
#endif

done:
    if (H5_LIKELY(have_mutex))
        if (H5_UNLIKELY(H5TS_mutex_unlock(&lock->mutex) < 0))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__rec_rwlock_rdunlock(H5TS_rec_rwlock_t *lock)
{
    H5TS_rec_entry_count_t *count;
    bool                    have_mutex = false;
    herr_t                  ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&lock->mutex) < 0))
        HGOTO_DONE(FAIL);
    have_mutex = true;

    
    if (H5_UNLIKELY(H5TS_REC_RWLOCK_READ != lock->lock_type))
        HGOTO_DONE(FAIL);

    
    assert(lock->is_key_registered);
    assert(lock->reader_thread_count > 0);
    assert(0 == lock->rec_write_lock_count);
    if (H5_UNLIKELY(H5TS_key_get_value(lock->rec_read_lock_count_key, (void **)&count) < 0))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(NULL == count))
        HGOTO_DONE(FAIL);
    assert(*count > 0);

    
    (*count)--;
#if H5TS_ENABLE_REC_RWLOCK_STATS
    H5TS__update_stats_rd_unlock(lock, count);
#endif

    
    if (0 == *count) {
        
        lock->reader_thread_count--;

        
        if (0 == lock->reader_thread_count) {
            lock->lock_type = H5TS_REC_RWLOCK_UNUSED;

            
            
            if (lock->waiting_writers_count > 0) {
                if (H5_UNLIKELY(H5TS_cond_signal(&lock->writers_cv) < 0))
                    HGOTO_DONE(FAIL);
            }
            else {
                if (H5_UNLIKELY(H5TS_cond_broadcast(&lock->readers_cv) < 0))
                    HGOTO_DONE(FAIL);
            }
        }
    }

done:
    if (H5_LIKELY(have_mutex))
        if (H5_UNLIKELY(H5TS_mutex_unlock(&lock->mutex) < 0))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__rec_rwlock_wrunlock(H5TS_rec_rwlock_t *lock)
{
    bool   have_mutex = false;
    herr_t ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&lock->mutex) < 0))
        HGOTO_DONE(FAIL);
    have_mutex = true;

    
    if (H5_UNLIKELY(H5TS_REC_RWLOCK_WRITE != lock->lock_type))
        HGOTO_DONE(FAIL);

    
    assert(0 == lock->reader_thread_count);
    assert(lock->rec_write_lock_count > 0);

    
    lock->rec_write_lock_count--;
#if H5TS_ENABLE_REC_RWLOCK_STATS
    H5TS__update_stats_wr_unlock(lock);
#endif

    
    if (0 == lock->rec_write_lock_count) {
        lock->lock_type = H5TS_REC_RWLOCK_UNUSED;

        
        
        if (lock->waiting_writers_count > 0) {
            if (H5_UNLIKELY(H5TS_cond_signal(&lock->writers_cv) < 0))
                HGOTO_DONE(FAIL);
        }
        else {
            if (H5_UNLIKELY(H5TS_cond_broadcast(&lock->readers_cv) < 0))
                HGOTO_DONE(FAIL);
        }
    }

done:
    if (H5_LIKELY(have_mutex))
        if (H5_UNLIKELY(H5TS_mutex_unlock(&lock->mutex) < 0))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#endif 
