/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Smodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5Spkg.h"      
#include "H5VMprivate.h" 

static herr_t   H5S__all_copy(H5S_t *dst, const H5S_t *src, bool share_selection);
static herr_t   H5S__all_release(H5S_t *space);
static htri_t   H5S__all_is_valid(const H5S_t *space);
static hssize_t H5S__all_serial_size(H5S_t *space);
static herr_t   H5S__all_serialize(H5S_t *space, uint8_t **p);
static herr_t   H5S__all_deserialize(H5S_t **space, const uint8_t **p, const size_t p_size, bool skip);
static herr_t   H5S__all_bounds(const H5S_t *space, hsize_t *start, hsize_t *end);
static herr_t   H5S__all_offset(const H5S_t *space, hsize_t *off);
static int      H5S__all_unlim_dim(const H5S_t *space);
static htri_t   H5S__all_is_contiguous(const H5S_t *space);
static htri_t   H5S__all_is_single(const H5S_t *space);
static htri_t   H5S__all_is_regular(H5S_t *space);
static htri_t   H5S__all_shape_same(H5S_t *space1, H5S_t *space2);
static htri_t   H5S__all_intersect_block(H5S_t *space, const hsize_t *start, const hsize_t *end);
static herr_t   H5S__all_adjust_u(H5S_t *space, const hsize_t *offset);
static herr_t   H5S__all_adjust_s(H5S_t *space, const hssize_t *offset);
static herr_t   H5S__all_project_scalar(const H5S_t *space, hsize_t *offset);
static herr_t   H5S__all_project_simple(const H5S_t *space, H5S_t *new_space, hsize_t *offset);
static herr_t   H5S__all_iter_init(H5S_t *space, H5S_sel_iter_t *iter);

static herr_t  H5S__all_iter_coords(const H5S_sel_iter_t *iter, hsize_t *coords);
static herr_t  H5S__all_iter_block(const H5S_sel_iter_t *iter, hsize_t *start, hsize_t *end);
static hsize_t H5S__all_iter_nelmts(const H5S_sel_iter_t *iter);
static htri_t  H5S__all_iter_has_next_block(const H5S_sel_iter_t *iter);
static herr_t  H5S__all_iter_next(H5S_sel_iter_t *sel_iter, size_t nelem);
static herr_t  H5S__all_iter_next_block(H5S_sel_iter_t *sel_iter);
static herr_t  H5S__all_iter_get_seq_list(H5S_sel_iter_t *iter, size_t maxseq, size_t maxbytes, size_t *nseq,
                                          size_t *nbytes, hsize_t *off, size_t *len);
static herr_t  H5S__all_iter_release(H5S_sel_iter_t *sel_iter);

const H5S_select_class_t H5S_sel_all[1] = {{
    H5S_SEL_ALL,

    
    H5S__all_copy,
    H5S__all_release,
    H5S__all_is_valid,
    H5S__all_serial_size,
    H5S__all_serialize,
    H5S__all_deserialize,
    H5S__all_bounds,
    H5S__all_offset,
    H5S__all_unlim_dim,
    NULL,
    H5S__all_is_contiguous,
    H5S__all_is_single,
    H5S__all_is_regular,
    H5S__all_shape_same,
    H5S__all_intersect_block,
    H5S__all_adjust_u,
    H5S__all_adjust_s,
    H5S__all_project_scalar,
    H5S__all_project_simple,
    H5S__all_iter_init,
}};

static const H5S_sel_iter_class_t H5S_sel_iter_all[1] = {{
    H5S_SEL_ALL,

    
    H5S__all_iter_coords,
    H5S__all_iter_block,
    H5S__all_iter_nelmts,
    H5S__all_iter_has_next_block,
    H5S__all_iter_next,
    H5S__all_iter_next_block,
    H5S__all_iter_get_seq_list,
    H5S__all_iter_release,
}};

static herr_t
H5S__all_iter_init(H5S_t H5_ATTR_UNUSED *space, H5S_sel_iter_t *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space && H5S_SEL_ALL == H5S_GET_SELECT_TYPE(space));
    assert(iter);

    
    iter->u.all.elmt_offset = 0;
    iter->u.all.byte_offset = 0;

    
    iter->type = H5S_sel_iter_all;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_iter_coords(const H5S_sel_iter_t *iter, hsize_t *coords)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(iter);
    assert(coords);

    
    if (H5VM_array_calc(iter->u.all.elmt_offset, iter->rank, iter->dims, coords) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't retrieve coordinates");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5S__all_iter_block(const H5S_sel_iter_t *iter, hsize_t *start, hsize_t *end)
{
    unsigned u; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);
    assert(start);
    assert(end);

    for (u = 0; u < iter->rank; u++) {
        
        
        start[u] = 0;

        
        
        end[u] = iter->dims[u] - 1;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static hsize_t
H5S__all_iter_nelmts(const H5S_sel_iter_t *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(iter->elmt_left)
} 

static htri_t
H5S__all_iter_has_next_block(const H5S_sel_iter_t H5_ATTR_UNUSED *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(false)
} 

static herr_t
H5S__all_iter_next(H5S_sel_iter_t *iter, size_t nelem)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);
    assert(nelem > 0);

    
    iter->u.all.elmt_offset += nelem;
    iter->u.all.byte_offset += (nelem * iter->elmt_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_iter_next_block(H5S_sel_iter_t H5_ATTR_UNUSED *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5S__all_iter_get_seq_list(H5S_sel_iter_t *iter, size_t H5_ATTR_UNUSED maxseq, size_t maxelem, size_t *nseq,
                           size_t *nelem, hsize_t *off, size_t *len)
{
    size_t elem_used; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);
    assert(maxseq > 0);
    assert(maxelem > 0);
    assert(nseq);
    assert(nelem);
    assert(off);
    assert(len);

    
    H5_CHECK_OVERFLOW(iter->elmt_left, hsize_t, size_t);
    elem_used = MIN(maxelem, (size_t)iter->elmt_left);
    assert(elem_used > 0);

    
    off[0] = iter->u.all.byte_offset;
    len[0] = elem_used * iter->elmt_size;

    
    *nseq = 1;

    
    *nelem = elem_used;

    
    iter->elmt_left -= elem_used;
    iter->u.all.elmt_offset += elem_used;
    iter->u.all.byte_offset += len[0];

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_iter_release(H5S_sel_iter_t H5_ATTR_UNUSED *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_release(H5S_t *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);

    
    space->select.num_elem = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_copy(H5S_t *dst, const H5S_t H5_ATTR_UNUSED *src, bool H5_ATTR_UNUSED share_selection)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(src);
    assert(dst);

    
    dst->select.num_elem = (hsize_t)H5S_GET_EXTENT_NPOINTS(dst);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5S__all_is_valid(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    FUNC_LEAVE_NOAPI(true)
} 

static hssize_t
H5S__all_serial_size(H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    
    FUNC_LEAVE_NOAPI(16)
} 

static herr_t
H5S__all_serialize(H5S_t *space, uint8_t **p)
{
    uint8_t *pp = (*p); 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(p);
    assert(pp);

    
    UINT32ENCODE(pp, (uint32_t)H5S_GET_SELECT_TYPE(space)); 
    UINT32ENCODE(pp, (uint32_t)H5S_ALL_VERSION_1);          
    UINT32ENCODE(pp, (uint32_t)0);                          
    UINT32ENCODE(pp, (uint32_t)0);                          

    
    *p = pp;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_deserialize(H5S_t **space, const uint8_t **p, const size_t p_size, bool skip)
{
    uint32_t version;                           
    H5S_t   *tmp_space = NULL;                  
    herr_t         ret_value = SUCCEED;         
    const uint8_t *p_end     = *p + p_size - 1; 
    FUNC_ENTER_PACKAGE

    assert(p);
    assert(*p);

    

    
    if (!*space) {
        if (NULL == (tmp_space = H5S_create(H5S_SIMPLE)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create dataspace");
    } 
    else
        tmp_space = *space;

    
    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *p, sizeof(uint32_t), p_end))
        HGOTO_ERROR(H5E_DATASPACE, H5E_OVERFLOW, FAIL, "buffer overflow while decoding selection version");
    UINT32DECODE(*p, version);

    if (version < H5S_ALL_VERSION_1 || version > H5S_ALL_VERSION_LATEST)
        HGOTO_ERROR(H5E_DATASPACE, H5E_BADVALUE, FAIL, "bad version number for all selection");

    
    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *p, 8, p_end))
        HGOTO_ERROR(H5E_DATASPACE, H5E_OVERFLOW, FAIL, "buffer overflow while decoding header");
    *p += 8;

    
    if (H5S_select_all(tmp_space, true) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTDELETE, FAIL, "can't change selection");

    
    if (!*space)
        *space = tmp_space;

done:
    
    if (!*space && tmp_space)
        if (H5S_close(tmp_space) < 0)
            HDONE_ERROR(H5E_DATASPACE, H5E_CANTFREE, FAIL, "can't close dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5S__all_bounds(const H5S_t *space, hsize_t *start, hsize_t *end)
{
    unsigned rank; 
    unsigned i;    

    FUNC_ENTER_PACKAGE_NOERR

    assert(space);
    assert(start);
    assert(end);

    
    rank = space->extent.rank;

    
    for (i = 0; i < rank; i++) {
        start[i] = 0;
        end[i]   = space->extent.size[i] - 1;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_offset(const H5S_t H5_ATTR_UNUSED *space, hsize_t *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);
    assert(offset);

    
    *offset = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5S__all_unlim_dim(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(-1)
} 

static htri_t
H5S__all_is_contiguous(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    FUNC_LEAVE_NOAPI(true)
} 

static htri_t
H5S__all_is_single(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    FUNC_LEAVE_NOAPI(true)
} 

static htri_t
H5S__all_is_regular(H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);

    FUNC_LEAVE_NOAPI(true)
} 

static htri_t
H5S__all_shape_same(H5S_t *space1, H5S_t *space2)
{
    int    space1_dim;       
    int    space2_dim;       
    htri_t ret_value = true; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space1);
    assert(space2);

    
    space1_dim = (int)space1->extent.rank - 1;
    space2_dim = (int)space2->extent.rank - 1;

    
    while (space2_dim >= 0) {
        if (space1->extent.size[space1_dim] != space2->extent.size[space2_dim])
            HGOTO_DONE(false);

        space1_dim--;
        space2_dim--;
    } 

    
    while (space1_dim >= 0) {
        if (space1->extent.size[space1_dim] != 1)
            HGOTO_DONE(false);

        space1_dim--;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5S__all_intersect_block(H5S_t H5_ATTR_UNUSED *space, const hsize_t H5_ATTR_UNUSED *start,
                         const hsize_t H5_ATTR_UNUSED *end)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(H5S_SEL_ALL == H5S_GET_SELECT_TYPE(space));
    assert(start);
    assert(end);

    FUNC_LEAVE_NOAPI(true)
} 

static herr_t
H5S__all_adjust_u(H5S_t H5_ATTR_UNUSED *space, const hsize_t H5_ATTR_UNUSED *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(offset);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_adjust_s(H5S_t H5_ATTR_UNUSED *space, const hssize_t H5_ATTR_UNUSED *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(offset);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_project_scalar(const H5S_t H5_ATTR_UNUSED *space, hsize_t *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space && H5S_SEL_ALL == H5S_GET_SELECT_TYPE(space));
    assert(offset);

    
    *offset = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__all_project_simple(const H5S_t H5_ATTR_UNUSED *base_space, H5S_t *new_space,
                        hsize_t H5_ATTR_UNUSED *offset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(base_space && H5S_SEL_ALL == H5S_GET_SELECT_TYPE(base_space));
    assert(new_space);
    assert(offset);

    
    if (H5S_select_all(new_space, true) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSET, FAIL, "unable to set all selection");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5S_select_all(H5S_t *space, bool rel_prev)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(space);

    
    if (rel_prev)
        if (H5S_SELECT_RELEASE(space) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTDELETE, FAIL, "can't release selection");

    
    space->select.num_elem = (hsize_t)H5S_GET_EXTENT_NPOINTS(space);

    
    space->select.type = H5S_sel_all;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Sselect_all(hid_t spaceid)
{
    H5S_t *space;               
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (space = (H5S_t *)H5I_object_verify(spaceid, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataspace");

    
    if (H5S_select_all(space, true) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTDELETE, FAIL, "can't change selection");

done:
    FUNC_LEAVE_API(ret_value)
} 
