/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h" 
#ifdef H5_HAVE_PARALLEL
#include "H5ACprivate.h" 
#endif                   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5MFprivate.h" 
#include "H5PBprivate.h" 
#include "H5SLprivate.h" 
#include "H5VMprivate.h" 

#define H5D_CHUNK_GET_FIRST_NODE(dinfo)                                                                      \
    (dinfo->layout_io_info.chunk_map->use_single                                                             \
         ? (H5SL_node_t *)(1)                                                                                \
         : H5SL_first(dinfo->layout_io_info.chunk_map->dset_sel_pieces))
#define H5D_CHUNK_GET_NODE_INFO(dinfo, node)                                                                 \
    (dinfo->layout_io_info.chunk_map->use_single ? dinfo->layout_io_info.chunk_map->single_piece_info        \
                                                 : (H5D_piece_info_t *)H5SL_item(node))
#define H5D_CHUNK_GET_NEXT_NODE(dinfo, node)                                                                 \
    (dinfo->layout_io_info.chunk_map->use_single ? (H5SL_node_t *)NULL : H5SL_next(node))
#define H5D_CHUNK_GET_NODE_COUNT(dinfo)                                                                      \
    (dinfo->layout_io_info.chunk_map->use_single                                                             \
         ? (size_t)1                                                                                         \
         : H5SL_count(dinfo->layout_io_info.chunk_map->dset_sel_pieces))

#define H5D_CHUNK_STORAGE_INDEX_CHK(storage)                                                                 \
    do {                                                                                                     \
        assert((H5D_CHUNK_IDX_EARRAY == (storage)->idx_type && H5D_COPS_EARRAY == (storage)->ops) ||         \
               (H5D_CHUNK_IDX_FARRAY == (storage)->idx_type && H5D_COPS_FARRAY == (storage)->ops) ||         \
               (H5D_CHUNK_IDX_BT2 == (storage)->idx_type && H5D_COPS_BT2 == (storage)->ops) ||               \
               (H5D_CHUNK_IDX_BTREE == (storage)->idx_type && H5D_COPS_BTREE == (storage)->ops) ||           \
               (H5D_CHUNK_IDX_SINGLE == (storage)->idx_type && H5D_COPS_SINGLE == (storage)->ops) ||         \
               (H5D_CHUNK_IDX_NONE == (storage)->idx_type && H5D_COPS_NONE == (storage)->ops));              \
    } while (0)

#define H5D_CHUNK_ENCODE_SIZE_CHECK(layout, length, err)                                                     \
    do {                                                                                                     \
        if ((layout)->version <= H5O_LAYOUT_VERSION_4) {                                                     \
            if ((length) > UINT32_MAX)                                                                       \
                HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, err,                                                  \
                            "chunk size is greater than UINT32_MAX and can't be encoded with this file "     \
                            "format version - see H5Pset_libver_bounds()");                                  \
                                                                                                             \
            if ((layout)->version == H5O_LAYOUT_VERSION_4 &&                                                 \
                (layout)->storage.u.chunk.idx_type != H5D_CHUNK_IDX_SINGLE) {                                \
                unsigned allow_chunk_size_len;                       \
                unsigned new_chunk_size_len;                                 \
                                                                                                             \
                                   \
                                          \
                allow_chunk_size_len = 1 + ((H5VM_log2_gen((uint64_t)((layout)->u.chunk.size)) + 8) / 8);    \
                if (allow_chunk_size_len > 8)                                                                \
                    allow_chunk_size_len = 8;                                                                \
                                                                                                             \
                                                                          \
                new_chunk_size_len = (H5VM_log2_gen((uint64_t)(length)) + 8) / 8;                            \
                if (new_chunk_size_len > 8)                                                                  \
                    HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, err,                                              \
                                "encoded chunk size is more than 8 bytes?!?");                               \
                                                                                                             \
                                                      \
                if (new_chunk_size_len > allow_chunk_size_len)                                               \
                    HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL,                                             \
                                "filter increased chunk size by too much and it cannot be encoded with "     \
                                "this file format version - see H5Pset_libver_bounds()");                    \
            }                                                                                                \
        }                                                                                                    \
    } while (0)

#define H5D_RDCC_DISABLE_FILTERS 0x01U 
#define H5D_RDCC_NEWLY_DISABLED_FILTERS                                                                      \
    0x02U 

typedef struct H5D_rdcc_ent_t {
    bool                   locked;                   
    bool                   dirty;                    
    bool                   deleted;                  
    unsigned               edge_chunk_state;         
    hsize_t                scaled[H5O_LAYOUT_NDIMS]; 
    hsize_t                rd_count;                 
    hsize_t                wr_count;                 
    H5F_block_t            chunk_block;              
    hsize_t                chunk_idx;                
    uint8_t               *chunk;                    
    unsigned               idx;                      
    struct H5D_rdcc_ent_t *next;                     
    struct H5D_rdcc_ent_t *prev;                     
    struct H5D_rdcc_ent_t *tmp_next;                 
    struct H5D_rdcc_ent_t *tmp_prev;                 
} H5D_rdcc_ent_t;
typedef H5D_rdcc_ent_t *H5D_rdcc_ent_ptr_t; 

typedef struct H5D_chunk_it_ud1_t {
    H5D_chunk_common_ud_t     common;          
    const H5D_chk_idx_info_t *idx_info;        
    const H5D_io_info_t      *io_info;         
    const H5D_dset_io_info_t *dset_info;       
    const hsize_t            *space_dim;       
    const bool               *shrunk_dim;      
    H5S_t                    *chunk_space;     
    hsize_t                   elmts_per_chunk; 
    hsize_t                  *hyper_start;     
    H5D_fill_buf_info_t       fb_info;         
    bool                      fb_info_init;    
} H5D_chunk_it_ud1_t;

typedef struct H5D_chunk_it_ud2_t {
    
    H5D_chunk_common_ud_t common; 

    
    haddr_t *chunk_addr; 
} H5D_chunk_it_ud2_t;

typedef struct H5D_chunk_it_ud3_t {
    H5D_chunk_common_ud_t common;       
    H5F_t                *file_src;     
    H5D_chk_idx_info_t   *idx_info_dst; 
    void                 *buf;          
    void                 *bkg;          
    size_t                buf_size;     
    bool                  do_convert;   

    
    const H5T_t *dt_src;           
    const H5T_t *dt_dst;           
    const H5T_t *dt_mem;           
    H5T_path_t  *tpath_src_mem;    
    H5T_path_t  *tpath_mem_dst;    
    void        *reclaim_buf;      
    size_t       reclaim_buf_size; 
    size_t       nelmts;           
    H5S_t       *buf_space;        

    
    const H5O_pline_t *pline;      
    unsigned           dset_ndims; 
    const hsize_t     *dset_dims;  

    
    H5O_copy_t *cpy_info; 

    
    bool     chunk_in_cache;
    uint8_t *chunk; 
} H5D_chunk_it_ud3_t;

typedef struct H5D_chunk_it_ud4_t {
    FILE    *stream;           
    bool     header_displayed; 
    unsigned ndims;            
    hsize_t *chunk_dim;        
} H5D_chunk_it_ud4_t;

typedef struct H5D_chunk_it_ud5_t {
    H5D_chk_idx_info_t *new_idx_info; 
    unsigned            dset_ndims;   
    hsize_t            *dset_dims;    
} H5D_chunk_it_ud5_t;

typedef struct H5D_chunk_readvv_ud_t {
    unsigned char *rbuf; 
    const H5D_t   *dset; 
} H5D_chunk_readvv_ud_t;

typedef struct H5D_chunk_info_iter_ud_t {
    hsize_t  scaled[H5O_LAYOUT_NDIMS]; 
    hsize_t  ndims;                    
    hsize_t  nbytes;                   
    unsigned filter_mask;              
    haddr_t  chunk_addr;               
    hsize_t  chunk_idx;                
    hsize_t  curr_idx;                 
    unsigned idx_hint;                 
    bool     found;                    
} H5D_chunk_info_iter_ud_t;

#ifdef H5_HAVE_PARALLEL

typedef struct H5D_chunk_coll_fill_info_t {
    size_t num_chunks; 
    struct chunk_coll_fill_info {
        haddr_t addr;       
        size_t  chunk_size; 
        bool    unfiltered_partial_chunk;
    } *chunk_info;
} H5D_chunk_coll_fill_info_t;
#endif 

typedef struct H5D_chunk_iter_ud_t {
    H5D_chunk_iter_op_t op;        
    void               *op_data;   
    H5O_layout_chunk_t *chunk;     
    haddr_t             base_addr; 
} H5D_chunk_iter_ud_t;

static herr_t H5D__chunk_construct(H5F_t *f, H5D_t *dset);
static herr_t H5D__chunk_init(H5F_t *f, H5D_t *dset, hid_t dapl_id, bool open_op);
static herr_t H5D__chunk_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__chunk_io_init_selections(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__chunk_mdio_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__chunk_read(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__chunk_write(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t H5D__chunk_flush(H5D_t *dset);
static herr_t H5D__chunk_io_term(H5D_io_info_t *io_info, H5D_dset_io_info_t *di);
static herr_t H5D__chunk_dest(H5D_t *dset);

static int H5D__get_num_chunks_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata);
static int H5D__get_chunk_info_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata);
static int H5D__get_chunk_info_by_coord_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata);
static int H5D__chunk_iter_cb(const H5D_chunk_rec_t *chunk_rec, void *udata);

static ssize_t H5D__nonexistent_readvv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                                       size_t chunk_max_nseq, size_t *chunk_curr_seq, size_t chunk_len_arr[],
                                       hsize_t chunk_offset_arr[], size_t mem_max_nseq, size_t *mem_curr_seq,
                                       size_t mem_len_arr[], hsize_t mem_offset_arr[]);

static int H5D__chunk_format_convert_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata);

static herr_t   H5D__chunk_set_info_real(H5O_layout_chunk_t *layout, unsigned ndims, const hsize_t *curr_dims,
                                         const hsize_t *max_dims);
static herr_t   H5D__chunk_set_sizes(H5D_t *dset);
static herr_t   H5D__chunk_cinfo_cache_reset(H5D_chunk_cached_t *last);
static herr_t   H5D__chunk_cinfo_cache_update(H5D_chunk_cached_t *last, const H5D_chunk_ud_t *udata);
static bool     H5D__chunk_cinfo_cache_found(const H5D_chunk_cached_t *last, H5D_chunk_ud_t *udata);
static herr_t   H5D__create_piece_map_single(H5D_dset_io_info_t *di, H5D_io_info_t *io_info);
static herr_t   H5D__create_piece_file_map_all(H5D_dset_io_info_t *di, H5D_io_info_t *io_info);
static herr_t   H5D__create_piece_file_map_hyper(H5D_dset_io_info_t *di, H5D_io_info_t *io_info);
static herr_t   H5D__create_piece_mem_map_1d(const H5D_dset_io_info_t *di);
static herr_t   H5D__create_piece_mem_map_hyper(const H5D_dset_io_info_t *di);
static herr_t   H5D__piece_file_cb(void *elem, const H5T_t *type, unsigned ndims, const hsize_t *coords,
                                   void *_opdata);
static herr_t   H5D__piece_mem_cb(void *elem, const H5T_t *type, unsigned ndims, const hsize_t *coords,
                                  void *_opdata);
static herr_t   H5D__chunk_may_use_select_io(H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info);
static unsigned H5D__chunk_hash_val(const H5D_shared_t *shared, const hsize_t *scaled);
static herr_t   H5D__chunk_flush_entry(const H5D_t *dset, H5D_rdcc_ent_t *ent, bool reset);
static herr_t   H5D__chunk_cache_evict(const H5D_t *dset, H5D_rdcc_ent_t *ent, bool flush);
static void    *H5D__chunk_lock(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                                H5D_chunk_ud_t *udata, bool relax, bool prev_unfilt_chunk);
static herr_t   H5D__chunk_unlock(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                                  const H5D_chunk_ud_t *udata, bool dirty, void *chunk, hsize_t naccessed);
static herr_t   H5D__chunk_cache_prune(const H5D_t *dset, size_t size);
static herr_t   H5D__chunk_prune_fill(H5D_chunk_it_ud1_t *udata, bool new_unfilt_chunk);
#ifdef H5_HAVE_PARALLEL
static herr_t H5D__chunk_collective_fill(const H5D_t *dset, H5D_chunk_coll_fill_info_t *chunk_fill_info,
                                         const void *fill_buf, const void *partial_chunk_fill_buf);
static int    H5D__chunk_cmp_coll_fill_info(const void *_entry1, const void *_entry2);
#endif 

static int H5D__chunk_dump_index_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata);

const H5D_layout_ops_t H5D_LOPS_CHUNK[1] = {{
    H5D__chunk_construct,      
    H5D__chunk_init,           
    H5D__chunk_is_space_alloc, 
    H5D__chunk_is_data_cached, 
    H5D__chunk_io_init,        
    H5D__chunk_mdio_init,      
    H5D__chunk_read,           
    H5D__chunk_write,          
    NULL,                      
    NULL,                      
    H5D__chunk_flush,          
    H5D__chunk_io_term,        
    H5D__chunk_dest            
}};

static const H5D_layout_ops_t H5D_LOPS_NONEXISTENT[1] = {
    {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, H5D__nonexistent_readvv, NULL, NULL, NULL, NULL}};

H5FL_SEQ_DEFINE_STATIC(H5D_rdcc_ent_ptr_t);

H5FL_DEFINE_STATIC(H5D_rdcc_ent_t);

H5FL_DEFINE_STATIC(H5D_chunk_map_t);

H5FL_DEFINE(H5D_piece_info_t);

H5FL_BLK_DEFINE_STATIC(chunk);

H5FL_EXTERN(H5S_sel_iter_t);

herr_t
H5D__chunk_direct_write(H5D_t *dset, uint32_t filters, hsize_t *offset, size_t data_size, const void *buf)
{
    const H5O_layout_t *layout = &(dset->shared->layout); 
    H5D_chunk_ud_t      udata;                            
    H5F_block_t         old_chunk;                        
    H5D_chk_idx_info_t  idx_info;                         
    hsize_t             scaled[H5S_MAX_RANK];             
    bool                need_insert = false;   
    herr_t              ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(layout->type == H5D_CHUNKED);

    
    if (!H5D__chunk_is_space_alloc(&layout->storage))
        if (H5D__alloc_storage(dset, H5D_ALLOC_WRITE, false, NULL) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize storage");

    
    H5VM_chunk_scaled(dset->shared->ndims, offset, layout->u.chunk.dim, scaled);
    scaled[dset->shared->ndims] = 0;

    
    if (H5D__chunk_lookup(dset, scaled, &udata) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

    
    assert((H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length > 0) ||
           (!H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length == 0));

    
    
    old_chunk.offset = udata.chunk_block.offset;
    old_chunk.length = udata.chunk_block.length;

    

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &(dset->shared->dcpl_cache.pline);
    idx_info.layout = &(dset->shared->layout);

    
    udata.chunk_block.length = data_size;

    
    if (H5_UNLIKELY((size_t)udata.chunk_block.length != data_size))
        HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "chunk size too big to fit in hsize_t");

    if (0 == idx_info.pline->nused && H5_addr_defined(old_chunk.offset))
        
        need_insert = false;
    else {
        
        if (H5D__chunk_file_alloc(&idx_info, &old_chunk, &udata.chunk_block, &need_insert, scaled) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "unable to allocate chunk");

        
        H5D__chunk_cinfo_cache_update(&dset->shared->cache.chunk.last, &udata);
    } 

    
    if (!H5_addr_defined(udata.chunk_block.offset))
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk address isn't defined");

    
    if (UINT_MAX != udata.idx_hint) {
        const H5D_rdcc_t *rdcc = &(dset->shared->cache.chunk); 

        if (H5D__chunk_cache_evict(dset, rdcc->slot[udata.idx_hint], false) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTREMOVE, FAIL, "unable to evict chunk");
    } 

    
    if (H5F_shared_block_write(H5F_SHARED(dset->oloc.file), H5FD_MEM_DRAW, udata.chunk_block.offset,
                               data_size, buf) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to write raw data to file");

    
    if (need_insert && layout->storage.u.chunk.ops->insert) {
        
        udata.filter_mask = filters;

        if ((layout->storage.u.chunk.ops->insert)(&idx_info, &udata, dset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL, "unable to insert chunk addr into index");
    } 

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D__chunk_direct_read(const H5D_t *dset, hsize_t *offset, uint32_t *filters, void *buf, size_t *nalloc)
{
    const H5O_layout_t *layout = &(dset->shared->layout);      
    const H5D_rdcc_t   *rdcc   = &(dset->shared->cache.chunk); 
    H5D_chunk_ud_t      udata;                                 
    hsize_t             scaled[H5S_MAX_RANK];                  
    herr_t              ret_value = SUCCEED;                   

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(dset && H5D_CHUNKED == layout->type);
    assert(offset);
    assert(filters);
    assert(buf || nalloc);

    *filters = 0;

    
    if (!H5D__chunk_is_space_alloc(&layout->storage) && !H5D__chunk_is_data_cached(dset->shared))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "storage is not initialized");

    
    H5VM_chunk_scaled(dset->shared->ndims, offset, layout->u.chunk.dim, scaled);
    scaled[dset->shared->ndims] = 0;

    
    udata.filter_mask        = 0;
    udata.chunk_block.offset = HADDR_UNDEF;
    udata.chunk_block.length = 0;
    udata.idx_hint           = UINT_MAX;

    
    if (H5D__chunk_lookup(dset, scaled, &udata) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

    
    assert((H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length > 0) ||
           (!H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length == 0));

    
    if (H5_UNLIKELY((hsize_t)((size_t)udata.chunk_block.length) != udata.chunk_block.length))
        HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "chunk size does not fit in size_t");

    
    if (UINT_MAX != udata.idx_hint) {
        H5D_rdcc_ent_t *ent = rdcc->slot[udata.idx_hint];
        bool            flush;

        
        assert(udata.idx_hint < rdcc->nslots);
        assert(rdcc->slot[udata.idx_hint]);

        flush = (ent->dirty == true) ? true : false;

        
        if (H5D__chunk_cache_evict(dset, rdcc->slot[udata.idx_hint], flush) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTREMOVE, FAIL, "unable to evict chunk");

        
        udata.filter_mask        = 0;
        udata.chunk_block.offset = HADDR_UNDEF;
        udata.chunk_block.length = 0;
        udata.idx_hint           = UINT_MAX;

        
        if (H5D__chunk_lookup(dset, scaled, &udata) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");
    }

    
    if (!H5_addr_defined(udata.chunk_block.offset))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "chunk address isn't defined");

    
    if (udata.chunk_block.length > 0 && buf && (!nalloc || *nalloc >= udata.chunk_block.length))
        
        if (H5F_shared_block_read(H5F_SHARED(dset->oloc.file), H5FD_MEM_DRAW, udata.chunk_block.offset,
                                  (size_t)udata.chunk_block.length, buf) < 0)
            HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "unable to read raw data chunk");

    
    if (nalloc)
        *nalloc = (size_t)udata.chunk_block.length;

    
    *filters = udata.filter_mask;

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D__get_chunk_storage_size(H5D_t *dset, const hsize_t *offset, hsize_t *storage_size)
{
    const H5O_layout_t *layout = &(dset->shared->layout);      
    const H5D_rdcc_t   *rdcc   = &(dset->shared->cache.chunk); 
    hsize_t             scaled[H5S_MAX_RANK];                  
    H5D_chunk_ud_t      udata;                                 
    herr_t              ret_value = SUCCEED;                   

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(dset && H5D_CHUNKED == layout->type);
    assert(offset);
    assert(storage_size);

    
    if (!(*layout->ops->is_space_alloc)(&layout->storage))
        HGOTO_DONE(SUCCEED);

    
    H5VM_chunk_scaled(dset->shared->ndims, offset, layout->u.chunk.dim, scaled);
    scaled[dset->shared->ndims] = 0;

    
    udata.chunk_block.offset = HADDR_UNDEF;
    udata.chunk_block.length = 0;
    udata.idx_hint           = UINT_MAX;

    
    if (H5D__chunk_lookup(dset, scaled, &udata) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

    
    assert((H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length > 0) ||
           (!H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length == 0));

    
    if (!H5_addr_defined(udata.chunk_block.offset) && UINT_MAX == udata.idx_hint)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "chunk storage is not allocated");

    
    if (dset->shared->dcpl_cache.pline.nused > 0) {
        
        if (UINT_MAX != udata.idx_hint) {
            H5D_rdcc_ent_t *ent = rdcc->slot[udata.idx_hint];

            
            assert(udata.idx_hint < rdcc->nslots);
            assert(rdcc->slot[udata.idx_hint]);

            
            if (ent->dirty == true) {
                
                if (H5D__chunk_cache_evict(dset, rdcc->slot[udata.idx_hint], true) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTREMOVE, FAIL, "unable to evict chunk");

                
                udata.chunk_block.offset = HADDR_UNDEF;
                udata.chunk_block.length = 0;
                udata.idx_hint           = UINT_MAX;

                
                if (H5D__chunk_lookup(dset, scaled, &udata) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");
            }
        }

        
        if (!H5_addr_defined(udata.chunk_block.offset))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "chunk address isn't defined");

        
        *storage_size = udata.chunk_block.length;
    }
    
    else
        *storage_size = dset->shared->layout.u.chunk.size;

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static herr_t
H5D__chunk_set_info_real(H5O_layout_chunk_t *layout, unsigned ndims, const hsize_t *curr_dims,
                         const hsize_t *max_dims)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(layout);
    assert(curr_dims);

    
    if (ndims == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "number of dimensions cannot be zero");

    
    layout->nchunks     = 1;
    layout->max_nchunks = 1;
    for (unsigned u = 0; u < ndims; u++) {
        
        layout->chunks[u] = ((curr_dims[u] + layout->dim[u]) - 1) / layout->dim[u];
        if (H5S_UNLIMITED == max_dims[u])
            layout->max_chunks[u] = H5S_UNLIMITED;
        else {
            
            if (layout->dim[u] == 0)
                HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "dimension size must be > 0, dim = %u ", u);

            layout->max_chunks[u] = ((max_dims[u] + layout->dim[u]) - 1) / layout->dim[u];
        }

        
        layout->nchunks *= layout->chunks[u];
        layout->max_nchunks *= layout->max_chunks[u];
    }

    
    H5VM_array_down(ndims, layout->chunks, layout->down_chunks);
    H5VM_array_down(ndims, layout->max_chunks, layout->max_down_chunks);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_set_info(const H5D_t *dset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    if (H5D__chunk_set_info_real(&dset->shared->layout.u.chunk, dset->shared->ndims, dset->shared->curr_dims,
                                 dset->shared->max_dims) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set layout's chunk info");

    
    if (dset->shared->layout.storage.u.chunk.ops->resize &&
        (dset->shared->layout.storage.u.chunk.ops->resize)(&dset->shared->layout.u.chunk) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to resize chunk index information");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_set_sizes(H5D_t *dset)
{
    uint64_t chunk_size;            
    unsigned max_enc_bytes_per_dim; 
    unsigned u;                     
    herr_t   ret_value = SUCCEED;   

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->shared->layout.u.chunk.ndims > 0);

    
    
    if (dset->shared->layout.u.chunk.dim[dset->shared->layout.u.chunk.ndims - 1]) {
        if (dset->shared->layout.u.chunk.dim[dset->shared->layout.u.chunk.ndims - 1] !=
            (hsize_t)H5T_GET_SIZE(dset->shared->type))
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL,
                        "stored datatype size in chunk layout does not match datatype description");
    }
    else
        dset->shared->layout.u.chunk.dim[dset->shared->layout.u.chunk.ndims - 1] =
            (hsize_t)H5T_GET_SIZE(dset->shared->type);

    
    max_enc_bytes_per_dim = 0;
    for (u = 0; u < (unsigned)dset->shared->layout.u.chunk.ndims; u++) {
        unsigned enc_bytes_per_dim; 

        
        enc_bytes_per_dim = (H5VM_log2_gen(dset->shared->layout.u.chunk.dim[u]) + 8) / 8;

        
        if (enc_bytes_per_dim > max_enc_bytes_per_dim)
            max_enc_bytes_per_dim = enc_bytes_per_dim;
    } 
    assert(max_enc_bytes_per_dim > 0 && max_enc_bytes_per_dim <= 8);

    
    if (dset->shared->layout.u.chunk.enc_bytes_per_dim) {
        if (dset->shared->layout.u.chunk.enc_bytes_per_dim != max_enc_bytes_per_dim)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL,
                        "stored chunk dimension encoding length does not match value calculated from chunk "
                        "dimensions");
    }
    else
        dset->shared->layout.u.chunk.enc_bytes_per_dim = max_enc_bytes_per_dim;

    
    
    for (u = 1, chunk_size = (uint64_t)dset->shared->layout.u.chunk.dim[0];
         u < dset->shared->layout.u.chunk.ndims; u++)
        chunk_size *= (uint64_t)dset->shared->layout.u.chunk.dim[u];

    
    dset->shared->layout.u.chunk.size = (hsize_t)chunk_size;

    
    if ((uint64_t)dset->shared->layout.u.chunk.size != chunk_size)
        HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "chunk size too big to fit in hsize_t");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_construct(H5F_t *f, H5D_t *dset)
{
    H5O_layout_t *layout;
    unsigned      version_req = H5O_LAYOUT_VERSION_1; 
    unsigned version_perf = H5O_LAYOUT_VERSION_1; 
    unsigned u;                                   
    herr_t   ret_value = SUCCEED;                 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(dset);
    assert(dset->shared);

    
    layout = &dset->shared->layout;

    
    if (0 == dset->shared->layout.u.chunk.ndims)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "no chunk information set?");
    if (layout->u.chunk.ndims != dset->shared->ndims)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "dimensionality of chunks doesn't match the dataspace");

    
    layout->u.chunk.ndims++;

    
    if (H5D__chunk_set_sizes(dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "unable to set chunk sizes");
    assert((unsigned)(layout->u.chunk.ndims) <= NELMTS(layout->u.chunk.dim));

    
    
    version_req = layout->version;

    
    version_perf = H5O_LAYOUT_VERSION_4;

    
    if (layout->u.chunk.size > (hsize_t)0xffffffff) {
        if (H5O_layout_ver_bounds[H5F_HIGH_BOUND(f)] < H5O_LAYOUT_VERSION_5)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                        "chunk size > 4GB requires H5F_LIBVER_V200 or later format version - see "
                        "H5Pset_libver_bounds()");

        
        version_req = MAX(version_req, H5O_LAYOUT_VERSION_5);
    }
#ifndef NDEBUG
    else {
        
        for (u = 0; u < dset->shared->ndims; u++)
            assert(layout->u.chunk.dim[u] <= (hsize_t)0xffffffff);
    }
#endif 

    
    if ((H5O_layout_ver_bounds[H5F_LOW_BOUND(f)] >= H5O_LAYOUT_VERSION_4) ||
        (version_req >= H5O_LAYOUT_VERSION_4)) {
        unsigned unlim_count = 0;    
        bool     single      = true; 

        
        assert(dset->shared->ndims > 0);

        
        for (u = 0; u < dset->shared->ndims; u++) {
            if (dset->shared->max_dims[u] == H5S_UNLIMITED)
                unlim_count++;
            if (dset->shared->curr_dims[u] != dset->shared->max_dims[u] ||
                dset->shared->curr_dims[u] != layout->u.chunk.dim[u])
                single = false;
        }

        
        if (unlim_count) {          
            if (1 == unlim_count) { 
                
                layout->u.chunk.idx_type         = H5D_CHUNK_IDX_EARRAY;
                layout->storage.u.chunk.idx_type = H5D_CHUNK_IDX_EARRAY;
                layout->storage.u.chunk.ops      = H5D_COPS_EARRAY;

                
                
                layout->u.chunk.u.earray.cparam.max_nelmts_bits       = H5D_EARRAY_MAX_NELMTS_BITS;
                layout->u.chunk.u.earray.cparam.idx_blk_elmts         = H5D_EARRAY_IDX_BLK_ELMTS;
                layout->u.chunk.u.earray.cparam.sup_blk_min_data_ptrs = H5D_EARRAY_SUP_BLK_MIN_DATA_PTRS;
                layout->u.chunk.u.earray.cparam.data_blk_min_elmts    = H5D_EARRAY_DATA_BLK_MIN_ELMTS;
                layout->u.chunk.u.earray.cparam.max_dblk_page_nelmts_bits =
                    H5D_EARRAY_MAX_DBLOCK_PAGE_NELMTS_BITS;

                
                if (dset->shared->dcpl_cache.pline.nused)
                    version_perf = H5O_LAYOUT_VERSION_5;
            }
            else { 
                
                layout->u.chunk.idx_type         = H5D_CHUNK_IDX_BT2;
                layout->storage.u.chunk.idx_type = H5D_CHUNK_IDX_BT2;
                layout->storage.u.chunk.ops      = H5D_COPS_BT2;

                
                
                layout->u.chunk.u.btree2.cparam.node_size     = H5D_BT2_NODE_SIZE;
                layout->u.chunk.u.btree2.cparam.split_percent = H5D_BT2_SPLIT_PERC;
                layout->u.chunk.u.btree2.cparam.merge_percent = H5D_BT2_MERGE_PERC;

                
                if (dset->shared->dcpl_cache.pline.nused)
                    version_perf = H5O_LAYOUT_VERSION_5;
            }
        }
        else { 
            
            if (single) {
                layout->u.chunk.idx_type         = H5D_CHUNK_IDX_SINGLE;
                layout->storage.u.chunk.idx_type = H5D_CHUNK_IDX_SINGLE;
                layout->storage.u.chunk.ops      = H5D_COPS_SINGLE;

                
                if (dset->shared->dcpl_cache.pline.nused)
                    version_perf = H5O_LAYOUT_VERSION_5;
            }
            else if (!dset->shared->dcpl_cache.pline.nused &&
                     dset->shared->dcpl_cache.fill.alloc_time == H5D_ALLOC_TIME_EARLY) {

                
                layout->u.chunk.idx_type         = H5D_CHUNK_IDX_NONE;
                layout->storage.u.chunk.idx_type = H5D_CHUNK_IDX_NONE;
                layout->storage.u.chunk.ops      = H5D_COPS_NONE;

                
            }
            else {
                
                layout->u.chunk.idx_type         = H5D_CHUNK_IDX_FARRAY;
                layout->storage.u.chunk.idx_type = H5D_CHUNK_IDX_FARRAY;
                layout->storage.u.chunk.ops      = H5D_COPS_FARRAY;

                
                
                layout->u.chunk.u.farray.cparam.max_dblk_page_nelmts_bits =
                    H5D_FARRAY_MAX_DBLK_PAGE_NELMTS_BITS;

                
                if (dset->shared->dcpl_cache.pline.nused)
                    version_perf = H5O_LAYOUT_VERSION_5;
            }
        }
    }

    
    layout->version =
        MAX3(layout->version, version_req, MIN(H5O_layout_ver_bounds[H5F_LOW_BOUND(f)], version_perf));

    
    assert(layout->version <= H5O_layout_ver_bounds[H5F_HIGH_BOUND(f)]);

    
    if (dset->shared->dcpl_cache.efl.nused > 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "external storage not supported with chunked layout");

    
    for (u = 0; u < layout->u.chunk.ndims - 1; u++) {
        
        if (0 == layout->u.chunk.dim[u])
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "chunk size must be > 0, dim = %u ", u);

        
        if (dset->shared->curr_dims[u] && dset->shared->max_dims[u] != H5S_UNLIMITED &&
            dset->shared->max_dims[u] < layout->u.chunk.dim[u])
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                        "chunk size must be <= maximum dimension size for fixed-sized dimensions");
    } 

    
    if (H5D_chunk_idx_reset(&layout->storage.u.chunk, true) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to reset chunked storage index");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_init(H5F_t *f, H5D_t *dset, hid_t dapl_id, bool open_op)
{
    H5D_chk_idx_info_t idx_info;                            
    H5D_rdcc_t        *rdcc = &(dset->shared->cache.chunk); 
    H5P_genplist_t    *dapl;                                
    H5O_storage_chunk_t *sc        = &(dset->shared->layout.storage.u.chunk);
    bool                 idx_init  = false;
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(dset);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

    if (NULL == (dapl = (H5P_genplist_t *)H5I_object(dapl_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for fapl ID");

    
    if (H5P_get(dapl, H5D_ACS_DATA_CACHE_NUM_SLOTS_NAME, &rdcc->nslots) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get data cache number of slots");
    if (rdcc->nslots == H5D_CHUNK_CACHE_NSLOTS_DEFAULT)
        rdcc->nslots = H5F_RDCC_NSLOTS(f);

    if (H5P_get(dapl, H5D_ACS_DATA_CACHE_BYTE_SIZE_NAME, &rdcc->nbytes_max) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get data cache byte size");
    if (rdcc->nbytes_max == H5D_CHUNK_CACHE_NBYTES_DEFAULT)
        rdcc->nbytes_max = H5F_RDCC_NBYTES(f);

    if (H5P_get(dapl, H5D_ACS_PREEMPT_READ_CHUNKS_NAME, &rdcc->w0) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get preempt read chunks");
    if (rdcc->w0 < 0)
        rdcc->w0 = H5F_RDCC_W0(f);

    
    if (!rdcc->nbytes_max || !rdcc->nslots)
        rdcc->nbytes_max = rdcc->nslots = 0;
    else {
        rdcc->slot = H5FL_SEQ_CALLOC(H5D_rdcc_ent_ptr_t, rdcc->nslots);
        if (NULL == rdcc->slot)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        H5D__chunk_cinfo_cache_reset(&(rdcc->last));
    } 

    
    if (dset->shared->ndims > 1) {
        unsigned u; 

        for (u = 0; u < dset->shared->ndims; u++) {
            hsize_t scaled_power2up; 

            
            if (dset->shared->layout.u.chunk.dim[u] == 0)
                HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ", u);

            
            rdcc->scaled_dims[u] = (dset->shared->curr_dims[u] + dset->shared->layout.u.chunk.dim[u] - 1) /
                                   dset->shared->layout.u.chunk.dim[u];

            if (!(scaled_power2up = H5VM_power2up(rdcc->scaled_dims[u])))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get the next power of 2");

            
            rdcc->scaled_power2up[u] = scaled_power2up;

            
            rdcc->scaled_encode_bits[u] = H5VM_log2_gen(rdcc->scaled_power2up[u]);
        } 
    }     

    
    idx_info.f      = f;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if (sc->ops->init && (sc->ops->init)(&idx_info, dset->shared->space, dset->oloc.addr) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize indexing information");
    idx_init = true;

    
    if (H5D__chunk_set_info(dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set # of chunks for dataset");

    
    if (open_op && (H5D__chunk_set_sizes(dset) < 0))
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "unable to set chunk sizes");

    
    if (dset->shared->layout.u.chunk.size > (hsize_t)0xffffffff &&
        dset->shared->layout.u.chunk.idx_type == H5D_CHUNK_IDX_BTREE)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "chunk size must be < 4GB with v1 b-tree index");

done:
    if (FAIL == ret_value) {
        if (rdcc->slot)
            rdcc->slot = H5FL_SEQ_FREE(H5D_rdcc_ent_ptr_t, rdcc->slot);

        if (idx_init && sc->ops->dest && (sc->ops->dest)(&idx_info) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to release chunk index info");
    }
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5D__chunk_is_space_alloc(const H5O_storage_t *storage)
{
    const H5O_storage_chunk_t *sc        = &(storage->u.chunk);
    bool                       ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

    
    ret_value = (sc->ops->is_space_alloc)(sc);

    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5D__chunk_is_data_cached(const H5D_shared_t *shared_dset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(shared_dset);

    FUNC_LEAVE_NOAPI(shared_dset->cache.chunk.nused > 0)
} 

static herr_t
H5D__chunk_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    const H5D_t     *dataset = dinfo->dset;         
    H5D_chunk_map_t *fm;                            
    hssize_t         old_offset[H5O_LAYOUT_NDIMS];  
    htri_t           file_space_normalized = false; 
    unsigned         f_ndims;                       
    int              sm_ndims; 
    unsigned         u;        
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (dinfo->layout_io_info.chunk_map = H5FL_MALLOC(H5D_chunk_map_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "unable to allocate chunk map");
    fm = dinfo->layout_io_info.chunk_map;

    
    dinfo->layout = &(dataset->shared->layout);

    
    fm->last_index      = (hsize_t)-1;
    fm->last_piece_info = NULL;

    
    fm->mchunk_tmpl       = NULL;
    fm->dset_sel_pieces   = NULL;
    fm->single_space      = NULL;
    fm->single_piece_info = NULL;

    
    fm->msel_type = H5S_SEL_ERROR;
    fm->fsel_type = H5S_SEL_ERROR;

    
    if ((sm_ndims = H5S_GET_EXTENT_NDIMS(dinfo->mem_space)) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "unable to get dimension number");
    
    H5_CHECKED_ASSIGN(fm->m_ndims, unsigned, sm_ndims, int);

    
    fm->f_ndims = f_ndims = dataset->shared->layout.u.chunk.ndims - 1;

    
    
    if ((file_space_normalized = H5S_hyper_normalize_offset(dinfo->file_space, old_offset)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to normalize selection");

    
    for (u = 0; u < f_ndims; u++)
        
        fm->chunk_dim[u] = dinfo->layout->u.chunk.dim[u];

    if (H5D__chunk_io_init_selections(io_info, dinfo) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create file and memory chunk selections");

    
    if (io_info->use_select_io != H5D_SELECTION_IO_MODE_OFF)
        if (H5D__chunk_may_use_select_io(io_info, dinfo) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if selection I/O is possible");

    
    if (io_info->use_select_io != H5D_SELECTION_IO_MODE_OFF &&
        !(dinfo->type_info.is_xform_noop && dinfo->type_info.is_conv_noop)) {
        H5SL_node_t *chunk_node; 

        
        chunk_node = H5D_CHUNK_GET_FIRST_NODE(dinfo);
        while (chunk_node) {
            H5D_piece_info_t *piece_info; 

            
            piece_info = H5D_CHUNK_GET_NODE_INFO(dinfo, chunk_node);

            
            H5D_INIT_PIECE_TCONV(io_info, dinfo, piece_info)

            
            chunk_node = H5D_CHUNK_GET_NEXT_NODE(dinfo, chunk_node);
        }
    }

#ifdef H5_HAVE_PARALLEL
    
    if (H5F_SHARED_HAS_FEATURE(io_info->f_sh, H5FD_FEAT_HAS_MPI) &&
        H5F_shared_get_coll_metadata_reads(io_info->f_sh) &&
        H5D__chunk_is_space_alloc(&dataset->shared->layout.storage)) {
        H5O_storage_chunk_t *sc = &(dataset->shared->layout.storage.u.chunk);
        H5D_chk_idx_info_t   idx_info;
        bool                 index_is_open;

        idx_info.f      = dataset->oloc.file;
        idx_info.pline  = &dataset->shared->dcpl_cache.pline;
        idx_info.layout = &dataset->shared->layout;

        assert(sc && sc->ops && sc->ops->is_open);
        if (sc->ops->is_open(&idx_info, &index_is_open) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to check if dataset chunk index is open");

        if (!index_is_open) {
            assert(sc->ops->open);
            if (sc->ops->open(&idx_info) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to open dataset chunk index");
        }

        
        if (sc->ops->load_metadata && sc->ops->load_metadata(&idx_info) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to load additional chunk index metadata");
    }
#endif

done:
    if (file_space_normalized == true)
        if (H5S_hyper_denormalize_offset(dinfo->file_space, old_offset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't denormalize selection");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_io_init_selections(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    H5D_chunk_map_t   *fm;                 
    const H5D_t       *dataset;            
    const H5T_t       *mem_type;           
    H5S_t             *tmp_mspace = NULL;  
    bool               iter_init  = false; 
    char               bogus;              
    H5D_io_info_wrap_t io_info_wrap;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(io_info);
    assert(dinfo);

    
    fm = dinfo->layout_io_info.chunk_map;
    assert(fm);
    dataset  = dinfo->dset;
    mem_type = dinfo->type_info.mem_type;

    
    
    if (dinfo->nelmts == 1
#ifdef H5_HAVE_PARALLEL
        && !(io_info->using_mpi_vfd)
#endif 
        && H5S_SEL_ALL != H5S_GET_SELECT_TYPE(dinfo->file_space)) {
        
        fm->use_single = true;

        
        if (NULL == dataset->shared->cache.chunk.single_space) {
            
            if ((dataset->shared->cache.chunk.single_space = H5S_copy(dinfo->file_space, true, false)) ==
                NULL)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy file space");

            
            if (H5S_set_extent_real(dataset->shared->cache.chunk.single_space, fm->chunk_dim) < 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSET, FAIL, "can't adjust chunk dimensions");

            
            if (H5S_select_all(dataset->shared->cache.chunk.single_space, true) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "unable to set all selection");
        } 
        fm->single_space = dataset->shared->cache.chunk.single_space;
        assert(fm->single_space);

        
        if (NULL == dataset->shared->cache.chunk.single_piece_info)
            if (NULL == (dataset->shared->cache.chunk.single_piece_info = H5FL_MALLOC(H5D_piece_info_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate chunk info");
        fm->single_piece_info = dataset->shared->cache.chunk.single_piece_info;
        assert(fm->single_piece_info);

        
        fm->mchunk_tmpl = NULL;

        
        if (H5D__create_piece_map_single(dinfo, io_info) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                        "unable to create chunk selections for single element");
    } 
    else {
        bool sel_hyper_flag; 

        
        if (NULL == dataset->shared->cache.chunk.sel_chunks)
            if (NULL == (dataset->shared->cache.chunk.sel_chunks = H5SL_create(H5SL_TYPE_HSIZE, NULL)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL, "can't create skip list for chunk selections");
        fm->dset_sel_pieces = dataset->shared->cache.chunk.sel_chunks;
        assert(fm->dset_sel_pieces);

        
        fm->use_single = false;

        
        if ((fm->fsel_type = H5S_GET_SELECT_TYPE(dinfo->file_space)) < H5S_SEL_NONE)
            HGOTO_ERROR(H5E_DATASET, H5E_BADSELECT, FAIL, "unable to get type of selection");
        if ((fm->msel_type = H5S_GET_SELECT_TYPE(dinfo->mem_space)) < H5S_SEL_NONE)
            HGOTO_ERROR(H5E_DATASET, H5E_BADSELECT, FAIL, "unable to get type of selection");

        
        if (fm->fsel_type == H5S_SEL_POINTS || fm->fsel_type == H5S_SEL_NONE)
            sel_hyper_flag = false;
        else
            sel_hyper_flag = true;

        
        if (sel_hyper_flag) {
            
            if (H5S_SEL_ALL == fm->fsel_type) {
                if (H5D__create_piece_file_map_all(dinfo, io_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create file chunk selections");
            } 
            else {
                
                assert(fm->fsel_type == H5S_SEL_HYPERSLABS);

                if (H5D__create_piece_file_map_hyper(dinfo, io_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create file chunk selections");
            } 
        }     
        else {
            H5S_sel_iter_op_t iter_op; 

            
            io_info_wrap.io_info = io_info;
            io_info_wrap.dinfo   = dinfo;
            iter_op.op_type      = H5S_SEL_ITER_OP_LIB;
            iter_op.u.lib_op     = H5D__piece_file_cb;

            
            if (H5S_select_iterate(&bogus, dataset->shared->type, dinfo->file_space, &iter_op,
                                   &io_info_wrap) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create file chunk selections");

            
            fm->last_index      = (hsize_t)-1;
            fm->last_piece_info = NULL;
        } 

        
        if (sel_hyper_flag && H5S_SELECT_SHAPE_SAME(dinfo->file_space, dinfo->mem_space) == true) {
            
            fm->mchunk_tmpl = NULL;

            
            if (H5D__create_piece_mem_map_hyper(dinfo) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create memory chunk selections");
        } 
        else if (sel_hyper_flag && fm->f_ndims == 1 && fm->m_ndims == 1 &&
                 H5S_SELECT_IS_REGULAR(dinfo->mem_space) && H5S_SELECT_IS_SINGLE(dinfo->mem_space)) {
            if (H5D__create_piece_mem_map_1d(dinfo) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create file chunk selections");
        } 
        else {
            H5S_sel_iter_op_t iter_op;   
            size_t            elmt_size; 

            
            if ((tmp_mspace = H5S_copy(dinfo->mem_space, true, false)) == NULL)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy memory space");

            
            if (H5S_select_none(tmp_mspace) < 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "unable to de-select memory space");

            
            fm->mchunk_tmpl = tmp_mspace;

            
            if (0 == (elmt_size = H5T_get_size(mem_type)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADSIZE, FAIL, "datatype size invalid");
            if (H5S_select_iter_init(&(fm->mem_iter), dinfo->mem_space, elmt_size, 0) < 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "unable to initialize selection iterator");
            iter_init = true; 

            
            io_info_wrap.io_info = io_info;
            io_info_wrap.dinfo   = dinfo;
            iter_op.op_type      = H5S_SEL_ITER_OP_LIB;
            iter_op.u.lib_op     = H5D__piece_mem_cb;

            
            if (H5S_select_iterate(&bogus, dataset->shared->type, dinfo->file_space, &iter_op,
                                   &io_info_wrap) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create memory chunk selections");
        } 
    }     

done:
    
    if (ret_value < 0) {
        if (tmp_mspace && !fm->mchunk_tmpl)
            if (H5S_close(tmp_mspace) < 0)
                HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL,
                            "can't release memory chunk dataspace template");
        if (H5D__chunk_io_term(io_info, dinfo) < 0)
            HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL, "unable to release chunk mapping");
    } 

    if (iter_init && H5S_SELECT_ITER_RELEASE(&(fm->mem_iter)) < 0)
        HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL, "unable to release selection iterator");

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5D__chunk_mem_alloc(size_t size, void *pline)
{
    H5O_pline_t *_pline    = (H5O_pline_t *)pline;
    void        *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(size);

    if (_pline && _pline->nused)
        ret_value = H5MM_malloc(size);
    else
        ret_value = H5FL_BLK_MALLOC(chunk, size);

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5D__chunk_mem_xfree(void *chk, const void *pline)
{
    const H5O_pline_t *_pline = (const H5O_pline_t *)pline;

    FUNC_ENTER_PACKAGE_NOERR

    if (chk) {
        if (_pline && _pline->nused)
            H5MM_xfree(chk);
        else
            chk = H5FL_BLK_FREE(chunk, chk);
    } 

    FUNC_LEAVE_NOAPI(NULL)
} 

void
H5D__chunk_mem_free(void *chk, void *pline)
{
    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    H5D__chunk_mem_xfree(chk, pline);

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
}

void *
H5D__chunk_mem_realloc(void *chk, size_t size, const H5O_pline_t *pline)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(size);
    assert(pline);

    if (pline->nused > 0)
        ret_value = H5MM_realloc(chk, size);
    else
        ret_value = H5FL_BLK_REALLOC(chunk, chk, size);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__free_piece_info(void *item, void H5_ATTR_UNUSED *key, void H5_ATTR_UNUSED *opdata)
{
    H5D_piece_info_t *piece_info = (H5D_piece_info_t *)item;

    FUNC_ENTER_PACKAGE_NOERR

    assert(piece_info);

    
    if (!piece_info->fspace_shared)
        (void)H5S_close(piece_info->fspace);
    else
        H5S_select_all(piece_info->fspace, true);

    
    if (!piece_info->mspace_shared && piece_info->mspace)
        (void)H5S_close((H5S_t *)piece_info->mspace);

    
    piece_info = H5FL_FREE(H5D_piece_info_t, piece_info);

    FUNC_LEAVE_NOAPI(0)
} 

static herr_t
H5D__create_piece_map_single(H5D_dset_io_info_t *di, H5D_io_info_t *io_info)
{
    H5D_chunk_map_t  *fm;                          
    H5D_piece_info_t *piece_info;                  
    hsize_t           coords[H5O_LAYOUT_NDIMS];    
    hsize_t           sel_start[H5O_LAYOUT_NDIMS]; 
    hsize_t           sel_end[H5O_LAYOUT_NDIMS];   
    unsigned          u;                           
    herr_t            ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    fm = di->layout_io_info.chunk_map;

    
    assert(fm);
    assert(fm->f_ndims > 0);

    
    if (H5S_SELECT_BOUNDS(di->file_space, sel_start, sel_end) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't get file selection bound info");

    
    piece_info               = fm->single_piece_info;
    piece_info->piece_points = 1;

    
    for (u = 0; u < fm->f_ndims; u++) {
        
        if (di->layout->u.chunk.dim[u] == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ", u);
        assert(sel_start[u] == sel_end[u]);
        piece_info->scaled[u] = sel_start[u] / di->layout->u.chunk.dim[u];
        coords[u]             = piece_info->scaled[u] * di->layout->u.chunk.dim[u];
    } 
    piece_info->scaled[fm->f_ndims] = 0;

    
    piece_info->index =
        H5VM_array_offset_pre(fm->f_ndims, di->layout->u.chunk.down_chunks, piece_info->scaled);

    
    if (H5S_select_copy(fm->single_space, di->file_space, false) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy file selection");

    
    if (H5S_SELECT_ADJUST_U(fm->single_space, coords) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSELECT, FAIL, "can't adjust chunk selection");

    
    piece_info->fspace = fm->single_space;

    
    piece_info->fspace_shared = true;

    
    piece_info->mspace = di->mem_space;

    
    piece_info->mspace_shared = true;

    
    piece_info->in_place_tconv = false;
    piece_info->buf_off        = 0;

    
    piece_info->filtered_dset = di->dset->shared->dcpl_cache.pline.nused > 0;

    
    piece_info->dset_info = di;

    
    io_info->piece_count++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__create_piece_file_map_all(H5D_dset_io_info_t *di, H5D_io_info_t *io_info)
{
    H5D_chunk_map_t *fm;                      
    H5S_t           *tmp_fchunk = NULL;       
    hsize_t          file_dims[H5S_MAX_RANK]; 
    hsize_t          sel_points;              
    hsize_t zeros[H5S_MAX_RANK];   
    hsize_t  coords[H5S_MAX_RANK]; 
    hsize_t  end[H5S_MAX_RANK];    
    hsize_t  scaled[H5S_MAX_RANK]; 
    hsize_t  chunk_index;          
    hsize_t  curr_partial_clip[H5S_MAX_RANK]; 
    hsize_t  partial_dim_size[H5S_MAX_RANK];  
    bool     is_partial_dim[H5S_MAX_RANK];    
    bool     filtered_dataset;                
    unsigned num_partial_dims;                
    unsigned u;                               
    herr_t   ret_value = SUCCEED;             

    FUNC_ENTER_PACKAGE

    
    fm = di->layout_io_info.chunk_map;

    
    assert(fm);
    assert(fm->f_ndims > 0);

    
    sel_points = di->nelmts;

    
    if (H5S_get_simple_extent_dims(di->file_space, file_dims, NULL) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't get file selection bound info");

    
    num_partial_dims = 0;
    memset(zeros, 0, sizeof(zeros));
    for (u = 0; u < fm->f_ndims; u++) {
        
        if (di->layout->u.chunk.dim[u] == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ", u);

        
        scaled[u] = 0;
        coords[u] = 0;
        end[u]    = fm->chunk_dim[u] - 1;

        
        partial_dim_size[u] = file_dims[u] % fm->chunk_dim[u];
        if (file_dims[u] < fm->chunk_dim[u]) {
            curr_partial_clip[u] = partial_dim_size[u];
            is_partial_dim[u]    = true;
            num_partial_dims++;
        } 
        else {
            curr_partial_clip[u] = fm->chunk_dim[u];
            is_partial_dim[u]    = false;
        } 
    }     

    
    chunk_index = 0;

    
    filtered_dataset = di->dset->shared->dcpl_cache.pline.nused > 0;

    
    if (NULL == (tmp_fchunk = H5S_create_simple(fm->f_ndims, fm->chunk_dim, NULL)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL, "unable to create dataspace for chunk");

    
    while (sel_points) {
        H5D_piece_info_t *new_piece_info; 
        hsize_t           chunk_points;   

        

        
        if (NULL == (new_piece_info = H5FL_MALLOC(H5D_piece_info_t)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate piece info");

        

        
        new_piece_info->index = chunk_index;

        
        if (NULL == (new_piece_info->fspace = H5S_copy(tmp_fchunk, true, false)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy chunk dataspace");
        new_piece_info->fspace_shared = false;

        
        if (num_partial_dims > 0)
            if (H5S_select_hyperslab(new_piece_info->fspace, H5S_SELECT_SET, zeros, NULL, curr_partial_clip,
                                     NULL) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "can't create chunk selection");

        
        new_piece_info->mspace        = NULL;
        new_piece_info->mspace_shared = false;

        
        H5MM_memcpy(new_piece_info->scaled, scaled, sizeof(hsize_t) * fm->f_ndims);
        new_piece_info->scaled[fm->f_ndims] = 0;

        
        new_piece_info->dset_info = di;

        
        new_piece_info->in_place_tconv = false;
        new_piece_info->buf_off        = 0;

        new_piece_info->filtered_dset = filtered_dataset;

        
        if (H5SL_insert(fm->dset_sel_pieces, new_piece_info, &new_piece_info->index) < 0) {
            H5D__free_piece_info(new_piece_info, NULL, NULL);
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINSERT, FAIL, "can't insert chunk into skip list");
        } 

        
        io_info->piece_count++;

        
        chunk_points                 = H5S_GET_SELECT_NPOINTS(new_piece_info->fspace);
        new_piece_info->piece_points = chunk_points;

        
        sel_points -= chunk_points;

        
        if (sel_points > 0) {
            int curr_dim; 

            
            chunk_index++;

            
            curr_dim = (int)fm->f_ndims - 1;

            
            coords[curr_dim] += fm->chunk_dim[curr_dim];
            scaled[curr_dim]++;
            end[curr_dim] += fm->chunk_dim[curr_dim];

            
            if (coords[curr_dim] >= file_dims[curr_dim]) {
                do {
                    
                    coords[curr_dim] = 0;
                    scaled[curr_dim] = 0;
                    end[curr_dim]    = fm->chunk_dim[curr_dim] - 1;

                    
                    if (is_partial_dim[curr_dim] && end[curr_dim] < file_dims[curr_dim]) {
                        
                        assert(num_partial_dims > 0);

                        
                        curr_partial_clip[curr_dim] = fm->chunk_dim[curr_dim];
                        is_partial_dim[curr_dim]    = false;
                        num_partial_dims--;
                    } 

                    
                    curr_dim--;

                    
                    if (curr_dim >= 0) {
                        
                        coords[curr_dim] += fm->chunk_dim[curr_dim];
                        scaled[curr_dim]++;
                        end[curr_dim] = (coords[curr_dim] + fm->chunk_dim[curr_dim]) - 1;
                    } 
                } while (curr_dim >= 0 && (coords[curr_dim] >= file_dims[curr_dim]));
            } 

            
            if (curr_dim >= 0) {
                
                if (!is_partial_dim[curr_dim] && file_dims[curr_dim] <= end[curr_dim]) {
                    
                    curr_partial_clip[curr_dim] = partial_dim_size[curr_dim];
                    is_partial_dim[curr_dim]    = true;
                    num_partial_dims++;

                    
                    assert(num_partial_dims <= fm->f_ndims);
                } 
            }     
        }         
    }             

done:
    
    if (tmp_fchunk && H5S_close(tmp_fchunk) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "can't release temporary dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__create_piece_file_map_hyper(H5D_dset_io_info_t *dinfo, H5D_io_info_t *io_info)
{
    H5D_chunk_map_t *fm;                             
    H5S_t           *tmp_fchunk = NULL;              
    hsize_t          sel_start[H5O_LAYOUT_NDIMS];    
    hsize_t          sel_end[H5O_LAYOUT_NDIMS];      
    hsize_t          sel_points;                     
    hsize_t          start_coords[H5O_LAYOUT_NDIMS]; 
    hsize_t          coords[H5O_LAYOUT_NDIMS];       
    hsize_t          end[H5O_LAYOUT_NDIMS];          
    hsize_t          chunk_index;                    
    hsize_t          start_scaled[H5S_MAX_RANK];     
    hsize_t          scaled[H5S_MAX_RANK];           
    bool             filtered_dataset;               
    int              curr_dim;                       
    unsigned         u;                              
    herr_t           ret_value = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    fm = dinfo->layout_io_info.chunk_map;

    
    assert(fm);
    assert(fm->f_ndims > 0);

    
    sel_points = dinfo->nelmts;

    
    if (H5S_SELECT_BOUNDS(dinfo->file_space, sel_start, sel_end) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't get file selection bound info");

    
    for (u = 0; u < fm->f_ndims; u++) {
        
        if (dinfo->layout->u.chunk.dim[u] == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ", u);
        scaled[u] = start_scaled[u] = sel_start[u] / dinfo->layout->u.chunk.dim[u];
        coords[u] = start_coords[u] = scaled[u] * dinfo->layout->u.chunk.dim[u];
        end[u]                      = (coords[u] + fm->chunk_dim[u]) - 1;
    } 

    
    chunk_index = H5VM_array_offset_pre(fm->f_ndims, dinfo->layout->u.chunk.down_chunks, scaled);

    
    filtered_dataset = dinfo->dset->shared->dcpl_cache.pline.nused > 0;

    
    while (sel_points) {
        
        if (true == H5S_SELECT_INTERSECT_BLOCK(dinfo->file_space, coords, end)) {
            H5D_piece_info_t *new_piece_info; 
            hsize_t           chunk_points;   

            
            if (H5S_combine_hyperslab(dinfo->file_space, H5S_SELECT_AND, coords, NULL, fm->chunk_dim, NULL,
                                      &tmp_fchunk) < 0)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL,
                            "unable to combine file space selection with chunk block");

            
            if (H5S_set_extent_real(tmp_fchunk, fm->chunk_dim) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "can't adjust chunk dimensions");

            
            if (H5S_SELECT_ADJUST_U(tmp_fchunk, coords) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "can't adjust chunk selection");

            

            
            if (NULL == (new_piece_info = H5FL_MALLOC(H5D_piece_info_t)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate chunk info");

            

            
            new_piece_info->index = chunk_index;

            
            new_piece_info->fspace        = tmp_fchunk;
            new_piece_info->fspace_shared = false;
            tmp_fchunk                    = NULL;

            
            new_piece_info->mspace        = NULL;
            new_piece_info->mspace_shared = false;

            
            H5MM_memcpy(new_piece_info->scaled, scaled, sizeof(hsize_t) * fm->f_ndims);
            new_piece_info->scaled[fm->f_ndims] = 0;

            
            new_piece_info->dset_info = dinfo;

            
            new_piece_info->in_place_tconv = false;
            new_piece_info->buf_off        = 0;

            new_piece_info->filtered_dset = filtered_dataset;

            
            io_info->piece_count++;

            
            if (H5SL_insert(fm->dset_sel_pieces, new_piece_info, &new_piece_info->index) < 0) {
                H5D__free_piece_info(new_piece_info, NULL, NULL);
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINSERT, FAIL, "can't insert piece into skip list");
            } 

            
            chunk_points                 = H5S_GET_SELECT_NPOINTS(new_piece_info->fspace);
            new_piece_info->piece_points = chunk_points;

            
            sel_points -= chunk_points;

            
            if (sel_points == 0)
                HGOTO_DONE(SUCCEED);
        } 

        
        chunk_index++;

        
        curr_dim = (int)fm->f_ndims - 1;

        
        coords[curr_dim] += fm->chunk_dim[curr_dim];
        end[curr_dim] += fm->chunk_dim[curr_dim];
        scaled[curr_dim]++;

        
        if (coords[curr_dim] > sel_end[curr_dim]) {
            do {
                
                scaled[curr_dim] = start_scaled[curr_dim];
                coords[curr_dim] = start_coords[curr_dim];
                end[curr_dim]    = (coords[curr_dim] + fm->chunk_dim[curr_dim]) - 1;

                
                curr_dim--;

                
                if (curr_dim >= 0) {
                    
                    scaled[curr_dim]++;
                    coords[curr_dim] += fm->chunk_dim[curr_dim];
                    end[curr_dim] = (coords[curr_dim] + fm->chunk_dim[curr_dim]) - 1;
                } 
            } while (curr_dim >= 0 && (coords[curr_dim] > sel_end[curr_dim]));

            
            chunk_index = H5VM_array_offset_pre(fm->f_ndims, dinfo->layout->u.chunk.down_chunks, scaled);
        } 
    }     

done:
    
    if (ret_value < 0)
        if (tmp_fchunk && H5S_close(tmp_fchunk) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "can't release temporary dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__create_piece_mem_map_hyper(const H5D_dset_io_info_t *dinfo)
{
    H5D_chunk_map_t  *fm;                           
    H5D_piece_info_t *piece_info;                   
    H5SL_node_t      *curr_node;                    
    hsize_t           file_sel_start[H5S_MAX_RANK]; 
    hsize_t           file_sel_end[H5S_MAX_RANK];   
    hsize_t           mem_sel_start[H5S_MAX_RANK];  
    hsize_t           mem_sel_end[H5S_MAX_RANK];    
    hssize_t          adjust[H5S_MAX_RANK];         
    unsigned          u;                            
    herr_t            ret_value = SUCCEED;          

    FUNC_ENTER_PACKAGE

    
    assert(dinfo->layout_io_info.chunk_map->f_ndims > 0);

    
    fm = dinfo->layout_io_info.chunk_map;

    
    if (H5SL_count(fm->dset_sel_pieces) == 1) {
        
        curr_node = H5SL_first(fm->dset_sel_pieces);

        
        piece_info = (H5D_piece_info_t *)H5SL_item(curr_node);
        assert(piece_info);

        
        piece_info->mspace = dinfo->mem_space;

        
        piece_info->mspace_shared = true;
    } 
    else {
        
        if (H5S_SELECT_BOUNDS(dinfo->file_space, file_sel_start, file_sel_end) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't get file selection bound info");

        
        if (H5S_SELECT_BOUNDS(dinfo->mem_space, mem_sel_start, mem_sel_end) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't get file selection bound info");

        
        assert(fm->m_ndims == fm->f_ndims);
        for (u = 0; u < fm->f_ndims; u++) {
            H5_CHECK_OVERFLOW(file_sel_start[u], hsize_t, hssize_t);
            H5_CHECK_OVERFLOW(mem_sel_start[u], hsize_t, hssize_t);
            adjust[u] = (hssize_t)file_sel_start[u] - (hssize_t)mem_sel_start[u];
        } 

        
        assert(fm->dset_sel_pieces);
        curr_node = H5SL_first(fm->dset_sel_pieces);
        while (curr_node) {
            hsize_t      coords[H5S_MAX_RANK];       
            hssize_t     piece_adjust[H5S_MAX_RANK]; 
            H5S_sel_type chunk_sel_type;             

            
            piece_info = (H5D_piece_info_t *)H5SL_item(curr_node);
            assert(piece_info);

            
            for (u = 0; u < fm->f_ndims; u++)
                coords[u] = piece_info->scaled[u] * dinfo->layout->u.chunk.dim[u];

            

            
            if ((piece_info->mspace = H5S_copy(dinfo->mem_space, true, false)) == NULL)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy memory space");

            
            if ((chunk_sel_type = H5S_GET_SELECT_TYPE(piece_info->fspace)) < H5S_SEL_NONE)
                HGOTO_ERROR(H5E_DATASET, H5E_BADSELECT, FAIL, "unable to get type of selection");

            
            if (H5S_SEL_ALL == chunk_sel_type) {
                
                for (u = 0; u < fm->f_ndims; u++)
                    coords[u] = (hsize_t)((hssize_t)coords[u] - adjust[u]);

                
                if (H5S_select_hyperslab(piece_info->mspace, H5S_SELECT_SET, coords, NULL, fm->chunk_dim,
                                         NULL) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "can't create chunk memory selection");
            } 
            else {
                
                assert(H5S_SEL_HYPERSLABS == chunk_sel_type);

                
                if (H5S_SELECT_COPY(piece_info->mspace, piece_info->fspace, false) < 0)
                    HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy selection");

                
                for (u = 0; u < fm->f_ndims; u++) {
                    
                    H5_CHECK_OVERFLOW(coords[u], hsize_t, hssize_t);
                    piece_adjust[u] = adjust[u] - (hssize_t)coords[u];
                } 

                
                if (H5S_SELECT_ADJUST_S(piece_info->mspace, piece_adjust) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to adjust selection");
            } 

            
            curr_node = H5SL_next(curr_node);
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__create_piece_mem_map_1d(const H5D_dset_io_info_t *dinfo)
{
    H5D_chunk_map_t  *fm;                  
    H5D_piece_info_t *piece_info;          
    H5SL_node_t      *curr_node;           
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dinfo->layout_io_info.chunk_map->f_ndims > 0);

    
    fm = dinfo->layout_io_info.chunk_map;
    assert(fm);

    
    if (H5SL_count(fm->dset_sel_pieces) == 1) {
        
        curr_node = H5SL_first(fm->dset_sel_pieces);

        
        piece_info = (H5D_piece_info_t *)H5SL_item(curr_node);
        assert(piece_info);

        
        piece_info->mspace = dinfo->mem_space;

        
        piece_info->mspace_shared = true;
    } 
    else {
        hsize_t mem_sel_start[H5S_MAX_RANK]; 
        hsize_t mem_sel_end[H5S_MAX_RANK];   

        assert(fm->m_ndims == 1);

        if (H5S_SELECT_BOUNDS(dinfo->mem_space, mem_sel_start, mem_sel_end) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "can't get file selection bound info");

        
        curr_node = H5SL_first(fm->dset_sel_pieces);
        while (curr_node) {
            hsize_t chunk_points; 
            hsize_t tmp_count = 1;

            
            piece_info = (H5D_piece_info_t *)H5SL_item(curr_node);
            assert(piece_info);

            
            if ((piece_info->mspace = H5S_copy(dinfo->mem_space, true, false)) == NULL)
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy memory space");

            chunk_points = H5S_GET_SELECT_NPOINTS(piece_info->fspace);

            if (H5S_select_hyperslab(piece_info->mspace, H5S_SELECT_SET, mem_sel_start, NULL, &tmp_count,
                                     &chunk_points) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "can't create chunk memory selection");

            mem_sel_start[0] += chunk_points;

            
            curr_node = H5SL_next(curr_node);
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__piece_file_cb(void H5_ATTR_UNUSED *elem, const H5T_t H5_ATTR_UNUSED *type, unsigned ndims,
                   const hsize_t *coords, void *_opdata)
{
    H5D_io_info_wrap_t *opdata  = (H5D_io_info_wrap_t *)_opdata;
    H5D_io_info_t      *io_info = (H5D_io_info_t *)opdata->io_info;    
    H5D_dset_io_info_t *dinfo   = (H5D_dset_io_info_t *)opdata->dinfo; 
    H5D_chunk_map_t    *fm;                                            
    H5D_piece_info_t   *piece_info;                        
    hsize_t             coords_in_chunk[H5O_LAYOUT_NDIMS]; 
    hsize_t             chunk_index;                       
    hsize_t             scaled[H5S_MAX_RANK];              
    unsigned            u;                                 
    herr_t              ret_value = SUCCEED;               

    FUNC_ENTER_PACKAGE

    
    fm = dinfo->layout_io_info.chunk_map;

    
    chunk_index = H5VM_chunk_index_scaled(ndims, coords, dinfo->layout->u.chunk.dim,
                                          dinfo->layout->u.chunk.down_chunks, scaled);

    
    if (chunk_index == fm->last_index) {
        
        piece_info = fm->last_piece_info;
    } 
    else {
        
        if (NULL == (piece_info = (H5D_piece_info_t *)H5SL_search(fm->dset_sel_pieces, &chunk_index))) {
            H5S_t *fspace; 

            
            if (NULL == (piece_info = H5FL_MALLOC(H5D_piece_info_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate chunk info");

            

            
            piece_info->index = chunk_index;

            
            if ((fspace = H5S_create_simple(fm->f_ndims, fm->chunk_dim, NULL)) == NULL) {
                piece_info = H5FL_FREE(H5D_piece_info_t, piece_info);
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "unable to create dataspace for chunk");
            } 

            
            if (H5S_select_none(fspace) < 0) {
                (void)H5S_close(fspace);
                piece_info = H5FL_FREE(H5D_piece_info_t, piece_info);
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "unable to de-select dataspace");
            } 

            
            piece_info->fspace        = fspace;
            piece_info->fspace_shared = false;

            
            piece_info->mspace        = NULL;
            piece_info->mspace_shared = false;

            
            piece_info->piece_points = 0;

            
            H5MM_memcpy(piece_info->scaled, scaled, sizeof(hsize_t) * fm->f_ndims);
            piece_info->scaled[fm->f_ndims] = 0;

            
            piece_info->in_place_tconv = false;
            piece_info->buf_off        = 0;

            piece_info->filtered_dset = dinfo->dset->shared->dcpl_cache.pline.nused > 0;

            
            piece_info->dset_info = dinfo;

            
            if (H5SL_insert(fm->dset_sel_pieces, piece_info, &piece_info->index) < 0) {
                H5D__free_piece_info(piece_info, NULL, NULL);
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINSERT, FAIL, "can't insert chunk into dataset skip list");
            } 

            
            io_info->piece_count++;
        } 

        
        fm->last_index      = chunk_index;
        fm->last_piece_info = piece_info;
    } 

    
    for (u = 0; u < fm->f_ndims; u++)
        coords_in_chunk[u] = coords[u] - (scaled[u] * dinfo->layout->u.chunk.dim[u]);

    
    if (H5S_select_elements(piece_info->fspace, H5S_SELECT_APPEND, (size_t)1, coords_in_chunk) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSELECT, FAIL, "unable to select element");

    
    piece_info->piece_points++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__piece_mem_cb(void H5_ATTR_UNUSED *elem, const H5T_t H5_ATTR_UNUSED *type, unsigned ndims,
                  const hsize_t *coords, void *_opdata)
{
    H5D_io_info_wrap_t *opdata = (H5D_io_info_wrap_t *)_opdata;
    H5D_dset_io_info_t *dinfo  = (H5D_dset_io_info_t *)opdata->dinfo; 
    H5D_piece_info_t   *piece_info;                  
    H5D_chunk_map_t    *fm;                          
    hsize_t             coords_in_mem[H5S_MAX_RANK]; 
    hsize_t             chunk_index;                 
    herr_t              ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    fm = dinfo->layout_io_info.chunk_map;

    
    chunk_index =
        H5VM_chunk_index(ndims, coords, dinfo->layout->u.chunk.dim, dinfo->layout->u.chunk.down_chunks);

    
    if (chunk_index == fm->last_index) {
        
        piece_info = fm->last_piece_info;
    } 
    else {
        
        
        if (NULL == (piece_info = (H5D_piece_info_t *)H5SL_search(fm->dset_sel_pieces, &chunk_index)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_NOTFOUND, H5_ITER_ERROR,
                        "can't locate piece in dataset skip list");

        
        if (NULL == piece_info->mspace)
            
            if (NULL == (piece_info->mspace = H5S_copy(fm->mchunk_tmpl, false, false)))
                HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, H5_ITER_ERROR, "unable to copy file space");

        
        fm->last_index      = chunk_index;
        fm->last_piece_info = piece_info;
    } 

    
    if (H5S_SELECT_ITER_COORDS(&fm->mem_iter, coords_in_mem) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, H5_ITER_ERROR, "unable to get iterator coordinates");

    
    if (fm->msel_type == H5S_SEL_POINTS) {
        if (H5S_select_elements(piece_info->mspace, H5S_SELECT_APPEND, (size_t)1, coords_in_mem) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSELECT, H5_ITER_ERROR, "unable to select element");
    } 
    else {
        if (H5S_hyper_add_span_element(piece_info->mspace, fm->m_ndims, coords_in_mem) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSELECT, H5_ITER_ERROR, "unable to select element");
    } 

    
    if (H5S_SELECT_ITER_NEXT(&fm->mem_iter, (size_t)1) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTNEXT, H5_ITER_ERROR, "unable to move to next iterator location");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_mdio_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    H5SL_node_t      *piece_node;          
    H5D_piece_info_t *piece_info;          
    H5D_chunk_ud_t    udata;               
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    piece_node = H5D_CHUNK_GET_FIRST_NODE(dinfo);

    
    while (piece_node) {
        
        if (NULL == (piece_info = (H5D_piece_info_t *)H5D_CHUNK_GET_NODE_INFO(dinfo, piece_node)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "couldn't get piece info from list");

        
        if (H5D__chunk_lookup(dinfo->dset, piece_info->scaled, &udata) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

        
        piece_info->faddr = udata.chunk_block.offset;

        
        if (H5_addr_defined(piece_info->faddr)) {
            assert(io_info->sel_pieces);
            assert(io_info->pieces_added < io_info->piece_count);

            
            io_info->sel_pieces[io_info->pieces_added++] = piece_info;

            if (piece_info->filtered_dset)
                io_info->filtered_pieces_added++;
        }

        
        piece_node = H5D_CHUNK_GET_NEXT_NODE(dinfo, piece_node);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5D__chunk_cacheable(const H5D_io_info_t H5_ATTR_PARALLEL_USED *io_info, H5D_dset_io_info_t *dset_info,
                     haddr_t caddr, bool write_op)
{
    const H5D_t *dataset     = NULL;  
    bool         has_filters = false; 
    htri_t       ret_value   = FAIL;  

    FUNC_ENTER_PACKAGE

    
    assert(dset_info);
    dataset = dset_info->dset;
    assert(dataset);

    
    if (dataset->shared->dcpl_cache.pline.nused > 0) {
        if (dataset->shared->layout.u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) {
            has_filters =
                !H5D__chunk_is_partial_edge_chunk(dataset->shared->ndims, dataset->shared->layout.u.chunk.dim,
                                                  dset_info->store->chunk.scaled, dataset->shared->curr_dims);
        } 
        else
            has_filters = true;
    } 

    if (has_filters)
        ret_value = true;
    else {
#ifdef H5_HAVE_PARALLEL
        
        if (io_info->using_mpi_vfd && (H5F_ACC_RDWR & H5F_INTENT(dataset->oloc.file)))
            ret_value = false;
        else {
#endif 
            
            
            if (!(((hsize_t)((size_t)dataset->shared->layout.u.chunk.size) ==
                   dataset->shared->layout.u.chunk.size) &&
                  ((size_t)dataset->shared->layout.u.chunk.size <=
                   dataset->shared->cache.chunk.nbytes_max))) {
                if (write_op && !H5_addr_defined(caddr)) {
                    const H5O_fill_t *fill = &(dataset->shared->dcpl_cache.fill); 
                    H5D_fill_value_t  fill_status;                                

                    
                    if (H5P_is_fill_value_defined(fill, &fill_status) < 0)
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't tell if fill value defined");

                    
                    if (fill->fill_time == H5D_FILL_TIME_ALLOC ||
                        (fill->fill_time == H5D_FILL_TIME_IFSET &&
                         (fill_status == H5D_FILL_VALUE_USER_DEFINED ||
                          fill_status == H5D_FILL_VALUE_DEFAULT))) {
                        
                        if (H5_UNLIKELY((hsize_t)((size_t)dataset->shared->layout.u.chunk.size) !=
                                        dataset->shared->layout.u.chunk.size))
                            HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL,
                                        "must fill chunk but it is too big to fit in size_t - try using "
                                        "early allocation");
                        ret_value = true;
                    }
                    else
                        ret_value = false;
                }
                else
                    ret_value = false;
            }
            else
                ret_value = true;
#ifdef H5_HAVE_PARALLEL
        } 
#endif    
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_may_use_select_io(H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info)
{
    const H5D_t *dataset   = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);

    dataset = dset_info->dset;
    assert(dataset);

    
    if (dataset->shared->dcpl_cache.pline.nused > 0) {
        io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
        io_info->no_selection_io_cause |= H5D_SEL_IO_DATASET_FILTER;
    }
    else {
        bool page_buf_enabled;

        
        if (H5PB_enabled(io_info->f_sh, H5FD_MEM_DRAW, &page_buf_enabled) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if page buffer is enabled");
        if (page_buf_enabled) {
            
            io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
            io_info->no_selection_io_cause |= H5D_SEL_IO_PAGE_BUFFER;
        }
        else {
            
#ifdef H5_HAVE_PARALLEL
            
            if (!(io_info->using_mpi_vfd && (H5F_ACC_RDWR & H5F_INTENT(dataset->oloc.file)))) {
#endif 
                
                
                if (((hsize_t)((size_t)dataset->shared->layout.u.chunk.size) ==
                     dataset->shared->layout.u.chunk.size) &&
                    ((size_t)dataset->shared->layout.u.chunk.size <=
                     dataset->shared->cache.chunk.nbytes_max)) {
                    io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
                    io_info->no_selection_io_cause |= H5D_SEL_IO_CHUNK_CACHE;
                }
#ifdef H5_HAVE_PARALLEL
            } 
#endif        
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_read(H5D_io_info_t *io_info, H5D_dset_io_info_t *dset_info)
{
    H5SL_node_t       *chunk_node;                  
    H5D_io_info_t      nonexistent_io_info;         
    H5D_dset_io_info_t nonexistent_dset_info;       
    H5D_dset_io_info_t ctg_dset_info;               
    H5D_dset_io_info_t cpt_dset_info;               
    hsize_t            src_accessed_bytes  = 0;     
    bool               skip_missing_chunks = false; 
    H5S_t            **chunk_mem_spaces    = NULL;  
    H5S_t             *chunk_mem_spaces_local[8];   
    H5S_t            **chunk_file_spaces = NULL;    
    H5S_t             *chunk_file_spaces_local[8];  
    haddr_t           *chunk_addrs = NULL;          
    haddr_t            chunk_addrs_local[8];        
    herr_t             ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->buf.vp);

    
    H5MM_memcpy(&nonexistent_io_info, io_info, sizeof(nonexistent_io_info));
    H5MM_memcpy(&nonexistent_dset_info, dset_info, sizeof(nonexistent_dset_info));
    nonexistent_dset_info.layout_ops = *H5D_LOPS_NONEXISTENT;
    nonexistent_io_info.dsets_info   = &nonexistent_dset_info;
    nonexistent_io_info.count        = 1;

    {
        const H5O_fill_t *fill = &(dset_info->dset->shared->dcpl_cache.fill); 
        H5D_fill_value_t  fill_status;                                        

        
        if (H5P_is_fill_value_defined(fill, &fill_status) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't tell if fill value defined");

        
        if (fill->fill_time == H5D_FILL_TIME_NEVER ||
            (fill->fill_time == H5D_FILL_TIME_IFSET && fill_status != H5D_FILL_VALUE_USER_DEFINED &&
             fill_status != H5D_FILL_VALUE_DEFAULT))
            skip_missing_chunks = true;
    }

    
    if (io_info->use_select_io == H5D_SELECTION_IO_MODE_ON) {
        size_t num_chunks       = 0;
        size_t element_sizes[2] = {dset_info->type_info.src_type_size, 0};
        void  *bufs[2]          = {dset_info->buf.vp, NULL};

        
        if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
            
            num_chunks = H5D_CHUNK_GET_NODE_COUNT(dset_info);

            
            assert(sizeof(chunk_mem_spaces_local) / sizeof(chunk_mem_spaces_local[0]) ==
                   sizeof(chunk_file_spaces_local) / sizeof(chunk_file_spaces_local[0]));
            assert(sizeof(chunk_mem_spaces_local) / sizeof(chunk_mem_spaces_local[0]) ==
                   sizeof(chunk_addrs_local) / sizeof(chunk_addrs_local[0]));
            if (num_chunks > (sizeof(chunk_mem_spaces_local) / sizeof(chunk_mem_spaces_local[0]))) {
                if (NULL == (chunk_mem_spaces = H5MM_malloc(num_chunks * sizeof(H5S_t *))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "memory allocation failed for memory space list");
                if (NULL == (chunk_file_spaces = H5MM_malloc(num_chunks * sizeof(H5S_t *))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "memory allocation failed for file space list");
                if (NULL == (chunk_addrs = H5MM_malloc(num_chunks * sizeof(haddr_t))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "memory allocation failed for chunk address list");
            } 
            else {
                chunk_mem_spaces  = chunk_mem_spaces_local;
                chunk_file_spaces = chunk_file_spaces_local;
                chunk_addrs       = chunk_addrs_local;
            } 

            
            num_chunks = 0;
        } 

        
        chunk_node = H5D_CHUNK_GET_FIRST_NODE(dset_info);
        while (chunk_node) {
            H5D_piece_info_t *chunk_info; 
            H5D_chunk_ud_t    udata;      

            
            chunk_info = H5D_CHUNK_GET_NODE_INFO(dset_info, chunk_node);

            
            if (H5D__chunk_lookup(dset_info->dset, chunk_info->scaled, &udata) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

            
            assert(UINT_MAX == udata.idx_hint);

            
            assert((H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length > 0) ||
                   (!H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length == 0));

            
            if (H5_addr_defined(udata.chunk_block.offset)) {
                
                if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
                    chunk_mem_spaces[num_chunks]  = chunk_info->mspace;
                    chunk_file_spaces[num_chunks] = chunk_info->fspace;
                    chunk_addrs[num_chunks]       = udata.chunk_block.offset;
                    num_chunks++;
                } 
                else {
                    
                    assert(io_info->mem_spaces);
                    assert(io_info->file_spaces);
                    assert(io_info->addrs);
                    assert(io_info->element_sizes);
                    assert(io_info->rbufs);
                    assert(io_info->pieces_added < io_info->piece_count);

                    io_info->mem_spaces[io_info->pieces_added]    = chunk_info->mspace;
                    io_info->file_spaces[io_info->pieces_added]   = chunk_info->fspace;
                    io_info->addrs[io_info->pieces_added]         = udata.chunk_block.offset;
                    io_info->element_sizes[io_info->pieces_added] = element_sizes[0];
                    io_info->rbufs[io_info->pieces_added]         = bufs[0];
                    if (io_info->sel_pieces)
                        io_info->sel_pieces[io_info->pieces_added] = chunk_info;
                    io_info->pieces_added++;

                    if (io_info->sel_pieces && chunk_info->filtered_dset)
                        io_info->filtered_pieces_added++;
                }
            } 
            else if (!skip_missing_chunks) {
                
                nonexistent_dset_info.layout_io_info.contig_piece_info = chunk_info;
                nonexistent_dset_info.file_space                       = chunk_info->fspace;
                nonexistent_dset_info.mem_space                        = chunk_info->mspace;
                nonexistent_dset_info.nelmts                           = chunk_info->piece_points;

                
                H5_CHECKED_ASSIGN(nonexistent_dset_info.type_info.request_nelmts, size_t,
                                  nonexistent_dset_info.nelmts, hsize_t);

                
                if ((dset_info->io_ops.single_read)(&nonexistent_io_info, &nonexistent_dset_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "chunked read failed");
            } 

            
            chunk_node = H5D_CHUNK_GET_NEXT_NODE(dset_info, chunk_node);
        } 

        
        if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
            
            H5_CHECK_OVERFLOW(num_chunks, size_t, uint32_t);
            if (H5F_shared_select_read(H5F_SHARED(dset_info->dset->oloc.file), H5FD_MEM_DRAW,
                                       (uint32_t)num_chunks, chunk_mem_spaces, chunk_file_spaces, chunk_addrs,
                                       element_sizes, bufs) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "chunk selection read failed");

            
            if (chunk_mem_spaces != chunk_mem_spaces_local) {
                assert(chunk_file_spaces != chunk_file_spaces_local);
                assert(chunk_addrs != chunk_addrs_local);
                chunk_mem_spaces  = H5MM_xfree(chunk_mem_spaces);
                chunk_file_spaces = H5MM_xfree(chunk_file_spaces);
                chunk_addrs       = H5MM_xfree(chunk_addrs);
            } 
        }     

#ifdef H5_HAVE_PARALLEL
        
        io_info->actual_io_mode |= H5D_MPIO_CHUNK_COLLECTIVE;
#endif 
    }  
    else {
        H5D_io_info_t ctg_io_info; 
        H5D_storage_t ctg_store;   
        H5D_io_info_t cpt_io_info; 
        H5D_storage_t cpt_store;   
        bool          cpt_dirty;   

        
        H5MM_memcpy(&ctg_io_info, io_info, sizeof(ctg_io_info));
        H5MM_memcpy(&ctg_dset_info, dset_info, sizeof(ctg_dset_info));
        ctg_dset_info.store      = &ctg_store;
        ctg_dset_info.layout_ops = *H5D_LOPS_CONTIG;
        ctg_io_info.dsets_info   = &ctg_dset_info;
        ctg_io_info.count        = 1;

        
        ctg_store.contig.dset_size = dset_info->dset->shared->layout.u.chunk.size;

        
        H5MM_memcpy(&cpt_io_info, io_info, sizeof(cpt_io_info));
        H5MM_memcpy(&cpt_dset_info, dset_info, sizeof(cpt_dset_info));
        cpt_dset_info.store      = &cpt_store;
        cpt_dset_info.layout_ops = *H5D_LOPS_COMPACT;
        cpt_io_info.dsets_info   = &cpt_dset_info;
        cpt_io_info.count        = 1;

        
        cpt_store.compact.dirty = &cpt_dirty;

        
        chunk_node = H5D_CHUNK_GET_FIRST_NODE(dset_info);
        while (chunk_node) {
            H5D_piece_info_t *chunk_info; 
            H5D_chunk_ud_t    udata;      
            htri_t            cacheable;  

            
            chunk_info = H5D_CHUNK_GET_NODE_INFO(dset_info, chunk_node);

            
            if (H5D__chunk_lookup(dset_info->dset, chunk_info->scaled, &udata) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

            
            assert((H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length > 0) ||
                   (!H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length == 0));

            
            if (H5_addr_defined(udata.chunk_block.offset) || UINT_MAX != udata.idx_hint ||
                !skip_missing_chunks) {
                H5D_io_info_t *chk_io_info;  
                void          *chunk = NULL; 

                
                dset_info->store->chunk.scaled = chunk_info->scaled;

                
                if ((cacheable = H5D__chunk_cacheable(io_info, dset_info, udata.chunk_block.offset, false)) <
                    0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't tell if chunk is cacheable");
                if (cacheable) {
                    

                    
                    H5_CHECK_OVERFLOW(dset_info->type_info.src_type_size,  size_t,  hsize_t);
                    src_accessed_bytes =
                        chunk_info->piece_points * (hsize_t)dset_info->type_info.src_type_size;

                    
                    if (NULL == (chunk = H5D__chunk_lock(io_info, dset_info, &udata, false, false)))
                        HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "unable to read raw data chunk");

                    
                    cpt_store.compact.buf = chunk;

                    
                    chk_io_info = &cpt_io_info;
                } 
                else if (H5_addr_defined(udata.chunk_block.offset)) {
                    
                    ctg_store.contig.dset_addr = udata.chunk_block.offset;

                    
                    chk_io_info = &ctg_io_info;
                } 
                else {
                    
                    chk_io_info = &nonexistent_io_info;
                } 

                
                assert(chk_io_info->count == 1);
                chk_io_info->dsets_info[0].layout_io_info.contig_piece_info = chunk_info;
                chk_io_info->dsets_info[0].file_space                       = chunk_info->fspace;
                chk_io_info->dsets_info[0].mem_space                        = chunk_info->mspace;
                chk_io_info->dsets_info[0].nelmts                           = chunk_info->piece_points;
                if ((dset_info->io_ops.single_read)(chk_io_info, &chk_io_info->dsets_info[0]) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "chunked read failed");

                
                if (chunk &&
                    H5D__chunk_unlock(io_info, dset_info, &udata, false, chunk, src_accessed_bytes) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "unable to unlock raw data chunk");
            } 

            
            chunk_node = H5D_CHUNK_GET_NEXT_NODE(dset_info, chunk_node);
        } 
    }     

done:
    
    if (ret_value < 0) {
        if (chunk_mem_spaces != chunk_mem_spaces_local)
            chunk_mem_spaces = H5MM_xfree(chunk_mem_spaces);
        if (chunk_file_spaces != chunk_file_spaces_local)
            chunk_file_spaces = H5MM_xfree(chunk_file_spaces);
        if (chunk_addrs != chunk_addrs_local)
            chunk_addrs = H5MM_xfree(chunk_addrs);
    } 

    
    assert(!chunk_mem_spaces || chunk_mem_spaces == chunk_mem_spaces_local);
    assert(!chunk_file_spaces || chunk_file_spaces == chunk_file_spaces_local);
    assert(!chunk_addrs || chunk_addrs == chunk_addrs_local);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_write(H5D_io_info_t *io_info, H5D_dset_io_info_t *dset_info)
{
    H5SL_node_t       *chunk_node;                
    H5D_io_info_t      ctg_io_info;               
    H5D_dset_io_info_t ctg_dset_info;             
    H5D_storage_t      ctg_store;                 
    H5D_io_info_t      cpt_io_info;               
    H5D_dset_io_info_t cpt_dset_info;             
    H5D_storage_t      cpt_store;                 
    bool               cpt_dirty;                 
    hsize_t            dst_accessed_bytes = 0;    
    H5S_t            **chunk_mem_spaces   = NULL; 
    H5S_t             *chunk_mem_spaces_local[8]; 
    H5S_t            **chunk_file_spaces = NULL;  
    H5S_t             *chunk_file_spaces_local[8]; 
    haddr_t           *chunk_addrs = NULL;         
    haddr_t            chunk_addrs_local[8];       
    herr_t             ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->buf.cvp);

    
    H5MM_memcpy(&ctg_io_info, io_info, sizeof(ctg_io_info));
    H5MM_memcpy(&ctg_dset_info, dset_info, sizeof(ctg_dset_info));
    ctg_dset_info.store      = &ctg_store;
    ctg_dset_info.layout_ops = *H5D_LOPS_CONTIG;
    ctg_io_info.dsets_info   = &ctg_dset_info;
    ctg_io_info.count        = 1;

    
    ctg_store.contig.dset_size = dset_info->dset->shared->layout.u.chunk.size;

    
    H5MM_memcpy(&cpt_io_info, io_info, sizeof(cpt_io_info));
    H5MM_memcpy(&cpt_dset_info, dset_info, sizeof(cpt_dset_info));
    cpt_dset_info.store      = &cpt_store;
    cpt_dset_info.layout_ops = *H5D_LOPS_COMPACT;
    cpt_io_info.dsets_info   = &cpt_dset_info;
    cpt_io_info.count        = 1;

    
    cpt_store.compact.dirty = &cpt_dirty;

    
    if (io_info->use_select_io == H5D_SELECTION_IO_MODE_ON) {
        size_t      num_chunks       = 0;
        size_t      element_sizes[2] = {dset_info->type_info.dst_type_size, 0};
        const void *bufs[2]          = {dset_info->buf.cvp, NULL};

        
        if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
            
            num_chunks = H5D_CHUNK_GET_NODE_COUNT(dset_info);

            
            assert(sizeof(chunk_mem_spaces_local) / sizeof(chunk_mem_spaces_local[0]) ==
                   sizeof(chunk_file_spaces_local) / sizeof(chunk_file_spaces_local[0]));
            assert(sizeof(chunk_mem_spaces_local) / sizeof(chunk_mem_spaces_local[0]) ==
                   sizeof(chunk_addrs_local) / sizeof(chunk_addrs_local[0]));
            if (num_chunks > (sizeof(chunk_mem_spaces_local) / sizeof(chunk_mem_spaces_local[0]))) {
                if (NULL == (chunk_mem_spaces = H5MM_malloc(num_chunks * sizeof(H5S_t *))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "memory allocation failed for memory space list");
                if (NULL == (chunk_file_spaces = H5MM_malloc(num_chunks * sizeof(H5S_t *))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "memory allocation failed for file space list");
                if (NULL == (chunk_addrs = H5MM_malloc(num_chunks * sizeof(haddr_t))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                "memory allocation failed for chunk address list");
            } 
            else {
                chunk_mem_spaces  = chunk_mem_spaces_local;
                chunk_file_spaces = chunk_file_spaces_local;
                chunk_addrs       = chunk_addrs_local;
            } 

            
            num_chunks = 0;
        } 

        
        chunk_node = H5D_CHUNK_GET_FIRST_NODE(dset_info);
        while (chunk_node) {
            H5D_piece_info_t  *chunk_info; 
            H5D_chk_idx_info_t idx_info;   
            H5D_chunk_ud_t     udata;      
            htri_t             cacheable;  
            bool need_insert = false;      

            
            chunk_info = H5D_CHUNK_GET_NODE_INFO(dset_info, chunk_node);

            
            if (H5D__chunk_lookup(dset_info->dset, chunk_info->scaled, &udata) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

            
            assert(UINT_MAX == udata.idx_hint);

            
            assert((H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length > 0) ||
                   (!H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length == 0));

            
            dset_info->store->chunk.scaled = chunk_info->scaled;

            
            if ((cacheable = H5D__chunk_cacheable(io_info, dset_info, udata.chunk_block.offset, true)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't tell if chunk is cacheable");
            if (cacheable) {
                
                void *chunk;               
                bool  entire_chunk = true; 

                
                H5_CHECK_OVERFLOW(dset_info->type_info.dst_type_size,  size_t,  hsize_t);
                dst_accessed_bytes = chunk_info->piece_points * (hsize_t)dset_info->type_info.dst_type_size;

                
                if (dst_accessed_bytes != ctg_store.contig.dset_size ||
                    (chunk_info->piece_points * dset_info->type_info.src_type_size) !=
                        ctg_store.contig.dset_size ||
                    dset_info->layout_io_info.chunk_map->fsel_type == H5S_SEL_POINTS)
                    entire_chunk = false;

                
                if (NULL == (chunk = H5D__chunk_lock(io_info, dset_info, &udata, entire_chunk, false)))
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "unable to read raw data chunk");

                
                cpt_store.compact.buf = chunk;

                
                cpt_dset_info.layout_io_info.contig_piece_info = chunk_info;
                cpt_dset_info.file_space                       = chunk_info->fspace;
                cpt_dset_info.mem_space                        = chunk_info->mspace;
                cpt_dset_info.nelmts                           = chunk_info->piece_points;

                
                H5_CHECKED_ASSIGN(cpt_dset_info.type_info.request_nelmts, size_t, cpt_dset_info.nelmts,
                                  hsize_t);

                
                if ((dset_info->io_ops.single_write)(&cpt_io_info, &cpt_dset_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "chunked write failed");

                
                if (H5D__chunk_unlock(io_info, dset_info, &udata, true, chunk, dst_accessed_bytes) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "unable to unlock raw data chunk");
            } 
            else {
                
                if (!H5_addr_defined(udata.chunk_block.offset)) {
                    
                    idx_info.f      = dset_info->dset->oloc.file;
                    idx_info.pline  = &(dset_info->dset->shared->dcpl_cache.pline);
                    idx_info.layout = &(dset_info->dset->shared->layout);

                    
                    udata.chunk_block.length = dset_info->dset->shared->layout.u.chunk.size;

                    
                    if (H5D__chunk_file_alloc(&idx_info, NULL, &udata.chunk_block, &need_insert,
                                              chunk_info->scaled) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL,
                                    "unable to insert/resize chunk on chunk level");

                    
                    if (!H5_addr_defined(udata.chunk_block.offset))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "chunk address isn't defined");

                    
                    H5D__chunk_cinfo_cache_update(&dset_info->dset->shared->cache.chunk.last, &udata);

                    
                    if (need_insert && dset_info->dset->shared->layout.storage.u.chunk.ops->insert)
                        if ((dset_info->dset->shared->layout.storage.u.chunk.ops->insert)(&idx_info, &udata,
                                                                                          NULL) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL,
                                        "unable to insert chunk addr into index");
                } 

                
                if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
                    chunk_mem_spaces[num_chunks]  = chunk_info->mspace;
                    chunk_file_spaces[num_chunks] = chunk_info->fspace;
                    chunk_addrs[num_chunks]       = udata.chunk_block.offset;
                    num_chunks++;
                } 
                else {
                    
                    assert(io_info->mem_spaces);
                    assert(io_info->file_spaces);
                    assert(io_info->addrs);
                    assert(io_info->element_sizes);
                    assert(io_info->wbufs);
                    assert(io_info->pieces_added < io_info->piece_count);

                    io_info->mem_spaces[io_info->pieces_added]    = chunk_info->mspace;
                    io_info->file_spaces[io_info->pieces_added]   = chunk_info->fspace;
                    io_info->addrs[io_info->pieces_added]         = udata.chunk_block.offset;
                    io_info->element_sizes[io_info->pieces_added] = element_sizes[0];
                    io_info->wbufs[io_info->pieces_added]         = bufs[0];
                    if (io_info->sel_pieces)
                        io_info->sel_pieces[io_info->pieces_added] = chunk_info;
                    io_info->pieces_added++;

                    if (io_info->sel_pieces && chunk_info->filtered_dset)
                        io_info->filtered_pieces_added++;
                }
            } 

            
            chunk_node = H5D_CHUNK_GET_NEXT_NODE(dset_info, chunk_node);
        } 

        
        if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
            
            H5_CHECK_OVERFLOW(num_chunks, size_t, uint32_t);
            if (H5F_shared_select_write(H5F_SHARED(dset_info->dset->oloc.file), H5FD_MEM_DRAW,
                                        (uint32_t)num_chunks, chunk_mem_spaces, chunk_file_spaces,
                                        chunk_addrs, element_sizes, bufs) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "chunk selection write failed");

            
            if (chunk_mem_spaces != chunk_mem_spaces_local) {
                assert(chunk_file_spaces != chunk_file_spaces_local);
                assert(chunk_addrs != chunk_addrs_local);
                chunk_mem_spaces  = H5MM_xfree(chunk_mem_spaces);
                chunk_file_spaces = H5MM_xfree(chunk_file_spaces);
                chunk_addrs       = H5MM_xfree(chunk_addrs);
            } 
        }     

#ifdef H5_HAVE_PARALLEL
        
        io_info->actual_io_mode |= H5D_MPIO_CHUNK_COLLECTIVE;
#endif 
    }  
    else {
        
        chunk_node = H5D_CHUNK_GET_FIRST_NODE(dset_info);
        while (chunk_node) {
            H5D_piece_info_t  *chunk_info;  
            H5D_chk_idx_info_t idx_info;    
            H5D_io_info_t     *chk_io_info; 
            void              *chunk;       
            H5D_chunk_ud_t     udata;       
            htri_t             cacheable;   
            bool need_insert = false;       

            
            chunk_info = H5D_CHUNK_GET_NODE_INFO(dset_info, chunk_node);

            
            if (H5D__chunk_lookup(dset_info->dset, chunk_info->scaled, &udata) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

            
            assert((H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length > 0) ||
                   (!H5_addr_defined(udata.chunk_block.offset) && udata.chunk_block.length == 0));

            
            dset_info->store->chunk.scaled = chunk_info->scaled;

            
            if ((cacheable = H5D__chunk_cacheable(io_info, dset_info, udata.chunk_block.offset, true)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't tell if chunk is cacheable");
            if (cacheable) {
                
                bool entire_chunk = true; 

                
                H5_CHECK_OVERFLOW(dset_info->type_info.dst_type_size,  size_t,  hsize_t);
                dst_accessed_bytes = chunk_info->piece_points * (hsize_t)dset_info->type_info.dst_type_size;

                
                if (dst_accessed_bytes != ctg_store.contig.dset_size ||
                    (chunk_info->piece_points * dset_info->type_info.src_type_size) !=
                        ctg_store.contig.dset_size ||
                    dset_info->layout_io_info.chunk_map->fsel_type == H5S_SEL_POINTS)
                    entire_chunk = false;

                
                if (NULL == (chunk = H5D__chunk_lock(io_info, dset_info, &udata, entire_chunk, false)))
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "unable to read raw data chunk");

                
                cpt_store.compact.buf = chunk;

                
                chk_io_info = &cpt_io_info;
            } 
            else {
                
                if (!H5_addr_defined(udata.chunk_block.offset)) {
                    
                    idx_info.f      = dset_info->dset->oloc.file;
                    idx_info.pline  = &(dset_info->dset->shared->dcpl_cache.pline);
                    idx_info.layout = &(dset_info->dset->shared->layout);

                    
                    udata.chunk_block.length = dset_info->dset->shared->layout.u.chunk.size;

                    
                    if (H5D__chunk_file_alloc(&idx_info, NULL, &udata.chunk_block, &need_insert,
                                              chunk_info->scaled) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL,
                                    "unable to insert/resize chunk on chunk level");

                    
                    if (!H5_addr_defined(udata.chunk_block.offset))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "chunk address isn't defined");

                    
                    H5D__chunk_cinfo_cache_update(&dset_info->dset->shared->cache.chunk.last, &udata);
                } 

                
                ctg_store.contig.dset_addr = udata.chunk_block.offset;

                
                chunk = NULL;

                
                chk_io_info = &ctg_io_info;
            } 

            
            assert(chk_io_info->count == 1);
            chk_io_info->dsets_info[0].layout_io_info.contig_piece_info = chunk_info;
            chk_io_info->dsets_info[0].file_space                       = chunk_info->fspace;
            chk_io_info->dsets_info[0].mem_space                        = chunk_info->mspace;
            chk_io_info->dsets_info[0].nelmts                           = chunk_info->piece_points;
            if ((dset_info->io_ops.single_write)(chk_io_info, &chk_io_info->dsets_info[0]) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "chunked write failed");

            
            if (chunk) {
                if (H5D__chunk_unlock(io_info, dset_info, &udata, true, chunk, dst_accessed_bytes) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "unable to unlock raw data chunk");
            } 
            else {
                if (need_insert && dset_info->dset->shared->layout.storage.u.chunk.ops->insert)
                    if ((dset_info->dset->shared->layout.storage.u.chunk.ops->insert)(&idx_info, &udata,
                                                                                      NULL) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL,
                                    "unable to insert chunk addr into index");
            } 

            
            chunk_node = H5D_CHUNK_GET_NEXT_NODE(dset_info, chunk_node);
        } 
    }     

done:
    
    if (ret_value < 0) {
        if (chunk_mem_spaces != chunk_mem_spaces_local)
            chunk_mem_spaces = H5MM_xfree(chunk_mem_spaces);
        if (chunk_file_spaces != chunk_file_spaces_local)
            chunk_file_spaces = H5MM_xfree(chunk_file_spaces);
        if (chunk_addrs != chunk_addrs_local)
            chunk_addrs = H5MM_xfree(chunk_addrs);
    } 

    
    assert(!chunk_mem_spaces || chunk_mem_spaces == chunk_mem_spaces_local);
    assert(!chunk_file_spaces || chunk_file_spaces == chunk_file_spaces_local);
    assert(!chunk_addrs || chunk_addrs == chunk_addrs_local);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_flush(H5D_t *dset)
{
    H5D_rdcc_t     *rdcc = &(dset->shared->cache.chunk);
    H5D_rdcc_ent_t *ent, *next;
    unsigned        nerrors   = 0;       
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    for (ent = rdcc->head; ent; ent = next) {
        next = ent->next;
        if (H5D__chunk_flush_entry(dset, ent, false) < 0)
            nerrors++;
    } 
    if (nerrors)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to flush one or more raw data chunks");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_io_term(H5D_io_info_t H5_ATTR_UNUSED *io_info, H5D_dset_io_info_t *di)
{
    H5D_chunk_map_t *fm;                  
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(di);

    
    fm = di->layout_io_info.chunk_map;

    
    if (fm->use_single) {
        
        assert(fm->dset_sel_pieces == NULL);
        assert(fm->last_piece_info == NULL);
        assert(fm->single_piece_info);
        assert(fm->single_piece_info->fspace_shared);
        assert(fm->single_piece_info->mspace_shared);

        
        H5S_select_all(fm->single_space, true);
    } 
    else {
        
        if (fm->dset_sel_pieces) {
            if (H5SL_free(fm->dset_sel_pieces, H5D__free_piece_info, NULL) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTNEXT, FAIL, "can't free dataset skip list");
        } 
        else if (fm->last_piece_info) {
            if (H5D__free_piece_info(fm->last_piece_info, NULL, NULL) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't free piece info");
            fm->last_piece_info = NULL;
        } 
    }     

    
    if (fm->mchunk_tmpl)
        if (H5S_close(fm->mchunk_tmpl) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL,
                        "can't release memory chunk dataspace template");

    
    di->layout_io_info.chunk_map = H5FL_FREE(H5D_chunk_map_t, di->layout_io_info.chunk_map);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_dest(H5D_t *dset)
{
    H5D_chk_idx_info_t   idx_info;                            
    H5D_rdcc_t          *rdcc = &(dset->shared->cache.chunk); 
    H5D_rdcc_ent_t      *ent = NULL, *next = NULL;            
    int                  nerrors   = 0;                       
    H5O_storage_chunk_t *sc        = &(dset->shared->layout.storage.u.chunk);
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(dset);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

    
    for (ent = rdcc->head; ent; ent = next) {
        next = ent->next;
        if (H5D__chunk_cache_evict(dset, ent, true) < 0)
            nerrors++;
    } 

    
    if (nerrors)
        HDONE_ERROR(H5E_IO, H5E_CANTFLUSH, FAIL, "unable to flush one or more raw data chunks");

    
    if (rdcc->slot)
        rdcc->slot = H5FL_SEQ_FREE(H5D_rdcc_ent_ptr_t, rdcc->slot);
    memset(rdcc, 0, sizeof(H5D_rdcc_t));

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if (sc->ops->dest && (sc->ops->dest)(&idx_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to release chunk index info");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D_chunk_idx_reset(H5O_storage_chunk_t *storage, bool reset_addr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(storage);
    assert(storage->ops);
    H5D_CHUNK_STORAGE_INDEX_CHK(storage);

    
    if ((storage->ops->reset)(storage, reset_addr) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to reset chunk index info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_cinfo_cache_reset(H5D_chunk_cached_t *last)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(last);

    
    last->valid = false;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__chunk_cinfo_cache_update(H5D_chunk_cached_t *last, const H5D_chunk_ud_t *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(last);
    assert(udata);
    assert(udata->common.layout);
    assert(udata->common.scaled);

    
    H5MM_memcpy(last->scaled, udata->common.scaled, sizeof(hsize_t) * udata->common.layout->ndims);
    last->addr        = udata->chunk_block.offset;
    last->nbytes      = udata->chunk_block.length;
    last->chunk_idx   = udata->chunk_idx;
    last->filter_mask = udata->filter_mask;

    
    last->valid = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static bool
H5D__chunk_cinfo_cache_found(const H5D_chunk_cached_t *last, H5D_chunk_ud_t *udata)
{
    bool ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(last);
    assert(udata);
    assert(udata->common.layout);
    assert(udata->common.scaled);

    
    if (last->valid) {
        unsigned u; 

        
        for (u = 0; u < udata->common.layout->ndims; u++)
            if (last->scaled[u] != udata->common.scaled[u])
                HGOTO_DONE(false);

        
        udata->chunk_block.offset = last->addr;
        udata->chunk_block.length = last->nbytes;
        udata->chunk_idx          = last->chunk_idx;
        udata->filter_mask        = last->filter_mask;

        
        HGOTO_DONE(true);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_create(const H5D_t *dset )
{
    H5D_chk_idx_info_t   idx_info; 
    H5O_storage_chunk_t *sc        = &(dset->shared->layout.storage.u.chunk);
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(H5D_CHUNKED == dset->shared->layout.type);
    assert(dset->shared->layout.u.chunk.ndims > 0 && dset->shared->layout.u.chunk.ndims <= H5O_LAYOUT_NDIMS);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

#ifndef NDEBUG
    {
        unsigned u; 

        for (u = 0; u < dset->shared->layout.u.chunk.ndims; u++)
            assert(dset->shared->layout.u.chunk.dim[u] > 0);
    }
#endif

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if ((sc->ops->create)(&idx_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't create chunk index");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static unsigned
H5D__chunk_hash_val(const H5D_shared_t *shared, const hsize_t *scaled)
{
    hsize_t  val;                   
    unsigned ndims = shared->ndims; 
    unsigned ret   = 0;             
    unsigned u;                     

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(shared);
    assert(scaled);

    
    val = scaled[0];
    for (u = 1; u < ndims; u++) {
        val <<= shared->cache.chunk.scaled_encode_bits[u];
        val ^= scaled[u];
    } 

    
    ret = (unsigned)(val % shared->cache.chunk.nslots);

    FUNC_LEAVE_NOAPI(ret)
} 

herr_t
H5D__chunk_lookup(const H5D_t *dset, const hsize_t *scaled, H5D_chunk_ud_t *udata)
{
    H5D_rdcc_ent_t      *ent   = NULL; 
    H5O_storage_chunk_t *sc    = &(dset->shared->layout.storage.u.chunk);
    unsigned             idx   = 0;     
    bool                 found = false; 
#ifdef H5_HAVE_PARALLEL
    H5P_coll_md_read_flag_t md_reads_file_flag;
    bool                    md_reads_context_flag;
    bool                    restore_md_reads_state = false;
#endif
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->shared->layout.u.chunk.ndims > 0);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);
    assert(scaled);
    assert(udata);

    
    udata->common.layout  = &(dset->shared->layout.u.chunk);
    udata->common.storage = sc;
    udata->common.scaled  = scaled;

    
    udata->chunk_block.offset = HADDR_UNDEF;
    udata->chunk_block.length = 0;
    udata->filter_mask        = 0;
    udata->new_unfilt_chunk   = false;

    
    if (dset->shared->cache.chunk.nslots > 0) {
        
        idx = H5D__chunk_hash_val(dset->shared, scaled);

        
        ent = dset->shared->cache.chunk.slot[idx];
        if (ent) {
            unsigned u; 

            
            found = true;

            
            for (u = 0; u < dset->shared->ndims; u++)
                if (scaled[u] != ent->scaled[u]) {
                    found = false;
                    break;
                } 
        }         
    }             

    
    if (found) {
        udata->idx_hint           = idx;
        udata->chunk_block.offset = ent->chunk_block.offset;
        udata->chunk_block.length = ent->chunk_block.length;
        udata->chunk_idx          = ent->chunk_idx;
    } 
    else {
        
        udata->idx_hint = UINT_MAX;

        
        if (!H5D__chunk_cinfo_cache_found(&dset->shared->cache.chunk.last, udata)) {
            H5D_chk_idx_info_t idx_info; 

            
            idx_info.f      = dset->oloc.file;
            idx_info.pline  = &dset->shared->dcpl_cache.pline;
            idx_info.layout = &dset->shared->layout;

#ifdef H5_HAVE_PARALLEL
            if (H5F_HAS_FEATURE(idx_info.f, H5FD_FEAT_HAS_MPI)) {
                
                if (H5F_get_coll_metadata_reads(idx_info.f)) {
#ifndef NDEBUG
                    bool index_is_open;

                    
                    sc->ops->is_open(&idx_info, &index_is_open);
                    assert(index_is_open);
#endif

                    md_reads_file_flag    = H5P_FORCE_FALSE;
                    md_reads_context_flag = false;
                    H5F_set_coll_metadata_reads(idx_info.f, &md_reads_file_flag, &md_reads_context_flag);
                    restore_md_reads_state = true;
                }
            }
#endif 

            
            if ((sc->ops->get_addr)(&idx_info, udata) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't query chunk address");

                
#ifdef H5_HAVE_PARALLEL
            if (!((H5F_HAS_FEATURE(idx_info.f, H5FD_FEAT_HAS_MPI)) &&
                  (H5F_INTENT(dset->oloc.file) & H5F_ACC_RDWR) && dset->shared->dcpl_cache.pline.nused))
#endif
                H5D__chunk_cinfo_cache_update(&dset->shared->cache.chunk.last, udata);
        } 
    }     

done:
#ifdef H5_HAVE_PARALLEL
    
    if (restore_md_reads_state)
        H5F_set_coll_metadata_reads(dset->oloc.file, &md_reads_file_flag, &md_reads_context_flag);
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_flush_entry(const H5D_t *dset, H5D_rdcc_ent_t *ent, bool reset)
{
    void                *buf                = NULL; 
    bool                 point_of_no_return = false;
    H5O_storage_chunk_t *sc                 = &(dset->shared->layout.storage.u.chunk);
    herr_t               ret_value          = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(dset);
    assert(dset->shared);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);
    assert(ent);
    assert(!ent->locked);

    buf = ent->chunk;
    if (ent->dirty) {
        H5D_chk_idx_info_t idx_info;            
        H5D_chunk_ud_t     udata;               
        bool               must_alloc  = false; 
        bool               need_insert = false; 

        
        udata.common.layout      = &dset->shared->layout.u.chunk;
        udata.common.storage     = sc;
        udata.common.scaled      = ent->scaled;
        udata.chunk_block.offset = ent->chunk_block.offset;
        udata.chunk_block.length = dset->shared->layout.u.chunk.size;
        udata.filter_mask        = 0;
        udata.chunk_idx          = ent->chunk_idx;

        
        if (dset->shared->dcpl_cache.pline.nused && !(ent->edge_chunk_state & H5D_RDCC_DISABLE_FILTERS)) {
            H5Z_EDC_t err_detect; 
            H5Z_cb_t  filter_cb;  
            size_t    alloc;      
            size_t    nbytes;     

            
            H5_CHECKED_ASSIGN(alloc, size_t, udata.chunk_block.length, hsize_t);

            
            if (H5CX_get_err_detect(&err_detect) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get error detection info");
            if (H5CX_get_filter_cb(&filter_cb) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get I/O filter callback function");

            if (!reset) {
                
                if (NULL == (buf = H5MM_malloc(alloc)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for pipeline");
                H5MM_memcpy(buf, ent->chunk, alloc);
            } 
            else {
                
                point_of_no_return = true;
                ent->chunk         = NULL;
            } 
            H5_CHECKED_ASSIGN(nbytes, size_t, udata.chunk_block.length, hsize_t);
            if (H5Z_pipeline(&(dset->shared->dcpl_cache.pline), 0, &(udata.filter_mask), err_detect,
                             filter_cb, &nbytes, &alloc, &buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTFILTER, FAIL, "output pipeline failed");

            H5_CHECKED_ASSIGN(udata.chunk_block.length, hsize_t, nbytes, size_t);

            
            must_alloc = true;
        } 
        else if (!H5_addr_defined(udata.chunk_block.offset)) {
            
            must_alloc = true;

            
            ent->edge_chunk_state &= ~H5D_RDCC_NEWLY_DISABLED_FILTERS;
        } 
        else if (ent->edge_chunk_state & H5D_RDCC_NEWLY_DISABLED_FILTERS) {
            
            must_alloc = true;

            
            ent->edge_chunk_state &= ~H5D_RDCC_NEWLY_DISABLED_FILTERS;
        } 

        assert(!(ent->edge_chunk_state & H5D_RDCC_NEWLY_DISABLED_FILTERS));

        
        if (must_alloc) {
            
            idx_info.f      = dset->oloc.file;
            idx_info.pline  = &dset->shared->dcpl_cache.pline;
            idx_info.layout = &dset->shared->layout;

            
            if (H5D__chunk_file_alloc(&idx_info, &(ent->chunk_block), &udata.chunk_block, &need_insert,
                                      ent->scaled) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL,
                            "unable to insert/resize chunk on chunk level");

            
            ent->chunk_block.offset = udata.chunk_block.offset;
            ent->chunk_block.length = udata.chunk_block.length;
        } 

        
        assert(H5_addr_defined(udata.chunk_block.offset));
        H5_CHECK_OVERFLOW(udata.chunk_block.length, hsize_t, size_t);
        if (H5F_shared_block_write(H5F_SHARED(dset->oloc.file), H5FD_MEM_DRAW, udata.chunk_block.offset,
                                   (size_t)udata.chunk_block.length, buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to write raw data to file");

        
        if (need_insert && sc->ops->insert)
            if ((sc->ops->insert)(&idx_info, &udata, dset) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL, "unable to insert chunk addr into index");

        
        H5D__chunk_cinfo_cache_update(&dset->shared->cache.chunk.last, &udata);

        
        ent->dirty = false;

        
        dset->shared->cache.chunk.stats.nflushes++;
    } 

    
    if (reset) {
        point_of_no_return = false;
        if (buf == ent->chunk)
            buf = NULL;
        if (ent->chunk != NULL)
            ent->chunk = (uint8_t *)H5D__chunk_mem_xfree(ent->chunk,
                                                         ((ent->edge_chunk_state & H5D_RDCC_DISABLE_FILTERS)
                                                              ? NULL
                                                              : &(dset->shared->dcpl_cache.pline)));
    } 

done:
    
    if (buf != ent->chunk)
        H5MM_xfree(buf);

    
    if (ret_value < 0 && point_of_no_return)
        if (ent->chunk)
            ent->chunk = (uint8_t *)H5D__chunk_mem_xfree(ent->chunk,
                                                         ((ent->edge_chunk_state & H5D_RDCC_DISABLE_FILTERS)
                                                              ? NULL
                                                              : &(dset->shared->dcpl_cache.pline)));

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_cache_evict(const H5D_t *dset, H5D_rdcc_ent_t *ent, bool flush)
{
    H5D_rdcc_t *rdcc      = &(dset->shared->cache.chunk);
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(ent);
    assert(!ent->locked);
    assert(ent->idx < rdcc->nslots);

    if (flush) {
        
        if (H5D__chunk_flush_entry(dset, ent, true) < 0)
            HDONE_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "cannot flush indexed storage buffer");
    } 
    else {
        
        if (ent->chunk != NULL)
            ent->chunk = (uint8_t *)H5D__chunk_mem_xfree(ent->chunk,
                                                         ((ent->edge_chunk_state & H5D_RDCC_DISABLE_FILTERS)
                                                              ? NULL
                                                              : &(dset->shared->dcpl_cache.pline)));
    } 

    
    if (ent->prev)
        ent->prev->next = ent->next;
    else
        rdcc->head = ent->next;
    if (ent->next)
        ent->next->prev = ent->prev;
    else
        rdcc->tail = ent->prev;
    ent->prev = ent->next = NULL;

    
    if (ent->tmp_prev) {
        assert(rdcc->tmp_head->tmp_next);
        ent->tmp_prev->tmp_next = ent->tmp_next;
        if (ent->tmp_next) {
            ent->tmp_next->tmp_prev = ent->tmp_prev;
            ent->tmp_next           = NULL;
        } 
        ent->tmp_prev = NULL;
    } 
    else
        
        rdcc->slot[ent->idx] = NULL;

    
    assert(rdcc->slot[ent->idx] != ent);
    ent->idx = UINT_MAX;
    rdcc->nbytes_used -= (size_t)dset->shared->layout.u.chunk.size;
    --rdcc->nused;

    
    ent = H5FL_FREE(H5D_rdcc_ent_t, ent);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_cache_prune(const H5D_t *dset, size_t size)
{
    const H5D_rdcc_t *rdcc  = &(dset->shared->cache.chunk);
    size_t            total = rdcc->nbytes_max;
    const int         nmeth = 2;           
    int               w[1];                
    H5D_rdcc_ent_t   *p[2], *cur;          
    H5D_rdcc_ent_t   *n[2];                
    int               nerrors   = 0;       
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    w[0] = (int)(rdcc->nused * rdcc->w0);
    p[0] = rdcc->head;
    p[1] = NULL;

    while ((p[0] || p[1]) && (rdcc->nbytes_used + size) > total) {
        int i; 

        
        for (i = 0; i < nmeth - 1; i++)
            if (0 == w[i])
                p[i + 1] = rdcc->head;

        
        for (i = 0; i < nmeth; i++)
            n[i] = p[i] ? p[i]->next : NULL;

        
        for (i = 0; i < nmeth && (rdcc->nbytes_used + size) > total; i++) {
            if (0 == i && p[0] && !p[0]->locked &&
                ((0 == p[0]->rd_count && 0 == p[0]->wr_count) ||
                 (0 == p[0]->rd_count && dset->shared->layout.u.chunk.size == p[0]->wr_count) ||
                 (dset->shared->layout.u.chunk.size == p[0]->rd_count && 0 == p[0]->wr_count))) {
                
                cur = p[0];
            }
            else if (1 == i && p[1] && !p[1]->locked) {
                
                cur = p[1];
            }
            else {
                
                cur = NULL;
            }

            if (cur) {
                int j; 

                for (j = 0; j < nmeth; j++) {
                    if (p[j] == cur)
                        p[j] = NULL;
                    if (n[j] == cur)
                        n[j] = cur->next;
                } 
                if (H5D__chunk_cache_evict(dset, cur, true) < 0)
                    nerrors++;
            } 
        }     

        
        for (i = 0; i < nmeth; i++)
            p[i] = n[i];
        for (i = 0; i < nmeth - 1; i++)
            w[i] -= 1;
    } 

    if (nerrors)
        HGOTO_ERROR(H5E_IO, H5E_CANTFLUSH, FAIL, "unable to preempt one or more raw data cache entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5D__chunk_lock(const H5D_io_info_t H5_ATTR_NDEBUG_UNUSED *io_info, const H5D_dset_io_info_t *dset_info,
                H5D_chunk_ud_t *udata, bool relax, bool prev_unfilt_chunk)
{
    const H5D_t *dset;      
    H5O_pline_t *pline;     
    H5O_pline_t *old_pline; 
    const H5O_layout_t *layout;                  
    const H5O_fill_t   *fill;                    
    H5D_fill_buf_info_t fb_info;                 
    bool                fb_info_init = false;    
    H5D_rdcc_t         *rdcc;                    
    H5D_rdcc_ent_t     *ent;                     
    size_t              chunk_size;              
    bool                disable_filters = false; 
    void               *chunk           = NULL;  
    void               *ret_value       = NULL;  

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->store);
    dset = dset_info->dset;
    assert(dset);
    assert(udata);
    assert(!(udata->new_unfilt_chunk && prev_unfilt_chunk));

    
    pline     = &(dset->shared->dcpl_cache.pline);
    old_pline = pline;
    layout    = &(dset->shared->layout);
    fill      = &(dset->shared->dcpl_cache.fill);
    rdcc      = &(dset->shared->cache.chunk);

    assert(!rdcc->tmp_head);

    
    assert(layout->u.chunk.size > 0);
    if (H5_UNLIKELY((hsize_t)((size_t)layout->u.chunk.size) != layout->u.chunk.size))
        HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, NULL, "chunk too big to fit in size_t");
    chunk_size = (size_t)layout->u.chunk.size;

    
    if (UINT_MAX != udata->idx_hint) {
        
        assert(udata->idx_hint < rdcc->nslots);
        assert(rdcc->slot[udata->idx_hint]);

        
        ent = rdcc->slot[udata->idx_hint];

#ifndef NDEBUG
        {
            unsigned u; 

            
            for (u = 0; u < layout->u.chunk.ndims - 1; u++)
                assert(dset_info->store->chunk.scaled[u] == ent->scaled[u]);
        }
#endif 

        
        rdcc->stats.nhits++;

        
        if (pline->nused) {
            
            if (udata->new_unfilt_chunk) {
                
                assert(layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS);
                assert(!(ent->edge_chunk_state & H5D_RDCC_DISABLE_FILTERS));
                assert(old_pline->nused);

                
                pline = NULL;

                
                if (NULL == (chunk = H5D__chunk_mem_alloc(chunk_size, pline)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                                "memory allocation failed for raw data chunk");
                H5MM_memcpy(chunk, ent->chunk, chunk_size);
                ent->chunk = (uint8_t *)H5D__chunk_mem_xfree(ent->chunk, old_pline);
                ent->chunk = (uint8_t *)chunk;
                chunk      = NULL;

                
                ent->edge_chunk_state |= H5D_RDCC_DISABLE_FILTERS;
                ent->edge_chunk_state |= H5D_RDCC_NEWLY_DISABLED_FILTERS;
            } 
            else if (prev_unfilt_chunk) {
                
                assert(layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS);
                assert((ent->edge_chunk_state & H5D_RDCC_DISABLE_FILTERS));
                assert(pline->nused);

                
                old_pline = NULL;

                
                if (NULL == (chunk = H5D__chunk_mem_alloc(chunk_size, pline)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                                "memory allocation failed for raw data chunk");
                H5MM_memcpy(chunk, ent->chunk, chunk_size);

                ent->chunk = (uint8_t *)H5D__chunk_mem_xfree(ent->chunk, old_pline);
                ent->chunk = (uint8_t *)chunk;
                chunk      = NULL;

                
                ent->edge_chunk_state &= ~(H5D_RDCC_DISABLE_FILTERS | H5D_RDCC_NEWLY_DISABLED_FILTERS);
            } 
        }     

        
        if (ent->next) {
            if (ent->next->next)
                ent->next->next->prev = ent;
            else
                rdcc->tail = ent;
            ent->next->prev = ent->prev;
            if (ent->prev)
                ent->prev->next = ent->next;
            else
                rdcc->head = ent->next;
            ent->prev       = ent->next;
            ent->next       = ent->next->next;
            ent->prev->next = ent;
        } 
    }     
    else {
        haddr_t chunk_addr;  
        hsize_t chunk_alloc; 

        
        chunk_addr  = udata->chunk_block.offset;
        chunk_alloc = udata->chunk_block.length;

        
        if (pline->nused) {
            if (udata->new_unfilt_chunk) {
                assert(layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS);

                
                disable_filters = true;
                pline           = NULL;
            } 
            else if (prev_unfilt_chunk) {
                assert(layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS);

                
                old_pline = NULL;
            } 
            else if (layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) {
                
                if (H5D__chunk_is_partial_edge_chunk(dset->shared->ndims, layout->u.chunk.dim,
                                                     dset_info->store->chunk.scaled,
                                                     dset->shared->curr_dims)) {
                    
                    disable_filters = true;
                    old_pline       = NULL;
                    pline           = NULL;
                } 
            }     
        }         

        if (relax) {
            
            rdcc->stats.nhits++;

            if (NULL == (chunk = H5D__chunk_mem_alloc(chunk_size, pline)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for raw data chunk");

            
            memset(chunk, 0, chunk_size);
        } 
        else {
            

            
            if (H5_addr_defined(chunk_addr)) {
                size_t my_chunk_alloc; 
                size_t buf_alloc;      

                
                H5_CHECKED_ASSIGN(my_chunk_alloc, size_t, chunk_alloc, hsize_t);
                H5_CHECKED_ASSIGN(buf_alloc, size_t, chunk_alloc, hsize_t);

                
                if (NULL == (chunk = H5D__chunk_mem_alloc(my_chunk_alloc,
                                                          (udata->new_unfilt_chunk ? old_pline : pline))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                                "memory allocation failed for raw data chunk");
                if (H5F_shared_block_read(H5F_SHARED(dset->oloc.file), H5FD_MEM_DRAW, chunk_addr,
                                          my_chunk_alloc, chunk) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_READERROR, NULL, "unable to read raw data chunk");

                if (old_pline && old_pline->nused) {
                    H5Z_EDC_t err_detect; 
                    H5Z_cb_t  filter_cb;  

                    
                    if (H5CX_get_err_detect(&err_detect) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "can't get error detection info");
                    if (H5CX_get_filter_cb(&filter_cb) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "can't get I/O filter callback function");

                    if (H5Z_pipeline(old_pline, H5Z_FLAG_REVERSE, &(udata->filter_mask), err_detect,
                                     filter_cb, &my_chunk_alloc, &buf_alloc, &chunk) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTFILTER, NULL, "data pipeline read failed");

                    
                    if (udata->new_unfilt_chunk) {
                        void *tmp_chunk = chunk;

                        if (NULL == (chunk = H5D__chunk_mem_alloc(my_chunk_alloc, pline))) {
                            (void)H5D__chunk_mem_xfree(tmp_chunk, old_pline);
                            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                                        "memory allocation failed for raw data chunk");
                        } 
                        H5MM_memcpy(chunk, tmp_chunk, chunk_size);
                        (void)H5D__chunk_mem_xfree(tmp_chunk, old_pline);
                    } 
                }     

                
                rdcc->stats.nmisses++;
            } 
            else {
                H5D_fill_value_t fill_status;

                
                assert(fill->alloc_time != H5D_ALLOC_TIME_EARLY);

                
                if (NULL == (chunk = H5D__chunk_mem_alloc(chunk_size, pline)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                                "memory allocation failed for raw data chunk");

                if (H5P_is_fill_value_defined(fill, &fill_status) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, NULL, "can't tell if fill value defined");

                if (fill->fill_time == H5D_FILL_TIME_ALLOC ||
                    (fill->fill_time == H5D_FILL_TIME_IFSET &&
                     (fill_status == H5D_FILL_VALUE_USER_DEFINED || fill_status == H5D_FILL_VALUE_DEFAULT))) {
                    

                    
                    
                    if (H5D__fill_init(&fb_info, chunk, NULL, NULL, NULL, NULL,
                                       &dset->shared->dcpl_cache.fill, dset->shared->type, (size_t)0,
                                       chunk_size) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "can't initialize fill buffer info");
                    fb_info_init = true;

                    
                    if (fb_info.has_vlen_fill_type)
                        
                        if (H5D__fill_refill_vl(&fb_info, fb_info.elmts_per_buf) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, NULL, "can't refill fill value buffer");
                } 
                else
                    memset(chunk, 0, chunk_size);

                
                rdcc->stats.ninits++;
            } 
        }     

        
        if (rdcc->nslots > 0 && chunk_size <= rdcc->nbytes_max) {
            
            udata->idx_hint = H5D__chunk_hash_val(dset->shared, udata->common.scaled);

            
            ent = rdcc->slot[udata->idx_hint];
            if (!ent || !ent->locked) {
                
                if (ent) {
                    if (H5D__chunk_cache_evict(dset, ent, true) < 0)
                        HGOTO_ERROR(H5E_IO, H5E_CANTINIT, NULL, "unable to preempt chunk from cache");
                } 
                if (H5D__chunk_cache_prune(dset, chunk_size) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_CANTINIT, NULL, "unable to preempt chunk(s) from cache");

                
                if (NULL == (ent = H5FL_CALLOC(H5D_rdcc_ent_t)))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, NULL, "can't allocate raw data chunk entry");

                ent->edge_chunk_state = disable_filters ? H5D_RDCC_DISABLE_FILTERS : 0;
                if (udata->new_unfilt_chunk)
                    ent->edge_chunk_state |= H5D_RDCC_NEWLY_DISABLED_FILTERS;

                
                ent->chunk_block.offset = chunk_addr;
                ent->chunk_block.length = chunk_alloc;
                ent->chunk_idx          = udata->chunk_idx;
                H5MM_memcpy(ent->scaled, udata->common.scaled, sizeof(hsize_t) * layout->u.chunk.ndims);
                H5_CHECKED_ASSIGN(ent->rd_count, hsize_t, chunk_size, size_t);
                H5_CHECKED_ASSIGN(ent->wr_count, hsize_t, chunk_size, size_t);
                ent->chunk = (uint8_t *)chunk;

                
                assert(NULL == rdcc->slot[udata->idx_hint]);
                rdcc->slot[udata->idx_hint] = ent;
                ent->idx                    = udata->idx_hint;
                rdcc->nbytes_used += chunk_size;
                rdcc->nused++;

                
                if (rdcc->tail) {
                    rdcc->tail->next = ent;
                    ent->prev        = rdcc->tail;
                    rdcc->tail       = ent;
                } 
                else
                    rdcc->head = rdcc->tail = ent;
                ent->tmp_next = NULL;
                ent->tmp_prev = NULL;

            } 
            else
                
                ent = NULL;
        }    
        else 
            ent = NULL;
    } 

    
    if (ent) {
        assert(!ent->locked);
        ent->locked = true;
        chunk       = ent->chunk;
    } 
    else
        
        udata->idx_hint = UINT_MAX;

    
    ret_value = chunk;

done:
    
    if (fb_info_init && H5D__fill_term(&fb_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, NULL, "Can't release fill buffer info");

    
    if (!ret_value)
        if (chunk)
            chunk = H5D__chunk_mem_xfree(chunk, pline);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__chunk_unlock(const H5D_io_info_t H5_ATTR_NDEBUG_UNUSED *io_info, const H5D_dset_io_info_t *dset_info,
                  const H5D_chunk_ud_t *udata, bool dirty, void *chunk, hsize_t naccessed)
{
    const H5O_layout_t *layout; 
    const H5D_rdcc_t   *rdcc;
    const H5D_t        *dset;                
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(udata);

    
    layout = &(dset_info->dset->shared->layout);
    rdcc   = &(dset_info->dset->shared->cache.chunk);
    dset   = dset_info->dset;

    if (UINT_MAX == udata->idx_hint) {
        
        bool is_unfiltered_edge_chunk = false; 

        
        if (udata->new_unfilt_chunk) {
            assert(layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS);

            is_unfiltered_edge_chunk = true;
        } 
        else if (layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) {
            
            is_unfiltered_edge_chunk =
                H5D__chunk_is_partial_edge_chunk(dset->shared->ndims, layout->u.chunk.dim,
                                                 dset_info->store->chunk.scaled, dset->shared->curr_dims);
        } 

        if (dirty) {
            H5D_rdcc_ent_t fake_ent; 

            memset(&fake_ent, 0, sizeof(fake_ent));
            fake_ent.dirty = true;
            if (is_unfiltered_edge_chunk)
                fake_ent.edge_chunk_state = H5D_RDCC_DISABLE_FILTERS;
            if (udata->new_unfilt_chunk)
                fake_ent.edge_chunk_state |= H5D_RDCC_NEWLY_DISABLED_FILTERS;
            H5MM_memcpy(fake_ent.scaled, udata->common.scaled, sizeof(hsize_t) * layout->u.chunk.ndims);
            assert(layout->u.chunk.size > 0);
            fake_ent.chunk_idx          = udata->chunk_idx;
            fake_ent.chunk_block.offset = udata->chunk_block.offset;
            fake_ent.chunk_block.length = udata->chunk_block.length;
            fake_ent.chunk              = (uint8_t *)chunk;

            if (H5D__chunk_flush_entry(dset, &fake_ent, true) < 0)
                HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "cannot flush indexed storage buffer");
        } 
        else {
            if (chunk)
                chunk = H5D__chunk_mem_xfree(
                    chunk, (is_unfiltered_edge_chunk ? NULL : &(dset->shared->dcpl_cache.pline)));
        } 
    }     
    else {
        H5D_rdcc_ent_t *ent; 

        
        assert(udata->idx_hint < rdcc->nslots);
        assert(rdcc->slot[udata->idx_hint]);
        assert(rdcc->slot[udata->idx_hint]->chunk == chunk);

        
        ent = rdcc->slot[udata->idx_hint];
        assert(ent->locked);
        if (dirty) {
            ent->dirty = true;
            ent->wr_count -= MIN(ent->wr_count, naccessed);
        } 
        else
            ent->rd_count -= MIN(ent->rd_count, naccessed);
        ent->locked = false;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5D__chunk_allocated_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata)
{
    hsize_t *nbytes = (hsize_t *)_udata;

    FUNC_ENTER_PACKAGE_NOERR

    *(hsize_t *)nbytes += chunk_rec->nbytes;

    FUNC_LEAVE_NOAPI(H5_ITER_CONT)
} 

herr_t
H5D__chunk_allocated(const H5D_t *dset, hsize_t *nbytes)
{
    H5D_chk_idx_info_t   idx_info;                            
    const H5D_rdcc_t    *rdcc = &(dset->shared->cache.chunk); 
    H5D_rdcc_ent_t      *ent;                                 
    hsize_t              chunk_bytes = 0;                     
    H5O_storage_chunk_t *sc          = &(dset->shared->layout.storage.u.chunk);
    herr_t               ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(dset);
    assert(dset->shared);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

    
    for (ent = rdcc->head; ent; ent = ent->next)
        
        if (H5D__chunk_flush_entry(dset, ent, false) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "cannot flush indexed storage buffer");

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if ((sc->ops->iterate)(&idx_info, H5D__chunk_allocated_cb, &chunk_bytes) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                    "unable to retrieve allocated chunk information from index");

    
    *nbytes = chunk_bytes;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_allocate(const H5D_t *dset, bool full_overwrite, const hsize_t old_dim[])
{
    H5D_chk_idx_info_t     idx_info;                                       
    const H5D_chunk_ops_t *ops = dset->shared->layout.storage.u.chunk.ops; 
    hsize_t min_unalloc[H5O_LAYOUT_NDIMS]; 
    hsize_t max_unalloc[H5O_LAYOUT_NDIMS]; 
    hsize_t           scaled[H5O_LAYOUT_NDIMS]; 
    size_t            orig_chunk_size = 0;      
    hsize_t           chunk_size;               
    unsigned          filter_mask = 0;          
    H5O_layout_t     *layout      = &(dset->shared->layout);           
    H5O_pline_t      *pline       = &(dset->shared->dcpl_cache.pline); 
    H5O_pline_t       def_pline   = H5O_CRT_PIPELINE_DEF;              
    const H5O_fill_t *fill        = &(dset->shared->dcpl_cache.fill);  
    H5D_fill_value_t  fill_status;                                     
    bool              should_fill     = false; 
    void             *unfilt_fill_buf = NULL;  
    void            **fill_buf        = NULL;  
#ifdef H5_HAVE_PARALLEL
    bool blocks_written = false; 
    bool using_mpi =
        false; 
    H5D_chunk_coll_fill_info_t chunk_fill_info; 
#endif                                          
    bool                carry; 
    unsigned            space_ndims;                     
    const hsize_t      *space_dim;                       
    const hsize_t      *chunk_dim = layout->u.chunk.dim; 
    unsigned            op_dim;                          
    H5D_fill_buf_info_t fb_info;                         
    bool                fb_info_init = false; 
    bool has_unfilt_edge_chunks = false; 
    bool unfilt_edge_chunk_dim[H5O_LAYOUT_NDIMS]; 
    hsize_t edge_chunk_scaled[H5O_LAYOUT_NDIMS];  
    unsigned             nunfilt_edge_chunk_dims = 0; 
    H5O_storage_chunk_t *sc                      = &(layout->storage.u.chunk); 
    herr_t               ret_value               = SUCCEED;                    

    FUNC_ENTER_PACKAGE

    
    assert(dset && H5D_CHUNKED == layout->type);
    assert(layout->u.chunk.ndims > 0 && layout->u.chunk.ndims <= H5O_LAYOUT_NDIMS);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

    
    space_dim   = dset->shared->curr_dims;
    space_ndims = dset->shared->ndims;

    
    scaled[space_ndims] = (hsize_t)0;

    
    for (op_dim = 0; op_dim < (unsigned)space_ndims; op_dim++)
        if (space_dim[op_dim] == 0) {
            
            H5D__chunk_cinfo_cache_reset(&dset->shared->cache.chunk.last);
            HGOTO_DONE(SUCCEED);
        } 

#ifdef H5_HAVE_PARALLEL
    
    if (H5F_HAS_FEATURE(dset->oloc.file, H5FD_FEAT_HAS_MPI)) {
        
        using_mpi = true;

        
        chunk_fill_info.num_chunks = 0;
        chunk_fill_info.chunk_info = NULL;
    }  
#endif 

    
    for (op_dim = 0; op_dim < space_ndims; op_dim++) {
        
        if (chunk_dim[op_dim] == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ", op_dim);
        min_unalloc[op_dim] = (old_dim[op_dim] + chunk_dim[op_dim] - 1) / chunk_dim[op_dim];
        max_unalloc[op_dim] = (space_dim[op_dim] - 1) / chunk_dim[op_dim];

        
        if ((layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) && pline->nused > 0 &&
            space_dim[op_dim] % chunk_dim[op_dim] != 0) {
            has_unfilt_edge_chunks        = true;
            unfilt_edge_chunk_dim[op_dim] = true;
            edge_chunk_scaled[op_dim]     = max_unalloc[op_dim];
        } 
        else
            unfilt_edge_chunk_dim[op_dim] = false;
    } 

    
    if (H5P_is_fill_value_defined(fill, &fill_status) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't tell if fill value defined");

    
    if ((!full_overwrite &&
         (fill->fill_time == H5D_FILL_TIME_ALLOC ||
          (fill->fill_time == H5D_FILL_TIME_IFSET &&
           (fill_status == H5D_FILL_VALUE_USER_DEFINED || fill_status == H5D_FILL_VALUE_DEFAULT)))) ||
        pline->nused > 0)
        should_fill = true;

    
    if (should_fill) {
        
        orig_chunk_size = (size_t)layout->u.chunk.size;

        
        if (H5_UNLIKELY((hsize_t)orig_chunk_size != layout->u.chunk.size))
            HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "chunk size too big to fit in size_t");

        
        
        if (H5D__fill_init(&fb_info, NULL, H5D__chunk_mem_alloc, pline, H5D__chunk_mem_free, pline,
                           &dset->shared->dcpl_cache.fill, dset->shared->type, (size_t)0,
                           orig_chunk_size) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize fill buffer info");
        fb_info_init = true;

        
        fill_buf = &fb_info.fill_buf;

        
        
        if (!fb_info.has_vlen_fill_type && pline->nused > 0) {
            H5Z_EDC_t err_detect; 
            H5Z_cb_t  filter_cb;  
            size_t    buf_size = orig_chunk_size;

            
            if (has_unfilt_edge_chunks) {
                if (NULL == (unfilt_fill_buf = H5D__chunk_mem_alloc(orig_chunk_size, &def_pline)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                "memory allocation failed for raw data chunk");
                H5MM_memcpy(unfilt_fill_buf, fb_info.fill_buf, orig_chunk_size);
            } 

            
            if (H5CX_get_err_detect(&err_detect) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get error detection info");
            if (H5CX_get_filter_cb(&filter_cb) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get I/O filter callback function");

            
            if (H5Z_pipeline(pline, 0, &filter_mask, err_detect, filter_cb, &orig_chunk_size, &buf_size,
                             &fb_info.fill_buf) < 0)
                HGOTO_ERROR(H5E_PLINE, H5E_WRITEERROR, FAIL, "output pipeline failed");
        } 

        
        H5_CHECKED_ASSIGN(chunk_size, hsize_t, orig_chunk_size, size_t);
    } 
    else
        
        chunk_size = layout->u.chunk.size;

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    
    for (op_dim = 0; op_dim < space_ndims; op_dim++) {
        H5D_chunk_ud_t udata; 
        unsigned       u;     
        int            i;     

        
        if (min_unalloc[op_dim] > max_unalloc[op_dim])
            continue;
        else {
            
            memset(scaled, 0, (space_ndims * sizeof(scaled[0])));
            scaled[op_dim] = min_unalloc[op_dim];

            if (has_unfilt_edge_chunks) {
                
                nunfilt_edge_chunk_dims = 0;
                for (u = 0; u < space_ndims; u++)
                    if (unfilt_edge_chunk_dim[u] && scaled[u] == edge_chunk_scaled[u])
                        nunfilt_edge_chunk_dims++;

                
                if (should_fill && !fb_info.has_vlen_fill_type) {
                    assert(fb_info_init);
                    assert(unfilt_fill_buf);
                    assert(orig_chunk_size);
                    if (nunfilt_edge_chunk_dims) {
                        fill_buf   = &unfilt_fill_buf;
                        chunk_size = layout->u.chunk.size;
                    } 
                    else {
                        fill_buf   = &fb_info.fill_buf;
                        chunk_size = orig_chunk_size;
                        assert(chunk_size == (size_t)orig_chunk_size);
                    } 
                }     
            }         

            carry = false;
        } 

        while (!carry) {
            bool need_insert = false; 

            
            if (H5D__chunk_lookup(dset, scaled, &udata) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");
#ifndef NDEBUG
            
            if (H5D_CHUNK_IDX_NONE != sc->idx_type)
                assert(!H5_addr_defined(udata.chunk_block.offset));

            
            {
                unsigned v; 
                bool     outside_orig = false;

                for (v = 0; v < space_ndims; v++) {
                    assert((scaled[v] * chunk_dim[v]) < space_dim[v]);
                    if ((scaled[v] * chunk_dim[v]) >= old_dim[v])
                        outside_orig = true;
                } 
                assert(outside_orig);
            } 
#endif        

            
            if (fb_info_init && fb_info.has_vlen_fill_type) {
                
                assert(should_fill);
                assert(!unfilt_fill_buf);
                assert(orig_chunk_size);
#ifdef H5_HAVE_PARALLEL
                assert(!using_mpi); 
#endif

                
                if (fb_info.fill_buf_size < orig_chunk_size) {
                    if (NULL ==
                        (fb_info.fill_buf = H5D__chunk_mem_realloc(fb_info.fill_buf, orig_chunk_size, pline)))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                    "memory reallocation failed for raw data chunk");
                    fb_info.fill_buf_size = orig_chunk_size;
                } 

                
                if (H5D__fill_refill_vl(&fb_info, fb_info.elmts_per_buf) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "can't refill fill value buffer");

                
                if ((pline->nused > 0) && !nunfilt_edge_chunk_dims) {
                    H5Z_EDC_t err_detect; 
                    H5Z_cb_t  filter_cb;  
                    size_t    nbytes = orig_chunk_size;

                    
                    if (H5CX_get_err_detect(&err_detect) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get error detection info");
                    if (H5CX_get_filter_cb(&filter_cb) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get I/O filter callback function");

                    
                    if (H5Z_pipeline(pline, 0, &filter_mask, err_detect, filter_cb, &nbytes,
                                     &fb_info.fill_buf_size, &fb_info.fill_buf) < 0)
                        HGOTO_ERROR(H5E_PLINE, H5E_WRITEERROR, FAIL, "output pipeline failed");

                    
                    H5_CHECKED_ASSIGN(chunk_size, hsize_t, nbytes, size_t);
                } 
                else
                    chunk_size = layout->u.chunk.size;

                assert(*fill_buf == fb_info.fill_buf);
            } 

            
            udata.common.layout      = &layout->u.chunk;
            udata.common.storage     = sc;
            udata.common.scaled      = scaled;
            udata.chunk_block.offset = HADDR_UNDEF;
            udata.chunk_block.length = chunk_size;
            udata.filter_mask        = filter_mask;

            
            if (H5D__chunk_file_alloc(&idx_info, NULL, &udata.chunk_block, &need_insert, scaled) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL,
                            "unable to insert/resize chunk on chunk level");
            assert(H5_addr_defined(udata.chunk_block.offset));

            
            if (should_fill) {
                
                assert(fb_info_init);
                assert(udata.chunk_block.length == chunk_size);

#ifdef H5_HAVE_PARALLEL
                
                if (using_mpi) {
                    

                    
                    if (0 == chunk_fill_info.num_chunks % 1024) {
                        void *tmp_realloc;

                        if (NULL == (tmp_realloc = H5MM_realloc(chunk_fill_info.chunk_info,
                                                                (chunk_fill_info.num_chunks + 1024) *
                                                                    sizeof(struct chunk_coll_fill_info))))
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for chunk fill info");

                        chunk_fill_info.chunk_info = tmp_realloc;
                    }

                    
                    chunk_fill_info.chunk_info[chunk_fill_info.num_chunks].addr = udata.chunk_block.offset;
                    H5_CHECKED_ASSIGN(chunk_fill_info.chunk_info[chunk_fill_info.num_chunks].chunk_size,
                                      size_t, chunk_size, hsize_t);
                    chunk_fill_info.chunk_info[chunk_fill_info.num_chunks].unfiltered_partial_chunk =
                        (*fill_buf == unfilt_fill_buf);
                    chunk_fill_info.num_chunks++;

                    
                    blocks_written = true;
                } 
                else
#endif 
                {
                    H5_CHECK_OVERFLOW(chunk_size, hsize_t, size_t);
                    if (H5F_shared_block_write(H5F_SHARED(dset->oloc.file), H5FD_MEM_DRAW,
                                               udata.chunk_block.offset, (size_t)chunk_size, *fill_buf) < 0)
                        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "unable to write raw data to file");
                }
            } 

            
            if (need_insert && ops->insert)
                if ((ops->insert)(&idx_info, &udata, dset) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL, "unable to insert chunk addr into index");

            
            carry = true;
            for (i = ((int)space_ndims - 1); i >= 0; --i) {
                scaled[i]++;
                if (scaled[i] > max_unalloc[i]) {
                    if ((unsigned)i == op_dim)
                        scaled[i] = min_unalloc[i];
                    else
                        scaled[i] = 0;

                    
                    if (unfilt_edge_chunk_dim[i] && edge_chunk_scaled[i] == max_unalloc[i] &&
                        scaled[i] < edge_chunk_scaled[i]) {
                        nunfilt_edge_chunk_dims--;
                        if (should_fill && nunfilt_edge_chunk_dims == 0 && !fb_info.has_vlen_fill_type) {
                            assert(
                                !H5D__chunk_is_partial_edge_chunk(space_ndims, chunk_dim, scaled, space_dim));
                            assert(orig_chunk_size);
                            fill_buf = &fb_info.fill_buf;
                            H5_CHECKED_ASSIGN(chunk_size, hsize_t, orig_chunk_size, size_t);
                        } 
                    }     
                }         
                else {
                    
                    if (unfilt_edge_chunk_dim[i] && scaled[i] == edge_chunk_scaled[i]) {
                        assert(edge_chunk_scaled[i] == max_unalloc[i]);
                        nunfilt_edge_chunk_dims++;
                        if (should_fill && nunfilt_edge_chunk_dims == 1 && !fb_info.has_vlen_fill_type) {
                            assert(
                                H5D__chunk_is_partial_edge_chunk(space_ndims, chunk_dim, scaled, space_dim));
                            fill_buf   = &unfilt_fill_buf;
                            chunk_size = layout->u.chunk.size;
                        } 
                    }     

                    carry = false;
                    break;
                } 
            }     
        }         

        
        if (min_unalloc[op_dim] == 0)
            break;
        else
            max_unalloc[op_dim] = min_unalloc[op_dim] - 1;
    } 

#ifdef H5_HAVE_PARALLEL
    
    if (using_mpi && blocks_written)
        if (H5D__chunk_collective_fill(dset, &chunk_fill_info, fb_info.fill_buf, unfilt_fill_buf) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "unable to write raw data to file");
#endif 

    
    H5D__chunk_cinfo_cache_reset(&dset->shared->cache.chunk.last);

done:
    
    if (fb_info_init && H5D__fill_term(&fb_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release fill buffer info");

    
    unfilt_fill_buf = H5D__chunk_mem_xfree(unfilt_fill_buf, &def_pline);

#ifdef H5_HAVE_PARALLEL
    if (using_mpi && chunk_fill_info.chunk_info)
        H5MM_free(chunk_fill_info.chunk_info);
#endif

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_update_old_edge_chunks(H5D_t *dset, hsize_t old_dim[])
{
    hsize_t old_edge_chunk_sc[H5O_LAYOUT_NDIMS]; 
    hsize_t max_edge_chunk_sc[H5O_LAYOUT_NDIMS]; 
    bool new_full_dim[H5O_LAYOUT_NDIMS];         
    const H5O_layout_t *layout = &(dset->shared->layout); 
    hsize_t             chunk_sc[H5O_LAYOUT_NDIMS];       
    const hsize_t      *chunk_dim = layout->u.chunk.dim;  
    unsigned            space_ndims;                      
    const hsize_t      *space_dim;                        
    unsigned            op_dim;                           
    H5D_io_info_t       chk_io_info;                      
    H5D_chunk_ud_t      chk_udata;                        
    H5D_storage_t       chk_store;                        
    H5D_dset_io_info_t  chk_dset_info;                    
    void               *chunk;                            
    bool                carry; 
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset && H5D_CHUNKED == layout->type);
    assert(layout->u.chunk.ndims > 0 && layout->u.chunk.ndims <= H5O_LAYOUT_NDIMS);
    H5D_CHUNK_STORAGE_INDEX_CHK(&layout->storage.u.chunk);
    assert(dset->shared->dcpl_cache.pline.nused > 0);
    assert(layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS);

    
    space_dim   = dset->shared->curr_dims;
    space_ndims = dset->shared->ndims;

    
    chunk_sc[space_ndims] = (hsize_t)0;

    
    for (op_dim = 0; op_dim < space_ndims; op_dim++)
        if ((space_dim[op_dim] < chunk_dim[op_dim]) || old_dim[op_dim] == 0) {
            
            H5D__chunk_cinfo_cache_reset(&dset->shared->cache.chunk.last);
            HGOTO_DONE(SUCCEED);
        } 

    
    chk_store.chunk.scaled = chunk_sc;

    chk_io_info.op_type = H5D_IO_OP_READ;

    chk_dset_info.dset     = dset;
    chk_dset_info.store    = &chk_store;
    chk_dset_info.buf.vp   = NULL;
    chk_io_info.dsets_info = &chk_dset_info;

    
    for (op_dim = 0; op_dim < space_ndims; op_dim++) {
        
        new_full_dim[op_dim] = false;

        
        if (chunk_dim[op_dim] == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ", op_dim);

        
        old_edge_chunk_sc[op_dim] = (old_dim[op_dim] / chunk_dim[op_dim]);

        
        max_edge_chunk_sc[op_dim] = MIN((old_dim[op_dim] - 1) / chunk_dim[op_dim],
                                        MAX((space_dim[op_dim] / chunk_dim[op_dim]), 1) - 1);

        
        if (old_dim[op_dim] % chunk_dim[op_dim] == 0)
            continue;

        
        if ((space_dim[op_dim] / chunk_dim[op_dim]) >= (old_edge_chunk_sc[op_dim] + 1))
            new_full_dim[op_dim] = true;
    } 

    
    for (op_dim = 0; op_dim < space_ndims; op_dim++) {
        
        if (!new_full_dim[op_dim])
            continue;
        else {
            assert(max_edge_chunk_sc[op_dim] == old_edge_chunk_sc[op_dim]);

            
            memset(chunk_sc, 0, (space_ndims * sizeof(chunk_sc[0])));
            chunk_sc[op_dim] = old_edge_chunk_sc[op_dim];

            carry = false;
        } 

        while (!carry) {
            int i; 

            
            assert(H5D__chunk_is_partial_edge_chunk(space_ndims, chunk_dim, chunk_sc, old_dim) &&
                   !H5D__chunk_is_partial_edge_chunk(space_ndims, chunk_dim, chunk_sc, space_dim));

            
            if (H5D__chunk_lookup(dset, chunk_sc, &chk_udata) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");

            
            if (H5_addr_defined(chk_udata.chunk_block.offset) || (UINT_MAX != chk_udata.idx_hint)) {
                
                if (NULL ==
                    (chunk = (void *)H5D__chunk_lock(&chk_io_info, &chk_dset_info, &chk_udata, false, true)))
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to lock raw data chunk");

                
                if (H5D__chunk_unlock(&chk_io_info, &chk_dset_info, &chk_udata, true, chunk, (hsize_t)0) < 0)
                    HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "unable to unlock raw data chunk");
            } 

            
            carry = true;
            for (i = ((int)space_ndims - 1); i >= 0; --i) {
                if ((unsigned)i != op_dim) {
                    ++chunk_sc[i];
                    if (chunk_sc[i] > (hsize_t)max_edge_chunk_sc[i])
                        chunk_sc[i] = 0;
                    else {
                        carry = false;
                        break;
                    } 
                }     
            }         
        }             

        
        if (old_edge_chunk_sc[op_dim] == 0)
            break;
        else
            --max_edge_chunk_sc[op_dim];
    } 

    
    H5D__chunk_cinfo_cache_reset(&dset->shared->cache.chunk.last);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_PARALLEL

static herr_t
H5D__chunk_collective_fill(const H5D_t *dset, H5D_chunk_coll_fill_info_t *chunk_fill_info,
                           const void *fill_buf, const void *partial_chunk_fill_buf)
{
    MPI_Comm         mpi_comm = MPI_COMM_NULL; 
    int              mpi_rank = (-1);          
    int              mpi_size = (-1);          
    int              mpi_code;                 
    size_t           num_blocks;               
    size_t           leftover_blocks;          
    int              blocks;                   
    int              leftover;                 
    H5FD_mpio_xfer_t prev_xfer_mode;           
    bool             have_xfer_mode = false;   
    size_t           i;                        
    haddr_t         *io_addrs = NULL;
    size_t          *io_sizes = NULL;
    const void     **io_wbufs = NULL;
    H5FD_mem_t       io_types[2];
    bool             all_same_block_len = true;
    bool             need_sort          = false;
    size_t           io_2sizes[2];
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(chunk_fill_info->num_chunks != 0);

    
    if (partial_chunk_fill_buf)
        assert(dset->shared->layout.u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS);

    
    if (MPI_COMM_NULL == (mpi_comm = H5F_mpi_get_comm(dset->oloc.file)))
        HGOTO_ERROR(H5E_INTERNAL, H5E_MPI, FAIL, "Can't retrieve MPI communicator");

    
    if ((mpi_rank = H5F_mpi_get_rank(dset->oloc.file)) < 0)
        HGOTO_ERROR(H5E_INTERNAL, H5E_MPI, FAIL, "Can't retrieve MPI rank");

    
    if ((mpi_size = H5F_mpi_get_size(dset->oloc.file)) < 0)
        HGOTO_ERROR(H5E_INTERNAL, H5E_MPI, FAIL, "Can't retrieve MPI size");

    
    if (mpi_size == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "Resulted in division by zero");

    num_blocks =
        (size_t)(chunk_fill_info->num_chunks / (size_t)mpi_size); 

    
    leftover_blocks = (size_t)(chunk_fill_info->num_chunks % (size_t)mpi_size);

    
    H5_CHECKED_ASSIGN(blocks, int, num_blocks, size_t);
    H5_CHECKED_ASSIGN(leftover, int, leftover_blocks, size_t);

    
    if (num_blocks > 0 || leftover > mpi_rank) {

        if (NULL == (io_addrs = H5MM_malloc((size_t)(blocks + 1) * sizeof(*io_addrs))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                        "couldn't allocate space for I/O addresses vector");

        if (NULL == (io_wbufs = H5MM_malloc((size_t)(blocks + 1) * sizeof(*io_wbufs))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "couldn't allocate space for I/O buffers vector");
    }

    
    for (i = 1; i < chunk_fill_info->num_chunks; i++) {
        if (chunk_fill_info->chunk_info[i].addr < chunk_fill_info->chunk_info[i - 1].addr)
            need_sort = true;

        if (chunk_fill_info->chunk_info[i].chunk_size != chunk_fill_info->chunk_info[i - 1].chunk_size)
            all_same_block_len = false;
    }

    
    if (need_sort)
        qsort(chunk_fill_info->chunk_info, chunk_fill_info->num_chunks, sizeof(struct chunk_coll_fill_info),
              H5D__chunk_cmp_coll_fill_info);

    
    if (all_same_block_len) {
        io_2sizes[0] = chunk_fill_info->chunk_info[0].chunk_size;
        io_2sizes[1] = 0;
    }
    else {
        if (NULL == (io_sizes = H5MM_malloc((size_t)(blocks + 1) * sizeof(*io_sizes))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "couldn't allocate space for I/O sizes vector");
    }

    
    io_types[0] = H5FD_MEM_DRAW;
    io_types[1] = H5FD_MEM_NOLIST;

    
    for (i = 0; i < (size_t)blocks; i++) {
        size_t idx = i + (size_t)(mpi_rank * blocks);

        io_addrs[i] = chunk_fill_info->chunk_info[idx].addr;

        if (!all_same_block_len)
            io_sizes[i] = chunk_fill_info->chunk_info[idx].chunk_size;

        if (chunk_fill_info->chunk_info[idx].unfiltered_partial_chunk)
            io_wbufs[i] = partial_chunk_fill_buf;
        else
            io_wbufs[i] = fill_buf;
    }

    
    if (leftover > mpi_rank) {
        io_addrs[blocks] = chunk_fill_info->chunk_info[(blocks * mpi_size) + mpi_rank].addr;

        if (!all_same_block_len)
            io_sizes[blocks] = chunk_fill_info->chunk_info[(blocks * mpi_size) + mpi_rank].chunk_size;

        if (chunk_fill_info->chunk_info[(blocks * mpi_size) + mpi_rank].unfiltered_partial_chunk) {
            assert(partial_chunk_fill_buf);
            io_wbufs[blocks] = partial_chunk_fill_buf;
        }
        else
            io_wbufs[blocks] = fill_buf;

        blocks++;
    }

    
    if (H5CX_get_io_xfer_mode(&prev_xfer_mode) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set transfer mode");
    have_xfer_mode = true;

    
    if (H5CX_set_io_xfer_mode(H5FD_MPIO_COLLECTIVE) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set transfer mode");

    
    if (MPI_SUCCESS != (mpi_code = MPI_Barrier(mpi_comm)))
        HMPI_GOTO_ERROR(FAIL, "MPI_Barrier failed", mpi_code)

    
    if (H5F_shared_vector_write(H5F_SHARED(dset->oloc.file), (uint32_t)blocks, io_types, io_addrs,
                                all_same_block_len ? io_2sizes : io_sizes, io_wbufs) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "vector write call failed");

done:
    if (have_xfer_mode)
        
        if (H5CX_set_io_xfer_mode(prev_xfer_mode) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set transfer mode");

    H5MM_xfree(io_addrs);
    H5MM_xfree(io_wbufs);
    H5MM_xfree(io_sizes);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5D__chunk_cmp_coll_fill_info(const void *_entry1, const void *_entry2)
{
    const struct chunk_coll_fill_info *entry1;
    const struct chunk_coll_fill_info *entry2;
    int                                ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    entry1 = (const struct chunk_coll_fill_info *)_entry1;
    entry2 = (const struct chunk_coll_fill_info *)_entry2;

    ret_value = H5_addr_cmp(entry1->addr, entry2->addr);

    FUNC_LEAVE_NOAPI(ret_value)
} 

#endif 

static herr_t
H5D__chunk_prune_fill(H5D_chunk_it_ud1_t *udata, bool new_unfilt_chunk)
{
    const H5D_io_info_t *io_info         = udata->io_info;          
    const H5D_t         *dset            = udata->dset_info->dset;  
    const H5O_layout_t  *layout          = &(dset->shared->layout); 
    unsigned             rank            = udata->common.layout->ndims - 1; 
    const hsize_t       *scaled          = udata->common.scaled;            
    H5S_sel_iter_t      *chunk_iter      = NULL;  
    bool                 chunk_iter_init = false; 
    hsize_t              sel_nelmts;              
    hsize_t              count[H5O_LAYOUT_NDIMS]; 
    size_t               chunk_size;              
    void                *chunk;                   
    H5D_chunk_ud_t       chk_udata;               
    hsize_t              bytes_accessed;          
    unsigned             u;                       
    herr_t               ret_value = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    assert(layout->u.chunk.size > 0);
    H5_CHECKED_ASSIGN(chunk_size, size_t, layout->u.chunk.size, hsize_t);

    
    if (H5D__chunk_lookup(dset, scaled, &chk_udata) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk address");
    chk_udata.new_unfilt_chunk = new_unfilt_chunk;

    
    if (!H5_addr_defined(chk_udata.chunk_block.offset) && UINT_MAX == chk_udata.idx_hint)
        HGOTO_DONE(SUCCEED);

    
    if (!udata->fb_info_init) {
        H5_CHECK_OVERFLOW(udata->elmts_per_chunk, hsize_t, size_t);
        if (H5D__fill_init(&udata->fb_info, NULL, NULL, NULL, NULL, NULL, &dset->shared->dcpl_cache.fill,
                           dset->shared->type, (size_t)udata->elmts_per_chunk, chunk_size) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize fill buffer info");
        udata->fb_info_init = true;
    } 

    
    for (u = 0; u < rank; u++) {
        count[u] = MIN(layout->u.chunk.dim[u], (udata->space_dim[u] - (scaled[u] * layout->u.chunk.dim[u])));
        assert(count[u] > 0);
    } 

    
    if (H5S_select_all(udata->chunk_space, true) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "unable to select space");

    
    if (H5S_select_hyperslab(udata->chunk_space, H5S_SELECT_NOTB, udata->hyper_start, NULL, count, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSELECT, FAIL, "unable to select hyperslab");

    
    if (NULL == (chunk = (void *)H5D__chunk_lock(io_info, udata->dset_info, &chk_udata, false, false)))
        HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to lock raw data chunk");

    
    
    
    

    
    sel_nelmts = H5S_GET_SELECT_NPOINTS(udata->chunk_space);
    H5_CHECK_OVERFLOW(sel_nelmts, hsize_t, size_t);

    
    if (udata->fb_info.has_vlen_fill_type)
        
        if (H5D__fill_refill_vl(&udata->fb_info, (size_t)sel_nelmts) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "can't refill fill value buffer");

    
    if (NULL == (chunk_iter = H5FL_MALLOC(H5S_sel_iter_t)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate chunk selection iterator");

    
    H5_CHECK_OVERFLOW(layout->u.chunk.dim[rank], hsize_t, size_t);
    if (H5S_select_iter_init(chunk_iter, udata->chunk_space, (size_t)layout->u.chunk.dim[rank], 0) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize chunk selection information");
    chunk_iter_init = true;

    
    if (H5D__scatter_mem(udata->fb_info.fill_buf, chunk_iter, (size_t)sel_nelmts, chunk ) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "scatter failed");

    
    
    bytes_accessed = sel_nelmts * layout->u.chunk.dim[rank];

    
    if (H5D__chunk_unlock(io_info, udata->dset_info, &chk_udata, true, chunk, bytes_accessed) < 0)
        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "unable to unlock raw data chunk");

done:
    
    if (chunk_iter_init && H5S_SELECT_ITER_RELEASE(chunk_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (chunk_iter)
        chunk_iter = H5FL_FREE(H5S_sel_iter_t, chunk_iter);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_prune_by_extent(H5D_t *dset, const hsize_t *old_dim)
{
    hsize_t min_mod_chunk_sc[H5O_LAYOUT_NDIMS]; 
    hsize_t max_mod_chunk_sc[H5O_LAYOUT_NDIMS]; 
    hssize_t max_fill_chunk_sc[H5O_LAYOUT_NDIMS]; 
    bool fill_dim[H5O_LAYOUT_NDIMS]; 
    hsize_t min_partial_chunk_sc[H5O_LAYOUT_NDIMS]; 
    bool new_unfilt_dim[H5O_LAYOUT_NDIMS]; 
    H5D_chk_idx_info_t  idx_info;          
    H5D_io_info_t       chk_io_info;       
    H5D_dset_io_info_t  chk_dset_info;     
    H5D_storage_t       chk_store;         
    const H5O_layout_t *layout = &(dset->shared->layout);      
    const H5D_rdcc_t   *rdcc   = &(dset->shared->cache.chunk); 
    unsigned            space_ndims;                           
    const hsize_t      *space_dim;                             
    unsigned            op_dim;                                
    bool                shrunk_dim[H5O_LAYOUT_NDIMS];          
    H5D_chunk_it_ud1_t  udata;                                 
    bool udata_init = false;         
    H5D_chunk_common_ud_t idx_udata; 
    H5S_t                *chunk_space = NULL;            
    hsize_t               chunk_dim[H5O_LAYOUT_NDIMS];   
    hsize_t               scaled[H5O_LAYOUT_NDIMS];      
    hsize_t               hyper_start[H5O_LAYOUT_NDIMS]; 
    hsize_t               elmts_per_chunk;               
    bool     disable_edge_filters = false; 
    bool     new_unfilt_chunk     = false; 
    unsigned u;                            
    const H5O_storage_chunk_t *sc        = &(layout->storage.u.chunk);
    herr_t                     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset && H5D_CHUNKED == layout->type);
    assert(layout->u.chunk.ndims > 0 && layout->u.chunk.ndims <= H5O_LAYOUT_NDIMS);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

    
    space_dim   = dset->shared->curr_dims;
    space_ndims = dset->shared->ndims;

    
    scaled[space_ndims] = (hsize_t)0;

    
    for (op_dim = 0; op_dim < (unsigned)space_ndims; op_dim++)
        if (old_dim[op_dim] == 0) {
            
            H5D__chunk_cinfo_cache_reset(&dset->shared->cache.chunk.last);
            HGOTO_DONE(SUCCEED);
        } 

    
    
    
    
    
    elmts_per_chunk = 1;
    for (u = 0; u < space_ndims; u++) {
        elmts_per_chunk *= layout->u.chunk.dim[u];
        chunk_dim[u]  = layout->u.chunk.dim[u];
        shrunk_dim[u] = (space_dim[u] < old_dim[u]);
    } 

    
    if (NULL == (chunk_space = H5S_create_simple(space_ndims, chunk_dim, NULL)))
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create simple dataspace");

    
    
    memset(hyper_start, 0, sizeof(hyper_start));

    
    chk_store.chunk.scaled = scaled;

    chk_io_info.op_type = H5D_IO_OP_READ;

    chk_dset_info.dset     = dset;
    chk_dset_info.store    = &chk_store;
    chk_dset_info.buf.vp   = NULL;
    chk_io_info.dsets_info = &chk_dset_info;
    chk_io_info.count      = 1;

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    memset(&udata, 0, sizeof udata);
    udata.common.layout   = &layout->u.chunk;
    udata.common.storage  = sc;
    udata.common.scaled   = scaled;
    udata.io_info         = &chk_io_info;
    udata.dset_info       = &chk_dset_info;
    udata.idx_info        = &idx_info;
    udata.space_dim       = space_dim;
    udata.shrunk_dim      = shrunk_dim;
    udata.elmts_per_chunk = elmts_per_chunk;
    udata.chunk_space     = chunk_space;
    udata.hyper_start     = hyper_start;
    udata_init            = true;

    
    idx_udata.layout  = &layout->u.chunk;
    idx_udata.storage = sc;

    
    disable_edge_filters = (layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) &&
                           (idx_info.pline->nused > 0);

    
    memset(min_mod_chunk_sc, 0, sizeof(min_mod_chunk_sc));
    memset(max_mod_chunk_sc, 0, sizeof(max_mod_chunk_sc));
    for (op_dim = 0; op_dim < (unsigned)space_ndims; op_dim++) {
        
        if (chunk_dim[op_dim] == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ", op_dim);

        
        max_mod_chunk_sc[op_dim] = (old_dim[op_dim] - 1) / chunk_dim[op_dim];

        
        if (0 == space_dim[op_dim])
            max_fill_chunk_sc[op_dim] = -1;
        else
            max_fill_chunk_sc[op_dim] =
                (hssize_t)(((MIN(space_dim[op_dim], old_dim[op_dim]) - 1) / chunk_dim[op_dim]));

        if (shrunk_dim[op_dim]) {
            
            min_mod_chunk_sc[op_dim] = space_dim[op_dim] / chunk_dim[op_dim];

            
            if ((hssize_t)min_mod_chunk_sc[op_dim] == max_fill_chunk_sc[op_dim]) {
                fill_dim[op_dim] = true;

                
                if (disable_edge_filters && old_dim[op_dim] >= (min_mod_chunk_sc[op_dim] + 1))
                    new_unfilt_dim[op_dim] = true;
                else
                    new_unfilt_dim[op_dim] = false;
            } 
            else {
                fill_dim[op_dim]       = false;
                new_unfilt_dim[op_dim] = false;
            } 
        }     
        else {
            fill_dim[op_dim]       = false;
            new_unfilt_dim[op_dim] = false;
        } 

        
        if (disable_edge_filters)
            min_partial_chunk_sc[op_dim] = old_dim[op_dim] / chunk_dim[op_dim];
    } 

    
    for (op_dim = 0; op_dim < (unsigned)space_ndims; op_dim++) {
        bool dims_outside_fill[H5O_LAYOUT_NDIMS]; 
        int  ndims_outside_fill; 
        bool carry; 

        
        if (!shrunk_dim[op_dim])
            continue;
        else {
            assert(max_mod_chunk_sc[op_dim] >= min_mod_chunk_sc[op_dim]);

            
            memset(scaled, 0, (space_ndims * sizeof(scaled[0])));
            scaled[op_dim] = min_mod_chunk_sc[op_dim];

            
            ndims_outside_fill = 0;
            for (u = 0; u < space_ndims; u++)
                if ((hssize_t)scaled[u] > max_fill_chunk_sc[u]) {
                    dims_outside_fill[u] = true;
                    ndims_outside_fill++;
                } 
                else
                    dims_outside_fill[u] = false;
        } 

        carry = false;
        while (!carry) {
            int i; 

            udata.common.scaled = scaled;

            if (0 == ndims_outside_fill) {
                assert(fill_dim[op_dim]);
                assert(scaled[op_dim] == min_mod_chunk_sc[op_dim]);

                
                assert(H5D__chunk_is_partial_edge_chunk(space_ndims, layout->u.chunk.dim, scaled, space_dim));

                
                if (new_unfilt_dim[op_dim]) {
                    new_unfilt_chunk = true;
                    for (u = 0; u < space_ndims; u++)
                        if (scaled[u] == min_partial_chunk_sc[u]) {
                            new_unfilt_chunk = false;
                            break;
                        } 
                }         

                
                assert(!new_unfilt_dim[op_dim] ||
                       (!new_unfilt_chunk != !H5D__chunk_is_partial_edge_chunk(
                                                 space_ndims, layout->u.chunk.dim, scaled, old_dim)));
                assert(!new_unfilt_chunk || new_unfilt_dim[op_dim]);

                
                if (H5D__chunk_prune_fill(&udata, new_unfilt_chunk) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to write fill value");
            } 
            else {
                H5D_chunk_ud_t chk_udata; 

#ifndef NDEBUG
                
                {
                    bool outside_dim = false;

                    for (u = 0; u < space_ndims; u++)
                        if ((scaled[u] * chunk_dim[u]) >= space_dim[u]) {
                            outside_dim = true;
                            break;
                        } 
                    assert(outside_dim);
                } 
#endif            

                
                if (H5D__chunk_lookup(dset, scaled, &chk_udata) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error looking up chunk");

                
                if (UINT_MAX != chk_udata.idx_hint)
                    if (H5D__chunk_cache_evict(dset, rdcc->slot[chk_udata.idx_hint], false) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTREMOVE, FAIL, "unable to evict chunk");

                
                if (H5_addr_defined(chk_udata.chunk_block.offset)) {
                    
                    idx_udata.scaled = udata.common.scaled;

                    
                    if ((sc->ops->remove)(&idx_info, &idx_udata) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTDELETE, FAIL,
                                    "unable to remove chunk entry from index");
                } 
            }     

            
            carry = true;
            for (i = (int)(space_ndims - 1); i >= 0; --i) {
                scaled[i]++;
                if (scaled[i] > max_mod_chunk_sc[i]) {
                    
                    if ((unsigned)i == op_dim) {
                        scaled[i] = min_mod_chunk_sc[i];
                        if (dims_outside_fill[i] && fill_dim[i]) {
                            dims_outside_fill[i] = false;
                            ndims_outside_fill--;
                        } 
                    }     
                    else {
                        scaled[i] = 0;
                        if (dims_outside_fill[i] && max_fill_chunk_sc[i] >= 0) {
                            dims_outside_fill[i] = false;
                            ndims_outside_fill--;
                        } 
                    }     
                }         
                else {
                    
                    if (!dims_outside_fill[i] && (hssize_t)scaled[i] > max_fill_chunk_sc[i]) {
                        dims_outside_fill[i] = true;
                        ndims_outside_fill++;
                    } 

                    
                    carry = false;
                    break;
                } 
            }     
        }         

        
        if (min_mod_chunk_sc[op_dim] == 0)
            break;
        else
            max_mod_chunk_sc[op_dim] = min_mod_chunk_sc[op_dim] - 1;
    } 

    
    H5D__chunk_cinfo_cache_reset(&dset->shared->cache.chunk.last);

done:
    
    if (chunk_space && H5S_close(chunk_space) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release dataspace");
    if (udata_init)
        if (udata.fb_info_init && H5D__fill_term(&udata.fb_info) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release fill buffer info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_PARALLEL

static int
H5D__chunk_addrmap_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata)
{
    H5D_chunk_it_ud2_t *udata = (H5D_chunk_it_ud2_t *)_udata;    
    unsigned            rank  = udata->common.layout->ndims - 1; 
    hsize_t             chunk_index;

    FUNC_ENTER_PACKAGE_NOERR

    
    chunk_index = H5VM_array_offset_pre(rank, udata->common.layout->down_chunks, chunk_rec->scaled);

    
    udata->chunk_addr[chunk_index] = chunk_rec->chunk_addr;

    FUNC_LEAVE_NOAPI(H5_ITER_CONT)
} 

herr_t
H5D__chunk_addrmap(const H5D_t *dset, haddr_t chunk_addr[])
{
    H5D_chk_idx_info_t   idx_info; 
    H5D_chunk_it_ud2_t   udata;    
    H5O_storage_chunk_t *sc;
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(dset);
    assert(dset->shared);
    sc = &(dset->shared->layout.storage.u.chunk);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);
    assert(chunk_addr);

    
    memset(&udata, 0, sizeof(udata));
    udata.common.layout  = &dset->shared->layout.u.chunk;
    udata.common.storage = sc;
    udata.chunk_addr     = chunk_addr;

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if ((sc->ops->iterate)(&idx_info, H5D__chunk_addrmap_cb, &udata) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                    "unable to iterate over chunk index to build address map");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

herr_t
H5D__chunk_delete(H5F_t *f, H5O_t *oh, H5O_layout_t *layout)
{
    H5D_chk_idx_info_t idx_info;            
    H5O_pline_t        pline;               
    bool               pline_read = false;  
    htri_t             exists;              
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(layout);
    H5D_CHUNK_STORAGE_INDEX_CHK(&layout->storage.u.chunk);

    
    if ((exists = H5O_msg_exists_oh(oh, H5O_PLINE_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to check for object header message");
    else if (exists) {
        if (NULL == H5O_msg_read_oh(f, oh, H5O_PLINE_ID, &pline))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get I/O pipeline message");
        pline_read = true;
    } 
    else
        memset(&pline, 0, sizeof(pline));

    
    idx_info.f      = f;
    idx_info.pline  = &pline;
    idx_info.layout = layout;

    
    if ((layout->storage.u.chunk.ops->idx_delete)(&idx_info) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTDELETE, FAIL, "unable to delete chunk index");

done:
    
    if (pline_read)
        if (H5O_msg_reset(H5O_PLINE_ID, &pline) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset I/O pipeline message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_update_cache(H5D_t *dset)
{
    H5D_rdcc_t     *rdcc = &(dset->shared->cache.chunk); 
    H5D_rdcc_ent_t *ent, *next;                          
    H5D_rdcc_ent_t  tmp_head;                            
    H5D_rdcc_ent_t *tmp_tail;                            
    herr_t          ret_value = SUCCEED;                 

    FUNC_ENTER_PACKAGE

    
    assert(dset && H5D_CHUNKED == dset->shared->layout.type);
    assert(dset->shared->layout.u.chunk.ndims > 0 && dset->shared->layout.u.chunk.ndims <= H5O_LAYOUT_NDIMS);

    
    assert((dset->shared->layout.u.chunk.ndims - 1) > 1);

    
    (void)memset(&tmp_head, 0, sizeof(tmp_head));
    rdcc->tmp_head = &tmp_head;
    tmp_tail       = &tmp_head;

    
    for (ent = rdcc->head; ent; ent = next) {
        unsigned old_idx; 

        
        next = ent->next;

        
        old_idx  = ent->idx; 
        ent->idx = H5D__chunk_hash_val(dset->shared, ent->scaled);

        if (old_idx != ent->idx) {
            H5D_rdcc_ent_t *old_ent; 

            
            old_ent = rdcc->slot[ent->idx];
            if (old_ent != NULL) {
                assert(old_ent->locked == false);
                assert(old_ent->deleted == false);

                
                assert(!old_ent->tmp_next);
                assert(!old_ent->tmp_prev);
                tmp_tail->tmp_next = old_ent;
                old_ent->tmp_prev  = tmp_tail;
                tmp_tail           = old_ent;
            } 

            
            rdcc->slot[ent->idx] = ent;

            
            if (ent->tmp_prev) {
                assert(tmp_head.tmp_next);
                assert(tmp_tail != &tmp_head);
                ent->tmp_prev->tmp_next = ent->tmp_next;
                if (ent->tmp_next) {
                    ent->tmp_next->tmp_prev = ent->tmp_prev;
                    ent->tmp_next           = NULL;
                } 
                else {
                    assert(tmp_tail == ent);
                    tmp_tail = ent->tmp_prev;
                } 
                ent->tmp_prev = NULL;
            } 
            else
                rdcc->slot[old_idx] = NULL;
        } 
    }     

    
    tmp_tail = NULL;

    
    while (tmp_head.tmp_next) {
        ent = tmp_head.tmp_next;

        
        if (H5D__chunk_cache_evict(dset, ent, true) < 0)
            HGOTO_ERROR(H5E_IO, H5E_CANTFLUSH, FAIL, "unable to flush one or more raw data chunks");
    } 

done:
    
    rdcc->tmp_head = NULL;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5D__chunk_copy_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata)
{
    H5D_chunk_it_ud3_t *udata = (H5D_chunk_it_ud3_t *)_udata; 
    H5D_chunk_ud_t      udata_dst;                            
    bool                is_vlen     = false;                  
    bool                fix_ref     = false; 
    bool                need_insert = false; 

    
    const H5T_t       *dt_src   = udata->dt_src;
    void              *bkg      = udata->bkg;      
    void              *buf      = udata->buf;      
    size_t             buf_size = udata->buf_size; 
    const H5O_pline_t *pline    = udata->pline;    

    
    bool     must_filter = false;      
    size_t   nbytes;                   
    H5Z_cb_t filter_cb;                
    int      ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    H5_CHECKED_ASSIGN(nbytes, size_t, chunk_rec->nbytes, hsize_t);

    
    filter_cb.op_data = NULL;
    filter_cb.func    = NULL; 

    
    
    if (pline && pline->nused) {
        must_filter = true;
        if ((udata->common.layout->flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) &&
            H5D__chunk_is_partial_edge_chunk(udata->dset_ndims, udata->common.layout->dim, chunk_rec->scaled,
                                             udata->dset_dims))
            must_filter = false;
    }

    
    if (udata->do_convert) {
        if (H5T_detect_class(dt_src, H5T_VLEN, false) > 0)
            is_vlen = true;
        else if ((H5T_get_class(dt_src, false) == H5T_REFERENCE) &&
                 (udata->file_src != udata->idx_info_dst->f))
            fix_ref = true;
        else
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, H5_ITER_ERROR, "unable to copy dataset elements");
    } 

    
    if (nbytes > buf_size) {
        void *new_buf; 

        
        if (NULL == (new_buf = H5MM_realloc(udata->buf, nbytes)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, H5_ITER_ERROR,
                        "memory allocation failed for raw data chunk");
        udata->buf = new_buf;
        if (udata->bkg) {
            if (NULL == (new_buf = H5MM_realloc(udata->bkg, nbytes)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, H5_ITER_ERROR,
                            "memory allocation failed for raw data chunk");
            udata->bkg = new_buf;
            if (!udata->cpy_info->expand_ref)
                memset((uint8_t *)udata->bkg + buf_size, 0, (size_t)(nbytes - buf_size));

            bkg = udata->bkg;
        } 

        buf             = udata->buf;
        udata->buf_size = buf_size = nbytes;
    } 

    if (udata->chunk_in_cache && udata->chunk) {
        assert(!H5_addr_defined(chunk_rec->chunk_addr));
        H5MM_memcpy(buf, udata->chunk, nbytes);
        udata->chunk = NULL;
    }
    else {
        H5D_rdcc_ent_t *ent = NULL; 
        unsigned        idx;        
        unsigned        u;          
        H5D_shared_t   *shared_fo = (H5D_shared_t *)udata->cpy_info->shared_fo;

        
        if (shared_fo && shared_fo->cache.chunk.nslots > 0) {
            
            idx = H5D__chunk_hash_val(shared_fo, chunk_rec->scaled);

            
            ent = shared_fo->cache.chunk.slot[idx];
            if (ent) {
                
                udata->chunk_in_cache = true;

                
                for (u = 0; u < shared_fo->ndims; u++)
                    if (chunk_rec->scaled[u] != ent->scaled[u]) {
                        udata->chunk_in_cache = false;
                        break;
                    } 
            }         
        }             

        if (udata->chunk_in_cache) {

            if (NULL == ent)
                HGOTO_ERROR(H5E_IO, H5E_BADVALUE, H5_ITER_ERROR, "NULL chunk entry pointer");

            assert(H5_addr_defined(chunk_rec->chunk_addr));
            assert(H5_addr_defined(ent->chunk_block.offset));

            H5_CHECKED_ASSIGN(nbytes, size_t, shared_fo->layout.u.chunk.size, hsize_t);
            H5MM_memcpy(buf, ent->chunk, nbytes);
        }
        else {
            
            if (H5F_block_read(udata->file_src, H5FD_MEM_DRAW, chunk_rec->chunk_addr, nbytes, buf) < 0)
                HGOTO_ERROR(H5E_IO, H5E_READERROR, H5_ITER_ERROR, "unable to read raw data chunk");
        }
    }

    
    if (must_filter && (is_vlen || fix_ref) && !udata->chunk_in_cache) {
        unsigned filter_mask = chunk_rec->filter_mask;

        if (H5Z_pipeline(pline, H5Z_FLAG_REVERSE, &filter_mask, H5Z_NO_EDC, filter_cb, &nbytes, &buf_size,
                         &buf) < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTFILTER, H5_ITER_ERROR, "data pipeline read failed");
    } 

    
    if (is_vlen) {
        H5T_path_t  *tpath_src_mem    = udata->tpath_src_mem;
        H5T_path_t  *tpath_mem_dst    = udata->tpath_mem_dst;
        const H5T_t *dt_dst           = udata->dt_dst;
        const H5T_t *dt_mem           = udata->dt_mem;
        H5S_t       *buf_space        = udata->buf_space;
        void        *reclaim_buf      = udata->reclaim_buf;
        size_t       reclaim_buf_size = udata->reclaim_buf_size;

        
        if (H5T_convert(tpath_src_mem, dt_src, dt_mem, udata->nelmts, (size_t)0, (size_t)0, buf, bkg) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, H5_ITER_ERROR, "datatype conversion failed");

        
        H5MM_memcpy(reclaim_buf, buf, reclaim_buf_size);

        
        memset(bkg, 0, buf_size);

        
        if (H5T_convert(tpath_mem_dst, dt_mem, dt_dst, udata->nelmts, (size_t)0, (size_t)0, buf, bkg) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, H5_ITER_ERROR, "datatype conversion failed");

        
        if (H5T_reclaim(dt_mem, buf_space, reclaim_buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, H5_ITER_ERROR, "unable to reclaim variable-length data");
    } 
    else if (fix_ref) {
        
        
        if (udata->cpy_info->expand_ref) {
            
            if (H5O_copy_expand_ref(udata->file_src, dt_src, buf, nbytes, udata->idx_info_dst->f, bkg,
                                    udata->cpy_info) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, H5_ITER_ERROR, "unable to copy reference attribute");
        } 

        
        H5MM_memcpy(buf, bkg, buf_size);
    } 

    
    udata_dst.common.layout      = &udata->idx_info_dst->layout->u.chunk;
    udata_dst.common.storage     = &udata->idx_info_dst->layout->storage.u.chunk;
    udata_dst.common.scaled      = chunk_rec->scaled;
    udata_dst.chunk_block.offset = HADDR_UNDEF;
    udata_dst.chunk_block.length = chunk_rec->nbytes;
    udata_dst.filter_mask        = chunk_rec->filter_mask;

    
    if (must_filter && (is_vlen || fix_ref || udata->chunk_in_cache)) {
        if (H5Z_pipeline(pline, 0, &(udata_dst.filter_mask), H5Z_NO_EDC, filter_cb, &nbytes, &buf_size,
                         &buf) < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTFILTER, H5_ITER_ERROR, "output pipeline failed");

        H5_CHECKED_ASSIGN(udata_dst.chunk_block.length, hsize_t, nbytes, size_t);
        udata->buf      = buf;
        udata->buf_size = buf_size;
    } 

    udata->chunk_in_cache = false;

    udata_dst.chunk_idx =
        H5VM_array_offset_pre(udata_dst.common.layout->ndims - 1, udata_dst.common.layout->max_down_chunks,
                              udata_dst.common.scaled);

    
    if (H5D__chunk_file_alloc(udata->idx_info_dst, NULL, &udata_dst.chunk_block, &need_insert,
                              udata_dst.common.scaled) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL, "unable to insert/resize chunk on chunk level");

    
    assert(H5_addr_defined(udata_dst.chunk_block.offset));
    if (H5F_block_write(udata->idx_info_dst->f, H5FD_MEM_DRAW, udata_dst.chunk_block.offset, nbytes, buf) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, H5_ITER_ERROR, "unable to write raw data to file");

    
    H5_BEGIN_TAG(H5AC__COPIED_TAG)

    
    if (need_insert && udata->idx_info_dst->layout->storage.u.chunk.ops->insert)
        if ((udata->idx_info_dst->layout->storage.u.chunk.ops->insert)(udata->idx_info_dst, &udata_dst,
                                                                       NULL) < 0)
            HGOTO_ERROR_TAG(H5E_DATASET, H5E_CANTINSERT, H5_ITER_ERROR,
                            "unable to insert chunk addr into index");

    
    H5_END_TAG

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_copy(H5F_t *f_src, H5O_layout_t *layout_src, H5F_t *f_dst, H5O_layout_t *layout_dst,
                const H5S_extent_t *ds_extent_src, H5T_t *dt_src, const H5O_pline_t *pline_src,
                H5O_copy_t *cpy_info)
{
    H5D_chunk_it_ud3_t udata;                       
    H5D_chk_idx_info_t idx_info_dst;                
    H5D_chk_idx_info_t idx_info_src;                
    int                sndims;                      
    hsize_t            curr_dims[H5O_LAYOUT_NDIMS]; 
    hsize_t            max_dims[H5O_LAYOUT_NDIMS];  
    H5O_pline_t        _pline;                      
    const H5O_pline_t *pline;                       
    H5T_path_t        *tpath_src_mem = NULL;        
    H5T_path_t        *tpath_mem_dst = NULL;        
    H5T_t             *dt_dst        = NULL;        
    H5T_t             *dt_mem        = NULL;        
    size_t             buf_size;                    
    size_t             reclaim_buf_size;            
    void              *buf             = NULL;      
    void              *bkg             = NULL;      
    void              *reclaim_buf     = NULL;      
    H5S_t             *buf_space       = NULL;      
    size_t             nelmts          = 0;         
    bool               do_convert      = false;     
    bool               copy_setup_done = false;     
    herr_t             ret_value       = SUCCEED;   

    FUNC_ENTER_PACKAGE

    
    assert(f_src);
    assert(layout_src);
    H5D_CHUNK_STORAGE_INDEX_CHK(&layout_src->storage.u.chunk);
    assert(f_dst);
    assert(layout_dst);
    H5D_CHUNK_STORAGE_INDEX_CHK(&layout_dst->storage.u.chunk);
    assert(ds_extent_src);
    assert(dt_src);

    
    if (NULL == pline_src) {
        memset(&_pline, 0, sizeof(_pline));
        pline = &_pline;
    } 
    else
        pline = pline_src;

    
    if (H5D_chunk_idx_reset(&layout_dst->storage.u.chunk, true) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to reset chunked storage index in dest");

    
    {
        unsigned ndims; 

        
        if ((sndims = H5S_extent_get_dims(ds_extent_src, curr_dims, max_dims)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get dataspace dimensions");
        H5_CHECKED_ASSIGN(ndims, unsigned, sndims, int);

        
        if (H5D__chunk_set_info_real(&layout_src->u.chunk, ndims, curr_dims, max_dims) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set source layout's chunk info");

        
        if (H5D__chunk_set_info_real(&layout_dst->u.chunk, ndims, curr_dims, max_dims) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set destination layout's chunk info");
    } 

    
    idx_info_src.f      = f_src;
    idx_info_src.pline  = pline;
    idx_info_src.layout = layout_src;

    idx_info_dst.f      = f_dst;
    idx_info_dst.pline  = pline; 
    idx_info_dst.layout = layout_dst;

    
    if ((layout_src->storage.u.chunk.ops->copy_setup)(&idx_info_src, &idx_info_dst) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                    "unable to set up index-specific chunk copying information");
    copy_setup_done = true;

    
    if (H5T_detect_class(dt_src, H5T_VLEN, false) > 0) {
        size_t   mem_dt_size; 
        size_t   tmp_dt_size; 
        size_t   max_dt_size; 
        hsize_t  buf_dim;     
        uint64_t nelmts_64;   
        unsigned u;

        
        if (NULL == (dt_mem = H5T_copy(dt_src, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy");

        
        if (NULL == (dt_dst = H5T_copy(dt_src, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy");
        if (H5T_set_loc(dt_dst, H5F_VOL_OBJ(f_dst), H5T_LOC_DISK) < 0) {
            (void)H5T_close_real(dt_dst);
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "cannot mark datatype on disk");
        } 

        
        if (NULL == (tpath_src_mem = H5T_path_find(dt_src, dt_mem)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to convert between src and mem datatypes");
        if (NULL == (tpath_mem_dst = H5T_path_find(dt_mem, dt_dst)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to convert between mem and dst datatypes");

        
        if (0 == (max_dt_size = H5T_get_size(dt_src)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        if (0 == (mem_dt_size = H5T_get_size(dt_mem)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        max_dt_size = MAX(max_dt_size, mem_dt_size);
        if (0 == (tmp_dt_size = H5T_get_size(dt_dst)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        max_dt_size = MAX(max_dt_size, tmp_dt_size);

        
        nelmts_64 = 1;
        for (u = 0; u < (layout_src->u.chunk.ndims - 1); u++)
            nelmts_64 *= layout_src->u.chunk.dim[u];
        nelmts = (size_t)nelmts_64;
        if ((uint64_t)nelmts != nelmts_64)
            HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "number of chunk elements too big to fit in size_t");

        
        buf_dim = nelmts;
        if (NULL == (buf_space = H5S_create_simple((unsigned)1, &buf_dim, NULL)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create simple dataspace");

        
        buf_size         = nelmts * max_dt_size;
        reclaim_buf_size = nelmts * mem_dt_size;

        
        if (NULL == (reclaim_buf = H5MM_malloc(reclaim_buf_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for raw data chunk");

        
        do_convert = true;
    } 
    else {
        if (H5T_get_class(dt_src, false) == H5T_REFERENCE) {
            
            do_convert = true;
        } 

        H5_CHECKED_ASSIGN(buf_size, size_t, layout_src->u.chunk.size, hsize_t);
        reclaim_buf_size = 0;
    } 

    
    if (do_convert) {
        
        if (NULL == (bkg = H5MM_malloc(buf_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for raw data chunk");

        
        if (!cpy_info->expand_ref && ((H5T_get_class(dt_src, false) == H5T_REFERENCE) && (f_src != f_dst)))
            
            memset(bkg, 0, buf_size);
    } 

    
    if (NULL == (buf = H5MM_malloc(buf_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for raw data chunk");

    
    memset(&udata, 0, sizeof udata);
    udata.common.layout    = &layout_src->u.chunk;
    udata.common.storage   = &layout_src->storage.u.chunk;
    udata.file_src         = f_src;
    udata.idx_info_dst     = &idx_info_dst;
    udata.buf              = buf;
    udata.bkg              = bkg;
    udata.buf_size         = buf_size;
    udata.dt_src           = dt_src;
    udata.dt_dst           = dt_dst;
    udata.dt_mem           = dt_mem;
    udata.do_convert       = do_convert;
    udata.tpath_src_mem    = tpath_src_mem;
    udata.tpath_mem_dst    = tpath_mem_dst;
    udata.reclaim_buf      = reclaim_buf;
    udata.reclaim_buf_size = reclaim_buf_size;
    udata.buf_space        = buf_space;
    udata.nelmts           = nelmts;
    udata.pline            = pline;
    udata.dset_ndims       = (unsigned)sndims;
    udata.dset_dims        = curr_dims;
    udata.cpy_info         = cpy_info;
    udata.chunk_in_cache   = false;
    udata.chunk            = NULL;

    
    if ((layout_src->storage.u.chunk.ops->iterate)(&idx_info_src, H5D__chunk_copy_cb, &udata) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADITER, FAIL, "unable to iterate over chunk index to copy data");

    
    if (udata.cpy_info->shared_fo) {
        H5D_rdcc_ent_t *ent, *next;
        H5D_chunk_rec_t chunk_rec;
        H5D_shared_t   *shared_fo = (H5D_shared_t *)udata.cpy_info->shared_fo;

        chunk_rec.nbytes      = layout_src->u.chunk.size;
        chunk_rec.filter_mask = 0;
        chunk_rec.chunk_addr  = HADDR_UNDEF;

        for (ent = shared_fo->cache.chunk.head; ent; ent = next) {
            if (!H5_addr_defined(ent->chunk_block.offset)) {
                H5MM_memcpy(chunk_rec.scaled, ent->scaled, sizeof(chunk_rec.scaled));
                udata.chunk          = ent->chunk;
                udata.chunk_in_cache = true;
                if (H5D__chunk_copy_cb(&chunk_rec, &udata) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy chunk data in cache");
            }
            next = ent->next;
        } 
    }

    
    buf = udata.buf;
    bkg = udata.bkg;

done:
    if (dt_dst && (H5T_close(dt_dst) < 0))
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    if (dt_mem && (H5T_close(dt_mem) < 0))
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    if (buf_space && H5S_close(buf_space) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary dataspace");
    if (buf)
        H5MM_xfree(buf);
    if (bkg)
        H5MM_xfree(bkg);
    if (reclaim_buf)
        H5MM_xfree(reclaim_buf);

    
    if (copy_setup_done)
        if (layout_src->storage.u.chunk.ops->copy_shutdown &&
            (layout_src->storage.u.chunk.ops->copy_shutdown)(&layout_src->storage.u.chunk,
                                                             &layout_dst->storage.u.chunk) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to shut down index copying info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_bh_info(const H5O_loc_t *loc, H5O_t *oh, H5O_layout_t *layout, hsize_t *index_size)
{
    H5D_chk_idx_info_t   idx_info;     
    H5S_t               *space = NULL; 
    H5O_pline_t          pline;        
    H5O_storage_chunk_t *sc = &(layout->storage.u.chunk);
    htri_t               exists;                  
    bool                 idx_info_init = false;   
    bool                 pline_read    = false;   
    herr_t               ret_value     = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(layout);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);
    assert(index_size);

    
    if ((exists = H5O_msg_exists_oh(oh, H5O_PLINE_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to read object header");
    else if (exists) {
        if (NULL == H5O_msg_read_oh(loc->file, oh, H5O_PLINE_ID, &pline))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't find I/O pipeline message");
        pline_read = true;
    } 
    else
        memset(&pline, 0, sizeof(pline));

    
    idx_info.f      = loc->file;
    idx_info.pline  = &pline;
    idx_info.layout = layout;

    
    if (NULL == (space = H5S_read(loc)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to load dataspace info from dataset header");

    
    if (sc->ops->init && (sc->ops->init)(&idx_info, space, loc->addr) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize indexing information");
    idx_info_init = true;

    
    if (sc->ops->size && (sc->ops->size)(&idx_info, index_size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to retrieve chunk index info");

done:
    
    if (idx_info_init && sc->ops->dest && (sc->ops->dest)(&idx_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to release chunk index info");
    if (pline_read && H5O_msg_reset(H5O_PLINE_ID, &pline) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset I/O pipeline message");
    if (space && H5S_close(space) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5D__chunk_dump_index_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata)
{
    H5D_chunk_it_ud4_t *udata = (H5D_chunk_it_ud4_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    if (udata->stream) {
        unsigned u; 

        
        if (!udata->header_displayed) {
            Rfprintf(udata->stream, "           Flags    Bytes     Address          Logical Offset\n");
            Rfprintf(udata->stream, "        ========== ======== ========== ==============================\n");

            
            udata->header_displayed = true;
        } 

        
        Rfprintf(udata->stream, "        0x%08x %8" PRIuHSIZE " %10" PRIuHADDR " [", chunk_rec->filter_mask,
                chunk_rec->nbytes, chunk_rec->chunk_addr);
        for (u = 0; u < udata->ndims; u++)
            Rfprintf(udata->stream, "%s%" PRIuHSIZE, (u ? ", " : ""),
                    (chunk_rec->scaled[u] * udata->chunk_dim[u]));
        Rfputs("]\n", udata->stream);
    } 

    FUNC_LEAVE_NOAPI(H5_ITER_CONT)
} 

herr_t
H5D__chunk_dump_index(H5D_t *dset, FILE *stream)
{
    H5O_storage_chunk_t *sc        = &(dset->shared->layout.storage.u.chunk);
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    H5D_CHUNK_STORAGE_INDEX_CHK(sc);

    
    if (stream) {
        H5D_chk_idx_info_t idx_info; 
        H5D_chunk_it_ud4_t udata;    

        
        if ((sc->ops->dump)(sc, stream) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "unable to dump chunk index info");

        
        idx_info.f      = dset->oloc.file;
        idx_info.pline  = &dset->shared->dcpl_cache.pline;
        idx_info.layout = &dset->shared->layout;

        
        udata.stream           = stream;
        udata.header_displayed = false;
        udata.ndims            = dset->shared->layout.u.chunk.ndims;
        udata.chunk_dim        = dset->shared->layout.u.chunk.dim;

        
        if ((sc->ops->iterate)(&idx_info, H5D__chunk_dump_index_cb, &udata) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADITER, FAIL,
                        "unable to iterate over chunk index to dump chunk info");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5D_CHUNK_DEBUG

herr_t
H5D__chunk_stats(const H5D_t *dset, bool headers)
{
    H5D_rdcc_t *rdcc = &(dset->shared->cache.chunk);
    double      miss_rate;
    char        ascii[32];
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    if (!H5DEBUG(AC))
        HGOTO_DONE(SUCCEED);

    if (headers) {
        Rfprintf(H5DEBUG(AC), "H5D: raw data cache statistics\n");
        Rfprintf(H5DEBUG(AC), "   %-18s %8s %8s %8s %8s+%-8s\n", "Layer", "Hits", "Misses", "MissRate",
                "Inits", "Flushes");
        Rfprintf(H5DEBUG(AC), "   %-18s %8s %8s %8s %8s-%-8s\n", "-----", "----", "------", "--------",
                "-----", "-------");
    }

#ifdef H5AC_DEBUG
    if (H5DEBUG(AC))
        headers = true;
#endif

    if (headers) {
        if (rdcc->stats.nhits > 0 || rdcc->stats.nmisses > 0) {
            miss_rate = 100.0 * rdcc->stats.nmisses / (rdcc->stats.nhits + rdcc->stats.nmisses);
        }
        else {
            miss_rate = 0.0;
        }
        if (miss_rate > 100) {
            snprintf(ascii, sizeof(ascii), "%7d%%", (int)(miss_rate + 0.5));
        }
        else {
            snprintf(ascii, sizeof(ascii), "%7.2f%%", miss_rate);
        }

        Rfprintf(H5DEBUG(AC), "   %-18s %8u %8u %7s %8d+%-9ld\n", "raw data chunks", rdcc->stats.nhits,
                rdcc->stats.nmisses, ascii, rdcc->stats.ninits,
                (long)(rdcc->stats.nflushes) - (long)(rdcc->stats.ninits));
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

static herr_t
H5D__nonexistent_readvv_cb(hsize_t H5_ATTR_UNUSED dst_off, hsize_t src_off, size_t len, void *_udata)
{
    H5D_chunk_readvv_ud_t *udata = (H5D_chunk_readvv_ud_t *)_udata; 
    H5D_fill_buf_info_t    fb_info;                                 
    bool                   fb_info_init = false;   
    herr_t                 ret_value    = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5D__fill_init(&fb_info, (udata->rbuf + src_off), NULL, NULL, NULL, NULL,
                       &udata->dset->shared->dcpl_cache.fill, udata->dset->shared->type, (size_t)0, len) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize fill buffer info");
    fb_info_init = true;

    
    if (fb_info.has_vlen_fill_type && H5D__fill_refill_vl(&fb_info, fb_info.elmts_per_buf) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "can't refill fill value buffer");

done:
    
    if (fb_info_init && H5D__fill_term(&fb_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release fill buffer info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static ssize_t
H5D__nonexistent_readvv(const H5D_io_info_t H5_ATTR_NDEBUG_UNUSED *io_info,
                        const H5D_dset_io_info_t *dset_info, size_t chunk_max_nseq, size_t *chunk_curr_seq,
                        size_t chunk_len_arr[], hsize_t chunk_off_arr[], size_t mem_max_nseq,
                        size_t *mem_curr_seq, size_t mem_len_arr[], hsize_t mem_off_arr[])
{
    H5D_chunk_readvv_ud_t udata;          
    ssize_t               ret_value = -1; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(chunk_curr_seq);
    assert(chunk_len_arr);
    assert(chunk_off_arr);
    assert(mem_curr_seq);
    assert(mem_len_arr);
    assert(mem_off_arr);

    
    udata.rbuf = (unsigned char *)dset_info->buf.vp;
    udata.dset = dset_info->dset;

    
    if ((ret_value = H5VM_opvv(chunk_max_nseq, chunk_curr_seq, chunk_len_arr, chunk_off_arr, mem_max_nseq,
                               mem_curr_seq, mem_len_arr, mem_off_arr, H5D__nonexistent_readvv_cb, &udata)) <
        0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPERATE, FAIL, "can't perform vectorized fill value init");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5D__chunk_is_partial_edge_chunk(unsigned dset_ndims, const hsize_t *chunk_dims, const hsize_t scaled[],
                                 const hsize_t *dset_dims)
{
    unsigned u;                 
    bool     ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(scaled);
    assert(dset_ndims > 0);
    assert(dset_dims);
    assert(chunk_dims);

    
    for (u = 0; u < dset_ndims; u++)
        if (((scaled[u] + 1) * chunk_dims[u]) > dset_dims[u])
            HGOTO_DONE(true);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_file_alloc(const H5D_chk_idx_info_t *idx_info, const H5F_block_t *old_chunk,
                      H5F_block_t *new_chunk, bool *need_insert, const hsize_t *scaled)
{
    bool   alloc_chunk = false;   
    herr_t ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->layout);
    assert(new_chunk);
    assert(need_insert);

    *need_insert = false;

    
    if (idx_info->pline->nused > 0) {
        
        assert(idx_info->layout->storage.u.chunk.idx_type != H5D_CHUNK_IDX_NONE);

        
        
        H5D_CHUNK_ENCODE_SIZE_CHECK(idx_info->layout, new_chunk->length, FAIL);

        if (old_chunk && H5_addr_defined(old_chunk->offset)) {
            
            assert(!H5_addr_defined(new_chunk->offset) || H5_addr_eq(new_chunk->offset, old_chunk->offset));

            
            if (new_chunk->length != old_chunk->length) {
                
                
                if (!(H5F_INTENT(idx_info->f) & H5F_ACC_SWMR_WRITE))
                    if (H5MF_xfree(idx_info->f, H5FD_MEM_DRAW, old_chunk->offset, old_chunk->length) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to free chunk");
                alloc_chunk = true;
            } 
            else {
                
                if (!H5_addr_defined(new_chunk->offset))
                    new_chunk->offset = old_chunk->offset;
            } 
        }     
        else {
            assert(!H5_addr_defined(new_chunk->offset));
            alloc_chunk = true;
        } 
    }     
    else {
        assert(!H5_addr_defined(new_chunk->offset));
        assert(new_chunk->length == idx_info->layout->u.chunk.size);
        alloc_chunk = true;
    } 

    
    if (alloc_chunk) {
        switch (idx_info->layout->storage.u.chunk.idx_type) {
            case H5D_CHUNK_IDX_NONE: {
                H5D_chunk_ud_t udata;

                udata.common.scaled = scaled;
                if ((idx_info->layout->storage.u.chunk.ops->get_addr)(idx_info, &udata) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't query chunk address");
                new_chunk->offset = udata.chunk_block.offset;
                assert(new_chunk->length == udata.chunk_block.length);
                break;
            }

            case H5D_CHUNK_IDX_EARRAY:
            case H5D_CHUNK_IDX_FARRAY:
            case H5D_CHUNK_IDX_BT2:
            case H5D_CHUNK_IDX_BTREE:
            case H5D_CHUNK_IDX_SINGLE:
                assert(new_chunk->length > 0);
                new_chunk->offset = H5MF_alloc(idx_info->f, H5FD_MEM_DRAW, new_chunk->length);
                if (!H5_addr_defined(new_chunk->offset))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "file allocation failed");
                *need_insert = true;
                break;

            case H5D_CHUNK_IDX_NTYPES:
            default:
                assert(0 && "This should never be executed!");
                break;
        } 
    }     

    assert(H5_addr_defined(new_chunk->offset));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5D__chunk_format_convert_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata)
{
    H5D_chunk_it_ud5_t *udata = (H5D_chunk_it_ud5_t *)_udata; 
    H5D_chk_idx_info_t *new_idx_info;                         
    H5D_chunk_ud_t      insert_udata;                         
    haddr_t             chunk_addr;                           
    size_t              nbytes;                               
    void               *buf       = NULL;                     
    int                 ret_value = H5_ITER_CONT;             

    FUNC_ENTER_PACKAGE

    
    new_idx_info = udata->new_idx_info;
    H5_CHECKED_ASSIGN(nbytes, size_t, chunk_rec->nbytes, hsize_t);
    chunk_addr = chunk_rec->chunk_addr;

    if (new_idx_info->pline->nused &&
        (new_idx_info->layout->u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) &&
        (H5D__chunk_is_partial_edge_chunk(udata->dset_ndims, new_idx_info->layout->u.chunk.dim,
                                          chunk_rec->scaled, udata->dset_dims))) {

        
        

        unsigned filter_mask = chunk_rec->filter_mask;
        H5Z_cb_t filter_cb;          
        size_t   read_size = nbytes; 

        assert(read_size == new_idx_info->layout->u.chunk.size);

        
        filter_cb.op_data = NULL;
        filter_cb.func    = NULL; 

        
        if (NULL == (buf = H5MM_malloc(read_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, H5_ITER_ERROR,
                        "memory allocation failed for raw data chunk");

        
        if (H5F_block_read(new_idx_info->f, H5FD_MEM_DRAW, chunk_addr, read_size, buf) < 0)
            HGOTO_ERROR(H5E_IO, H5E_READERROR, H5_ITER_ERROR, "unable to read raw data chunk");

        
        if (H5Z_pipeline(new_idx_info->pline, 0, &filter_mask, H5Z_NO_EDC, filter_cb, &nbytes, &read_size,
                         &buf) < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTFILTER, H5_ITER_ERROR, "output pipeline failed");

#if H5_SIZEOF_SIZE_T > 4
        
        if (nbytes > ((size_t)0xffffffff))
            HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, H5_ITER_ERROR, "chunk too large for 32-bit length");
#endif 

        
        if ((chunk_addr = H5MF_alloc(new_idx_info->f, H5FD_MEM_DRAW, (hsize_t)nbytes)) == HADDR_UNDEF)
            HGOTO_ERROR(H5E_DATASET, H5E_NOSPACE, H5_ITER_ERROR, "file allocation failed for filtered chunk");
        assert(H5_addr_defined(chunk_addr));

        
        if (H5F_block_write(new_idx_info->f, H5FD_MEM_DRAW, chunk_addr, nbytes, buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, H5_ITER_ERROR, "unable to write raw data to file");
    } 

    
    insert_udata.chunk_block.offset = chunk_addr;
    insert_udata.chunk_block.length = nbytes;
    insert_udata.filter_mask        = chunk_rec->filter_mask;
    insert_udata.common.scaled      = chunk_rec->scaled;
    insert_udata.common.layout      = &new_idx_info->layout->u.chunk;
    insert_udata.common.storage     = &new_idx_info->layout->storage.u.chunk;

    
    if ((new_idx_info->layout->storage.u.chunk.ops->insert)(new_idx_info, &insert_udata, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, H5_ITER_ERROR, "unable to insert chunk addr into index");

done:
    if (buf)
        H5MM_xfree(buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_format_convert(H5D_t *dset, H5D_chk_idx_info_t *idx_info, H5D_chk_idx_info_t *new_idx_info)
{
    H5D_chunk_it_ud5_t udata;               
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    udata.new_idx_info = new_idx_info;
    udata.dset_ndims   = dset->shared->ndims;
    udata.dset_dims    = dset->shared->curr_dims;

    
    if ((idx_info->layout->storage.u.chunk.ops->iterate)(idx_info, H5D__chunk_format_convert_cb, &udata) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADITER, FAIL, "unable to iterate over chunk index to chunk info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5D__chunk_index_empty_cb(const H5D_chunk_rec_t H5_ATTR_UNUSED *chunk_rec, void *_udata)
{
    bool *empty     = (bool *)_udata;
    int   ret_value = H5_ITER_STOP;

    FUNC_ENTER_PACKAGE_NOERR

    *empty = false;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_index_empty(const H5D_t *dset, bool *empty)
{
    H5D_chk_idx_info_t idx_info;            
    H5D_rdcc_ent_t    *ent;                 
    const H5D_rdcc_t  *rdcc      = NULL;    
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    assert(dset);
    assert(dset->shared);
    assert(empty);

    rdcc = &(dset->shared->cache.chunk); 
    assert(rdcc);

    
    for (ent = rdcc->head; ent; ent = ent->next)
        
        if (H5D__chunk_flush_entry(dset, ent, false) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "cannot flush indexed storage buffer");

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    *empty = true;

    if (H5_addr_defined(idx_info.layout->storage.u.chunk.idx_addr)) {
        
        if ((dset->shared->layout.storage.u.chunk.ops->iterate)(&idx_info, H5D__chunk_index_empty_cb, empty) <
            0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                        "unable to retrieve allocated chunk information from index");
    }

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static int
H5D__get_num_chunks_cb(const H5D_chunk_rec_t H5_ATTR_UNUSED *chunk_rec, void *_udata)
{
    hsize_t *num_chunks = (hsize_t *)_udata;
    int      ret_value  = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(num_chunks);

    (*num_chunks)++;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__get_num_chunks(const H5D_t *dset, const H5S_t H5_ATTR_UNUSED *space, hsize_t *nchunks)
{
    H5D_chk_idx_info_t idx_info;            
    hsize_t            num_chunks = 0;      
    H5D_rdcc_ent_t    *ent;                 
    const H5D_rdcc_t  *rdcc      = NULL;    
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    assert(dset);
    assert(dset->shared);
    assert(space);
    assert(nchunks);

    rdcc = &(dset->shared->cache.chunk); 
    assert(rdcc);

    
    for (ent = rdcc->head; ent; ent = ent->next)
        
        if (H5D__chunk_flush_entry(dset, ent, false) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "cannot flush indexed storage buffer");

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if (!H5_addr_defined(idx_info.layout->storage.u.chunk.idx_addr))
        *nchunks = 0;
    else {
        
        if ((dset->shared->layout.storage.u.chunk.ops->iterate)(&idx_info, H5D__get_num_chunks_cb,
                                                                &num_chunks) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                        "unable to retrieve allocated chunk information from index");
        *nchunks = num_chunks;
    }

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static int
H5D__get_chunk_info_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata)
{
    H5D_chunk_info_iter_ud_t *chunk_info = (H5D_chunk_info_iter_ud_t *)_udata;
    int                       ret_value  = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(chunk_rec);
    assert(chunk_info);

    
    if (chunk_info->curr_idx == chunk_info->chunk_idx) {
        hsize_t ii = 0; 

        
        chunk_info->filter_mask = chunk_rec->filter_mask;
        chunk_info->chunk_addr  = chunk_rec->chunk_addr;
        chunk_info->nbytes      = chunk_rec->nbytes;
        for (ii = 0; ii < chunk_info->ndims; ii++)
            chunk_info->scaled[ii] = chunk_rec->scaled[ii];
        chunk_info->found = true;

        
        ret_value = H5_ITER_STOP;
    }
    
    else
        chunk_info->curr_idx++;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__get_chunk_info(const H5D_t *dset, const H5S_t H5_ATTR_UNUSED *space, hsize_t chk_index, hsize_t *offset,
                    unsigned *filter_mask, haddr_t *addr, hsize_t *size)
{
    H5D_chk_idx_info_t idx_info;            
    const H5D_rdcc_t  *rdcc = NULL;         
    H5D_rdcc_ent_t    *ent;                 
    hsize_t            ii        = 0;       
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    assert(dset);
    assert(dset->shared);
    assert(space);

    
    rdcc = &(dset->shared->cache.chunk);
    assert(rdcc);

    
    for (ent = rdcc->head; ent; ent = ent->next)
        
        if (H5D__chunk_flush_entry(dset, ent, false) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "cannot flush indexed storage buffer");

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if (addr)
        *addr = HADDR_UNDEF;
    if (size)
        *size = 0;

    
    if (H5_addr_defined(idx_info.layout->storage.u.chunk.idx_addr)) {

        H5D_chunk_info_iter_ud_t udata;

        
        udata.chunk_idx   = chk_index;
        udata.curr_idx    = 0;
        udata.ndims       = dset->shared->ndims;
        udata.nbytes      = 0;
        udata.filter_mask = 0;
        udata.chunk_addr  = HADDR_UNDEF;
        udata.found       = false;

        
        if ((dset->shared->layout.storage.u.chunk.ops->iterate)(&idx_info, H5D__get_chunk_info_cb, &udata) <
            0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                        "unable to retrieve allocated chunk information from index");

        
        if (udata.found) {
            if (filter_mask)
                *filter_mask = udata.filter_mask;
            if (addr)
                *addr = udata.chunk_addr + H5F_BASE_ADDR(dset->oloc.file);
            if (size)
                *size = udata.nbytes;
            if (offset)
                for (ii = 0; ii < udata.ndims; ii++)
                    offset[ii] = udata.scaled[ii] * dset->shared->layout.u.chunk.dim[ii];
        }
    }

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static int
H5D__get_chunk_info_by_coord_cb(const H5D_chunk_rec_t *chunk_rec, void *_udata)
{
    H5D_chunk_info_iter_ud_t *chunk_info = (H5D_chunk_info_iter_ud_t *)_udata;
    bool                      different  = false;       
    hsize_t                   ii;                       
    int                       ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(chunk_rec);
    assert(chunk_info);

    
    for (ii = 0; ii < chunk_info->ndims && !different; ii++)
        if (chunk_info->scaled[ii] != chunk_rec->scaled[ii])
            different = true;

    
    if (!different) {
        chunk_info->nbytes      = chunk_rec->nbytes;
        chunk_info->filter_mask = chunk_rec->filter_mask;
        chunk_info->chunk_addr  = chunk_rec->chunk_addr;
        chunk_info->found       = true;

        
        ret_value = H5_ITER_STOP;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__get_chunk_info_by_coord(const H5D_t *dset, const hsize_t *offset, unsigned *filter_mask, haddr_t *addr,
                             hsize_t *size)
{
    const H5O_layout_t *layout = NULL;       
    const H5D_rdcc_t   *rdcc   = NULL;       
    H5D_rdcc_ent_t     *ent;                 
    H5D_chk_idx_info_t  idx_info;            
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(dset);
    assert(dset->shared);
    assert(offset);

    
    layout = &(dset->shared->layout);
    rdcc   = &(dset->shared->cache.chunk);
    assert(layout);
    assert(rdcc);
    assert(H5D_CHUNKED == layout->type);

    
    for (ent = rdcc->head; ent; ent = ent->next)
        
        if (H5D__chunk_flush_entry(dset, ent, false) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "cannot flush indexed storage buffer");

    
    if (addr)
        *addr = HADDR_UNDEF;
    if (size)
        *size = 0;

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = &dset->shared->layout;

    
    if (H5_addr_defined(idx_info.layout->storage.u.chunk.idx_addr)) {

        H5D_chunk_info_iter_ud_t udata;

        
        H5VM_chunk_scaled(dset->shared->ndims, offset, layout->u.chunk.dim, udata.scaled);
        udata.scaled[dset->shared->ndims] = 0;

        
        udata.ndims       = dset->shared->ndims;
        udata.nbytes      = 0;
        udata.filter_mask = 0;
        udata.chunk_addr  = HADDR_UNDEF;
        udata.found       = false;

        
        if ((dset->shared->layout.storage.u.chunk.ops->iterate)(&idx_info, H5D__get_chunk_info_by_coord_cb,
                                                                &udata) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                        "unable to retrieve information of the chunk by its scaled coordinates");

        
        if (udata.found) {
            if (filter_mask)
                *filter_mask = udata.filter_mask;
            if (addr)
                *addr = udata.chunk_addr + H5F_BASE_ADDR(dset->oloc.file);
            if (size)
                *size = udata.nbytes;
        }
    }

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static int
H5D__chunk_iter_cb(const H5D_chunk_rec_t *chunk_rec, void *udata)
{
    const H5D_chunk_iter_ud_t *data  = (H5D_chunk_iter_ud_t *)udata;
    const H5O_layout_chunk_t  *chunk = data->chunk;
    hsize_t                    offset[H5O_LAYOUT_NDIMS];
    int                        ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE_NOERR

    
    for (unsigned i = 0; i < chunk->ndims; i++)
        offset[i] = chunk_rec->scaled[i] * chunk->dim[i];

    
    H5_BEFORE_USER_CB_NOERR(FAIL)
        {
            ret_value = (data->op)(offset, (unsigned)chunk_rec->filter_mask,
                                   data->base_addr + chunk_rec->chunk_addr, chunk_rec->nbytes, data->op_data);
        }
    H5_AFTER_USER_CB_NOERR(FAIL)

    
    if (ret_value < 0)
        HERROR(H5E_DATASET, H5E_CANTNEXT, "iteration operator failed");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__chunk_iter(H5D_t *dset, H5D_chunk_iter_op_t op, void *op_data)
{
    const H5D_rdcc_t  *rdcc   = NULL;       
    H5O_layout_t      *layout = NULL;       
    H5D_rdcc_ent_t    *ent;                 
    H5D_chk_idx_info_t idx_info;            
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(dset);
    assert(dset->shared);

    
    layout = &(dset->shared->layout);
    rdcc   = &(dset->shared->cache.chunk);
    assert(layout);
    assert(rdcc);
    assert(H5D_CHUNKED == layout->type);

    
    for (ent = rdcc->head; ent; ent = ent->next)
        
        if (H5D__chunk_flush_entry(dset, ent, false) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "cannot flush indexed storage buffer");

    
    idx_info.f      = dset->oloc.file;
    idx_info.pline  = &dset->shared->dcpl_cache.pline;
    idx_info.layout = layout;

    
    if (H5_addr_defined(idx_info.layout->storage.u.chunk.idx_addr)) {
        H5D_chunk_iter_ud_t ud;

        
        ud.op        = op;
        ud.op_data   = op_data;
        ud.chunk     = &dset->shared->layout.u.chunk;
        ud.base_addr = H5F_BASE_ADDR(dset->oloc.file);

        
        if ((ret_value = (layout->storage.u.chunk.ops->iterate)(&idx_info, H5D__chunk_iter_cb, &ud)) < 0)
            HERROR(H5E_DATASET, H5E_CANTNEXT, "chunk iteration failed");
    }

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D__chunk_get_offset_copy(const H5D_t *dset, const hsize_t *offset, hsize_t *offset_copy)
{
    unsigned u;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(dset);
    assert(offset);
    assert(offset_copy);

    
    memset(offset_copy, 0, H5O_LAYOUT_NDIMS * sizeof(hsize_t));

    for (u = 0; u < dset->shared->ndims; u++) {
        
        if (offset[u] > dset->shared->curr_dims[u])
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "offset exceeds dimensions of dataset");

        
        if (offset[u] % dset->shared->layout.u.chunk.dim[u])
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "offset doesn't fall on chunks's boundary");

        offset_copy[u] = offset[u];
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
