/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5module.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5CXprivate.h" 
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5FSprivate.h" 
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  
#include "H5PLprivate.h" 
#include "H5SLprivate.h" 
#include "H5Tprivate.h"  
#include "H5TSprivate.h" 

static FILE r_stdout_sentinel;
static FILE r_stderr_sentinel;

H5_DLL FILE *Rstdout = &r_stdout_sentinel;
H5_DLL FILE *Rstderr = &r_stderr_sentinel;

H5_DLL int Rfprintf(FILE *stream, const char *format, ...) {
    int ret = 0;
    va_list ap;
    va_start(ap, format);

    if (stream == Rstdout) {
        
        Rvprintf(format, ap);
    }
    else if (stream == Rstderr) {
        
        REvprintf(format, ap);
    }
    else {
        
        ret = vfprintf(stream, format, ap);
    }

    va_end(ap);
    return ret;
}

H5_DLL int Rfputs(const char *s, FILE *stream) {
    
    int ret = 0; 

    if (stream == Rstdout) {
        
        Rprintf("%s", s);
    }
    else if (stream == Rstderr) {
        
        REprintf("%s", s);
    }
    else {
        
        ret = Rfputs(s, stream);
    }

    return ret;
}

H5_DLL void Rabort(void) {
    Rf_error("HDF5 library called Rabort()");
}

H5_DLL void Rexit(int status) {
    Rf_error("HDF5 library called Rexit() with status %d", status);
}

typedef struct H5_atclose_node_t {
    H5_atclose_func_t         func; 
    void                     *ctx;  
    struct H5_atclose_node_t *next; 
} H5_atclose_node_t;

static void H5__debug_mask(const char *);
#ifdef H5_HAVE_PARALLEL
static int H5__mpi_delete_cb(MPI_Comm comm, int keyval, void *attr_val, int *flag);
#endif 
static herr_t H5_check_version(unsigned majnum, unsigned minnum, unsigned relnum);

bool H5_PKG_INIT_VAR = false;

const char H5libhdf5_settings[] = 
    "HDF5 Library Version: 2.0.0\n"
    "   configured on: R-Package-Build\n"
    "   configured by: R-User\n"
    "      build mode: C-Only\n"
    "support: none";

const char H5build_settings[] = 
    "HDF5 Library Version: 2.0.0\n"
    "   configured on: R-Package-Build\n";

static const unsigned VERS_MINOR_EXCEPTIONS[] = {999};

static const unsigned VERS_MINOR_EXCEPTIONS_SIZE = 1;

bool H5_libinit_g = false; 
bool H5_libterm_g = false; 

char        H5_lib_vers_info_g[] = H5_VERS_INFO;
static bool H5_dont_atexit_g     = false;
H5_debug_t  H5_debug_g; 

static H5_atclose_node_t *H5_atclose_head = NULL;

H5FL_DEFINE_STATIC(H5_atclose_node_t);

herr_t
H5__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (!H5_INIT_GLOBAL && !H5_TERM_GLOBAL)
        if (H5_init_library() < 0)
            HGOTO_ERROR(H5E_LIB, H5E_CANTINIT, FAIL, "unable to initialize library");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5_init_library(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5_INIT_GLOBAL || H5_TERM_GLOBAL)
        HGOTO_DONE(SUCCEED);

    
    
    H5_check_version(H5_VERS_MAJOR, H5_VERS_MINOR, H5_VERS_RELEASE);

    
    H5_INIT_GLOBAL = true;

    
    HDcompile_assert(sizeof(size_t) == sizeof(ssize_t));

#ifdef H5_HAVE_PARALLEL
    {
        int mpi_initialized;
        int mpi_finalized;
        int mpi_code;

        MPI_Initialized(&mpi_initialized);
        MPI_Finalized(&mpi_finalized);

        
        if (mpi_initialized && !mpi_finalized) {
            int key_val;

            if (MPI_SUCCESS != (mpi_code = MPI_Comm_create_keyval(
                                    MPI_COMM_NULL_COPY_FN, (MPI_Comm_delete_attr_function *)H5__mpi_delete_cb,
                                    &key_val, NULL)))
                HMPI_GOTO_ERROR(FAIL, "MPI_Comm_create_keyval failed", mpi_code)

            if (MPI_SUCCESS != (mpi_code = MPI_Comm_set_attr(MPI_COMM_SELF, key_val, NULL)))
                HMPI_GOTO_ERROR(FAIL, "MPI_Comm_set_attr failed", mpi_code)

            if (MPI_SUCCESS != (mpi_code = MPI_Comm_free_keyval(&key_val)))
                HMPI_GOTO_ERROR(FAIL, "MPI_Comm_free_keyval failed", mpi_code)
        }
    }
#endif 

    
    memset(&H5_debug_g, 0, sizeof H5_debug_g);
    H5_debug_g.pkg[H5_PKG_A].name  = "a";
    H5_debug_g.pkg[H5_PKG_AC].name = "ac";
    H5_debug_g.pkg[H5_PKG_B].name  = "b";
    H5_debug_g.pkg[H5_PKG_D].name  = "d";
    H5_debug_g.pkg[H5_PKG_E].name  = "e";
    H5_debug_g.pkg[H5_PKG_F].name  = "f";
    H5_debug_g.pkg[H5_PKG_G].name  = "g";
    H5_debug_g.pkg[H5_PKG_HG].name = "hg";
    H5_debug_g.pkg[H5_PKG_HL].name = "hl";
    H5_debug_g.pkg[H5_PKG_I].name  = "i";
    H5_debug_g.pkg[H5_PKG_M].name  = "m";
    H5_debug_g.pkg[H5_PKG_MF].name = "mf";
    H5_debug_g.pkg[H5_PKG_MM].name = "mm";
    H5_debug_g.pkg[H5_PKG_O].name  = "o";
    H5_debug_g.pkg[H5_PKG_P].name  = "p";
    H5_debug_g.pkg[H5_PKG_S].name  = "s";
    H5_debug_g.pkg[H5_PKG_T].name  = "t";
    H5_debug_g.pkg[H5_PKG_V].name  = "v";
    H5_debug_g.pkg[H5_PKG_VL].name = "vl";
    H5_debug_g.pkg[H5_PKG_Z].name  = "z";

    
    if (!H5_dont_atexit_g) {

#ifdef H5_HAVE_THREADSAFE_API
        
        (void)atexit(H5TS_term_package);
#endif 

        
        (void)atexit(H5_term_library);

        H5_dont_atexit_g = true;
    } 

    
    if (H5E_init() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize error interface");
    if (H5FD_init() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize VFL interface");
    if (H5VL_init_phase1() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize vol interface");
    if (H5P_init_phase1() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize property list interface");
    if (H5L_init() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize link interface");
    if (H5O_init() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize object interface");
    if (H5FS_init() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize FS interface");
    if (H5S_init() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize dataspace interface");
    if (H5T_init() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize datatype interface");

    
    if (H5P_init_phase2() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize property list interface");
    if (H5VL_init_phase2() < 0)
        HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "unable to initialize vol interface");

    
    H5__debug_mask("-all");
    H5__debug_mask(getenv("HDF5_DEBUG"));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5_term_library(void)
{
    int         pending, ntries = 0, n;
    size_t      at = 0;
    char        loop[1024];
    H5E_auto2_t func;
    H5CX_node_t api_ctx = {{0}, NULL}; 

    
    H5_API_SETUP_PUBLIC_API_VARS
    H5_API_LOCK

    
    if (!H5_INIT_GLOBAL)
        goto done;

    
    H5_TERM_GLOBAL = true;

    
    H5CX_push(&api_ctx);

    
    (void)H5E_get_default_auto_func(&func);

    
    if (H5_atclose_head) {
        H5_atclose_node_t *curr_atclose; 

        
        curr_atclose = H5_atclose_head;
        while (curr_atclose) {
            H5_atclose_node_t *tmp_atclose; 

            
            H5_BEFORE_USER_CB_NOCHECK
                {
                    
                    (*curr_atclose->func)(curr_atclose->ctx);
                }
            H5_AFTER_USER_CB_NOCHECK

            
            tmp_atclose  = curr_atclose;
            curr_atclose = curr_atclose->next;
            H5FL_FREE(H5_atclose_node_t, tmp_atclose);
        } 

        
        H5_atclose_head = NULL;
    } 

    
#define DOWN(F)                                                                                              \
    (((n = H5##F##_term_package()) && (at + 8) < sizeof loop)                                                \
         ? (snprintf(loop + at, sizeof(loop) - at, "%s%s", (at ? "," : ""), #F), at += strlen(loop + at), n) \
         : ((n > 0 && (at + 5) < sizeof loop) ? (snprintf(loop + at, sizeof(loop) - at, "..."), at += strlen(loop + at), n) \
                                              : n))

    do {
        pending = 0;

        

        
        pending += DOWN(ES);

        
        if (pending == 0) {
            
            pending += DOWN(L);

            
            pending += DOWN(A_top);
            pending += DOWN(D_top);
            pending += DOWN(G_top);
            pending += DOWN(M_top);
            pending += DOWN(S_top);
            pending += DOWN(T_top);
        } 

        
        if (pending == 0)
            pending += DOWN(F);

        
        if (pending == 0)
            pending += DOWN(P);

        
        if (pending == 0) {
            
            pending += DOWN(A);
            pending += DOWN(D);
            pending += DOWN(G);
            pending += DOWN(M);
            pending += DOWN(S);
            pending += DOWN(T);
        } 

        
        if (pending == 0) {
            pending += DOWN(AC);
            
            pending += DOWN(Z);
            pending += DOWN(FD);
            pending += DOWN(VL);
            
            if (pending == 0)
                pending += DOWN(PL);
            
            if (pending == 0)
                pending += DOWN(E);
            
            if (pending == 0)
                pending += DOWN(I);
            
            if (pending == 0)
                pending += DOWN(SL);
            
            if (pending == 0)
                pending += DOWN(FL);
            
            if (pending == 0)
                pending += DOWN(CX);
        } 
    } while (pending && ntries++ < 100);

    if (pending) {
        
        if (func) {
            Rfprintf(Rstderr, "HDF5: infinite loop closing library\n");
            Rfprintf(Rstderr, "      %s\n", loop);
#ifndef NDEBUG
            Rabort();
#endif
        }
    }

    
    while (H5_debug_g.open_stream) {
        H5_debug_open_stream_t *tmp_open_stream;

        tmp_open_stream = H5_debug_g.open_stream;
        (void)fclose(H5_debug_g.open_stream->stream);
        H5_debug_g.open_stream = H5_debug_g.open_stream->next;
        (void)H5MM_free(tmp_open_stream);
    } 

    
    H5_TERM_GLOBAL = false;

    
    H5_INIT_GLOBAL = false;

    

done:
    
    H5_API_UNLOCK

    return;
} 

herr_t
H5dont_atexit(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API_NOINIT_NOERR

    if (H5_dont_atexit_g)
        ret_value = FAIL;
    else
        H5_dont_atexit_g = true;

    FUNC_LEAVE_API_NOERR(ret_value)
} 

herr_t
H5garbage_collect(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (H5FL_garbage_coll() < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "can't garbage collect objects");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5set_free_list_limits(int reg_global_lim, int reg_list_lim, int arr_global_lim, int arr_list_lim,
                       int blk_global_lim, int blk_list_lim)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (H5FL_set_free_list_limits(reg_global_lim, reg_list_lim, arr_global_lim, arr_list_lim, blk_global_lim,
                                  blk_list_lim, blk_global_lim, blk_list_lim) < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTSET, FAIL, "can't set garbage collection limits");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5get_free_list_sizes(size_t *reg_size , size_t *arr_size , size_t *blk_size ,
                      size_t *fac_size )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5FL_get_free_list_sizes(reg_size, arr_size, blk_size, fac_size) < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "can't get garbage collection sizes");

done:
    FUNC_LEAVE_API(ret_value)
} 

static void
H5__debug_mask(const char *s)
{
    FILE  *stream = Rstderr;
    char   pkg_name[32], *rest;
    size_t i;
    bool   clear;

    while (s && *s) {

        if (isalpha(*s) || '-' == *s || '+' == *s) {

            
            if ('-' == *s) {
                clear = true;
                s++;
            }
            else if ('+' == *s) {
                clear = false;
                s++;
            }
            else {
                clear = false;
            } 

            
            for (i = 0; isalpha(*s); i++, s++)
                if (i < sizeof pkg_name)
                    pkg_name[i] = *s;
            pkg_name[MIN(sizeof(pkg_name) - 1, i)] = '\0';

            
            if (!strcmp(pkg_name, "trace")) {
                H5_debug_g.trace = clear ? NULL : stream;
            }
            else if (!strcmp(pkg_name, "ttop")) {
                H5_debug_g.trace = stream;
                H5_debug_g.ttop  = (bool)!clear;
            }
            else if (!strcmp(pkg_name, "ttimes")) {
                H5_debug_g.trace  = stream;
                H5_debug_g.ttimes = (bool)!clear;
            }
            else if (!strcmp(pkg_name, "all")) {
                for (i = 0; i < (size_t)H5_NPKGS; i++)
                    H5_debug_g.pkg[i].stream = clear ? NULL : stream;
            }
            else {
                for (i = 0; i < (size_t)H5_NPKGS; i++) {
                    if (!strcmp(H5_debug_g.pkg[i].name, pkg_name)) {
                        H5_debug_g.pkg[i].stream = clear ? NULL : stream;
                        break;
                    } 
                }     
                if (i >= (size_t)H5_NPKGS)
                    Rfprintf(Rstderr, "HDF5_DEBUG: ignored %s\n", pkg_name);
            } 
        }
        else if (isdigit(*s)) {
            int                     fd = (int)strtol(s, &rest, 0);
            H5_debug_open_stream_t *open_stream;

            if ((stream = HDfdopen(fd, "w")) != NULL) {
                (void)HDsetvbuf(stream, NULL, _IOLBF, (size_t)0);

                if (NULL ==
                    (open_stream = (H5_debug_open_stream_t *)H5MM_malloc(sizeof(H5_debug_open_stream_t)))) {
                    (void)fclose(stream);
                    return;
                } 

                open_stream->stream    = stream;
                open_stream->next      = H5_debug_g.open_stream;
                H5_debug_g.open_stream = open_stream;
            } 

            s = rest;
        }
        else {
            s++;
        } 
    }     
} 

#ifdef H5_HAVE_PARALLEL

static int
H5__mpi_delete_cb(MPI_Comm H5_ATTR_UNUSED comm, int H5_ATTR_UNUSED keyval, void H5_ATTR_UNUSED *attr_val,
                  int H5_ATTR_UNUSED *flag)
{
    H5_term_library();
    return MPI_SUCCESS;
}
#endif 

herr_t
H5get_libversion(unsigned *majnum , unsigned *minnum , unsigned *relnum )
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (majnum)
        *majnum = H5_VERS_MAJOR;
    if (minnum)
        *minnum = H5_VERS_MINOR;
    if (relnum)
        *relnum = H5_VERS_RELEASE;

done:
    FUNC_LEAVE_API(ret_value)
} 

#define VERSION_MISMATCH_WARNING                                                                             \
    "Warning! ***HDF5 library version mismatched error***\n"                                                 \
    "The HDF5 header files used to compile this application do not match\n"                                  \
    "the version used by the HDF5 library to which this application is linked.\n"                            \
    "Data corruption or segmentation faults may occur if the application continues.\n"                       \
    "This can happen when an application was compiled by one version of HDF5 but\n"                          \
    "linked with a different version of static or shared HDF5 library.\n"                                    \
    "You should recompile the application or check your shared library related\n"                            \
    "settings such as 'LD_LIBRARY_PATH'.\n"
#define MINOR_VERSION_MISMATCH_WARNING                                                                       \
    "Warning! ***HDF5 library minor version mismatched error***\n"                                           \
    "The HDF5 header files used to compile this application are not compatible with\n"                       \
    "the version used by the HDF5 library to which this application is linked.\n"                            \
    "Data corruption or segmentation faults may occur if the application continues.\n"                       \
    "This can happen when an application was compiled by one version of HDF5 but\n"                          \
    "linked with an incompatible version of static or shared HDF5 library.\n"                                \
    "You should recompile the application or check your shared library related\n"                            \
    "settings such as 'LD_LIBRARY_PATH'.\n"
#define MINOR_VERSION_FORWARD_COMPATIBLE_WARNING                                                             \
    "Warning! ***HDF5 library minor version forward compatibility error***\n"                                \
    "The HDF5 header files used to compile this application are from a newer\n"                              \
    "version of the HDF5 library than the one to which this application is linked.\n"                        \
    "Data corruption or segmentation faults may occur if the application continues.\n"                       \
    "This can happen when an application was compiled by a newer version of HDF5 but\n"                      \
    "linked with an older version of static or shared HDF5 library.\n"                                       \
    "You should recompile the application or check your shared library related\n"                            \
    "settings such as 'LD_LIBRARY_PATH'.\n"

static herr_t
H5_check_version(unsigned majnum, unsigned minnum, unsigned relnum)
{
    char                lib_str[256];
    char                substr[]                 = H5_VERS_SUBRELEASE;
    static bool         checked                  = false; 
    static unsigned int disable_version_check    = 0;     
    static const char  *version_mismatch_warning = VERSION_MISMATCH_WARNING;
    static const char  *minor_version_mismatch_warning           = MINOR_VERSION_MISMATCH_WARNING;
    static const char  *minor_version_forward_compatible_warning = MINOR_VERSION_FORWARD_COMPATIBLE_WARNING;
    herr_t              ret_value                                = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    if (checked)
        HGOTO_DONE(SUCCEED);

    {
        const char *s; 

        
        s = getenv("HDF5_DISABLE_VERSION_CHECK");

        if (s && isdigit(*s))
            disable_version_check = (unsigned int)strtol(s, NULL, 0);
    }

    
    if (H5_VERS_MAJOR != majnum) {
        switch (disable_version_check) {
            case 0:
                Rfprintf(Rstderr, "%s%s", version_mismatch_warning,
                        "You can, at your own risk, disable this warning by setting the environment\n"
                        "variable 'HDF5_DISABLE_VERSION_CHECK' to a value of '1'.\n"
                        "Setting it to 2 or higher will suppress the warning messages totally.\n");
                
                Rfprintf(Rstderr, "Headers are %u.%u.%u, library is %u.%u.%u\n", majnum, minnum, relnum,
                        (unsigned)H5_VERS_MAJOR, (unsigned)H5_VERS_MINOR, (unsigned)H5_VERS_RELEASE);
                
                Rfprintf(Rstderr, "%s", H5build_settings);

                
                Rfputs("Bye...\n", Rstderr);
                Rabort();
                break;
            case 1:
                
                
                Rfprintf(Rstderr,
                        "%s'HDF5_DISABLE_VERSION_CHECK' "
                        "environment variable is set to %d, application will\n"
                        "continue at your own risk.\n",
                        version_mismatch_warning, disable_version_check);
                
                Rfprintf(Rstderr, "Headers are %u.%u.%u, library is %u.%u.%u\n", majnum, minnum, relnum,
                        (unsigned)H5_VERS_MAJOR, (unsigned)H5_VERS_MINOR, (unsigned)H5_VERS_RELEASE);
                
                Rfprintf(Rstderr, "%s", H5build_settings);
                break;
            default:
                
                break;
        } 

    } 

    
    
    if (H5_VERS_MINOR != minnum) {
        for (unsigned i = 0; i < VERS_MINOR_EXCEPTIONS_SIZE; i++) {
            
            if (VERS_MINOR_EXCEPTIONS[i] == minnum || VERS_MINOR_EXCEPTIONS[i] == H5_VERS_MINOR) {
                switch (disable_version_check) {
                    case 0:
                        Rfprintf(Rstderr, "%s%s", minor_version_mismatch_warning,
                                "You can, at your own risk, disable this warning by setting the environment\n"
                                "variable 'HDF5_DISABLE_VERSION_CHECK' to a value of '1'.\n"
                                "Setting it to 2 or higher will suppress the warning messages totally.\n");
                        
                        Rfprintf(Rstderr, "Headers are %u.%u.%u, library is %u.%u.%u\n", majnum, minnum, relnum,
                                (unsigned)H5_VERS_MAJOR, (unsigned)H5_VERS_MINOR, (unsigned)H5_VERS_RELEASE);

                        
                        Rfputs("Bye...\n", Rstderr);
                        Rabort();
                        break;
                    case 1:
                        
                        
                        Rfprintf(Rstderr,
                                "%s'HDF5_DISABLE_VERSION_CHECK' "
                                "environment variable is set to %d, application will\n"
                                "continue at your own risk.\n",
                                minor_version_mismatch_warning, disable_version_check);
                        
                        Rfprintf(Rstderr, "Headers are %u.%u.%u, library is %u.%u.%u\n", majnum, minnum, relnum,
                                (unsigned)H5_VERS_MAJOR, (unsigned)H5_VERS_MINOR, (unsigned)H5_VERS_RELEASE);
                        break;
                    default:
                        
                        break;
                } 

            } 

        } 

        
        if (minnum > H5_VERS_MINOR) {
            switch (disable_version_check) {
                case 0:
                    Rfprintf(Rstderr, "%s%s", minor_version_forward_compatible_warning,
                            "You can, at your own risk, disable this warning by setting the environment\n"
                            "variable 'HDF5_DISABLE_VERSION_CHECK' to a value of '1'.\n"
                            "Setting it to 2 or higher will suppress the warning messages totally.\n");
                    
                    Rfprintf(Rstderr, "Headers are %u.%u.%u, library is %u.%u.%u\n", majnum, minnum, relnum,
                            (unsigned)H5_VERS_MAJOR, (unsigned)H5_VERS_MINOR, (unsigned)H5_VERS_RELEASE);

                    
                    Rfputs("Bye...\n", Rstderr);
                    Rabort();
                    break;
                case 1:
                    
                    
                    Rfprintf(Rstderr,
                            "%s'HDF5_DISABLE_VERSION_CHECK' "
                            "environment variable is set to %d, application will\n"
                            "continue at your own risk.\n",
                            minor_version_forward_compatible_warning, disable_version_check);
                    
                    Rfprintf(Rstderr, "Headers are %u.%u.%u, library is %u.%u.%u\n", majnum, minnum, relnum,
                            (unsigned)H5_VERS_MAJOR, (unsigned)H5_VERS_MINOR, (unsigned)H5_VERS_RELEASE);
                    break;
                default:
                    
                    break;
            } 
        }

    } 

    
    checked = true;

    if (!disable_version_check) {
        
        snprintf(lib_str, sizeof(lib_str), "HDF5 library version: %d.%d.%d%s%s", H5_VERS_MAJOR, H5_VERS_MINOR,
                 H5_VERS_RELEASE, (*substr ? "-" : ""), substr);

        if (strcmp(lib_str, H5_lib_vers_info_g) != 0) {
            Rfputs("Warning!  Library version information error.\n"
                  "The HDF5 library version information are not "
                  "consistent in its source code.\nThis is NOT a fatal error "
                  "but should be corrected.  Setting the environment\n"
                  "variable 'HDF5_DISABLE_VERSION_CHECK' to a value of 1 "
                  "will suppress\nthis warning.\n",
                  Rstderr);
            Rfprintf(Rstderr,
                    "Library version information are:\n"
                    "H5_VERS_MAJOR=%d, H5_VERS_MINOR=%d, H5_VERS_RELEASE=%d, "
                    "H5_VERS_SUBRELEASE=%s,\nH5_VERS_INFO=%s\n",
                    H5_VERS_MAJOR, H5_VERS_MINOR, H5_VERS_RELEASE, H5_VERS_SUBRELEASE, H5_VERS_INFO);
        } 
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5check_version(unsigned majnum, unsigned minnum, unsigned relnum)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API_NOINIT_NOERR

    
    
    H5_check_version(majnum, minnum, relnum);

    FUNC_LEAVE_API_NOERR(ret_value)
} 

herr_t
H5open(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API_NOPUSH(FAIL)

    

done:
    FUNC_LEAVE_API_NOPUSH(ret_value)
} 

herr_t
H5atclose(H5_atclose_func_t func, void *ctx)
{
    H5_atclose_node_t *new_atclose;         
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == func)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL func pointer");

    
    if (NULL == (new_atclose = H5FL_MALLOC(H5_atclose_node_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't allocate 'atclose' node");

    
    new_atclose->func = func;
    new_atclose->ctx  = ctx;

    
    new_atclose->next = H5_atclose_head;
    H5_atclose_head   = new_atclose;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5close(void)
{
    
    FUNC_ENTER_API_NAMECHECK_ONLY

    H5_term_library();

    FUNC_LEAVE_API_NAMECHECK_ONLY(SUCCEED)
} 

void *H5_ATTR_MALLOC
H5allocate_memory(size_t size, bool clear)
{
    void *ret_value = NULL;

    FUNC_ENTER_API_NOINIT

    if (0 == size)
        HGOTO_DONE(NULL);

    if (clear)
        ret_value = H5MM_calloc(size);
    else
        ret_value = H5MM_malloc(size);

done:
    FUNC_LEAVE_API_NOINIT(ret_value)
} 

void *
H5resize_memory(void *mem, size_t size)
{
    void *ret_value = NULL;

    FUNC_ENTER_API_NOINIT

    ret_value = H5MM_realloc(mem, size);

    FUNC_LEAVE_API_NOINIT(ret_value)
} 

herr_t
H5free_memory(void *mem)
{
    FUNC_ENTER_API_NOINIT

    
    H5MM_xfree(mem);

    FUNC_LEAVE_API_NOINIT(SUCCEED)
} 

herr_t
H5is_library_threadsafe(bool *is_ts )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API_NOINIT

    if (is_ts) {
#ifdef H5_HAVE_THREADSAFE_API
        *is_ts = true;
#else  
        *is_ts = false;
#endif 
    }
    else
        ret_value = FAIL;

    FUNC_LEAVE_API_NOINIT(ret_value)
} 

herr_t
H5is_library_terminating(bool *is_terminating )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API_NOINIT

    assert(is_terminating);

    if (is_terminating)
        *is_terminating = H5_TERM_GLOBAL;
    else
        ret_value = FAIL;

    FUNC_LEAVE_API_NOINIT(ret_value)
} 

herr_t
H5_user_cb_prepare(H5_user_cb_state_t *state)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5E_user_cb_prepare(&state->h5e_state) < 0)
        HGOTO_ERROR(H5E_LIB, H5E_CANTSET, FAIL, "unable to prepare H5E package for user callback");

#ifdef H5_HAVE_CONCURRENCY
    
    if (H5TS_user_cb_prepare() < 0)
        HGOTO_ERROR(H5E_LIB, H5E_CANTSET, FAIL, "unable to prepare H5TS package for user callback");
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5_user_cb_restore(const H5_user_cb_state_t *state)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5E_user_cb_restore(&state->h5e_state) < 0)
        HGOTO_ERROR(H5E_LIB, H5E_CANTRESTORE, FAIL, "unable to restore H5E package after user callback");

#ifdef H5_HAVE_CONCURRENCY
    
    if (H5TS_user_cb_restore() < 0)
        HGOTO_ERROR(H5E_LIB, H5E_CANTRESTORE, FAIL, "unable to restore H5TS package after user callback");
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
