##' Boruvka-Kruskal algorithm for finding the spanning tree of minimum
##'    weight of a weighted (connected) graph.
##'
##' The Boruvka-Kruskal algorithm in a weighted, connected graph of
##'    \eqn{n} vertices consists of growing an initially empty forest
##'    by adding to it the least weight edge not yet added, while
##'    being careful to not close any cycles.  The algorithm ends when
##'    the size of the forest is \eqn{n-1}; then it becomes connected
##'    and, being acyclic, the forest is a tree.
##'
##' This algorithm is exact, that is, it always finds a minimum
##'    solution.  Please note that the minimum may not be unique, and
##'    different solutions with the same weight can be found by
##'    changing the edge ordering when some weights coincide.
##'
##' This routine can plot the algorithm stepwise for illustrative
##'    purposes.  To do that, set the parameter "plot.steps" to TRUE
##'    and adjust the "pause" parameter to some convenient value if
##'    desired (its default value is 2).  The "color" parameter, a
##'    list of R colors, is used to color the vertices of the graph:
##'    The first color marks the vertices outside the growing forest
##'    and the remaining colors are used for the different components
##'    of the forest, which coalesce in a single component (the
##'    minimum spanning tree) as the algorithm progresses.  The
##'    default value of the "color" parameter is c("gray60",
##'    rainbow(12)), which might not be enough for large graphs.
##'    Finally, by setting the "pdf" parameter to TRUE, this routine
##'    can also output the individual plots as a bunch of pdf files
##'    which can be included in a LaTeX animation.
##' 
##' @title Minimum spanning tree (Boruvka-Kruskal algorithm)
##' @param g A graph in igraph format.
##' @param d A list of weights assigned to the edges of g in edgeid
##'   order.
##' @param plot.steps Boolean indicating whether the routine should
##'   plot the intermediate steps of the algorithm to see the growing
##'   of the forest into the minimum spanning tree.  It defaults to
##'   FALSE.
##' @param color A set of colors to see the different components of
##'   the forest as it grows, see details below.  It is only used when
##'   plot.steps is set to TRUE.
##' @param pause Time interval (in seconds) between successive steps
##'   of the algorithm.  It is only used when plot.steps is set to
##'   TRUE and it defaults to 2.
##' @param pdf Boolean indicating whether the routine should eject pdf
##'   plots of intermediate steps of the algorithm, see details below.
##'   It is only used when plot.steps is set to TRUE and it defaults
##'   to FALSE.
##' @param ... Further parameters to pass to the plotting routine,
##'   especially the layout parameter.
##' @return A list containing the minimum spanning tree in igraph
##'   format (components $tree and $arbol) and the corresponding
##'   minimum weight (components $weight and $peso).
##' @importFrom grDevices dev.off
##' @importFrom grDevices rainbow
##' @importFrom graphics mtext
##' @author Cesar Asensio
##' @examples
##' library(igraph)
##' g <- make_graph("Frucht")
##' set.seed(1)
##' d <- round(runif(gsize(g), min = 1, max = 10))  # Random weights
##' T <- krus(g, d)
##' T$weight   # = 47
##' ## To plot the result:
##' ## z <- layout_with_gem(g)
##' ## T <- krus(g, d, plot.steps = TRUE, layout = z)
##' 
##' @encoding UTF-8
##' @md 
##' @export 
krus <- function(g, d, plot.steps = FALSE,
                 color = c("gray60", rainbow(12)),
                 pause = 2, pdf = FALSE, ...) {
  n <- gorder(g)       # Number of vertices of "g"
  cv <- rep(0,n)       # Vertex colors starting at 0
  co <- 1              # First positive color
  T <- c()             # Empty forest (it will grow into the minimum spanning tree)
  w <- 0               # Forest weight
  q <- gsize(g)        # Number of edges of "g"
  Eg <- as_edgelist(g) # Edges of "g" (matrix of size  nx2)
  doi <- sort(d, index = TRUE)  # We sort weights...
  Eg <- Eg[doi$ix,]             # ...and edges in the same way
  do <- doi$x          # Sorted weights
  k <- 1
  
  if (plot.steps) {    # Plot the initial graph if desired
    if (pdf) {         # Plot a pdf version to build a pdfLaTeX animation
      pdf(file = paste(c("kruskal-example-", k, ".pdf"), collapse = ""))
      k <- k+1
    }
    plot(g, edge.label = d, vertex.color = color[1], ...)
    ## We should close the pdf device or pause to see the result stepwise
    if (pdf) { dev.off() } else { Sys.sleep(pause) }
  }

  for (i in 1:q) {  # We explore ordered edges:
    e <- Eg[i,]     # Edge of minimum weigth of those not added to the forest
    c1 <- cv[e[1]]  # Color of vertices joined by "e"
    c2 <- cv[e[2]]  #  " "
    if (c1 == 0 & c2 == 0) { # New component of the forest
      cv[e] <- co            # We use a positive color...
      co <- co+1             # ...and prepare the next color
      T <- c(T,e)            # We grow the forest...
      w <- w + do[i]         # ...and its weight
    } else if (c1 == 0 & c2 > 0) { # New vertex e[1]
      cv[e[1]] <- c2         # We add it to the component...
      T <- c(T,e)            # ...we grow the forest...
      w <- w + do[i]         # ...and its weight            
    } else if (c1 > 0 & c2 == 0) { # New vertex e[2]
      cv[e[2]] <- c1         # We add it to the component...
      T <- c(T,e)            # ...we grow the forest...
      w <- w + do[i]         # ...and its weight
    } else if (c1 > 0 & c2 > 0 & c1 != c2) { # We connect components
      cv[cv==c1] <- c2       # We color c1 with c2
      T <- c(T,e)            # ...we grow the forest...
      w <- w + do[i]         # ...and its weight  
    } else { next } # We skip this case (otherwise, we would close a cycle)

    Tg <- make_graph(T, n = n, directed = FALSE) # Forest as a graph

    if (plot.steps) { # Plot each step of the algorithm if desired
      if (pdf) {      # Plot a pdf version to build a pdfLaTeX animation
        pdf(file = paste(c("kruskal-example-", k, ".pdf"), collapse = ""))
        k <- k+1
      }
      plot(g, edge.label = d, ...)
      plot(Tg, add = TRUE, edge.color = "cyan4", edge.width = 3,
           vertex.color = color[cv+1],...)
      mtext(paste("Peso =", w), 1)
      ## We should close the pdf device or pause to see the result stepwise
      if (pdf) { dev.off() } else { Sys.sleep(pause) }
    }
    
    if (length(T) == n-1) { break } # When |T|=n-1, T will be a tree
  }

  list(arbol = Tg, peso = w, tree = Tg, weight = w)
}

##' Jarnik-Prim algorithm for finding the spanning tree of minimum
##'    weight of a weighted (connected) graph.
##'
##' The Jarnik-Prim algorithm in a weighted, connected graph of
##'    \eqn{n} vertices consists of growing an initially empty tree by
##'    adding to it the least weight edge in the cut set of the
##'    already added vertices, starting with a single vertex.
##'    Choosing each edge from this cutset guarantees that the added
##'    edges will form indeed a tree, that is, no cycle can be formed
##'    by this choice.  The algorithm ends when the tree contains all
##'    \eqn{n} vertices.
##'
##' This algorithm is exact, that is, it always finds a minimum
##'    solution.  Please note that the minimum may not be unique, and
##'    different solutions with the same weight can be found by
##'    changing the initial vertex or the edge choice when some
##'    weights coincide.
##'
##' This routine can plot the algorithm stepwise for illustrative
##'    purposes.  To do that, set the parameter "plot.steps" to TRUE
##'    and adjust the "pause" parameter to some convenient value if
##'    desired (its default value is 2).  The "color" parameter, a
##'    list of five R colors, is used to color the vertices and edges
##'    of the graph:  The first color marks the vertices outside the
##'    growing tree and the second those inside it.  The third color
##'    marks the edges incident to two vertices outside the tree, the
##'    fourth color marks the edges in the cut set of the added
##'    vertices and the fifth color marks the edges incident to two
##'    vertices inside the tree.  The edges of the minimum weight
##'    spanning tree are colored with a fixed style and color.  The
##'    default value of the "color" parameter is c("pink", "red",
##'    "gray50", "violet", "gray80").  Finally, by setting the "pdf"
##'    parameter to TRUE, this routine can also output the individual
##'    plots as a bunch of pdf files which can be included in a LaTeX
##'    animation.
##' 
##' @title Minimum spanning tree (Jarnik-Prim algorithm)
##' @param g A graph in igraph format.
##' @param d A list of weights assigned to the edges of g in edgeid
##'   order.
##' @param v Initial vertex to use in the first step of the algorithm.
##'   It defaults to 1.
##' @param plot.steps Boolean indicating whether the routine should
##'   plot the intermediate steps of the algorithm to see the growing
##'   of the tree into the minimum spanning tree.  It defaults to
##'   FALSE.
##' @param color A set of colors to mark the vertices and edges added
##'   to the tree as it grows, and to visualize the cut of the added
##'   vertices set, see details below.  It is only used when
##'   plot.steps is set to TRUE.
##' @param pause Time interval (in seconds) between successive steps
##'   of the algorithm.  It is only used when plot.steps is set to
##'   TRUE and it defaults to 2.
##' @param pdf Boolean indicating whether the routine should eject pdf
##'   plots of intermediate steps of the algorithm, see details below.
##'   It is only used when plot.steps is set to TRUE and it defaults
##'   to FALSE.
##' @param ... Further parameters to pass to the plotting routine,
##'   especially the layout parameter.
##' @return A list containing the minimum spanning tree in igraph
##'   format (components $tree and $arbol) and the corresponding
##'   minimum weight (components $weight and $peso).
##' @importFrom grDevices dev.off
##' @importFrom graphics mtext
##' @author Cesar Asensio
##' @examples
##' library(igraph)
##' g <- make_graph("Frucht")
##' set.seed(1)
##' d <- round(runif(gsize(g), min = 1, max = 10))  # Random weights
##' T <- prim(g, d)
##' T$weight   # = 47
##' ## To plot the result:
##' ## z <- layout_with_gem(g)
##' ## T <- prim(g, d, plot.steps = TRUE, layout = z)
##'
##' @encoding UTF-8
##' @md 
##' @export 
prim <- function(g, d, v = 1, plot.steps = FALSE,
                 color = c("pink", "red", "gray50", "violet", "gray80"),
                 pause = 2, pdf = FALSE, ...) {
  n <- gorder(g)        # Number of vertices of "g"
  cv <- rep(0,n)        # Vertex colors starting at 0
  T <- c()              # Empty tree (it will grow into the minimum spanning tree)
  w <- 0                # Tree weight
  q <- gsize(g)         # Number of edges of "g"
  Eg <- as_edgelist(g)  # Edges of "g" (matrix of size  nx2)
  cE <- rbind(1:q, rep(0,q))        # Edge colors
  U <- c(v)             # Vertices inside the tree (only v initially)
  u <- v                # Vertex recently added to the tree

  ## All vertices and edges are colored with the null color (first and
  ## third of the list "color", respectively).
  v.co <- color[1:2]
  v.cols <- rep(v.co[1], n)
  e.co <- color[3:5]
  e.cols <- rep(e.co[1], q)
  e.wids <- rep(1,q)         # Edge widths, all set to 1 initially
  e.stys <- rep("solid", q)  # Edge styles, all set to "solid" initially
  j <- 1
  
  if (plot.steps) {    # Plot the initial graph if desired
    if (pdf) {         # Plot a pdf version to build a pdfLaTeX animation
      pdf(file = paste(c("prim-example-", j, ".pdf"), collapse = ""))
      j <- j+1
    }
    plot(g, edge.label = d, vertex.color = color[1], ...)
    ## We should close the pdf device or pause to see the result stepwise
    if (pdf) { dev.off() } else { Sys.sleep(pause) }
  }

  while (length(U) < n) {          # When |U| = n the algorithm ends
    cutu <- Eg[,1]==u | Eg[,2]==u  # Cut of vertex "u"
    co <- cE[2,cutu]               # Colors of the previous cut
    cE[2,cutu] <- co + 1           # Increase the previous colors
    c1 <- cE[2,]==1                # U-cut: edges of color 1

    ## Update characteristics of U vertices and U-cut edges
    v.cols[U] <- v.co[2]
    e.cols[c1] <- e.co[2]
    e.wids[c1] <- 2
    e.stys[c1] <- "dashed"
    
    c2 <- cE[2,]==2                # U-inner edges get color 2
    e.cols[c2] <- e.co[3]

    Tg <- make_graph(T, n = n, directed = FALSE) # Tree as a graph

    if (plot.steps) { # Plot each step of the algorithm if desired
      if (pdf) {
        pdf(file = paste(c("prim-example-", j, ".pdf"), collapse = ""))
        j <- j+1
      }
      plot(g, edge.label = d, vertex.color = v.cols,
           edge.color = e.cols, edge.width = e.wids,
           edge.lty = e.stys, ...)
      plot(Tg, add = TRUE, edge.color = "cyan4", edge.width = 3,
           vertex.color = v.cols, ...)
      mtext(paste("Peso =",w), 1)
      ## We should close the pdf device or pause to see the result stepwise
      if (pdf) { dev.off() } else { Sys.sleep(pause) }
    }

    k <- which.min(d[c1])          # Location of minimum weight in cut
    cE1 <- cE[1,c1]                # Locations of color 1 edges (U-cut) 
    e <- Eg[cE1[k],]               # Edge of minimum weight in U-cut
    T <- c(T,e)                    # We add the previous edge to the tree
    u <- e[!(e %in% U)]            # We find which vertex in e is outside U...
    U <- c(U,u)                    # ...and we add it to U
    w <- w + d[cE1[k]]             # Update weight
  }

  Tg <- make_graph(T, n = n, directed = FALSE) # Tree as a graph

  if (plot.steps) { # Plot the final result
    v.cols[U] <- v.co[2]
    e.cols[c1] <- e.co[2]
    c2 <- cE[2,]==2
    e.cols[c2] <- e.co[3]
    if (pdf) {
      pdf(file = paste(c("prim-example-", j, ".pdf"), collapse = ""))
    }
    plot(g, edge.label = d, vertex.color = v.cols,
         edge.color = e.cols, edge.width = e.wids,
         edge.lty = e.stys, ...)
    plot(Tg, add = TRUE, edge.color = "cyan4", edge.width = 3,
         vertex.color = v.cols, ...)
    mtext(paste("Peso =", w), 1)
    if (pdf) { dev.off() }
  }

  list(arbol = Tg, peso = w, tree = Tg, weight = w)
}

##' Dijkstra's algorithm finding the shortest paths from a root vertex
##' to the remaining vertices of a graph using a spanning tree
##'
##' An implementation of Dijkstra's algorithm.  Is is an exact
##'    algorithm to find the shortest path from a given (root) vertex
##'    to each vertex of a given weighted graph.  The shortest paths
##'    form a spanning tree, and the distances from the root to the
##'    remaining vertices are computed as sums of the weights of the
##'    edges in each path.  Please note that the minimum may not be
##'    unique, and different solutions can be found by changing the
##'    visiting order of neighbors when some distances coincide.
##' 
##' This routine can plot the algorithm stepwise for illustrative
##'    purposes.  To do that, set the parameter "plot.steps" to TRUE
##'    and adjust the "pause" parameter to some convenient value if
##'    desired (its default value is 2).  The "color" parameter, a
##'    list of five R colors, is used to color the vertices and vertex
##'    labels of the graph, as well as the edges of the spanning tree:
##'    The first (third) color marks the vertices (vertex labels)
##'    outside the growing tree and the second (fourth) those inside
##'    it.  The fifth color marks the edges forming the growing tree.
##'    The default value of the "color" parameter is
##'    c("navajowhite", "orangered3", "black", "white", "cyan4").
##'    Finally, by setting the "pdf" parameter to TRUE, this routine
##'    can also output the individual plots as a bunch of pdf files
##'    which can be included in a LaTeX animation.
##' 
##' @title Dijkstra' algorithm for shortest paths
##' @param g An igraph graph.
##' @param d Weights (lengths) of the edges of \eqn{g}.  It should be
##'   given as a \eqn{n\times n} symmetric distance matrix, with
##'   \eqn{n} being the number of vertices of \eqn{g}.  Please note
##'   that only the distances corresponding to edges of the graph
##'   \eqn{g} will be taken into account.
##' @param r Starting vertex --- root of the output tree.  It defaults
##'   to 1.
##' @param plot.steps Boolean indicating whether the routine should
##'   plot the intermediate steps of the algorithm to see the growing
##'   of the tree into the minimum distance spanning tree.  It
##'   defaults to FALSE.
##' @param color A set of colors to mark the vertices, vertex labels
##'   and edges added to the tree as it grows, see details below.  It
##'   is only used when plot.steps is set to TRUE.
##' @param pause Time interval (in seconds) between successive steps
##'   of the algorithm.  It is only used when plot.steps is set to
##'   TRUE and it defaults to 2.
##' @param pdf Boolean indicating whether the routine should eject pdf
##'   plots of intermediate steps of the algorithm, see details below.
##'   It is only used when plot.steps is set to TRUE and it defaults
##'   to FALSE.
##' @param ... Further parameters to pass to the plotting routine,
##'   especially the layout parameter.
##' @return A list with components: $tree, which is the spanning tree
##'   leading to minimum distances with r as root in igraph format;
##'   $distances, which is a \eqn{2\times n} matrix with distances
##'   from the root vertex to the remaining vertices, and $parents,
##'   which contains the parent of each vertex in the tree, except for
##'   the root which has no parent, so its entry is NA.
##' @importFrom grDevices dev.off
##' @author Cesar Asensio
##' @seealso [igraph::shortest_paths()] in the [igraph::igraph]
##'   package.
##' @examples
##' library(igraph)
##' g <- make_graph("Frucht")
##' n <- gorder(g)
##' set.seed(1)
##' d <- matrix(round(runif(n^2, min = 1, max = 10)), nrow = n)  # Distances
##' d <- d + t(d); for (i in 1:n) { d[i,i] <- 0 }          # Distance matrix
##' Td <- dijk(g, d, r = 1)
##' Td$distances
##' Td$parents
##' ## To plot the result:
##' ## z <- layout_with_gem(g)
##' ## T <- dijk(g, d, r = 1, plot.steps = TRUE, layout = z)
##' 
##' @encoding UTF-8
##' @md 
##' @export 
dijk <- function(g, d, r = 1,
                 plot.steps = FALSE,
                 color = c("navajowhite","orangered3", "black", "white", "cyan4"),
                 pause = 2, pdf = FALSE, ...) { # Dijkstra's algorithm
  n <- gorder(g)
  Vg <- V(g)
  U <- c()                # Will contain vertices added to the tree
  L <- rbind(1:n,         # Vertex labels...
             rep(Inf,n))  # ..and infinite initial distances...
  L[2,r] <- 0             # ...except for the root, which is 0
  p <- rep(NA,n)          # Predecessors

  ## Converting the distance matrix in a weight list.  We
  ## respect edgeid order.
  Eg <- as_edgelist(g)
  dl <- c()
  for (e in 1:nrow(Eg)) { dl <- c(dl,d[Eg[e,1],Eg[e,2]]) }

  ## Al vertices are colored with the null color (first of the
  ## list "color", a light color by default).  The added
  ## vertices will be colored using a dark color by default
  ## (second of the list "color"), so we need to adjust the
  ## added vertex label color accordingly from black (color[3])
  ## to white (color[4]).
  v.cols <- rep(color[1],n)
  v.labs <- rep(color[3],n)
  i <- 1
  
  if (plot.steps) {    # Plot the initial graph if desired
    if (pdf) {         # Plot a pdf version to build a pdfLaTeX animation
      pdf(file = paste(c("dijk-example-", i, ".pdf"), collapse = ""))
      i <- i+1
    }
    plot(g, edge.label = dl,
         vertex.color = color[1], vertex.label.color = color[3], ...)
    ## We should close the pdf device or pause to see the result stepwise
    if (pdf) { dev.off() } else { Sys.sleep(pause) }
  }

  T <- c()                           # Tree
  while (length(U) < n) {            # When |U| = n we will stop
    Uc <- setdiff(Vg,U)            # Non-added vertices
    k <- which.min(L[2,Uc])        # Take the minimum distance vertex...
    u <- L[1,Uc][k]                # ...which is this...
    U <- c(U,u)                    # ...and add it to the tree
    UciNu <- intersect(Uc,neighbors(g,u))  # Non-added vertices
    for (w in UciNu) {             # Update L:
      if (L[2,w] > L[2,u] + d[u,w]) {
        L[2,w] <- L[2,u] + d[u,w]
        p[w] <- u
      }
    }
    if (u != r) { T <- c(T,p[u],u) }

    Tg <- make_graph(T, n = n)
    
    if (plot.steps) { # Plot each step of the algorithm if desired
      v.cols[U] <- color[2]
      v.labs[U] <- color[4]
      if (pdf) {
        pdf(file = paste(c("dijkstra-example-", i, ".pdf"), collapse = ""))
        i <- i+1
      }
      plot(g, edge.label = dl,
           vertex.color = v.cols, vertex.label.color = v.labs, ...)
      plot(Tg, add = TRUE,
           edge.color = color[5], edge.width = 3,
           vertex.color = v.cols, vertex.label.color = v.labs, ...)
      ## We should close the pdf device or pause to see the result stepwise
      if (pdf) { dev.off() } else { Sys.sleep(pause) }
    }
  }

  list(tree = Tg, distances = L, parents = p)
}
