% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gisco-get-nuts.R
\encoding{UTF-8}
\name{gisco_get_nuts}
\alias{gisco_get_nuts}
\title{Territorial units for statistics (NUTS) dataset}
\source{
\url{https://gisco-services.ec.europa.eu/distribution/v2/}.

Copyright:
\url{https://ec.europa.eu/eurostat/web/gisco/geodata/administrative-units}.
}
\usage{
gisco_get_nuts(
  year = 2024,
  epsg = 4326,
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  resolution = 20,
  spatialtype = "RG",
  country = NULL,
  nuts_id = NULL,
  nuts_level = c("all", "0", "1", "2", "3"),
  ext = "gpkg"
)
}
\arguments{
\item{year}{character string or number. Release year of the file. One of
\Sexpr[stage=render,results=rd]{giscoR:::db_values("nuts",
"year",TRUE)}.}

\item{epsg}{character string or number. Projection of the map: 4-digit
\href{https://epsg.io/}{EPSG code}. One of:
\itemize{
\item \code{"4326"}: \href{https://epsg.io/4326}{WGS84}.
\item \code{"3035"}: \href{https://epsg.io/3035}{ETRS89 / ETRS-LAEA}.
\item \code{"3857"}: \href{https://epsg.io/3857}{Pseudo-Mercator}.
}}

\item{cache}{logical. Whether to do caching. Default is \code{TRUE}. See
\strong{Caching strategies} section in \code{\link[=gisco_set_cache_dir]{gisco_set_cache_dir()}}.}

\item{update_cache}{logical. Should the cached file be refreshed?. Default
is \code{FALSE}. When set to \code{TRUE} it would force a new download.}

\item{cache_dir}{character string. A path to a cache directory. See
\strong{Caching strategies} section in \code{\link[=gisco_set_cache_dir]{gisco_set_cache_dir()}}.}

\item{verbose}{logical. If \code{TRUE} displays informational messages.}

\item{resolution}{character string or number. Resolution of the geospatial
data. One of:
\itemize{
\item \code{"60"}: 1:60 million.
\item \code{"20"}: 1:20 million.
\item \code{"10"}: 1:10 million.
\item \code{"03"}: 1:3 million.
\item \code{"01"}: 1:1 million.
}}

\item{spatialtype}{character string. Type of geometry to be returned. Options
available are:
\itemize{
\item \code{"RG"}: Regions - \code{MULTIPOLYGON/POLYGON} object.
\item \code{"LB"}: Labels - \code{POINT} object.
\item \code{"BN"}: Boundaries - \code{LINESTRING} object.
}

\strong{Note that} arguments \code{country}, \code{nuts_level} and \code{nuts_id} would be
only applied when \code{spatialtype} is \code{"RG"} or \code{"LB"}.}

\item{country}{character vector of country codes. It could be either a
vector of country names, a vector of ISO3 country codes or a vector of
Eurostat country codes. See also \code{\link[countrycode:countrycode]{countrycode::countrycode()}}.}

\item{nuts_id}{Optional. A character vector of NUTS IDs.}

\item{nuts_level}{character string. NUTS level. One of \code{0},
\code{1}, \code{2}, \code{3} or \code{all} for all levels.}

\item{ext}{character. Extension of the file (default \code{"gpkg"}). One of
\Sexpr[stage=render,results=rd]{giscoR:::db_values("nuts",
"ext",TRUE)}.}
}
\value{
A \code{\link[sf:sf]{sf}} object.
}
\description{
The GISCO statistical unit dataset represents the NUTS (nomenclature of
territorial units for statistics) and statistical regions by means of
multipart polygon, polyline and point topology. The NUTS geographical
information is completed by attribute tables and a set of cartographic
help lines to better visualise multipart polygonal regions.

The NUTS are a hierarchical system divided into 3 levels:
\itemize{
\item NUTS 1: major socio-economic regions
\item NUTS 2: basic regions for the application of regional policies
\item NUTS 3: small regions for specific diagnoses.
}

Also, there is a NUTS 0 level, which usually corresponds to the national
boundaries.

\strong{Please note that} this function gets data from the aggregated GISCO
NUTS file, that contains data of all the countries at the requested NUTS
level(s). If you prefer to download individual NUTS files, please use
\code{\link[=gisco_get_unit_nuts]{gisco_get_unit_nuts()}}.
}
\details{
The NUTS nomenclature is a hierarchical classification of statistical
regions and subdivides the EU economic territory into regions of three
different levels (NUTS 1, 2 and 3, moving respectively from larger to smaller
territorial units). NUTS 1 is the most aggregated level. An additional
Country level (NUTS 0) is also available for countries where the nation at
statistical level does not coincide with the administrative boundaries.

The NUTS classification has been officially established through Commission
Delegated Regulation 2019/1755. A non-official NUTS-like classification has
been defined for the EFTA countries, candidate countries and potential
candidates based on a bilateral agreement between Eurostat and the respective
statistical agencies.

An introduction to the NUTS classification is available here:
\url{https://ec.europa.eu/eurostat/web/nuts/overview}.
}
\section{Note}{
Please check the download and usage provisions on \code{\link[=gisco_attributions]{gisco_attributions()}}.
}

\examples{
nuts2 <- gisco_get_nuts(nuts_level = 2)

library(ggplot2)

ggplot(nuts2) +
  geom_sf() +
  # ETRS89 / ETRS-LAEA
  coord_sf(
    crs = 3035, xlim = c(2377294, 7453440),
    ylim = c(1313597, 5628510)
  ) +
  labs(title = "NUTS-2 levels")
# NUTS-3 for Germany
germany_nuts3 <- gisco_get_nuts(nuts_level = 3, country = "Germany")

ggplot(germany_nuts3) +
  geom_sf() +
  labs(
    title = "NUTS-3 levels",
    subtitle = "Germany",
    caption = gisco_attributions()
  )


# Select specific regions
select_nuts <- gisco_get_nuts(nuts_id = c("ES2", "FRJ", "FRL", "ITC"))

ggplot(select_nuts) +
  geom_sf(aes(fill = CNTR_CODE)) +
  scale_fill_viridis_d()
}
\seealso{
\link{gisco_nuts_2024}, \code{\link[eurostat:get_eurostat_geospatial]{eurostat::get_eurostat_geospatial()}}.

See \code{\link[=gisco_bulk_download]{gisco_bulk_download()}} to perform a bulk download of datasets.

See \code{\link[=gisco_get_unit_nuts]{gisco_get_unit_nuts()}} to download single files.

See \code{\link[=gisco_id_api_nuts]{gisco_id_api_nuts()}} to download via GISCO ID service API.

Other statistical units datasets:
\code{\link{gisco_get_census}()},
\code{\link{gisco_get_coastal_lines}()},
\code{\link{gisco_get_lau}()},
\code{\link{gisco_get_urban_audit}()}
}
\concept{stats}
