% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/excursions.R
\name{excursions}
\alias{excursions}
\title{Excursion Sets and Contour Credibility Regions for Random Fields}
\usage{
excursions(
  alpha,
  u,
  mu,
  Q,
  type,
  n.iter = 10000,
  Q.chol,
  F.limit,
  vars,
  rho,
  reo,
  method = "EB",
  ind,
  max.size,
  verbose = 0,
  max.threads = 0,
  seed,
  prune.ind = FALSE
)
}
\arguments{
\item{alpha}{Error probability for the excursion set.}

\item{u}{Excursion or contour level.}

\item{mu}{Expectation vector.}

\item{Q}{Precision matrix.}

\item{type}{Type of region:
\describe{
\item{'>'}{positive excursion region}
\item{'<'}{negative excursion region}
\item{'!='}{contour avoiding region}
\item{'='}{contour credibility region}}}

\item{n.iter}{Number or iterations in the MC sampler that is used for approximating probabilities. The default value is 10000.}

\item{Q.chol}{The Cholesky factor of the precision matrix (optional).}

\item{F.limit}{The limit value for the computation of the F function. F is set to NA for all nodes where F<1-F.limit. Default is F.limit = \code{alpha}.}

\item{vars}{Precomputed marginal variances (optional).}

\item{rho}{Marginal excursion probabilities (optional). For contour regions, provide \eqn{P(X>u)}.}

\item{reo}{Reordering (optional).}

\item{method}{Method for handeling the latent Gaussian structure:
\describe{
\item{'EB'}{Empirical Bayes (default)}
\item{'QC'}{Quantile correction, rho must be provided if QC is used.}}}

\item{ind}{Indices of the nodes that should be analysed (optional).}

\item{max.size}{Maximum number of nodes to include in the set of interest (optional).}

\item{verbose}{Set to TRUE for verbose mode (optional).}

\item{max.threads}{Decides the number of threads the program can use. Set to 0 for using the maximum number of threads allowed by the system (default).}

\item{seed}{Random seed (optional).}

\item{prune.ind}{If \code{TRUE} and \code{ind} is supplied, then the result object is pruned to
contain only the active nodes specified by \code{ind}.}
}
\value{
\code{excursions} returns an object of class "excurobj" with the following elements
\item{E}{Excursion set, contour credible region, or contour avoiding set}
\item{G}{Contour map set. \eqn{G=1} for all nodes where the \eqn{mu > u}.}
\item{M}{Contour avoiding set. \eqn{M=-1} for all non-significant nodes. \eqn{M=0} for nodes where the process is significantly below \code{u} and \eqn{M=1} for all nodes where the field is significantly above \code{u}. Which values that should be present depends on what type of set that is calculated.}
\item{F}{The excursion function corresponding to the set \code{E} calculated or values up to \code{F.limit}}
\item{rho}{Marginal excursion probabilities}
\item{mean}{The mean \code{mu}.}
\item{vars}{Marginal variances.}
\item{meta}{A list containing various information about the calculation.}
}
\description{
\code{excursions} is one of the main functions in the package with the same name.
For an introduction to the package, see \code{\link[=excursions-package]{excursions-package()}}.
The function is used for calculating excursion sets, contour credible regions,
and contour avoiding sets for latent Gaussian models. Details on the function and the
package are given in the sections below.
}
\details{
The estimation of the region is done using sequential importance sampling with
\code{n.iter} samples. The procedure requires computing the marginal variances of
the field, which should be supplied if available. If not, they are computed using
the Cholesky factor of the precision matrix. The cost of this step can therefore be
reduced by supplying the Cholesky factor if it is available.

The latent structure in the latent Gaussian model can be handled in several different
ways. The default strategy is the EB method, which is
exact for problems with Gaussian posterior distributions. For problems with
non-Gaussian posteriors, the QC method can be used for improved results. In order to use
the QC method, the true marginal excursion probabilities must be supplied using the
argument \code{rho}.
Other more
complicated methods for handling non-Gaussian posteriors must be implemented manually
unless \code{INLA} is used to fit the model. If the model is fitted using \code{INLA},
the method \code{excursions.inla} can be used. See the Package section for further details
about the different options.
}
\examples{
## Create a tridiagonal precision matrix
n <- 21
Q.x <- sparseMatrix(
  i = c(1:n, 2:n), j = c(1:n, 1:(n - 1)), x = c(rep(1, n), rep(-0.1, n - 1)),
  dims = c(n, n), symmetric = TRUE
)
## Set the mean value function
mu.x <- seq(-5, 5, length = n)

## calculate the level 0 positive excursion function
res.x <- excursions(
  alpha = 1, u = 0, mu = mu.x, Q = Q.x,
  type = ">", verbose = 1, max.threads = 2
)

## Plot the excursion function and the marginal excursion probabilities
plot(res.x$F,
  type = "l",
  main = "Excursion function (black) and marginal probabilites (red)"
)
lines(res.x$rho, col = 2)
}
\references{
Bolin, D. and Lindgren, F. (2015) \emph{Excursion and contour uncertainty regions for latent Gaussian models}, JRSS-series B, vol 77, no 1, pp 85-106.

Bolin, D. and Lindgren, F. (2018), \emph{Calculating Probabilistic Excursion Sets and Related Quantities Using excursions}, Journal of Statistical Software, vol 86, no 1, pp 1-20.
}
\seealso{
\code{\link[=excursions-package]{excursions-package()}}, \code{\link[=excursions.inla]{excursions.inla()}}, \code{\link[=excursions.mc]{excursions.mc()}}
}
\author{
David Bolin \email{davidbolin@gmail.com} and Finn Lindgren \email{finn.lindgren@gmail.com}
}
