% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/validation.R
\name{create_method}
\alias{create_method}
\title{Create a cross-fitting method specification}
\usage{
create_method(
  target,
  list_nuisance = NULL,
  folds,
  repeats,
  mode = c("estimate", "predict"),
  eval_fold = if (mode == "estimate") 1L else 0L,
  fold_allocation = c("independence", "overlap", "disjoint"),
  aggregate_panels = NULL,
  aggregate_repeats = NULL
)
}
\arguments{
\item{target}{A function representing the target functional. It must
accept nuisance predictions as arguments (named after nuisances) and
optionally a \code{data} argument.}

\item{list_nuisance}{Optional named list of nuisance specifications
created by \code{\link{create_nuisance}}.}

\item{folds}{Positive integer giving the number of folds \eqn{K}. May
be \code{NULL}, in which case \code{\link{crossfit_multi}} will
infer a minimal feasible \code{K} from the dependency structure.}

\item{repeats}{Positive integer giving the number of repetitions.}

\item{mode}{Cross-fitting mode. Either \code{"estimate"} (target
returns numeric estimates) or \code{"predict"} (target returns a
cross-fitted predictor).}

\item{eval_fold}{Integer giving the width (in folds) of the
evaluation window for the target. Must be \code{> 0} for
\code{mode = "estimate"} and \code{0} for \code{mode = "predict"}.
If omitted, the default is \code{1L} for \code{"estimate"} and
\code{0L} for \code{"predict"}.}

\item{fold_allocation}{Fold allocation strategy; one of
\code{"independence"}, \code{"overlap"}, or \code{"disjoint"}.}

\item{aggregate_panels}{Aggregation function for panel-level
results, typically one of \code{\link{mean_estimate}},
\code{\link{median_estimate}}, \code{\link{mean_predictor}},
\code{\link{median_predictor}}, or a custom function. May be
\code{NULL}, in which case a global default can be supplied via
\code{\link{crossfit_multi}}.}

\item{aggregate_repeats}{Aggregation function for repetition-level
results, typically one of \code{\link{mean_estimate}},
\code{\link{median_estimate}}, \code{\link{mean_predictor}},
\code{\link{median_predictor}}, or a custom function. May be
\code{NULL}, in which case a global default can be supplied via
\code{\link{crossfit_multi}}.}
}
\value{
A method specification list suitable for use in
\code{\link{crossfit}} or \code{\link{crossfit_multi}}.
}
\description{
Helper to create a method specification for
\code{\link{crossfit}} / \code{\link{crossfit_multi}}. A method
bundles together:
\itemize{
\item a target functional \code{target()},
\item a named list of nuisance specifications,
\item cross-fitting geometry (\code{folds}, \code{repeats},
\code{eval_fold}, \code{mode}, \code{fold_allocation}),
\item and panel / repetition aggregation functions.
}
}
\details{
The returned list is validated by \code{validate_method()} to ensure
structural soundness, but the validated object is not stored: you are
free to modify the returned method before passing it to
\code{\link{crossfit}} or \code{\link{crossfit_multi}}.

By default, \code{eval_fold} is chosen to be \code{1L} when
\code{mode = "estimate"} and \code{0L} when \code{mode = "predict"}.
If you override \code{eval_fold}, it must satisfy these constraints:
positive integer for \code{"estimate"}, zero for \code{"predict"}.
}
\examples{
set.seed(1)
n <- 50
x <- rnorm(n)
y <- x + rnorm(n)

# Nuisance: regression for E[Y | X]
nuis_y <- create_nuisance(
  fit = function(data, ...) lm(y ~ x, data = data),
  predict = function(model, data, ...) predict(model, newdata = data)
)

# Target: mean squared error of the nuisance predictor
target_mse <- function(data, nuis_y, ...) {
  mean((data$y - nuis_y)^2)
}

m <- create_method(
  target = target_mse,
  list_nuisance = list(nuis_y = nuis_y),
  folds = 2,
  repeats = 1,
  eval_fold = 1L,
  mode = "estimate",
  fold_allocation = "independence",
  aggregate_panels  = mean_estimate,
  aggregate_repeats = mean_estimate
)

str(m)
}
