% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nhclu_dbscan.R
\name{nhclu_dbscan}
\alias{nhclu_dbscan}
\title{Non-hierarchical clustering: DBSCAN}
\usage{
nhclu_dbscan(
  dissimilarity,
  index = names(dissimilarity)[3],
  minPts = NULL,
  eps = NULL,
  plot = TRUE,
  algorithm_in_output = TRUE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{dissimilarity}{The output object from \code{\link[=dissimilarity]{dissimilarity()}} or
\code{\link[=similarity_to_dissimilarity]{similarity_to_dissimilarity()}}, or a \code{dist} object. If a \code{data.frame} is
used, the first two columns should represent pairs of sites (or any pair of
nodes), and the subsequent column(s) should contain the dissimilarity indices.}

\item{index}{The name or number of the dissimilarity column to use. By
default, the third column name of \code{dissimilarity} is used.}

\item{minPts}{A \code{numeric} vector or a single \code{numeric} value specifying the
\code{minPts} argument of \code{\link[dbscan:dbscan]{dbscan::dbscan()}}. \code{minPts} is the minimum number of
points to form a dense region. By default, it is set to the natural logarithm
of the number of sites in \code{dissimilarity}. See Details for guidance on
choosing this parameter.}

\item{eps}{A \code{numeric} vector or a single \code{numeric} value specifying the \code{eps}
argument of \code{\link[dbscan:dbscan]{dbscan::dbscan()}}. \code{eps} specifies how similar points should be
to each other to be considered part of a cluster. See Details for guidance on
choosing this parameter.}

\item{plot}{A \code{boolean} indicating whether the k-nearest neighbor distance
plot should be displayed.}

\item{algorithm_in_output}{A \code{boolean} indicating whether the original output
of \link[dbscan:dbscan]{dbscan::dbscan} should be included in the output. Defaults to \code{TRUE} (see
Value).}

\item{verbose}{A \code{boolean} indicating whether to
display progress messages. Set to \code{FALSE} to suppress these messages.}

\item{...}{Additional arguments to be passed to \code{dbscan()} (see
\link[dbscan:dbscan]{dbscan::dbscan}).}
}
\value{
A \code{list} of class \code{bioregion.clusters} with five components:
\enumerate{
\item{\strong{name}: A \code{character} string containing the name of the algorithm.}
\item{\strong{args}: A \code{list} of input arguments as provided by the user.}
\item{\strong{inputs}: A \code{list} of characteristics of the clustering process.}
\item{\strong{algorithm}: A \code{list} of all objects associated with the clustering
procedure, such as original cluster objects (only if
\code{algorithm_in_output = TRUE}).}
\item{\strong{clusters}: A \code{data.frame} containing the clustering results.}}

If \code{algorithm_in_output = TRUE}, the \code{algorithm} slot includes the output of
\link[dbscan:dbscan]{dbscan::dbscan}.
}
\description{
This function performs non-hierarchical clustering based on dissimilarity
using the Density-Based Spatial Clustering of Applications with Noise
(DBSCAN) algorithm.
}
\details{
The DBSCAN (Density-Based Spatial Clustering of Applications with Noise)
algorithm clusters points based on the density of neighbors around each
data point. It requires two main arguments: \code{minPts}, the minimum number of
points to identify a core, and \code{eps}, the radius used to find neighbors.

\strong{Choosing minPts:} This determines how many points are necessary to form a
cluster. For example, what is the minimum number of sites expected in a
bioregion? Choose a value sufficiently large for your dataset and expectations.

\strong{Choosing eps:} This determines how similar sites should be to form a
cluster. If \code{eps} is too small, most points will be considered too distinct
and marked as noise. If \code{eps} is too large, clusters may merge. The value of
\code{eps} depends on \code{minPts}. It is recommended to choose \code{eps} by identifying
a knee in the k-nearest neighbor distance plot.

By default, the function attempts to find a knee in this curve
automatically, but the result is uncertain. Users should inspect the graph
and modify \code{eps} accordingly. To explore \code{eps} values, run the function
initially without defining \code{eps}, review the recommendations, and adjust
as needed based on clustering results.
}
\examples{
comat <- matrix(sample(0:1000, size = 500, replace = TRUE, prob = 1/1:1001),
20, 25)
rownames(comat) <- paste0("Site",1:20)
colnames(comat) <- paste0("Species",1:25)

dissim <- dissimilarity(comat, metric = "all")

clust1 <- nhclu_dbscan(dissim, index = "Simpson")
clust2 <- nhclu_dbscan(dissim, index = "Simpson", eps = 0.2)
clust3 <- nhclu_dbscan(dissim, index = "Simpson", minPts = c(5, 10, 15, 20),
     eps = c(.1, .15, .2, .25, .3))
     
}
\references{
Hahsler M, Piekenbrock M & Doran D (2019) Dbscan: Fast density-based
clustering with R. \emph{Journal of Statistical Software}, 91(1), 1--30.
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a4_2_non_hierarchical_clustering.html}.

Associated functions:
\link{nhclu_clara} \link{nhclu_clarans} \link{nhclu_kmeans} \link{nhclu_pam} \link{nhclu_affprop}
}
\author{
Boris Leroy (\email{leroy.boris@gmail.com}) \cr
Pierre Denelle (\email{pierre.denelle@gmail.com}) \cr
Maxime Lenormand (\email{maxime.lenormand@inrae.fr})
}
