#' Check if the package is being loaded via devtools::load_all()
#'
#' @return TRUE if loaded via devtools, FALSE otherwise
#' @keywords internal
is_devtools_load <- function() {
  identical(Sys.getenv("DEVTOOLS_LOAD"), "true")
}

#' @title Get valid box path
#' @description
#' Construct a box path that will allow box imports for the current package. This is done by adding the package path to the box path option if it is not already there.
#' @param libname The path to the library.
#' @param pkgname The name of the package.
#' @keywords internal
get_valid_boxpath <- function(libname, pkgname) {
  current_box_path <- getOption("box.path", character(0))

  pkg_path <- file.path(libname, pkgname)
  dev_path <- file.path(pkg_path, "inst")

  boxpath_defined <- function(path) {
    if (length(current_box_path) == 0) {
      return(FALSE)
    }
    grepl(path, current_box_path)
  }

  if (all(c(
    boxpath_defined(pkg_path),
    boxpath_defined(dev_path)
  ))) {
    return(current_box_path)
  } # Already valid

  unique(c(current_box_path, pkg_path, dev_path))
}

#' Reference runtime dependencies so R CMD check detects Imports usage.
#'
#' These packages are required by the modules shipped under `inst/artma`
#' and are accessed via `pkg::fun()` at runtime. Because those modules are
#' not parsed when `R CMD check` scans the package's R sources, we touch the
#' symbols here to make the dependency explicit.
#'
#' @keywords internal
#' @noRd
register_runtime_dependencies <- function() {
  invisible(list(
    ggplot2::ggplot,
    ggtext::element_markdown,
    NlcOptim::solnl,
    lmtest::coeftest,
    plm::plm,
    plm::vcovHC,
    sandwich::vcovCL,
    sandwich::vcovHC,
    yaml::read_yaml,
    yaml::write_yaml
  ))
}

#' @title .onLoad hook for package initialization
#' @description Called when the package is loaded.
#' @param libname The path to the library.
#' @param pkgname The name of the package.
#' @return `NULL` Sets up the package on load
#' @keywords internal
.onLoad <- function(libname, pkgname) {
  op <- options()
  op.artma <- list(
    # artma.abc = xyz
  )

  # Optional set
  toset <- !(names(op.artma) %in% names(op))
  if (any(toset)) options(op.artma[toset])

  # Mandatory set
  options(box.path = get_valid_boxpath(libname, pkgname))

  register_runtime_dependencies()

  invisible()
}

#' @title .onUnload hook for package detachment
#' @description Called when the package is detached.
#' @note The box imports no longer work after the package is detached.
#' @return `NULL` Cleans up the package on unload
#' @keywords internal
.onUnload <- function(libpath) {
  # Remove options with the "artma" prefix
  opts_to_remove <- names(options())[startsWith(names(options()), "artma")]
  options(stats::setNames(rep(list(NULL), length(opts_to_remove)), opts_to_remove))

  invisible()
}
